from collections import Callable
from math import ceil, exp
from unittest import TestCase

from auxiliary import ExtendedTestCase

from math2.calc import newton
from math2.econ import Bond, CompoundInterest, ContinuousInterest, EffectiveInterest, Mortgage, NominalInterest, \
    SubperiodInterest, fp, pa, \
    pf, pg
from math2.misc import interpolate


class PS1TestCase(TestCase):
    def test_1(self) -> None:
        self.assertLess(8000 + 1200, 6800 + 2500)
        self.assertAlmostEqual(newton(lambda mv: 8000 + 1200 - (mv + 2500), 6800), 6700)

    def test_2(self) -> None:
        self.assertAlmostEqual(newton(lambda t: 7500 - t * 1700, 0), 4.411764705882353)
        self.assertAlmostEqual(newton(lambda t: 9000 - t * 2200, 0), 4.09090909)

    def test_3(self) -> None:
        self.assertAlmostEqual((1 + 0.05 / 1000000) ** 1000000, exp(0.05))

    def test_4(self) -> None:
        e = EffectiveInterest(0.034)
        p, t = 100000, 4

        self.assertAlmostEqual(e.to_nominal(2).rate, 0.03371581102178567)
        self.assertAlmostEqual(e.to_nominal(12).rate, 0.033481397886386155)
        self.assertAlmostEqual(e.to_nominal(365).rate, 0.03343630748129267)
        self.assertAlmostEqual(e.to_continuous().rate, 0.03343477608623742)

        self.assertAlmostEqual(p * e.to_nominal(2).to_factor(t), p * e.to_nominal(12).to_factor(t))
        self.assertAlmostEqual(p * e.to_nominal(12).to_factor(t), p * e.to_nominal(365).to_factor(t))
        self.assertAlmostEqual(p * e.to_nominal(365).to_factor(t), p * e.to_continuous().to_factor(t))
        self.assertAlmostEqual(p * e.to_continuous().to_factor(t), 114309.4552336)

    def test_5(self) -> None:
        fv, c, cd = 100, 0.023, 0.02

        self.assertAlmostEqual(ContinuousInterest(c).to_effective().rate, 0.02326653954721758)
        self.assertAlmostEqual(fv / ContinuousInterest(c).to_factor(0.5), 98.8565872247913)
        self.assertAlmostEqual(fv / ContinuousInterest(c).to_factor(0.75), 98.28979294344309)
        self.assertAlmostEqual(fv / ContinuousInterest(cd).to_factor(0.5), 99.0049833749168)
        self.assertAlmostEqual(fv / ContinuousInterest(cd).to_factor(0.75), 98.51119396030627)

    def test_6(self) -> None:
        self.assertAlmostEqual(EffectiveInterest(0.08).to_subperiod(12).rate, 0.00643403011000343)
        self.assertAlmostEqual(NominalInterest(0.035, 252).to_effective().rate, 0.03561719190449408)
        self.assertAlmostEqual(NominalInterest(0.04, 4).to_continuous().rate, 0.039801323412672354)
        self.assertAlmostEqual(CompoundInterest.from_factor(SubperiodInterest(0.015, 12).to_factor(1), 4)
                               .to_continuous().rate, 0.04466583748125169)
        self.assertAlmostEqual(CompoundInterest.from_factor(CompoundInterest.from_factor(
            NominalInterest(0.012, 3).to_factor(1), 1 / 4).to_factor(3), 1).to_nominal(6).rate, 0.1454477030768886)


class PS2TestCase(ExtendedTestCase):
    def test_1(self) -> None:
        factor = NominalInterest(0.15, 4).to_factor(4 / 12) * ContinuousInterest(0.11).to_factor(5 / 12)

        self.assertAlmostEqual(CompoundInterest.from_factor(factor, 9 / 12).rate, 0.134915472328168)

    def test_2(self) -> None:
        p, i, n = 100, 0.05, 10

        self.assertAlmostEqual(p * pa(i, n), p * ((1 + i) ** n - 1) / (i * (1 + i) ** n))

    def test_3(self) -> None:
        cases = [
            (1, 99.52, 0.05943811, 0.05773868),
            (4, 99.01, 0.03029791, 0.02984799),
            (8, 97.64, 0.03647384, 0.03582441),
            (12, 95.85, 0.04329682, 0.04238572),
        ]

        for m, p, x, y in cases:
            self.assertAlmostEqual(newton(lambda y: p * EffectiveInterest(y).to_factor(m / 12) - 100, 0), x)
            self.assertAlmostEqual(newton(lambda y: p * ContinuousInterest(y).to_factor(m / 12) - 100, 0), y)

        self.assertAlmostEqual(interpolate(7, 4, 8, 0.03029791, 0.03647384), 0.0349298575)
        self.assertAlmostEqual(interpolate(7, 4, 8, 0.02984799, 0.03582441), 0.034330305)

        self.assertAlmostEqual(newton(lambda y: 1.03029791 ** (4. / 12) * (1 + y) ** (8. / 12) - 1.04329682, 0),
                               0.04985765)

    def test_4(self) -> None:
        data = {
            2: 99.11,
            6: 98.72,
            9: 96.85,
            12: 95.97,
        }

        r: dict[float, float] = {
            t: newton(lambda y: p * ContinuousInterest(y).to_factor(t / 12) - 100, (0))
            for t, p in data.items()
        }

        r[4] = interpolate(4, 2, 6, r[2], r[6])
        r[5.5] = interpolate(5.5, 2, 6, r[2], r[6])
        r[7] = interpolate(7, 6, 9, r[6], r[9])
        r[10] = interpolate(10, 9, 12, r[9], r[12])

        self.assertAlmostEqual(100 * ContinuousInterest(r[2]).to_factor(-1. / 12), 99.55400544428134)
        self.assertAlmostEqual(100 * ContinuousInterest(r[4]).to_factor(-4. / 12), 98.68531340424114)
        self.assertAlmostEqual(100 * ContinuousInterest(r[5.5]).to_factor(-5.5 / 12), 98.66834503291024)
        self.assertAlmostEqual(100 * ContinuousInterest(r[7]).to_factor(-7 / 12), 98.18488742486127)
        self.assertAlmostEqual(100 * ContinuousInterest(r[10]).to_factor(-10 / 12), 96.54750684004732)
        self.assertAlmostEqual(2000 * pf(r[2], 1. / 12) + 500 * pf(r[4], 4. / 12) - 1200 * pf(r[7], 7. / 12) - 1000
                               * pf(r[10], 10. / 12) + 500 * pf(r[12], 1), 820.3872407904064)

    def test_5(self) -> None:
        i = 0.02
        options = (
            15500 * pa(i, 10),
            140000,
            155000 * pf(i, 5),
            170000 * pf(i, 10),
        )

        self.assertSequenceAlmostEqual(options, (139230.06759675476, 140000, 140388.27552363696, 139459.21097877636))
        self.assertEqual(max(range(4), key=lambda i: options[i]), 2)

    def test_6(self) -> None:
        i = 0.02
        options = (
            155000 * pf(i, 5),
            10000 * pa(i, 10) + 1500 * pg(i, 10),
        )

        self.assertAlmostEqual(options[1], 148258.50062422457)
        self.assertEqual(max(range(2), key=lambda i: options[i]), 1)

    def test_7(self) -> None:
        i = 0.02
        option = 10000 * pa(i, 10, g=0.05)

        self.assertAlmostEqual(option, 112086.97925088322)


# class PS3TestCase(TestCase):
#     def test_1(self) -> None:
#         m = Mortgage.from_dtv(2995000, 0.2)
#         i1 = NominalInterest(0.02, 2)
#         i2 = NominalInterest(0.04, 2)
#         p = m.payment(i1)
#
#         self.assertAlmostEqual(p, 10145.891129693951)
#         self.assertAlmostEqual(p * 12 * 3, 365252.08066898223)
#         self.assertAlmostEqual(m.pay(i1, 5).payment(i2), 12128.043601452593)
#
#     def test_2(self) -> None:
#         self.assertAlmostEqual(Bond(100, 0, 2, 3 / 12).present_worth(EffectiveInterest(0.07)), 98.32275876142123)
#         self.assertAlmostEqual(Bond(100, 0, 2, 5 / 12).present_worth(EffectiveInterest(0.07)), 97.17391685967232)
#         self.assertAlmostEqual(Bond(100, 0, 2, 3).present_worth(EffectiveInterest(0.07)), 81.35006443077528)
#         self.assertAlmostEqual(Bond.from_rate(100, 0.04, 2, 3).present_worth(EffectiveInterest(0.07)),
#                                92.00717047033226)
#         self.assertAlmostEqual(Bond.from_rate(100, 0.06, 2, 3.25).present_worth(EffectiveInterest(0.07)),
#                                97.13753584095278)
#
#     def test_3(self) -> None:
#         self.assertAlmostEqual(
#             newton(lambda y: Bond.from_rate(100, 0.07, 2, 3).present_worth(EffectiveInterest(y)) - 100, 0.1), 0.07)
#         self.assertAlmostEqual(Bond.from_rate(100, 0.04, 2, 3).present_worth(EffectiveInterest(0.05)) + 100 * 0.04 / 2,
#                                99.24593731921009)
#         self.assertAlmostEqual(
#             newton(lambda y: Bond.from_rate(100, 0.03, 2, 2.25).present_worth(EffectiveInterest(y)) - 100, 0.1), 0.03)
#         self.assertAlmostEqual(Bond.from_rate(100, 0.07, 2, 2.25).present_worth(EffectiveInterest(0.05)),
#                                104.20662940110009)
#         self.assertAlmostEqual(
#             newton(lambda c: Bond.from_rate(100, c, 2, 2.25).present_worth(EffectiveInterest(0.03)) - 114, 0.1),
#             0.09481118)
#
#     def test_4(self) -> None:  # TODO
#         ...
#
#     def test_5(self) -> None:
#         y = newton(lambda y: Bond.from_rate(100, 0.07, 2, 7.5).present_worth(EffectiveInterest(y)) * fp(y / 2, 0.5) - 108, 0.1)
#
#         b: Callable[[float], float] = lambda c: Bond.from_rate(1000, c, 2, 9).present_worth(EffectiveInterest(y))
#         c = ceil(newton(lambda c: 9500000 / 2 - (4400 * b(c)), 0.1) / 0.0025) * 0.0025
#         self.assertAlmostEqual(c, 0.0725)
#         self.assertAlmostEqual(4400 * b(c), 4802235.185695931)
#         c = ceil(newton(lambda c: 9500000 / 2 / (1 - 0.008) - (4400 * b(c)), (0.1)) / 0.0025) * 0.0025
#         self.assertAlmostEqual(c, 0.0725)
#         self.assertAlmostEqual(4400 * b(c) * (1 - 0.008), 4763817.304210364)
#
#     def test_6(self) -> None:
#         self.assertAlmostEqual(Mortgage.from_down(500000, 50000).payment(NominalInterest(0.060755, 2)),
#                                2899.3558026129626)
#         self.assertAlmostEqual(Mortgage.from_down(500000, 50000).pay(NominalInterest(0.060755, 2), 5, 700).payment(
#             NominalInterest(0.060755, 2)), 3490.3113416458878)
#         self.assertLess(Mortgage.from_down(500000, 50000, 25).pay(NominalInterest(0.060755, 2), 3).principal,
#                         Mortgage.from_down(500000, 50000, 25).pay(NominalInterest(0.060755, 2), 3, 700).principal)

# class PS6TestCase(ExtendedTestCase):
#     def test_1(self) -> None:
#         data = ((-41000, 6100, 7),
#                 (-32000, 6700, 7),
#                 (-28000, 5700, 5),
#                 (-28000, 12600, 5),
#                 (-36000, 9000, 7),
#                 (-27000, 10600, 6),
#                 (-53000, 6700, 5),
#                 (-50000, 15000, 6),
#                 (-32000, 6900, 7),
#                 (-42000, 14600, 5))
#
#         irrs = list(map(lambda d: SimpleProject(*d).irr, data))
#
#         self.assertSequenceAlmostEqual(irrs, (
#             0.010261108929599895,
#             0.10584583010815002,
#             0.005929015028005828,
#             0.3494328573992243,
#             0.16326709023510008,
#             0.31754169406374866,
#             - 0.13571830650187303,
#             0.1990541470961173,
#             0.114956469240095,
#             0.2178733729868983,
#         ))
#
#         points = sorted(range(len(data)), key=lambda pt: -irrs[pt])
#
#         cost = 0
#         marr = 0
#
#         for pt in points:
#             cost -= data[pt][0]
#
#             if cost > 100000:
#                 break
#
#             marr = irrs[pt]
#
#         self.assertAlmostEqual(marr, 0.2178733729868983)
#
#     def test_2(self) -> None:
#         self.assertEqual(from_table(
#             [[],
#              [0.17],
#              [0.14, 0.075],
#              [0.19, 0.209, 0.286],
#              [0.2, 0.127, 0.257, 0.229],
#              [0.18, 0.177, 0.192, 0.158, 0.117],
#              [0.13, 0.128, 0.132, 0.106, 0.081, 0.062]],
#             0.12,
#         ), 4)
#
#         self.assertEqual(from_table(
#             [[],
#              [0.14],
#              [0.20, 0.29],
#              [0.24, 0.32, 0.36],
#              [0.21, 0.24, 0.22, 0.11],
#              [0.17, 0.18, 0.15, 0.08, 0.06],
#              [0.17, 0.18, 0.16, 0.12, 0.13, 0.19]],
#             0.12,
#         ), 3)
#
#     def test_3(self) -> None:
#         table = [[],
#                  [0.1096],
#                  [0.132, 0.286],
#                  [0.1205, 0.17, -0.058],
#                  [0.1293, 0.189, 0.112, 0.228],
#                  [0.1286, 0.177, 0.112, 0.187, 0.113],
#                  [0.1113, 0.113, 0.079, 0.094, 0.069, 0.063]]
#
#         self.assertEqual(from_table(table, 0.04), 6)
#         self.assertEqual(from_table(table, 0.06), 6)
#         self.assertEqual(from_table(table, 0.08), 5)
#         self.assertEqual(from_table(table, 0.10), 5)
#         self.assertEqual(from_table(table, 0.12), 2)
#         self.assertEqual(from_table(table, 0.14), 0)
#
#     def test_4(self) -> None:
#         """
#
#         :return:
#         """
#         ...
#
#     def test_5(self) -> None:
#         """
#         A startup pharmaceutical company, Lexcol Pharma, has passed all but the last stage of regulatory
# approval for its patented drug that has been in development for 8 years. Lexcol will know in two
# years if their drug passes the last regulatory approval stage. An unsuccessful approval would
# essentially close the company, resulting in little measurable value. If successful, however, to
# manufacture the drug, Lexcol will need to invest $500 million in a unique facility (consider this cost
# to be a one-time investment cost to be paid as soon as the design of the facility is to begin) and the
# time to design and construct the facility is 4 years (i.e. if the investment were made now and the
# regulatory approval were successful, then cash-flows will start in the 5th year as per regular
# convention). The cash-flows for the company are estimated to be $200 million per year until the
# patent expires, after which, the cash-flows will likely drop substantially, and for valuations
# purposes, can be estimated as $10 million per year perpetually. The patent is expected to expire
# 12 years from now. Assume that Lexcol is fully equity financed (and will remain so) by wealthy,
# well diversified “angel” investors. A financial research analyst who specializes in the pharma
# industry has estimated an 8% hurdle rate (MARR) to value the cash-flows, assuming successful
# regulatory approval. Clearly, Lexcol has a choice to start the development of the manufacturing
# facility now, or after regulatory approval (recall that the $500 million will need to be invested as
# soon as the decision is made to design and construct the facility). How confident would Lexcol
# have to be to pass the last stage regulatory approval to be indifferent to building the facility now
# or waiting until after the approval?
#         :return:
#         """


# if __name__ == '__main__':
#     main()
