# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hyperion']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'hyperion-py',
    'version': '0.7.4',
    'description': 'Hyperion Ambient Lighting Python Package',
    'long_description': '<img src="images/hyperion-logo.png"\n     alt="Hyperion logo"\n     width="20%"\n     align="right"\n     style="float: right; margin: 10px 0px 20px 20px;" />\n     \n[![PyPi](https://img.shields.io/pypi/v/hyperion-py.svg?style=flat-square)](https://pypi.org/project/hyperion-py/)\n[![PyPi](https://img.shields.io/pypi/pyversions/hyperion-py.svg?style=flat-square)](https://pypi.org/project/hyperion-py/)\n[![Build Status](https://img.shields.io/github/workflow/status/dermotduffy/hyperion-py/Build?style=flat-square)](https://github.com/dermotduffy/hyperion-py/actions/workflows/build.yaml)\n[![Test Coverage](https://img.shields.io/codecov/c/gh/dermotduffy/hyperion-py?style=flat-square)](https://codecov.io/gh/dermotduffy/hyperion-py)\n[![License](https://img.shields.io/github/license/dermotduffy/hyperion-py.svg?style=flat-square)](LICENSE)\n[![BuyMeCoffee](https://img.shields.io/badge/buy%20me%20a%20coffee-donate-yellow.svg?style=flat-square)](https://www.buymeacoffee.com/dermotdu)\n\n# Hyperion Library\n\nPython library for\n[Hyperion-NG](https://github.com/hyperion-project/hyperion.ng). See [JSON\nAPI](https://docs.hyperion-project.org/en/json/) for more details about the\ninputs and outputs of this library.\n\n# Installation\n\n```bash\n$ pip3 install hyperion-py\n```\n\n# Usage\n\n## Data model philosophy\n\nWhilst not universally true, this library attempts to precisely represent the\ndata model, API and parameters as defined in the [Hyperion JSON\ndocumentation](https://docs.hyperion-project.org/en/json/). Thus no attempt is\nmade (intentionally) to present convenient accessors/calls at a finer level of\ngranularity than the model already supports. This is to ensure the client has a\ndecent chance at staying functional regardless of underlying data model changes\nfrom the server, and the responsibility to match the changes to the server\'s\ndata model (e.g. new Hyperion server features) belong to the caller.\n\n### Constructor Arguments\n\nThe following arguments may be passed to the `HyperionClient` constructor:\n\n|Argument|Type|Default|Description|\n|--------|----|-------|-----------|\n|host    |`str`||Host or IP to connect to|\n|port    |`int`|19444|Port to connect to|\n|default_callback|`callable`|None|A callable for Hyperion callbacks. See [callbacks](#callbacks)|\n|callbacks|`dict`|None|A dictionary of callables keyed by the update name. See [callbacks](#callbacks)|\n|token|`str`|None|An authentication token|\n|instance|`int`|0|An instance id to switch to upon connection|\n|origin|`str`|"hyperion-py"|An arbitrary string describing the calling application|\n|timeout_secs|`float`|5.0|The number of seconds to wait for a server response or connection attempt before giving up. See [timeouts](#timeouts)|\n|retry_secs|`float`|30.0|The number of seconds between connection attempts|\n|raw_connection|`bool`|False|If True, the connect call will establish the network connection but not attempt to authenticate, switch to the required instance or load state. The client must call `async_client_login` to login, `async_client_switch_instance` to switch to the configured instance and `async_get_serverinfo` to load the state manually. This may be useful if the caller wishes to communicate with the server prior to authentication.|\n\n### Connection, disconnection and client control calls\n\n   * `async_client_connect()`: Connect the client.\n   * `async_client_disconnect()`: Disconnect the client.\n   * `async_client_login()`: Login a connected client. Automatically called by\n     `async_client_connect()` unless the `raw_connection` constructor argument is True.\n   * `async_client_switch_instance()`: Switch to the configured instance on the Hyperion\n     server. Automatically called by `async_client_connect()` unless the `raw_connection`\n     constructor argument is True.\n\n### Native API Calls\n\nAll API calls can be found in\n[client.py](https://github.com/dermotduffy/hyperion-py/blob/master/hyperion/client.py).\nAll async calls start with `async_`.\n\n|Send request and await response|Send request only|Documentation|\n|-------------------------------|-----------------|-------------|\n|async_clear|async_send_clear|[Docs](https://docs.hyperion-project.org/en/json/Control.html#clear)|\n|async_image_stream_start|async_send_image_stream_start|[Docs](https://docs.hyperion-project.org/en/json/Control.html#live-image-stream)|\n|async_image_stream_stop|async_send_image_stream_stop|[Docs](https://docs.hyperion-project.org/en/json/Control.html#live-image-stream)|\n|async_is_auth_required|async_send_is_auth_required|[Docs](https://docs.hyperion-project.org/en/json/Authorization.html#authorization-check)|\n|async_led_stream_start|async_send_led_stream_start|[Docs](https://docs.hyperion-project.org/en/json/Control.html#live-led-color-stream)|\n|async_led_stream_stop|async_send_led_stream_stop|[Docs](https://docs.hyperion-project.org/en/json/Control.html#live-led-color-stream)|\n|async_login|async_send_login|[Docs](https://docs.hyperion-project.org/en/json/Authorization.html#login-with-token)|\n|async_logout|async_send_logout|[Docs](https://docs.hyperion-project.org/en/json/Authorization.html#logout)|\n|async_request_token|async_send_request_token|[Docs](https://docs.hyperion-project.org/en/json/Authorization.html#request-a-token)|\n|async_request_token_abort|async_send_request_token_abort|[Docs](https://docs.hyperion-project.org/en/json/Authorization.html#request-a-token)|\n|async_get_serverinfo|async_send_get_serverinfo|[Docs](https://docs.hyperion-project.org/en/json/ServerInfo.html#parts)|\n|async_set_adjustment|async_send_set_adjustment|[Docs](https://docs.hyperion-project.org/en/json/Control.html#adjustments)|\n|async_set_color|async_send_set_color|[Docs](https://docs.hyperion-project.org/en/json/Control.html#set-color)|\n|async_set_component|async_send_set_component|[Docs](https://docs.hyperion-project.org/en/json/Control.html#control-components)|\n|async_set_effect|async_send_set_effect|[Docs](https://docs.hyperion-project.org/en/json/Control.html#set-effect)|\n|async_set_image|async_send_set_image|[Docs](https://docs.hyperion-project.org/en/json/Control.html#set-image)|\n|async_set_led_mapping_type|async_send_set_led_mapping_type|[Docs](https://docs.hyperion-project.org/en/json/Control.html#led-mapping)|\n|async_set_sourceselect|async_send_set_sourceselect|[Docs](https://docs.hyperion-project.org/en/json/Control.html#source-selection)|\n|async_set_videomode|async_send_set_videomode|[Docs](https://docs.hyperion-project.org/en/json/Control.html#video-mode)|\n|async_start_instance|async_send_start_instance|[Docs](https://docs.hyperion-project.org/en/json/Control.html#control-instances)|\n|async_stop_instance|async_send_stop_instance|[Docs](https://docs.hyperion-project.org/en/json/Control.html#control-instances)|\n|async_switch_instance|async_send_switch_instance|[Docs](https://docs.hyperion-project.org/en/json/Control.html#api-instance-handling)|\n|async_sysinfo|async_send_sysinfo|[Docs](https://docs.hyperion-project.org/en/json/ServerInfo.html#system-hyperion)|\n\nNote that the `command` and `subcommand` keys shown in the above linked\ndocumentation will automatically be included in the calls the client sends, and\ndo not need to be specified.\n\n## Client inputs / outputs\n\nThe API parameters and output are all as defined in the [JSON API\ndocumentation](https://docs.hyperion-project.org/en/json/).\n\n## Example usage:\n\n```python\n#!/usr/bin/env python\n"""Simple Hyperion client read demonstration."""\n\nimport asyncio\n\nfrom hyperion import client, const\n\nHOST = "hyperion"\n\n\nasync def print_brightness() -> None:\n    """Print Hyperion brightness."""\n\n    async with client.HyperionClient(HOST) as hyperion_client:\n        assert hyperion_client\n\n        adjustment = hyperion_client.adjustment\n        assert adjustment\n\n        print("Brightness: %i%%" % adjustment[0][const.KEY_BRIGHTNESS])\n\n\nif __name__ == "__main__":\n    asyncio.get_event_loop().run_until_complete(print_brightness())\n```\n\n## Running in the background\n\nA background `asyncio task` runs to process all post-connection inbound data\n(e.g. request responses, or subscription updates from state changes on the\nserver side). This background task must either be started post-connection, or\nstart (and it will itself establish connection).\n\nOptionally, this background task can call callbacks back to the user.\n\n### Waiting for responses\n\nIf the user makes a call that does not have `_send_` in the name (see table\nabove), the function call will wait for the response and return it to the\ncaller. This matching of request & response is done via the `tan` parameter. If\nnot specified, the client will automatically attach a `tan` integer, and this\nwill be visible in the returned output data. This matching is necessary to\ndifferentiate between responses due to requests, and "spontaneous data" from\nsubscription updates.\n\n#### Example: Waiting for a response\n\n```python\n#!/usr/bin/env python\n"""Simple Hyperion client request demonstration."""\n\nimport asyncio\n\nfrom hyperion import client\n\nHOST = "hyperion"\n\n\nasync def print_if_auth_required() -> None:\n    """Print whether auth is required."""\n\n    hc = client.HyperionClient(HOST)\n    await hc.async_client_connect()\n\n    result = await hc.async_is_auth_required()\n    print("Result: %s" % result)\n\n    await hc.async_client_disconnect()\n\n\nasyncio.get_event_loop().run_until_complete(print_if_auth_required())\n```\n\nOutput:\n\n```\nResult: {\'command\': \'authorize-tokenRequired\', \'info\': {\'required\': False}, \'success\': True, \'tan\': 1}\n```\n\n#### Example: Sending commands\n\nA slightly more complex example that sends commands (clears the Hyperion source\nselect at a given priority, then sets color at that same priority).\n\n```python\n#!/usr/bin/env python\n"""Simple Hyperion client request demonstration."""\n\nimport asyncio\nimport logging\nimport sys\n\nfrom hyperion import client\n\nHOST = "hyperion"\nPRIORITY = 20\n\n\nasync def set_color() -> None:\n    """Set red color on Hyperion."""\n\n    async with client.HyperionClient(HOST) as hc:\n        assert hc\n\n        if not await hc.async_client_connect():\n            logging.error("Could not connect to: %s", HOST)\n            return\n\n        if not client.ResponseOK(\n            await hc.async_clear(priority=PRIORITY)\n        ) or not client.ResponseOK(\n            await hc.async_set_color(\n                color=[255, 0, 0], priority=PRIORITY, origin=sys.argv[0]\n            )\n        ):\n            logging.error("Could not clear/set_color on: %s", HOST)\n            return\n\n\nlogging.basicConfig(stream=sys.stdout, level=logging.DEBUG)\nasyncio.get_event_loop().run_until_complete(set_color())\n```\n\n#### Example: Starting and switching instances\n\nThe following example will start a stopped instance, wait for it to be ready,\nthen switch to it. Uses [callbacks](#callbacks), discussed below.\n\n\n```python\n#!/usr/bin/env python\n"""Simple Hyperion client request demonstration."""\n\nfrom __future__ import annotations\n\nimport asyncio\nimport logging\nimport sys\nfrom typing import Any\n\nfrom hyperion import client\n\nHOST = "hyperion"\nPRIORITY = 20\n\n\nasync def instance_start_and_switch() -> None:\n    """Wait for an instance to start."""\n\n    instance_ready = asyncio.Event()\n\n    def instance_update(json: dict[str, Any]) -> None:\n        for data in json["data"]:\n            if data["instance"] == 1 and data["running"]:\n                instance_ready.set()\n\n    async with client.HyperionClient(\n        HOST, callbacks={"instance-update": instance_update}\n    ) as hc:\n        assert hc\n\n        if not client.ResponseOK(await hc.async_start_instance(instance=1)):\n            logging.error("Could not start instance on: %s", HOST)\n            return\n\n        # Blocks waiting for the instance to start.\n        await instance_ready.wait()\n\n        if not client.ResponseOK(await hc.async_switch_instance(instance=1)):\n            logging.error("Could not switch instance on: %s", HOST)\n            return\n\n\nlogging.basicConfig(stream=sys.stdout, level=logging.DEBUG)\nasyncio.get_event_loop().run_until_complete(instance_start_and_switch())\n```\n\n<a name="callbacks"></a>\n### Callbacks\n\nThe client can be configured to callback as the Hyperion server reports new\nvalues. There are two classes of callbacks supported:\n\n   * **default_callback**: This callback will be called when a more specific callback is not specified.\n   * **callbacks**: A dict of callbacks keyed on the Hyperion subscription \'command\' (see [JSON API documentation](https://docs.hyperion-project.org/en/json/))\n\nCallbacks can be specified in the `HyperionClient` constructor\n(`default_callback=` or `callbacks=` arguments) or after construction via the\n`set_callbacks()` and `set_default_callback()` methods.\n\nAs above, the `callbacks` dict is keyed on the relevant Hyperion subscription\n`command` (e.g. `components-update`, `priorities-update`). The client also\nprovides a custom callback with command `client-update` of the following\nform:\n\n```python\n{"command": "client-update",\n "connected": True,\n "logged-in": True,\n "instance": 0,\n "loaded-state": True}\n```\n\nThis can be used to take special action as the client connects or disconnects from the server.\n\n#### Example: Callbacks\n\n```python\n#!/usr/bin/env python\n"""Simple Hyperion client callback demonstration."""\n\nfrom __future__ import annotations\n\nimport asyncio\nfrom typing import Any\n\nfrom hyperion import client\n\nHOST = "hyperion"\n\n\ndef callback(json: dict[str, Any]) -> None:\n    """Sample callback function."""\n\n    print("Received Hyperion callback: %s" % json)\n\n\nasync def show_callback() -> None:\n    """Show a default callback is called."""\n\n    async with client.HyperionClient(HOST, default_callback=callback):\n        pass\n\n\nif __name__ == "__main__":\n    asyncio.get_event_loop().run_until_complete(show_callback())\n```\n\nOutput, showing the progression of connection stages:\n\n```\nReceived Hyperion callback: {\'connected\': True, \'logged-in\': False, \'instance\': None, \'loaded-state\': False, \'command\': \'client-update\'}\nReceived Hyperion callback: {\'connected\': True, \'logged-in\': True, \'instance\': None, \'loaded-state\': False, \'command\': \'client-update\'}\nReceived Hyperion callback: {\'connected\': True, \'logged-in\': True, \'instance\': 0, \'loaded-state\': False, \'command\': \'client-update\'}\nReceived Hyperion callback: {\'command\': \'serverinfo\', ... }\nReceived Hyperion callback: {\'connected\': True, \'logged-in\': True, \'instance\': 0, \'loaded-state\': True, \'command\': \'client-update\'}\n```\n\n## ThreadedHyperionClient\n\nA `ThreadedHyperionClient` is also provided as a convenience wrapper to for\nnon-async code. The `ThreadedHyperionClient` wraps the async calls with\nnon-async versions (methods are named as shown above, except do not start with\n`async_`).\n\n### Waiting for the thread to initialize the client\n\nThe thread must be given a chance to initialize the client prior to interaction\nwith it. This method call will block the caller until the client has been initialized.\n\n   * wait_for_client_init()\n\n### Example use of Threaded client\n\n```python\n#!/usr/bin/env python\n"""Simple Threaded Hyperion client demonstration."""\n\nfrom hyperion import client, const\n\nHOST = "hyperion"\n\nif __name__ == "__main__":\n    hyperion_client = client.ThreadedHyperionClient(HOST)\n\n    # Start the asyncio loop in a new thread.\n    hyperion_client.start()\n\n    # Wait for the client to initialize in the new thread.\n    hyperion_client.wait_for_client_init()\n\n    # Connect the client.\n    hyperion_client.client_connect()\n\n    print("Brightness: %i%%" % hyperion_client.adjustment[0][const.KEY_BRIGHTNESS])\n\n    # Disconnect the client.\n    hyperion_client.client_disconnect()\n\n    # Stop the loop (will stop the thread).\n    hyperion_client.stop()\n\n    # Join the created thread.\n    hyperion_client.join()\n```\n\nOutput:\n\n```\nBrightness: 59%\n```\n\n## Exceptions / Errors \n\n### Philosophy\n\nHyperionClient strives not to throw an exception regardless of network\ncircumstances, reconnection will automatically happen in the background.\nExceptions are only raised (intentionally) for instances of likely programmer\nerror.\n\n### HyperionError\n\nNot directly raised, but other exceptions inherit from this.\n\n### HyperionClientTanNotAvailable\n\nException raised if a `tan` parameter is provided to an API call, but that\n`tan` parameter is already being used by another in-progress call. Users\nshould either not specify `tan` at all (and the client library will\nautomatically manage it in an incremental fashion), or if specified manually,\nit is the caller\'s responsibility to ensure no two simultaneous calls share a\n`tan` (as otherwise the client would not be able to match the call to the\nresponse, and this exception will be raised automatically prior to the call).\n\n### "Task was destroyed but it is pending!"\n\nIf a `HyperionClient` object is connected but destroyed prior to disconnection, a warning message may be printed ("Task was destroyed but it is pending!"). To avoid this, ensure to always call `async_client_disconnect` prior to destruction of a connected client. Alternatively use the async context manager:\n\n```python\nasync with client.HyperionClient(TEST_HOST, TEST_PORT) as hc:\n    if not hc:\n        return\n    ...\n```\n\n<a name="timeouts"></a>\n## Timeouts\n\nThe client makes liberal use of timeouts, which may be specified at multiple levels:\n\n   * In the client constructor argument `timeout_secs`, used for connection and requests.\n   * In each request using a `timeout_secs` argument to the individual calls\n\nTimeout values:\n\n   * `None`: If `None` is used as a timeout, the client will wait forever.\n   * `0`: If `0` is used as a timeout, the client default (specified in the constructor) will be used.\n   * `>0.0`: This number of seconds (or partial seconds) will be used.\n\nBy default, all requests will honour the `timeout_secs` specified in the client constructor unless explicitly overridden and defaults to 5 seconds (see [const.py](https://github.com/dermotduffy/hyperion-py/blob/master/hyperion/const.py#L95)). The one exception to this is the `async_send_request_token` which has a much larger default (180 seconds, see [const.py](https://github.com/dermotduffy/hyperion-py/blob/master/hyperion/const.py#L96)) as this request involves the user needing the interact with the Hyperion UI prior to the call being able to return.\n\n\n## Helpers\n\n### ResponseOK\n\nA handful of convenience callable classes are provided to determine whether\nserver responses were successful.\n\n   * `ResponseOK`: Whether any Hyperion command response was successful (general).\n   * `ServerInfoResponseOK`: Whether a `async_get_serverinfo` was successful.\n   * `LoginResponseOK`: Whether an `async_login` was successful.\n   * `SwitchInstanceResponseOK`: Whether an `async_switch_instance` command was successful.\n\n#### Example usage\n\n```\nif not client.ResponseOK(await hc.async_clear(priority=PRIORITY))\n```\n\n### Auth ID\n\nWhen requesting an auth token, a 5-character ID can be specified to ensure the\nadmin user is authorizing the right request from the right origin. By default\nthe `async_request_token` will randomly generate an ID, but if one is required\nto allow the user to confirm a match, it can be explicitly provided. In this case,\nthis helper method is made available.\n\n   * `generate_random_auth_id`: Generate a random 5-character auth ID for external display and inclusion in a call to `async_request_token`.\n\n#### Example usage\n\n```\nauth_id  = hc.generate_random_auth_id()\nhc.async_send_login(comment="Trustworthy actor", id=auth_id)\n# Show auth_id to the user to allow them to verify the origin of the request,\n# then have them visit the Hyperion UI.\n```\n',
    'author': 'Dermot Duffy',
    'author_email': 'dermot.duffy@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dermotduffy/hyperion-py',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
