"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: (_b = props.expiration) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.days(365),
                }],
            websiteIndexDocument: 'index.html',
        });
        // CloudFront distribution
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
            },
            certificate,
            domainNames: [domainName],
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const handler = new shortener_function_1.ShortenerFunction(this, 'handler', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        bucket.grantPut(handler);
        bucket.grantPutAcl(handler);
        table.grant(handler, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.LambdaRestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            handler,
            defaultMethodOptions: props.apiGatewayAuthorizer
                ? { authorizer: props.apiGatewayAuthorizer }
                : undefined,
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.4.25" };
//# sourceMappingURL=data:application/json;base64,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