import logging
import os
import sys
from argparse import ArgumentParser
from logging.config import dictConfig

from google.protobuf import text_format

from pythie_serving import create_grpc_server
from pythie_serving.tensorflow_proto.tensorflow_serving.config import (
    model_server_config_pb2,
)


def run():
    model_choice_set = {"xgboost", "lightgbm", "treelite", "sklearn", "table"}
    model_choice_str = ",".join(model_choice_set)

    parser = ArgumentParser(description=f"A GRPC server to serve different kind of model amongst: {model_choice_str}")
    parser.add_argument(
        "model_config_file_path",
        type=str,
        help="Path to a model config file of the format "
        "https://www.tensorflow.org/tfx/serving/serving_config#model_server_configuration",
    )
    parser.add_argument(
        "--worker-count",
        default=1,
        type=int,
        help="Number of concurrent threads for the GRPC server. "
        "Make sure the chosen model_platform is thread safe before increasing this number",
    )
    parser.add_argument(
        "--maximum-concurrent-rpcs",
        default=1,
        type=int,
        help="The maximum number of concurrent RPCs this server "
        "will service before returning RESOURCE_EXHAUSTED status, or -1 to indicate no limit.",
    )
    parser.add_argument("--port", default=9090, type=int, help="Port number to listen to")

    # set up logger
    dictConfig(
        {
            "disable_existing_loggers": True,
            "version": 1,
            "formatters": {
                "console_formatter": {
                    "format": "%(asctime)s - %(name)s - %(levelname)s - %(message)s",
                    "datefmt": "%Y-%m-%d %H:%M:%S",
                    "()": "logging.Formatter",
                }
            },
            "handlers": {
                "stdout_handler": {
                    "class": "logging.StreamHandler",
                    "formatter": "console_formatter",
                    "level": "INFO",
                    "stream": sys.stdout,
                }
            },
            "root": {
                "level": "INFO",
                "handlers": ["stdout_handler"],
                "formatter": ["console_formatter"],
            },
        }
    )

    ns = parser.parse_args()

    if not os.path.exists(ns.model_config_file_path):
        raise ValueError(f"No config file found at {ns.model_config_file_path}")

    model_server_config = model_server_config_pb2.ModelServerConfig()
    with open(ns.model_config_file_path) as opened_config_file:
        text_format.Parse(opened_config_file.read(), model_server_config)

    maximum_concurrent_rpcs = ns.maximum_concurrent_rpcs
    if maximum_concurrent_rpcs < 0:
        maximum_concurrent_rpcs = None  # grpc.server takes None to accept unlimited amount of connections

    server = create_grpc_server(
        model_server_config=model_server_config,
        worker_count=ns.worker_count,
        maximum_concurrent_rpcs=maximum_concurrent_rpcs,
        port=ns.port,
    )

    logger = logging.getLogger("pythie_serving")

    server.start()

    timeout = os.environ.get("GRPC_SERVER_TIMEOUT", None)
    # wait _for_termination() returns True if a timeout was supplied and it was reached. False otherwise.
    timeout_reached = server.wait_for_termination(timeout=float(timeout) if timeout is not None else None)
    if timeout_reached:
        logger.warning(f"Server shut down after timeout of {timeout}s was reached")


if __name__ == "__main__":
    run()
