"""
Example showing how to synthesize a 4-qubit QFT program using QFAST.
"""

import numpy as np

from qfast import *


# The 4-qubit QFT unitary matrix.
qft4 = np.array(
[[ 0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ,
   0.25             +0.j               ,  0.25             +0.j               ],
 [ 0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ,
   0.25             +0.j               , -0.25             +0.j               ],
 [ 0.25             +0.j               ,  0.               +0.25j             ,
  -0.25             +0.j               ,  0.               -0.25j             ,
   0.25             +0.j               ,  0.               +0.25j             ,
  -0.25             +0.j               ,  0.               -0.25j             ,
   0.25             +0.j               ,  0.               +0.25j             ,
  -0.25             +0.j               ,  0.               -0.25j             ,
   0.25             +0.j               ,  0.               +0.25j             ,
  -0.25             +0.j               ,  0.               -0.25j             ],
 [ 0.25             +0.j               ,  0.               -0.25j             ,
  -0.25             +0.j               ,  0.               +0.25j             ,
   0.25             +0.j               ,  0.               -0.25j             ,
  -0.25             +0.j               ,  0.               +0.25j             ,
   0.25             +0.j               ,  0.               -0.25j             ,
  -0.25             +0.j               ,  0.               +0.25j             ,
   0.25             +0.j               ,  0.               -0.25j             ,
  -0.25             +0.j               ,  0.               +0.25j             ],
 [ 0.25             +0.j               ,  0.176776695296637+0.176776695296637j,
   0.               +0.25j             , -0.176776695296637+0.176776695296637j,
  -0.25             +0.j               , -0.176776695296637-0.176776695296637j,
   0.               -0.25j             ,  0.176776695296637-0.176776695296637j,
   0.25             +0.j               ,  0.176776695296637+0.176776695296637j,
   0.               +0.25j             , -0.176776695296637+0.176776695296637j,
  -0.25             +0.j               , -0.176776695296637-0.176776695296637j,
   0.               -0.25j             ,  0.176776695296637-0.176776695296637j],
 [ 0.25             +0.j               , -0.176776695296637-0.176776695296637j,
   0.               +0.25j             ,  0.176776695296637-0.176776695296637j,
  -0.25             +0.j               ,  0.176776695296637+0.176776695296637j,
   0.               -0.25j             , -0.176776695296637+0.176776695296637j,
   0.25             +0.j               , -0.176776695296637-0.176776695296637j,
   0.               +0.25j             ,  0.176776695296637-0.176776695296637j,
  -0.25             +0.j               ,  0.176776695296637+0.176776695296637j,
   0.               -0.25j             , -0.176776695296637+0.176776695296637j],
 [ 0.25             +0.j               , -0.176776695296637+0.176776695296637j,
   0.               -0.25j             ,  0.176776695296637+0.176776695296637j,
  -0.25             +0.j               ,  0.176776695296637-0.176776695296637j,
   0.               +0.25j             , -0.176776695296637-0.176776695296637j,
   0.25             +0.j               , -0.176776695296637+0.176776695296637j,
   0.               -0.25j             ,  0.176776695296637+0.176776695296636j,
  -0.25             +0.j               ,  0.176776695296637-0.176776695296637j,
   0.               +0.25j             , -0.176776695296637-0.176776695296636j],
 [ 0.25             +0.j               ,  0.176776695296637-0.176776695296637j,
   0.               -0.25j             , -0.176776695296637-0.176776695296637j,
  -0.25             +0.j               , -0.176776695296637+0.176776695296637j,
   0.               +0.25j             ,  0.176776695296637+0.176776695296637j,
   0.25             +0.j               ,  0.176776695296637-0.176776695296637j,
   0.               -0.25j             , -0.176776695296637-0.176776695296636j,
  -0.25             +0.j               , -0.176776695296637+0.176776695296637j,
   0.               +0.25j             ,  0.176776695296637+0.176776695296636j],
 [ 0.25             +0.j               ,  0.230969883127822+0.095670858091272j,
   0.176776695296637+0.176776695296637j,  0.095670858091273+0.230969883127822j,
   0.               +0.25j             , -0.095670858091272+0.230969883127822j,
  -0.176776695296637+0.176776695296637j, -0.230969883127821+0.095670858091273j,
  -0.25             +0.j               , -0.230969883127822-0.095670858091272j,
  -0.176776695296637-0.176776695296637j, -0.095670858091273-0.230969883127822j,
   0.               -0.25j             ,  0.095670858091272-0.230969883127822j,
   0.176776695296637-0.176776695296637j,  0.230969883127822-0.095670858091273j],
 [ 0.25             +0.j               , -0.230969883127822-0.095670858091272j,
   0.176776695296637+0.176776695296637j, -0.095670858091273-0.230969883127822j,
   0.               +0.25j             ,  0.095670858091272-0.230969883127822j,
  -0.176776695296637+0.176776695296637j,  0.230969883127821-0.095670858091273j,
  -0.25             +0.j               ,  0.230969883127822+0.095670858091272j,
  -0.176776695296637-0.176776695296637j,  0.095670858091273+0.230969883127822j,
   0.               -0.25j             , -0.095670858091272+0.230969883127822j,
   0.176776695296637-0.176776695296637j, -0.230969883127822+0.095670858091273j],
 [ 0.25             +0.j               , -0.095670858091272+0.230969883127822j,
  -0.176776695296637-0.176776695296637j,  0.230969883127821-0.095670858091273j,
   0.               +0.25j             , -0.230969883127822-0.095670858091272j,
   0.176776695296637-0.176776695296637j,  0.095670858091273+0.230969883127821j,
  -0.25             +0.j               ,  0.095670858091272-0.230969883127822j,
   0.176776695296637+0.176776695296637j, -0.230969883127822+0.095670858091273j,
   0.               -0.25j             ,  0.230969883127822+0.095670858091272j,
  -0.176776695296637+0.176776695296637j, -0.095670858091273-0.230969883127821j],
 [ 0.25             +0.j               ,  0.095670858091272-0.230969883127822j,
  -0.176776695296637-0.176776695296637j, -0.230969883127821+0.095670858091273j,
   0.               +0.25j             ,  0.230969883127822+0.095670858091272j,
   0.176776695296637-0.176776695296637j, -0.095670858091273-0.230969883127821j,
  -0.25             +0.j               , -0.095670858091272+0.230969883127822j,
   0.176776695296637+0.176776695296637j,  0.230969883127822-0.095670858091273j,
   0.               -0.25j             , -0.230969883127822-0.095670858091272j,
  -0.176776695296637+0.176776695296637j,  0.095670858091273+0.230969883127821j],
 [ 0.25             +0.j               ,  0.095670858091273+0.230969883127822j,
  -0.176776695296637+0.176776695296637j, -0.230969883127822-0.095670858091272j,
   0.               -0.25j             ,  0.230969883127821-0.095670858091273j,
   0.176776695296637+0.176776695296637j, -0.095670858091272+0.230969883127822j,
  -0.25             +0.j               , -0.095670858091273-0.230969883127822j,
   0.176776695296637-0.176776695296637j,  0.230969883127822+0.095670858091272j,
   0.               +0.25j             , -0.230969883127822+0.095670858091273j,
  -0.176776695296637-0.176776695296637j,  0.095670858091272-0.230969883127822j],
 [ 0.25             +0.j               , -0.095670858091273-0.230969883127822j,
  -0.176776695296637+0.176776695296637j,  0.230969883127822+0.095670858091272j,
   0.               -0.25j             , -0.230969883127822+0.095670858091273j,
   0.176776695296637+0.176776695296636j,  0.095670858091272-0.230969883127822j,
  -0.25             +0.j               ,  0.095670858091273+0.230969883127822j,
   0.176776695296637-0.176776695296637j, -0.230969883127822-0.095670858091272j,
   0.               +0.25j             ,  0.230969883127822-0.095670858091273j,
  -0.176776695296637-0.176776695296637j, -0.095670858091272+0.230969883127822j],
 [ 0.25             +0.j               , -0.230969883127821+0.095670858091273j,
   0.176776695296637-0.176776695296637j, -0.095670858091272+0.230969883127822j,
   0.               -0.25j             ,  0.095670858091273+0.230969883127821j,
  -0.176776695296637-0.176776695296637j,  0.230969883127822+0.095670858091272j,
  -0.25             +0.j               ,  0.230969883127821-0.095670858091273j,
  -0.176776695296637+0.176776695296637j,  0.095670858091272-0.230969883127822j,
   0.               +0.25j             , -0.095670858091273-0.230969883127821j,
   0.176776695296637+0.176776695296637j, -0.230969883127822-0.095670858091272j],
 [ 0.25             +0.j               ,  0.230969883127822-0.095670858091273j,
   0.176776695296637-0.176776695296637j,  0.095670858091272-0.230969883127822j,
   0.               -0.25j             , -0.095670858091273-0.230969883127821j,
  -0.176776695296637-0.176776695296637j, -0.230969883127822-0.095670858091272j,
  -0.25             +0.j               , -0.230969883127822+0.095670858091273j,
  -0.176776695296637+0.176776695296637j, -0.095670858091272+0.230969883127822j,
   0.               +0.25j             ,  0.095670858091273+0.230969883127821j,
   0.176776695296637+0.176776695296637j,  0.230969883127822+0.095670858091272j]]
)


def synthesize ( utry ):
    """
    Synthesize a unitary matrix and return qasm code using QFAST with
    qiskit's kak native tool.

    Args:
        utry (np.ndarray): The unitary matrix to synthesize

    Returns:
        (str): Qasm code implementing utry
    """

    circ = Circuit( utry )
    block_size = get_native_tool( "kak" ).get_native_block_size()

    # Perform decomposition until the circuit is represented in blocks
    # of at most size block_size
    circ.hierarchically_decompose( block_size )

    # Perform instantiation converting blocks to small qasm circuits
    qasm_list = [ instantiation( "kak", block.utry ) for block in circ.blocks ]
    locations = circ.get_locations()

    # Recombine all small circuits into a large one
    out_qasm = recombination( qasm_list, locations )

    return out_qasm


# Synthesize the qft4 unitary and print results
qasm = synthesize( qft4 )
print( qasm )