import * as iot from '@aws-cdk/aws-iot-alpha';
import { CommonActionProps } from './common-action-props';
/**
 * MQTT Quality of Service (QoS) indicates the level of assurance for delivery of an MQTT Message.
 *
 * @see https://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html#mqtt-qos
 */
export declare enum MqttQualityOfService {
    /**
     * QoS level 0. Sent zero or more times.
     * This level should be used for messages that are sent over reliable communication links or that can be missed without a problem.
     */
    ZERO_OR_MORE_TIMES = 0,
    /**
     * QoS level 1. Sent at least one time, and then repeatedly until a PUBACK response is received.
     * The message is not considered complete until the sender receives a PUBACK response to indicate successful delivery.
     */
    AT_LEAST_ONCE = 1
}
/**
 * Configuration properties of an action to republish MQTT messages.
 */
export interface IotRepublishMqttActionProps extends CommonActionProps {
    /**
     * The Quality of Service (QoS) level to use when republishing messages.
     *
     * @see https://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html#mqtt-qos
     *
     * @default MqttQualityOfService.ZERO_OR_MORE_TIMES
     */
    readonly qualityOfService?: MqttQualityOfService;
}
/**
 * The action to put the record from an MQTT message to republish another MQTT topic.
 */
export declare class IotRepublishMqttAction implements iot.IAction {
    private readonly topic;
    private readonly qualityOfService?;
    private readonly role?;
    /**
     * @param topic The MQTT topic to which to republish the message.
     * @param props Optional properties to not use default.
     */
    constructor(topic: string, props?: IotRepublishMqttActionProps);
    bind(rule: iot.ITopicRule): iot.ActionConfig;
}
