"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        // For some reason path mapping to 'things/{thingName}/shadow/name/{shadowName}' results in 403 error, hence this shortcut
        this.addResourceMethod(namedShadowResource, props, 'topics/$aws/things/{thingName}/shadow/name/{shadowName}/update', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
        if (props.apiGatewayCreateApiKey === true) {
            const cfnMethod = apiMethod.node.findChild('Resource');
            cfnMethod.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W59',
                            reason: 'When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE'
                        }]
                }
            };
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
//# sourceMappingURL=data:application/json;base64,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