"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const redirect_function_1 = require("./redirect-function");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: props.expiration ?? aws_cdk_lib_1.Duration.days(365),
                }],
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            bucketName: props.bucketName ?? `cloudstructs-url-shortener-${domainName}`,
        });
        // Redirect function
        const redirectFunction = new redirect_function_1.RedirectFunction(this, 'Redirect');
        bucket.grantRead(redirectFunction);
        // CloudFront distribution
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: redirectFunction,
                    },
                ],
            },
            certificate,
            domainNames: [domainName],
            httpVersion: 'http2and3',
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const shortenerFunction = new shortener_function_1.ShortenerFunction(this, 'Shortener', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        if (props.corsAllowOrigins) {
            shortenerFunction.addEnvironment('CORS_ALLOW_ORIGINS', props.corsAllowOrigins.join(' '));
        }
        bucket.grantPut(shortenerFunction);
        table.grant(shortenerFunction, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.RestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
            defaultCorsPreflightOptions: props.corsAllowOrigins
                ? { allowOrigins: props.corsAllowOrigins }
                : undefined,
        });
        this.api.root.addMethod('ANY', new apigateway.LambdaIntegration(shortenerFunction), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.api.root
            .addResource('{proxy+}')
            .addMethod('ANY', new apigateway.LambdaIntegration(shortenerFunction), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.6.13" };
//# sourceMappingURL=data:application/json;base64,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