from pydantic import BaseModel, create_model
from typing import Any, Type


class ModelSerializer:
    
    @staticmethod
    def model_from_schema(schema: dict, model_name: str) -> Type[BaseModel]:
        """
        Recreate a Pydantic model from a given schema.
        
        Args:
            schema (dict): The schema dictionary generated by .schema() method of a Pydantic model.
            model_name (str): The name of the Pydantic model to create.

        Returns:
            Type[BaseModel]: A new Pydantic model based on the provided schema.
        """
        # Prepare field definitions
        fields = {}
        required_fields = set(schema.get('required', []))

        for field_name, field_info in schema.get('properties', {}).items():
            # Extract field type
            field_type = field_info.get('type', 'Any')

            # Map JSON schema types to Python types
            if field_type == 'string':
                python_type = str
            elif field_type == 'integer':
                python_type = int
            elif field_type == 'number':
                python_type = float
            elif field_type == 'boolean':
                python_type = bool
            elif field_type == 'array':
                python_type = list
            elif field_type == 'object':
                python_type = dict  # This can be improved to handle nested models
            else:
                python_type = Any
            
            # Check if the field is required
            is_required = field_name in required_fields
            fields[field_name] = (python_type, ...) if is_required else (python_type, None)

        # Create and return the model
        return create_model(model_name, **fields)