import * as acm from '@aws-cdk/aws-certificatemanager';
import { Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as route53 from '@aws-cdk/aws-route53';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { EngineVersion } from './version';
/**
 * Configures the capacity of the cluster such as the instance type and the number of instances.
 *
 * @stability stable
 */
export interface CapacityConfig {
    /**
     * The number of instances to use for the master node.
     *
     * @default - no dedicated master nodes
     * @stability stable
     */
    readonly masterNodes?: number;
    /**
     * The hardware configuration of the computer that hosts the dedicated master node, such as `m3.medium.search`. For valid values, see [Supported Instance Types] (https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html) in the Amazon OpenSearch Service Developer Guide.
     *
     * @default - r5.large.search
     * @stability stable
     */
    readonly masterNodeInstanceType?: string;
    /**
     * The number of data nodes (instances) to use in the Amazon OpenSearch Service domain.
     *
     * @default - 1
     * @stability stable
     */
    readonly dataNodes?: number;
    /**
     * The instance type for your data nodes, such as `m3.medium.search`. For valid values, see [Supported Instance Types](https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html) in the Amazon OpenSearch Service Developer Guide.
     *
     * @default - r5.large.search
     * @stability stable
     */
    readonly dataNodeInstanceType?: string;
    /**
     * The number of UltraWarm nodes (instances) to use in the Amazon OpenSearch Service domain.
     *
     * @default - no UltraWarm nodes
     * @stability stable
     */
    readonly warmNodes?: number;
    /**
     * The instance type for your UltraWarm node, such as `ultrawarm1.medium.search`. For valid values, see [UltraWarm Storage Limits] (https://docs.aws.amazon.com/opensearch-service/latest/developerguide/limits.html#limits-ultrawarm) in the Amazon OpenSearch Service Developer Guide.
     *
     * @default - ultrawarm1.medium.search
     * @stability stable
     */
    readonly warmInstanceType?: string;
}
/**
 * Specifies zone awareness configuration options.
 *
 * @stability stable
 */
export interface ZoneAwarenessConfig {
    /**
     * Indicates whether to enable zone awareness for the Amazon OpenSearch Service domain.
     *
     * When you enable zone awareness, Amazon OpenSearch Service allocates the nodes and replica
     * index shards that belong to a cluster across two Availability Zones (AZs)
     * in the same region to prevent data loss and minimize downtime in the event
     * of node or data center failure. Don't enable zone awareness if your cluster
     * has no replica index shards or is a single-node cluster. For more information,
     * see [Configuring a Multi-AZ Domain]
     * (https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html)
     * in the Amazon OpenSearch Service Developer Guide.
     *
     * @default - false
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * If you enabled multiple Availability Zones (AZs), the number of AZs that you want the domain to use.
     *
     * Valid values are 2 and 3.
     *
     * @default - 2 if zone awareness is enabled.
     * @stability stable
     */
    readonly availabilityZoneCount?: number;
}
/**
 * The configurations of Amazon Elastic Block Store (Amazon EBS) volumes that are attached to data nodes in the Amazon OpenSearch Service domain.
 *
 * For more information, see
 * [Amazon EBS]
 * (https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AmazonEBS.html)
 * in the Amazon Elastic Compute Cloud Developer Guide.
 *
 * @stability stable
 */
export interface EbsOptions {
    /**
     * Specifies whether Amazon EBS volumes are attached to data nodes in the Amazon OpenSearch Service domain.
     *
     * @default - true
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * The number of I/O operations per second (IOPS) that the volume supports.
     *
     * This property applies only to the Provisioned IOPS (SSD) EBS
     * volume type.
     *
     * @default - iops are not set.
     * @stability stable
     */
    readonly iops?: number;
    /**
     * The size (in GiB) of the EBS volume for each data node.
     *
     * The minimum and
     * maximum size of an EBS volume depends on the EBS volume type and the
     * instance type to which it is attached.  For  valid values, see
     * [EBS volume size limits]
     * (https://docs.aws.amazon.com/opensearch-service/latest/developerguide/limits.html#ebsresource)
     * in the Amazon OpenSearch Service Developer Guide.
     *
     * @default 10
     * @stability stable
     */
    readonly volumeSize?: number;
    /**
     * The EBS volume type to use with the Amazon OpenSearch Service domain, such as standard, gp2, io1.
     *
     * @default gp2
     * @stability stable
     */
    readonly volumeType?: ec2.EbsDeviceVolumeType;
}
/**
 * Configures log settings for the domain.
 *
 * @stability stable
 */
export interface LoggingOptions {
    /**
     * Specify if slow search logging should be set up.
     *
     * Requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.
     *
     * @default - false
     * @stability stable
     */
    readonly slowSearchLogEnabled?: boolean;
    /**
     * Log slow searches to this log group.
     *
     * @default - a new log group is created if slow search logging is enabled
     * @stability stable
     */
    readonly slowSearchLogGroup?: logs.ILogGroup;
    /**
     * Specify if slow index logging should be set up.
     *
     * Requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.
     *
     * @default - false
     * @stability stable
     */
    readonly slowIndexLogEnabled?: boolean;
    /**
     * Log slow indices to this log group.
     *
     * @default - a new log group is created if slow index logging is enabled
     * @stability stable
     */
    readonly slowIndexLogGroup?: logs.ILogGroup;
    /**
     * Specify if Amazon OpenSearch Service application logging should be set up.
     *
     * Requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.
     *
     * @default - false
     * @stability stable
     */
    readonly appLogEnabled?: boolean;
    /**
     * Log Amazon OpenSearch Service application logs to this log group.
     *
     * @default - a new log group is created if app logging is enabled
     * @stability stable
     */
    readonly appLogGroup?: logs.ILogGroup;
    /**
     * Specify if Amazon OpenSearch Service audit logging should be set up.
     *
     * Requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later and fine grained access control to be enabled.
     *
     * @default - false
     * @stability stable
     */
    readonly auditLogEnabled?: boolean;
    /**
     * Log Amazon OpenSearch Service audit logs to this log group.
     *
     * @default - a new log group is created if audit logging is enabled
     * @stability stable
     */
    readonly auditLogGroup?: logs.ILogGroup;
}
/**
 * Whether the domain should encrypt data at rest, and if so, the AWS Key Management Service (KMS) key to use.
 *
 * Can only be used to create a new domain,
 * not update an existing one. Requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.
 *
 * @stability stable
 */
export interface EncryptionAtRestOptions {
    /**
     * Specify true to enable encryption at rest.
     *
     * @default - encryption at rest is disabled.
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * Supply if using KMS key for encryption at rest.
     *
     * @default - uses default aws/es KMS key.
     * @stability stable
     */
    readonly kmsKey?: kms.IKey;
}
/**
 * Configures Amazon OpenSearch Service to use Amazon Cognito authentication for OpenSearch Dashboards.
 *
 * @see https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html
 * @stability stable
 */
export interface CognitoOptions {
    /**
     * The Amazon Cognito identity pool ID that you want Amazon OpenSearch Service to use for OpenSearch Dashboards authentication.
     *
     * @stability stable
     */
    readonly identityPoolId: string;
    /**
     * A role that allows Amazon OpenSearch Service to configure your user pool and identity pool.
     *
     * It must have the `AmazonESCognitoAccess` policy attached to it.
     *
     * @see https://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html#cognito-auth-prereq
     * @stability stable
     */
    readonly role: iam.IRole;
    /**
     * The Amazon Cognito user pool ID that you want Amazon OpenSearch Service to use for OpenSearch Dashboards authentication.
     *
     * @stability stable
     */
    readonly userPoolId: string;
}
/**
 * The minimum TLS version required for traffic to the domain.
 *
 * @stability stable
 */
export declare enum TLSSecurityPolicy {
    /**
     * Cipher suite TLS 1.0.
     *
     * @stability stable
     */
    TLS_1_0 = "Policy-Min-TLS-1-0-2019-07",
    /**
     * Cipher suite TLS 1.2.
     *
     * @stability stable
     */
    TLS_1_2 = "Policy-Min-TLS-1-2-2019-07"
}
/**
 * Specifies options for fine-grained access control.
 *
 * @stability stable
 */
export interface AdvancedSecurityOptions {
    /**
     * ARN for the master user.
     *
     * Only specify this or masterUserName, but not both.
     *
     * @default - fine-grained access control is disabled
     * @stability stable
     */
    readonly masterUserArn?: string;
    /**
     * Username for the master user.
     *
     * Only specify this or masterUserArn, but not both.
     *
     * @default - fine-grained access control is disabled
     * @stability stable
     */
    readonly masterUserName?: string;
    /**
     * Password for the master user.
     *
     * You can use `SecretValue.plainText` to specify a password in plain text or
     * use `secretsmanager.Secret.fromSecretAttributes` to reference a secret in
     * Secrets Manager.
     *
     * @default - A Secrets Manager generated password
     * @stability stable
     */
    readonly masterUserPassword?: cdk.SecretValue;
}
/**
 * Configures a custom domain endpoint for the Amazon OpenSearch Service domain.
 *
 * @stability stable
 */
export interface CustomEndpointOptions {
    /**
     * The custom domain name to assign.
     *
     * @stability stable
     */
    readonly domainName: string;
    /**
     * The certificate to use.
     *
     * @default - create a new one
     * @stability stable
     */
    readonly certificate?: acm.ICertificate;
    /**
     * The hosted zone in Route53 to create the CNAME record in.
     *
     * @default - do not create a CNAME
     * @stability stable
     */
    readonly hostedZone?: route53.IHostedZone;
}
/**
 * Properties for an Amazon OpenSearch Service domain.
 *
 * @stability stable
 */
export interface DomainProps {
    /**
     * Domain access policies.
     *
     * @default - No access policies.
     * @stability stable
     */
    readonly accessPolicies?: iam.PolicyStatement[];
    /**
     * Additional options to specify for the Amazon OpenSearch Service domain.
     *
     * @default - no advanced options are specified
     * @see https://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options
     * @stability stable
     */
    readonly advancedOptions?: {
        [key: string]: (string);
    };
    /**
     * Configures Amazon OpenSearch Service to use Amazon Cognito authentication for OpenSearch Dashboards.
     *
     * @default - Cognito not used for authentication to OpenSearch Dashboards.
     * @stability stable
     */
    readonly cognitoDashboardsAuth?: CognitoOptions;
    /**
     * Enforces a particular physical domain name.
     *
     * @default - A name will be auto-generated.
     * @stability stable
     */
    readonly domainName?: string;
    /**
     * The configurations of Amazon Elastic Block Store (Amazon EBS) volumes that are attached to data nodes in the Amazon OpenSearch Service domain.
     *
     * @default - 10 GiB General Purpose (SSD) volumes per node.
     * @stability stable
     */
    readonly ebs?: EbsOptions;
    /**
     * The cluster capacity configuration for the Amazon OpenSearch Service domain.
     *
     * @default - 1 r5.large.search data node; no dedicated master nodes.
     * @stability stable
     */
    readonly capacity?: CapacityConfig;
    /**
     * The cluster zone awareness configuration for the Amazon OpenSearch Service domain.
     *
     * @default - no zone awareness (1 AZ)
     * @stability stable
     */
    readonly zoneAwareness?: ZoneAwarenessConfig;
    /**
     * The Elasticsearch/OpenSearch version that your domain will leverage.
     *
     * @stability stable
     */
    readonly version: EngineVersion;
    /**
     * Encryption at rest options for the cluster.
     *
     * @default - No encryption at rest
     * @stability stable
     */
    readonly encryptionAtRest?: EncryptionAtRestOptions;
    /**
     * Configuration log publishing configuration options.
     *
     * @default - No logs are published
     * @stability stable
     */
    readonly logging?: LoggingOptions;
    /**
     * Specify true to enable node to node encryption.
     *
     * Requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.
     *
     * @default - Node to node encryption is not enabled.
     * @stability stable
     */
    readonly nodeToNodeEncryption?: boolean;
    /**
     * The hour in UTC during which the service takes an automated daily snapshot of the indices in the Amazon OpenSearch Service domain.
     *
     * Only applies for Elasticsearch versions
     * below 5.3.
     *
     * @default - Hourly automated snapshots not used
     * @stability stable
     */
    readonly automatedSnapshotStartHour?: number;
    /**
     * Place the domain inside this VPC.
     *
     * @default - Domain is not placed in a VPC.
     * @see https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The list of security groups that are associated with the VPC endpoints for the domain.
     *
     * Only used if `vpc` is specified.
     *
     * @default - One new security group is created.
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The specific vpc subnets the domain will be placed in.
     *
     * You must provide one subnet for each Availability Zone
     * that your domain uses. For example, you must specify three subnet IDs for a three Availability Zone
     * domain.
     *
     * Only used if `vpc` is specified.
     *
     * @default - All private subnets.
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection[];
    /**
     * True to require that all traffic to the domain arrive over HTTPS.
     *
     * @default - false
     * @stability stable
     */
    readonly enforceHttps?: boolean;
    /**
     * The minimum TLS version required for traffic to the domain.
     *
     * @default - TLSSecurityPolicy.TLS_1_0
     * @stability stable
     */
    readonly tlsSecurityPolicy?: TLSSecurityPolicy;
    /**
     * Specifies options for fine-grained access control.
     *
     * Requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later. Enabling fine-grained access control
     * also requires encryption of data at rest and node-to-node encryption, along with
     * enforced HTTPS.
     *
     * @default - fine-grained access control is disabled
     * @stability stable
     */
    readonly fineGrainedAccessControl?: AdvancedSecurityOptions;
    /**
     * Configures the domain so that unsigned basic auth is enabled.
     *
     * If no master user is provided a default master user
     * with username `admin` and a dynamically generated password stored in KMS is created. The password can be retrieved
     * by getting `masterUserPassword` from the domain instance.
     *
     * Setting this to true will also add an access policy that allows unsigned
     * access, enable node to node encryption, encryption at rest. If conflicting
     * settings are encountered (like disabling encryption at rest) enabling this
     * setting will cause a failure.
     *
     * @default - false
     * @stability stable
     */
    readonly useUnsignedBasicAuth?: boolean;
    /**
     * To upgrade an Amazon OpenSearch Service domain to a new version, rather than replacing the entire domain resource, use the EnableVersionUpgrade update policy.
     *
     * @default - false
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html#cfn-attributes-updatepolicy-upgradeopensearchdomain
     * @stability stable
     */
    readonly enableVersionUpgrade?: boolean;
    /**
     * Policy to apply when the domain is removed from the stack.
     *
     * @default RemovalPolicy.RETAIN
     * @stability stable
     */
    readonly removalPolicy?: cdk.RemovalPolicy;
    /**
     * To configure a custom domain configure these options.
     *
     * If you specify a Route53 hosted zone it will create a CNAME record and use DNS validation for the certificate
     *
     * @default - no custom domain endpoint will be configured
     * @stability stable
     */
    readonly customEndpoint?: CustomEndpointOptions;
}
/**
 * An interface that represents an Amazon OpenSearch Service domain - either created with the CDK, or an existing one.
 *
 * @stability stable
 */
export interface IDomain extends cdk.IResource {
    /**
     * Arn of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     * @attribute true
     */
    readonly domainArn: string;
    /**
     * Domain name of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     * @attribute true
     */
    readonly domainName: string;
    /**
     * Identifier of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     * @attribute true
     */
    readonly domainId: string;
    /**
     * Endpoint of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     * @attribute true
     */
    readonly domainEndpoint: string;
    /**
     * Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantReadWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexRead(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexReadWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathRead(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathReadWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this domain.
     *
     * @stability stable
     */
    metric(metricName: string, props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusRed(props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusYellow(props?: MetricOptions): Metric;
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @stability stable
     */
    metricFreeStorageSpace(props?: MetricOptions): Metric;
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @stability stable
     */
    metricClusterIndexWritesBlocked(props?: MetricOptions): Metric;
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @stability stable
     */
    metricNodes(props?: MetricOptions): Metric;
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricAutomatedSnapshotFailure(props?: MetricOptions): Metric;
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyError(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyInaccessible(props?: MetricOptions): Metric;
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricSearchableDocuments(props?: MetricOptions): Metric;
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricSearchLatency(props?: MetricOptions): Metric;
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricIndexingLatency(props?: MetricOptions): Metric;
}
/**
 * A new or imported domain.
 */
declare abstract class DomainBase extends cdk.Resource implements IDomain {
    abstract readonly domainArn: string;
    abstract readonly domainName: string;
    abstract readonly domainId: string;
    abstract readonly domainEndpoint: string;
    /**
     * Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantReadWrite(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexRead(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexReadWrite(index: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathRead(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathReadWrite(path: string, identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this domain.
     *
     * @stability stable
     */
    metric(metricName: string, props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusRed(props?: MetricOptions): Metric;
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusYellow(props?: MetricOptions): Metric;
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @stability stable
     */
    metricFreeStorageSpace(props?: MetricOptions): Metric;
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @stability stable
     */
    metricClusterIndexWritesBlocked(props?: MetricOptions): Metric;
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @stability stable
     */
    metricNodes(props?: MetricOptions): Metric;
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricAutomatedSnapshotFailure(props?: MetricOptions): Metric;
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterCPUUtilization(props?: MetricOptions): Metric;
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterJVMMemoryPressure(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyError(props?: MetricOptions): Metric;
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyInaccessible(props?: MetricOptions): Metric;
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricSearchableDocuments(props?: MetricOptions): Metric;
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricSearchLatency(props?: MetricOptions): Metric;
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricIndexingLatency(props?: MetricOptions): Metric;
    private grant;
}
/**
 * Reference to an Amazon OpenSearch Service domain.
 *
 * @stability stable
 */
export interface DomainAttributes {
    /**
     * The ARN of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainArn: string;
    /**
     * The domain endpoint of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainEndpoint: string;
}
/**
 * Provides an Amazon OpenSearch Service domain.
 *
 * @stability stable
 */
export declare class Domain extends DomainBase implements IDomain, ec2.IConnectable {
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @stability stable
     */
    static fromDomainEndpoint(scope: Construct, id: string, domainEndpoint: string): IDomain;
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @stability stable
     */
    static fromDomainAttributes(scope: Construct, id: string, attrs: DomainAttributes): IDomain;
    /**
     * Arn of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainArn: string;
    /**
     * Domain name of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainName: string;
    /**
     * Identifier of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainId: string;
    /**
     * Endpoint of the Amazon OpenSearch Service domain.
     *
     * @stability stable
     */
    readonly domainEndpoint: string;
    /**
     * Log group that slow searches are logged to.
     *
     * @stability stable
     * @attribute true
     */
    readonly slowSearchLogGroup?: logs.ILogGroup;
    /**
     * Log group that slow indices are logged to.
     *
     * @stability stable
     * @attribute true
     */
    readonly slowIndexLogGroup?: logs.ILogGroup;
    /**
     * Log group that application logs are logged to.
     *
     * @stability stable
     * @attribute true
     */
    readonly appLogGroup?: logs.ILogGroup;
    /**
     * Log group that audit logs are logged to.
     *
     * @stability stable
     * @attribute true
     */
    readonly auditLogGroup?: logs.ILogGroup;
    /**
     * Master user password if fine grained access control is configured.
     *
     * @stability stable
     */
    readonly masterUserPassword?: cdk.SecretValue;
    private readonly domain;
    private readonly _connections;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DomainProps);
    /**
     * Manages network connections to the domain.
     *
     * This will throw an error in case the domain
     * is not placed inside a VPC.
     *
     * @stability stable
     */
    get connections(): ec2.Connections;
}
export {};
