"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy,
            },
            ruleSetName: props.ruleSet.receiptRuleSetName,
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16',
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID,
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        },
                    }),
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet,
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log('SES Notification:\n', JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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