"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: (_b = props.tableName) !== null && _b !== void 0 ? _b : cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: (_c = props.sortStyle) !== null && _c !== void 0 ? _c : this.getDefaultSortStyle(props.tableColumns),
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.17.0-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,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