import os
import sys
import time
import argparse
from Bio.Seq import Seq 
from Bio import GenBank 
from Bio import SeqIO
from Bio.SeqRecord import SeqRecord

from featurExtract.extract_CDS import get_cds_gb
from featurExtract.extract_gene import get_gene_gb
from featurExtract.extract_rRNA import get_rRNA_gb
from featurExtract.extract_tRNA import get_tRNA_gb

def create_db(genbank_path):
    if not os.path.exists(genbank_path):
        print('args.genbank path not exist; please check.')
        sys.exit(1)
    for record in SeqIO.parse(genbank_path,'gb'):
        yield record

def UTR(args):
    '''
    parameters:
     args: arguments from argparse 
    '''
    utr(args)


def uORF(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_uorf(args)


def CDS(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_cds_gb(args)


def dORF(args):
    '''
    parameters:
     args: arugmensts from argparse
    '''
    get_dorf(db)


def exon(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_exon(args)

def intron(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_intron(args)


def promoter(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_promoter(args)

def gene(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_gene_gb(args)

def rRNA(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_rRNA_gb(args)

def tRNA(args):
    '''
    parameters:
     args: arguments from argparse
    '''
    get_tRNA_gb(args)

parser = argparse.ArgumentParser()
subparsers = parser.add_subparsers(help='sub-command help')
# promoter subcommand
parser_promoter = subparsers.add_parser('promoter', help='extract promoter in genome or gene')
parser_promoter.add_argument('-d', '--database', type=str, help='database generated by subcommand create')
parser_promoter.add_argument('-g', '--gene', type=str, help='specific gene; if not given, return whole genes')
parser_promoter.add_argument('-l', '--promoter_length', type=int, help='promoter length before TSS')
parser_promoter.add_argument('-u', '--utr5_upper_length', type=int, help='utr5 length after TSS')
parser_promoter.add_argument('-f', '--genome', type=str, help='genome fasta')
parser_promoter.add_argument('-o', '--output', type=str, help = 'output csv file path')
parser_promoter.add_argument('-p', '--print', action="store_true", help='boolean type, stdin')
parser_promoter.add_argument('-v', '--version', help = 'promoterExtract version', action = "store_true")
parser_promoter.set_defaults(func=promoter)

# gene subcommand 
parser_gene = subparsers.add_parser('gene', help='extract gene sequence in genome or gene')
parser_gene.add_argument('-g', '--genbank', type=str, help='GenBank file path')
parser_gene.add_argument('-f', '--format', choices = ['dna','protein'], type=str, help='output file path')
parser_gene.add_argument('-o', '--output', type=str, help = 'output csv file path')
parser_gene.add_argument('-p', '--print', action="store_true", help='boolean type, stdin')
parser_gene.set_defaults(func=gene)


# rRNA subcommand 
parser_rRNA = subparsers.add_parser('rRNA', help='extract rRNA in genome or gene')
parser_rRNA.add_argument('-g', '--genbank', type=str, help='GenBank file path')
parser_rRNA.add_argument('-f', '--format', choices = ['dna','protein'], type=str, help='output file path')
parser_rRNA.add_argument('-o', '--output', type=str, help = 'output csv file path')
parser_rRNA.add_argument('-p', '--print', action="store_true", help='boolean type, stdin')
parser_rRNA.set_defaults(func=rRNA)


# tRNA subcommand 
parser_tRNA = subparsers.add_parser('tRNA', help='extract tRNA in genome or gene')
parser_tRNA.add_argument('-g', '--genbank', type=str, help='GenBank file path')
parser_tRNA.add_argument('-f', '--format', choices = ['dna','protein'], type=str, help='output file path')
parser_tRNA.add_argument('-o', '--output', type=str, help = 'output csv file path')
parser_tRNA.add_argument('-p', '--print', action="store_true", help='boolean type, stdin')
parser_tRNA.set_defaults(func=tRNA)

# UTR subcommand
parser_utr = subparsers.add_parser('UTR', help='extract untranslated region sequence in genome or gene')
parser_utr.add_argument('-d', '--database', type=str, help='database generated by subcommand create')
parser_utr.add_argument('-f', '--genome', type=str, help='genome fasta file')
parser_utr.add_argument('-t', '--transcript', type=str, help='specific transcript id; if not given, \
                       whole transcript will return')
parser_utr.add_argument('-o', '--output', type=str, help='output file path')
parser_utr.add_argument('-p', '--print', action="store_true", help='boolean type, stdout')
parser_utr.set_defaults(func=UTR)

# CDS subcommand
parser_cds = subparsers.add_parser('CDS', help='extract coding sequence from GenBank')
parser_cds.add_argument('-g', '--genbank', type=str, help='GenBank file path')
parser_cds.add_argument('-o', '--output', type=str, help='output file path')
parser_cds.add_argument('-f', '--format', choices = ['dna','protein'], type=str, help='output file path')
parser_cds.add_argument('-p', '--print', action="store_true", help='boolean type; stdin')
parser_cds.set_defaults(func=CDS)

args = parser.parse_args()
print('[%s runing ...]'%(time.strftime("%a %b %d %H:%M:%S %Y", time.localtime())))
args.func(args)
print('[%s finished ...]'%(time.strftime("%a %b %d %H:%M:%S %Y", time.localtime())))
