import os
import time
import unittest

import timeout_decorator

from mavis.assemble import Contig, assemble, filter_contigs
from mavis.interval import Interval
from mavis.constants import reverse_complement
from mavis.validate.constants import DEFAULTS
from mavis.util import LOG

from . import MockObject, RUN_FULL
from ..util import get_data


class TestFilterContigs(unittest.TestCase):
    @timeout_decorator.timeout(30)
    def test_large_set(self):
        contigs = []
        with open(get_data('similar_contigs.txt'), 'r') as fh:
            for line in fh.readlines():
                contigs.append(Contig(line.strip(), 1))  # give a dummy score of 1
        start_time = int(time.time())
        filtered = filter_contigs(contigs, 0.10)
        end_time = int(time.time())
        print('duration:', end_time - start_time)
        print()
        for c in filtered:
            print(c.seq)
        self.assertEqual(3, len(filtered))  # figure out amount later. need to optimize timing


class TestContigRemap(unittest.TestCase):
    def setUp(self):
        self.contig = Contig(' ' * 60, None)
        self.contig.add_mapped_sequence(MockObject(reference_start=0, reference_end=10))
        self.contig.add_mapped_sequence(MockObject(reference_start=0, reference_end=20))
        self.contig.add_mapped_sequence(MockObject(reference_start=50, reference_end=60))

    def test_depth_even_coverage(self):
        covg = self.contig.remap_depth(Interval(1, 10))
        self.assertEqual(2, covg)

    def test_depth_mixed_coverage(self):
        covg = self.contig.remap_depth(Interval(1, 20))
        self.assertEqual(1.5, covg)

    def test_depth_no_coverage(self):
        covg = self.contig.remap_depth(Interval(21, 49))
        self.assertEqual(0, covg)

    def test_depth_whole_contig_coverage(self):
        self.assertAlmostEqual(40 / 60, self.contig.remap_depth())

    def test_depth_weighted_read(self):
        self.contig.add_mapped_sequence(MockObject(reference_start=0, reference_end=10), 5)
        self.assertAlmostEqual(42 / 60, self.contig.remap_depth())

    def test_depth_bad_query_range(self):
        with self.assertRaises(ValueError):
            self.contig.remap_depth(Interval(0, 10))
        with self.assertRaises(ValueError):
            self.contig.remap_depth(Interval(1, len(self.contig.seq) + 1))

    def test_coverage(self):
        self.assertEqual(0.5, self.contig.remap_coverage())


class TestAssemble(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        # load files here so they do not count towar timeout checking
        sequences = set()
        with open(get_data('long_filter_assembly.txt'), 'r') as fh:
            sequences.update([s.strip() for s in fh.readlines() if s])
        cls.long_filter_seq = sequences
        sequences = set()
        with open(get_data('large_assembly.txt'), 'r') as fh:
            sequences.update([line.strip() for line in fh.readlines()])
        cls.large_assembly_seq = sequences

    def setUp(self):
        self.log = lambda *x, **k: print(x, k)

    def test1(self):
        sequences = [
            'TCTTTTTCTTTCTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAGTCTTACTGACAAGCATGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCGTAAATCGCCGTGCTTGGTGCTTAATGTACTT',
            'AAGTACATTAAGCACCAAGCACGGCGATTTACGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACATGCTTGTCAGTAAGACTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAGAAAGAAAAAGA',
            'CCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACT',
            'AGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGG',
            'TTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAATCTCCATGCTTGGTTGTTAAACTACTAATGCCTCACACGGGTCATCAG',
            'CTGATGACCCGTGTGAGGCATTAGTAGTTTAACAACCAAGCATGGAGATTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAA',
            'GGGGGAGGGGAGATTAGCACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGAC',
            'GTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTGCTAATCTCCCCTCCCCC',
            'GAGACTCCATCTCTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTTTTTCTTTCTTTCTTTCTTTCTTTCTATCTATCTATCTTGCATATTTTTACTTATTAAATTAGTTCTGTCCATCCAAT',
            'ATTGGATGGACAGAACTAATTTAATAAGTAAAAATATGCAAGATAGATAGATAGAAAGAAAGAAAGAAAGAAAGAAAAAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAGAGATGGAGTCTC',
            'CATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTTTTCTTTTTTCTTCCTG',
            'CAGGAAGAAAAAAGAAAAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATG',
            'CTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTG',
            'CAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAG',
            'ACTTTGCTTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTC',
            'GAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAAGCAAAGT',
            'TACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGA',
            'TCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTA',
            'TCCACAGTTCTCCACACTAACACAGGGCTAGTCTGTCTGTCCTCCTGTCTGGATTTGTTTCTTGCTTCCTCGCTGTCGTCCTGACTCTGTGCATCTATCGTGCCTTCCGTCTGTCTTACTTGGTTCCTTTGTGTTGGTATGTGAGGCTTT',
            'AAAGCCTCACATACCAACACAAAGGAACCAAGTAAGACAGACGGAAGGCACGATAGATGCACAGAGTCAGGACGACAGCGAGGAAGCAAGAAACAAATCCAGACAGGAGGACAGACAGACTAGCCCTGTGTTAGTGTGGAGAACTGTGGA',
            'AACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTTCTAGC',
            'GCTAGAAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTT',
            'CCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTTTTTCTATTTTTTCTTCTTCCTTTCTCT',
            'AGAGAAAGGAAGAAGAAAAAATAGAAAAAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGG',
            'TCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATC',
            'GATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGA',
            'CATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTTTTTTTGTTTTTTA',
            'TAAAAAACAAAAAAAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATG',
            'CCGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGCCTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGTCTCGTCCCAGCTTAATCTTATTGACCAGCAGGTTACCTTCTTTTTATTTTTGTT',
            'AACAAAAATAAAAAGAAGGTAACCTGCTGGTCAATAAGATTAAGCTGGGACGAGACAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGGCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACGG',
            'CTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCT',
            'AGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAG',
            'CATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAA',
            'TTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATG',
            'GCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAA',
            'TTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGC',
            'AACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAATCTCCATGCTTGGTTGTTAAAA',
            'TTTTAACAACCAAGCATGGAGATTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTT',
            'TCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAATCTCCATGCTTG',
            'CAAGCATGGAGATTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGA',
            'CCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAATCTCC',
            'GGAGATTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGG',
            'ACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGAAATCTCCAT',
            'ATGGAGATTTCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGT',
            'CTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACCTTCTTTTTATTTTTGTTTTTAAACCACATTGATCTGA',
            'TCAGATCAATGTGGTTTAAAAACAAAAATAAAAAGAAGGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAG',
            'TTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTN',
            'NAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAA',
            'TTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTT',
            'AAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAA',
            'GCTTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCT',
            'AGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAAGC',
            'TCCCTCTCTCGCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTC',
            'GAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGCGAGAGAGGGA',
            'TCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTCATTTCTTTCTTTCTTTTTCTTCCTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTC',
            'GAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAGGAAGAAAAAGAAAGAAAGAAATGAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGA',
            'TAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGACAAGCAGGTTACC',
            'GGTAACCTGCTTGTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTA',
            'GGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTC',
            'GAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCC',
            'TGGGGCAGGGGAGATTAGGACAACCTTCACCAGTTCGTTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTT',
            'AAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAACGAACTGGTGAAGGTTGTCCTAATCTCCCCTGCCCCA',
            'AGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTATTCTATCTTCTTCCTGACTCTTCCTAGCTTAATCTTAATGAC',
            'GTCATTAAGATTAAGCTAGGAAGAGTCAGGAAGAAGATAGAATAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACT',
            'CTTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTT',
            'AAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAAG',
            'TTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTATTCTAT',
            'ATAGAATAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAA',
            'CATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGGCTAGTCTGTCTGTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTCTTTCTTTCTTTCTTTCTTTTCTATCTTCTTCCTT',
            'AAGGAAGAAGATAGAAAAGAAAGAAAGAAAGAAAGAAAGAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGACAGACAGACTAGCCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATG',
            'GTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTCTTTCTATTCT',
            'AGAATAGAAAGAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCAC',
            'TGCTTCCCTTGTGCCCCTTTCCCTACCTGGGCCAAACTCAGAAGAGCTGGGGGAGGGGAGATTAGGACAACCTTCACCAGTTCATTTCCCTCTCTCTCTGCTTTCCACAGTTCTCCACACTAACAAAGGACTAGTCTTTTTCTTTCTTTC',
            'GAAAGAAAGAAAAAGACTAGTCCTTTGTTAGTGTGGAGAACTGTGGAAAGCAGAGAGAGAGGGAAATGAACTGGTGAAGGTTGTCCTAATCTCCCCTCCCCCAGCTCTTCTGAGTTTGGCCCAGGTAGGGAAAGGGGCACAAGGGAAGCA',
            'AGTAGCGTAATGAACTAAGTGAAGATGTGAAACAGAATTTTAACTTAATAACTTCAATCAGCAACACCCATCGACTCTTCCTAAACTCAAATCCCATGCCCTGCTAAATTATGGGACAAATCACTATACTCTGGATCTAACCAGTCAAGA',
            'TCTTGACTGGTTAGATCCAGAGTATAGTGATTTGTCCCATAATTTAGCAGGGCATGGGATTTGAGTTTAGGAAGAGTCGATGGGTGTTGCTGATTGAAGTTATTAAGTTAAAATTCTGTTTCACATCTTCACTTAGTTCATTACGCTACT',
            'CATGAAGGAAGAAATTCATCTTCTGTCGAAAAGACAGTAGGAGTATCCAAGAAAGTTCAATGAGGTTTAGATGAAAGCCCAAGTGTGACACTAGTACTCTAACAAGTATGCATGGAGAGTTCATATCAACGTGATGTAAAAACAAATATA',
            'TATATTTGTTTTTACATCACGTTGATATGAACTCTCCATGCATACTTGTTAGAGTACTAGTGTCACACTTGGGCTTTCATCTAAACCTCATTGAACTTTCTTGGATACTCCTACTGTCTTTTCGACAGAAGATGAATTTCTTCCTTCATG',
            'GTCCTCAGCCACAGTTCCCTGCTTGCCTTGGCTCTCCTCCAGCCAATTATCTCCTTCTCATTTGGTACTGCTTGCCCTGGGGTGATTGCTTGAGTGGGTGTGACCTGTGGTTGGTCTCACTGGGTCTGGTTAAAGTCCTGTTGTGTGCTC',
            'GAGCACACAACAGGACTTTAACCAGACCCAGTGAGACCAACCACAGGTCACACCCACTCAAGCAATCACCCCAGGGCAAGCAGTACCAAATGAGAAGGAGATAATTGGCTGGAGGAGAGCCAAGGCAAGCAGGGAACTGTGGCTGAGGAC',
            'GGCCCTGGAACTTGTCTGTCTGTCTGTTGATTTGGAATTGACAGTGGTTGCAGACCTTTAAGTCAAACCTTTCCTCTTGATCCCAATGTGCCCTTCGCTTTCTCTAAAAGGTTCTCCCTTCCTCTTATTTTTCCTTATCCTCTTCTCCAT',
            'ATGGAGAAGAGGATAAGGAAAAATAAGAGGAAGGGAGAACCTTTTAGAGAAAGCGAAGGGCACATTGGGATCAAGAGGAAAGGTTTGACTTAAAGGTCTGCAACCACTGTCAATTCCAAATCAACAGACAGACAGACAAGTTCCAGGGCC',
        ]
        kmer_size = 0.75 * len(list(sequences)[0])
        assembly = assemble(
            sequences,
            kmer_size,
            min_edge_trim_weight=3,
            remap_min_match=0.95,
            min_contig_length=150,
            remap_min_exact_match=6,
            assembly_max_paths=20,
            assembly_min_uniq=0.01,
            log=self.log,
        )
        for contig in assembly:
            print(contig.seq)
        self.assertTrue(assembly)

    def test_assembly_low_center(self):
        sequences = {
            'AGTTGGAGCATCTAAGCATGATTTCTTGGGAGATACGGCCATTGGTGTTTTTTCTCAACCTGTCACTAGAGAGAGATACAGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTG',
            'CAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACTGTATCTCTCTCTAGTGACAGGTTGAGAAAAAACACCAATGGCCGTATCTCCCAAGAAATCATGCTTAGATGCTCCAACT',
            'AAGCATGATTTCTTGGGAGATACGGCCATTGGTGTTTTTTCTCAACCTGTCACTAGAGAGAGATACAGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGG',
            'CCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACTGTATCTCTCTCTAGTGACAGGTTGAGAAAAAACACCAATGGCCGTATCTCCCAAGAAATCATGCTT',
            'TTGGAGCATCTAAGCATGATTTCTTGGGAGATACGGCCATTGGTGTTTTTTCTCAACCTGTCACTAGAGAGAGATACAGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAA',
            'TTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACTGTATCTCTCTCTAGTGACAGGTTGAGAAAAAACACCAATGGCCGTATCTCCCAAGAAATCATGCTTAGATGCTCCAA',
            'CATGATTTCTTGGGAGATACGGCCATTGGTGTTTTTTCTCAACCTGTCACTAGAGAGAGATACAGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCC',
            'GGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACTGTATCTCTCTCTAGTGACAGGTTGAGAAAAAACACCAATGGCCGTATCTCCCAAGAAATCATG',
            'CCCTACTACTGAGCCCTTGTCCCAGGGACAAGAAGGAACATGCTGTGTTGCTTGACAGTGAAGTGAACCACCAAGAAATACAAAGAATGTGTGATGAAGAGCCCAGAGGTAAGGCGGAATAGGTAAGGGGATGCCATCTCAGTCTGGCAG',
            'CTGCCAGACTGAGATGGCATCCCCTTACCTATTCCGCCTTACCTCTGGGCTCTTCATCACACATTCTTTGTATTTCTTGGTGGTTCACTTCACTGTCAAGCAACACAGCATGTTCCTTCTTGTCCCTGGGACAAGGGCTCAGTAGTAGGG',
            'CTAAGGCTCATGTTCTTCTTACTGAGCCCTACTACTGAGCCCTTGTCCCAGGGACAAGAAGGAACATGCTGTGTTGCTTGACAGTGAAGTGAACCACCAAGAAATACAAAGAATGTGTGTTGAAGAGCCCAGAGGTAATGCGGCATGGGG',
            'CCCCATGCCGCATTACCTCTGGGCTCTTCAACACACATTCTTTGTATTTCTTGGTGGTTCACTTCACTGTCAAGCAACACAGCATGTTCCTTCTTGTCCCTGGGACAAGGGCTCAGTAGTAGGGCTCAGTAAGAAGAACATGAGCCTTAG',
            'GGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGTCCCGCCTTTGACTGATTCAGACACAGTGAGGATCTTATGGATGAGACAGAGATGACTGGAAGAGGTTGAGTAGGGAACATGTCTTGTCC',
            'GGACAAGACATGTTCCCTACTCAACCTCTTCCAGTCATCTCTGTCTCATCCATAAGATCCTCACTGTGTCTGAATCAGTCAAAGGCGGGACCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCC',
            'GGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGGTAACATGTCTTTTCC',
            'GGAAAAGACATGTTACCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCC',
            'AGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGT',
            'ACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACT',
            'CTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGG',
            'CCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAG',
            'GCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTA',
            'TACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGC',
            'GGCCCGTCTGTGTAGGATGCAGACACAGTGAGGATCATATGGAGGAGACAGAGATGACTGGAAGAGGTTGAGGAGGGGACATGTCTGTGCCAGCTTTCCTAATGCTTCATCATCGGAAGAGCCAGGGGTATAGAAAATGGAATTAAAAGC',
            'GCTTTTAATTCCATTTTCTATACCCCTGGCTCTTCCGATGATGAAGCATTAGGAAAGCTGGCACAGACATGTCCCCTCCTCAACCTCTTCCAGTCATCTCTGTCTCCTCCATATGATCCTCACTGTGTCTGCATCCTACACAGACGGGCC',
            'CTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGGGAACATGTCT',
            'AGACATGTTCCCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAG',
            'GTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTT',
            'AACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGAC',
            'AAGACTTGTTTTTCTCATCATAACACAGTCCAGTATTTATGTTCTGAAATATCCATGTGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGGTGACTGGAAGAGGTTGAGTAGGGAACATGTCTGTTCCAG',
            'CTGGAACAGACATGTTCCCTACTCAACCTCTTCCAGTCACCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCACATGGATATTTCAGAACATAAATACTGGACTGTGTTATGATGAGAAAAACAAGTCTT',
            'TTTCCTAGGGGAAGACTTGTTTGTCCCAGCATAACACAGTCCAGTTTGTATGGTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGGGACCAT',
            'ATGGTCCCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGACCATACAAACTGGACTGTGTTATGCTGGGACAAACAAGTCTTCCCCTAGGAAA',
            'GTTTTCCTAGGGGAAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGGGAAC',
            'GTTCCCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTTCCCCTAGGAAAAC',
            'AAGACTTGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCGTTGACTGATGCAGACACAGTGAGGATCTTATGGAAGAGACAGAGATGACTGGAAGAGGTTGAGTAGGGAACATGTCTGTTCCAG',
            'CTGGAACAGACATGTTCCCTACTCAACCTCTTCCAGTCATCTCTGTCTCTTCCATAAGATCCTCACTGTGTCTGCATCAGTCAACGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACAAGTCTT',
            'CTGGACTGTGTTATGATCGGAAAAACAGGTCAACCCCTAGGAAAGCGGTGGCTTGCCGGTACCGCCCTCTACTGACATGCTGAGCAGAAACACCAATGCCCCTACCCCCCACGCAGTCATGCTTAGCTGCTCCCACTCCTTTCCGCCCTT',
            'AAGGGCGGAAAGGAGTGGGAGCAGCTAAGCATGACTGCGTGGGGGGTAGGGGCATTGGTGTTTCTGCTCAGCATGTCAGTAGAGGGCGGTACCGGCAAGCCACCGCTTTCCTAGGGGTTGACCTGTTTTTCCGATCATAACACAGTCCAG',
            'TAGGTTTCATCATGCTTAGAATTTGATTATCTAGCACCCTGTCATTCTCAATCCATTATCCTGATTTCTTCTCTATAGCACTTATCACTTCACAACATTTTATTTTATATGAATTTGTTTATCTGTTATATAGATGCCCCACTGAAATAT',
            'ATATTTCAGTGGGGCATCTATATAACAGATAAACAAATTCATATAAAATAAAATGTTGTGAAGTGATAAGTGCTATAGAGAAGAAATCAGGATAATGGATTGAGAATGACAGGGTGCTAGATAATCAAATTCTAAGCATGATGAAACCTA',
            'CGGCCATTGGTGTTTTTTCTCAACCTGTCACTAGAGAGAGATACAGTCAAGCCTCCGTTTTCCTAGGGGAAGACTTGTTTTTCTACTCTGGGTGGAGAAAAATTATTAAAAAGTCTTGATTATCAGAATTTGGCCCCTAGTTTTTCTCAT',
            'ATGAGAAAAACTAGGGGCCAAATTCTGATAATCAAGACTTTTTAATAATTTTTCTCCACCCAGAGTAGAAAAACAAGTCTTCCCCTAGGAAAACGGAGGCTTGACTGTATCTCTCTCTAGTGACAGGTTGAGAAAAAACACCAATGGCCG',
            'CAGAACCTCAAAATACTGCCTGGTACCAATAAATATTTGTTAGGTAAGTAAAGCTGATCATTGTATTAATCATTTCACTTATATTTATGGACTGATCATGGTAGTCAGGCCCTGAGAAATAAAACAGAGCTCATAACCTGGCGGTTCGAA',
            'TTCGAACCGCCAGGTTATGAGCTCTGTTTTATTTCTCAGGGCCTGACTACCATGATCAGTCCATAAATATAAGTGAAATGATTAATACAATGATCAGCTTTACTTACCTAACAAATATTTATTGGTACCAGGCAGTATTTTGAGGTTCTG',
            'TAGAGTTGGGTCTTAAAAGATGAAGGAAGGGGGCACACTGGGTCCCAGTAAGACAAGAAGAGACTATGTGCTGGACATGGCGCTCAGTGATTTACATGTATACAATGCCTCATTTAGTACTCAGAAGAACTGGAAGAAGATGTATTATTA',
            'TAATAATACATCTTCTTCCAGTTCTTCTGAGTACTAAATGAGGCATTGTATACATGTAAATCACTGAGCGCCATGTCCAGCACATAGTCTCTTCTTGTCTTACTGGGACCCAGTGTGCCCCCTTCCTTCATCTTTTAAGACCCAACTCTA',
            'TTTTCTACTCTGGGTGGAGAAAAATTATTAAAAAGTCTTGATTATCAGAATTTGGCCCCTAGTTTTTCTCATCATAACACAGTCCAGTATGTATGTTCTGAAATATCCATGGGCCCGCCTTTGACTGATGCAGACACAGTGAGGATCTTA',
            'TAAGATCCTCACTGTGTCTGCATCAGTCAAAGGCGGGCCCATGGATATTTCAGAACATACATACTGGACTGTGTTATGATGAGAAAAACTAGGGGCCAAATTCTGATAATCAAGACTTTTTAATAATTTTTCTCCACCCAGAGTAGAAAA',
        }
        for seq in sequences:
            assert reverse_complement(seq) in sequences
        kmer_size = 0.75 * len(list(sequences)[0])
        assemblies = assemble(
            sequences,
            kmer_size,
            min_edge_trim_weight=2,
            remap_min_match=0.95,
            remap_min_exact_match=6,
            assembly_max_paths=20,
            assembly_min_uniq=0.01,
            log=self.log,
        )
        for assembly in assemblies:
            print(assembly.seq)
        self.assertEqual(2, len(assemblies))

    def test_low_evidence(self):
        seqs = [
            'AGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGCCATTCATTCTGTGTTTTTTTATATAGCAGTTATTATTTTAAAG',
            'TCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGCCATTCATTCTGTGTTTTTTTATATAGCAGTTATTATTTTAAAGCAGTTATTATTCTTATATT',
            'GAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGCCATTCATTCTGTGTCTTTTTATATAGCAGTTATTATTTTAAAGCAGTTATTATTCTTATATTTCTTATTTTT',
            'GCCTTCATGAGTAGGATTATTGCCCATTTTAAAAAAAGGTCCATGAGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAA',
            'CTTCATGAGTAGGATTATTGCCCATTTTAAAAAAAGGTCCATGAGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGC',
            'AGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGCCATTCATTCTGTGTTTTTTTATAGAGCAGTTATTATTTTAAAG',
            'GCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACTGTGAGAAATACATTTCTGTTCACATAAGCCATTCATTCTGTGTTTTTTTATATAGCAGTTATTATTTTAAAGC',
            'GGTATTAGGAAGTGAGACAATTAGGAGGTAATTAGGTCATGAGAGTGGAGCCTTCATGAGTAGGATTATTGCCCATTTTAAAAAAAGGTCCATGAGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTT',
            'TAGGAAGTGAGACAATTAGGAGGTAATTAGGTCATGAGAGTGGAGCCTTCATGAGTAGGATTATTGCCCATTTTAAAAAAAGGTCCATGAGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGAT',
            'AGGAGGTAATTAGGTCATGAGAGTGGAGCCTTCATGAGTAGGATTATTGCCCATTTTAAAAAAAGGTCCATGAGCACTTTCTTGCCTTTTATCTATCATCTGAGGACACATGCTGGGCACTCTGATTTCAGATTTCCATCCTCCAGAACT',
            'AAAGTTGGTTTTTCACAAAATTGGGATTGAAATTACAGCCACCATTCGATGGTATTAGGAAGGGAGGCAATTAGGAGGTACTTGGGTCAGGAGAGGGGAGCCTGCATGAGTAGGATTAATGGCCATTTTAAAAAACGATCAATGTGCACG',
        ]
        sequences = set(seqs)
        for seq in seqs:
            sequences.add(reverse_complement(seq))
        print('assembly size', len(sequences))
        kmer_size = 0.75 * len(list(seqs)[0])
        assemblies = assemble(
            sequences,
            kmer_size,
            min_edge_trim_weight=2,
            remap_min_match=0.95,
            remap_min_overlap=150 * 0.9,
            min_contig_length=150,
            remap_min_exact_match=6,
            assembly_max_paths=20,
            assembly_min_uniq=0.01,
            log=self.log,
        )
        for assembly in assemblies:
            print(assembly.seq, assembly.remap_score())
        self.assertEqual(2, len(assemblies))

    def test_multiple_events(self):
        sequences = {
            'GGCGGCCGAGTGTAACCAGGTCTTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCG',
            'CGAGTGTAACCAGGTCTTCGATATATAAAAACCCTAGGTCGGCCGGGCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCG',
            'GTGTAACCAGGTTTTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGA',
            'AACCAGGTCTTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGA',
            'ACCAGGTCTTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAG',
            'CCAGGTCTTCGATATATAAACACCCTAGGTCGGCCGGTCGGCCGTGTTAGAGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGA',
            'TTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTTCGCGCCCGCGGGAGAGAGAGAGAGAG',
            'ATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGAT',
            'ATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGAT',
            'AGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGAC',
            'GTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGA',
            'GAGACACACACACACACATGTATACCCGTTCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATA',
            'ACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAG',
            'ACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAG',
            'CCACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAG',
            'GTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAG',
            'ACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAG',
            'CCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGA',
            'CCAGGTCTTCGATATATAAACACCCTAGGTCGGCCGGTCGGCCGTGTTAGAGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGA',
            'TTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTTCGCGCCCGCGGGAGAGAGAGAGAGAG',
            'ATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGAT',
            'ATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGAT',
            'AGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGAC',
            'GTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGA',
            'GAGACACACACACACACATGTATACCCGTTCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATA',
            'ACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAG',
            'ACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAG',
            'CCACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAG',
            'GTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAG',
            'ACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAG',
            'CCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGA',
            'TGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGG',
            'CGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAG',
            'GCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAGACC',
            'CCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAGACCA',
            'CGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAGACCACCAG',
            'GATAGATATATATATCACAGACCTGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAGACCCCCAGAGAGAGAGAGGGATG',
        }
        seqs = list(sequences)
        for seq in seqs:
            sequences.add(reverse_complement(seq))
        print('assembly size', len(sequences))
        kmer_size = 0.75 * len(list(seqs)[0])
        assemblies = assemble(
            sequences,
            kmer_size,
            min_edge_trim_weight=DEFAULTS.assembly_min_edge_trim_weight,
            remap_min_match=0.95,
            remap_min_overlap=75 * 0.9,
            min_contig_length=75,
            remap_min_exact_match=DEFAULTS.assembly_min_exact_match_to_remap,
            assembly_max_paths=DEFAULTS.assembly_max_paths,
            assembly_min_uniq=0.01,
            log=self.log,
        )
        print('assemblies', len(assemblies))
        for assembly in assemblies:
            print(assembly.seq, assembly.remap_score())
            print(reverse_complement(assembly.seq))
        expected = 'ACCAGGTCTTCGATATATAAAAACCCTAGGTCGGCCGGTCGGCCGTGTTAGTGAGACACACACACACACATGTATACCCGTGCGCGCCCGCGGGAGAGAGAGAGAGAGAGATATATATATAGCAGACCAGGAGAGCGAGAGCGAGAGAGATATAGAGAGATCGCGCGCGAGAGAGATAGGAGACC'
        self.assertEqual(expected, assemblies[0].seq)
        self.assertEqual(1, len(assemblies))

    @timeout_decorator.timeout(300)
    @unittest.skipIf(
        not RUN_FULL,
        'slower tests will not be run unless the environment variable RUN_FULL is given',
    )
    def test_large_assembly(self):
        # simply testing that this will complete before the timeout
        sequences = self.large_assembly_seq
        kmer_size = 150 * DEFAULTS.assembly_kmer_size
        print('read inputs')
        contigs = assemble(
            sequences,
            kmer_size,
            min_edge_trim_weight=DEFAULTS.assembly_min_edge_trim_weight,
            assembly_max_paths=DEFAULTS.assembly_max_paths,
            min_contig_length=150,
            log=LOG,
            remap_min_exact_match=30,
            assembly_min_uniq=DEFAULTS.assembly_min_uniq,
        )
        for contig in contigs:
            print(len(contig.seq), contig.remap_score())
            print(contig.seq)
        self.assertTrue(len(contigs))

    def test_assemble_short_contig(self):
        sequences = {
            'ACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTATGT',
            'ACATAAACTGTATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCCG',
            'CGACGGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTA',
            'TAAACTGTATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCCGTCG',
            'CGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCATGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCATGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCG',
            'ACTGGAGGACGAGGTCTCCCATAGTATGAAGGAGATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACGGCTAGATCGGAAGAG',
            'CTCTTCCGATCTAGCCGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATCTCCTTCATACTATGGGAGACCTCGTCCTCCAGT',
            'TCTAGCGACCGGCCCGTGTCACTGGAGGACGAGGTCTCCCATAGTATGAAGGAGATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTAGATCGGAAGAGCGT',
            'ACGCTCTTCCGATCTAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATCTCCTTCATACTATGGGAGACCTCGTCCTCCAGTGACACGGGCCGGTCGCTAGA',
            'ACGTCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACACACTGGACCATGGTTTTGCAGGAAATGTGAATCTCAGGAGGGAGCAGCCAGAGTGAGATGTGAGCTTTGTCCC',
            'GGGACAAAGCTCACATCTCACTCTGGCTGCTCCCTCCTGAGATTCACATTTCCTGCAAAACCATGGTCCAGTGTGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGACGT',
            'TTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGCGTGATGGTATGCAATTCCTGTTGGTTAGCCTCCTCTGAAAACAGATCGGAAGAGCGTCGT',
            'ACGACGCTCTTCCGATCTGTTTTCAGAGGAGGCTAACCAACAGGAATTGCATACCATCACGCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAA',
            'TGGCGGAGTCCGTTGCTGACCATGCGGGAGTGGCGGGGCTGCTGCAGCGCCTGTGGGGGGGCCGCGGCAGGCCACCACCAGCCTAGGTGGCTGAAGAGGTAGATCGGAAGAGCGTCGTGTAGGGG',
            'CCCCTACACGACGCTCTTCCGATCTACCTCTTCAGCCACCTAGGCTGGTGGTGGCCTGCCGCGGCCCCCCCACAGGCGCTGCAGCAGCCCCGCCACTCCCGCATGGTCAGCAACGGACTCCGCCA',
            'CGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCATGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGGAGA',
            'TCTCCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCATGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACG',
            'AGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGGATGGAGCTTTAAAAAGAACAGATAATGGGGGTTGGGCCCATGTGGAGATCGGAAGAGCGTCGTG',
            'CACGACGCTCTTCCGATCTCCACATGGGCCCAACCCCCATTATCTGTTCTTTTTAAAGCTCCATCCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCT',
            'AAGGAGATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACGGCTGCAGCGTCGCGGTGCATCAAGAGATGTGAAAGATCGGAAG',
            'CTTCCGATCTTTCACATCTCTTGATGCACCGCGACGCTGCAGCCGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATCTCCTT',
            'GGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATGTTATTGCTTAC',
            'GTAAGCAATAACATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCC',
            'AAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGGATGGAGCTTTAAAAAGAACAGATAAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTTATCTGTTCTTTTTAAAGCTCCATCCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTT',
            'CGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCATGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGATCGGAAGA',
            'TCTTCCGATCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCATGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCG',
            'GAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGGATGGAGCTTTAAAAAGAACAGATAATGGGGGTTGGGCCCATGTGGTTTGTGCCCTGTAGATCGGAAGAGCGTC',
            'GACGCTCTTCCGATCTACAGGGCACAAACCACATGGGCCCAACCCCCATTATCTGTTCTTTTTAAAGCTCCATCCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTC',
            'CAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATGTGAACTTTGTCCCCATAAGGATGGAGCTTTAAAAAGAACAGATAATGGGGGTTGGGCCCATGTGGTTTGTGCAGATCGGAAGA',
            'TCTTCCGATCTGCACAAACCACATGGGCCCAACCCCCATTATCTGTTCTTTTTAAAGCTCCATCCTTATGGGGACAAAGTTCACATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTG',
            'AGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCAGATCGGAAGAGCGTCGTGTAG',
            'CTACACGACGCTCTTCCGATCTGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCT',
            'TGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATGGTATGCAATTCCTGTTGGTTAGCCTCATCTGAAAACGTCACTCCTGGATACATAGAACATCACAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTGTGATGTTCTATGTATCCAGGAGTGACGTTTTCAGATGAGGCTAACCAACAGGAATTGCATACCATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCA',
            'TAGCGACCGGCCCGTGTCACTGGAGGACGAGGTCTCCCATAGTATGAAGGAGATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCAGATCGG',
            'CCGATCTGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATCTCCTTCATACTATGGGAGACCTCGTCCTCCAGTGACACGGGCCGGTCGCTA',
            'GATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGAGATCGGAA',
            'TTCCGATCTCAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATC',
            'CCCGTGTCACTGGAGGACGAGGTCTCCCATAGTATGAAGGAGATGATTGGAGGCTGTTGCGTTTGCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACAGATCGGA',
            'TCCGATCTGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGCAAACGCAACAGCCTCCAATCATCTCCTTCATACTATGGGAGACCTCGTCCTCCAGTGACACGGG',
            'CGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTATGTAACTTGATGGAAGAA',
            'TTCTTCCATCAAGTTACATAAACTGTATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACG',
            'CTCAGGAGAGAGCAGCCAGAGTGATGGTATGCAATTCCTGTTGGTTAGCCTCATCTGAAAACGTCACTCCTGGATACATAGAACATCACTGCGCATGTGCATCTCCCCAAGATCGGAAGAGCGTC',
            'GACGCTCTTCCGATCTTGGGGAGATGCACATGCGCAGTGATGTTCTATGTATCCAGGAGTGACGTTTTCAGATGAGGCTAACCAACAGGAATTGCATACCATCACTCTGGCTGCTCTCTCCTGAG',
            'AGTAGCCACTGTACCGTGGGTTTGCAGGAAATGTGAATCTCAGGAGAGGGCAGCCAGAGTGAGAGGTGAACTTTGTCCCGAAAAGGATGTAGCTGAAAAAAGGACGATGAAGGGGGGTTGGACCC',
            'GGGTCCAACCCCCCTTCATCGTCCTTTTTTCAGCTACATCCTTTTCGGGACAAAGTTCACCTCTCACTCTGGCTGCCCTCTCCTGAGATTCACATTTCCTGCAAACCCACGGTACAGTGGCTACT',
            'GCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCATGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGAGATCGGAAGAGCGTCGTGTAGGGAAAGAGTG',
            'CACTCTTTCCCTACACGACGCTCTTCCGATCTCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCATGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGC',
            'ATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCAGATCGGAAGAGCGGTTCAGCAGGAATGCCG',
            'CGGCATTCCTGCTGAACCGCTCTTCCGATCTGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGAT',
            'AACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAAGATCGGAAGAGCGGTTCA',
            'TGAACCGCTCTTCCGATCTTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTT',
            'CTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGAGATCGGA',
            'TCCGATCTCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAG',
            'ACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAAGATCGGAAGAGCGGTTCAGCAGGA',
            'TCCTGCTGAACCGCTCTTCCGATCTTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGT',
            'CAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCAGATCGGAAGAGCGGTTCAGCAGGA',
            'TCCTGCTGAACCGCTCTTCCGATCTGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTG',
            'GGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGATCGGAAGAGCGGTTCAG',
            'CTGAACCGCTCTTCCGATCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCC',
            'TAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTAGATCGGAAGAGCGGTTCAGCAG',
            'CTGCTGAACCGCTCTTCCGATCTAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTA',
            'CTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGAAGATCGGAAGAGCGGTTCAG',
            'CTGAACCGCTCTTCCGATCTTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAG',
            'CTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAG',
            'AATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGAGATCGGAAGAGCGGTTCAGCAG',
            'CTGCTGAACCGCTCTTCCGATCTCCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATT',
            'ACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGATTGAGCACTTGTCTTTTTGGAGTAAATCCATTTTTCTTCCATAAAGTTACATAAACTGTATCCGTGATCCTGATACATGTCCAAA',
            'TTTGGACATGTATCAGGATCACGGATACAGTTTATGTAACTTTATGGAAGAAAAATGGATTTACTCCAAAAAGACAAGTGCTCAATCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGT',
            'AGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGCATCTGTGCTCCTGA',
            'TCAGGAGCACAGATGCAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCT',
            'GTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTAGATCGGAAGA',
            'TCTTCCGATCTAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGAC',
            'CTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGGGAATTTCATTTTTTTCCATCCAGTTTCATAAACTTTATTTTTTCTTCTTATTCCAGGCCAAAGGGGTTTTGGAGGTTTCTTG',
            'CAAGAAACCTCCAAAACCCCTTTGGCCTGGAATAAGAAGAAAAAATAAAGTTTATGAAACTGGATGGAAAAAAATGAAATTCCCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAG',
            'CTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGATCGGA',
            'TCCGATCTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAG',
            'AGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGAGATCGGAAGAGCGGTTCAGCAGGAATGCCGAG',
            'CTCGGCATTCCTGCTGAACCGCTCTTCCGATCTCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCT',
            'GATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCATTGTTGCCGCATTCCAAGCGGTTAGATCGGAAGAGCGGTTCAGCAGGAAAGC',
            'GCTTTCCTGCTGAACCGCTCTTCCGATCTAACCGCTTGGAATGCGGCAACAATGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATC',
            'GGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTT',
            'AAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCC',
            'GCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTAGATCG',
            'CGATCTAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGC',
            'CTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAAGATCGGAAGA',
            'TCTTCCGATCTTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAG',
            'GCAATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAAGATCGGAAGAGCGGTTC',
            'GAACCGCTCTTCCGATCTTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATTGC',
            'GCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCAG',
            'CTGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGC',
            'TATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAAGATCGGAAGAGCGGTTCAGCAGG',
            'CCTGCTGAACCGCTCTTCCGATCTTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATA',
            'CTATAAACTAGTCCCAATTTCCTTCATGGGCTTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGAGATCGGAAGAGCGGTTCAGCAGGAATGC',
            'GCATTCCTGCTGAACCGCTCTTCCGATCTCTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAAGCCCATGAAGGAAATTGGGACTAGTTTATAG',
            'ATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGAAGATCGGAA',
            'TTCCGATCTTCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAAT',
            'CCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTAGATCGGAAGAGCGGTTCAGC',
            'GCTGAACCGCTCTTCCGATCTAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGG',
            'ATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGAGATCGGAAG',
            'CTTCCGATCTCAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCAT',
            'TTCATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAAGATCGGAAGAGCGGTTCAGCAGGAATGCCGA',
            'TCGGCATTCCTGCTGAACCGCTCTTCCGATCTTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATGAA',
            'TAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTCATTACAGGAACACTTCCCATTTGTGGAAGATCGGAAGAG',
            'CTCTTCCGATCTTCCACAAATGGGAAGTGTTCCTGTAATGACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTA',
            'GTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCAGATCGGAAGAGCGGTTCAGCAGGAATGCC',
            'GGCATTCCTGCTGAACCGCTCTTCCGATCTGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAAC',
            'TTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGAGATCGGAAGAGCGGTTCAGCAGGAATGCC',
            'GGCATTCCTGCTGAACCGCTCTTCCGATCTCCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAA',
            'TTTTTGGTGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGAGATCGGAAGAGC',
            'GCTCTTCCGATCTCCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACACCAAAAA',
            'TTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTAGATCGGAAGA',
            'TCTTCCGATCTAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAA',
            'TTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCATTGTTGCCGCATTCCAAGCGGTTGTTGGTGCAACCTTTGGTTGCCAGTTAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTAACTGGCAACCAAAGGTTGCACCAACAACCGCTTGGAATGCGGCAACAATGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAA',
            'ATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTAGATCGGAAGAGCGGTTCAGCAG',
            'CTGCTGAACCGCTCTTCCGATCTAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCAT',
            'ATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGCCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTCATTACAGGAACACTTCCCATTTAGATCGGAAGAGCGGT',
            'ACCGCTCTTCCGATCTAAATGGGAAGTGTTCCTGTAATGACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGGCCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTAT',
            'GCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTAGATCGGAAGAGCGGTTCAGCAGGAATGCC',
            'GGCATTCCTGCTGAACCGCTCTTCCGATCTAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGC',
            'TGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTCATTACAGGAACACTTCCCATTTGTGGAGGAATTCCATAGATCGG',
            'CCGATCTATGGAATTCCTCCACAAATGGGAAGTGTTCCTGTAATGACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACA',
            'ATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTCATTAAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTTAATGACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAAT',
            'CTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAAGATCGGAAGAGCGGTTCAGCAGGAATGCC',
            'GGCATTCCTGCTGAACCGCTCTTCCGATCTTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAG',
            'TTAGTCTTAAATCATAGCAATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGAGATCGGAAG',
            'CTTCCGATCTCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATTGCTATGATTTAAGACTAA',
            'ATCATAGCAATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAAGATCGGAAGAGCGGTT',
            'AACCGCTCTTCCGATCTTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATTGCTATGAT',
            'TTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGAGATCGGAAGAGC',
            'GCTCTTCCGATCTCCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAA',
            'TTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTAGATCG',
            'CGATCTACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAA',
            'ATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTAGATCGGAA',
            'TTCCGATCTAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTAT',
            'TACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGAGATCG',
            'CGATCTCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTA',
            'CTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAAGATCGGAAGAGCGGTTC',
            'GAACCGCTCTTCCGATCTTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAG',
            'TAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGAGATCGG',
            'CCGATCTCAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTA',
            'GACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGAGATCGGAAGAGCGGTTCAGCAGGAAT',
            'ATTCCTGCTGAACCGCTCTTCCGATCTCAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTC',
            'CTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGAGATCGG',
            'CCGATCTCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAG',
            'TCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAAGATCG',
            'CGATCTTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGA',
            'AATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAAGATCGGAAGAG',
            'CTCTTCCGATCTTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATT',
            'CTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGAGATCGGAAGAGCGGT',
            'ACCGCTCTTCCGATCTCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAG',
            'CATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCCTTTTTCTTCCATCAAGT',
            'ACTTGATGGAAGAAAAAGGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATG',
            'TAGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTAGATCGGAAGA',
            'TCTTCCGATCTAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCTA',
            'TTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCACTCTGGCTGCTCTCTCCTGAGATTCACATTTCCTGCAAAACCACGGTCCAGTGGGTACTTGAA',
            'TTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAA',
            'AAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATAGATCGG',
            'CCGATCTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTT',
            'TATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAAGATCG',
            'CGATCTTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATA',
            'AGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGAGATCGGAAGAGCGGTTCAGCAGGAAT',
            'ATTCCTGCTGAACCGCTCTTCCGATCTCAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACT',
            'TAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTAGATCG',
            'CGATCTACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTA',
            'TAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCAAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTTGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTA',
            'CACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAGATCGG',
            'CCGATCTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTG',
            'TTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATACTAAGAATGATGTAAATTGGAG',
            'CTCCAATTTACATCATTCTTAGTATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAA',
            'TTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTAGATCG',
            'CGATCTACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAA',
            'CTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTAGATCGGAAGA',
            'TCTTCCGATCTAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAG',
            'GATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGAGATCGGAAG',
            'CTTCCGATCTCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATC',
            'CTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGAGATCGGAAGAGC',
            'GCTCTTCCGATCTCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAG',
            'GATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCATTGTTGCCGCATTCCAAGCGGTTAGATCGGAAGAGCGGTTCAGCAGGAATGC',
            'GCATTCCTGCTGAACCGCTCTTCCGATCTAACCGCTTGGAATGCGGCAACAATGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATC',
            'AATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAAGATCGGAAGAGCGGTTC',
            'GAACCGCTCTTCCGATCTTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATT',
            'ATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGAGATCG',
            'CGATCTCTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTAT',
            'CAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAAGATCG',
            'CGATCTTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTG',
            'TAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGAGATCGGAAGAGCGGTTC',
            'GAACCGCTCTTCCGATCTCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTA',
            'GTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTCTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAAC',
            'TGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTAGATCGGAAGAGCGGTTC',
            'GAACCGCTCTTCCGATCTAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCA',
            'AAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCAGATCGGAAGAGCG',
            'CGCTCTTCCGATCTGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTT',
            'AAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTAGATCGGAAGAGCGGTTCAGCAGGA',
            'TCCTGCTGAACCGCTCTTCCGATCTAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTT',
            'GAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGAGATCGGAAGAGCGGTTCAGCAGGAATGCCGAGAC',
            'GTCTCGGCATTCCTGCTGAACCGCTCTTCCGATCTCTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTC',
            'TAGCAATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGAGATCGGAAGAGC',
            'GCTCTTCCGATCTCTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATTGCTA',
            'CACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGAGATCGGAAGAGCGGTTCA',
            'TGAACCGCTCTTCCGATCTCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTG',
            'TATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATAGATCGGAAGAGCGGTTCAGCAG',
            'CTGCTGAACCGCTCTTCCGATCTATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATA',
            'TGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGAGATCGGAAG',
            'CTTCCGATCTCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCA',
            'ATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTAGATCGG',
            'CCGATCTAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAAT',
            'ATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGAAGATCGGAAGAGCGGTTCAGCAGGA',
            'TCCTGCTGAACCGCTCTTCCGATCTTCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAAT',
            'TTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCTGTGCTAGATCGGAAGAGCGGTTCAGCAGGAATGCCG',
            'CGGCATTCCTGCTGAACCGCTCTTCCGATCTAGCACAGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAA',
            'TCTTCCATCAAGTTACATAAACTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGAGATCGGAAGAGCGGTTCAGCAGG',
            'CCTGCTGAACCGCTCTTCCGATCTCGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAGTTTATGTAACTTGATGGAAGA',
            'TGAGTTACTTGTAGCATTCTAATGGAAGAAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGAGATCGGAAGA',
            'TCTTCCGATCTCTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTTCTTCCATTAGAATGCTACAAGTAACTCA',
            'CTGTATCTGTGCTCCTGATACAGGGCCAAAGGGGTTTGGAGGTCTCATGACAGGCTGGCTGTATATTAAGGTTGGTTGCGTCATTACAGGAACACTTCCCATTTGTGGAGGCTGACTCTCTCTTC',
            'GAAGAGAGAGTCAGCCTCCACAAATGGGAAGTGTTCCTGTAATGACGCAACCAACCTTAATATACAGCCAGCCTGTCATGAGACCTCCAAACCCCTTTGGCCCTGTATCAGGAGCACAGATACAG',
            'GCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTATCATTGTTGCCGCATTCCAAGCGGTTGTTGGTGCAACCTTTGGTTGCCAGTTAAGATCGGAAGAGCGGTT',
            'AACCGCTCTTCCGATCTTAACTGGCAACCAAAGGTTGCACCAACAACCGCTTGGAATGCGGCAACAATGATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGC',
            'CTTAAATCATAGCAATTCTTGGCTTTATAAACTGTATTGATACGTTCTCCTATAAACTAGTCCCAATTTCCTTCATGGGCCTTCACTGAGTTACTTGTAGCATTCTAATGGAAGAAGATCGGAAG',
            'CTTCCGATCTTCTTCCATTAGAATGCTACAAGTAACTCAGTGAAGGCCCATGAAGGAAATTGGGACTAGTTTATAGGAGAACGTATCAATACAGTTTATAAAGCCAAGAATTGCTATGATTTAAG',
            'GCTCAGACGAGAGAGGCTGGGCCGAGAACCCGCTGGTTTATTGCGACGGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGG',
            'CCTGCAAAACCACGGTCCAGTGGGTACTTGAACAATGCCATAGCAAGCTTGATGCACCGCGACGCTGCAGCCGTGCCCGTCGCAATAAACCAGCGGGTTCTCGGCCCAGCCTCTCTCGTCTGAGC',
            'AAGAGAGTTTAAGAGATTTAAGAGACAGCAGTTTTGGATTTTGCTGGAAGTAAGGATTTTGCTGCTTGAGCACTTGTCTTTTTGGAGTAATTCCATTTTCTTCCATCAAGTTACATAAACTGTAT',
            'ATACAGTTTATGTAACTTGATGGAAGAAAATGGAATTACTCCAAAAAGACAAGTGCTCAAGCAGCAAAATCCTTACTTCCAGCAAAATCCAAAACTGCTGTCTCTTAAATCTCTTAAACTCTCTT',
        }
        contigs = assemble(
            sequences,
            kmer_size=125 * 0.74,
            min_edge_trim_weight=3,
            assembly_max_paths=8,
            assembly_min_uniq=0.1,
            min_contig_length=125,
            remap_min_exact_match=15,
            log=LOG,
        )

        target = 'GGGCACGGCTGCAGCGTCGCGGTGCATCAAGCTTGCTATGGCATTGTTCAAGTACCCACTGGACCGTGGTTTTGCAGGAAATGTGAATCTCAGGAGAGAGCAGCCAGAGTGATACAGTTTATGTAACTTGATGGAAGAA'
        print('target', target)
        for contig in contigs:
            print(len(contig.seq), contig.remap_score(), contig.seq)
        self.assertTrue({target, reverse_complement(target)} & {c.seq for c in contigs})

    @timeout_decorator.timeout(60)
    @unittest.skipIf(
        not RUN_FULL,
        'slower tests will not be run unless the environment variable RUN_FULL is given',
    )
    def test_long_filter_bug(self):
        sequences = self.long_filter_seq
        contigs = assemble(sequences, 111, 3, 8, 0.1, 0.1, log=LOG)
        for c in contigs:
            print(c.seq, c.remap_score())
        self.assertTrue(len(contigs))
