# PyMedit
`PyMedit` is a python package which allows to process triangle and tetrahedral meshes
in the INRIA `.mesh` file format (see e.g. the reference documentations of Cecile
Dobrzynski [here](https://hal.inria.fr/hal-00681813/file/RT-422.pdf)).

<img src="https://gitlab.com/florian.feppon/pymedit/-/raw/master/screenshot_pymedit2.png" align="center" alt="Output" width="50%">

*A finite element mesh generated by pymedit.examples.generate3DMesh*
## Features
- Provides `Mesh` and `Mesh3D` classes for reading and writing meshes in ASCII or 
binary `.mesh` and `.meshb` format
- Provides classes for reading and writing solution files in ASCII or binary `.sol`
or `.solb` formats: `P1Function`, `P1Function3D`, `P0Function`, `P0Function3D`,
`P1Vector`, `P1Vector3D`
- Connectivity accessible from user friendly attributes: `.triangles`, `.tetrahedra`,
`.vertices`, and so on
- Elementary operations implemented efficiently with vectorized operations (`trunc`,
`trunc3DMesh`, `connectedComponents`, `connectedComponents3D` etc...)
- Tunable verbosity via the `debug` attribute
- Interface with the [Mmg](https://www.mmgtools.org/) and `advect` and `mshdist` of 
the [ISCD toolbox](https://github.com/ISCDtoolbox) for convenient remeshing,
advection or signed distance function computation in python
- 2D plots with [matplotlib](https://matplotlib.org/)
and 3D plots with [medit](https://github.com/ISCDtoolbox/Medit)
- Generation of 3D mesh with tagged triangles 
    with `generate3DMesh` and export to VTK with `saveToVtk`
  (requires [pyfreefem](https://gitlab.com/florian.feppon/pyfreefem/) and 
  a recent installation of [FreeFEM](https://freefem.org/))

## Installation

### With pip

```bash
pip install pymedit
```

### Manual installation

Clone this repository and use the command `pip install -e DIR` where `DIR` is the 
path to `pymedit` to install it into 
your python environment.


## Running examples
Examples are available in the folder `examples` and can be run from the command
line with
```bash
python -m pymedit.examples.ex0
python -m pymedit.examples.mesh
python -m pymedit.examples.P1
python -m pymedit.examples.P0
python -m pymedit.examples.iscd2d
python -m pymedit.examples.mesh3D
python -m pymedit.examples.P1_3D
python -m pymedit.examples.iscd3d
python -m pymedit.examples.generate3DMesh
``` 
Use `--debug n` with `n` an integer for more verbose outputs. 

You can test everything
at once with
```bash
python -m pymedit.examples.test_all
``` 

Check in particular the files `pymedit/examples/iscd2d.py` and
`pymedit/examples/iscd3d.py` for examples of use of `mmg2d`, `mmg3d`, `advect` and
`mshdist`.

## Requirements
Runs with python 3.6 and the following libraries:
* numpy (>=1.12.1)
* scipy (>=0.19.1)
* cvxopt (>=1.2.1)
* colored (>=1.3.93)   *(for colored output)*

Extra dependencies:

- For using `savetovtk`, a recent version of `FreeFEM` (>=4.0) should be available from the
  command line.
- For using `mmg2d`, `mmg3d`, `advect`, `mshdist`, the related software should be
  available from the command line, see [Mmg](https://www.mmgtools.org/) and  
  the [ISCD toolbox](https://github.com/ISCDtoolbox). 
- `xdotool` is required for passing key press inputs to medit vizualization software
- Plots in 3D are available if  
   [medit](https://github.com/ISCDtoolbox/Medit) is available from the command
   line


## Quick example
File `pymedit/examples/ex0.py`
```python
import numpy as np
from pymedit import cube, mmg3d, P1Function3D, trunc3DMesh

M = cube(30,30,30)
M.plot("Initial mesh", keys="bceZZZZ") #keys are for setting the medit window
M.debug = 4 # For debugging and mmg3d output

# Setting a P1 level set function
phi = lambda x: np.cos(10*x[0])*np.cos(10*x[1])*np.cos(10*x[2])-0.3
phiP1 = P1Function3D(M,phi)

# Remesh according to the level set
newM = mmg3d(M, 0.02, 0.05, 1.3, sol=phiP1, ls=True) 

# Trunc the negative subdomain of the level set
Mf = trunc3DMesh(newM, 3)

Mf.plot(title="Plotting Truncated mesh",keys="bceZZZZ")

Mf.save("Thf.meshb") #Saving in binary format 
```

Running the script should produce the following image

<img src="screenshot_pymedit.png" align="center" alt="Output" width="50%">

and the following output:
```
$ python -m pymedit.examples.ex0
Initial mesh
Saved /tmp/tmpreyvbp8j/Th.solb in 0.00s.
Saved /tmp/tmpreyvbp8j/Th.solb
mmg3d_O3 -nr -hmin 0.02 -hmax 0.05 -hgrad 1.3 -hausd 0.002 -ls -sol /tmp/tmpreyvbp8j/Th.solb /tmp/tmpreyvbp8j/Th.meshb                                                                                                                  
  -- MMG3D, Release 5.4.3 (Feb. 26, 2020) 
     Copyright (c) Bdx INP/CNRS/Inria/UPMC, 2004-
     Apr  6 2020 18:56:14
     git branch: develop
     git commit: b1b0331db11598ff548b382ce036465f49c883c1
     git date:   2020-04-01 20:50:37 +0200


  -- INPUT DATA
  %% /tmp/tmpreyvbp8j/Th.meshb OPENED
  %% /tmp/tmpreyvbp8j/Th.solb OPENED
  -- DATA READING COMPLETED.     0.139s

  &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&
   MODULE MMG3D: 5.4.3 (Feb. 26, 2020)
  &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&
     git branch: develop
     git commit: b1b0331db11598ff548b382ce036465f49c883c1
     git date:   2020-04-01 20:50:37 +0200


  -- MMG3DLS: INPUT DATA
  --  INPUT DATA COMPLETED.     0.000s

  -- PHASE 1 : ISOSURFACE DISCRETIZATION

  -- MESH QUALITY  162000
     BEST   0.657267  AVRG.   0.657267  WRST.   0.657267 (3394)
  -- PHASE 1 COMPLETED.     0.281s

  -- PHASE 2 : ANALYSIS
  -- PHASE 2 COMPLETED.     4.095s

  -- PHASE 3 : MESH IMPROVEMENT
  *** Manifold implicit surface.
                           14606 splitted,    16006 collapsed,    43043 swapped, 6 iter.

  -- GRADATION : 1.300000 (2.300000)
                             387 splitted,     1965 collapsed,     2536 swapped, 6 iter.
        34173 splitted,    22422 collapsed,    85225 swapped,   348982 moved, 7 iter. 
  *** Manifold implicit surface.
  -- PHASE 3 COMPLETED.     22.502s

  -- MESH QUALITY  302987
     BEST   0.999087  AVRG.   0.789762  WRST.   0.243372 (108431)

  -- MESH PACKED UP
     NUMBER OF VERTICES      54729   CORNERS       44
     NUMBER OF TETRAHEDRA   302987
     NUMBER OF EDGES          1282   RIDGES      1282
     NUMBER OF TRIANGLES     27447

   MMG3DLS: ELAPSED TIME  27.035s

  &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&
   END OF MODULE MMG3D
  &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&


  -- WRITING DATA FILE /tmp/tmpreyvbp8j/Th.o.meshb
  %% /tmp/tmpreyvbp8j/Th.o.meshb OPENED
  %% /tmp/tmpreyvbp8j/Th.o.sol OPENED
  -- WRITING COMPLETED

   ELAPSED TIME  27.256s

Finished in (27.26s)
Loading /tmp/tmpreyvbp8j/Th.o.meshb
Creating empty mesh
Truncating mesh from region 3
Added 1155 edges
Truncated mesh in 0.42s.
Plotting Truncated mesh
medit /tmp/tmpc35iy48n/Th.meshb
Wrote Thf.meshb in 0.18s.
Saved Thf.meshb
```


