# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['streamlit_webrtc']

package_data = \
{'': ['*'],
 'streamlit_webrtc': ['frontend/build/*', 'frontend/build/static/js/*']}

install_requires = \
['aiortc>=1.1.2,<2.0.0', 'streamlit>=0.84.1']

extras_require = \
{':python_version < "3.8"': ['typing_extensions>=3.7.4,<5.0.0']}

setup_kwargs = {
    'name': 'streamlit-webrtc',
    'version': '0.36.1',
    'description': '',
    'long_description': '# streamlit-webrtc\n**Handling and transmitting real-time video/audio streams over the network with Streamlit** [![Open in Streamlit](https://static.streamlit.io/badges/streamlit_badge_black_white.svg)](https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py)\n\n[![Tests](https://github.com/whitphx/streamlit-webrtc/workflows/Tests/badge.svg?branch=main)](https://github.com/whitphx/streamlit-webrtc/actions?query=workflow%3ATests+branch%3Amain)\n[![Frontend Tests](https://github.com/whitphx/streamlit-webrtc/workflows/Frontend%20tests/badge.svg?branch=main)](https://github.com/whitphx/streamlit-webrtc/actions?query=workflow%3A%22Frontend+tests%22+branch%3Amain)\n\n[![PyPI](https://img.shields.io/pypi/v/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - License](https://img.shields.io/pypi/l/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/streamlit-webrtc)](https://pypi.org/project/streamlit-webrtc/)\n\n[![GitHub Sponsors](https://img.shields.io/github/sponsors/whitphx?label=Sponsor%20me%20on%20GitHub%20Sponsors&style=social)](https://github.com/sponsors/whitphx)\n\n<a href="https://www.buymeacoffee.com/whitphx" target="_blank"><img src="https://cdn.buymeacoffee.com/buttons/v2/default-yellow.png" alt="Buy Me A Coffee" width="180" height="50" ></a>\n\n<table>\n<tr>\n<td width="48%">\n  <a href="https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py">\n    <img src="https://aws1.discourse-cdn.com/business7/uploads/streamlit/original/2X/a/af111a7393c77cb69d7712ac8e71ca862feaeb24.gif" />\n  </a>\n</id>\n<td width="48%">\n  <a href="https://share.streamlit.io/whitphx/style-transfer-web-app/main/app.py">\n    <img src="https://aws1.discourse-cdn.com/business7/uploads/streamlit/original/2X/b/b3cb8aa60eb746366e06726a9137720583c02c3a.gif" />\n  </a>\n</id>\n</tr>\n</table>\n\n## Examples\n### [⚡️Showcase including following examples and more](https://github.com/whitphx/streamlit-webrtc-example): [🎈Online demo](https://share.streamlit.io/whitphx/streamlit-webrtc-example/main/app.py)\n\n* Object detection\n* OpenCV filter\n* Uni-directional video streaming\n* Audio processing\n\nYou can try out this sample app using the following commands on your local env.\n```\n$ pip install streamlit-webrtc opencv-python-headless matplotlib pydub\n$ streamlit run https://raw.githubusercontent.com/whitphx/streamlit-webrtc-example/main/app.py\n```\n\n### [⚡️Real-time Speech-to-Text](https://github.com/whitphx/streamlit-stt-app): [🎈Online demo](https://share.streamlit.io/whitphx/streamlit-stt-app/main/app_deepspeech.py)\n\nIt converts your voice into text in real time.\nThis app is self-contained; it does not depend on any external API.\n\n### [⚡️Real-time video style transfer](https://github.com/whitphx/style-transfer-web-app): [🎈Online demo](https://share.streamlit.io/whitphx/style-transfer-web-app/main/app.py)\nIt applies a wide variety of style transfer filters to real-time video streams.\n\n### [⚡️Video chat](https://github.com/whitphx/streamlit-video-chat-example)\n(Online demo not available)\n\nYou can create video chat apps with ~100 lines of Python code.\n\n### [⚡️Tokyo 2020 Pictogram](https://github.com/whitphx/Tokyo2020-Pictogram-using-MediaPipe): [🎈Online demo](https://share.streamlit.io/whitphx/tokyo2020-pictogram-using-mediapipe/streamlit-app)\n[MediaPipe](https://google.github.io/mediapipe/) is used for pose estimation.\n\n## Install\n```shell\n$ pip install -U streamlit-webrtc\n```\n\n## Quick tutorial\nCreate `app.py` with the content below.\n```py\nfrom streamlit_webrtc import webrtc_streamer\n\nwebrtc_streamer(key="sample")\n```\nUnlike other Streamlit components, `webrtc_streamer()` requires the `key` argument as a unique identifier. Set an arbitrary string to it.\n\nThen run it with Streamlit and open http://localhost:8501/.\n```shell\n$ streamlit run app.py\n```\n\nYou see the app view, so click the "START" button.\n\nThen, video and audio streaming starts. If asked for permissions to access the camera and microphone, allow it.\n![Basic example of streamlit-webrtc](./docs/images/streamlit_webrtc_basic.gif)\n\nNext, edit `app.py` as below and run it again.\n```py\nfrom streamlit_webrtc import webrtc_streamer\nimport av\n\n\nclass VideoProcessor:\n    def recv(self, frame):\n        img = frame.to_ndarray(format="bgr24")\n\n        flipped = img[::-1,:,:]\n\n        return av.VideoFrame.from_ndarray(flipped, format="bgr24")\n\n\nwebrtc_streamer(key="example", video_processor_factory=VideoProcessor)\n```\n\nNow the video is vertically flipped.\n![Vertically flipping example](./docs/images/streamlit_webrtc_flipped.gif)\n\nAs an example above, you can edit the video frames by defining a class with a callback method `recv(self, frame)` and passing it to the `video_processor_factory` argument.\nThe callback receives and returns a frame. The frame is an instance of [`av.VideoFrame`](https://pyav.org/docs/develop/api/video.html#av.video.frame.VideoFrame) (or [`av.AudioFrame`](https://pyav.org/docs/develop/api/audio.html#av.audio.frame.AudioFrame) when dealing with audio) of [`PyAV` library](https://pyav.org/).\n\nYou can inject any kinds of image (or audio) processing inside the callback.\nSee examples above for more applications.\n\nNote that there are some limitations in this callback. See the section below.\n\n## Limitations\nThe callback methods (`VideoProcessor.recv()` and similar ones) are executed in threads different from the main thread, so there are some limitations:\n* Streamlit methods (`st.*` such as `st.write()`) do not work inside the callbacks.\n* Variables outside the callbacks cannot be referred to from inside, and vice versa.\n  * It\'s impossible even with the `global` keyword, which also does not work in the callbacks properly.\n* You have to care about thread-safety when accessing the same objects both from outside and inside the callbacks.\n\n### A technique to pass values between inside and outside the callbacks\nAs stated above, you cannot directly pass variables from/to outside and inside the callback and have to consider about thread-safety.\n\nUsual cases are\n* to change some parameters used in the callback to new values passed from the main scope.\n* to refer to the results of some processing inside the callback from the main scope.\n\nThe solution is to use the properties of the processor object which is accessible via the context object returned from `webrtc_streamer()` as below.\n```python\nclass VideoProcessor:\n    def __init__(self):\n        self.some_value = 0.5\n\n    def recv(self, frame):\n        img = frame.to_ndarray(format="bgr24")\n\n        ...\n        self.do_something(img, self.some_value)  # `some_value` is used here\n        ...\n\n        return av.VideoFrame.from_ndarray(img, format="bgr24")\n\n\nctx = webrtc_streamer(key="example", video_processor_factory=VideoProcessor)\n\nif ctx.video_processor:\n    ctx.video_processor.some_value = st.slider(...)  # `some_value` is set here\n```\n\nIf the passed value is a complex object, you may also have to consider about using something like [`threading.Lock`](https://docs.python.org/3/library/threading.html#threading.Lock) or [`queue.Queue`](https://docs.python.org/3/library/queue.html#queue.Queue) for thread-safety.\n\n[The sample app, `app.py`](https://github.com/whitphx/streamlit-webrtc/blob/main/app.py) has many cases where this technique is used and can be a hint for this topic.\n\n## Serving from remote host\nWhen deploying apps to remote servers, there are some things you need to be aware of.\n\n### HTTPS\n`streamlit-webrtc` uses [`getUserMedia()`](https://developer.mozilla.org/en-US/docs/Web/API/MediaDevices/getUserMedia) API to access local media devices, and this method does not work in an insecure context.\n\n[This document](https://developer.mozilla.org/en-US/docs/Web/API/MediaDevices/getUserMedia#privacy_and_security) says\n> A secure context is, in short, a page loaded using HTTPS or the file:/// URL scheme, or a page loaded from localhost.\n\nSo, when hosting your app on a remote server, it must be served via HTTPS if your app is using webcam or microphone.\nIf not, you will encounter an error when starting using the device. For example, it\'s something like below on Chrome.\n> Error: navigator.mediaDevices is undefined. It seems the current document is not loaded securely.\n\n[Streamlit Cloud](https://streamlit.io/cloud) is a recommended way for HTTPS serving. You can easily deploy Streamlit apps with it, and most importantly for this topic, it serves the apps via HTTPS automatically by defualt.\n\n### Network connectivity\nVideo streaming does not work in some network environments.\nFor example, in some office or public networks, there are firewalls which drop the WebRTC packets.\n\nIn such environments, setting up a [TURN server](https://webrtc.org/getting-started/turn-server) is a solution. See https://github.com/whitphx/streamlit-webrtc/issues/335#issuecomment-897326755.\n\n## Logging\nFor logging, this library uses the standard `logging` module and follows the practice described in [the official logging tutorial](https://docs.python.org/3/howto/logging.html#advanced-logging-tutorial). Then the logger names are the same as the module names - `streamlit_webrtc` or `streamlit_webrtc.*`.\n\nSo you can get the logger instance with `logging.getLogger("streamlit_webrtc")` through which you can control the logs from this library.\n\nFor example, if you want to set the log level on this library\'s logger as WARNING, you can use the following code.\n```python\nst_webrtc_logger = logging.getLogger("streamlit_webrtc")\nst_webrtc_logger.setLevel(logging.WARNING)\n```\n\nIn practice, `aiortc`, a third-party package this library is internally using, also emits many INFO level logs and you may want to control its logs too.\nYou can do it in the same way as below.\n```python\naioice_logger = logging.getLogger("aioice")\naioice_logger.setLevel(logging.WARNING)\n```\n\n## API\nCurrently there is no documentation about the interface. See the example [app.py](./app.py) for the usage.\nThe API is not finalized yet and can be changed without backward compatiblity in the future releases until v1.0.\n\n### For users since versions `<0.20`\n`VideoTransformerBase` and its `transform` method have been marked as deprecated in v0.20.0. Please use `VideoProcessorBase#recv()` instead.\nNote that the signature of the `recv` method is different from the `transform` in that the `recv` has to return an instance of `av.VideoFrame` or `av.AudioFrame`.\n\nAlso, `webrtc_streamer()`\'s `video_transformer_factory` and `async_transform` arguments are deprecated, so use `video_processor_factory` and `async_processing` respectively.\n\nSee the samples in [app.py](./app.py) for their usage.\n\n## Resources\n* [Developing web-based real-time video/audio processing apps quickly with Streamlit](https://www.whitphx.info/posts/20211231-streamlit-webrtc-video-app-tutorial/)\n  * A tutorial for real-time video app development using `streamlit-webrtc`.\n  * Crosspost on dev.to: https://dev.to/whitphx/developing-web-based-real-time-videoaudio-processing-apps-quickly-with-streamlit-4k89\n* [New Component: streamlit-webrtc, a new way to deal with real-time media streams (Streamlit Community)](https://discuss.streamlit.io/t/new-component-streamlit-webrtc-a-new-way-to-deal-with-real-time-media-streams/8669)\n  * This is a forum topic where `streamlit-webrtc` has been introduced and discussed about.\n',
    'author': 'Yuichiro Tsuchiya',
    'author_email': 't.yic.yt@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/whitphx/streamlit-webrtc',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
