#!/usr/bin/env python3
"""Create sample neural recording data using DRIADA's synthetic data generation."""

import numpy as np
import os
import sys

# Add src to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'src')))

from driada.experiment.synthetic import generate_2d_manifold_exp

# Create example data directory
output_dir = os.path.dirname(os.path.abspath(__file__))

print("Generating synthetic 2D spatial experiment...")

# Generate synthetic 2D spatial navigation experiment
# This creates realistic calcium imaging data with place cells
exp = generate_2d_manifold_exp(
    n_neurons=50,
    duration=333.333333,  # exactly 10000 frames at 30 fps
    fps=30.0,
    field_sigma=0.1,  # Size of place fields
    step_size=0.02,   # Movement step size
    momentum=0.8,     # Movement smoothness
    baseline_rate=0.1,
    peak_rate=1.0,
    noise_std=0.05,
    grid_arrangement=True,  # Arrange place fields in grid
    seed=42
)

print(f"Generated experiment with {exp.n_cells} neurons and {exp.n_frames} frames")

# Extract data for saving
x_pos = exp.dynamic_features['x'].data
y_pos = exp.dynamic_features['y'].data

# Calculate speed from position
dx = np.gradient(x_pos)
dy = np.gradient(y_pos)
speed = np.sqrt(dx**2 + dy**2) * exp.static_features['fps']

data_dict = {
    'calcium': exp.calcium.data,
    'position': np.vstack([x_pos, y_pos]),
    'x_pos': x_pos,
    'y_pos': y_pos,
    'speed': speed
    # Note: fps should be provided as static_features when loading, not stored in NPZ
}

# Add trial structure to simulate different behavioral epochs
n_timepoints = exp.calcium.data.shape[1]
trial_duration = 300  # 10 seconds per trial
n_trials = n_timepoints // trial_duration
# Use numeric encoding: 0=A, 1=B, 2=C, 3=D
trial_type = np.zeros(n_timepoints, dtype=int)

for i in range(n_trials):
    start = i * trial_duration
    end = min((i + 1) * trial_duration, n_timepoints)
    trial_type[start:end] = i % 4  # Cycle through 0, 1, 2, 3

# Fill any remaining timepoints
if n_trials * trial_duration < n_timepoints:
    trial_type[n_trials * trial_duration:] = n_trials % 4

data_dict['trial_type'] = trial_type

# Add head direction as circular variable
# Generate smooth head direction from velocity
dx = np.gradient(data_dict['x_pos'])
dy = np.gradient(data_dict['y_pos'])
head_direction = np.arctan2(dy, dx)
# Smooth it
from scipy.ndimage import gaussian_filter1d
head_direction = gaussian_filter1d(head_direction, sigma=5, mode='wrap')
data_dict['head_direction'] = head_direction

# Ensure exactly 10000 frames by padding if necessary
target_frames = 10000
current_frames = data_dict['calcium'].shape[1]

if current_frames < target_frames:
    # Pad with the last frame values
    pad_width = target_frames - current_frames
    
    # Pad calcium data
    data_dict['calcium'] = np.pad(data_dict['calcium'], 
                                  ((0, 0), (0, pad_width)), 
                                  mode='edge')
    
    # Pad 1D arrays
    for key in ['x_pos', 'y_pos', 'speed', 'trial_type', 'head_direction']:
        if key in data_dict:
            data_dict[key] = np.pad(data_dict[key], (0, pad_width), mode='edge')
    
    # Pad 2D position array
    data_dict['position'] = np.pad(data_dict['position'], 
                                   ((0, 0), (0, pad_width)), 
                                   mode='edge')

print(f"Final data shape: {data_dict['calcium'].shape}")

# Save as NPZ file
output_path = os.path.join(output_dir, 'sample_recording.npz')
np.savez(
    output_path,
    **data_dict,
    description='Sample calcium imaging data with place cells generated by DRIADA'
)

print(f"\nCreated {output_path} with:")
print(f"  - calcium: {data_dict['calcium'].shape} (n_neurons x n_timepoints)")
print(f"  - position: {data_dict['position'].shape} (2D trajectory)")
print(f"  - speed: {data_dict['speed'].shape}")
print(f"  - trial_type: {trial_type.shape} (discrete labels)")
print(f"  - head_direction: {data_dict['head_direction'].shape} (circular variable)")
print(f"  Note: fps (30.0) should be provided via static_features when loading")
print(f"\nFile size: {os.path.getsize(output_path) / 1024 / 1024:.1f} MB")

# Also save the experiment object as pickle
from driada.experiment import save_exp_to_pickle

# Need to recreate experiment with padded data
# First add the extra features to the experiment
from driada.information import TimeSeries

# Add trial_type as a feature (already numeric)
exp.dynamic_features['trial_type'] = TimeSeries(
    data_dict['trial_type'][:exp.n_frames],  # Use original length
    ts_type='categorical',  # Discrete categorical data
    name='trial_type'
)

# Add head_direction
exp.dynamic_features['head_direction'] = TimeSeries(
    data_dict['head_direction'][:exp.n_frames],
    ts_type='circular',  # Continuous circular/angular data
    name='head_direction'
)

# Save experiment as pickle
pkl_path = os.path.join(output_dir, 'sample_experiment.pkl')
save_exp_to_pickle(exp, pkl_path)

print(f"\nAlso saved experiment as {pkl_path}")
print(f"Pickle file size: {os.path.getsize(pkl_path) / 1024 / 1024:.1f} MB")