# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['ldict', 'ldict.persistence']

package_data = \
{'': ['*']}

install_requires = \
['garoupa>=2.210907.7,<3.0.0',
 'lange>=0.2101.24,<0.2102.0',
 'orjson>=3.5.0,<4.0.0',
 'pdoc3>=0.10.0,<0.11.0',
 'uncompyle6>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'ldict',
    'version': '2.210913.3',
    'description': 'Lazy dict with universally unique identifier for values',
    'long_description': '![test](https://github.com/davips/ldict/workflows/test/badge.svg)\n[![codecov](https://codecov.io/gh/davips/ldict/branch/main/graph/badge.svg)](https://codecov.io/gh/davips/ldict)\n<div style="background-color: rgb(15, 20, 25);">\n\n# ldict\nUniquely identified lazy dict.\n\n[Latest version as a package](https://pypi.org/project/ldict)\n\n[Current code](https://github.com/davips/ldict)\n\n[API documentation](https://davips.github.io/ldict)\n\n## Overview\nWe consider that every value is generated by a process, starting from `empty`. The process is a sequence of\ntransformation steps that can be of two types:\nvalue insertion and function application. Value insertion is done using dict-like objects as shown below. The\noperator `>>` concatenates the steps chronologically. Each value and each function have unique deterministic\nidentifiers. Identifiers for future values are predictable through the magic\navailable [here](https://pypi.org/project/garoupa).\n![img.png](https://raw.githubusercontent.com/davips/ldict/main/examples/img.png)\n\nFunction application is done in the same way. The parameter names define the input fields, while the keys in the\nreturned dict define the output fields:\n![img_1.png](https://raw.githubusercontent.com/davips/ldict/main/examples/img_1.png)\n\nSimilarly, for anonymous functions:\n![img_5.png](https://raw.githubusercontent.com/davips/ldict/main/examples/img_5.png)\n\nFinally, the result is only evaluated at request:\n![img_6.png](https://raw.githubusercontent.com/davips/ldict/main/examples/img_6.png)\n![img_7.png](https://raw.githubusercontent.com/davips/ldict/main/examples/img_7.png)\n\n\n## Installation\n### ...as a standalone lib\n```bash\n# Set up a virtualenv. \npython3 -m venv venv\nsource venv/bin/activate\n\n# Install from PyPI...\npip install --upgrade pip\npip install -U ldict\n\n# ...or, install from updated source code.\npip install git+https://github.com/davips/ldict\n```\n\n### ...from source\n```bash\ngit clone https://github.com/davips/ldict\ncd ldict\npoetry install\n```\n\n## Examples\n**Merging two ldicts**\n<details>\n<p>\n\n```python3\nfrom ldict import ldict\n\na = ldict(x=3)\nprint(a)\n"""\n{\n    "id": "kr_4aee5c3bcac2c478be9901d57fd1ef8a9d002",\n    "ids": "kr_4aee5c3bcac2c478be9901d57fd1ef8a9d002",\n    "x": 3\n}\n"""\n```\n\n```python3\n\nb = ldict(y=5)\nprint(b)\n"""\n{\n    "id": "Uz_0af6d78f77734fad67e6de7cdba3ea368aae4",\n    "ids": "Uz_0af6d78f77734fad67e6de7cdba3ea368aae4",\n    "y": 5\n}\n"""\n```\n\n```python3\n\nprint(a >> b)\n"""\n{\n    "id": "c._2b0434ca422114262680df425b85cac028be6",\n    "ids": "kr_4aee5c3bcac2c478be9901d57fd1ef8a9d002 Uz_0af6d78f77734fad67e6de7cdba3ea368aae4",\n    "x": 3,\n    "y": 5\n}\n"""\n```\n\n\n</p>\n</details>\n\n**Lazily applying functions to ldict**\n<details>\n<p>\n\n```python3\nfrom ldict import ldict\n\na = ldict(x=3)\nprint(a)\n"""\n{\n    "id": "kr_4aee5c3bcac2c478be9901d57fd1ef8a9d002",\n    "ids": "kr_4aee5c3bcac2c478be9901d57fd1ef8a9d002",\n    "x": 3\n}\n"""\n```\n\n```python3\n\na = a >> ldict(y=5) >> {"z": 7} >> (lambda x, y, z: {"r": x ** y // z})\nprint(a)\n"""\n{\n    "id": "8jopGVdtSEyCk1NSKcrEF-Lfv8up9MQBdvkLxU2o",\n    "ids": "J3tsy4vUXPELySBicaAy-h-UK7Dp9MQBdvkLxU2o... +2 ...Ss_7dff0a161ba7462725cac7dcee71b67669f69",\n    "r": "→(x y z)",\n    "x": 3,\n    "y": 5,\n    "z": 7\n}\n"""\n```\n\n```python3\n\nprint(a.r)\n"""\n34\n"""\n```\n\n```python3\n\nprint(a)\n"""\n{\n    "id": "8jopGVdtSEyCk1NSKcrEF-Lfv8up9MQBdvkLxU2o",\n    "ids": "J3tsy4vUXPELySBicaAy-h-UK7Dp9MQBdvkLxU2o... +2 ...Ss_7dff0a161ba7462725cac7dcee71b67669f69",\n    "r": 34,\n    "x": 3,\n    "y": 5,\n    "z": 7\n}\n"""\n```\n\n\n</p>\n</details>\n\n**Parameterized functions and sampling**\n<details>\n<p>\n\n```python3\nfrom random import Random\n\nfrom ldict import Ø\nfrom ldict.cfg import cfg\n\n\n# A function provide input fields and, optionally, parameters.\n# For instance:\n# \'a\' is sampled from an arithmetic progression\n# \'b\' is sampled from a geometric progression\n# Here, the syntax for default parameter values is borrowed with a new meaning.\ndef fun(x, y, a=[-100, -99, -98, ..., 100], b=[0.0001, 0.001, 0.01, ..., 100000000]):\n    return {"z": a * x + b * y}\n\n\n# Creating an empty ldict. Alternatively: d = ldict().\nd = Ø >> {}\nd.show(colored=False)\n"""\n{\n    "id": "0000000000000000000000000000000000000000",\n    "ids": {}\n}\n"""\n```\n\n```python3\n\n# Putting some values. Alternatively: d = ldict(x=5, y=7).\nd["x"] = 5\nd["y"] = 7\nd.show(colored=False)\n"""\n{\n    "id": "I0_39c94b4dfbc7a8579ca1304eba25917204a5e",\n    "ids": {\n        "x": "Tz_d158c49297834fad67e6de7cdba3ea368aae4",\n        "y": "Rs_92162dea64a7462725cac7dcee71b67669f69"\n    },\n    "x": 5,\n    "y": 7\n}\n"""\n```\n\n```python3\n\n# Parameter values are uniformly sampled.\nd1 = d >> fun\nd1.show(colored=False)\nprint(d1.z)\n"""\n{\n    "id": "OtpiClhsdMcDugtb-hb9bUtynSGmEnxYSn3tq7Hj",\n    "ids": {\n        "z": "mJb2cg66d2EKVJcii7S2BTVb9HGmEnxYSn3tq7Hj",\n        "x": "Tz_d158c49297834fad67e6de7cdba3ea368aae4",\n        "y": "Rs_92162dea64a7462725cac7dcee71b67669f69"\n    },\n    "z": "→(a b x y)",\n    "x": 5,\n    "y": 7\n}\n699670.0\n"""\n```\n\n```python3\n\nd2 = d >> fun\nd2.show(colored=False)\nprint(d2.z)\n"""\n{\n    "id": "aYiQ.3WOifUwPW4v.pQb1FiK1aG1NdH0HvdQXDII",\n    "ids": {\n        "z": ".vyWPsRJmOApQrQBjfv5rEKnP-F1NdH0HvdQXDII",\n        "x": "Tz_d158c49297834fad67e6de7cdba3ea368aae4",\n        "y": "Rs_92162dea64a7462725cac7dcee71b67669f69"\n    },\n    "z": "→(a b x y)",\n    "x": 5,\n    "y": 7\n}\n-239.993\n"""\n```\n\n```python3\n\n# Parameter values can also be manually set.\ne = d >> cfg(a=5, b=10) >> fun\nprint(e.z)\n"""\n95\n"""\n```\n\n```python3\n\n# Not all parameters need to be set.\ne = d >> cfg(a=5) >> fun\nprint(e.z)\n"""\n7025.0\n"""\n```\n\n```python3\n\n# Each run will be a different sample for the missing parameters.\ne = e >> cfg(a=5) >> fun\nprint(e.z)\n"""\n25.007\n"""\n```\n\n```python3\n\n# We can define the initial state of the random sampler.\n# It will be in effect from its location place onwards in the expression.\ne = d >> cfg(a=5) >> Random(0) >> fun\nprint(e.z)\n"""\n699999990.0\n"""\n```\n\n```python3\n\n# All runs will yield the same result,\n# if starting from the same random number generator seed.\ne = e >> cfg(a=5) >> Random(0) >> fun\nprint(e.z)\n"""\n699999990.0\n"""\n```\n\n```python3\n\n# Reproducible different runs are achievable by using a single random number generator.\nrnd = Random(0)\ne = d >> cfg(a=5) >> rnd >> fun\nprint(e.z)\ne = d >> cfg(a=5) >> rnd >> fun  # Alternative syntax.\nprint(e.z)\n"""\n699999990.0\n35.0007\n"""\n```\n\n\n</p>\n</details>\n\n**Composition of sets of functions**\n<details>\n<p>\n\n```python3\nfrom random import Random\n\nfrom ldict import Ø\n\n\n# A multistep process can be defined without applying its functions\n\n\ndef g(x, y, a=[1, 2, 3, ..., 10], b=[0.00001, 0.0001, 0.001, ..., 100000]):\n    return {"z": a * x + b * y}\n\n\ndef h(z, c=[1, 2, 3]):\n    return {"z": c * z}\n\n\n# In the ldict framework \'data is function\',\n# so the alias ø represents the \'empty data object\' and the \'reflexive function\' at the same time.\n# In other words: \'inserting nothing\' has the same effect as \'doing nothing\'.\n# The operator \'*\' is an alias for \'>>\', used just to make the context clearer.\nfun = Ø * g * h  # ø enable the cartesian product of the subsequent sets of functions within the expression.\nprint(fun)\n"""\n«g × h»\n"""\n```\n\n```python3\n\n# The difference between \'ø * g * h\' and \'ldict(x=3) >> g >> h\' is that the functions in the latter are already applied\n# (resulting in an ldict object). The former still has its free parameters unsampled,\n# and results in an ordered set of composite functions.\n# It is a set because the parameter values of the functions are still undefined.\nd = {"x": 5, "y": 7} >> fun\nprint(d)\n"""\n{\n    "id": "eqvHDdMVIvkMG-2nWCklDe6meQKfycl56rr.3DjY",\n    "ids": "JBGd0t0GXbxjntOtes.e1ey..EKfycl56rr.3DjY... +1 ...Rs_92162dea64a7462725cac7dcee71b67669f69",\n    "z": "→(c z→(a b x y))",\n    "x": 5,\n    "y": 7\n}\n"""\n```\n\n```python3\n\nprint(d.z)\n"""\n1490.0\n"""\n```\n\n```python3\n\nd = {"x": 5, "y": 7} >> fun\nprint(d.z)\n"""\n2100015.0\n"""\n```\n\n```python3\n\n# Reproducible different runs by passing a stateful random number generator.\nrnd = Random(0)\ne = d >> rnd >> fun\nprint(e.z)\n"""\n105.0\n"""\n```\n\n```python3\n\ne = d >> rnd >> fun\nprint(e.z)\n"""\n14050.0\n"""\n```\n\n```python3\n\n# Repeating the same results.\nrnd = Random(0)\ne = d >> rnd >> fun\nprint(e.z)\n"""\n105.0\n"""\n```\n\n```python3\n\ne = d >> rnd >> fun\nprint(e.z)\n"""\n14050.0\n"""\n```\n\n\n</p>\n</details>\n\n**Transparent persistence**\n<details>\n<p>\n\n```python3\nimport shelve\nfrom collections import namedtuple\nfrom pprint import pprint\n\nfrom ldict import ldict, Ø\n# The cache can be set globally.\n# It is as simple as a dict, or any dict-like implementation mapping str to serializable content.\n# Implementations can, e.g., store data on disk or in a remote computer.\nfrom ldict.cfg import cfg\nfrom ldict.config import setcache\n\nsetcache({})\n\n\ndef fun(x, y):\n    print("Calculated!")  # Watch whether the value had to be calculated.\n    return {"z": x ** y}\n\n\n# The operator \'^\' indicates a relevant point during the process, i.e., a point where data should be stored.\n# It is mostly intended to avoid costly recalculations or log results.\n# The symbol points upwards, meaning data can momentarily come from or go outside of the process.\n# When the same process is repeated, only the first request will trigger calculation.\n# Local caching objects (dicts or dict-like database servers) can also be used.\n# They should be wrapped by square brackets to avoid ambiguity.\n# The list may contain many different caches, e.g.: [RAM, local, remote].\nmycache = {}\nremote = {}\nd = Ø >> {"x": 3, "y": 2} >> fun >> [mycache, remote]\nprint(d)\nprint(d.z, d.id)\n"""\n{\n    "id": "dpWeC4tFX.7oD1PMWLoyNAaH6gtNSvzvAw2XMZVi",\n    "ids": "GsDJe8CjPiVCEoJEoNzyfKAyyirNSvzvAw2XMZVi... +1 ...yI_a331070d4bcdde465f28ba37ba1310e928122",\n    "z": "→(^ x y)",\n    "x": 3,\n    "y": 2\n}\nCalculated!\n9 dpWeC4tFX.7oD1PMWLoyNAaH6gtNSvzvAw2XMZVi\n"""\n```\n\n```python3\n\n# The second request just retrieves the cached value.\nd = ldict(y=2, x=3) >> fun >> [remote]\nprint(d.z, d.id)\n"""\n9 dpWeC4tFX.7oD1PMWLoyNAaH6gtNSvzvAw2XMZVi\n"""\n```\n\n```python3\n\n# The caching operator can appear in multiple places in the expression, if intermediate values are of interest.\n# The ø is used as ldict-inducer when needed.\nd = ldict(y=2, x=3) >> fun ^ Ø >> (lambda x: {"x": x ** 2}) >> Ø >> {"w": 5, "k": 5} >> Ø >> [mycache]\nprint(d.z, d.id)\n"""\n9 QaRWaaqyTLRqBDzvIff.HdTGQVDeSMDamXXwaYMA\n"""\n```\n\n```python3\n\n# Persisting to disk is easily done via Python shelve.\nP = namedtuple("P", "x y")\na = [3, 2]\nb = [1, 4]\n\n\ndef measure_distance(a, b):\n    from math import sqrt\n    return {"distance": sqrt((a[0] - b[0]) ** 2 + (a[1] - b[1]) ** 2)}\n\n\nwith shelve.open("/tmp/my-cache-file.db") as db:\n    d = ldict(a=a, b=b) >> measure_distance >> [db]\n    pprint(dict(db))  # Cache is initially empty.\n    print(d.distance)\n    pprint(dict(db))\n    #  ...\n\n    # \'^\' syntax is also possible.\n    a = [7, 1]\n    b = [4, 3]\n    copy = lambda source=None, target=None, **kwargs: {target: kwargs[source]}\n    mean = lambda distance, other_distance: {"m": (distance + other_distance) / 2}\n    e = (\n            ldict(a=a, b=b)\n            >> measure_distance\n            >> {"other_distance": d.distance}\n            >> mean\n            ^ Ø\n            ^ cfg(source="m", target="m0")\n            >> copy\n            >> (lambda m: {"m": m ** 2})\n    )\n    print(e.m0, e.m)\n\n"""\n{\'3Q_85403c3464883af128dc24eef54294173d8ef\': [1, 4],\n \'E0_45bf7de0dcdfc012da8a0f556492e8880b09d\': [3, 2],\n \'KBQMiN2gHLwCewlu6HC67I1R2-m8hIbZ8IXI2c0c\': 2.8284271247461903,\n \'uSytg7O3BaGIggOzqYSBO3ees4KQw0Bf1SouJQht\': 2.8284271247461903}\n2.8284271247461903\n{\'3Q_85403c3464883af128dc24eef54294173d8ef\': [1, 4],\n \'E0_45bf7de0dcdfc012da8a0f556492e8880b09d\': [3, 2],\n \'KBQMiN2gHLwCewlu6HC67I1R2-m8hIbZ8IXI2c0c\': 2.8284271247461903,\n \'uSytg7O3BaGIggOzqYSBO3ees4KQw0Bf1SouJQht\': 2.8284271247461903}\n3.2169892001050897 10.349019513592784\n"""\n```\n\n\n</p>\n</details>\n\n<!--- ## Persistence\nExtra dependencies can be installed to support saving data to disk or to a server in the network. \n\n**[still an ongoing work...]**\n\n`poetry install -E full`\n--->\n\n## Concept\nA ldict is like a common Python dict, with extra functionality and lazy.\nIt is a mapping between string keys, called fields, and any serializable object.\nThe ldict `id` (identifier) and the field `ids` are also part of the mapping.  \n\nThe user can provide a unique identifier ([hosh](https://pypi.org/project/garoupa))\nfor each function or value object.\nOtherwise, they will be calculated through blake3 hashing of the content of data or bytecode of function.\nFor this reason, such functions should be simple, i.e.,\nwith minimal external dependencies, to avoid the unfortunate situation where two\nfunctions with identical local code actually perform different calculations through\ncalls to external code that implement different algorithms with the same name.\n<!--- Alternatively, a Hosh object can be passed inside the dict that is returned by the function, under the key "_id". --->\n\n## Grants\nThis work was partially supported by Fapesp under supervision of\nProf. André C. P. L. F. de Carvalho at CEPID-CeMEAI (Grants 2013/07375-0 – 2019/01735-0).\n\n</div>\n',
    'author': 'davips',
    'author_email': 'dpsabc@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
