# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['drf_localize',
 'drf_localize.applications',
 'drf_localize.commons',
 'drf_localize.commons.helpers',
 'drf_localize.core',
 'drf_localize.fixtures',
 'drf_localize.migrations']

package_data = \
{'': ['*']}

install_requires = \
['Django>=3.2.8,<4.0.0',
 'django-cors-headers>=3.10.0,<4.0.0',
 'django-filter>=21.1,<22.0',
 'djangorestframework>=3.12.4,<4.0.0',
 'drf-yasg>=1.20.0,<2.0.0',
 'jsonschema>=3.2.0,<4.0.0',
 'swagger_spec_validator>=2.7.4,<3.0.0']

setup_kwargs = {
    'name': 'drf-localize',
    'version': '0.1.4',
    'description': 'Package to provide localization experiences for mobile and api applications.',
    'long_description': '# 🌐 DRF Localize\n\nPackage to provide localization experiences for mobile and api applications.\n\n# ⚡️ Features\n\n✅ Localize keys for multiple languages<br/>\n✅ Generate localizable `iOS` , `Android`, `Web` compatible **zip** & **files**<br/>\n✅ Model `JSON` based fields localization<br/>\n✅ Library for localizable keys<br/>\n✅ REST API for localizable keys<br/>\n✅ Configurable\n\n# 🔐 Installation\n\nUse the package manager [pip](https://pip.pypa.io/en/stable/) to install drf_localize.\n\n```bash\npip install drf_localize\n```\n\nthen add it to your installed apps:\n\n```python\nINSTALLED_APPS = [\n    ...,\n    \'drf_localize\',\n    ...,\n]\n```\n\nthen run migrate:\n\n```bash\npython manage.py migrate\n```\n\nand load initial data:\n\n```bash\npython manage.py loaddata localizelanguages\npython manage.py loaddata localizekeys (optional)\n```\n\n# 🔨 Configuration\n\nConfiguration for **DRF Localize** is namespaced in a single django setting, named `DRF_LOCALIZE`, by default everything is configured out of the box.\n\n```python\nDRF_LOCALIZE = {\n    \'LANGUAGES\': \'ALL\',  # noqa\n    \'BASE_DIR\': BASE_DIR,  # noqa\n    \'API_KEY_HEADER_NAME\': \'HTTP_X_API_KEY\',\n    \'PAGINATION_CLASS\': \'rest_framework.pagination.PageNumberPagination\',\n    \'KEY_MODEL_CLASS\': \'drf_localize.models.LocalizeKey\',\n    \'LANGUAGE_MODEL_CLASS\': \'drf_localize.models.LocalizeLanguage\',\n    \'LANGUAGE_MODEL_CLASS_FIELD\': \'code\',\n}\n```\n\n# 🔎 Available Settings\n\n| Header                       | Description                                                           | Default                                         |\n|:-----------------------------|:----------------------------------------------------------------------|:------------------------------------------------|\n| `LANGUAGES`                  | **Specify language codes to use as array**.                           | ALL                                             |\n| `BASE_DIR`                   | **Specify internal directory path**.                                  | BASE_DIR                                        |\n| `API_KEY_HEADER_NAME`        | **Specify `X-API-Key` header**.                                       | HTTP_X_API_KEY                                  |\n| `PAGINATION_CLASS`           | **Specify pagination class**.                                         | rest_framework.pagination.PageNumberPagination  |\n| `KEY_MODEL_CLASS`            | **Specify key model class, must comply to `drf_localize` key model**. | drf_localize.models.LocalizeKey                 |\n| `LANGUAGE_MODEL_CLASS`       | **Specify language model class**.                                     | drf_localize.models.LocalizeLanguage            |\n| `LANGUAGE_MODEL_CLASS_FIELD` | **Specify language model class code field**.                          | code                                            |\n\n# 🔧 Usage\n\nSpecify `localize` fields in your model\n\n```python\nfrom django.db import models\n\n\nclass Blog(models.Model):  # noqa\n\n    # DRF Localize model specific constant, which is used to translate model fields\n    LOCALIZE_TRANSLATE = [\'title\']  # or LOCALIZE_TRANSLATE = [\'title\', \'description\']\n\n    # DRF Localize model specific constant, which is used to store translations (json)\n    LOCALIZE_FIELD = \'i18n\'\n\n    # Your custom model fields\n    title = models.CharField(max_length=254, null=True)\n    description = models.CharField(max_length=512, null=True)\n\n    # Referenced json field to store translations\n    i18n = models.JSONField(default=dict)\n```\n\nInherit `I18NModelSerializer` in your model serializer\n\n```python\nfrom apps.blog.models import Blog  # noqa\nfrom drf_localize.serializers import (\n    I18NModelSerializer\n)\n\n\nclass BlogSerializer(I18NModelSerializer):\n    class Meta:\n        model = Blog\n        fields = \'__all__\'\n\n```\n\nUse your model serializer in your endpoints as usual\n\n```python\nfrom apps.blog.models import Blog  # noqa\nfrom apps.blog.serializers import BlogSerializer  # noqa\nfrom rest_framework.viewsets import GenericViewSet\nfrom rest_framework.generics import (\n    ListAPIView,\n    CreateAPIView,\n    RetrieveUpdateDestroyAPIView,\n)\n\n\nclass BlogViewSet(ListAPIView, CreateAPIView, RetrieveUpdateDestroyAPIView, GenericViewSet):\n    queryset = Blog.objects.all()\n    serializer_class = BlogSerializer\n\n```\n\n# 📈 REST API\n\n### Standalone mode\n\nIf you are planning to use `drf_localize`\'s REST API endpoints add package urls in your project.\n\n```python\nfrom django.urls import (include, path, )\n\nurlpatterns = [\n    ...,\n    path(\'drf_localize\', include(\'drf_localize.urls\', namespace=\'localize\')),\n    ...,\n]\n```\n\n### Create localize key\n\n```http\nPOST /drf_localize/localize/keys\n```\n\n| Parameter | Type     | Description            |\n|:----------|:---------|:-----------------------|\n| `code`    | `string` | **Required**. *Unique* |\n| `i18n`    | `object` | **Optional**.          |\n\n```json\n{\n  "code": "DRF Localize",\n  "i18n": {\n    "en": "DRF Localize",\n    "ro": "DRF Localizare",\n    "ru": "DRF Локализация"\n  }\n}\n```\n\n#### Update localize key\n\n```http\nPATCH /drf_localize/localize/keys/:id\n```\n\n| Parameter | Type     | Description         |\n|:----------|:---------|:--------------------|\n| `:id`     | `string` | **Required**. |\n| `code`    | `string` | **Required**. *Unique* |\n| `i18n`    | `object` | **Optional**.       |\n\n```json\n{\n  "i18n": {\n    "en": "DRF Localizes",\n    "ro": "DRF Localizare",\n    "ru": "DRF Локализация"\n  }\n}\n```\n\n### Create localize namespace key\n\n```http\nPOST /drf_localize/localize/keys\n```\n\n| Parameter | Type     | Description            |\n|:----------|:---------|:-----------------------|\n| `code`    | `string` | **Required**. *Unique* |\n| `i18n`    | `object` | **Optional**.          |\n| `type`    | `string` | **Required**.          |\n\n```json\n{\n  "code": "global",\n  "i18n": {\n    "en": {\n      "localize": "DRF Localizes"\n    },\n    "ro": {\n      "localize": "DRF Localizare"\n    },\n    "ru": {\n      "localize": "DRF Локализация"\n    }\n  },\n  "type": "NAMESPACE"\n}\n```\n\n#### Update localize namespace key\n\n```http\nPATCH /drf_localize/localize/keys/:id\n```\n\n| Parameter | Type     | Description         |\n|:----------|:---------|:--------------------|\n| `:id`     | `string` | **Required**. |\n| `code`    | `string` | **Required**. *Unique* |\n| `i18n`    | `object` | **Optional**.       |\n| `type`    | `string` | **Required**.          |\n\n```json\n{\n  "i18n": {\n    "en": {\n      "localize": "DRF Localize"\n    },\n    "ro": {\n      "localize": "DRF Localizare"\n    },\n    "ru": {\n      "localize": "DRF Локализация"\n    }\n  },\n  "type": "NAMESPACE"\n}\n```\n\n#### Retrieve localize namespace keys\n\n```http\nGET /drf_localize/localize/keys?search=NAMESPACE\n```\n\n#### Download platform specific localize key file\n\n```http\nGET /drf_localize/localize/keys/:platform/:language/file\n```\n\n| Parameter   | Type     | Description                                 |\n|:------------|:---------|:--------------------------------------------|\n| `:platform` | `string` | **Required**. `ios`/ `android` / `web`      |\n| `:language` | `string` | **Required**. `en`/ any other language code |\n\n#### Download platform specific localize keys zip file\n\n```http\nGET /drf_localize/localize/keys/:platform/zip\n```\n\n| Parameter   | Type     | Description                                 |\n|:------------|:---------|:--------------------------------------------|\n| `:platform` | `string` | **Required**. `ios`/ `android` / `web`      |\n\n#### Download all platform specific localize keys in a single zip file\n\n```http\nGET /drf_localize/localize/keys/zip\n```\n\n### Service mode\n\nYou will need to add a middleware class:\n\n```python\n# This middleware listens to `X-API-Key` key header value and finds your application.\n\nMIDDLEWARE = [\n    ...,\n    \'drf_localize.middlewares.LocalizeApplicationMiddleware\',\n    ...,\n]\n```\n\nthen add `X-API-Key` header in standalone mode endpoints:\n\n| Header         | Type    | Description                        |\n|:---------------|:--------|:-----------------------------------|\n| `X-API-Key`| `string`| **Required**. Your application key |\n\n### Create application\n\n```http\nPOST /drf_localize/localize/applications\n```\n\n| Parameter | Type     | Description         |\n|:----------|:---------|:--------------------|\n| `title`   | `string` | **Optional**.|\n| `description`    | `string` | **Optional**.       |\n\n#### Retrieve application\n\n```http\nGET /drf_localize/localize/application\n```\n\n| Header         | Type    | Description                        |\n|:---------------|:--------|:-----------------------------------|\n| `X-API-Key`| `string`| **Required**. Your application key |\n\n#### Attach application languages\n\n```http\nPOST /drf_localize/localize/application/languages\n```\n\n| Parameter | Type     | Description   |\n|:----------|:---------|:--------------|\n| `languages_id`   | `array`  | **Required**. |\n\n| Header         | Type    | Description                        |\n|:---------------|:--------|:-----------------------------------|\n| `X-API-Key`| `string`| **Required**. Your application key |\n\n```json\n{\n  "languages_id": [\n    2,\n    1\n  ]\n}\n```\n\n#### Detach application languages\n\n```http\nDELETE /drf_localize/localize/application/languages\n```\n\n| Parameter | Type     | Description   |\n|:----------|:---------|:--------------|\n| `languages_id`   | `array`  | **Required**. |\n\n| Header         | Type    | Description                        |\n|:---------------|:--------|:-----------------------------------|\n| `X-API-Key`| `string`| **Required**. Your application key |\n\n```json\n{\n  "languages_id": [\n    1\n  ]\n}\n```\n\n#### Retrieve application languages\n\n```http\nGET /drf_localize/localize/application/languages\n```\n\n| Header         | Type    | Description                        |\n|:---------------|:--------|:-----------------------------------|\n| `X-API-Key`| `string`| **Required**. Your application key |\n\n```json\n[\n  {\n    "code": "ab",\n    "name": "Abkhaz",\n    "native": "аҧсуа"\n  }\n]\n```\n\n# 📦️ Library\n\n```python\nfrom drf_localize import localize\n\n# Set translatable key translation.\nlocalize.set_key(\'Welcome\').set_en(\'Welcome\').set_ro(\'Bine ati venit\').set_pt(\'Receber\')  # noqa\n\n# Set translatable namespace translation.\nlocalize.set_key_namespace(\'common\').set_en({\n    \'welcome\': \'Welcome\', \'order\': \'Order\'\n}).set_ro({\n    \'welcome\': \'Bine ati venit\', \'order\': \'Ordin\'\n}).set_pt({\n    \'welcome\': \'Receber\', \'order\': \'Pedido\'\n})\n\n# Build \'en\' translation file for iOS\nlocalize.build(language=\'en\')\nfile = localize.to_platform(platform=\'IOS\')  # noqa\n\n# Build translations zip file for iOS\nfile = localize.build_zip(platform=\'IOS\')  # noqa\n\n# Build translations zip file for ANDROID\nfile = localize.build_zip(platform=\'ANDROID\')  # noqa\n\n# Build translations zip file for WEB\nfile = localize.build_zip(platform=\'WEB\')  # noqa\n\n# Build translations zip file for every platform\nfile = localize.build_zip()  # noqa\n```\n\n## 📌 Library helpers\n\n```python\n\n# from drf_localize import localize_key_type\n\nclass LocalizeKeyType:\n    KEY_NAMESPACE = \'NAMESPACE\'\n    KEY_PLAIN = \'PLAIN\'\n\n    KEY_TYPES = [KEY_NAMESPACE, KEY_PLAIN]\n    KEY_CHOICES = (\n        (KEY_NAMESPACE, KEY_NAMESPACE),\n        (KEY_PLAIN, KEY_PLAIN)\n    )\n\n\n# from drf_localize import localize_platform\n\nclass LocalizePlatform:\n    PLATFORM_IOS = \'IOS\'\n    PLATFORM_ANDROID = \'ANDROID\'\n    PLATFORM_WEB = \'WEB\'\n\n    PLATFORM_TYPES = [PLATFORM_IOS, PLATFORM_ANDROID, PLATFORM_WEB]\n    PLATFORM_CHOICES = (\n        (PLATFORM_IOS, PLATFORM_IOS),\n        (PLATFORM_ANDROID, PLATFORM_ANDROID),\n        (PLATFORM_WEB, PLATFORM_WEB)\n    )\n```\n\n# ⚗️ Internal Serializer Classes\n\nUses `I18N` serializer to transform localize field\n\n> **Inherit this `I18NModelSerializer` serializer in your model serializer**\n\n```python\nfrom rest_framework.serializers import ModelSerializer\n\n\nclass I18NModelSerializer(ModelSerializer):  # noqa\n    ...\n```\n\nTransforms localize field into i18n json field with translations\n\n```python\nfrom rest_framework.serializers import Serializer\n\n\nclass I18N(Serializer):  # noqa\n    ...\n```\n\n# 👥 Contributing\n\nPull requests are always appreciated. Open issues addressing pull requests.\n\nFlow:\n\n1. Fork it\n2. Create your feature branch (`git checkout -b my-new-feature`)\n3. Test your changes to the best of your ability.\n4. Update the documentation to reflect your changes if they add or changes current functionality.\n5. Commit your changes (`git commit -am \'Added some feature\'`).\n6. Push to the branch (`git push origin my-new-feature`)\n7. Create new Pull Request',
    'author': 'Dorin Musteața',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<=3.10',
}


setup(**setup_kwargs)
