# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['apos']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'apos',
    'version': '0.1.1',
    'description': 'The backbone for message-driven applications.',
    'long_description': '# apos\n\nThe backbone for message-driven applications.\n\n## Summary\nThis library is designed to act as a software-level message broker, enabling a lightweight implementation of the publish-subscribe design pattern.\n\napos was born to accomplish two objectives:\n* Decouple the application layer from any interfaces\n* Develop reactive business functions\n\nWith apos you can develop a message-driven application. This means that commands, events, and queries are sent to apos, which in return executes the functions that subscribe to these messages. This means that an interface implementation, may it be a web-api or a CLI, would not directly call  application functions, but would rather send a message to apos, which will in return execute the business functions that subscribe to these messages. Equally, a business function would not call any other business function, but rather publishes an event, which other business functions can subscribe to and execute upon, controlled through apos.\n\n\n## Installation\nThe library can be found on PyPi:\nhttps://pypi.org/project/apos/\n\n\n```shell\npip3 install apos\n```\n\n## Decoupling the application layer\nIn this particular case we are looking at the boundry between the interface, which can be anything like a web-api or a AMQP endpoint, and the application layer, which is a bit more ambigious but lets say it contains the business functions/services that your application offers. Using apos, an interface implementation would not be aware of any business functions. This is the case because it would not call functions directly but rather publish a command, event, or query object to apos. Apos will in return execute the functions that subscribe to these messages. This decouples the interfaces almost completely from the business logic. In practice, this means that most of the application layer can be extended, refactored, or replaced without having to change any dependencies in any interface implementations. The interface implementations would only depend on the message objects and would rely on the application layer to provide the logic to react to these messages.\n\n### Publishing\n\n```python\nevents: List[apos.IEvent] = messenger.publish_command(\n            CreateUserCommand(user_name="Max"))\n```\nThe example above shows how an interface implementation can use the apos.Messenger to publish a command. The apos.Messenger will execute the Callable mapped to the name of the command Class. At the same time, it will record the event published by the executed business function, as well as any other events that are published by business functions that are executed as a result of subscribing to the prior events. These events are returned upon the completion of the command execution. It is up to the interface implementation what it wants to do with these events, but an example would be returning them in a HTTP response or publishing the messages to an external message broker like Kafka or RabbitMQ.\n\n```python\nresponse: RetrievedUserResponse = messenger.publish_query(\n            RetrieveUserQuery(user_name="Max"))\n```\nThe example above shows that the apos.Messenger equally supports query messages. Publishing queries works like publishing commands, the only difference being that it will return the response object returned by the business function that subscribes to the query.\n\n### Subscribing\n\n```python\nmessenger.subscribe_command(CreateUserCommand, create_user)\nmessenger.subscribe_event(UserCreatedEvent, email_user)\nmessenger.subscribe_query(RetrieveUserQuery, retrieve_user)\n```\nThe example above shows how the apos.Messenger is used to subscribe functions to message Classes. The apos.Messenger will record these subscriptions by mapping the Class names to the according Callables. Be mindful of the following restrictions:\n* Commands and queries are mapped one-to-one, only allowing one subscription per command\n* Events are mapped one-to-many, allowing multiple subscriptions per event\n\n\n\n## Reactive Business Functions\nLooking at a typical straight forward implementation of an application layer, you will usually see business functions calling other business functions. This becomes funny when its done across domain core aggregate boundries. For example, lets say we have a recruitment platform, where deactivating a user additionally requires a withdrawal of the users job applications an alert email to be sent. In this case, we would likely have a business function for deactivating a user, which would call the business functions for withdrawing job applications and sending an alert email. With this, we coupled the user deactivation to everying else that should happen as a consequence. The other way of doing this would be by having reactive business functions, where the completion of one function would emit an event, which other business functions can subscribe to aka react to. Coming back to the example, withdrawing the job applications and sending an alert email would be business functions that happen as a reaction to the user being deactivated. This way of developing reactive application increases maintainability and extendability because any business function can be added, removed, or refactored independent of any other function, as long as the events are defined. \n\n### Publishing\n\n```python\nmessenger.publish_event(\n    UserDeactivatedEvent(user_name="Max"))\n```\nThe example above shows how an event can be published. In this case, the business function for deactivating a user would publish the UserDeactivatedEvent upon completion.\n\n### Subscribing\n\n```python\nmessenger.subscribe_event(\n    UserDeactivatedEvent, withdraw_job_applications)\n```\nThe example above shows how the apos.Messenger can be used to subscribe a business function to an event. In this case, the business function for withdrawing job applications subscribes to the UserDeactivatedEvent. This means that if the user deactivation business function in the earlier example completes by publishing a UserDeactivatedEvent, the apos.Messenger would react by executing the business function for withdrawing the job applications. Upon calling the function, the event would be passed as an object as a parameter.\n\n\n## Complete Example\n```python\n\n@dataclass\nclass DeactivateUserCommand:\n    user_id: str\n\n@dataclass\nclass UserDeactivatedEvent:\n    user_id: str\n\n@dataclass\nclass ApplicationsWithdrawnEvent:\n    application_ids: List[str]\n\n\nclass DeactivateUser:\n\n    def __init__(self, messenger) -> None:\n        self._messenger = messenger\n\n    def __call__(self, command: DeactivateUserCommand) -> None:\n        # Implementation of user deactivation\n        self._messenger.publish_event(\n            UserDeactivatedEvent(user_id))\n\n\nclass WithdrawApplications:\n\n    def __init__(self, messenger) -> None:\n        self._messenger = messenger\n\n    def __call__(self, event: UserDeactivatedEvent) -> None:\n        # Implementation of job applications withdraw\n        self._messenger.publish_event(\n            ApplicationsWithdrawnEvent(application_ids))\n\n\nmessenger = apos.Messenger()\ndeactivate_user = DeactivateUser(messenger)\nwithdraw_applications = WithdrawApplications(messenger)\n\n# some interface implementation\nmessenger.subscribe_command(DeactivateUserCommand, deactivate_user)\nmessenger.subscribe_event(UserDeactivatedEvent, withdraw_applications)\n\n# some other interface implementation\nevents = messenger.execute_command(DeactivateUserCommand(user_id))\n\n```\n\nPlease mind, that the code above is missing some business logic implementation and variables to keep the example simple.\n\nOptionally, you can implement type-hinting by using the interfaces provided by apos as abstracted Classes. For example, apos provides the apos.IEvent Class, which a custome event Class can extend, and the apos.IEventHandler Class, which a business function Class can extend.  ',
    'author': 'Max Kossatz',
    'author_email': 'max@kossatzonline.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mkossatz/apos',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
