"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericSyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls Textract synchronous API.
 * Supports the Textract APIs:  'GENERIC' | 'ANALYZEID' | 'EXPENSE'
 * When GENERIC is called with features in the manifest definition, will call the AnalzyeDocument API.
 * Takes the configuration from "Payload"."manifest"
 * Will retry on recoverable errors based on textractAsyncCallMaxRetries
 * errors for retry: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
 *
 * Input: "Payload"."manifest"
 * Output: Textract JSON Schema at  s3_output_bucket/s3_output_prefix
 *
 * Example (Python)
 * ```python
 *         textract_sync_task = tcdk.TextractGenericSyncSfnTask(
            self,
            "TextractSync",
            s3_output_bucket=document_bucket.bucket_name,
            s3_output_prefix=s3_output_prefix,
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            lambda_log_level="DEBUG",
            timeout=Duration.hours(24),
            input=sfn.TaskInput.from_object({
                "Token":
                sfn.JsonPath.task_token,
                "ExecutionId":
                sfn.JsonPath.string_at('$$.Execution.Id'),
                "Payload":
                sfn.JsonPath.entire_payload,
            }),
            result_path="$.textract_result")
  ```
 */
class TextractGenericSyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 300 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 300 : props.lambdaMemory;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        var enableCloudWatchMetricsAndDashboard = props.enableCloudWatchMetricsAndDashboard === undefined ? false :
            props.enableCloudWatchMetricsAndDashboard;
        this.textractSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_sync/')),
            memorySize: lambdaMemory,
            architecture: lambda.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractSyncCallTask = new tasks.LambdaInvoke(this, 'TextractSyncCallTask', { lambdaFunction: this.textractSyncCallFunction });
        textractSyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Analyze*',
                'textract:Detect*',
            ],
            resources: ['*'],
        }));
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        this.textractSyncLambdaLogGroup = this.textractSyncCallFunction.logGroup;
        const workflow_chain = sfn.Chain.start(textractSyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure', 'states:SendTaskSuccess',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        if (enableCloudWatchMetricsAndDashboard) {
            const appName = this.node.tryGetContext('appName');
            const customMetricNamespace = 'TextractConstructGenericSync';
            // OPERATIONAL
            const syncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Duration',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                    .whereString('message', '=', `textract_sync_${textractAPI}_call_duration_in_ms:`),
                metricValue: '$durationMs',
            });
            this.syncDurationMetric = syncDurationMetricFilter.metric({ statistic: 'avg' });
            const syncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_sync_${textractAPI}_number_of_pages_processed:`),
                metricValue: '$pages',
            });
            this.syncNumberPagesMetric = syncNumberPagesMetricFilter.metric({ statistic: 'sum' });
            const syncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_sync_${textractAPI}_number_of_pages_send_to_process:`),
                metricValue: '$pages',
            });
            this.syncNumberPagesSendMetric = syncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
            const syncTimedOutMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-TimedOutFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'DurationSeconds',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message1', 'durationS', 'message2')
                    .whereString('message1', '=', 'Task timed out after').whereString('message2', '=', 'seconds'),
                metricValue: '$durationS',
            });
            this.syncTimedOutMetric = syncTimedOutMetricFilter.metric({ statistic: 'sum' });
            // OPERATIONAL STOP
            // CALCUATED OPERATIONAL METRICS
            const pagesPerSecond = new cloudwatch.MathExpression({
                expression: 'pages / (duration / 1000)',
                usingMetrics: {
                    pages: this.syncNumberPagesMetric,
                    duration: this.syncDurationMetric,
                },
            });
            // CALCUATED OPERATIONAL METRICS STOP
            const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Errors',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
                metricValue: '1',
            });
            const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'LimitExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
                metricValue: '1',
            });
            const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ThrottlingException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
                metricValue: '1',
            });
            const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ProvisionedThroughputExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
                metricValue: '1',
            });
            const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'InternalServerError',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
                metricValue: '1',
            });
            const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
            const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
            const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
            const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
            const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
            const textractStartDocumentTextThrottle = new cloudwatch.Metric({
                namespace: 'AWS/Textract',
                metricName: 'ThrottledCount',
                dimensionsMap: {
                    Operation: 'StartDocumentTextDetection',
                },
                statistic: 'sum',
            });
            // DASHBOARD LAMBDA
            const dashboardWidth = 24;
            // const widgetStandardHeight=9;
            // DASHBOARD
            new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
                end: 'end',
                periodOverride: cloudwatch.PeriodOverride.AUTO,
                start: 'start',
                widgets: [
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [this.syncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesProcessed', left: [this.syncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesSendToProcess', left: [this.syncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractSyncCallFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Errors', left: [this.textractSyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Invocations', left: [this.textractSyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Throttles', left: [this.textractSyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-TimedOut', left: [this.syncTimedOutMetric], width: Math.floor(dashboardWidth / 4) }),
                    ],
                ],
            });
        }
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericSyncSfnTask = TextractGenericSyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericSyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericSyncSfnTask", version: "0.0.9" };
TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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