"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CSVToAuroraTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const rds = require("aws-cdk-lib/aws-rds");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const rdsServerlessInit_1 = require("./rdsServerlessInit");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * CSVToAuroraTask is a demo construct to show import into a serverless Aurora DB.
 * At the moment it also creates the Aurora Serverless RDS DB, initializes a table structure the matches the output of the GenerateCSV construct.
 * The Step Functions flow expect a pointer to a CSV at "csv_output_location"."TextractOutputCSVPath" and uses that to execute a batch insert statement command.
 *
 * Example:
 * ```python
*  csv_to_aurora_task = tcdk.CSVToAuroraTask(
    self,
    "CsvToAurora",
    vpc=vpc,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    lambda_log_level="DEBUG",
    timeout=Duration.hours(24),
    input=sfn.TaskInput.from_object({
      "Token":
      sfn.JsonPath.task_token,
      "ExecutionId":
      sfn.JsonPath.string_at('$$.Execution.Id'),
      "Payload":
      sfn.JsonPath.entire_payload
    }),
    result_path="$.textract_result")
  ```
 *
 * Input: "csv_output_location"."TextractOutputCSVPath"
 * Output: CSV in Aurora Serverless DB, table name 'textractcsvimport"
 */
class CSVToAuroraTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, CSVToAuroraTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var enableCloudWatchMetricsAndDashboard = props.enableCloudWatchMetricsAndDashboard === undefined ? false :
            props.enableCloudWatchMetricsAndDashboard;
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 600 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 256 : props.lambdaMemory;
        var csvToAuroraMaxRetries = props.csvToAuroraMaxRetries === undefined ? 100 : props.csvToAuroraMaxRetries;
        var csvToAuroraBackoffRate = props.csvToAuroraBackoffRate === undefined ? 1.1 : props.csvToAuroraBackoffRate;
        var csvToAuroraInterval = props.csvToAuroraInterval === undefined ? 1 : props.csvToAuroraInterval;
        if (props.dbCluster === undefined) {
            if (props.vpc === undefined) {
                throw new Error('if no dbCluster is passed in, requires a vpc props.');
            }
            this.lambdaSecurityGroup = new ec2.SecurityGroup(this, 'LambdaSG', { allowAllOutbound: true, vpc: props.vpc });
            this.auroraSecurityGroup = new ec2.SecurityGroup(this, 'Aurora', { allowAllOutbound: true, vpc: props.vpc });
            this.auroraSecurityGroup.addIngressRule(this.auroraSecurityGroup, ec2.Port.tcp(5432), 'fromSameSG');
            this.auroraSecurityGroup.addIngressRule(this.auroraSecurityGroup, ec2.Port.tcp(443), 'fromSameSG');
            this.auroraSecurityGroup.addIngressRule(this.lambdaSecurityGroup, ec2.Port.tcp(5432), 'LambdaIngreess');
            this.auroraSecurityGroup.addIngressRule(this.lambdaSecurityGroup, ec2.Port.tcp(443), 'LambdaIngreess');
            // AURORA
            this.dbCluster = new rds.ServerlessCluster(this, id + 'AuroraPSQL', {
                engine: rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
                parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-postgresql10'),
                vpc: props.vpc,
                securityGroups: [this.auroraSecurityGroup],
                enableDataApi: true,
            });
            this.dbCluster.node.addDependency(props.vpc);
            const rdsServerlessInit = new rdsServerlessInit_1.RdsServerlessInit(this, 'RdsServerlessInit', {
                dbClusterSecretARN: this.dbCluster.secret.secretArn,
                dbClusterARN: this.dbCluster.clusterArn,
            });
            rdsServerlessInit.node.addDependency(this.dbCluster);
        }
        else {
            if (props.lambdaSecurityGroup != undefined && props.auroraSecurityGroup != undefined && props.dbCluster != undefined) {
                this.lambdaSecurityGroup = props.lambdaSecurityGroup;
                this.auroraSecurityGroup = props.lambdaSecurityGroup;
                this.dbCluster = props.dbCluster;
            }
            else {
                throw new Error('Need lambdaSeucrityGroup and auroraSecurityGroup and dbCluster');
            }
        }
        // LAMBDA PUT ON Cluster
        this.csvToAuroraFunction = new lambda.DockerImageFunction(this, 'CSVToAuroraFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/csv_to_aurora/')),
            memorySize: lambdaMemory,
            architecture: lambda.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            securityGroups: [this.lambdaSecurityGroup],
            vpc: props.vpc,
            environment: {
                SECRET_ARN: this.dbCluster.secret.secretArn,
                CLUSTER_ARN: this.dbCluster.clusterArn,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.csvToAuroraLambdaLogGroup = this.csvToAuroraFunction.logGroup;
        const csvToAuroraTask = new tasks.LambdaInvoke(this, 'TextractSyncCallTask', { lambdaFunction: this.csvToAuroraFunction });
        csvToAuroraTask.addRetry({
            maxAttempts: csvToAuroraMaxRetries,
            backoffRate: csvToAuroraBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(csvToAuroraInterval),
            errors: ['StatementTimeoutException', 'InternalServerErrorException', 'ServiceUnavailableError', 'Lambda.Unknown'],
        });
        // WORKFLOW
        const workflow_chain = sfn.Chain.start(csvToAuroraTask);
        this.stateMachine = new sfn.StateMachine(this, id + '-SFN', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'secretsmanager:GetSecretValue',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket',
            ],
            resources: ['*'],
        }));
        this.csvToAuroraFunction.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonRDSDataFullAccess'));
        // START DASHBOARD
        if (enableCloudWatchMetricsAndDashboard) {
            const appName = this.node.tryGetContext('appName');
            const customMetricNamespace = 'TextractIDPCSVToAurora';
            const csvToAuroraNumberRowsInsertedFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberOfRowsFilter`, {
                logGroup: this.csvToAuroraFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberOfRows',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'numberOfRows')
                    .whereString('message', '=', 'csv_to_aurora_insert_rows:'),
                metricValue: '$numberOfRows',
            });
            this.csvToAuroraNumberRowsInsertedMetric = csvToAuroraNumberRowsInsertedFilter.metric({ statistic: 'sum' });
        }
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
           * @internal
           */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
           * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
           *
           * the scoped access policy should be generated accordingly.
           *
           * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
           */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.CSVToAuroraTask = CSVToAuroraTask;
_a = JSII_RTTI_SYMBOL_1;
CSVToAuroraTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.CSVToAuroraTask", version: "0.0.9" };
CSVToAuroraTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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