"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToStepFunctionProps} props - user provided props for the construct
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.s3Bucket = defaults.buildS3Bucket(this, {
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        this.addCfnNagSuppress(this.s3Bucket);
        if (!props.hasOwnProperty('deployCloudTrail') || props.deployCloudTrail === true) {
            const trailBucket = defaults.buildS3Bucket(this, {}, 'CloudTrail');
            this.addCfnNagSuppress(trailBucket);
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: trailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket: this.s3Bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                this.s3Bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
    addCfnNagSuppress(bucket) {
        // Extract the CfnBucket from the s3Bucket
        const s3BucketResource = bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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