"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeleteImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implementation for aws:deleteImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html
 */
class DeleteImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:deleteImage';
        this.deleteImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        const imageId = this.deleteImageStep.imageId.resolveToString(inputs);
        console.log(`DeleteImage: Get snapshots of ${imageId}`);
        const snapshots = this.getSnapshots(imageId);
        console.log(`DeleteImage: Deregister ${imageId}`);
        this.deregisterImage(imageId);
        console.log(`DeleteImage: Delete snapshots for ${imageId}`);
        this.deleteSnapshotsOfImage(snapshots);
        console.log(`DeleteImage: ${imageId} is deleted`);
        return {};
    }
    getSnapshots(imageId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeImage', {
            service: 'EC2',
            pascalCaseApi: 'DescribeImages',
            apiParams: {
                ImageIds: [imageId],
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    selector: '$.Images[0]',
                    name: 'Image',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Describe image failed for ${imageId}: ${result.stackTrace}`);
        }
        const image = result.outputs?.['describeImage.Image'];
        return image?.BlockDeviceMappings.map(mapping => mapping.Ebs.SnapshotId) ?? [];
    }
    deregisterImage(imageId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deregisterImage', {
            service: 'EC2',
            pascalCaseApi: 'DeregisterImage',
            apiParams: {
                ImageId: imageId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Deregister image failed for ${imageId}: ${result.stackTrace}`);
        }
    }
    deleteSnapshotsOfImage(snapshotIds) {
        for (const snapshotId of snapshotIds) {
            if (snapshotId === null || snapshotId === undefined) {
                continue;
            }
            try {
                this.deleteSnapshot(snapshotId);
            }
            catch (e) {
                if (e.message.indexOf('InvalidSnapshot.InUse') < 0) {
                    throw e;
                }
                console.log('InvalidSnapshot.InUse case found. Attempt to retry after 2 seconds.');
                new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                    sleepSeconds: 2,
                }), this.props).invoke({});
                this.deleteSnapshot(snapshotId);
            }
        }
    }
    deleteSnapshot(snapshotId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deleteSnapshot', {
            service: 'EC2',
            pascalCaseApi: 'DeleteSnapshot',
            apiParams: {
                SnapshotId: snapshotId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Delete snapshot failed for ${snapshotId}: ${result.stackTrace}`);
        }
    }
}
exports.DeleteImageSimulation = DeleteImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
DeleteImageSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.DeleteImageSimulation", version: "0.0.25" };
//# sourceMappingURL=data:application/json;base64,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