"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CopyImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:copyImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-copyimage.html
 */
class CopyImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:copyImage';
        this.copyImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        console.log(`CopyImage: Creating copy of image ${this.copyImageStep.sourceImageId}`);
        const imageId = this.copyImage(inputs);
        console.log(`CopyImage: Waiting for ${imageId} to be available`);
        const state = this.waitForImageAvailable(imageId);
        console.log(`CopyImage: Image ${imageId} is available`);
        return {
            ImageId: imageId,
            ImageState: state,
        };
    }
    copyImage(inputs) {
        const apiParamMap = {
            Name: this.copyImageStep.imageName,
            SourceImageId: this.copyImageStep.sourceImageId,
            SourceRegion: this.copyImageStep.sourceRegion,
            ClientToken: this.copyImageStep.clientToken,
            Description: this.copyImageStep.imageDescription,
            Encrypted: this.copyImageStep.encrypted,
            KmsKeyId: this.copyImageStep.kmsKeyId,
        };
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord(apiParamMap, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'copyImage', {
            service: 'EC2',
            pascalCaseApi: 'CopyImage',
            apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    name: 'ImageId',
                    selector: '$.ImageId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Copy image failed for ${apiParams.SourceImageId}: ${result.stackTrace}`);
        }
        return result.outputs?.['copyImage.ImageId'];
    }
    waitForImageAvailable(imageId) {
        wait_for_and_assert_1.waitForAndAssertImageAvailable({
            imageId: imageId,
        }, this.props);
        return 'available';
    }
}
exports.CopyImageSimulation = CopyImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
CopyImageSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CopyImageSimulation", version: "0.0.25" };
//# sourceMappingURL=data:application/json;base64,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