"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomationStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const __1 = require("..");
const data_type_1 = require("../domain/data-type");
const on_failure_1 = require("../interface/on-failure");
const step_1 = require("./step");
/**
 * Parent class for AutomationSteps. These steps are using in an AutomationDocument.
 * You can instantiate steps using the AutomationBuilder for convenience.
 * You can use these steps to simulate their execution (mimics the SSM run) AND to build their yaml/json declaration.
 * Control flow of the subsequent step is determined by the currently executing step.
 * The flow of the execution therefore follows a chain-of-responsibility pattern.
 * The inputs received into a step AND the outputs of previous steps are merged to form inputs of subsequent steps.
 */
class AutomationStep extends step_1.Step {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.maxAttempts = props.maxAttempts ?? AutomationStep.DEFAULT_MAX_ATTEMPTS;
        this.timeoutSeconds = props.timeoutSeconds ?? AutomationStep.DEFAULT_TIMEOUT;
        this.isEnd = props.isEnd ?? false;
        this.onFailure = props.onFailure ?? new on_failure_1.Abort();
        this.onCancel = props.onCancel ?? new on_failure_1.Abort();
        this.explicitNextStep = props.explicitNextStep;
    }
    addToDocument(doc) {
        doc.addStep(this);
    }
    prepareSsmEntry(inputs) {
        // Maps maintain insertion order
        const ssmDef = new Map();
        if (this.description) {
            ssmDef.set('description', this.description);
        }
        ssmDef.set('name', this.name);
        ssmDef.set('action', this.action);
        ssmDef.set('inputs', inputs);
        const formattedOutputs = this.listOutputs().map(output => this.formatOutput(output));
        if (this.listOutputs().length > 0) {
            ssmDef.set('outputs', formattedOutputs);
        }
        if (this.isEnd) {
            ssmDef.set('isEnd', true);
        }
        if (this.maxAttempts != AutomationStep.DEFAULT_MAX_ATTEMPTS) {
            ssmDef.set('maxAttempts', this.maxAttempts);
        }
        if (this.timeoutSeconds != AutomationStep.DEFAULT_TIMEOUT) {
            ssmDef.set('timeoutSeconds', this.timeoutSeconds);
        }
        if (!(this.onCancel instanceof on_failure_1.Abort)) {
            ssmDef.set('onCancel', this.onCancel.toSsmValue());
        }
        if (!(this.onFailure instanceof on_failure_1.Abort)) {
            ssmDef.set('onFailure', this.onFailure.toSsmValue());
        }
        if (this.explicitNextStep != undefined) {
            ssmDef.set('nextStep', this.explicitNextStep.stepName);
        }
        const ssmDict = {};
        ssmDef.forEach((value, key) => {
            ssmDict[key] = value;
        });
        return ssmDict;
    }
    /**
     * Formats the provided output into an object that can be used to generate the SSM yaml/json representation.
     */
    formatOutput(output) {
        return {
            Name: output.name,
            Selector: output.selector,
            Type: new data_type_1.DataType(output.outputType).toSsmString(),
        };
    }
    variables() {
        return Object.assign({}, ...this.listOutputs()
            .map(out => ({ [out.name]: new __1.StringVariable(`${this.name}.${out.name}`) })));
    }
}
exports.AutomationStep = AutomationStep;
_a = JSII_RTTI_SYMBOL_1;
AutomationStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.AutomationStep", version: "0.0.25" };
AutomationStep.DEFAULT_TIMEOUT = 3600;
AutomationStep.DEFAULT_MAX_ATTEMPTS = 1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXV0b21hdGlvbi1zdGVwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3BhcmVudC1zdGVwcy9hdXRvbWF0aW9uLXN0ZXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSwwQkFBb0M7QUFHcEMsbURBQStDO0FBRS9DLHdEQUFxRTtBQUVyRSxpQ0FBeUM7QUE2Q3pDOzs7Ozs7O0dBT0c7QUFDSCxNQUFzQixjQUFlLFNBQVEsV0FBSTtJQWEvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCO1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxjQUFjLENBQUMsb0JBQW9CLENBQUM7UUFDNUUsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxJQUFJLGNBQWMsQ0FBQyxlQUFlLENBQUM7UUFDN0UsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksSUFBSSxrQkFBSyxFQUFFLENBQUM7UUFDaEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxJQUFJLElBQUksa0JBQUssRUFBRSxDQUFDO1FBQzlDLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7SUFDakQsQ0FBQztJQUVNLGFBQWEsQ0FBQyxHQUE4QjtRQUNqRCxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3BCLENBQUM7SUFFUyxlQUFlLENBQUMsTUFBK0I7UUFDdkQsZ0NBQWdDO1FBQ2hDLE1BQU0sTUFBTSxHQUFxQixJQUFJLEdBQUcsRUFBRSxDQUFDO1FBQzNDLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNwQixNQUFNLENBQUMsR0FBRyxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDN0M7UUFDRCxNQUFNLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDOUIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ2xDLE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRTdCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUNyRixJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2pDLE1BQU0sQ0FBQyxHQUFHLENBQUMsU0FBUyxFQUFFLGdCQUFnQixDQUFDLENBQUM7U0FDekM7UUFDRCxJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDZCxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztTQUMzQjtRQUNELElBQUksSUFBSSxDQUFDLFdBQVcsSUFBSSxjQUFjLENBQUMsb0JBQW9CLEVBQUU7WUFDM0QsTUFBTSxDQUFDLEdBQUcsQ0FBQyxhQUFhLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQzdDO1FBQ0QsSUFBSSxJQUFJLENBQUMsY0FBYyxJQUFJLGNBQWMsQ0FBQyxlQUFlLEVBQUU7WUFDekQsTUFBTSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7U0FDbkQ7UUFDRCxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxZQUFZLGtCQUFLLENBQUMsRUFBRTtZQUNyQyxNQUFNLENBQUMsR0FBRyxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxZQUFZLGtCQUFLLENBQUMsRUFBRTtZQUN0QyxNQUFNLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDdEQ7UUFDRCxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsSUFBSSxTQUFTLEVBQUU7WUFDdEMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3hEO1FBQ0QsTUFBTSxPQUFPLEdBQTRCLEVBQUUsQ0FBQztRQUM1QyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxFQUFFO1lBQzVCLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7UUFDdkIsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRUQ7O09BRUc7SUFDSyxZQUFZLENBQUMsTUFBYztRQUNqQyxPQUFPO1lBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1lBQ2pCLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUTtZQUN6QixJQUFJLEVBQUUsSUFBSSxvQkFBUSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxXQUFXLEVBQUU7U0FDcEQsQ0FBQztJQUNKLENBQUM7SUFFTSxTQUFTO1FBQ2QsT0FBTyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUU7YUFDM0MsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksa0JBQWMsQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNuRixDQUFDOztBQWhGSCx3Q0FpRkM7OztBQWhGaUIsOEJBQWUsR0FBRyxJQUFJLENBQUM7QUFDdkIsbUNBQW9CLEdBQUcsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBTdHJpbmdWYXJpYWJsZSB9IGZyb20gJy4uJztcbmltcG9ydCB7IElBdXRvbWF0aW9uQ29tcG9uZW50IH0gZnJvbSAnLi4vY29uc3RydWN0L2RvY3VtZW50LWNvbXBvbmVudCc7XG5pbXBvcnQgeyBBdXRvbWF0aW9uRG9jdW1lbnRCdWlsZGVyIH0gZnJvbSAnLi4vZG9jdW1lbnQvZG9jdW1lbnQtYnVpbGRlcic7XG5pbXBvcnQgeyBEYXRhVHlwZSB9IGZyb20gJy4uL2RvbWFpbi9kYXRhLXR5cGUnO1xuaW1wb3J0IHsgT3V0cHV0IH0gZnJvbSAnLi4vZG9tYWluL291dHB1dCc7XG5pbXBvcnQgeyBBYm9ydCwgT25DYW5jZWwsIE9uRmFpbHVyZSB9IGZyb20gJy4uL2ludGVyZmFjZS9vbi1mYWlsdXJlJztcbmltcG9ydCB7IFN0ZXBSZWYgfSBmcm9tICcuLi9pbnRlcmZhY2Uvc3RlcC1yZWYnO1xuaW1wb3J0IHsgU3RlcCwgU3RlcFByb3BzIH0gZnJvbSAnLi9zdGVwJztcblxuZXhwb3J0IGludGVyZmFjZSBBdXRvbWF0aW9uU3RlcFByb3BzIGV4dGVuZHMgU3RlcFByb3BzIHtcblxuICAvKipcbiAgICogV2hldGhlciB0byBzdG9wIGRvY3VtZW50IGV4ZWN1dGlvbiBhZnRlciB0aGlzIHN0ZXAuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpc0VuZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIChPcHRpb25hbCkgRmFsbGJhY2sgYWN0aW9uIHRvIHRha2UgaW4gdGhlIGV2ZW50IHRoYXQgdGhpcyBzdGVwIGZhaWxzLlxuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IG9uRmFpbHVyZT86IE9uRmFpbHVyZTtcblxuICAvKipcbiAgICogKE9wdGlvbmFsKSBGYWxsYmFjayBhY3Rpb24gdG8gdGFrZSBpbiB0aGUgZXZlbnQgdGhhdCB0aGlzIHN0ZXAgaXMgY2FuY2VsbGVkLlxuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IG9uQ2FuY2VsPzogT25DYW5jZWw7XG5cblxuICAvKipcbiAgICogKE9wdGlvbmFsKSBtYXggYXR0ZW1wdHMgdG8gcnVuIHRoaXMgc3RlcCBpZiB0aGVyZSBhcmUgZmFpbHVyZXMuXG4gICAqIEBkZWZhdWx0IFN0ZXAuREVGQVVMVF9NQVhfQVRURU1QVFNcbiAgICovXG4gIHJlYWRvbmx5IG1heEF0dGVtcHRzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiAoT3B0aW9uYWwpIHRpbWVvdXQgc2Vjb25kcyB0byBydW4gdGhpcyBzdGVwLlxuICAgKiBJbiBhIHNpbXVsYXRpb24gcnVuLCB0aGlzIHdpbGwgb25seSBiZSBlbmNvcmNlZCBhZnRlci10aGUtZmFjdCBidXQgZXhlY3V0aW9uIHdpbGwgbm90IGJlIHN0b3BwZWQgbWlkLXN0ZXAuXG4gICAqIEBkZWZhdWx0IFN0ZXAuREVGQVVMVF9USU1FT1VUXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0U2Vjb25kcz86IG51bWJlcjtcblxuICAvKipcbiAgICogKE9wdGlvbmFsKSBleHBsaWNpdCBzdGVwIHRvIGdvIHRvIGFmdGVyIHRoaXMgc3RlcCBjb21wbGV0ZXMuXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zeXN0ZW1zLW1hbmFnZXIvbGF0ZXN0L3VzZXJndWlkZS9hdXRvbWF0aW9uLWFjdGlvbnMuaHRtbCNuZXh0UHJvcFxuICAgKiBAZGVmYXVsdCB3aWxsIGltcGxpY2l0bHkgY2hvb3NlIHRoZSBuZXh0IHN0ZXAgaW4gdGhlIHNlcXVlbmNlIHRoYXQgdGhlIHN0ZXBzIGFyZSBhZGRlZCB0byB0aGUgZG9jdW1lbnQuXG4gICAqL1xuICByZWFkb25seSBleHBsaWNpdE5leHRTdGVwPzogU3RlcFJlZjtcblxufVxuXG4vKipcbiAqIFBhcmVudCBjbGFzcyBmb3IgQXV0b21hdGlvblN0ZXBzLiBUaGVzZSBzdGVwcyBhcmUgdXNpbmcgaW4gYW4gQXV0b21hdGlvbkRvY3VtZW50LlxuICogWW91IGNhbiBpbnN0YW50aWF0ZSBzdGVwcyB1c2luZyB0aGUgQXV0b21hdGlvbkJ1aWxkZXIgZm9yIGNvbnZlbmllbmNlLlxuICogWW91IGNhbiB1c2UgdGhlc2Ugc3RlcHMgdG8gc2ltdWxhdGUgdGhlaXIgZXhlY3V0aW9uIChtaW1pY3MgdGhlIFNTTSBydW4pIEFORCB0byBidWlsZCB0aGVpciB5YW1sL2pzb24gZGVjbGFyYXRpb24uXG4gKiBDb250cm9sIGZsb3cgb2YgdGhlIHN1YnNlcXVlbnQgc3RlcCBpcyBkZXRlcm1pbmVkIGJ5IHRoZSBjdXJyZW50bHkgZXhlY3V0aW5nIHN0ZXAuXG4gKiBUaGUgZmxvdyBvZiB0aGUgZXhlY3V0aW9uIHRoZXJlZm9yZSBmb2xsb3dzIGEgY2hhaW4tb2YtcmVzcG9uc2liaWxpdHkgcGF0dGVybi5cbiAqIFRoZSBpbnB1dHMgcmVjZWl2ZWQgaW50byBhIHN0ZXAgQU5EIHRoZSBvdXRwdXRzIG9mIHByZXZpb3VzIHN0ZXBzIGFyZSBtZXJnZWQgdG8gZm9ybSBpbnB1dHMgb2Ygc3Vic2VxdWVudCBzdGVwcy5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEF1dG9tYXRpb25TdGVwIGV4dGVuZHMgU3RlcCBpbXBsZW1lbnRzIElBdXRvbWF0aW9uQ29tcG9uZW50IHtcbiAgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfVElNRU9VVCA9IDM2MDA7XG4gIHN0YXRpYyByZWFkb25seSBERUZBVUxUX01BWF9BVFRFTVBUUyA9IDE7XG4gIHJlYWRvbmx5IG1heEF0dGVtcHRzOiBudW1iZXI7XG4gIHJlYWRvbmx5IHRpbWVvdXRTZWNvbmRzOiBudW1iZXI7XG4gIHJlYWRvbmx5IGlzRW5kOiBib29sZWFuO1xuICByZWFkb25seSBvbkZhaWx1cmU6IE9uRmFpbHVyZTtcbiAgcmVhZG9ubHkgb25DYW5jZWw6IE9uQ2FuY2VsO1xuICByZWFkb25seSBleHBsaWNpdE5leHRTdGVwPzogU3RlcFJlZjtcblxuICBuZXh0U3RlcD86IEF1dG9tYXRpb25TdGVwO1xuICBhbGxTdGVwc0luRXhlY3V0aW9uPzogQXV0b21hdGlvblN0ZXBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXV0b21hdGlvblN0ZXBQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIHRoaXMubWF4QXR0ZW1wdHMgPSBwcm9wcy5tYXhBdHRlbXB0cyA/PyBBdXRvbWF0aW9uU3RlcC5ERUZBVUxUX01BWF9BVFRFTVBUUztcbiAgICB0aGlzLnRpbWVvdXRTZWNvbmRzID0gcHJvcHMudGltZW91dFNlY29uZHMgPz8gQXV0b21hdGlvblN0ZXAuREVGQVVMVF9USU1FT1VUO1xuICAgIHRoaXMuaXNFbmQgPSBwcm9wcy5pc0VuZCA/PyBmYWxzZTtcbiAgICB0aGlzLm9uRmFpbHVyZSA9IHByb3BzLm9uRmFpbHVyZSA/PyBuZXcgQWJvcnQoKTtcbiAgICB0aGlzLm9uQ2FuY2VsID0gcHJvcHMub25DYW5jZWwgPz8gbmV3IEFib3J0KCk7XG4gICAgdGhpcy5leHBsaWNpdE5leHRTdGVwID0gcHJvcHMuZXhwbGljaXROZXh0U3RlcDtcbiAgfVxuXG4gIHB1YmxpYyBhZGRUb0RvY3VtZW50KGRvYzogQXV0b21hdGlvbkRvY3VtZW50QnVpbGRlcik6IHZvaWQge1xuICAgIGRvYy5hZGRTdGVwKHRoaXMpO1xuICB9XG5cbiAgcHJvdGVjdGVkIHByZXBhcmVTc21FbnRyeShpbnB1dHM6IHsgW25hbWU6IHN0cmluZ106IGFueSB9KTogeyBbbmFtZTogc3RyaW5nXTogYW55IH0ge1xuICAgIC8vIE1hcHMgbWFpbnRhaW4gaW5zZXJ0aW9uIG9yZGVyXG4gICAgY29uc3Qgc3NtRGVmOiBNYXA8c3RyaW5nLCBhbnk+ID0gbmV3IE1hcCgpO1xuICAgIGlmICh0aGlzLmRlc2NyaXB0aW9uKSB7XG4gICAgICBzc21EZWYuc2V0KCdkZXNjcmlwdGlvbicsIHRoaXMuZGVzY3JpcHRpb24pO1xuICAgIH1cbiAgICBzc21EZWYuc2V0KCduYW1lJywgdGhpcy5uYW1lKTtcbiAgICBzc21EZWYuc2V0KCdhY3Rpb24nLCB0aGlzLmFjdGlvbik7XG4gICAgc3NtRGVmLnNldCgnaW5wdXRzJywgaW5wdXRzKTtcblxuICAgIGNvbnN0IGZvcm1hdHRlZE91dHB1dHMgPSB0aGlzLmxpc3RPdXRwdXRzKCkubWFwKG91dHB1dCA9PiB0aGlzLmZvcm1hdE91dHB1dChvdXRwdXQpKTtcbiAgICBpZiAodGhpcy5saXN0T3V0cHV0cygpLmxlbmd0aCA+IDApIHtcbiAgICAgIHNzbURlZi5zZXQoJ291dHB1dHMnLCBmb3JtYXR0ZWRPdXRwdXRzKTtcbiAgICB9XG4gICAgaWYgKHRoaXMuaXNFbmQpIHtcbiAgICAgIHNzbURlZi5zZXQoJ2lzRW5kJywgdHJ1ZSk7XG4gICAgfVxuICAgIGlmICh0aGlzLm1heEF0dGVtcHRzICE9IEF1dG9tYXRpb25TdGVwLkRFRkFVTFRfTUFYX0FUVEVNUFRTKSB7XG4gICAgICBzc21EZWYuc2V0KCdtYXhBdHRlbXB0cycsIHRoaXMubWF4QXR0ZW1wdHMpO1xuICAgIH1cbiAgICBpZiAodGhpcy50aW1lb3V0U2Vjb25kcyAhPSBBdXRvbWF0aW9uU3RlcC5ERUZBVUxUX1RJTUVPVVQpIHtcbiAgICAgIHNzbURlZi5zZXQoJ3RpbWVvdXRTZWNvbmRzJywgdGhpcy50aW1lb3V0U2Vjb25kcyk7XG4gICAgfVxuICAgIGlmICghKHRoaXMub25DYW5jZWwgaW5zdGFuY2VvZiBBYm9ydCkpIHtcbiAgICAgIHNzbURlZi5zZXQoJ29uQ2FuY2VsJywgdGhpcy5vbkNhbmNlbC50b1NzbVZhbHVlKCkpO1xuICAgIH1cbiAgICBpZiAoISh0aGlzLm9uRmFpbHVyZSBpbnN0YW5jZW9mIEFib3J0KSkge1xuICAgICAgc3NtRGVmLnNldCgnb25GYWlsdXJlJywgdGhpcy5vbkZhaWx1cmUudG9Tc21WYWx1ZSgpKTtcbiAgICB9XG4gICAgaWYgKHRoaXMuZXhwbGljaXROZXh0U3RlcCAhPSB1bmRlZmluZWQpIHtcbiAgICAgIHNzbURlZi5zZXQoJ25leHRTdGVwJywgdGhpcy5leHBsaWNpdE5leHRTdGVwLnN0ZXBOYW1lKTtcbiAgICB9XG4gICAgY29uc3Qgc3NtRGljdDogeyBbbmFtZTogc3RyaW5nXTogYW55IH0gPSB7fTtcbiAgICBzc21EZWYuZm9yRWFjaCgodmFsdWUsIGtleSkgPT4ge1xuICAgICAgc3NtRGljdFtrZXldID0gdmFsdWU7XG4gICAgfSk7XG4gICAgcmV0dXJuIHNzbURpY3Q7XG4gIH1cblxuICAvKipcbiAgICogRm9ybWF0cyB0aGUgcHJvdmlkZWQgb3V0cHV0IGludG8gYW4gb2JqZWN0IHRoYXQgY2FuIGJlIHVzZWQgdG8gZ2VuZXJhdGUgdGhlIFNTTSB5YW1sL2pzb24gcmVwcmVzZW50YXRpb24uXG4gICAqL1xuICBwcml2YXRlIGZvcm1hdE91dHB1dChvdXRwdXQ6IE91dHB1dCk6IHtbbmFtZTogc3RyaW5nXTogc3RyaW5nfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIE5hbWU6IG91dHB1dC5uYW1lLFxuICAgICAgU2VsZWN0b3I6IG91dHB1dC5zZWxlY3RvcixcbiAgICAgIFR5cGU6IG5ldyBEYXRhVHlwZShvdXRwdXQub3V0cHV0VHlwZSkudG9Tc21TdHJpbmcoKSxcbiAgICB9O1xuICB9XG5cbiAgcHVibGljIHZhcmlhYmxlcygpOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIE9iamVjdC5hc3NpZ24oe30sIC4uLnRoaXMubGlzdE91dHB1dHMoKVxuICAgICAgLm1hcChvdXQgPT4gKHsgW291dC5uYW1lXTogbmV3IFN0cmluZ1ZhcmlhYmxlKGAke3RoaXMubmFtZX0uJHtvdXQubmFtZX1gKSB9KSkpO1xuICB9XG59XG4iXX0=