"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiRunCommandHook = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../../simulation/automation-step-simulation");
const string_list_variable_1 = require("../variables/string-list-variable");
const string_map_variable_1 = require("../variables/string-map-variable");
const string_variable_1 = require("../variables/string-variable");
/**
 * RunCommand implementation using AWS API
 */
class ApiRunCommandHook {
    constructor(awsInvoker, sleepHook) {
        this.awsInvoker = awsInvoker;
        this.sleepHook = sleepHook;
        this.props = { awsInvoker, sleepHook };
    }
    execute(props) {
        if (string_list_variable_1.isStringList(props.targets)) {
            if (props.targets.length > ApiRunCommandHook.MaximumMaxResultsValue) {
                throw new Error(`Only up to ${ApiRunCommandHook.MaximumMaxResultsValue} instance IDs can be specified.`);
            }
            console.log('RunCommand: Validating the instances are healthy');
            this.validateInstanceIdsAreHealthy(props.targets);
        }
        console.log(`RunCommand: Executing the document ${props.documentName}`);
        const executionId = this.runCommand(props);
        console.log(`RunCommand: Waiting for ${executionId} to complete`);
        const result = this.waitForCommand(executionId);
        console.log(`RunCommand: ${executionId} finished`);
        return result;
    }
    waitForCommand(executionId) {
        let status = this.getCommandStatus(executionId);
        while (status === null || ApiRunCommandHook.RunningStatuses.includes(status)) {
            console.log(`Command ${executionId} is not complete. Status: ${status}.`);
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
            status = this.getCommandStatus(executionId);
        }
        if (status !== 'Success') {
            throw new Error(`${executionId} ended in status ${status}`);
        }
        const pluginStatus = this.getSinglePluginStatus(executionId);
        return {
            commandId: executionId,
            status: status,
            responseCode: pluginStatus?.ResponseCode,
            output: pluginStatus?.Output,
        };
    }
    getCommandStatus(executionId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'listCommands', {
            service: 'SSM',
            pascalCaseApi: 'ListCommands',
            apiParams: {
                CommandId: executionId,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'commands',
                    selector: '$.Commands',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get command ${executionId} status: ${result.stackTrace}`);
        }
        const commands = result.outputs?.['listCommands.commands'];
        if (commands.length === 0) {
            return null;
        }
        return commands[0].Status;
    }
    /**
       * We only return status code and output if there is exactly one invocation and one plug-in
       */
    getSinglePluginStatus(executionId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'listCommandInvocations', {
            service: 'SSM',
            pascalCaseApi: 'ListCommandInvocations',
            apiParams: {
                CommandId: executionId,
                Details: true,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    name: 'result',
                    selector: '$',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get invocations for ${executionId}: ${result.stackTrace}`);
        }
        const response = result.outputs?.['listCommandInvocations.result'];
        if (!response.CommandInvocations || response.CommandInvocations.length !== 1) {
            return null;
        }
        const plugins = response.CommandInvocations[0].CommandPlugins;
        if (!plugins || plugins.length !== 1) {
            return null;
        }
        return plugins[0];
    }
    runCommand(props) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'sendCommand', {
            service: 'SSM',
            pascalCaseApi: 'SendCommand',
            apiParams: this.getSendCommandProps(props),
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    name: 'commandId',
                    selector: '$.Command.CommandId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to start command: ${result.stackTrace}`);
        }
        return result.outputs?.['sendCommand.commandId'];
    }
    getSendCommandProps(props) {
        const apiParamMap = {
            DocumentName: props.documentName,
            MaxConcurrency: props.maxConcurrency,
            MaxErrors: props.maxErrors,
            DocumentHash: props.documentHash,
            DocumentHashType: props.documentHashType,
            NotificationConfig: props.notificationConfig,
            TimeoutSeconds: props.timeoutSeconds,
            Comment: props.comment,
            OutputS3BucketName: props.outputS3BucketName,
            OutputS3KeyPrefix: props.outputS3KeyPrefix,
            ServiceRoleArn: props.serviceRoleArn,
            CloudWatchOutputConfig: props.cloudWatchOutputConfig,
        };
        const apiParams = {};
        for (const key of Object.keys(apiParamMap)) {
            const value = apiParamMap[key];
            if (value === undefined) {
                continue;
            }
            apiParams[key] = value;
        }
        if (string_list_variable_1.isStringList(props.targets)) {
            apiParams.InstanceIds = props.targets;
        }
        else {
            apiParams.Targets = props.targets;
        }
        if (props.parameters) {
            apiParams.Parameters = this.adaptRunCommandParameters(props.parameters);
        }
        return apiParams;
    }
    adaptRunCommandParameters(parameters) {
        const result = {};
        for (const key of Object.keys(parameters)) {
            const value = parameters[key];
            if (string_variable_1.isString(value)) {
                result[key] = [value];
                continue;
            }
            if (Array.isArray(value)) {
                if (!string_list_variable_1.isStringList(value)) {
                    throw new Error(`Parameter ${key} is a list but not a StringList. Only StringList is supported.`);
                }
                result[key] = value;
                continue;
            }
            if (string_map_variable_1.isStringMap(value)) {
                result[key] = [JSON.stringify(value)];
                continue;
            }
            throw new Error(`Parameter ${key} must be a String, StringList, or StringMap.`);
        }
        return result;
    }
    validateInstanceIdsAreHealthy(instanceIds) {
        const pingStatus = this.getPingStatusOfInstances(instanceIds);
        if (pingStatus.length !== instanceIds.length || !pingStatus.every(x => x === ApiRunCommandHook.SsmPingOnlineStatusValue)) {
            throw new Error('Not every instance has a healthy SSM agent');
        }
    }
    getPingStatusOfInstances(instanceIds) {
        const apiParams = {
            MaxResults: ApiRunCommandHook.MaximumMaxResultsValue,
            InstanceInformationFilterList: [{
                    key: ApiRunCommandHook.SsmInstanceIdsFilterKey,
                    valueSet: instanceIds,
                }],
        };
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeInstanceInfo', {
            service: 'SSM',
            pascalCaseApi: 'DescribeInstanceInformation',
            apiParams: apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'instanceInformation',
                    selector: '$.InstanceInformationList',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to load instance information: ${result.stackTrace}`);
        }
        return result.outputs?.['describeInstanceInfo.instanceInformation']?.map(x => x.PingStatus);
    }
}
exports.ApiRunCommandHook = ApiRunCommandHook;
_a = JSII_RTTI_SYMBOL_1;
ApiRunCommandHook[_a] = { fqn: "@cdklabs/cdk-ssm-documents.ApiRunCommandHook", version: "0.0.25" };
ApiRunCommandHook.MaximumMaxResultsValue = 50; // Only up to 50 instance IDs are allowed for aws:runCommand
ApiRunCommandHook.SsmInstanceIdsFilterKey = 'InstanceIds';
ApiRunCommandHook.SsmPingOnlineStatusValue = 'Online';
ApiRunCommandHook.RunningStatuses = ['Pending', 'InProgress', 'Cancelling'];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLXJ1bi1jb21tYW5kLWhvb2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvaW50ZXJmYWNlL3J1bi1jb21tYW5kLWhvb2svYXBpLXJ1bi1jb21tYW5kLWhvb2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FBb0M7QUFDcEMsc0RBQXNEO0FBQ3RELDhEQUEwRDtBQUMxRCw2RUFBd0U7QUFDeEUseUVBQXFFO0FBQ3JFLDRGQUF1RjtBQUl2Riw0RUFBaUU7QUFDakUsMEVBQStEO0FBQy9ELGtFQUF3RDtBQTRCeEQ7O0dBRUc7QUFDSCxNQUFhLGlCQUFpQjtJQVU1QixZQUFZLFVBQXVCLEVBQUUsU0FBcUI7UUFDeEQsSUFBSSxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7UUFDN0IsSUFBSSxDQUFDLFNBQVMsR0FBRyxTQUFTLENBQUM7UUFDM0IsSUFBSSxDQUFDLEtBQUssR0FBRyxFQUFFLFVBQVUsRUFBRSxTQUFTLEVBQUUsQ0FBQztJQUN6QyxDQUFDO0lBRUQsT0FBTyxDQUFDLEtBQXNCO1FBQzVCLElBQUksbUNBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDL0IsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQyxzQkFBc0IsRUFBRTtnQkFDbkUsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLGlCQUFpQixDQUFDLHNCQUFzQixpQ0FBaUMsQ0FBQyxDQUFDO2FBQzFHO1lBQ0QsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrREFBa0QsQ0FBQyxDQUFDO1lBQ2hFLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDbkQ7UUFFRCxPQUFPLENBQUMsR0FBRyxDQUFDLHNDQUFzQyxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUN4RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNDLE9BQU8sQ0FBQyxHQUFHLENBQUMsMkJBQTJCLFdBQVcsY0FBYyxDQUFDLENBQUM7UUFDbEUsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNoRCxPQUFPLENBQUMsR0FBRyxDQUFDLGVBQWUsV0FBVyxXQUFXLENBQUMsQ0FBQztRQUVuRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU8sY0FBYyxDQUFDLFdBQW1CO1FBQ3hDLElBQUksTUFBTSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNoRCxPQUFPLE1BQU0sS0FBSyxJQUFJLElBQUksaUJBQWlCLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUM1RSxPQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsV0FBVyw2QkFBNkIsTUFBTSxHQUFHLENBQUMsQ0FBQztZQUMxRSxJQUFJLHFEQUF3QixDQUFDLElBQUksc0JBQVMsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxPQUFPLEVBQUU7Z0JBQy9ELFlBQVksRUFBRSxDQUFDO2FBQ2hCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzNCLE1BQU0sR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDN0M7UUFDRCxJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLFdBQVcsb0JBQW9CLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDN0Q7UUFDRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFN0QsT0FBTztZQUNMLFNBQVMsRUFBRSxXQUFXO1lBQ3RCLE1BQU0sRUFBRSxNQUFNO1lBQ2QsWUFBWSxFQUFFLFlBQVksRUFBRSxZQUFZO1lBQ3hDLE1BQU0sRUFBRSxZQUFZLEVBQUUsTUFBTTtTQUM3QixDQUFDO0lBQ0osQ0FBQztJQUVPLGdCQUFnQixDQUFDLFdBQW1CO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUkscURBQXdCLENBQUMsSUFBSSx5QkFBVSxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLGNBQWMsRUFBRTtZQUN0RixPQUFPLEVBQUUsS0FBSztZQUNkLGFBQWEsRUFBRSxjQUFjO1lBQzdCLFNBQVMsRUFBRTtnQkFDVCxTQUFTLEVBQUUsV0FBVzthQUN2QjtZQUNELE9BQU8sRUFBRSxDQUFDO29CQUNSLFVBQVUsRUFBRSx3QkFBWSxDQUFDLFFBQVE7b0JBQ2pDLElBQUksRUFBRSxVQUFVO29CQUNoQixRQUFRLEVBQUUsWUFBWTtpQkFDdkIsQ0FBQztTQUNILENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzNCLElBQUksTUFBTSxDQUFDLFlBQVksS0FBSyw0QkFBWSxDQUFDLE9BQU8sRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixXQUFXLFlBQVksTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDdEY7UUFDRCxNQUFNLFFBQVEsR0FBb0IsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDNUUsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUFFLE9BQU8sSUFBSSxDQUFDO1NBQUU7UUFDM0MsT0FBTyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO0lBQzVCLENBQUM7SUFFRDs7U0FFSztJQUNHLHFCQUFxQixDQUFDLFdBQW1CO1FBQy9DLE1BQU0sTUFBTSxHQUFHLElBQUkscURBQXdCLENBQUMsSUFBSSx5QkFBVSxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLHdCQUF3QixFQUFFO1lBQ2hHLE9BQU8sRUFBRSxLQUFLO1lBQ2QsYUFBYSxFQUFFLHdCQUF3QjtZQUN2QyxTQUFTLEVBQUU7Z0JBQ1QsU0FBUyxFQUFFLFdBQVc7Z0JBQ3RCLE9BQU8sRUFBRSxJQUFJO2FBQ2Q7WUFDRCxPQUFPLEVBQUUsQ0FBQztvQkFDUixVQUFVLEVBQUUsd0JBQVksQ0FBQyxVQUFVO29CQUNuQyxJQUFJLEVBQUUsUUFBUTtvQkFDZCxRQUFRLEVBQUUsR0FBRztpQkFDZCxDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLDRCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLFdBQVcsS0FBSyxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUN2RjtRQUNELE1BQU0sUUFBUSxHQUFpQyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUNqRyxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQUUsT0FBTyxJQUFJLENBQUM7U0FBRTtRQUM5RixNQUFNLE9BQU8sR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDO1FBQzlELElBQUksQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQztTQUFFO1FBQ3RELE9BQU8sT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3BCLENBQUM7SUFFTyxVQUFVLENBQUMsS0FBc0I7UUFDdkMsTUFBTSxNQUFNLEdBQUcsSUFBSSxxREFBd0IsQ0FBQyxJQUFJLHlCQUFVLENBQUMsSUFBSSxtQkFBSyxFQUFFLEVBQUUsYUFBYSxFQUFFO1lBQ3JGLE9BQU8sRUFBRSxLQUFLO1lBQ2QsYUFBYSxFQUFFLGFBQWE7WUFDNUIsU0FBUyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUM7WUFDMUMsT0FBTyxFQUFFLENBQUM7b0JBQ1IsVUFBVSxFQUFFLHdCQUFZLENBQUMsTUFBTTtvQkFDL0IsSUFBSSxFQUFFLFdBQVc7b0JBQ2pCLFFBQVEsRUFBRSxxQkFBcUI7aUJBQ2hDLENBQUM7U0FDSCxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUMzQixJQUFJLE1BQU0sQ0FBQyxZQUFZLEtBQUssNEJBQVksQ0FBQyxPQUFPLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0QkFBNEIsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDbEU7UUFDRCxPQUFPLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFTyxtQkFBbUIsQ0FBQyxLQUFzQjtRQUNoRCxNQUFNLFdBQVcsR0FBd0I7WUFDdkMsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDMUIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM1QyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7WUFDNUMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUNyRCxDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQXdCLEVBQUUsQ0FBQztRQUMxQyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDMUMsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9CLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtnQkFBRSxTQUFTO2FBQUU7WUFDdEMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztTQUN4QjtRQUVELElBQUksbUNBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDL0IsU0FBUyxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1NBQ3ZDO2FBQU07WUFDTCxTQUFTLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7U0FDbkM7UUFDRCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsU0FBUyxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3pFO1FBRUQsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVPLHlCQUF5QixDQUFDLFVBQStCO1FBQy9ELE1BQU0sTUFBTSxHQUE2QixFQUFFLENBQUM7UUFDNUMsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ3pDLE1BQU0sS0FBSyxHQUFHLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUM5QixJQUFJLDBCQUFRLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ25CLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUN0QixTQUFTO2FBQ1Y7WUFDRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3hCLElBQUksQ0FBQyxtQ0FBWSxDQUFDLEtBQUssQ0FBQyxFQUFFO29CQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLGFBQWEsR0FBRyxnRUFBZ0UsQ0FBQyxDQUFDO2lCQUNuRztnQkFDRCxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO2dCQUNwQixTQUFTO2FBQ1Y7WUFDRCxJQUFJLGlDQUFXLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3RCLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztnQkFDdEMsU0FBUzthQUNWO1lBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxhQUFhLEdBQUcsOENBQThDLENBQUMsQ0FBQztTQUNqRjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFTyw2QkFBNkIsQ0FBQyxXQUFxQjtRQUN6RCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDOUQsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLFdBQVcsQ0FBQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLGlCQUFpQixDQUFDLHdCQUF3QixDQUFDLEVBQUU7WUFDeEgsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1NBQy9EO0lBQ0gsQ0FBQztJQUVPLHdCQUF3QixDQUFDLFdBQXFCO1FBQ3BELE1BQU0sU0FBUyxHQUF3QjtZQUNyQyxVQUFVLEVBQUUsaUJBQWlCLENBQUMsc0JBQXNCO1lBQ3BELDZCQUE2QixFQUFFLENBQUM7b0JBQzlCLEdBQUcsRUFBRSxpQkFBaUIsQ0FBQyx1QkFBdUI7b0JBQzlDLFFBQVEsRUFBRSxXQUFXO2lCQUN0QixDQUFDO1NBQ0gsQ0FBQztRQUVGLE1BQU0sTUFBTSxHQUFHLElBQUkscURBQXdCLENBQUMsSUFBSSx5QkFBVSxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLHNCQUFzQixFQUFFO1lBQzlGLE9BQU8sRUFBRSxLQUFLO1lBQ2QsYUFBYSxFQUFFLDZCQUE2QjtZQUM1QyxTQUFTLEVBQUUsU0FBUztZQUNwQixPQUFPLEVBQUUsQ0FBQztvQkFDUixVQUFVLEVBQUUsd0JBQVksQ0FBQyxRQUFRO29CQUNqQyxJQUFJLEVBQUUscUJBQXFCO29CQUMzQixRQUFRLEVBQUUsMkJBQTJCO2lCQUN0QyxDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLDRCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQzlFO1FBQ0QsT0FBUSxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUMsMENBQTBDLENBQXlDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3ZJLENBQUM7O0FBak5ILDhDQWtOQzs7O0FBak55Qix3Q0FBc0IsR0FBRyxFQUFFLENBQUMsQ0FBQyw0REFBNEQ7QUFDekYseUNBQXVCLEdBQUcsYUFBYSxDQUFDO0FBQ3hDLDBDQUF3QixHQUFHLFFBQVEsQ0FBQztBQUNwQyxpQ0FBZSxHQUFvQixDQUFDLFNBQVMsRUFBRSxZQUFZLEVBQUUsWUFBWSxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IERhdGFUeXBlRW51bSB9IGZyb20gJy4uLy4uL2RvbWFpbi9kYXRhLXR5cGUnO1xuaW1wb3J0IHsgUmVzcG9uc2VDb2RlIH0gZnJvbSAnLi4vLi4vZG9tYWluL3Jlc3BvbnNlLWNvZGUnO1xuaW1wb3J0IHsgQXdzQXBpU3RlcCB9IGZyb20gJy4uLy4uL3BhcmVudC1zdGVwcy9hdXRvbWF0aW9uL2F3cy1hcGktc3RlcCc7XG5pbXBvcnQgeyBTbGVlcFN0ZXAgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9zbGVlcC1zdGVwJztcbmltcG9ydCB7IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbiB9IGZyb20gJy4uLy4uL3NpbXVsYXRpb24vYXV0b21hdGlvbi1zdGVwLXNpbXVsYXRpb24nO1xuaW1wb3J0IHsgSUF3c0ludm9rZXIgfSBmcm9tICcuLi9hd3MtaW52b2tlcic7XG5pbXBvcnQgeyBJUnVuQ29tbWFuZEhvb2ssIFJ1bkNvbW1hbmRPdXRwdXRzLCBSdW5Db21tYW5kUHJvcHMgfSBmcm9tICcuLi9ydW4tY29tbWFuZC1ob29rJztcbmltcG9ydCB7IElTbGVlcEhvb2sgfSBmcm9tICcuLi9zbGVlcC1ob29rJztcbmltcG9ydCB7IGlzU3RyaW5nTGlzdCB9IGZyb20gJy4uL3ZhcmlhYmxlcy9zdHJpbmctbGlzdC12YXJpYWJsZSc7XG5pbXBvcnQgeyBpc1N0cmluZ01hcCB9IGZyb20gJy4uL3ZhcmlhYmxlcy9zdHJpbmctbWFwLXZhcmlhYmxlJztcbmltcG9ydCB7IGlzU3RyaW5nIH0gZnJvbSAnLi4vdmFyaWFibGVzL3N0cmluZy12YXJpYWJsZSc7XG5cbmludGVyZmFjZSBEZXNjcmliZUluc3RhbmNlSW5mb3JtYXRpb25SZXN1bHQge1xuICBQaW5nU3RhdHVzOiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBDb21tYW5kUmVzdWx0IHtcbiAgU3RhdHVzOiBDb21tYW5kU3RhdHVzO1xufVxuXG50eXBlIENvbW1hbmRTdGF0dXMgPSAnUGVuZGluZycgfCAnSW5Qcm9ncmVzcycgfCAnU3VjY2VzcycgfCAnQ2FuY2VsbGVkJyB8ICdGYWlsZWQnIHwgJ1RpbWVkT3V0JyB8ICdDYW5jZWxsaW5nJztcblxuaW50ZXJmYWNlIExpc3RDb21tYW5kSW52b2NhdGlvbnNSZXN1bHQge1xuICBDb21tYW5kSW52b2NhdGlvbnM/OiB7XG4gICAgQ29tbWFuZFBsdWdpbnM/OiBDb21tYW5kUGx1Z2luUmVzdWx0W107XG4gIH1bXTtcbn1cblxuaW50ZXJmYWNlIENvbW1hbmRQbHVnaW5SZXN1bHQge1xuICBSZXNwb25zZUNvZGU6IG51bWJlcjtcbiAgT3V0cHV0OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBpUnVuQ29tbWFuZFByb3BzIHtcbiAgcmVhZG9ubHkgYXdzSW52b2tlcjogSUF3c0ludm9rZXI7XG4gIHJlYWRvbmx5IHNsZWVwSG9vazogSVNsZWVwSG9vaztcbn1cblxuLyoqXG4gKiBSdW5Db21tYW5kIGltcGxlbWVudGF0aW9uIHVzaW5nIEFXUyBBUElcbiAqL1xuZXhwb3J0IGNsYXNzIEFwaVJ1bkNvbW1hbmRIb29rIGltcGxlbWVudHMgSVJ1bkNvbW1hbmRIb29rIHtcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgTWF4aW11bU1heFJlc3VsdHNWYWx1ZSA9IDUwOyAvLyBPbmx5IHVwIHRvIDUwIGluc3RhbmNlIElEcyBhcmUgYWxsb3dlZCBmb3IgYXdzOnJ1bkNvbW1hbmRcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU3NtSW5zdGFuY2VJZHNGaWx0ZXJLZXkgPSAnSW5zdGFuY2VJZHMnO1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTc21QaW5nT25saW5lU3RhdHVzVmFsdWUgPSAnT25saW5lJztcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgUnVubmluZ1N0YXR1c2VzOiBDb21tYW5kU3RhdHVzW10gPSBbJ1BlbmRpbmcnLCAnSW5Qcm9ncmVzcycsICdDYW5jZWxsaW5nJ107XG5cbiAgcmVhZG9ubHkgYXdzSW52b2tlcjogSUF3c0ludm9rZXI7XG4gIHJlYWRvbmx5IHNsZWVwSG9vazogSVNsZWVwSG9vaztcbiAgcmVhZG9ubHkgcHJvcHM6IEFwaVJ1bkNvbW1hbmRQcm9wcztcblxuICBjb25zdHJ1Y3Rvcihhd3NJbnZva2VyOiBJQXdzSW52b2tlciwgc2xlZXBIb29rOiBJU2xlZXBIb29rKSB7XG4gICAgdGhpcy5hd3NJbnZva2VyID0gYXdzSW52b2tlcjtcbiAgICB0aGlzLnNsZWVwSG9vayA9IHNsZWVwSG9vaztcbiAgICB0aGlzLnByb3BzID0geyBhd3NJbnZva2VyLCBzbGVlcEhvb2sgfTtcbiAgfVxuXG4gIGV4ZWN1dGUocHJvcHM6IFJ1bkNvbW1hbmRQcm9wcyk6IFJ1bkNvbW1hbmRPdXRwdXRzIHtcbiAgICBpZiAoaXNTdHJpbmdMaXN0KHByb3BzLnRhcmdldHMpKSB7XG4gICAgICBpZiAocHJvcHMudGFyZ2V0cy5sZW5ndGggPiBBcGlSdW5Db21tYW5kSG9vay5NYXhpbXVtTWF4UmVzdWx0c1ZhbHVlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgT25seSB1cCB0byAke0FwaVJ1bkNvbW1hbmRIb29rLk1heGltdW1NYXhSZXN1bHRzVmFsdWV9IGluc3RhbmNlIElEcyBjYW4gYmUgc3BlY2lmaWVkLmApO1xuICAgICAgfVxuICAgICAgY29uc29sZS5sb2coJ1J1bkNvbW1hbmQ6IFZhbGlkYXRpbmcgdGhlIGluc3RhbmNlcyBhcmUgaGVhbHRoeScpO1xuICAgICAgdGhpcy52YWxpZGF0ZUluc3RhbmNlSWRzQXJlSGVhbHRoeShwcm9wcy50YXJnZXRzKTtcbiAgICB9XG5cbiAgICBjb25zb2xlLmxvZyhgUnVuQ29tbWFuZDogRXhlY3V0aW5nIHRoZSBkb2N1bWVudCAke3Byb3BzLmRvY3VtZW50TmFtZX1gKTtcbiAgICBjb25zdCBleGVjdXRpb25JZCA9IHRoaXMucnVuQ29tbWFuZChwcm9wcyk7XG4gICAgY29uc29sZS5sb2coYFJ1bkNvbW1hbmQ6IFdhaXRpbmcgZm9yICR7ZXhlY3V0aW9uSWR9IHRvIGNvbXBsZXRlYCk7XG4gICAgY29uc3QgcmVzdWx0ID0gdGhpcy53YWl0Rm9yQ29tbWFuZChleGVjdXRpb25JZCk7XG4gICAgY29uc29sZS5sb2coYFJ1bkNvbW1hbmQ6ICR7ZXhlY3V0aW9uSWR9IGZpbmlzaGVkYCk7XG5cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgcHJpdmF0ZSB3YWl0Rm9yQ29tbWFuZChleGVjdXRpb25JZDogc3RyaW5nKTogUnVuQ29tbWFuZE91dHB1dHMge1xuICAgIGxldCBzdGF0dXMgPSB0aGlzLmdldENvbW1hbmRTdGF0dXMoZXhlY3V0aW9uSWQpO1xuICAgIHdoaWxlIChzdGF0dXMgPT09IG51bGwgfHwgQXBpUnVuQ29tbWFuZEhvb2suUnVubmluZ1N0YXR1c2VzLmluY2x1ZGVzKHN0YXR1cykpIHtcbiAgICAgIGNvbnNvbGUubG9nKGBDb21tYW5kICR7ZXhlY3V0aW9uSWR9IGlzIG5vdCBjb21wbGV0ZS4gU3RhdHVzOiAke3N0YXR1c30uYCk7XG4gICAgICBuZXcgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uKG5ldyBTbGVlcFN0ZXAobmV3IFN0YWNrKCksICdzbGVlcCcsIHtcbiAgICAgICAgc2xlZXBTZWNvbmRzOiAyLFxuICAgICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgICBzdGF0dXMgPSB0aGlzLmdldENvbW1hbmRTdGF0dXMoZXhlY3V0aW9uSWQpO1xuICAgIH1cbiAgICBpZiAoc3RhdHVzICE9PSAnU3VjY2VzcycpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtleGVjdXRpb25JZH0gZW5kZWQgaW4gc3RhdHVzICR7c3RhdHVzfWApO1xuICAgIH1cbiAgICBjb25zdCBwbHVnaW5TdGF0dXMgPSB0aGlzLmdldFNpbmdsZVBsdWdpblN0YXR1cyhleGVjdXRpb25JZCk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29tbWFuZElkOiBleGVjdXRpb25JZCxcbiAgICAgIHN0YXR1czogc3RhdHVzLFxuICAgICAgcmVzcG9uc2VDb2RlOiBwbHVnaW5TdGF0dXM/LlJlc3BvbnNlQ29kZSxcbiAgICAgIG91dHB1dDogcGx1Z2luU3RhdHVzPy5PdXRwdXQsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0Q29tbWFuZFN0YXR1cyhleGVjdXRpb25JZDogc3RyaW5nKTogQ29tbWFuZFN0YXR1cyB8IG51bGwge1xuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IEF3c0FwaVN0ZXAobmV3IFN0YWNrKCksICdsaXN0Q29tbWFuZHMnLCB7XG4gICAgICBzZXJ2aWNlOiAnU1NNJyxcbiAgICAgIHBhc2NhbENhc2VBcGk6ICdMaXN0Q29tbWFuZHMnLFxuICAgICAgYXBpUGFyYW1zOiB7XG4gICAgICAgIENvbW1hbmRJZDogZXhlY3V0aW9uSWQsXG4gICAgICB9LFxuICAgICAgb3V0cHV0czogW3tcbiAgICAgICAgb3V0cHV0VHlwZTogRGF0YVR5cGVFbnVtLk1BUF9MSVNULFxuICAgICAgICBuYW1lOiAnY29tbWFuZHMnLFxuICAgICAgICBzZWxlY3RvcjogJyQuQ29tbWFuZHMnLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZhaWxlZCB0byBnZXQgY29tbWFuZCAke2V4ZWN1dGlvbklkfSBzdGF0dXM6ICR7cmVzdWx0LnN0YWNrVHJhY2V9YCk7XG4gICAgfVxuICAgIGNvbnN0IGNvbW1hbmRzOiBDb21tYW5kUmVzdWx0W10gPSByZXN1bHQub3V0cHV0cz8uWydsaXN0Q29tbWFuZHMuY29tbWFuZHMnXTtcbiAgICBpZiAoY29tbWFuZHMubGVuZ3RoID09PSAwKSB7IHJldHVybiBudWxsOyB9XG4gICAgcmV0dXJuIGNvbW1hbmRzWzBdLlN0YXR1cztcbiAgfVxuXG4gIC8qKlxuICAgICAqIFdlIG9ubHkgcmV0dXJuIHN0YXR1cyBjb2RlIGFuZCBvdXRwdXQgaWYgdGhlcmUgaXMgZXhhY3RseSBvbmUgaW52b2NhdGlvbiBhbmQgb25lIHBsdWctaW5cbiAgICAgKi9cbiAgcHJpdmF0ZSBnZXRTaW5nbGVQbHVnaW5TdGF0dXMoZXhlY3V0aW9uSWQ6IHN0cmluZyk6IENvbW1hbmRQbHVnaW5SZXN1bHQgfCBudWxsIHtcbiAgICBjb25zdCByZXN1bHQgPSBuZXcgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uKG5ldyBBd3NBcGlTdGVwKG5ldyBTdGFjaygpLCAnbGlzdENvbW1hbmRJbnZvY2F0aW9ucycsIHtcbiAgICAgIHNlcnZpY2U6ICdTU00nLFxuICAgICAgcGFzY2FsQ2FzZUFwaTogJ0xpc3RDb21tYW5kSW52b2NhdGlvbnMnLFxuICAgICAgYXBpUGFyYW1zOiB7XG4gICAgICAgIENvbW1hbmRJZDogZXhlY3V0aW9uSWQsXG4gICAgICAgIERldGFpbHM6IHRydWUsXG4gICAgICB9LFxuICAgICAgb3V0cHV0czogW3tcbiAgICAgICAgb3V0cHV0VHlwZTogRGF0YVR5cGVFbnVtLlNUUklOR19NQVAsXG4gICAgICAgIG5hbWU6ICdyZXN1bHQnLFxuICAgICAgICBzZWxlY3RvcjogJyQnLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZhaWxlZCB0byBnZXQgaW52b2NhdGlvbnMgZm9yICR7ZXhlY3V0aW9uSWR9OiAke3Jlc3VsdC5zdGFja1RyYWNlfWApO1xuICAgIH1cbiAgICBjb25zdCByZXNwb25zZTogTGlzdENvbW1hbmRJbnZvY2F0aW9uc1Jlc3VsdCA9IHJlc3VsdC5vdXRwdXRzPy5bJ2xpc3RDb21tYW5kSW52b2NhdGlvbnMucmVzdWx0J107XG4gICAgaWYgKCFyZXNwb25zZS5Db21tYW5kSW52b2NhdGlvbnMgfHwgcmVzcG9uc2UuQ29tbWFuZEludm9jYXRpb25zLmxlbmd0aCAhPT0gMSkgeyByZXR1cm4gbnVsbDsgfVxuICAgIGNvbnN0IHBsdWdpbnMgPSByZXNwb25zZS5Db21tYW5kSW52b2NhdGlvbnNbMF0uQ29tbWFuZFBsdWdpbnM7XG4gICAgaWYgKCFwbHVnaW5zIHx8IHBsdWdpbnMubGVuZ3RoICE9PSAxKSB7IHJldHVybiBudWxsOyB9XG4gICAgcmV0dXJuIHBsdWdpbnNbMF07XG4gIH1cblxuICBwcml2YXRlIHJ1bkNvbW1hbmQocHJvcHM6IFJ1bkNvbW1hbmRQcm9wcyk6IHN0cmluZyB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ3NlbmRDb21tYW5kJywge1xuICAgICAgc2VydmljZTogJ1NTTScsXG4gICAgICBwYXNjYWxDYXNlQXBpOiAnU2VuZENvbW1hbmQnLFxuICAgICAgYXBpUGFyYW1zOiB0aGlzLmdldFNlbmRDb21tYW5kUHJvcHMocHJvcHMpLFxuICAgICAgb3V0cHV0czogW3tcbiAgICAgICAgb3V0cHV0VHlwZTogRGF0YVR5cGVFbnVtLlNUUklORyxcbiAgICAgICAgbmFtZTogJ2NvbW1hbmRJZCcsXG4gICAgICAgIHNlbGVjdG9yOiAnJC5Db21tYW5kLkNvbW1hbmRJZCcsXG4gICAgICB9XSxcbiAgICB9KSwgdGhpcy5wcm9wcykuaW52b2tlKHt9KTtcbiAgICBpZiAocmVzdWx0LnJlc3BvbnNlQ29kZSAhPT0gUmVzcG9uc2VDb2RlLlNVQ0NFU1MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRmFpbGVkIHRvIHN0YXJ0IGNvbW1hbmQ6ICR7cmVzdWx0LnN0YWNrVHJhY2V9YCk7XG4gICAgfVxuICAgIHJldHVybiByZXN1bHQub3V0cHV0cz8uWydzZW5kQ29tbWFuZC5jb21tYW5kSWQnXTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0U2VuZENvbW1hbmRQcm9wcyhwcm9wczogUnVuQ29tbWFuZFByb3BzKTogUmVjb3JkPHN0cmluZywgYW55PiB7XG4gICAgY29uc3QgYXBpUGFyYW1NYXA6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7XG4gICAgICBEb2N1bWVudE5hbWU6IHByb3BzLmRvY3VtZW50TmFtZSxcbiAgICAgIE1heENvbmN1cnJlbmN5OiBwcm9wcy5tYXhDb25jdXJyZW5jeSxcbiAgICAgIE1heEVycm9yczogcHJvcHMubWF4RXJyb3JzLFxuICAgICAgRG9jdW1lbnRIYXNoOiBwcm9wcy5kb2N1bWVudEhhc2gsXG4gICAgICBEb2N1bWVudEhhc2hUeXBlOiBwcm9wcy5kb2N1bWVudEhhc2hUeXBlLFxuICAgICAgTm90aWZpY2F0aW9uQ29uZmlnOiBwcm9wcy5ub3RpZmljYXRpb25Db25maWcsXG4gICAgICBUaW1lb3V0U2Vjb25kczogcHJvcHMudGltZW91dFNlY29uZHMsXG4gICAgICBDb21tZW50OiBwcm9wcy5jb21tZW50LFxuICAgICAgT3V0cHV0UzNCdWNrZXROYW1lOiBwcm9wcy5vdXRwdXRTM0J1Y2tldE5hbWUsXG4gICAgICBPdXRwdXRTM0tleVByZWZpeDogcHJvcHMub3V0cHV0UzNLZXlQcmVmaXgsXG4gICAgICBTZXJ2aWNlUm9sZUFybjogcHJvcHMuc2VydmljZVJvbGVBcm4sXG4gICAgICBDbG91ZFdhdGNoT3V0cHV0Q29uZmlnOiBwcm9wcy5jbG91ZFdhdGNoT3V0cHV0Q29uZmlnLFxuICAgIH07XG4gICAgY29uc3QgYXBpUGFyYW1zOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG4gICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMoYXBpUGFyYW1NYXApKSB7XG4gICAgICBjb25zdCB2YWx1ZSA9IGFwaVBhcmFtTWFwW2tleV07XG4gICAgICBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkgeyBjb250aW51ZTsgfVxuICAgICAgYXBpUGFyYW1zW2tleV0gPSB2YWx1ZTtcbiAgICB9XG5cbiAgICBpZiAoaXNTdHJpbmdMaXN0KHByb3BzLnRhcmdldHMpKSB7XG4gICAgICBhcGlQYXJhbXMuSW5zdGFuY2VJZHMgPSBwcm9wcy50YXJnZXRzO1xuICAgIH0gZWxzZSB7XG4gICAgICBhcGlQYXJhbXMuVGFyZ2V0cyA9IHByb3BzLnRhcmdldHM7XG4gICAgfVxuICAgIGlmIChwcm9wcy5wYXJhbWV0ZXJzKSB7XG4gICAgICBhcGlQYXJhbXMuUGFyYW1ldGVycyA9IHRoaXMuYWRhcHRSdW5Db21tYW5kUGFyYW1ldGVycyhwcm9wcy5wYXJhbWV0ZXJzKTtcbiAgICB9XG5cbiAgICByZXR1cm4gYXBpUGFyYW1zO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGFwdFJ1bkNvbW1hbmRQYXJhbWV0ZXJzKHBhcmFtZXRlcnM6IFJlY29yZDxzdHJpbmcsIGFueT4pOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT4ge1xuICAgIGNvbnN0IHJlc3VsdDogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+ID0ge307XG4gICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMocGFyYW1ldGVycykpIHtcbiAgICAgIGNvbnN0IHZhbHVlID0gcGFyYW1ldGVyc1trZXldO1xuICAgICAgaWYgKGlzU3RyaW5nKHZhbHVlKSkge1xuICAgICAgICByZXN1bHRba2V5XSA9IFt2YWx1ZV07XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgaWYgKEFycmF5LmlzQXJyYXkodmFsdWUpKSB7XG4gICAgICAgIGlmICghaXNTdHJpbmdMaXN0KHZhbHVlKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUGFyYW1ldGVyICR7a2V5fSBpcyBhIGxpc3QgYnV0IG5vdCBhIFN0cmluZ0xpc3QuIE9ubHkgU3RyaW5nTGlzdCBpcyBzdXBwb3J0ZWQuYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmVzdWx0W2tleV0gPSB2YWx1ZTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG4gICAgICBpZiAoaXNTdHJpbmdNYXAodmFsdWUpKSB7XG4gICAgICAgIHJlc3VsdFtrZXldID0gW0pTT04uc3RyaW5naWZ5KHZhbHVlKV07XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgdGhyb3cgbmV3IEVycm9yKGBQYXJhbWV0ZXIgJHtrZXl9IG11c3QgYmUgYSBTdHJpbmcsIFN0cmluZ0xpc3QsIG9yIFN0cmluZ01hcC5gKTtcbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVJbnN0YW5jZUlkc0FyZUhlYWx0aHkoaW5zdGFuY2VJZHM6IHN0cmluZ1tdKTogdm9pZCB7XG4gICAgY29uc3QgcGluZ1N0YXR1cyA9IHRoaXMuZ2V0UGluZ1N0YXR1c09mSW5zdGFuY2VzKGluc3RhbmNlSWRzKTtcbiAgICBpZiAocGluZ1N0YXR1cy5sZW5ndGggIT09IGluc3RhbmNlSWRzLmxlbmd0aCB8fCAhcGluZ1N0YXR1cy5ldmVyeSh4ID0+IHggPT09IEFwaVJ1bkNvbW1hbmRIb29rLlNzbVBpbmdPbmxpbmVTdGF0dXNWYWx1ZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTm90IGV2ZXJ5IGluc3RhbmNlIGhhcyBhIGhlYWx0aHkgU1NNIGFnZW50Jyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBnZXRQaW5nU3RhdHVzT2ZJbnN0YW5jZXMoaW5zdGFuY2VJZHM6IHN0cmluZ1tdKTogc3RyaW5nW10ge1xuICAgIGNvbnN0IGFwaVBhcmFtczogUmVjb3JkPHN0cmluZywgYW55PiA9IHtcbiAgICAgIE1heFJlc3VsdHM6IEFwaVJ1bkNvbW1hbmRIb29rLk1heGltdW1NYXhSZXN1bHRzVmFsdWUsXG4gICAgICBJbnN0YW5jZUluZm9ybWF0aW9uRmlsdGVyTGlzdDogW3tcbiAgICAgICAga2V5OiBBcGlSdW5Db21tYW5kSG9vay5Tc21JbnN0YW5jZUlkc0ZpbHRlcktleSxcbiAgICAgICAgdmFsdWVTZXQ6IGluc3RhbmNlSWRzLFxuICAgICAgfV0sXG4gICAgfTtcblxuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IEF3c0FwaVN0ZXAobmV3IFN0YWNrKCksICdkZXNjcmliZUluc3RhbmNlSW5mbycsIHtcbiAgICAgIHNlcnZpY2U6ICdTU00nLFxuICAgICAgcGFzY2FsQ2FzZUFwaTogJ0Rlc2NyaWJlSW5zdGFuY2VJbmZvcm1hdGlvbicsXG4gICAgICBhcGlQYXJhbXM6IGFwaVBhcmFtcyxcbiAgICAgIG91dHB1dHM6IFt7XG4gICAgICAgIG91dHB1dFR5cGU6IERhdGFUeXBlRW51bS5NQVBfTElTVCxcbiAgICAgICAgbmFtZTogJ2luc3RhbmNlSW5mb3JtYXRpb24nLFxuICAgICAgICBzZWxlY3RvcjogJyQuSW5zdGFuY2VJbmZvcm1hdGlvbkxpc3QnLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZhaWxlZCB0byBsb2FkIGluc3RhbmNlIGluZm9ybWF0aW9uOiAke3Jlc3VsdC5zdGFja1RyYWNlfWApO1xuICAgIH1cbiAgICByZXR1cm4gKHJlc3VsdC5vdXRwdXRzPy5bJ2Rlc2NyaWJlSW5zdGFuY2VJbmZvLmluc3RhbmNlSW5mb3JtYXRpb24nXSBhcyBEZXNjcmliZUluc3RhbmNlSW5mb3JtYXRpb25SZXN1bHRbXSk/Lm1hcCh4ID0+IHguUGluZ1N0YXR1cyk7XG4gIH1cbn0iXX0=