"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabContainerRunner = exports.InstanceInterruptionBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const ssm = require("aws-cdk-lib/aws-ssm");
const cr = require("aws-cdk-lib/custom-resources");
// eslint-disable-next-line import/no-extraneous-dependencies
const compare_versions_1 = require("compare-versions");
const constructs_1 = require("constructs");
/**
 * InstanceInterruptionBehavior enum.
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    /**
     * hibernate
     */
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    /**
     * stop
     */
    InstanceInterruptionBehavior["STOP"] = "stop";
    /**
     * terminate
     */
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = InstanceInterruptionBehavior = {}));
/**
 * GitlabContainerRunner Construct for create a Gitlab Runner.
 */
class GitlabContainerRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.gitlabRunnerImage) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Default Gitlab Runner Image use public.ecr.aws/gitlab/gitlab-runner:latest GitlabRunner Version Maybe > 16.0');
        }
        const spotFleetId = id;
        const defaultProps = {
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:latest',
            gitlaburl: 'https://gitlab.com/',
            ec2type: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            concurrentJobs: 1,
            runnerDescription: 'Docker Runner',
        };
        const runnerProps = { ...defaultProps, ...props };
        if (aws_cdk_lib_1.Token.isUnresolved(props.gitlabtoken)) {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`
      Your Gitlab Token pass by AWS Systems Manager Parameter Store or AWS Secret Manager can not check gitlab token. 
      @see https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html`);
        }
        else {
            if ((0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') && props.gitlabtoken.includes('glrt-') === false) {
                throw new Error('If gitlabRunnerVersion >= 15.10, gitlabtoken please give glrt-xxxxxxx @see https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html');
            }
        }
        const tokenParameterStore = new ssm.StringParameter(this, 'GitlabTokenParameter', {
            stringValue: 'GITLAB_TOKEN',
        });
        const shell = aws_ec2_1.UserData.forLinux();
        shell.addCommands(...this.createUserData(runnerProps, tokenParameterStore.parameterName));
        this.runnerRole =
            runnerProps.ec2iamrole ??
                new aws_iam_1.Role(this, 'runner-role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For Gitlab EC2 Runner Role',
                });
        this.validUntil = runnerProps.validUntil;
        tokenParameterStore.grantWrite(this.runnerRole);
        tokenParameterStore.grantRead(this.runnerRole);
        this.vpc =
            runnerProps.selfvpc ??
                new aws_ec2_1.Vpc(this, 'VPC', {
                    ipAddresses: aws_ec2_1.IpAddresses.cidr('10.0.0.0/16'),
                    maxAzs: 2,
                    subnetConfiguration: [
                        {
                            cidrMask: 26,
                            name: 'RunnerVPC',
                            subnetType: aws_ec2_1.SubnetType.PUBLIC,
                        },
                    ],
                    natGateways: 0,
                });
        this.defaultRunnerSG = new aws_ec2_1.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.vpc,
        });
        const spotOrOnDemand = runnerProps.spotFleet ?? false;
        if (spotOrOnDemand) {
            //--token Error('yes new spotfleet');
            const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
                roles: [this.runnerRole.roleName],
            });
            const imageId = aws_ec2_1.MachineImage.latestAmazonLinux2().getImage(this).imageId;
            const lt = new aws_ec2_1.CfnLaunchTemplate(this, 'LaunchTemplate', {
                launchTemplateData: {
                    imageId,
                    instanceType: runnerProps.ec2type,
                    blockDeviceMappings: [
                        {
                            deviceName: '/dev/xvda',
                            ebs: runnerProps.spotEbsConfig ?? {
                                volumeSize: 60,
                            },
                        },
                    ],
                    userData: aws_cdk_lib_1.Fn.base64(shell.render()),
                    keyName: runnerProps.keyName,
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [
                                {
                                    key: 'Name',
                                    value: `${aws_cdk_lib_1.Stack.of(this).stackName}/spotFleetGitlabRunner/${spotFleetId}`,
                                },
                            ],
                        },
                    ],
                    instanceMarketOptions: {
                        marketType: 'spot',
                        spotOptions: {
                            instanceInterruptionBehavior: runnerProps.instanceInterruptionBehavior ??
                                InstanceInterruptionBehavior.TERMINATE,
                        },
                    },
                    securityGroupIds: this.defaultRunnerSG.connections.securityGroups.map((m) => m.securityGroupId),
                    iamInstanceProfile: {
                        arn: instanceProfile.attrArn,
                    },
                    metadataOptions: runnerProps.enabledIMDSv2 ? {
                        httpTokens: 'required',
                        httpPutResponseHopLimit: 2,
                    } : undefined,
                },
            });
            const spotFleetRole = new aws_iam_1.Role(this, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            const vpcSubnetSelection = runnerProps.vpcSubnet ?? {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            };
            const subnetConfig = this.vpc
                .selectSubnets(vpcSubnetSelection)
                .subnets.map((s) => ({
                subnetId: s.subnetId,
            }));
            this.cfnSpotFleet = new aws_ec2_1.CfnSpotFleet(this, id, {
                spotFleetRequestConfigData: {
                    launchTemplateConfigs: [
                        {
                            launchTemplateSpecification: {
                                launchTemplateId: lt.ref,
                                version: lt.attrLatestVersionNumber,
                            },
                            overrides: subnetConfig,
                        },
                    ],
                    iamFleetRole: spotFleetRole.roleArn,
                    targetCapacity: 1,
                    validUntil: aws_cdk_lib_1.Lazy.string({ produce: () => this.validUntil }),
                    terminateInstancesWithExpiration: true,
                },
            });
            const onEvent = new lambda.Function(this, 'OnEvent', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.on_event',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
            });
            const isComplete = new lambda.Function(this, 'IsComplete', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.is_complete',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
                role: onEvent.role,
            });
            const myProvider = new cr.Provider(this, 'MyProvider', {
                onEventHandler: onEvent,
                isCompleteHandler: isComplete,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ec2:DescribeSpotFleetInstances'],
                resources: ['*'],
            }));
            const fleetInstancesId = new aws_cdk_lib_1.CustomResource(this, 'GetInstanceId', {
                serviceToken: myProvider.serviceToken,
                properties: {
                    SpotFleetRequestId: this.cfnSpotFleet.ref,
                },
            });
            fleetInstancesId.node.addDependency(this.cfnSpotFleet);
            this.spotFleetInstanceId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('InstanceId'));
            this.spotFleetRequestId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('SpotInstanceRequestId'));
            new aws_cdk_lib_1.CfnOutput(this, 'InstanceId', { value: this.spotFleetInstanceId });
            new aws_cdk_lib_1.CfnOutput(this, 'SpotFleetId', { value: this.cfnSpotFleet.ref });
        }
        else {
            this.runnerEc2 = new aws_ec2_1.Instance(this, 'GitlabRunner', {
                instanceType: new aws_ec2_1.InstanceType(runnerProps.ec2type),
                instanceName: 'Gitlab-Runner',
                vpc: this.vpc,
                vpcSubnets: runnerProps.vpcSubnet ?? {
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux2(),
                role: this.runnerRole,
                userData: shell,
                securityGroup: this.defaultRunnerSG,
                blockDevices: [
                    {
                        deviceName: '/dev/xvda',
                        volume: runnerProps.onDemandEbsConfig ?? aws_ec2_1.BlockDeviceVolume.ebs(60),
                    },
                ],
            });
            if (runnerProps.enabledIMDSv2) {
                const template = new aws_ec2_1.LaunchTemplate(this, 'Template', {
                    httpTokens: aws_ec2_1.LaunchTemplateHttpTokens.REQUIRED,
                    httpPutResponseHopLimit: 2,
                });
                const cfnInstance = this.runnerEc2.node.defaultChild;
                cfnInstance.launchTemplate = {
                    launchTemplateId: template.launchTemplateId,
                    version: template.latestVersionNumber,
                };
            }
            new aws_cdk_lib_1.CfnOutput(this, 'Runner-Instance-ID', {
                value: this.runnerEc2.instanceId,
            });
        }
        const unregisterRunnerOnEvent = new lambda.Function(this, 'unregisterRunnerOnEvent', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'unregister_runner.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: aws_cdk_lib_1.Duration.seconds(60),
        });
        const unregisterRunnerProvider = new cr.Provider(this, 'unregisterRunnerProvider', {
            onEventHandler: unregisterRunnerOnEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterRunnerCR = new aws_cdk_lib_1.CustomResource(this, 'unregisterRunnerCR', {
            resourceType: 'Custom::unregisterRunnerProvider',
            serviceToken: unregisterRunnerProvider.serviceToken,
            properties: {
                TokenParameterStoreName: tokenParameterStore.parameterName,
                GitlabUrl: runnerProps.gitlaburl,
            },
        });
        tokenParameterStore.grantRead(unregisterRunnerOnEvent);
        unregisterRunnerCR.node.addDependency(tokenParameterStore);
        this.runnerRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        new aws_cdk_lib_1.CfnOutput(this, 'Runner-Role-Arn', {
            value: this.runnerRole.roleArn,
        });
    }
    /**
     * Add expire time function for spotfleet runner !!! .
     *
     * @param duration - Block duration.
     */
    expireAfter(duration) {
        const date = new Date();
        date.setSeconds(date.getSeconds() + duration.toSeconds());
        this.validUntil = date.toISOString();
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @param tokenParameterStoreName - the tokenParameterStoreName to put gitlab runner token.
     * @returns Array.
     */
    createUserData(props, tokenParameterStoreName) {
        return [
            'yum update -y ',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlaburl} ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? '--token' : '--registration-token'} ${props.gitlabtoken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "${props.runnerDescription}" \
      ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? undefined : `--tag-list "${props.tags?.join(',')}" `} --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
            `sed -i 's/concurrent = .*/concurrent = ${props.concurrentJobs}/g' /home/ec2-user/.gitlab-runner/config.toml`,
            'TOKEN=$(cat /home/ec2-user/.gitlab-runner/config.toml | grep \'token \' | awk \'{print $3}\'| tr -d \'"\')',
            `aws ssm put-parameter --name ${tokenParameterStoreName} --value $TOKEN --overwrite --region ${aws_cdk_lib_1.Stack.of(this).region}`,
        ];
    }
}
exports.GitlabContainerRunner = GitlabContainerRunner;
_a = JSII_RTTI_SYMBOL_1;
GitlabContainerRunner[_a] = { fqn: "cdk-gitlab-runner.GitlabContainerRunner", version: "2.3.115" };
//# sourceMappingURL=data:application/json;base64,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