# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typegql', 'typegql.builder', 'typegql.client']

package_data = \
{'': ['*']}

install_requires = \
['graphql-core>=3']

extras_require = \
{u'client': ['aiohttp>=3.4,<4.0']}

setup_kwargs = {
    'name': 'typegql',
    'version': '4.0.2',
    'description': 'A Python GraphQL library that makes use of type hinting and concurrency support with the new async/await syntax.',
    'long_description': '.. role:: python(code)\n    :language: python\n\nTypeGQL\n========\n\nA Python `GraphQL <https://graphql.org>`_ library that makes use of type hinting and concurrency support with the new async/await syntax.\nWith the help of type hints and dataclass it\'s easy to build a GraphQL schema using nothing but Python objects and primitives\n\n**Consider the following:**\n\n.. code-block:: python\n\n    from graphql import (\n        GraphQLSchema, GraphQLObjectType, GraphQLField, GraphQLString)\n\n    schema = GraphQLSchema(\n        query=GraphQLObjectType(\n            name=\'RootQueryType\',\n            fields={\n                \'hello\': GraphQLField(\n                    GraphQLString,\n                    resolve=lambda obj, info: \'world\')\n            }))\n\n**Versus:**\n\n.. code-block:: python\n\n    from dataclasses import dataclass\n    from typegql import Schema\n\n    @dataclass(init=False)\n    class RootQuery:\n        hello: str\n\n    def resolve_hello(info):\n        return \'world\n\n     schema = Schema(query=RootQuery)\n\nClearly the second one looks more "Pythonic" and it\'s easier to maintain for complex structures\n\nInstallation\n============\n\n.. code-block:: python\n\n    pip install typegql\n\n\nUsage\n=====\n\nThe following demonstrates how to use **typegql** for implementing a *GraphQL API* for a library of books.\nThe example can be found in *typegql/core/examples* and you can run it with Sanic by executing ``python <path_to_example>/server.py``\n\nDefine your query\n-----------------\n\n.. code-block:: python\n\n    from dataclasses import dataclass\n    from typing import List\n    from typegql import Connection\n    from typegql.examples.library.types import Author, Category\n    from typegql.examples.library.types import Book\n    from typegql.examples.library import db\n\n\n    @dataclass(init=False, repr=False)\n    class Query:\n        books: List[Book]\n        authors: List[Author]\n        categories: List[Category]\n\n        books_connection: Connection[Book]\n\n        async def resolve_authors(self, info, **kwargs):\n            return db.get(\'authors\')\n\n        async def resolve_books(self, info, **kwargs):\n            return db.get(\'books\')\n\n        async def resolve_categories(self, info, **kwargs):\n            return db.get(\'categories\')\n\n       async def resolve_books_connection(self, info, **kwargs):\n            data = db.get(\'books\')\n            return {\n                \'edges\': [{\n                    \'node\': node\n                } for node in data]}\n\n\nDefine your types\n-----------------\n\n.. code-block:: python\n\n    from dataclasses import dataclass, field\n    from datetime import datetime\n    from decimal import Decimal\n    from enum import Enum\n    from typing import List\n\n    from examples.library import db\n\n\n    class Gender(Enum):\n        MALE = \'male\'\n        FEMALE = \'female\'\n\n\n    @dataclass\n    class GeoLocation:\n        latitude: Decimal\n        longitude: Decimal\n\n\n    @dataclass\n    class Author:\n        """Person that is usually a writer"""\n\n        id: ID = field(metadata={\'readonly\': True})\n        name: str\n        gender: Optional[Gender] = None\n        geo: Optional[GeoLocation] = None\n        books: Optional[List[Book]] = None\n\n\n    @dataclass\n    class Category:\n        id: ID = field(metadata={\'readonly\': True})\n        name: str\n\n\n    @dataclass\n    class Book:\n        """A book... for reading :|"""\n\n        id: ID = field(metadata={\'readonly\': True})\n        author_id: ID\n        title: str\n        author: Optional[Author] = field(default=None, metadata={\'description\': \'The author of this book\'})\n        categories: Optional[List[Category]] = None\n        published: Optional[datetime] = None\n        tags: Optional[List[str]] = None\n\n        def __post_init__(self):\n            self.published = datetime.strptime(self.published, \'%Y-%m-%d %H:%M:%S\')\n\n        async def resolve_author(self, info):\n            data = filter(lambda x: x[\'id\'] == self.author_id, db.get(\'authors\'))\n            data = next(data)\n            author = Author(**data)\n            author.gender = Gender(author.gender)\n            if \'geo\' in data:\n                author.geo = GeoLocation(**data.get(\'geo\'))\n            return author\n\n        async def resolve_categories(self, selections, name=None):\n            data = filter(lambda x: x[\'id\'] in self.categories, db.get(\'categories\'))\n            for d in data:  # showcasing async generator\n                yield Category(**d)\n\n        def resolve_tags(self, selections):\n            return [\'testing\', \'purpose\']\n\n\nRun your query\n--------------\n\n.. code-block:: python\n\n    from typegql import Schema\n    from examples.library.query import Query\n\n\n    schema = Schema(Query)\n    query = \'\'\'\n    query BooksConnection {\n      books_connection {\n        edges {\n          node {\n            id\n            title\n            published\n            author {\n              id\n              name\n            }\n          }\n        }\n      }\n    }\n    \'\'\'\n\n    async def run():\n        result = await schema.run(query)\n\nClient\n======\n\nTypeGQL supports DSL client for working with a GraphQL API.\nThe client automatically converts snake to camelcase. Set `camelcase=False` if this is not desired.\n\n.. code-block:: python\n\n    pip install typegql[client]\n\nFor example:\n\n\n.. code-block:: python\n\n    from typegql.client import Client\n\n    async with Client(url, camelcase=True) as client:\n        await client.introspection()\n        dsl = client.dsl\n        query = dsl.Query.books_connection.select(dsl.BooksConnection.total_count)\n        doc = dsl.query(query)\n\n        status, result = await client.execute(doc)\n\nChange Log\n==========\n4.0.2 [2020-04-06]\n------------------\n- updates graphql-core to 3.1.0\n\n4.0.1 [2020-03-17]\n------------------\n- fixes enum return value\n\n4.0.0 [2020-03-16]\n------------------\n- BREAKING CHANGES:\n    - `Schema` now accepts a more granular list of custom graphql types. Signature is:\n\n.. code-block:: python\n\n        def __init__(self,\n                 query: Type,\n                 mutation: Optional[Type] = None,\n                 subscription: Optional[Type] = None,\n                 scalars: Optional[GraphQLScalarMap] = None,\n                 enums: Optional[GraphQLEnumMap] = None,\n                 interfaces: Optional[GraphQLInterfaceMap] = None,\n                 query_types: Optional[GraphQLObjectTypeMap] = None,\n                 mutation_types: Optional[GraphQLInputObjectTypeMap] = None,\n                 camelcase=True):\n\n3.1.0 [2020-01-29]\n------------------\n- fixes an issue with camelcase parameters when a `load` method is provided\n\n3.0.9 [2020-01-13]\n------------------\n- fix client execution function\n- update `dsl` selections to work with `FrozenList`\n\n3.0.8 [2020-01-07]\n------------------\n- added ability to ignore attributes in building the schema by using `field(metadata={\'skip\': True})`. This can be useful when you don\'t want to expose certain fields in GraphQL, like a user\'s `password` for example.\n\n3.0.5 [2019-12-16]\n------------------\n- added support for subscriptions\n\n3.0.4 [2019-12-04]\n------------------\n- updates `graphql-core-next` to `grapqhl-core` >= 3\n- use Sequence instead of List where possible\n\n3.0.3 [2019-11-29]\n------------------\n- fixed a bug where a custom connection arguments don\'t include the relay pagination arguments as well\n\n3.0.2 [2019-11-26]\n------------------\n- PEP 561 compliant\n\n3.0.1 [2019-11-26]\n------------------\n- BREAKING: Removed `Graph` as a baseclass\n- now makes use of `dataclasses.dataclass` and `dataclasess.fields` for building the `Schema`\n- bug fixing and improvements\n\n2.0.9 [2019-10-29]\n------------------\n- changed the name of an input object from ObjectMuation to ObjectInput\n\n2.0.8 [2019-10-15]\n------------------\n- allows forward reference between graph types (ie: `Book` has an `author` and an `Author` has `books`).\n    this only works with python 3.7(using `from __future__ import annotations`, or python 3.8\n\n2.0.6 [2019-06-24]\n------------------\n- updates uvloop dependency\n\n2.0.5 [2019-04-24]\n------------------\n- fixed a bug when sending `introspection schema`\n\n2.0.4 [2019-04-24]\n------------------\n- updates assert for introspection add message with status and result\n- adds support for enum objects in `resolve_field_velue_or_error`\n\n2.0.3 [2019-02-08]\n------------------\n- changes `Connection`, `Edge`, `Node` and `PageInfo` to interfaces `IConnection`, `IEdge`, etc.\n- implements default `Connection` and `PageInfo` objects\n- removes `has_next`, `has_previous` from `PageInfo`\n\n2.0.1 [2019-01-19]\n------------------\n- all properties that don\'t have a `Field` instance assigned to them will be ignored by the `Schema`\n- updates docs & example to reflect 2.0 changes\n- fixed a bug when using a `List` argument in mutations\n\n1.0.7 [2018-12-09]\n------------------\n- bug fixing\n- adds support for camelcase in Client\n\n1.0.1 [2018-11-19]\n------------------\n- adds support for client DSL\n\nInitial\n-------\n- added `graphql-core-next <https://github.com/graphql-python/graphql-core-next>`_ as a baseline for all GraphQL operations\n\n\nTODO\n====\n- testing\n- travis\n- more testing\n- please help with testing :|\n',
    'author': 'Ciprian Tarta',
    'author_email': 'ciprian@cipriantarta.ro',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cipriantarta/typegql',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
