# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# Copyright 2021 Daniel Mark Gass, see __about__.py for license information.
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
"""Test bytes transform API conformance."""

from baseline import Baseline

from plum.bytes import BytesX
from plum.conformance import Case, CaseData


class TestGreedy(Case):

    """Test greedy."""

    data = CaseData(
        fmt=BytesX("greedy_bytes"),
        bindata=bytes(range(20)),
        nbytes=None,
        values=(bytes(range(20)),),
        dump=Baseline(
            r"""
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------------+
            | Offset | Access  | Value                                                         | Bytes                                           | Format       |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------------+
            |        |         |                                                               |                                                 | greedy_bytes |
            |  0     | [0:16]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e\x0f' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f |              |
            | 16     | [16:20] | b'\x10\x11\x12\x13'                                           | 10 11 12 13                                     |              |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------------+
            """
        ),
        excess="N/A",
        shortage="N/A",
    )


class TestFixedSize(Case):

    """Test fixed size."""

    data = CaseData(
        fmt=BytesX("fixed", nbytes=20),
        bindata=bytes(range(20)),
        nbytes=20,
        values=(bytes(range(20)),),
        dump=Baseline(
            r"""
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            | Offset | Access  | Value                                                         | Bytes                                           | Format |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            |        |         |                                                               |                                                 | fixed  |
            |  0     | [0:16]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e\x0f' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f |        |
            | 16     | [16:20] | b'\x10\x11\x12\x13'                                           | 10 11 12 13                                     |        |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            """
        ),
        excess=Baseline(
            r"""
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            | Offset | Access  | Value                                                         | Bytes                                           | Format |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            |        |         |                                                               |                                                 | fixed  |
            |  0     | [0:16]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e\x0f' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f |        |
            | 16     | [16:20] | b'\x10\x11\x12\x13'                                           | 10 11 12 13                                     |        |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
            | 20     |         | <excess bytes>                                                | 99                                              |        |
            +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+

            ExcessMemoryError occurred during unpack operation:

            1 unconsumed bytes
            """
        ),
        shortage=Baseline(
            r"""
        +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
        | Offset | Access  | Value                                                         | Bytes                                           | Format |
        +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+
        |        |         | <insufficient bytes>                                          |                                                 | fixed  |
        |  0     | [0:16]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e\x0f' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f |        |
        | 16     | [16:19] | b'\x10\x11\x12'                                               | 10 11 12                                        |        |
        +--------+---------+---------------------------------------------------------------+-------------------------------------------------+--------+

        InsufficientMemoryError occurred during unpack operation:

        1 too few bytes to unpack 'fixed' (20 needed, only 19 available)
        """
        ),
    )


class TestPadded(Case):

    """Test fixed size with pad."""

    data = CaseData(
        fmt=BytesX("padded", nbytes=20, pad=b"\x99"),
        bindata=bytes(range(15)) + b"\x99" * 5,
        nbytes=20,
        values=(bytes(range(15)),),
        dump=Baseline(
            r"""
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            | Offset | Access  | Value                                                     | Bytes                                        | Format |
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            |        |         |                                                           |                                              | padded |
            |  0     | [0:15]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e |        |
            | 15     | --pad-- | b'\x99\x99\x99\x99\x99'                                   | 99 99 99 99 99                               |        |
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            """
        ),
        excess=Baseline(
            r"""
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            | Offset | Access  | Value                                                     | Bytes                                        | Format |
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            |        |         |                                                           |                                              | padded |
            |  0     | [0:15]  | b'\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b\x0c\r\x0e' | 00 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e |        |
            | 15     | --pad-- | b'\x99\x99\x99\x99\x99'                                   | 99 99 99 99 99                               |        |
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+
            | 20     |         | <excess bytes>                                            | 99                                           |        |
            +--------+---------+-----------------------------------------------------------+----------------------------------------------+--------+

            ExcessMemoryError occurred during unpack operation:

            1 unconsumed bytes
            """
        ),
        shortage="N/A",
    )
