"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const asg = require("aws-cdk-lib/aws-autoscaling");
const ec2 = require("aws-cdk-lib/aws-ec2");
const gitlab_runner_autoscaling_1 = require("./gitlab-runner-autoscaling");
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, 'TestStackAutoscaling', { env });
const vpc = ec2.Vpc.fromLookup(stack, 'DefaultVpc', {
    isDefault: true,
});
const defaultCapacity = parseInt(stack.node.tryGetContext('CAPACITY'));
const runner = new gitlab_runner_autoscaling_1.GitlabRunnerAutoscaling(stack, 'TestRunnerAutoscaling', {
    gitlabToken: stack.node.tryGetContext('GITLAB_TOKEN'),
    ebsSize: 10,
    vpc: vpc,
    dockerVolumes: [{
            hostPath: '/tmp/cache',
            containerPath: '/tmp/cache',
        }],
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
    spotInstance: true,
    alarms: [
        {
            AlarmName: 'GitlabRunnerDiskUsage',
            MetricName: 'disk_used_percent',
            Threshold: 50,
        },
    ],
});
/**
 * Scheduled scaling
 * https://docs.aws.amazon.com/cdk/api/latest/docs/aws-autoscaling-readme.html#scheduled-scaling
 */
runner.autoscalingGroup.scaleOnSchedule('StopOnWeekends', {
    schedule: asg.Schedule.cron({ weekDay: 'fri', hour: '18', minute: '0' }),
    minCapacity: 0,
    maxCapacity: 0,
});
runner.autoscalingGroup.scaleOnSchedule('WorkOnWeekdays', {
    schedule: asg.Schedule.cron({ weekDay: 'mon', hour: '9', minute: '0' }),
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
});
new aws_cdk_lib_1.CfnOutput(stack, 'role', { value: runner.instanceRole.roleArn });
//# sourceMappingURL=data:application/json;base64,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