# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['html2image', 'html2image.browsers']

package_data = \
{'': ['*']}

entry_points = \
{'console_scripts': ['hti = html2image:main', 'html2image = html2image:main']}

setup_kwargs = {
    'name': 'html2image',
    'version': '2.0.1',
    'description': 'Package acting as a wrapper around the headless mode of existing web browsers to generate images from URLs and from HTML+CSS strings or files.',
    'long_description': '<img src="readme_assets/html2image_black.png" align="right" alt="html2image logo" title="html2image" height="128"/>\n\n# HTML2Image\n[\n![PyPI](https://img.shields.io/pypi/v/html2image.svg)\n![PyPI](https://img.shields.io/pypi/pyversions/html2image.svg)\n![PyPI](https://img.shields.io/github/license/vgalin/html2image.svg)\n](https://pypi.org/project/html2image/)\n[\n![GitHub](https://img.shields.io/github/v/release/vgalin/html2image?include_prereleases)\n![GitHub](https://img.shields.io/github/languages/code-size/vgalin/html2image)\n](https://github.com/vgalin/html2image)\n\nHTML2Image is a lightweight Python package that acts as a wrapper around the headless mode of existing web browsers to generate images from URLs and from HTML+CSS strings or files.\n\nThis package has been tested on Windows, Ubuntu (desktop and server) and MacOS. It is currently in a work in progress stage. If you encounter any problem or difficulties while using it, feel free to open an issue on the GitHub page of this project. Feedback is also welcome!\n\n\n## Principle\n\nMost web browsers have a Headless Mode, which is a way to run them without displaying any graphical interface. Headless mode is mainly used for automated testings but also comes in handy if you want to take screenshots of web pages that are exact replicas of what you would see on your screen if you were using the browser yourself.\n\nHowever, for the sake of taking screenshots, headless mode is not very convenient to use. HTML2Image aims to hide the inconveniences of the browsers\' headless modes while adding useful features such as allowing to create an image from as little as a string.\n\nFor more information about headless modes :\n-   (Chrome) [https://developers.google.com/web/updates/2017/04/headless-chrome](https://developers.google.com/web/updates/2017/04/headless-chrome)\n-   (Firefox) [https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode)\n\n## Installation\nHTML2Image is published on PyPI and can be installed through pip:\n\n```console\npip install --upgrade html2image\n```\n\nIn addition to this package, at least one of the following browsers **must** be installed on your machine :\n-   Google Chrome (Windows, MacOS)\n-   Chromium Brower (Linux)\n\n## Usage\n\n### First, import the package and instantiate it\n```python\nfrom html2image import Html2Image\nhti = Html2Image()\n```\n\n<details>\n<summary> Multiple arguments can be passed to the constructor (click to expand):</summary>\n\n-   `browser` :  Browser that will be used, set by default to `\'chrome\'` (the only browser supported by HTML2Image at the moment)\n-   `browser_executable` : The path or the command that can be used to find the executable of a specific browser.\n-   `output_path` : Path to the folder to which taken screenshots will be outputed. Default is the current working directory of your python program.\n-   `size` : 2-Tuple reprensenting the size of the screenshots that will be taken. Default value is `(1920, 1080)`.\n-   `temp_path` : Path that will be used to put together different resources when screenshotting strings of files. Default value is `%TEMP%/html2image` on Windows, and `/tmp/html2image` on Linux and MacOS.\n\nExample:\n```python\nhti = Html2Image(size=(500, 200))\n```\n\nYou can also change these values later: \n``` python\nhti.size = (500, 200)\n```\n</details>\n<br>\n\n### Then take a screenshot\n\nThe `screenshot` method is the basis of this package, most of the time, you won\'t need to use anything else. It can take screenshots of a lot of things :\n- URLs via the `url` parameter;\n- HTML and CSS **files** via the `html_file` and `css_file` parameters;\n- HTML and CSS **strings** via the `html_str` and `css_str` parameters;\n- and "other" types of files via the `other_file` parameter (try it with .svg files!).\n\nAnd you can also (optional):\n- Change the size of the screenshots using the `size` parameter;\n- Save the screenshots as a specific name using the `save_as` parameter.\n\n*N.B. : The `screenshot` method returns a **list** containing the path(s) of the screenshot(s) it took.*\n\n### A few examples\n\n- **URL to image**\n```python\nhti.screenshot(url=\'https://www.python.org\', save_as=\'python_org.png\')\n```\n\n- **HTML & CSS strings to image**\n```python\nhtml = """<h1> An interesting title </h1> This page will be red"""\ncss = "body {background: red;}"\n\nhti.screenshot(html_str=html, css_str=css, save_as=\'red_page.png\')\n```\n\n- **HTML & CSS files to image**\n```python\nhti.screenshot(\n    html_file=\'blue_page.html\', css_file=\'blue_background.css\',\n    save_as=\'blue_page.png\'\n)\n```\n\n- **Other files to image**\n```python\nhti.screenshot(other_file=\'star.svg\')\n```\n\n- **Change the screenshots\' size**\n```python\nhti.screenshot(other_file=\'star.svg\', size=(500, 500))\n```\n\n---\n\n<details>\n<summary> Click to show all the images generated with all the code above </summary>\n<img src="readme_assets/sample_url_to_img.png" alt="sample_url_to_img.png"/>\n<img src="readme_assets/sample_strings_to_img.png" alt="sample_strings_to_img"/>\n<img src="readme_assets/sample_files_to_img.png" alt="sample_files_to_img"/>\n<img src="readme_assets/sample_other.png" alt="sample_other_to_img"/>\n<img src="readme_assets/sample_other_50_50.png" alt="sample_other_50_50"/>\n\n</details>\n\n\n---\n\n- **Change the directory to which the screenshots are saved**\n```python\nhti = Html2Image(output_path=\'my_screenshot_folder\')\n```\n**OR**\n```python\nhti.output_path = \'my_screenshot_folder\'\n```\n\n*N.B. : the output path will be changed for all future screenshots.*\n\n---\n\n#### Use lists in place of any parameters while using the `screenshot` method\n- Screenshot multiple objects using only one filename, or one filename per file:\n```python\n# create three files from one filename\nhti.screenshot(html_str=[\'A\', \'B\', \'C\'], save_as=\'ABC.png\')\n# outputs ABC_0.png, ABC_1.png, ABC_2.png\n\n# create three files from from different filenames\nhti.screenshot(html_str=[\'A\', \'B\', \'C\'], save_as=[\'A.png\', \'B.png\', \'C.png\'])\n# outputs A.png, B.png, C.png\n```\n- Take multiple screenshots with the same size\n```python\n# take four screenshots with a resolution of 100*50\nhti.screenshot(\n    html_str=[\'A\', \'B\', \'C\', \'D\']\n    size=(100, 50)\n)\n```\n- Take multiple screenshots with different sizes\n```python\n# take four screenshots with different resolutions from three given sizes\nhti.screenshot(\n    html_str=[\'A\', \'B\', \'C\', \'D\'],\n    size=[(100, 50), (100, 100), (50, 50)]\n)\n# respectively 100*50, 100*100, 50*50, 50*50\n# if not enough sizes are given, the last size in the list will be repeated\n\n```\n\n- Apply CSS string(s) to multiple HTML string(s)\n```python\n# screenshot two html strings and apply css strings on both\nhti.screenshot(\n    html_str=[\'A\', \'B\'],\n    css_str=\'body {background: red;}\'\n)\n\n# screenshot two html strings and apply multiple css strings on both\nhti.screenshot(\n    html_str=[\'A\', \'B\'],\n    css_str=[\'body {background: red;}\', \'body {font-size: 50px;}\']\n)\n\n# screenshot one html string and apply multiple css strings on it\nhti.screenshot(\n    html_str=\'A\',\n    css_str=[\'body {background: red;}\', \'body {font-size: 50px;}\']\n)\n```\n\n---\n\n- **Retrieve the path of the generated file(s)**  \nThe `screenshot` method returns a list containing the path(s) of the screenshot(s):\n\n```python\npaths = hti.screenshot(\n    html_str=[\'A\', \'B\', \'C\'],\n    save_as="letters.png",\n)\n\nprint(paths)\n# >>> [\'D:\\\\myFiles\\\\letters_0.png\', \'D:\\\\myFiles\\\\letters_1.png\', \'D:\\\\myFiles\\\\letters_2.png\']\n```\n\n---\n\n#### Change browser flags\nIn some cases, you may need to change the *flags* that are used to run the headless mode of a browser.\n\nFlags can be used to:\n- Change the default background color of the pages;\n- Hide the scrollbar;\n- Add delay before taking a screenshot;\n- Allow you to use Html2Image when you\'re root, as you will have to specify the `--no-sandbox` flag;\n\nYou can find the full list of Chrome / Chromium flags [here](https://peter.sh/experiments/chromium-command-line-switches/).\n\nThere is two ways to specify custom flags:\n```python\n# At the object instanciation\nhti = Html2image(custom_flags=[\'--my_flag\', \'--my_other_flag=value\'])\n\n# Afterwards\nhti.browser.flags = [\'--my_flag\', \'--my_other_flag=value\']\n```\n\n- **Flags example use-case: adding a delay before taking a screenshot**\n\nWith Chrome / Chromium, screenshots are fired directly after there is no more "pending network fetches", but you may sometimes want to add a delay before taking a screenshot, to wait for animations to end for example. \nThere is a flag for this purpose, `--virtual-time-budget=VALUE_IN_MILLISECONDS`. You can use it like so:\n\n```python\nhti = Html2Image(\n    custom_flags=[\'--virtual-time-budget=10000\', \'--hide-scrollbars\']\n)\n\nhti.screenshot(url=\'http://example.org\')\n```\n\n- **Default flags**\n\nFor ease of use, some flags are set by default. However default flags are not used if you decide to specify `custom_flags` or change the value of `browser.flags`:\n\n```python\n# Taking a look at the default flags\n>>> hti = Html2Image()\n>>> hti.browser.flags\n[\'--default-background-color=0\', \'--hide-scrollbars\']\n\n# Changing the value of browser.flags gets rid of the default flags.\n>>> hti.browser.flags = [\'--1\', \'--2\']\n>>> hti.browser.flags\n[\'--1\', \'--2\'] \n\n# Using the custom_flags parameter gets rid of the default flags.\n>>> hti = Html2Image(custom_flags=[\'--a\', \'--b\'])\n>>> hti.browser.flags\n[\'--a\', \'--b\']\n```\n\n## Using the CLI\nHTML2image comes with a Command Line Interface which you can use to generate screenshots from files and urls on the go.\n\nThe CLI is a work in progress and may be subject to changes.\nYou can call it by typing `hti` or `html2image` into a terminal.\n\n\n| argument | description | example |\n| - | - | - |\n| -h, --help | Shows the help message | `hti -h` |\n| -U, --urls | Screenshots a list of URLs | `hti -U https://www.python.org` |\n| -H, --html | Screenshots a list of HTML files | `hti -H file.html` |\n| -C, --css | Attaches a CSS files to the HTML ones | `hti -H file.html -C style.css` |\n| -O, --other | Screenshots a list of files of type "other" | `hti -O star.svg` |\n| -S, --save-as | A list of the screenshot filename(s)  | `hti -O star.svg -S star.png` |\n| -s, --size | A list of the screenshot size(s) | `hti -O star.svg -s 50,50`|\n| -o, --output_path| Change the output path of the screenshots (default is current working directory) | `hti star.svg -o screenshot_dir` |\n| -q, --quiet| Disable all CLI\'s outputs | `hti --quiet` |\n| -v, --verbose| More details, can help debugging | `hti --verbose` |\n| --chrome_path| Specify a different chrome path ||\n| --temp_path| Specify a different temp path (where the files are loaded)||\n\n<br>\n\n## Testing\n\nOnly basic testing is available at the moment. To run tests, install the requirements (Pillow) and run PyTest at the root of the project:\n``` console\npip install -r requirements-test.txt\npython -m pytest\n```\n\n\n## FAQ\n\n- Can I automatically take a full page screenshot?  \n**Sadly no**, it is not easily possible. Html2Image relies on the headless mode of Chrome/Chromium browsers to take screenshots and there is no way to "ask" for a full page screenshot at the moment. If you know a way to take one (by estimating the page size for example) I would be happy to see it, so please open an issue or a discussion!\n\n- Can I add delay before taking a screenshot?   \n**Yes** you can, please take a look at the `Change browser flags` section of the readme.\n\n- Can I speed up the screenshot taking process?  \n**Yes**, when you are taking a lot of screenshots, you can achieve better "performances" using Parallel Processing or Multiprocessing methods. You can find an [example of it here](https://github.com/vgalin/html2image/issues/28#issuecomment-862608053).\n\n- Can I make a cookie modal disappear?  \n**Yes and no**. **No** because there is no options to do it magically and [extensions are not supported in headless Chrome](https://bugs.chromium.org/p/chromium/issues/detail?id=706008#c5) (The [`I don\'t care about cookies`](https://www.i-dont-care-about-cookies.eu/) extension would have been useful in this case). **Yes** because you can make any element of a page disappear by retrieving its source code, modifying it as you wish, and finally screenshotting the modified source code.\n## TODO List\n-   A nice CLI (currently in a WIP state).\n-   Support of other browsers (such as Firefox when their screenshot feature will work).\n-   PDF generation?\n-   Contributing, issue templates, pull request template, code of conduct.\n\n---\n\n*If you see any typos or notice things that are odly said, feel free to create an issue or a pull request.*',
    'author': 'vgalin',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vgalin/html2image',
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
