"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyLayers = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, extensionLayerVersion) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = index_1.runtimeLookup[runtime];
        const isARM = lam.architecture === aws_lambda_1.Architecture.ARM_64;
        const isNode = lambdaRuntimeType === index_1.RuntimeType.NODE;
        if (lambdaRuntimeType === index_1.RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === index_1.RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                const errorMessage = index_1.getMissingLayerVersionErrorMsg(lam.node.id, "Python", "python");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = index_1.getLambdaLayerArn(region, pythonLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                const errorMessage = index_1.getMissingLayerVersionErrorMsg(lam.node.id, "Node.js", "node");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = index_1.getLambdaLayerArn(region, nodeLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = index_1.getExtensionLayerArn(region, extensionLayerVersion, isARM);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    const layerId = index_1.generateLayerId(isExtensionLayer, lam.functionArn, runtime);
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
//# sourceMappingURL=data:application/json;base64,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