"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleSettingPropDefaults = exports.checkForMultipleApiKeys = exports.validateProps = void 0;
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
function validateProps(props) {
    loglevel_1.default.debug("Validating props...");
    checkForMultipleApiKeys(props);
    const siteList = [
        "datadoghq.com",
        "datadoghq.eu",
        "us3.datadoghq.com",
        "us5.datadoghq.com",
        "ddog-gov.com",
    ];
    if (props.site !== undefined &&
        !siteList.includes(props.site.toLowerCase()) &&
        !(props.site.startsWith("${Token[") && props.site.endsWith("]}"))) {
        throw new Error("Warning: Invalid site URL. Must be either datadoghq.com, datadoghq.eu, us3.datadoghq.com, us5.datadoghq.com, or ddog-gov.com.");
    }
    if (props.apiKey === undefined &&
        props.apiKmsKey === undefined &&
        props.apiKeySecretArn === undefined &&
        props.flushMetricsToLogs === false) {
        throw new Error("When `flushMetricsToLogs` is false, `apiKey`, `apiKeySecretArn`, or `apiKmsKey` must also be set.");
    }
    if (props.extensionLayerVersion !== undefined) {
        if (props.apiKey === undefined && props.apiKeySecretArn === undefined && props.apiKmsKey === undefined) {
            throw new Error("When `extensionLayer` is set, `apiKey`, `apiKeySecretArn`, or `apiKmsKey` must also be set.");
        }
    }
}
exports.validateProps = validateProps;
function checkForMultipleApiKeys(props) {
    let multipleApiKeysMessage;
    if (props.apiKey !== undefined && props.apiKmsKey !== undefined && props.apiKeySecretArn !== undefined) {
        multipleApiKeysMessage = "`apiKey`, `apiKmsKey`, and `apiKeySecretArn`";
    }
    else if (props.apiKey !== undefined && props.apiKmsKey !== undefined) {
        multipleApiKeysMessage = "`apiKey` and `apiKmsKey`";
    }
    else if (props.apiKey !== undefined && props.apiKeySecretArn !== undefined) {
        multipleApiKeysMessage = "`apiKey` and `apiKeySecretArn`";
    }
    else if (props.apiKmsKey !== undefined && props.apiKeySecretArn !== undefined) {
        multipleApiKeysMessage = "`apiKmsKey` and `apiKeySecretArn`";
    }
    if (multipleApiKeysMessage) {
        throw new Error(`${multipleApiKeysMessage} should not be set at the same time.`);
    }
}
exports.checkForMultipleApiKeys = checkForMultipleApiKeys;
function handleSettingPropDefaults(props) {
    let addLayers = props.addLayers;
    let enableDatadogTracing = props.enableDatadogTracing;
    let injectLogContext = props.injectLogContext;
    const logLevel = props.logLevel;
    let enableDatadogLogs = props.enableDatadogLogs;
    let captureLambdaPayload = props.captureLambdaPayload;
    if (addLayers === undefined) {
        loglevel_1.default.debug(`No value provided for addLayers, defaulting to ${constants_1.DefaultDatadogProps.addLayers}`);
        addLayers = constants_1.DefaultDatadogProps.addLayers;
    }
    if (enableDatadogTracing === undefined) {
        loglevel_1.default.debug(`No value provided for enableDatadogTracing, defaulting to ${constants_1.DefaultDatadogProps.enableDatadogTracing}`);
        enableDatadogTracing = constants_1.DefaultDatadogProps.enableDatadogTracing;
    }
    if (injectLogContext === undefined) {
        loglevel_1.default.debug(`No value provided for injectLogContext, defaulting to ${constants_1.DefaultDatadogProps.injectLogContext}`);
        injectLogContext = constants_1.DefaultDatadogProps.injectLogContext;
    }
    if (logLevel === undefined) {
        loglevel_1.default.debug(`No value provided for logLevel`);
    }
    if (enableDatadogLogs === undefined) {
        loglevel_1.default.debug(`No value provided for enableDatadogLogs, defaulting to ${constants_1.DefaultDatadogProps.enableDatadogLogs}`);
        enableDatadogLogs = constants_1.DefaultDatadogProps.enableDatadogLogs;
    }
    if (captureLambdaPayload === undefined) {
        loglevel_1.default.debug(`No value provided for captureLambdaPayload, default to ${constants_1.DefaultDatadogProps.captureLambdaPayload}`);
        captureLambdaPayload = constants_1.DefaultDatadogProps.captureLambdaPayload;
    }
    return {
        addLayers: addLayers,
        enableDatadogTracing: enableDatadogTracing,
        injectLogContext: injectLogContext,
        logLevel: logLevel,
        enableDatadogLogs: enableDatadogLogs,
        captureLambdaPayload: captureLambdaPayload,
    };
}
exports.handleSettingPropDefaults = handleSettingPropDefaults;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWRvZ1NoYXJlZExvZ2ljLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2NvbW1vbi9kYXRhZG9nU2hhcmVkTG9naWMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7R0FNRzs7O0FBRUgsdUNBQTJCO0FBQzNCLDJDQUFrRDtBQUdsRCxTQUFnQixhQUFhLENBQUMsS0FBbUI7SUFDL0Msa0JBQUcsQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUVqQyx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMvQixNQUFNLFFBQVEsR0FBYTtRQUN6QixlQUFlO1FBQ2YsY0FBYztRQUNkLG1CQUFtQjtRQUNuQixtQkFBbUI7UUFDbkIsY0FBYztLQUNmLENBQUM7SUFDRixJQUNFLEtBQUssQ0FBQyxJQUFJLEtBQUssU0FBUztRQUN4QixDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUM1QyxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsRUFDakU7UUFDQSxNQUFNLElBQUksS0FBSyxDQUNiLCtIQUErSCxDQUNoSSxDQUFDO0tBQ0g7SUFFRCxJQUNFLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUztRQUMxQixLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVM7UUFDN0IsS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTO1FBQ25DLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxLQUFLLEVBQ2xDO1FBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixtR0FBbUcsQ0FDcEcsQ0FBQztLQUNIO0lBQ0QsSUFBSSxLQUFLLENBQUMscUJBQXFCLEtBQUssU0FBUyxFQUFFO1FBQzdDLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUU7WUFDdEcsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO1NBQ2hIO0tBQ0Y7QUFDSCxDQUFDO0FBcENELHNDQW9DQztBQUVELFNBQWdCLHVCQUF1QixDQUFDLEtBQW1CO0lBQ3pELElBQUksc0JBQXNCLENBQUM7SUFDM0IsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtRQUN0RyxzQkFBc0IsR0FBRyw4Q0FBOEMsQ0FBQztLQUN6RTtTQUFNLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUU7UUFDdEUsc0JBQXNCLEdBQUcsMEJBQTBCLENBQUM7S0FDckQ7U0FBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxlQUFlLEtBQUssU0FBUyxFQUFFO1FBQzVFLHNCQUFzQixHQUFHLGdDQUFnQyxDQUFDO0tBQzNEO1NBQU0sSUFBSSxLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtRQUMvRSxzQkFBc0IsR0FBRyxtQ0FBbUMsQ0FBQztLQUM5RDtJQUVELElBQUksc0JBQXNCLEVBQUU7UUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLHNCQUFzQixzQ0FBc0MsQ0FBQyxDQUFDO0tBQ2xGO0FBQ0gsQ0FBQztBQWZELDBEQWVDO0FBRUQsU0FBZ0IseUJBQXlCLENBQUMsS0FBbUI7SUFDM0QsSUFBSSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztJQUNoQyxJQUFJLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQztJQUN0RCxJQUFJLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztJQUM5QyxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO0lBQ2hDLElBQUksaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO0lBQ2hELElBQUksb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBRXRELElBQUksU0FBUyxLQUFLLFNBQVMsRUFBRTtRQUMzQixrQkFBRyxDQUFDLEtBQUssQ0FBQyxrREFBa0QsK0JBQW1CLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUM3RixTQUFTLEdBQUcsK0JBQW1CLENBQUMsU0FBUyxDQUFDO0tBQzNDO0lBQ0QsSUFBSSxvQkFBb0IsS0FBSyxTQUFTLEVBQUU7UUFDdEMsa0JBQUcsQ0FBQyxLQUFLLENBQUMsNkRBQTZELCtCQUFtQixDQUFDLG9CQUFvQixFQUFFLENBQUMsQ0FBQztRQUNuSCxvQkFBb0IsR0FBRywrQkFBbUIsQ0FBQyxvQkFBb0IsQ0FBQztLQUNqRTtJQUNELElBQUksZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1FBQ2xDLGtCQUFHLENBQUMsS0FBSyxDQUFDLHlEQUF5RCwrQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7UUFDM0csZ0JBQWdCLEdBQUcsK0JBQW1CLENBQUMsZ0JBQWdCLENBQUM7S0FDekQ7SUFDRCxJQUFJLFFBQVEsS0FBSyxTQUFTLEVBQUU7UUFDMUIsa0JBQUcsQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztLQUM3QztJQUNELElBQUksaUJBQWlCLEtBQUssU0FBUyxFQUFFO1FBQ25DLGtCQUFHLENBQUMsS0FBSyxDQUFDLDBEQUEwRCwrQkFBbUIsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDN0csaUJBQWlCLEdBQUcsK0JBQW1CLENBQUMsaUJBQWlCLENBQUM7S0FDM0Q7SUFDRCxJQUFJLG9CQUFvQixLQUFLLFNBQVMsRUFBRTtRQUN0QyxrQkFBRyxDQUFDLEtBQUssQ0FBQywwREFBMEQsK0JBQW1CLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1FBQ2hILG9CQUFvQixHQUFHLCtCQUFtQixDQUFDLG9CQUFvQixDQUFDO0tBQ2pFO0lBRUQsT0FBTztRQUNMLFNBQVMsRUFBRSxTQUFTO1FBQ3BCLG9CQUFvQixFQUFFLG9CQUFvQjtRQUMxQyxnQkFBZ0IsRUFBRSxnQkFBZ0I7UUFDbEMsUUFBUSxFQUFFLFFBQVE7UUFDbEIsaUJBQWlCLEVBQUUsaUJBQWlCO1FBQ3BDLG9CQUFvQixFQUFFLG9CQUFvQjtLQUMzQyxDQUFDO0FBQ0osQ0FBQztBQXhDRCw4REF3Q0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogVW5sZXNzIGV4cGxpY2l0bHkgc3RhdGVkIG90aGVyd2lzZSBhbGwgZmlsZXMgaW4gdGhpcyByZXBvc2l0b3J5IGFyZSBsaWNlbnNlZFxuICogdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlIFZlcnNpb24gMi4wLlxuICpcbiAqIFRoaXMgcHJvZHVjdCBpbmNsdWRlcyBzb2Z0d2FyZSBkZXZlbG9wZWQgYXQgRGF0YWRvZyAoaHR0cHM6Ly93d3cuZGF0YWRvZ2hxLmNvbS8pLlxuICogQ29weXJpZ2h0IDIwMjEgRGF0YWRvZywgSW5jLlxuICovXG5cbmltcG9ydCBsb2cgZnJvbSBcImxvZ2xldmVsXCI7XG5pbXBvcnQgeyBEZWZhdWx0RGF0YWRvZ1Byb3BzIH0gZnJvbSBcIi4vY29uc3RhbnRzXCI7XG5pbXBvcnQgeyBEYXRhZG9nUHJvcHMsIERhdGFkb2dTdHJpY3RQcm9wcyB9IGZyb20gXCIuL2ludGVyZmFjZXNcIjtcblxuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlUHJvcHMocHJvcHM6IERhdGFkb2dQcm9wcykge1xuICBsb2cuZGVidWcoXCJWYWxpZGF0aW5nIHByb3BzLi4uXCIpO1xuXG4gIGNoZWNrRm9yTXVsdGlwbGVBcGlLZXlzKHByb3BzKTtcbiAgY29uc3Qgc2l0ZUxpc3Q6IHN0cmluZ1tdID0gW1xuICAgIFwiZGF0YWRvZ2hxLmNvbVwiLFxuICAgIFwiZGF0YWRvZ2hxLmV1XCIsXG4gICAgXCJ1czMuZGF0YWRvZ2hxLmNvbVwiLFxuICAgIFwidXM1LmRhdGFkb2docS5jb21cIixcbiAgICBcImRkb2ctZ292LmNvbVwiLFxuICBdO1xuICBpZiAoXG4gICAgcHJvcHMuc2l0ZSAhPT0gdW5kZWZpbmVkICYmXG4gICAgIXNpdGVMaXN0LmluY2x1ZGVzKHByb3BzLnNpdGUudG9Mb3dlckNhc2UoKSkgJiZcbiAgICAhKHByb3BzLnNpdGUuc3RhcnRzV2l0aChcIiR7VG9rZW5bXCIpICYmIHByb3BzLnNpdGUuZW5kc1dpdGgoXCJdfVwiKSlcbiAgKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgXCJXYXJuaW5nOiBJbnZhbGlkIHNpdGUgVVJMLiBNdXN0IGJlIGVpdGhlciBkYXRhZG9naHEuY29tLCBkYXRhZG9naHEuZXUsIHVzMy5kYXRhZG9naHEuY29tLCB1czUuZGF0YWRvZ2hxLmNvbSwgb3IgZGRvZy1nb3YuY29tLlwiLFxuICAgICk7XG4gIH1cblxuICBpZiAoXG4gICAgcHJvcHMuYXBpS2V5ID09PSB1bmRlZmluZWQgJiZcbiAgICBwcm9wcy5hcGlLbXNLZXkgPT09IHVuZGVmaW5lZCAmJlxuICAgIHByb3BzLmFwaUtleVNlY3JldEFybiA9PT0gdW5kZWZpbmVkICYmXG4gICAgcHJvcHMuZmx1c2hNZXRyaWNzVG9Mb2dzID09PSBmYWxzZVxuICApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIldoZW4gYGZsdXNoTWV0cmljc1RvTG9nc2AgaXMgZmFsc2UsIGBhcGlLZXlgLCBgYXBpS2V5U2VjcmV0QXJuYCwgb3IgYGFwaUttc0tleWAgbXVzdCBhbHNvIGJlIHNldC5cIixcbiAgICApO1xuICB9XG4gIGlmIChwcm9wcy5leHRlbnNpb25MYXllclZlcnNpb24gIT09IHVuZGVmaW5lZCkge1xuICAgIGlmIChwcm9wcy5hcGlLZXkgPT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcGlLZXlTZWNyZXRBcm4gPT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcGlLbXNLZXkgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiV2hlbiBgZXh0ZW5zaW9uTGF5ZXJgIGlzIHNldCwgYGFwaUtleWAsIGBhcGlLZXlTZWNyZXRBcm5gLCBvciBgYXBpS21zS2V5YCBtdXN0IGFsc28gYmUgc2V0LlwiKTtcbiAgICB9XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNoZWNrRm9yTXVsdGlwbGVBcGlLZXlzKHByb3BzOiBEYXRhZG9nUHJvcHMpIHtcbiAgbGV0IG11bHRpcGxlQXBpS2V5c01lc3NhZ2U7XG4gIGlmIChwcm9wcy5hcGlLZXkgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcGlLbXNLZXkgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcGlLZXlTZWNyZXRBcm4gIT09IHVuZGVmaW5lZCkge1xuICAgIG11bHRpcGxlQXBpS2V5c01lc3NhZ2UgPSBcImBhcGlLZXlgLCBgYXBpS21zS2V5YCwgYW5kIGBhcGlLZXlTZWNyZXRBcm5gXCI7XG4gIH0gZWxzZSBpZiAocHJvcHMuYXBpS2V5ICE9PSB1bmRlZmluZWQgJiYgcHJvcHMuYXBpS21zS2V5ICE9PSB1bmRlZmluZWQpIHtcbiAgICBtdWx0aXBsZUFwaUtleXNNZXNzYWdlID0gXCJgYXBpS2V5YCBhbmQgYGFwaUttc0tleWBcIjtcbiAgfSBlbHNlIGlmIChwcm9wcy5hcGlLZXkgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcGlLZXlTZWNyZXRBcm4gIT09IHVuZGVmaW5lZCkge1xuICAgIG11bHRpcGxlQXBpS2V5c01lc3NhZ2UgPSBcImBhcGlLZXlgIGFuZCBgYXBpS2V5U2VjcmV0QXJuYFwiO1xuICB9IGVsc2UgaWYgKHByb3BzLmFwaUttc0tleSAhPT0gdW5kZWZpbmVkICYmIHByb3BzLmFwaUtleVNlY3JldEFybiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgbXVsdGlwbGVBcGlLZXlzTWVzc2FnZSA9IFwiYGFwaUttc0tleWAgYW5kIGBhcGlLZXlTZWNyZXRBcm5gXCI7XG4gIH1cblxuICBpZiAobXVsdGlwbGVBcGlLZXlzTWVzc2FnZSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgJHttdWx0aXBsZUFwaUtleXNNZXNzYWdlfSBzaG91bGQgbm90IGJlIHNldCBhdCB0aGUgc2FtZSB0aW1lLmApO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBoYW5kbGVTZXR0aW5nUHJvcERlZmF1bHRzKHByb3BzOiBEYXRhZG9nUHJvcHMpOiBEYXRhZG9nU3RyaWN0UHJvcHMge1xuICBsZXQgYWRkTGF5ZXJzID0gcHJvcHMuYWRkTGF5ZXJzO1xuICBsZXQgZW5hYmxlRGF0YWRvZ1RyYWNpbmcgPSBwcm9wcy5lbmFibGVEYXRhZG9nVHJhY2luZztcbiAgbGV0IGluamVjdExvZ0NvbnRleHQgPSBwcm9wcy5pbmplY3RMb2dDb250ZXh0O1xuICBjb25zdCBsb2dMZXZlbCA9IHByb3BzLmxvZ0xldmVsO1xuICBsZXQgZW5hYmxlRGF0YWRvZ0xvZ3MgPSBwcm9wcy5lbmFibGVEYXRhZG9nTG9ncztcbiAgbGV0IGNhcHR1cmVMYW1iZGFQYXlsb2FkID0gcHJvcHMuY2FwdHVyZUxhbWJkYVBheWxvYWQ7XG5cbiAgaWYgKGFkZExheWVycyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKGBObyB2YWx1ZSBwcm92aWRlZCBmb3IgYWRkTGF5ZXJzLCBkZWZhdWx0aW5nIHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5hZGRMYXllcnN9YCk7XG4gICAgYWRkTGF5ZXJzID0gRGVmYXVsdERhdGFkb2dQcm9wcy5hZGRMYXllcnM7XG4gIH1cbiAgaWYgKGVuYWJsZURhdGFkb2dUcmFjaW5nID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBlbmFibGVEYXRhZG9nVHJhY2luZywgZGVmYXVsdGluZyB0byAke0RlZmF1bHREYXRhZG9nUHJvcHMuZW5hYmxlRGF0YWRvZ1RyYWNpbmd9YCk7XG4gICAgZW5hYmxlRGF0YWRvZ1RyYWNpbmcgPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZURhdGFkb2dUcmFjaW5nO1xuICB9XG4gIGlmIChpbmplY3RMb2dDb250ZXh0ID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBpbmplY3RMb2dDb250ZXh0LCBkZWZhdWx0aW5nIHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5pbmplY3RMb2dDb250ZXh0fWApO1xuICAgIGluamVjdExvZ0NvbnRleHQgPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLmluamVjdExvZ0NvbnRleHQ7XG4gIH1cbiAgaWYgKGxvZ0xldmVsID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBsb2dMZXZlbGApO1xuICB9XG4gIGlmIChlbmFibGVEYXRhZG9nTG9ncyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKGBObyB2YWx1ZSBwcm92aWRlZCBmb3IgZW5hYmxlRGF0YWRvZ0xvZ3MsIGRlZmF1bHRpbmcgdG8gJHtEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZURhdGFkb2dMb2dzfWApO1xuICAgIGVuYWJsZURhdGFkb2dMb2dzID0gRGVmYXVsdERhdGFkb2dQcm9wcy5lbmFibGVEYXRhZG9nTG9ncztcbiAgfVxuICBpZiAoY2FwdHVyZUxhbWJkYVBheWxvYWQgPT09IHVuZGVmaW5lZCkge1xuICAgIGxvZy5kZWJ1ZyhgTm8gdmFsdWUgcHJvdmlkZWQgZm9yIGNhcHR1cmVMYW1iZGFQYXlsb2FkLCBkZWZhdWx0IHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5jYXB0dXJlTGFtYmRhUGF5bG9hZH1gKTtcbiAgICBjYXB0dXJlTGFtYmRhUGF5bG9hZCA9IERlZmF1bHREYXRhZG9nUHJvcHMuY2FwdHVyZUxhbWJkYVBheWxvYWQ7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGFkZExheWVyczogYWRkTGF5ZXJzLFxuICAgIGVuYWJsZURhdGFkb2dUcmFjaW5nOiBlbmFibGVEYXRhZG9nVHJhY2luZyxcbiAgICBpbmplY3RMb2dDb250ZXh0OiBpbmplY3RMb2dDb250ZXh0LFxuICAgIGxvZ0xldmVsOiBsb2dMZXZlbCxcbiAgICBlbmFibGVEYXRhZG9nTG9nczogZW5hYmxlRGF0YWRvZ0xvZ3MsXG4gICAgY2FwdHVyZUxhbWJkYVBheWxvYWQ6IGNhcHR1cmVMYW1iZGFQYXlsb2FkLFxuICB9O1xufVxuIl19