import numpy as np


class TestInterpolatorLoadValues:
    """
    Base class for loading values for a 3D sinc function test.

    Storage for interpolation and extrapolation data to be test against.
    These data are saved to 12 significant figures. self.data is generated by applying the sinc function to an
    (NB_X = 10, NB_Y = 10, NB_Z = 10) dimension xyz data, which is used as the spline knots. The precalc_interpolation
    is setup for cubic, and linear interpolation separately using the generate_3d_splines script moved into Cherab.
    The data for nearest extrapolation is also generated through Cherab, but linear extrapolation is handled
    differently (Cherab duplicates the edge border to calculate derivatives, these interpolators handle each edge
    as a special case). Interpolation data is generated on a 19X19X19 grid to overlap consistently with the spline
    knots, but have distinct interpolation points to be sampled in between spline knots. The extrapolation positions
    are 1 outside from the middle of each edge of the 3D grid in every direction. Linear extrapolation from both
    interpolator types were generated from these interpolators on 12/07/2021, so are not mathematical tests and
    just highlight if something changes unexpectedly.
    """
    def __init__(self):
        # Define in setup_cubic or setup_linear
        self.precalc_interpolation = None

        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = None

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = None

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = None


class TestInterpolatorLoadNormalValues(TestInterpolatorLoadValues):
    """
    Loading normal sized values for a 3D sinc function test.

    For description of data storage, see TestInterpolatorLoadValues.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        self.data: np.array = np.array(
            [[[-5.750924620219E-02, -1.378569409163E-02, 4.356156928097E-02, 6.830161067344E-02,
         7.076114333481E-02, 7.076114333481E-02, 6.830161067344E-02, 4.356156928097E-02,
         -1.378569409163E-02, -5.750924620219E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [-5.750924620219E-02, -1.378569409163E-02, 4.356156928097E-02, 6.830161067344E-02,
         7.076114333481E-02, 7.076114333481E-02, 6.830161067344E-02, 4.356156928097E-02,
         -1.378569409163E-02, -5.750924620219E-02]],

       [[-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [5.864644239018E-02, 5.012671341215E-02, -3.233945733892E-02, -8.287206522483E-02,
         -9.132287823503E-02, -9.132287823503E-02, -8.287206522483E-02, -3.233945733892E-02,
         5.012671341215E-02, 5.864644239018E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [5.864644239018E-02, 5.012671341215E-02, -3.233945733892E-02, -8.287206522483E-02,
         -9.132287823503E-02, -9.132287823503E-02, -8.287206522483E-02, -3.233945733892E-02,
         5.012671341215E-02, 5.864644239018E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02]],

       [[4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02]],

       [[6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-9.132287823503E-02, 4.831031939387E-02, 1.096062925707E-01, -9.690247379716E-02,
         -2.087231323860E-01, -2.087231323860E-01, -9.690247379716E-02, 1.096062925707E-01,
         4.831031939387E-02, -9.132287823503E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-9.132287823503E-02, 4.831031939387E-02, 1.096062925707E-01, -9.690247379716E-02,
         -2.087231323860E-01, -2.087231323860E-01, -9.690247379716E-02, 1.096062925707E-01,
         4.831031939387E-02, -9.132287823503E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02]],

       [[7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-5.367000806700E-02, 1.276578283168E-01, -9.690247379716E-02, -1.307271192057E-01,
         4.983093384723E-01, 4.983093384723E-01, -1.307271192057E-01, -9.690247379716E-02,
         1.276578283168E-01, -5.367000806700E-02],
        [-5.367000806700E-02, 1.276578283168E-01, -9.690247379716E-02, -1.307271192057E-01,
         4.983093384723E-01, 4.983093384723E-01, -1.307271192057E-01, -9.690247379716E-02,
         1.276578283168E-01, -5.367000806700E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02]],

       [[7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-5.367000806700E-02, 1.276578283168E-01, -9.690247379716E-02, -1.307271192057E-01,
         4.983093384723E-01, 4.983093384723E-01, -1.307271192057E-01, -9.690247379716E-02,
         1.276578283168E-01, -5.367000806700E-02],
        [-5.367000806700E-02, 1.276578283168E-01, -9.690247379716E-02, -1.307271192057E-01,
         4.983093384723E-01, 4.983093384723E-01, -1.307271192057E-01, -9.690247379716E-02,
         1.276578283168E-01, -5.367000806700E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02]],

       [[6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-9.132287823503E-02, 4.831031939387E-02, 1.096062925707E-01, -9.690247379716E-02,
         -2.087231323860E-01, -2.087231323860E-01, -9.690247379716E-02, 1.096062925707E-01,
         4.831031939387E-02, -9.132287823503E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-8.226118588934E-02, 9.999127997672E-02, 3.130524741593E-02, -2.087231323860E-01,
         -1.307271192057E-01, -1.307271192057E-01, -2.087231323860E-01, 3.130524741593E-02,
         9.999127997672E-02, -8.226118588934E-02],
        [-9.132287823503E-02, 4.831031939387E-02, 1.096062925707E-01, -9.690247379716E-02,
         -2.087231323860E-01, -2.087231323860E-01, -9.690247379716E-02, 1.096062925707E-01,
         4.831031939387E-02, -9.132287823503E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02]],

       [[4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-8.287206522483E-02, -7.601370334387E-03, 1.276578283168E-01, 3.130524741593E-02,
         -9.690247379716E-02, -9.690247379716E-02, 3.130524741593E-02, 1.276578283168E-01,
         -7.601370334387E-03, -8.287206522483E-02],
        [-6.137786872162E-02, -5.367000806700E-02, 9.999127997672E-02, 1.096062925707E-01,
         3.130524741593E-02, 3.130524741593E-02, 1.096062925707E-01, 9.999127997672E-02,
         -5.367000806700E-02, -6.137786872162E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334386E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02]],

       [[-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [5.864644239018E-02, 5.012671341215E-02, -3.233945733892E-02, -8.287206522483E-02,
         -9.132287823503E-02, -9.132287823503E-02, -8.287206522483E-02, -3.233945733892E-02,
         5.012671341215E-02, 5.864644239018E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [-1.191055434292E-03, -9.132287823503E-02, -7.601370334387E-03, 9.999127997672E-02,
         1.276578283168E-01, 1.276578283168E-01, 9.999127997672E-02, -7.601370334387E-03,
         -9.132287823503E-02, -1.191055434292E-03],
        [2.743644194787E-02, -8.287206522483E-02, -5.367000806700E-02, 4.831031939387E-02,
         9.999127997672E-02, 9.999127997672E-02, 4.831031939387E-02, -5.367000806700E-02,
         -8.287206522483E-02, 2.743644194787E-02],
        [6.482091617167E-02, -3.233945733892E-02, -9.132287823503E-02, -5.367000806700E-02,
         -7.601370334387E-03, -7.601370334387E-03, -5.367000806700E-02, -9.132287823503E-02,
         -3.233945733892E-02, 6.482091617167E-02],
        [5.864644239018E-02, 5.012671341215E-02, -3.233945733892E-02, -8.287206522483E-02,
         -9.132287823503E-02, -9.132287823503E-02, -8.287206522483E-02, -3.233945733892E-02,
         5.012671341215E-02, 5.864644239018E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02]],

       [[-5.750924620219E-02, -1.378569409163E-02, 4.356156928097E-02, 6.830161067344E-02,
         7.076114333481E-02, 7.076114333481E-02, 6.830161067344E-02, 4.356156928097E-02,
         -1.378569409163E-02, -5.750924620219E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [7.076114333481E-02, -1.191055434292E-03, -8.287206522483E-02, -8.226118588934E-02,
         -5.367000806700E-02, -5.367000806700E-02, -8.226118588934E-02, -8.287206522483E-02,
         -1.191055434292E-03, 7.076114333481E-02],
        [6.830161067344E-02, 2.743644194787E-02, -6.137786872162E-02, -9.132287823503E-02,
         -8.226118588934E-02, -8.226118588934E-02, -9.132287823503E-02, -6.137786872162E-02,
         2.743644194787E-02, 6.830161067344E-02],
        [4.356156928097E-02, 6.482091617167E-02, -1.191055434292E-03, -6.137786872162E-02,
         -8.287206522483E-02, -8.287206522483E-02, -6.137786872162E-02, -1.191055434292E-03,
         6.482091617167E-02, 4.356156928097E-02],
        [-1.378569409163E-02, 5.864644239018E-02, 6.482091617167E-02, 2.743644194787E-02,
         -1.191055434292E-03, -1.191055434292E-03, 2.743644194787E-02, 6.482091617167E-02,
         5.864644239018E-02, -1.378569409163E-02],
        [-5.750924620219E-02, -1.378569409163E-02, 4.356156928097E-02, 6.830161067344E-02,
         7.076114333481E-02, 7.076114333481E-02, 6.830161067344E-02, 4.356156928097E-02,
         -1.378569409163E-02, -5.750924620219E-02]]], dtype=np.float64)


    def setup_cubic(self):
        self.precalc_interpolation = np.array(
            [[[-5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02,
               4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02,
               7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02,
               6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02,
               -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02],
              [-3.649895211738E-02, -3.088974011050E-03, 2.657147806335E-02, 4.759923176017E-02,
               5.964570012010E-02, 5.828995984419E-02, 5.086584768151E-02, 4.258073038267E-02,
               3.539309463878E-02, 3.345900050891E-02, 3.539309463946E-02, 4.258073038068E-02,
               5.086584768574E-02, 5.828995985165E-02, 5.964570013079E-02, 4.759923175448E-02,
               2.657147805303E-02, -3.088974030447E-03, -3.649895213245E-02],
              [-1.378569410836E-02, 2.657147805303E-02, 5.864644238984E-02, 6.859721745322E-02,
               6.482091617241E-02, 4.830380225823E-02, 2.743644194715E-02, 1.078616361697E-02,
               -1.191055434163E-03, -4.769492607025E-03, -1.191055434376E-03, 1.078616361760E-02,
               2.743644194653E-02, 4.830380225562E-02, 6.482091616058E-02, 6.859721744395E-02,
               5.864644239405E-02, 2.657147806556E-02, -1.378569409104E-02],
              [1.607440700266E-02, 4.759923175276E-02, 6.859721744999E-02, 5.944385073981E-02,
               3.690531538030E-02, 9.452397728552E-03, -1.765322333815E-02, -3.551989669485E-02,
               -4.656675271099E-02, -5.018094388273E-02, -4.656675271113E-02, -3.551989669440E-02,
               -1.765322333876E-02, 9.452397726093E-03, 3.690531537103E-02, 5.944385073178E-02,
               6.859721745415E-02, 4.759923176233E-02, 1.607440701568E-02],
              [4.356156929915E-02, 5.964570014935E-02, 6.482091617121E-02, 3.690531537805E-02,
               -1.191055430596E-03, -3.406682302067E-02, -6.137786872165E-02, -7.588664280371E-02,
               -8.287206522451E-02, -8.555883978764E-02, -8.287206522502E-02, -7.588664280388E-02,
               -6.137786872209E-02, -3.406682302217E-02, -1.191055436745E-03, 3.690531537487E-02,
               6.482091617227E-02, 5.964570014017E-02, 4.356156929263E-02],
              [5.936207315787E-02, 5.828995986152E-02, 4.830380225713E-02, 9.452397726372E-03,
               -3.406682301951E-02, -6.297844907249E-02, -8.246762366680E-02, -8.898818051517E-02,
               -8.945863728280E-02, -9.033251398485E-02, -8.945863728310E-02, -8.898818051543E-02,
               -8.246762366704E-02, -6.297844907350E-02, -3.406682302335E-02, 9.452397724649E-03,
               4.830380225745E-02, 5.828995985726E-02, 5.936207315493E-02],
              [6.830161067188E-02, 5.086584768016E-02, 2.743644194736E-02, -1.765322333716E-02,
               -6.137786872188E-02, -8.246762366694E-02, -9.132287823500E-02, -8.866359515673E-02,
               -8.226118588935E-02, -8.112847434613E-02, -8.226118588939E-02, -8.866359515687E-02,
               -9.132287823508E-02, -8.246762366680E-02, -6.137786872115E-02, -1.765322333746E-02,
               2.743644194769E-02, 5.086584768251E-02, 6.830161067455E-02],
              [7.107762959038E-02, 4.258073038138E-02, 1.078616361751E-02, -3.551989669405E-02,
               -7.588664280397E-02, -8.898818051528E-02, -8.866359515675E-02, -7.909406561030E-02,
               -6.792741701972E-02, -6.533539475257E-02, -6.792741701977E-02, -7.909406561041E-02,
               -8.866359515684E-02, -8.898818051519E-02, -7.588664280371E-02, -3.551989669442E-02,
               1.078616361837E-02, 4.258073038353E-02, 7.107762959247E-02],
              [7.076114333566E-02, 3.539309463963E-02, -1.191055434244E-03, -4.656675271103E-02,
               -8.287206522482E-02, -8.945863728286E-02, -8.226118588935E-02, -6.792741701970E-02,
               -5.367000806700E-02, -5.009611083921E-02, -5.367000806699E-02, -6.792741701970E-02,
               -8.226118588946E-02, -8.945863728288E-02, -8.287206522469E-02, -4.656675271102E-02,
               -1.191055434066E-03, 3.539309463997E-02, 7.076114333521E-02],
              [7.106858491835E-02, 3.345900050925E-02, -4.769492607011E-03, -5.018094388270E-02,
               -8.555883978772E-02, -9.033251398488E-02, -8.112847434614E-02, -6.533539475257E-02,
               -5.009611083922E-02, -4.621706540085E-02, -5.009611083921E-02, -6.533539475257E-02,
               -8.112847434625E-02, -9.033251398490E-02, -8.555883978761E-02, -5.018094388272E-02,
               -4.769492606812E-03, 3.345900050959E-02, 7.106858491790E-02],
              [7.076114333396E-02, 3.539309463867E-02, -1.191055434516E-03, -4.656675271126E-02,
               -8.287206522498E-02, -8.945863728301E-02, -8.226118588933E-02, -6.792741701967E-02,
               -5.367000806702E-02, -5.009611083921E-02, -5.367000806697E-02, -6.792741701964E-02,
               -8.226118588937E-02, -8.945863728288E-02, -8.287206522485E-02, -4.656675271062E-02,
               -1.191055434693E-03, 3.539309463898E-02, 7.076114333498E-02],
              [7.107762959019E-02, 4.258073038204E-02, 1.078616361737E-02, -3.551989669443E-02,
               -7.588664280384E-02, -8.898818051525E-02, -8.866359515677E-02, -7.909406561028E-02,
               -6.792741701971E-02, -6.533539475258E-02, -6.792741701968E-02, -7.909406561026E-02,
               -8.866359515678E-02, -8.898818051514E-02, -7.588664280389E-02, -3.551989669383E-02,
               1.078616361796E-02, 4.258073038240E-02, 7.107762959115E-02],
              [6.830161067588E-02, 5.086584768557E-02, 2.743644194734E-02, -1.765322333791E-02,
               -6.137786872193E-02, -8.246762366704E-02, -9.132287823511E-02, -8.866359515673E-02,
               -8.226118588927E-02, -8.112847434610E-02, -8.226118588891E-02, -8.866359515666E-02,
               -9.132287823417E-02, -8.246762366571E-02, -6.137786872110E-02, -1.765322333624E-02,
               2.743644195026E-02, 5.086584767606E-02, 6.830161066853E-02],
              [5.936207315005E-02, 5.828995985678E-02, 4.830380225714E-02, 9.452397727007E-03,
               -3.406682302249E-02, -6.297844907372E-02, -8.246762366682E-02, -8.898818051514E-02,
               -8.945863728277E-02, -9.033251398490E-02, -8.945863728260E-02, -8.898818051505E-02,
               -8.246762366635E-02, -6.297844907261E-02, -3.406682302059E-02, 9.452397730918E-03,
               4.830380225822E-02, 5.828995984248E-02, 5.936207314396E-02],
              [4.356156927876E-02, 5.964570012580E-02, 6.482091617804E-02, 3.690531538273E-02,
               -1.191055434191E-03, -3.406682302106E-02, -6.137786872090E-02, -7.588664280368E-02,
               -8.287206522483E-02, -8.555883978772E-02, -8.287206522455E-02, -7.588664280379E-02,
               -6.137786872198E-02, -3.406682302096E-02, -1.191055432770E-03, 3.690531537512E-02,
               6.482091616700E-02, 5.964570013270E-02, 4.356156928137E-02],
              [1.607440701081E-02, 4.759923175430E-02, 6.859721745905E-02, 5.944385074129E-02,
               3.690531537985E-02, 9.452397728154E-03, -1.765322333673E-02, -3.551989669413E-02,
               -4.656675271111E-02, -5.018094388280E-02, -4.656675271075E-02, -3.551989669466E-02,
               -1.765322333821E-02, 9.452397728822E-03, 3.690531537920E-02, 5.944385072904E-02,
               6.859721744377E-02, 4.759923176384E-02, 1.607440701473E-02],
              [-1.378569400687E-02, 2.657147812158E-02, 5.864644239897E-02, 6.859721744841E-02,
               6.482091617160E-02, 4.830380225513E-02, 2.743644194722E-02, 1.078616361659E-02,
               -1.191055434532E-03, -4.769492607409E-03, -1.191055434106E-03, 1.078616361715E-02,
               2.743644195107E-02, 4.830380225332E-02, 6.482091617384E-02, 6.859721744401E-02,
               5.864644236959E-02, 2.657147806131E-02, -1.378569410514E-02],
              [-3.649895199161E-02, -3.088973940848E-03, 2.657147807892E-02, 4.759923176151E-02,
               5.964570013216E-02, 5.828995984835E-02, 5.086584768191E-02, 4.258073038125E-02,
               3.539309463849E-02, 3.345900050823E-02, 3.539309463952E-02, 4.258073038187E-02,
               5.086584768429E-02, 5.828995984560E-02, 5.964570013445E-02, 4.759923175558E-02,
               2.657147804251E-02, -3.088974026880E-03, -3.649895211890E-02],
              [-5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02,
               4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02,
               7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02,
               6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02,
               -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02]],

             [[-3.649895212153E-02, -3.088974014866E-03, 2.657147806395E-02, 4.759923176017E-02,
               5.964570012047E-02, 5.828995984426E-02, 5.086584768171E-02, 4.258073038296E-02,
               3.539309463846E-02, 3.345900050898E-02, 3.539309463962E-02, 4.258073038087E-02,
               5.086584768556E-02, 5.828995985127E-02, 5.964570013211E-02, 4.759923175469E-02,
               2.657147805074E-02, -3.088974026225E-03, -3.649895213205E-02],
              [-3.088974026933E-03, 2.602783331572E-02, 4.763924885384E-02, 4.831970810151E-02,
               3.832433747192E-02, 2.170304544150E-02, 3.645921392555E-03, -8.922263380603E-03,
               -1.715564614388E-02, -1.975584208556E-02, -1.715564614318E-02, -8.922263382291E-03,
               3.645921395820E-03, 2.170304544736E-02, 3.832433748101E-02, 4.831970809583E-02,
               4.763924883944E-02, 2.602783329743E-02, -3.088974039496E-03],
              [2.657147805463E-02, 4.763924884444E-02, 5.900823051145E-02, 4.160199046097E-02,
               1.385466987872E-02, -1.282052492368E-02, -3.643722190969E-02, -4.992325419372E-02,
               -5.712280000340E-02, -5.970849726526E-02, -5.712280000356E-02, -4.992325419316E-02,
               -3.643722191032E-02, -1.282052492602E-02, 1.385466986842E-02, 4.160199045518E-02,
               5.900823051442E-02, 4.763924885469E-02, 2.657147806777E-02],
              [4.759923175432E-02, 4.831970809471E-02, 4.160199045919E-02, 1.073979771818E-02,
               -2.390080272802E-02, -4.590027524467E-02, -6.012530565211E-02, -6.404296523560E-02,
               -6.343256195290E-02, -6.384596899061E-02, -6.343256195302E-02, -6.404296523519E-02,
               -6.012530565270E-02, -4.590027524684E-02, -2.390080273604E-02, 1.073979771297E-02,
               4.160199046219E-02, 4.831970810258E-02, 4.759923176423E-02],
              [5.964570014721E-02, 3.832433749599E-02, 1.385466987765E-02, -2.390080272983E-02,
               -5.712280000025E-02, -6.742425744267E-02, -6.664602760617E-02, -5.841121516060E-02,
               -4.898599926438E-02, -4.677849572190E-02, -4.898599926486E-02, -5.841121516075E-02,
               -6.664602760645E-02, -6.742425744391E-02, -5.712280000576E-02, -2.390080273272E-02,
               1.385466987862E-02, 3.832433748830E-02, 5.964570014214E-02],
              [5.828995986065E-02, 2.170304545593E-02, -1.282052492455E-02, -4.590027524639E-02,
               -6.742425744165E-02, -6.217010895109E-02, -4.608859723580E-02, -2.874753183622E-02,
               -1.407342396249E-02, -1.007152730337E-02, -1.407342396279E-02, -2.874753183645E-02,
               -4.608859723600E-02, -6.217010895194E-02, -6.742425744518E-02, -4.590027524797E-02,
               -1.282052492414E-02, 2.170304545240E-02, 5.828995985828E-02],
              [5.086584768094E-02, 3.645921391500E-03, -3.643722190959E-02, -6.012530565136E-02,
               -6.664602760632E-02, -4.608859723587E-02, -1.661020289226E-02, 7.096489196192E-03,
               2.454870319535E-02, 2.969356645633E-02, 2.454870319577E-02, 7.096489196257E-03,
               -1.661020289265E-02, -4.608859723611E-02, -6.664602760470E-02, -6.012530565117E-02,
               -3.643722190926E-02, 3.645921393057E-03, 5.086584768228E-02],
              [4.258073038191E-02, -8.922263381645E-03, -4.992325419328E-02, -6.404296523495E-02,
               -5.841121516077E-02, -2.874753183626E-02, 7.096489196180E-03, 3.166809106638E-02,
               4.805122989193E-02, 5.317057247892E-02, 4.805122989242E-02, 3.166809106655E-02,
               7.096489195911E-03, -2.874753183648E-02, -5.841121515962E-02, -6.404296523480E-02,
               -4.992325419243E-02, -8.922263380130E-03, 4.258073038292E-02],
              [3.539309463981E-02, -1.715564614309E-02, -5.712280000345E-02, -6.343256195294E-02,
               -4.898599926464E-02, -1.407342396253E-02, 2.454870319537E-02, 4.805122989196E-02,
               6.236191878104E-02, 6.708857072924E-02, 6.236191878100E-02, 4.805122989195E-02,
               2.454870319530E-02, -1.407342396255E-02, -4.898599926445E-02, -6.343256195296E-02,
               -5.712280000328E-02, -1.715564614275E-02, 3.539309463949E-02],
              [3.345900050944E-02, -1.975584208517E-02, -5.970849726525E-02, -6.384596899059E-02,
               -4.677849572196E-02, -1.007152730339E-02, 2.969356645633E-02, 5.317057247894E-02,
               6.708857072924E-02, 7.176294626336E-02, 6.708857072918E-02, 5.317057247891E-02,
               2.969356645624E-02, -1.007152730341E-02, -4.677849572178E-02, -6.384596899057E-02,
               -5.970849726517E-02, -1.975584208489E-02, 3.345900050910E-02],
              [3.539309463820E-02, -1.715564614410E-02, -5.712280000372E-02, -6.343256195318E-02,
               -4.898599926472E-02, -1.407342396264E-02, 2.454870319557E-02, 4.805122989204E-02,
               6.236191878102E-02, 6.708857072922E-02, 6.236191878037E-02, 4.805122989154E-02,
               2.454870319450E-02, -1.407342396208E-02, -4.898599926477E-02, -6.343256195336E-02,
               -5.712280000633E-02, -1.715564614648E-02, 3.539309463608E-02],
              [4.258073038148E-02, -8.922263381353E-03, -4.992325419341E-02, -6.404296523526E-02,
               -5.841121516060E-02, -2.874753183621E-02, 7.096489196432E-03, 3.166809106626E-02,
               4.805122989195E-02, 5.317057247891E-02, 4.805122989137E-02, 3.166809106564E-02,
               7.096489195510E-03, -2.874753183566E-02, -5.841121516106E-02, -6.404296523652E-02,
               -4.992325419413E-02, -8.922263382368E-03, 4.258073038075E-02],
              [5.086584768413E-02, 3.645921396046E-03, -3.643722190940E-02, -6.012530565188E-02,
               -6.664602760631E-02, -4.608859723592E-02, -1.661020289287E-02, 7.096489195857E-03,
               2.454870319542E-02, 2.969356645637E-02, 2.454870319517E-02, 7.096489196599E-03,
               -1.661020289137E-02, -4.608859723587E-02, -6.664602760916E-02, -6.012530565300E-02,
               -3.643722191072E-02, 3.645921382036E-03, 5.086584766974E-02],
              [5.828995985421E-02, 2.170304545215E-02, -1.282052492435E-02, -4.590027524586E-02,
               -6.742425744423E-02, -6.217010895209E-02, -4.608859723617E-02, -2.874753183650E-02,
               -1.407342396246E-02, -1.007152730340E-02, -1.407342396299E-02, -2.874753183551E-02,
               -4.608859723571E-02, -6.217010895196E-02, -6.742425744713E-02, -4.590027524706E-02,
               -1.282052492962E-02, 2.170304543419E-02, 5.828995984365E-02],
              [5.964570013105E-02, 3.832433747629E-02, 1.385466988353E-02, -2.390080272606E-02,
               -5.712280000348E-02, -6.742425744308E-02, -6.664602760540E-02, -5.841121516024E-02,
               -4.898599926468E-02, -4.677849572192E-02, -4.898599926912E-02, -5.841121516272E-02,
               -6.664602760528E-02, -6.742425744799E-02, -5.712280000045E-02, -2.390080272944E-02,
               1.385466987522E-02, 3.832433748143E-02, 5.964570013002E-02],
              [4.759923176137E-02, 4.831970809616E-02, 4.160199046726E-02, 1.073979772027E-02,
               -2.390080272833E-02, -4.590027524496E-02, -6.012530565130E-02, -6.404296523443E-02,
               -6.343256195316E-02, -6.384596899065E-02, -6.343256195646E-02, -6.404296523730E-02,
               -6.012530564855E-02, -4.590027524715E-02, -2.390080273398E-02, 1.073979770736E-02,
               4.160199046212E-02, 4.831970810938E-02, 4.759923176577E-02],
              [2.657147814155E-02, 4.763924890613E-02, 5.900823051923E-02, 4.160199045831E-02,
               1.385466987745E-02, -1.282052492658E-02, -3.643722190957E-02, -4.992325419389E-02,
               -5.712280000360E-02, -5.970849726563E-02, -5.712280000294E-02, -4.992325419386E-02,
               -3.643722191160E-02, -1.282052493424E-02, 1.385466988786E-02, 4.160199046293E-02,
               5.900823048818E-02, 4.763924884901E-02, 2.657147805003E-02],
              [-3.088973915079E-03, 2.602783337651E-02, 4.763924886642E-02, 4.831970810278E-02,
               3.832433748121E-02, 2.170304544467E-02, 3.645921392930E-03, -8.922263381936E-03,
               -1.715564614398E-02, -1.975584208607E-02, -1.715564614300E-02, -8.922263381171E-03,
               3.645921393510E-03, 2.170304544010E-02, 3.832433748849E-02, 4.831970810510E-02,
               4.763924882993E-02, 2.602783329981E-02, -3.088974027303E-03],
              [-3.649895198230E-02, -3.088973932833E-03, 2.657147807884E-02, 4.759923176142E-02,
               5.964570013161E-02, 5.828995984817E-02, 5.086584768120E-02, 4.258073038038E-02,
               3.539309463881E-02, 3.345900050856E-02, 3.539309463970E-02, 4.258073038310E-02,
               5.086584768503E-02, 5.828995984593E-02, 5.964570013567E-02, 4.759923176153E-02,
               2.657147803679E-02, -3.088974031520E-03, -3.649895212067E-02]],

             [[-1.378569409171E-02, 2.657147806499E-02, 5.864644238892E-02, 6.859721743952E-02,
               6.482091617400E-02, 4.830380225833E-02, 2.743644194838E-02, 1.078616361776E-02,
               -1.191055434395E-03, -4.769492607174E-03, -1.191055434461E-03, 1.078616361809E-02,
               2.743644194982E-02, 4.830380226187E-02, 6.482091618142E-02, 6.859721746023E-02,
               5.864644239491E-02, 2.657147806935E-02, -1.378569409693E-02],
              [2.657147806897E-02, 4.763924885560E-02, 5.900823050962E-02, 4.160199044942E-02,
               1.385466988014E-02, -1.282052492325E-02, -3.643722190880E-02, -4.992325419310E-02,
               -5.712280000353E-02, -5.970849726541E-02, -5.712280000366E-02, -4.992325419278E-02,
               -3.643722190739E-02, -1.282052492051E-02, 1.385466988635E-02, 4.160199046784E-02,
               5.900823051497E-02, 4.763924885845E-02, 2.657147806331E-02],
              [5.864644238220E-02, 5.900823050761E-02, 5.012671342054E-02, 1.151943297231E-02,
               -3.233945733901E-02, -6.223172114048E-02, -8.287206522478E-02, -9.025575972308E-02,
               -9.132287823459E-02, -9.237922986108E-02, -9.132287823494E-02, -9.025575972284E-02,
               -8.287206522443E-02, -6.223172114022E-02, -3.233945733668E-02, 1.151943297527E-02,
               5.012671341158E-02, 5.900823050668E-02, 5.864644238279E-02],
              [6.859721744398E-02, 4.160199045603E-02, 1.151943297457E-02, -3.223101197051E-02,
               -7.025699551714E-02, -8.224151800376E-02, -8.153908881051E-02, -7.238462716334E-02,
               -6.181990385399E-02, -5.935500573459E-02, -6.181990385443E-02, -7.238462716337E-02,
               -8.153908881001E-02, -8.224151800321E-02, -7.025699551532E-02, -3.223101196840E-02,
               1.151943296688E-02, 4.160199045441E-02, 6.859721744240E-02],
              [6.482091617052E-02, 1.385466987527E-02, -3.233945733906E-02, -7.025699551714E-02,
               -9.132287823488E-02, -7.906219681523E-02, -5.367000806679E-02, -2.828238481530E-02,
               -7.601370334360E-03, -1.842790617735E-03, -7.601370334002E-03, -2.828238481504E-02,
               -5.367000806747E-02, -7.906219681553E-02, -9.132287823518E-02, -7.025699551669E-02,
               -3.233945734237E-02, 1.385466987728E-02, 6.482091616928E-02],
              [4.830380225667E-02, -1.282052492603E-02, -6.223172114080E-02, -8.224151800395E-02,
               -7.906219681517E-02, -4.598683888723E-02, -4.084775800584E-03, 2.749289580984E-02,
               4.969838979374E-02, 5.642128549303E-02, 4.969838979409E-02, 2.749289581053E-02,
               -4.084775801535E-03, -4.598683888861E-02, -7.906219681523E-02, -8.224151800365E-02,
               -6.223172114109E-02, -1.282052492323E-02, 4.830380225714E-02],
              [2.743644194828E-02, -3.643722190875E-02, -8.287206522495E-02, -8.153908881065E-02,
               -5.367000806734E-02, -4.084775800988E-03, 4.831031939408E-02, 8.052457015186E-02,
               9.999127997664E-02, 1.064514000496E-01, 9.999127997744E-02, 8.052457015225E-02,
               4.831031939493E-02, -4.084775800583E-03, -5.367000806771E-02, -8.153908881159E-02,
               -8.287206522110E-02, -3.643722190737E-02, 2.743644194786E-02],
              [1.078616361804E-02, -4.992325419272E-02, -9.025575972294E-02, -7.238462716333E-02,
               -2.828238481553E-02, 2.749289580958E-02, 8.052457015182E-02, 1.073372671571E-01,
               1.205490947911E-01, 1.255521603711E-01, 1.205490947921E-01, 1.073372671577E-01,
               8.052457015369E-02, 2.749289581208E-02, -2.828238481642E-02, -7.238462716501E-02,
               -9.025575971987E-02, -4.992325419101E-02, 1.078616361864E-02],
              [-1.191055434311E-03, -5.712280000347E-02, -9.132287823459E-02, -6.181990385417E-02,
               -7.601370334354E-03, 4.969838979367E-02, 9.999127997676E-02, 1.205490947914E-01,
               1.276578283168E-01, 1.311161468594E-01, 1.276578283167E-01, 1.205490947912E-01,
               9.999127997706E-02, 4.969838979344E-02, -7.601370334407E-03, -6.181990385433E-02,
               -9.132287823419E-02, -5.712280000376E-02, -1.191055435457E-03],
              [-4.769492607076E-03, -5.970849726526E-02, -9.237922986086E-02, -5.935500573463E-02,
               -1.842790617800E-03, 5.642128549294E-02, 1.064514000496E-01, 1.255521603713E-01,
               1.311161468594E-01, 1.341992402106E-01, 1.311161468592E-01, 1.255521603712E-01,
               1.064514000500E-01, 5.642128549269E-02, -1.842790617926E-03, -5.935500573482E-02,
               -9.237922985975E-02, -5.970849726490E-02, -4.769492607615E-03],
              [-1.191055433243E-03, -5.712280000249E-02, -9.132287823467E-02, -6.181990385410E-02,
               -7.601370333846E-03, 4.969838979364E-02, 9.999127997620E-02, 1.205490947917E-01,
               1.276578283163E-01, 1.311161468590E-01, 1.276578283164E-01, 1.205490947922E-01,
               9.999127998282E-02, 4.969838979262E-02, -7.601370343455E-03, -6.181990386637E-02,
               -9.132287821433E-02, -5.712279999853E-02, -1.191055444137E-03],
              [1.078616361884E-02, -4.992325419194E-02, -9.025575972236E-02, -7.238462716312E-02,
               -2.828238481534E-02, 2.749289580972E-02, 8.052457015134E-02, 1.073372671572E-01,
               1.205490947909E-01, 1.255521603707E-01, 1.205490947900E-01, 1.073372671556E-01,
               8.052457014468E-02, 2.749289580503E-02, -2.828238481833E-02, -7.238462716743E-02,
               -9.025575974530E-02, -4.992325420463E-02, 1.078616361705E-02],
              [2.743644194669E-02, -3.643722190978E-02, -8.287206522568E-02, -8.153908881053E-02,
               -5.367000806571E-02, -4.084775798941E-03, 4.831031939615E-02, 8.052457015211E-02,
               9.999127997727E-02, 1.064514000499E-01, 9.999127997832E-02, 8.052457015240E-02,
               4.831031938557E-02, -4.084775800398E-03, -5.367000806970E-02, -8.153908881292E-02,
               -8.287206524633E-02, -3.643722191817E-02, 2.743644195000E-02],
              [4.830380225634E-02, -1.282052492495E-02, -6.223172114188E-02, -8.224151800368E-02,
               -7.906219681317E-02, -4.598683888499E-02, -4.084775799829E-03, 2.749289581013E-02,
               4.969838979381E-02, 5.642128549280E-02, 4.969838980279E-02, 2.749289581274E-02,
               -4.084775827958E-03, -4.598683890825E-02, -7.906219681778E-02, -8.224151800320E-02,
               -6.223172114463E-02, -1.282052493122E-02, 4.830380224711E-02],
              [6.482091617827E-02, 1.385466988168E-02, -3.233945733826E-02, -7.025699551640E-02,
               -9.132287823611E-02, -7.906219681637E-02, -5.367000806577E-02, -2.828238481543E-02,
               -7.601370334315E-03, -1.842790617956E-03, -7.601370331822E-03, -2.828238481478E-02,
               -5.367000806594E-02, -7.906219681535E-02, -9.132287824864E-02, -7.025699552772E-02,
               -3.233945735003E-02, 1.385466987023E-02, 6.482091616720E-02],
              [6.859721745869E-02, 4.160199046481E-02, 1.151943296867E-02, -3.223101197354E-02,
               -7.025699551823E-02, -8.224151800550E-02, -8.153908880816E-02, -7.238462716198E-02,
               -6.181990385401E-02, -5.935500573438E-02, -6.181990385072E-02, -7.238462716017E-02,
               -8.153908881053E-02, -8.224151800355E-02, -7.025699551305E-02, -3.223101198444E-02,
               1.151943296390E-02, 4.160199045888E-02, 6.859721745334E-02],
              [5.864644238720E-02, 5.900823050730E-02, 5.012671341596E-02, 1.151943296784E-02,
               -3.233945733989E-02, -6.223172114117E-02, -8.287206522407E-02, -9.025575972555E-02,
               -9.132287823381E-02, -9.237922986042E-02, -9.132287823895E-02, -9.025575972659E-02,
               -8.287206522924E-02, -6.223172111633E-02, -3.233945736698E-02, 1.151943294807E-02,
               5.012671336003E-02, 5.900823048980E-02, 5.864644240619E-02],
              [2.657147806632E-02, 4.763924885361E-02, 5.900823051827E-02, 4.160199045973E-02,
               1.385466987784E-02, -1.282052492446E-02, -3.643722191266E-02, -4.992325419540E-02,
               -5.712280000218E-02, -5.970849726467E-02, -5.712279998916E-02, -4.992325418971E-02,
               -3.643722191137E-02, -1.282052494051E-02, 1.385466986128E-02, 4.160199045065E-02,
               5.900823048358E-02, 4.763924884926E-02, 2.657147808980E-02],
              [-1.378569409165E-02, 2.657147806577E-02, 5.864644239914E-02, 6.859721745113E-02,
               6.482091617246E-02, 4.830380225762E-02, 2.743644194008E-02, 1.078616361549E-02,
               -1.191055432767E-03, -4.769492606708E-03, -1.191055401971E-03, 1.078616362758E-02,
               2.743644195091E-02, 4.830380220373E-02, 6.482091616552E-02, 6.859721745172E-02,
               5.864644238170E-02, 2.657147807189E-02, -1.378569406431E-02]],

             [[1.607440701473E-02, 4.759923176159E-02, 6.859721744875E-02, 5.944385072825E-02,
               3.690531538111E-02, 9.452397727812E-03, -1.765322333726E-02, -3.551989669422E-02,
               -4.656675271119E-02, -5.018094388287E-02, -4.656675271125E-02, -3.551989669398E-02,
               -1.765322333610E-02, 9.452397730952E-03, 3.690531538776E-02, 5.944385074483E-02,
               6.859721745215E-02, 4.759923176383E-02, 1.607440700824E-02],
              [4.759923176501E-02, 4.831970810317E-02, 4.160199045720E-02, 1.073979770834E-02,
               -2.390080272724E-02, -4.590027524501E-02, -6.012530565147E-02, -6.404296523510E-02,
               -6.343256195303E-02, -6.384596899075E-02, -6.343256195314E-02, -6.404296523487E-02,
               -6.012530565030E-02, -4.590027524259E-02, -2.390080272159E-02, 1.073979772316E-02,
               4.160199046031E-02, 4.831970810418E-02, 4.759923175831E-02],
              [6.859721744378E-02, 4.160199045534E-02, 1.151943297488E-02, -3.223101197017E-02,
               -7.025699551693E-02, -8.224151800361E-02, -8.153908881042E-02, -7.238462716343E-02,
               -6.181990385384E-02, -5.935500573454E-02, -6.181990385407E-02, -7.238462716318E-02,
               -8.153908881017E-02, -8.224151800348E-02, -7.025699551457E-02, -3.223101196760E-02,
               1.151943296750E-02, 4.160199045496E-02, 6.859721744397E-02],
              [5.944385073225E-02, 1.073979771344E-02, -3.223101196863E-02, -6.336210613032E-02,
               -7.803992804257E-02, -6.344434624254E-02, -3.808861275382E-02, -1.545550682691E-02,
               2.183684689331E-03, 7.217721869553E-03, 2.183684689036E-03, -1.545550682673E-02,
               -3.808861275339E-02, -6.344434624211E-02, -7.803992804130E-02, -6.336210612881E-02,
               -3.223101197514E-02, 1.073979771238E-02, 5.944385073084E-02],
              [3.690531537863E-02, -2.390080273080E-02, -7.025699551698E-02, -7.803992804269E-02,
               -6.181990385392E-02, -2.184399201675E-02, 2.304143895887E-02, 5.220189526903E-02,
               7.075468372830E-02, 7.671883932455E-02, 7.075468372851E-02, 5.220189526968E-02,
               2.304143895704E-02, -2.184399201665E-02, -6.181990385421E-02, -7.803992804162E-02,
               -7.025699552249E-02, -2.390080272978E-02, 3.690531537959E-02],
              [9.452397726771E-03, -4.590027524695E-02, -8.224151800391E-02, -6.344434624280E-02,
               -2.184399201663E-02, 2.574993055458E-02, 6.888755196724E-02, 8.693183182589E-02,
               9.363466868696E-02, 9.672805827696E-02, 9.363466868756E-02, 8.693183182701E-02,
               6.888755196647E-02, 2.574993055367E-02, -2.184399201698E-02, -6.344434624239E-02,
               -8.224151800096E-02, -4.590027524124E-02, 9.452397731244E-03],
              [-1.765322333702E-02, -6.012530565111E-02, -8.153908881071E-02, -3.808861275400E-02,
               2.304143895793E-02, 6.888755196665E-02, 1.005921787326E-01, 1.011634188777E-01,
               9.204081655063E-02, 9.097189627778E-02, 9.204081655158E-02, 1.011634188769E-01,
               1.005921787300E-01, 6.888755196510E-02, 2.304143895474E-02, -3.808861275650E-02,
               -8.153908881093E-02, -6.012530565951E-02, -1.765322335236E-02],
              [-3.551989669374E-02, -6.404296523452E-02, -7.238462716346E-02, -1.545550682701E-02,
               5.220189526844E-02, 8.693183182532E-02, 1.011634188777E-01, 8.452337729588E-02,
               6.176314526244E-02, 5.683811106072E-02, 6.176314526352E-02, 8.452337729469E-02,
               1.011634188777E-01, 8.693183182723E-02, 5.220189526615E-02, -1.545550682731E-02,
               -7.238462716058E-02, -6.404296523972E-02, -3.551989670593E-02],
              [-4.656675271117E-02, -6.343256195302E-02, -6.181990385387E-02, 2.183684689170E-03,
               7.075468372818E-02, 9.363466868692E-02, 9.204081655062E-02, 6.176314526255E-02,
               2.882470737182E-02, 2.092269372447E-02, 2.882470737102E-02, 6.176314526219E-02,
               9.204081655037E-02, 9.363466868575E-02, 7.075468372815E-02, 2.183684688956E-03,
               -6.181990385133E-02, -6.343256195297E-02, -4.656675271367E-02],
              [-5.018094388285E-02, -6.384596899068E-02, -5.935500573433E-02, 7.217721869544E-03,
               7.671883932435E-02, 9.672805827683E-02, 9.097189627789E-02, 5.683811106088E-02,
               2.092269372448E-02, 1.216654340533E-02, 2.092269372415E-02, 5.683811106125E-02,
               9.097189627769E-02, 9.672805827612E-02, 7.671883932416E-02, 7.217721869295E-03,
               -5.935500573086E-02, -6.384596899089E-02, -5.018094388674E-02],
              [-4.656675271024E-02, -6.343256195205E-02, -6.181990385355E-02, 2.183684689769E-03,
               7.075468372856E-02, 9.363466868703E-02, 9.204081654942E-02, 6.176314526330E-02,
               2.882470737069E-02, 2.092269372373E-02, 2.882470737028E-02, 6.176314526587E-02,
               9.204081656394E-02, 9.363466869474E-02, 7.075468371880E-02, 2.183684666928E-03,
               -6.181990380728E-02, -6.343256190940E-02, -4.656675266841E-02],
              [-3.551989669340E-02, -6.404296523407E-02, -7.238462716243E-02, -1.545550682629E-02,
               5.220189526850E-02, 8.693183182488E-02, 1.011634188766E-01, 8.452337729618E-02,
               6.176314526181E-02, 5.683811106045E-02, 6.176314525784E-02, 8.452337729073E-02,
               1.011634188692E-01, 8.693183182655E-02, 5.220189526394E-02, -1.545550683418E-02,
               -7.238462716710E-02, -6.404296521874E-02, -3.551989665830E-02],
              [-1.765322334055E-02, -6.012530565380E-02, -8.153908881130E-02, -3.808861275404E-02,
               2.304143896189E-02, 6.888755197351E-02, 1.005921787344E-01, 1.011634188767E-01,
               9.204081655239E-02, 9.097189627954E-02, 9.204081656764E-02, 1.011634188893E-01,
               1.005921787021E-01, 6.888755193657E-02, 2.304143895905E-02, -3.808861275615E-02,
               -8.153908884524E-02, -6.012530567395E-02, -1.765322335177E-02],
              [9.452397724621E-03, -4.590027524759E-02, -8.224151800489E-02, -6.344434624268E-02,
               -2.184399201258E-02, 2.574993056096E-02, 6.888755196933E-02, 8.693183182560E-02,
               9.363466868781E-02, 9.672805827782E-02, 9.363466870690E-02, 8.693183182742E-02,
               6.888755193058E-02, 2.574993049220E-02, -2.184399201931E-02, -6.344434624520E-02,
               -8.224151802883E-02, -4.590027528004E-02, 9.452397679747E-03],
              [3.690531538472E-02, -2.390080272599E-02, -7.025699551350E-02, -7.803992803940E-02,
               -6.181990385551E-02, -2.184399201839E-02, 2.304143895954E-02, 5.220189526795E-02,
               7.075468372796E-02, 7.671883932436E-02, 7.075468373145E-02, 5.220189526943E-02,
               2.304143895707E-02, -2.184399201923E-02, -6.181990392529E-02, -7.803992805902E-02,
               -7.025699549907E-02, -2.390080273766E-02, 3.690531534465E-02],
              [5.944385074365E-02, 1.073979772050E-02, -3.223101197004E-02, -6.336210613006E-02,
               -7.803992804413E-02, -6.344434624444E-02, -3.808861275233E-02, -1.545550682699E-02,
               2.183684689059E-03, 7.217721869754E-03, 2.183684691722E-03, -1.545550682560E-02,
               -3.808861275310E-02, -6.344434623933E-02, -7.803992801619E-02, -6.336210611106E-02,
               -3.223101198489E-02, 1.073979770079E-02, 5.944385071627E-02],
              [6.859721744906E-02, 4.160199045666E-02, 1.151943297469E-02, -3.223101197100E-02,
               -7.025699551705E-02, -8.224151800308E-02, -8.153908882174E-02, -7.238462717012E-02,
               -6.181990385473E-02, -5.935500573311E-02, -6.181990389670E-02, -7.238462715172E-02,
               -8.153908877110E-02, -8.224151790657E-02, -7.025699558976E-02, -3.223101200858E-02,
               1.151943298740E-02, 4.160199045960E-02, 6.859721743676E-02],
              [4.759923176403E-02, 4.831970810229E-02, 4.160199046815E-02, 1.073979772023E-02,
               -2.390080272699E-02, -4.590027524269E-02, -6.012530566362E-02, -6.404296523930E-02,
               -6.343256195244E-02, -6.384596898855E-02, -6.343256195407E-02, -6.404296520875E-02,
               -6.012530561617E-02, -4.590027523035E-02, -2.390080278035E-02, 1.073979769454E-02,
               4.160199047196E-02, 4.831970812325E-02, 4.759923179396E-02],
              [1.607440701618E-02, 4.759923176298E-02, 6.859721746138E-02, 5.944385074167E-02,
               3.690531538286E-02, 9.452397732280E-03, -1.765322334998E-02, -3.551989669664E-02,
               -4.656675270936E-02, -5.018094388014E-02, -4.656675266850E-02, -3.551989665375E-02,
               -1.765322330930E-02, 9.452397663657E-03, 3.690531534455E-02, 5.944385072329E-02,
               6.859721744358E-02, 4.759923179470E-02, 1.607440708711E-02]],

             [[4.356156927379E-02, 5.964570012346E-02, 6.482091617535E-02, 3.690531538307E-02,
               -1.191055438631E-03, -3.406682302464E-02, -6.137786872110E-02, -7.588664280318E-02,
               -8.287206522455E-02, -8.555883978744E-02, -8.287206522474E-02, -7.588664280331E-02,
               -6.137786872323E-02, -3.406682302612E-02, -1.191055431078E-03, 3.690531538558E-02,
               6.482091614509E-02, 5.964570010327E-02, 4.356156924574E-02],
              [5.964570012421E-02, 3.832433747305E-02, 1.385466988085E-02, -2.390080272608E-02,
               -5.712280000687E-02, -6.742425744581E-02, -6.664602760562E-02, -5.841121516010E-02,
               -4.898599926459E-02, -4.677849572179E-02, -4.898599926474E-02, -5.841121516022E-02,
               -6.664602760749E-02, -6.742425744741E-02, -5.712280000012E-02, -2.390080272259E-02,
               1.385466985590E-02, 3.832433745792E-02, 5.964570010279E-02],
              [6.482091617972E-02, 1.385466988733E-02, -3.233945734229E-02, -7.025699551868E-02,
               -9.132287823558E-02, -7.906219681528E-02, -5.367000806639E-02, -2.828238481516E-02,
               -7.601370334154E-03, -1.842790617579E-03, -7.601370334279E-03, -2.828238481499E-02,
               -5.367000806768E-02, -7.906219681599E-02, -9.132287823270E-02, -7.025699551554E-02,
               -3.233945735301E-02, 1.385466985817E-02, 6.482091615177E-02],
              [3.690531538567E-02, -2.390080272147E-02, -7.025699551945E-02, -7.803992804375E-02,
               -6.181990385449E-02, -2.184399201684E-02, 2.304143895891E-02, 5.220189526903E-02,
               7.075468372823E-02, 7.671883932455E-02, 7.075468372847E-02, 5.220189527001E-02,
               2.304143896057E-02, -2.184399201467E-02, -6.181990385197E-02, -7.803992804086E-02,
               -7.025699552630E-02, -2.390080274442E-02, 3.690531536190E-02],
              [-1.191055433694E-03, -5.712280000494E-02, -9.132287823557E-02, -6.181990385498E-02,
               -7.601370334226E-03, 4.969838979381E-02, 9.999127997659E-02, 1.205490947918E-01,
               1.276578283168E-01, 1.311161468593E-01, 1.276578283164E-01, 1.205490947914E-01,
               9.999127997719E-02, 4.969838979430E-02, -7.601370318531E-03, -6.181990384007E-02,
               -9.132287824069E-02, -5.712280000866E-02, -1.191055437527E-03],
              [-3.406682302200E-02, -6.742425744518E-02, -7.906219681582E-02, -2.184399201759E-02,
               4.969838979386E-02, 9.400421084309E-02, 1.192964320993E-01, 1.119722038729E-01,
               9.594822035786E-02, 9.302969389022E-02, 9.594822035643E-02, 1.119722038692E-01,
               1.192964320973E-01, 9.400421084178E-02, 4.969838979117E-02, -2.184399200847E-02,
               -7.906219681402E-02, -6.742425744429E-02, -3.406682302215E-02],
              [-6.137786871924E-02, -6.664602760422E-02, -5.367000806925E-02, 2.304143895703E-02,
               9.999127997781E-02, 1.192964321008E-01, 1.096062925701E-01, 7.105670827737E-02,
               3.130524741646E-02, 2.151761677175E-02, 3.130524742148E-02, 7.105670828489E-02,
               1.096062925686E-01, 1.192964321034E-01, 9.999127996757E-02, 2.304143894852E-02,
               -5.367000807981E-02, -6.664602761143E-02, -6.137786872006E-02],
              [-7.588664280080E-02, -5.841121515824E-02, -2.828238481776E-02, 5.220189526728E-02,
               1.205490947928E-01, 1.119722038747E-01, 7.105670827502E-02, 1.302475523048E-02,
               -3.882064949522E-02, -5.255531921819E-02, -3.882064949242E-02, 1.302475523757E-02,
               7.105670827421E-02, 1.119722038641E-01, 1.205490947857E-01, 5.220189525847E-02,
               -2.828238482027E-02, -5.841121516662E-02, -7.588664281097E-02],
              [-8.287206522469E-02, -4.898599926431E-02, -7.601370334623E-03, 7.075468372813E-02,
               1.276578283168E-01, 9.594822035793E-02, 3.130524741682E-02, -3.882064949755E-02,
               -9.690247379720E-02, -1.129284389489E-01, -9.690247379315E-02, -3.882064949350E-02,
               3.130524741559E-02, 9.594822035850E-02, 1.276578283065E-01, 7.075468371883E-02,
               -7.601370331493E-03, -4.898599926380E-02, -8.287206522583E-02],
              [-8.555883978757E-02, -4.677849572161E-02, -1.842790618081E-03, 7.671883932431E-02,
               1.311161468594E-01, 9.302969389037E-02, 2.151761677326E-02, -5.255531921948E-02,
               -1.129284389488E-01, -1.297341959139E-01, -1.129284389427E-01, -5.255531921586E-02,
               2.151761677120E-02, 9.302969389113E-02, 1.311161468521E-01, 7.671883931074E-02,
               -1.842790612652E-03, -4.677849571936E-02, -8.555883978745E-02],
              [-8.287206522079E-02, -4.898599926131E-02, -7.601370336039E-03, 7.075468372720E-02,
               1.276578283197E-01, 9.594822036505E-02, 3.130524743616E-02, -3.882064950290E-02,
               -9.690247379540E-02, -1.129284389495E-01, -9.690247382792E-02, -3.882064956807E-02,
               3.130524738701E-02, 9.594822038398E-02, 1.276578282813E-01, 7.075468361523E-02,
               -7.601370402067E-03, -4.898599929938E-02, -8.287206522834E-02],
              [-7.588664279949E-02, -5.841121515715E-02, -2.828238481671E-02, 5.220189526782E-02,
               1.205490947990E-01, 1.119722038804E-01, 7.105670829300E-02, 1.302475523359E-02,
               -3.882064949873E-02, -5.255531921871E-02, -3.882064952495E-02, 1.302475521090E-02,
               7.105670833171E-02, 1.119722039220E-01, 1.205490948462E-01, 5.220189528039E-02,
               -2.828238477987E-02, -5.841121515324E-02, -7.588664282472E-02],
              [-6.137786872785E-02, -6.664602761172E-02, -5.367000804881E-02, 2.304143897403E-02,
               9.999127997349E-02, 1.192964320969E-01, 1.096062925768E-01, 7.105670829818E-02,
               3.130524741581E-02, 2.151761677178E-02, 3.130524744449E-02, 7.105670832089E-02,
               1.096062925506E-01, 1.192964320942E-01, 9.999128014101E-02, 2.304143912547E-02,
               -5.367000811864E-02, -6.664602757321E-02, -6.137786860979E-02],
              [-3.406682303057E-02, -6.742425745130E-02, -7.906219679524E-02, -2.184399199874E-02,
               4.969838979030E-02, 9.400421084139E-02, 1.192964321132E-01, 1.119722038919E-01,
               9.594822035890E-02, 9.302969389162E-02, 9.594822034510E-02, 1.119722039013E-01,
               1.192964320370E-01, 9.400421079261E-02, 4.969838990329E-02, -2.184399190272E-02,
               -7.906219680837E-02, -6.742425742143E-02, -3.406682298535E-02],
              [-1.191055444583E-03, -5.712280000933E-02, -9.132287823992E-02, -6.181990385824E-02,
               -7.601370361761E-03, 4.969838976729E-02, 9.999127996545E-02, 1.205490947739E-01,
               1.276578283224E-01, 1.311161468648E-01, 1.276578283054E-01, 1.205490947655E-01,
               9.999127985629E-02, 4.969838976028E-02, -7.601370328022E-03, -6.181990384851E-02,
               -9.132287803691E-02, -5.712279977237E-02, -1.191055158805E-03],
              [3.690531536620E-02, -2.390080273686E-02, -7.025699552167E-02, -7.803992804657E-02,
               -6.181990388769E-02, -2.184399205532E-02, 2.304143896086E-02, 5.220189527488E-02,
               7.075468373523E-02, 7.671883933272E-02, 7.075468376211E-02, 5.220189534065E-02,
               2.304143886413E-02, -2.184399213537E-02, -6.181990379649E-02, -7.803992804047E-02,
               -7.025699538097E-02, -2.390080261938E-02, 3.690531546388E-02],
              [6.482091618360E-02, 1.385466989017E-02, -3.233945732811E-02, -7.025699550824E-02,
               -9.132287823997E-02, -7.906219681922E-02, -5.367000805754E-02, -2.828238479304E-02,
               -7.601370334135E-03, -1.842790617571E-03, -7.601370316603E-03, -2.828238479147E-02,
               -5.367000814368E-02, -7.906219676339E-02, -9.132287767065E-02, -7.025699500165E-02,
               -3.233945793432E-02, 1.385466958463E-02, 6.482091617096E-02],
              [5.964570014300E-02, 3.832433749211E-02, 1.385466989058E-02, -2.390080271628E-02,
               -5.712280000424E-02, -6.742425744675E-02, -6.664602757923E-02, -5.841121513787E-02,
               -4.898599926420E-02, -4.677849572248E-02, -4.898599930009E-02, -5.841121516303E-02,
               -6.664602769418E-02, -6.742425759594E-02, -5.712279958844E-02, -2.390080237601E-02,
               1.385466960018E-02, 3.832433738872E-02, 5.964570021521E-02],
              [4.356156929032E-02, 5.964570014048E-02, 6.482091618607E-02, 3.690531539543E-02,
               -1.191055431360E-03, -3.406682302387E-02, -6.137786867720E-02, -7.588664277964E-02,
               -8.287206522394E-02, -8.555883978896E-02, -8.287206531375E-02, -7.588664283118E-02,
               -6.137786880848E-02, -3.406682336366E-02, -1.191055144756E-03, 3.690531559754E-02,
               6.482091618400E-02, 5.964570022560E-02, 4.356156944608E-02]],

             [[5.936207313787E-02, 5.828995984270E-02, 4.830380226102E-02, 9.452397730714E-03,
               -3.406682302576E-02, -6.297844907573E-02, -8.246762366632E-02, -8.898818051474E-02,
               -8.945863728260E-02, -9.033251398459E-02, -8.945863728276E-02, -8.898818051488E-02,
               -8.246762366854E-02, -6.297844907725E-02, -3.406682301648E-02, 9.452397735866E-03,
               4.830380223503E-02, 5.828995982631E-02, 5.936207311695E-02],
              [5.828995984182E-02, 2.170304543863E-02, -1.282052492152E-02, -4.590027524307E-02,
               -6.742425744678E-02, -6.217010895364E-02, -4.608859723532E-02, -2.874753183580E-02,
               -1.407342396245E-02, -1.007152730321E-02, -1.407342396259E-02, -2.874753183599E-02,
               -4.608859723732E-02, -6.217010895517E-02, -6.742425743851E-02, -4.590027523766E-02,
               -1.282052494295E-02, 2.170304542686E-02, 5.828995982663E-02],
              [4.830380226644E-02, -1.282052491537E-02, -6.223172114361E-02, -8.224151800572E-02,
               -7.906219681522E-02, -4.598683888707E-02, -4.084775800167E-03, 2.749289580995E-02,
               4.969838979401E-02, 5.642128549327E-02, 4.969838979361E-02, 2.749289581052E-02,
               -4.084775800412E-03, -4.598683888722E-02, -7.906219681291E-02, -8.224151800226E-02,
               -6.223172115169E-02, -1.282052494278E-02, 4.830380223510E-02],
              [9.452397734617E-03, -4.590027523840E-02, -8.224151800605E-02, -6.344434624406E-02,
               -2.184399201672E-02, 2.574993055467E-02, 6.888755196750E-02, 8.693183182612E-02,
               9.363466868691E-02, 9.672805827707E-02, 9.363466868661E-02, 8.693183182646E-02,
               6.888755196979E-02, 2.574993055520E-02, -2.184399201524E-02, -6.344434624125E-02,
               -8.224151800522E-02, -4.590027525639E-02, 9.452397710263E-03],
              [-3.406682302120E-02, -6.742425744466E-02, -7.906219681666E-02, -2.184399201828E-02,
               4.969838979397E-02, 9.400421084318E-02, 1.192964320990E-01, 1.119722038725E-01,
               9.594822035800E-02, 9.302969389031E-02, 9.594822035416E-02, 1.119722038688E-01,
               1.192964321004E-01, 9.400421084400E-02, 4.969838979634E-02, -2.184399201929E-02,
               -7.906219682926E-02, -6.742425744593E-02, -3.406682301067E-02],
              [-6.297844907363E-02, -6.217010895314E-02, -4.598683888860E-02, 2.574993055312E-02,
               9.400421084333E-02, 1.021244353642E-01, 8.313590780512E-02, 4.401715541032E-02,
               6.256940895049E-03, -3.352929968738E-03, 6.256940896573E-03, 4.401715541179E-02,
               8.313590780480E-02, 1.021244353639E-01, 9.400421082991E-02, 2.574993054754E-02,
               -4.598683889458E-02, -6.217010895743E-02, -6.297844907515E-02],
              [-8.246762366334E-02, -4.608859723299E-02, -4.084775804385E-03, 6.888755196470E-02,
               1.192964321000E-01, 8.313590780596E-02, 1.717169886405E-02, -4.673523156609E-02,
               -9.787657034544E-02, -1.122576039981E-01, -9.787657032886E-02, -4.673523153200E-02,
               1.717169890330E-02, 8.313590783356E-02, 1.192964320915E-01, 6.888755191342E-02,
               -4.084775798774E-03, -4.608859723467E-02, -8.246762366568E-02],
              [-8.898818051105E-02, -2.874753183286E-02, 2.749289580597E-02, 8.693183182314E-02,
               1.119722038735E-01, 4.401715541167E-02, -4.673523157195E-02, -1.056041462995E-01,
               -1.446346315831E-01, -1.568720565874E-01, -1.446346315784E-01, -1.056041462836E-01,
               -4.673523155499E-02, 4.401715542635E-02, 1.119722038469E-01, 8.693183176126E-02,
               2.749289579175E-02, -2.874753185092E-02, -8.898818052628E-02],
              [-8.945863728284E-02, -1.407342396236E-02, 4.969838979305E-02, 9.363466868648E-02,
               9.594822035801E-02, 6.256940895111E-03, -9.787657034403E-02, -1.446346315842E-01,
               -1.671645939885E-01, -1.758255969440E-01, -1.671645939855E-01, -1.446346315837E-01,
               -9.787657034322E-02, 6.256940895132E-03, 9.594822036121E-02, 9.363466868884E-02,
               4.969838978938E-02, -1.407342396612E-02, -8.945863728550E-02],
              [-9.033251398484E-02, -1.007152730319E-02, 5.642128549226E-02, 9.672805827639E-02,
               9.302969389045E-02, -3.352929968574E-03, -1.122576039953E-01, -1.568720565863E-01,
               -1.758255969441E-01, -1.837715960626E-01, -1.758255969397E-01, -1.568720565862E-01,
               -1.122576039958E-01, -3.352929969231E-03, 9.302969390721E-02, 9.672805828612E-02,
               5.642128548892E-02, -1.007152730783E-02, -9.033251398938E-02],
              [-8.945863727839E-02, -1.407342395888E-02, 4.969838979123E-02, 9.363466868390E-02,
               9.594822036773E-02, 6.256940902751E-03, -9.787657033341E-02, -1.446346315801E-01,
               -1.671645939916E-01, -1.758255969456E-01, -1.671645939702E-01, -1.446346315898E-01,
               -9.787657032615E-02, 6.256940856878E-03, 9.594821990767E-02, 9.363466868575E-02,
               4.969838974598E-02, -1.407342402174E-02, -8.945863735750E-02],
              [-8.898818051067E-02, -2.874753183265E-02, 2.749289580793E-02, 8.693183182360E-02,
               1.119722038826E-01, 4.401715542160E-02, -4.673523154340E-02, -1.056041462757E-01,
               -1.446346315857E-01, -1.568720565866E-01, -1.446346316181E-01, -1.056041463400E-01,
               -4.673523153752E-02, 4.401715544427E-02, 1.119722037718E-01, 8.693183190921E-02,
               2.749289581984E-02, -2.874753189748E-02, -8.898818065688E-02],
              [-8.246762367447E-02, -4.608859724269E-02, -4.084775778606E-03, 6.888755198522E-02,
               1.192964320924E-01, 8.313590779850E-02, 1.717169889951E-02, -4.673523152869E-02,
               -9.787657034396E-02, -1.122576039964E-01, -9.787657039866E-02, -4.673523165833E-02,
               1.717169887271E-02, 8.313590786058E-02, 1.192964322549E-01, 6.888755208026E-02,
               -4.084775735714E-03, -4.608859716459E-02, -8.246762358499E-02],
              [-6.297844908367E-02, -6.217010896091E-02, -4.598683886263E-02, 2.574993057391E-02,
               9.400421083711E-02, 1.021244353597E-01, 8.313590784141E-02, 4.401715544175E-02,
               6.256940895258E-03, -3.352929969066E-03, 6.256940842622E-03, 4.401715537241E-02,
               8.313590779167E-02, 1.021244353645E-01, 9.400421092889E-02, 2.574993075668E-02,
               -4.598683890942E-02, -6.217010885564E-02, -6.297844887283E-02],
              [-3.406682303576E-02, -6.742425745161E-02, -7.906219682084E-02, -2.184399202135E-02,
               4.969838976293E-02, 9.400421082512E-02, 1.192964321023E-01, 1.119722038489E-01,
               9.594822036547E-02, 9.302969388729E-02, 9.594822038834E-02, 1.119722039491E-01,
               1.192964321870E-01, 9.400421088335E-02, 4.969838977509E-02, -2.184399203097E-02,
               -7.906219706652E-02, -6.742425734965E-02, -3.406682258364E-02],
              [9.452397709324E-03, -4.590027525693E-02, -8.224151800914E-02, -6.344434624691E-02,
               -2.184399205480E-02, 2.574993052657E-02, 6.888755197561E-02, 8.693183177445E-02,
               9.363466869839E-02, 9.672805827955E-02, 9.363466878950E-02, 8.693183200841E-02,
               6.888755190639E-02, 2.574993066603E-02, -2.184399203461E-02, -6.344434632804E-02,
               -8.224151847245E-02, -4.590027548083E-02, 9.452397732179E-03],
              [4.830380228359E-02, -1.282052490147E-02, -6.223172113134E-02, -8.224151799437E-02,
               -7.906219682180E-02, -4.598683888847E-02, -4.084775779105E-03, 2.749289584410E-02,
               4.969838979127E-02, 5.642128549401E-02, 4.969838982863E-02, 2.749289579012E-02,
               -4.084775927366E-03, -4.598683867179E-02, -7.906219688101E-02, -8.224151736141E-02,
               -6.223172032767E-02, -1.282052424573E-02, 4.830380283486E-02],
              [5.828995987769E-02, 2.170304546975E-02, -1.282052491373E-02, -4.590027523493E-02,
               -6.742425744342E-02, -6.217010895193E-02, -4.608859720836E-02, -2.874753180918E-02,
               -1.407342396446E-02, -1.007152730316E-02, -1.407342399275E-02, -2.874753185281E-02,
               -4.608859728557E-02, -6.217010893833E-02, -6.742425771904E-02, -4.590027500124E-02,
               -1.282052457150E-02, 2.170304588851E-02, 5.828996042445E-02],
              [5.936207317315E-02, 5.828995987299E-02, 4.830380226953E-02, 9.452397739812E-03,
               -3.406682301444E-02, -6.297844907272E-02, -8.246762363158E-02, -8.898818049328E-02,
               -8.945863728415E-02, -9.033251398543E-02, -8.945863737774E-02, -8.898818052876E-02,
               -8.246762363826E-02, -6.297844925954E-02, -3.406682348468E-02, 9.452397589606E-03,
               4.830380218507E-02, 5.828996005643E-02, 5.936207368426E-02]],

             [[6.830161066213E-02, 5.086584767314E-02, 2.743644194748E-02, -1.765322333770E-02,
               -6.137786872134E-02, -8.246762366660E-02, -9.132287823289E-02, -8.866359515589E-02,
               -8.226118588934E-02, -8.112847434614E-02, -8.226118588929E-02, -8.866359515663E-02,
               -9.132287823512E-02, -8.246762366705E-02, -6.137786871882E-02, -1.765322333222E-02,
               2.743644194896E-02, 5.086584768387E-02, 6.830161067476E-02],
              [5.086584767283E-02, 3.645921385629E-03, -3.643722190852E-02, -6.012530565097E-02,
               -6.664602760586E-02, -4.608859723560E-02, -1.661020289044E-02, 7.096489196928E-03,
               2.454870319537E-02, 2.969356645633E-02, 2.454870319588E-02, 7.096489196517E-03,
               -1.661020289250E-02, -4.608859723612E-02, -6.664602760305E-02, -6.012530564683E-02,
               -3.643722190873E-02, 3.645921394146E-03, 5.086584768288E-02],
              [2.743644195038E-02, -3.643722190841E-02, -8.287206522205E-02, -8.153908880819E-02,
               -5.367000806421E-02, -4.084775798847E-03, 4.831031939423E-02, 8.052457015202E-02,
               9.999127997729E-02, 1.064514000502E-01, 9.999127997738E-02, 8.052457015258E-02,
               4.831031939478E-02, -4.084775801375E-03, -5.367000806824E-02, -8.153908881188E-02,
               -8.287206521676E-02, -3.643722190006E-02, 2.743644195755E-02],
              [-1.765322333681E-02, -6.012530565197E-02, -8.153908880796E-02, -3.808861275171E-02,
               2.304143896171E-02, 6.888755196921E-02, 1.005921787327E-01, 1.011634188780E-01,
               9.204081655119E-02, 9.097189627842E-02, 9.204081655010E-02, 1.011634188767E-01,
               1.005921787337E-01, 6.888755196485E-02, 2.304143895211E-02, -3.808861275852E-02,
               -8.153908881194E-02, -6.012530564384E-02, -1.765322332082E-02],
              [-6.137786871733E-02, -6.664602760257E-02, -5.367000807331E-02, 2.304143895328E-02,
               9.999127997908E-02, 1.192964321016E-01, 1.096062925729E-01, 7.105670828037E-02,
               3.130524741595E-02, 2.151761677110E-02, 3.130524740384E-02, 7.105670825579E-02,
               1.096062925617E-01, 1.192964320992E-01, 9.999127997134E-02, 2.304143895628E-02,
               -5.367000807696E-02, -6.664602761307E-02, -6.137786872614E-02],
              [-8.246762366307E-02, -4.608859723277E-02, -4.084775806550E-03, 6.888755196196E-02,
               1.192964321016E-01, 8.313590780814E-02, 1.717169886113E-02, -4.673523157383E-02,
               -9.787657034405E-02, -1.122576039962E-01, -9.787657036242E-02, -4.673523157807E-02,
               1.717169888584E-02, 8.313590781483E-02, 1.192964321096E-01, 6.888755198882E-02,
               -4.084775822452E-03, -4.608859725189E-02, -8.246762367722E-02],
              [-9.132287820985E-02, -1.661020286927E-02, 4.831031939300E-02, 1.005921787315E-01,
               1.096062925786E-01, 1.717169887461E-02, -9.690247380610E-02, -1.657196009913E-01,
               -2.087231323907E-01, -2.227007147116E-01, -2.087231323834E-01, -1.657196009130E-01,
               -9.690247371515E-02, 1.717169892804E-02, 1.096062925835E-01, 1.005921787557E-01,
               4.831031939624E-02, -1.661020286675E-02, -9.132287818135E-02],
              [-8.866359512528E-02, 7.096489224471E-03, 8.052457015015E-02, 1.011634188759E-01,
               7.105670828863E-02, -4.673523155705E-02, -1.657196009753E-01, -1.900217695794E-01,
               -1.847268995331E-01, -1.871028118508E-01, -1.847268995111E-01, -1.900217694966E-01,
               -1.657196010147E-01, -4.673523156025E-02, 7.105670830511E-02, 1.011634189413E-01,
               8.052457014131E-02, 7.096489265301E-03, -8.866359500230E-02],
              [-8.226118588719E-02, 2.454870319729E-02, 9.999127997555E-02, 9.204081654978E-02,
               3.130524741561E-02, -9.787657034483E-02, -2.087231323887E-01, -1.847268995338E-01,
               -1.307271192065E-01, -1.209776175584E-01, -1.307271191984E-01, -1.847268995147E-01,
               -2.087231323763E-01, -9.787657034767E-02, 3.130524743617E-02, 9.204081657434E-02,
               9.999127999025E-02, 2.454870320351E-02, -8.226118588565E-02],
              [-8.112847434388E-02, 2.969356645833E-02, 1.064514000484E-01, 9.097189627704E-02,
               2.151761677100E-02, -1.122576039971E-01, -2.227007147096E-01, -1.871028118439E-01,
               -1.209776175584E-01, -1.082622304147E-01, -1.209776175531E-01, -1.871028118308E-01,
               -2.227007147085E-01, -1.122576040145E-01, 2.151761680508E-02, 9.097189632033E-02,
               1.064514000616E-01, 2.969356647192E-02, -8.112847432667E-02],
              [-8.226118591076E-02, 2.454870317483E-02, 9.999127997292E-02, 9.204081654608E-02,
               3.130524740352E-02, -9.787657037639E-02, -2.087231324560E-01, -1.847268995155E-01,
               -1.307271191972E-01, -1.209776175377E-01, -1.307271191288E-01, -1.847268995386E-01,
               -2.087231324182E-01, -9.787657051834E-02, 3.130524744355E-02, 9.204081668735E-02,
               9.999128039976E-02, 2.454870330068E-02, -8.226118610074E-02],
              [-8.866359518786E-02, 7.096489167069E-03, 8.052457014799E-02, 1.011634188735E-01,
               7.105670823905E-02, -4.673523160512E-02, -1.657196009449E-01, -1.900217694844E-01,
               -1.847268994999E-01, -1.871028118290E-01, -1.847268993835E-01, -1.900217697501E-01,
               -1.657196015525E-01, -4.673523233783E-02, 7.105670851802E-02, 1.011634190141E-01,
               8.052456964944E-02, 7.096489081008E-03, -8.866359488149E-02],
              [-9.132287826645E-02, -1.661020291907E-02, 4.831031933959E-02, 1.005921786870E-01,
               1.096062925314E-01, 1.717169885484E-02, -9.690247378091E-02, -1.657196009727E-01,
               -2.087231323745E-01, -2.227007147082E-01, -2.087231321154E-01, -1.657196010242E-01,
               -9.690247399425E-02, 1.717169875889E-02, 1.096062926768E-01, 1.005921787120E-01,
               4.831031944508E-02, -1.661020287930E-02, -9.132287826749E-02],
              [-8.246762370216E-02, -4.608859726576E-02, -4.084775864279E-03, 6.888755191261E-02,
               1.192964320595E-01, 8.313590777732E-02, 1.717169882522E-02, -4.673523157300E-02,
               -9.787657034527E-02, -1.122576040073E-01, -9.787657001368E-02, -4.673523138658E-02,
               1.717169879909E-02, 8.313590790067E-02, 1.192964319825E-01, 6.888755168103E-02,
               -4.084775745976E-03, -4.608859717256E-02, -8.246762359012E-02],
              [-6.137786871067E-02, -6.664602759128E-02, -5.367000803025E-02, 2.304143899133E-02,
               9.999127999330E-02, 1.192964321174E-01, 1.096062926176E-01, 7.105670832650E-02,
               3.130524741119E-02, 2.151761676532E-02, 3.130524750736E-02, 7.105670837205E-02,
               1.096062922329E-01, 1.192964316666E-01, 9.999127989883E-02, 2.304143896567E-02,
               -5.367000926921E-02, -6.664602846660E-02, -6.137786929411E-02],
              [-1.765322332327E-02, -6.012530563284E-02, -8.153908877178E-02, -3.808861271828E-02,
               2.304143895553E-02, 6.888755199557E-02, 1.005921787477E-01, 1.011634189063E-01,
               9.204081653733E-02, 9.097189626279E-02, 9.204081640400E-02, 1.011634186366E-01,
               1.005921780776E-01, 6.888755156212E-02, 2.304143887890E-02, -3.808861264301E-02,
               -8.153909001812E-02, -6.012530659795E-02, -1.765322408887E-02],
              [2.743644197732E-02, -3.643722188639E-02, -8.287206516654E-02, -8.153908874908E-02,
               -5.367000807273E-02, -4.084775793387E-03, 4.831031963687E-02, 8.052457026109E-02,
               9.999128000955E-02, 1.064514000705E-01, 9.999127957266E-02, 8.052456943627E-02,
               4.831031920318E-02, -4.084775883448E-03, -5.367000858705E-02, -8.153908923141E-02,
               -8.287206486717E-02, -3.643722166924E-02, 2.743644207970E-02],
              [5.086584771446E-02, 3.645921418096E-03, -3.643722183878E-02, -6.012530557917E-02,
               -6.664602760784E-02, -4.608859723016E-02, -1.661020285641E-02, 7.096489274458E-03,
               2.454870322806E-02, 2.969356647065E-02, 2.454870323551E-02, 7.096488354810E-03,
               -1.661020287833E-02, -4.608859729527E-02, -6.664602902200E-02, -6.012530645569E-02,
               -3.643722162348E-02, 3.645921506217E-03, 5.086584762092E-02],
              [6.830161070891E-02, 5.086584770959E-02, 2.743644203252E-02, -1.765322325215E-02,
               -6.137786871891E-02, -8.246762366243E-02, -9.132287839913E-02, -8.866359510841E-02,
               -8.226118585715E-02, -8.112847433875E-02, -8.226118538192E-02, -8.866359611420E-02,
               -9.132287799458E-02, -8.246762369299E-02, -6.137787102837E-02, -1.765322453880E-02,
               2.743644217104E-02, 5.086584767308E-02, 6.830161041900E-02]],

             [[7.107762957585E-02, 4.258073037039E-02, 1.078616362078E-02, -3.551989669155E-02,
               -7.588664280340E-02, -8.898818051497E-02, -8.866359515457E-02, -7.909406560943E-02,
               -6.792741701969E-02, -6.533539475257E-02, -6.792741701959E-02, -7.909406561032E-02,
               -8.866359515683E-02, -8.898818051545E-02, -7.588664280120E-02, -3.551989668918E-02,
               1.078616362000E-02, 4.258073038520E-02, 7.107762959368E-02],
              [4.258073036987E-02, -8.922263390742E-03, -4.992325418939E-02, -6.404296523194E-02,
               -5.841121516027E-02, -2.874753183602E-02, 7.096489198108E-03, 3.166809106715E-02,
               4.805122989196E-02, 5.317057247894E-02, 4.805122989269E-02, 3.166809106711E-02,
               7.096489196009E-03, -2.874753183650E-02, -5.841121515797E-02, -6.404296523055E-02,
               -4.992325419104E-02, -8.922263378625E-03, 4.258073038418E-02],
              [1.078616362392E-02, -4.992325418939E-02, -9.025575971947E-02, -7.238462716057E-02,
               -2.828238481229E-02, 2.749289581167E-02, 8.052457015198E-02, 1.073372671572E-01,
               1.205490947919E-01, 1.255521603720E-01, 1.205490947915E-01, 1.073372671572E-01,
               8.052457015339E-02, 2.749289581078E-02, -2.828238481598E-02, -7.238462716350E-02,
               -9.025575971976E-02, -4.992325417900E-02, 1.078616364259E-02],
              [-3.551989669117E-02, -6.404296523356E-02, -7.238462716018E-02, -1.545550682440E-02,
               5.220189527253E-02, 8.693183182824E-02, 1.011634188777E-01, 8.452337729595E-02,
               6.176314526303E-02, 5.683811106144E-02, 6.176314526306E-02, 8.452337729534E-02,
               1.011634188801E-01, 8.693183182471E-02, 5.220189526690E-02, -1.545550682929E-02,
               -7.238462717307E-02, -6.404296521987E-02, -3.551989665447E-02],
              [-7.588664279814E-02, -5.841121515604E-02, -2.828238482324E-02, 5.220189526258E-02,
               1.205490947949E-01, 1.119722038760E-01, 7.105670827465E-02, 1.302475523729E-02,
               -3.882064949688E-02, -5.255531921966E-02, -3.882064953157E-02, 1.302475520294E-02,
               7.105670829618E-02, 1.119722038694E-01, 1.205490948333E-01, 5.220189528752E-02,
               -2.828238482556E-02, -5.841121517569E-02, -7.588664282337E-02],
              [-8.898818051028E-02, -2.874753183230E-02, 2.749289580257E-02, 8.693183181979E-02,
               1.119722038771E-01, 4.401715541497E-02, -4.673523156921E-02, -1.056041462932E-01,
               -1.446346315847E-01, -1.568720565888E-01, -1.446346316296E-01, -1.056041463110E-01,
               -4.673523154079E-02, 4.401715543159E-02, 1.119722038237E-01, 8.693183182737E-02,
               2.749289581351E-02, -2.874753184270E-02, -8.898818053148E-02],
              [-8.866359512567E-02, 7.096489224791E-03, 8.052457015095E-02, 1.011634188770E-01,
               7.105670828843E-02, -4.673523155241E-02, -1.657196009878E-01, -1.900217695736E-01,
               -1.847268995387E-01, -1.871028118528E-01, -1.847268995886E-01, -1.900217695844E-01,
               -1.657196009641E-01, -4.673523145876E-02, 7.105670833818E-02, 1.011634189028E-01,
               8.052457015134E-02, 7.096489241223E-03, -8.866359506293E-02],
              [-7.909406557188E-02, 3.166809110110E-02, 1.073372671556E-01, 8.452337729458E-02,
               1.302475525075E-02, -1.056041462763E-01, -1.900217695618E-01, -1.130233204368E-01,
               -1.064405574677E-02, 1.177815848131E-02, -1.064405577390E-02, -1.130233204012E-01,
               -1.900217695694E-01, -1.056041462685E-01, 1.302475529776E-02, 8.452337734771E-02,
               1.073372674056E-01, 3.166809134520E-02, -7.909406529352E-02],
              [-6.792741701705E-02, 4.805122989434E-02, 1.205490947901E-01, 6.176314526195E-02,
               -3.882064949959E-02, -1.446346315871E-01, -1.847268995307E-01, -1.064405575110E-02,
               1.816770692942E-01, 2.274775653983E-01, 1.816770693077E-01, -1.064405572252E-02,
               -1.847268995440E-01, -1.446346315605E-01, -3.882064951218E-02, 6.176314527087E-02,
               1.205490947966E-01, 4.805122992207E-02, -6.792741696493E-02],
              [-6.533539474976E-02, 5.317057248146E-02, 1.255521603700E-01, 5.683811106018E-02,
               -5.255531922135E-02, -1.568720565890E-01, -1.871028118466E-01, 1.177815848678E-02,
               2.274775653991E-01, 2.793001125551E-01, 2.274775653991E-01, 1.177815849124E-02,
               -1.871028118731E-01, -1.568720565804E-01, -5.255531920985E-02, 5.683811107751E-02,
               1.255521603657E-01, 5.317057250714E-02, -6.533539469123E-02],
              [-6.792741704415E-02, 4.805122986916E-02, 1.205490947877E-01, 6.176314526038E-02,
               -3.882064952918E-02, -1.446346316295E-01, -1.847268996525E-01, -1.064405580702E-02,
               1.816770693165E-01, 2.274775654289E-01, 1.816770690158E-01, -1.064405600768E-02,
               -1.847268997223E-01, -1.446346314495E-01, -3.882064961585E-02, 6.176314499268E-02,
               1.205490946861E-01, 4.805122952733E-02, -6.792741767251E-02],
              [-7.909406564650E-02, 3.166809103285E-02, 1.073372671532E-01, 8.452337729293E-02,
               1.302475519408E-02, -1.056041463551E-01, -1.900217696790E-01, -1.130233203779E-01,
               -1.064405571762E-02, 1.177815851984E-02, -1.064405571398E-02, -1.130233208180E-01,
               -1.900217699010E-01, -1.056041468626E-01, 1.302475518085E-02, 8.452337690851E-02,
               1.073372673526E-01, 3.166809085944E-02, -7.909406623786E-02],
              [-8.866359519591E-02, 7.096489162707E-03, 8.052457008966E-02, 1.011634188251E-01,
               7.105670824028E-02, -4.673523158108E-02, -1.657196010921E-01, -1.900217696811E-01,
               -1.847268995414E-01, -1.871028118501E-01, -1.847268993455E-01, -1.900217695047E-01,
               -1.657196009241E-01, -4.673523170419E-02, 7.105670855125E-02, 1.011634188933E-01,
               8.052457038567E-02, 7.096489226673E-03, -8.866359533166E-02],
              [-8.898818055922E-02, -2.874753187354E-02, 2.749289573559E-02, 8.693183176283E-02,
               1.119722038189E-01, 4.401715537004E-02, -4.673523168501E-02, -1.056041463830E-01,
               -1.446346315971E-01, -1.568720565918E-01, -1.446346313449E-01, -1.056041461904E-01,
               -4.673523149315E-02, 4.401715534685E-02, 1.119722038834E-01, 8.693183155072E-02,
               2.749289611602E-02, -2.874753164305E-02, -8.898818043000E-02],
              [-7.588664281381E-02, -5.841121516416E-02, -2.828238476964E-02, 5.220189531258E-02,
               1.205490948125E-01, 1.119722039587E-01, 7.105670833450E-02, 1.302475526993E-02,
               -3.882064947492E-02, -5.255531920532E-02, -3.882064911611E-02, 1.302475532435E-02,
               7.105670729395E-02, 1.119722029907E-01, 1.205490950456E-01, 5.220189533168E-02,
               -2.828238465420E-02, -5.841121469366E-02, -7.588664202382E-02],
              [-3.551989670261E-02, -6.404296523621E-02, -7.238462711170E-02, -1.545550677744E-02,
               5.220189529173E-02, 8.693183190719E-02, 1.011634189068E-01, 8.452337731928E-02,
               6.176314526096E-02, 5.683811106464E-02, 6.176314532675E-02, 8.452337740555E-02,
               1.011634180384E-01, 8.693183146229E-02, 5.220189527859E-02, -1.545550677319E-02,
               -7.238462669190E-02, -6.404296437547E-02, -3.551989543206E-02],
              [1.078616365649E-02, -4.992325416346E-02, -9.025575965303E-02, -7.238462709707E-02,
               -2.828238482374E-02, 2.749289581605E-02, 8.052457040597E-02, 1.073372674728E-01,
               1.205490948154E-01, 1.255521603810E-01, 1.205490943125E-01, 1.073372664834E-01,
               8.052457000164E-02, 2.749289559490E-02, -2.828238192811E-02, -7.238462544654E-02,
               -9.025575932919E-02, -4.992325395815E-02, 1.078616370475E-02],
              [4.258073042507E-02, -8.922263347816E-03, -4.992325410530E-02, -6.404296514634E-02,
               -5.841121515804E-02, -2.874753182969E-02, 7.096489269439E-03, 3.166809126194E-02,
               4.805122991581E-02, 5.317057248858E-02, 4.805122997215E-02, 3.166809031751E-02,
               7.096489232857E-03, -2.874753192097E-02, -5.841121361736E-02, -6.404296449739E-02,
               -4.992325401194E-02, -8.922263276092E-03, 4.258073039986E-02],
              [7.107762963804E-02, 4.258073041915E-02, 1.078616372574E-02, -3.551989658162E-02,
               -7.588664278863E-02, -8.898818050755E-02, -8.866359526279E-02, -7.909406553441E-02,
               -6.792741699678E-02, -6.533539474332E-02, -6.792741636103E-02, -7.909406642640E-02,
               -8.866359491969E-02, -8.898818047505E-02, -7.588664248991E-02, -3.551989681164E-02,
               1.078616357365E-02, 4.258073036306E-02, 7.107762956389E-02]],

             [[7.076114333155E-02, 3.539309463577E-02, -1.191055431739E-03, -4.656675270908E-02,
               -8.287206522507E-02, -8.945863728305E-02, -8.226118588930E-02, -6.792741701965E-02,
               -5.367000806701E-02, -5.009611083921E-02, -5.367000806697E-02, -6.792741701966E-02,
               -8.226118588920E-02, -8.945863728263E-02, -8.287206522547E-02, -4.656675271188E-02,
               -1.191055427695E-03, 3.539309464677E-02, 7.076114334392E-02],
              [3.539309463623E-02, -1.715564614645E-02, -5.712280000133E-02, -6.343256195128E-02,
               -4.898599926486E-02, -1.407342396270E-02, 2.454870319541E-02, 4.805122989200E-02,
               6.236191878103E-02, 6.708857072924E-02, 6.236191878107E-02, 4.805122989200E-02,
               2.454870319554E-02, -1.407342396231E-02, -4.898599926542E-02, -6.343256195400E-02,
               -5.712279999801E-02, -1.715564613707E-02, 3.539309464675E-02],
              [-1.191055426103E-03, -5.712279999614E-02, -9.132287823491E-02, -6.181990385414E-02,
               -7.601370334046E-03, 4.969838979388E-02, 9.999127997674E-02, 1.205490947913E-01,
               1.276578283169E-01, 1.311161468594E-01, 1.276578283168E-01, 1.205490947913E-01,
               9.999127997692E-02, 4.969838979334E-02, -7.601370333691E-03, -6.181990385337E-02,
               -9.132287823300E-02, -5.712280000256E-02, -1.191055434331E-03],
              [-4.656675270370E-02, -6.343256194634E-02, -6.181990385421E-02, 2.183684689110E-03,
               7.075468372841E-02, 9.363466868707E-02, 9.204081655081E-02, 6.176314526296E-02,
               2.882470737192E-02, 2.092269372453E-02, 2.882470737188E-02, 6.176314526240E-02,
               9.204081654942E-02, 9.363466868443E-02, 7.075468372879E-02, 2.183684690012E-03,
               -6.181990385373E-02, -6.343256195148E-02, -4.656675270835E-02],
              [-8.287206522307E-02, -4.898599926321E-02, -7.601370331806E-03, 7.075468373031E-02,
               1.276578283168E-01, 9.594822035794E-02, 3.130524741587E-02, -3.882064949827E-02,
               -9.690247379714E-02, -1.129284389488E-01, -9.690247379792E-02, -3.882064949777E-02,
               3.130524741515E-02, 9.594822035782E-02, 1.276578283090E-01, 7.075468371988E-02,
               -7.601370330526E-03, -4.898599926235E-02, -8.287206522398E-02],
              [-8.945863728122E-02, -1.407342396118E-02, 4.969838979567E-02, 9.363466868850E-02,
               9.594822035779E-02, 6.256940894963E-03, -9.787657034478E-02, -1.446346315861E-01,
               -1.671645939885E-01, -1.758255969440E-01, -1.671645939868E-01, -1.446346315817E-01,
               -9.787657034363E-02, 6.256940894622E-03, 9.594822036880E-02, 9.363466869641E-02,
               4.969838979462E-02, -1.407342396318E-02, -8.945863728537E-02],
              [-8.226118588935E-02, 2.454870319554E-02, 9.999127997802E-02, 9.204081655190E-02,
               3.130524741551E-02, -9.787657034530E-02, -2.087231323859E-01, -1.847268995295E-01,
               -1.307271192063E-01, -1.209776175583E-01, -1.307271192052E-01, -1.847268995328E-01,
               -2.087231323695E-01, -9.787657033236E-02, 3.130524743216E-02, 9.204081656763E-02,
               9.999127998919E-02, 2.454870319340E-02, -8.226118590476E-02],
              [-6.792741701975E-02, 4.805122989214E-02, 1.205490947930E-01, 6.176314526445E-02,
               -3.882064949733E-02, -1.446346315857E-01, -1.847268995281E-01, -1.064405574133E-02,
               1.816770692938E-01, 2.274775653980E-01, 1.816770692907E-01, -1.064405574288E-02,
               -1.847268995117E-01, -1.446346315527E-01, -3.882064949057E-02, 6.176314527275E-02,
               1.205490947842E-01, 4.805122988833E-02, -6.792741701951E-02],
              [-5.367000806714E-02, 6.236191878092E-02, 1.276578283161E-01, 2.882470737116E-02,
               -9.690247379648E-02, -1.671645939881E-01, -1.307271192061E-01, 1.816770692972E-01,
               4.983093384721E-01, 5.769388956820E-01, 4.983093384749E-01, 1.816770692989E-01,
               -1.307271192044E-01, -1.671645939897E-01, -9.690247378566E-02, 2.882470738401E-02,
               1.276578283178E-01, 6.236191878103E-02, -5.367000806791E-02],
              [-5.009611083936E-02, 6.708857072912E-02, 1.311161468587E-01, 2.092269372384E-02,
               -1.129284389482E-01, -1.758255969436E-01, -1.209776175595E-01, 2.274775654005E-01,
               5.769388956820E-01, 6.641784598370E-01, 5.769388956837E-01, 2.274775654044E-01,
               -1.209776175574E-01, -1.758255969463E-01, -1.129284389420E-01, 2.092269373771E-02,
               1.311161468602E-01, 6.708857072936E-02, -5.009611083977E-02],
              [-5.367000805850E-02, 6.236191878795E-02, 1.276578283239E-01, 2.882470737707E-02,
               -9.690247378982E-02, -1.671645939886E-01, -1.307271192294E-01, 1.816770693011E-01,
               4.983093384708E-01, 5.769388956839E-01, 4.983093385397E-01, 1.816770693913E-01,
               -1.307271192016E-01, -1.671645940579E-01, -9.690247376485E-02, 2.882470747691E-02,
               1.276578284345E-01, 6.236191884354E-02, -5.367000805741E-02],
              [-6.792741700893E-02, 4.805122990084E-02, 1.205490947991E-01, 6.176314526916E-02,
               -3.882064949342E-02, -1.446346315848E-01, -1.847268995408E-01, -1.064405574160E-02,
               1.816770692987E-01, 2.274775654012E-01, 1.816770693750E-01, -1.064405568804E-02,
               -1.847268996718E-01, -1.446346317140E-01, -3.882064951065E-02, 6.176314528036E-02,
               1.205490947264E-01, 4.805122988050E-02, -6.792741697727E-02],
              [-8.226118588183E-02, 2.454870320244E-02, 9.999127998128E-02, 9.204081655429E-02,
               3.130524741315E-02, -9.787657034576E-02, -2.087231323640E-01, -1.847268995250E-01,
               -1.307271192046E-01, -1.209776175581E-01, -1.307271191882E-01, -1.847268995163E-01,
               -2.087231323813E-01, -9.787657034156E-02, 3.130524737942E-02, 9.204081650505E-02,
               9.999128008891E-02, 2.454870326451E-02, -8.226118585934E-02],
              [-8.945863727548E-02, -1.407342395551E-02, 4.969838979903E-02, 9.363466868986E-02,
               9.594822035668E-02, 6.256940894061E-03, -9.787657032559E-02, -1.446346315762E-01,
               -1.671645939903E-01, -1.758255969465E-01, -1.671645939125E-01, -1.446346315427E-01,
               -9.787657033149E-02, 6.256940904208E-03, 9.594822036864E-02, 9.363466869658E-02,
               4.969838984122E-02, -1.407342389327E-02, -8.945863718850E-02],
              [-8.287206521480E-02, -4.898599925710E-02, -7.601370339653E-03, 7.075468372381E-02,
               1.276578283356E-01, 9.594822037620E-02, 3.130524737365E-02, -3.882064953116E-02,
               -9.690247379474E-02, -1.129284389473E-01, -9.690247386015E-02, -3.882064953158E-02,
               3.130524754531E-02, 9.594822044161E-02, 1.276578283527E-01, 7.075468376669E-02,
               -7.601370233706E-03, -4.898599915604E-02, -8.287206511917E-02],
              [-4.656675269843E-02, -6.343256194321E-02, -6.181990386049E-02, 2.183684684013E-03,
               7.075468374764E-02, 9.363466872229E-02, 9.204081649681E-02, 6.176314520521E-02,
               2.882470737300E-02, 2.092269372215E-02, 2.882470730675E-02, 6.176314515209E-02,
               9.204081660656E-02, 9.363466882191E-02, 7.075468374937E-02, 2.183684721945E-03,
               -6.181990373532E-02, -6.343256176391E-02, -4.656675245442E-02],
              [-1.191055457627E-03, -5.712280002403E-02, -9.132287826520E-02, -6.181990387758E-02,
               -7.601370318339E-03, 4.969838980380E-02, 9.999127999067E-02, 1.205490947774E-01,
               1.276578283183E-01, 1.311161468604E-01, 1.276578282883E-01, 1.205490947693E-01,
               9.999127998363E-02, 4.969838971813E-02, -7.601370810044E-03, -6.181990436722E-02,
               -9.132287772520E-02, -5.712279971615E-02, -1.191055359107E-03],
              [3.539309461328E-02, -1.715564616621E-02, -5.712280003768E-02, -6.343256198112E-02,
               -4.898599924948E-02, -1.407342395194E-02, 2.454870318282E-02, 4.805122988286E-02,
               6.236191878209E-02, 6.708857072989E-02, 6.236191886101E-02, 4.805122992505E-02,
               2.454870317903E-02, -1.407342392236E-02, -4.898599961311E-02, -6.343256227596E-02,
               -5.712279953520E-02, -1.715564584955E-02, 3.539309476082E-02],
              [7.076114330639E-02, 3.539309461416E-02, -1.191055472435E-03, -4.656675274384E-02,
               -8.287206521028E-02, -8.945863727164E-02, -8.226118592860E-02, -6.792741702540E-02,
               -5.367000806633E-02, -5.009611083900E-02, -5.367000787769E-02, -6.792741693307E-02,
               -8.226118594013E-02, -8.945863713601E-02, -8.287206546646E-02, -4.656675286616E-02,
               -1.191054996523E-03, 3.539309494588E-02, 7.076114350452E-02]],

             [[7.106858491369E-02, 3.345900050499E-02, -4.769492603984E-03, -5.018094388034E-02,
               -8.555883978796E-02, -9.033251398505E-02, -8.112847434608E-02, -6.533539475251E-02,
               -5.009611083922E-02, -4.621706540085E-02, -5.009611083914E-02, -6.533539475250E-02,
               -8.112847434598E-02, -9.033251398465E-02, -8.555883978861E-02, -5.018094388386E-02,
               -4.769492599934E-03, 3.345900051698E-02, 7.106858492727E-02],
              [3.345900050538E-02, -1.975584208888E-02, -5.970849726269E-02, -6.384596898860E-02,
               -4.677849572217E-02, -1.007152730355E-02, 2.969356645637E-02, 5.317057247899E-02,
               6.708857072923E-02, 7.176294626335E-02, 6.708857072926E-02, 5.317057247900E-02,
               2.969356645650E-02, -1.007152730316E-02, -4.677849572310E-02, -6.384596899195E-02,
               -5.970849725937E-02, -1.975584207870E-02, 3.345900051687E-02],
              [-4.769492597823E-03, -5.970849725697E-02, -9.237922986116E-02, -5.935500573459E-02,
               -1.842790617506E-03, 5.642128549314E-02, 1.064514000496E-01, 1.255521603712E-01,
               1.311161468594E-01, 1.341992402106E-01, 1.311161468594E-01, 1.255521603714E-01,
               1.064514000500E-01, 5.642128549279E-02, -1.842790617148E-03, -5.935500573376E-02,
               -9.237922986180E-02, -5.970849726595E-02, -4.769492607739E-03],
              [-5.018094387443E-02, -6.384596898313E-02, -5.935500573466E-02, 7.217721869480E-03,
               7.671883932454E-02, 9.672805827698E-02, 9.097189627824E-02, 5.683811106129E-02,
               2.092269372459E-02, 1.216654340538E-02, 2.092269372514E-02, 5.683811106132E-02,
               9.097189627640E-02, 9.672805827461E-02, 7.671883932493E-02, 7.217721870364E-03,
               -5.935500573643E-02, -6.384596899092E-02, -5.018094388130E-02],
              [-8.555883978569E-02, -4.677849572030E-02, -1.842790614559E-03, 7.671883932702E-02,
               1.311161468593E-01, 9.302969389032E-02, 2.151761677208E-02, -5.255531921972E-02,
               -1.129284389488E-01, -1.297341959139E-01, -1.129284389497E-01, -5.255531922077E-02,
               2.151761677085E-02, 9.302969389044E-02, 1.311161468608E-01, 7.671883931800E-02,
               -1.842790614630E-03, -4.677849571879E-02, -8.555883978442E-02],
              [-9.033251398297E-02, -1.007152730181E-02, 5.642128549546E-02, 9.672805827883E-02,
               9.302969389023E-02, -3.352929968732E-03, -1.122576039961E-01, -1.568720565875E-01,
               -1.758255969442E-01, -1.837715960626E-01, -1.758255969420E-01, -1.568720565856E-01,
               -1.122576039957E-01, -3.352929969136E-03, 9.302969390393E-02, 9.672805828304E-02,
               5.642128548939E-02, -1.007152730268E-02, -9.033251398022E-02],
              [-8.112847434626E-02, 2.969356645640E-02, 1.064514000511E-01, 9.097189627931E-02,
               2.151761677106E-02, -1.122576039972E-01, -2.227007147076E-01, -1.871028118425E-01,
               -1.209776175580E-01, -1.082622304140E-01, -1.209776175621E-01, -1.871028118531E-01,
               -2.227007146840E-01, -1.122576039845E-01, 2.151761676118E-02, 9.097189627998E-02,
               1.064514000490E-01, 2.969356645531E-02, -8.112847434816E-02],
              [-6.533539475276E-02, 5.317057247901E-02, 1.255521603733E-01, 5.683811106288E-02,
               -5.255531921940E-02, -1.568720565878E-01, -1.871028118447E-01, 1.177815849103E-02,
               2.274775653984E-01, 2.793001125553E-01, 2.274775654104E-01, 1.177815849259E-02,
               -1.871028118404E-01, -1.568720565643E-01, -5.255531921284E-02, 5.683811105913E-02,
               1.255521603642E-01, 5.317057247876E-02, -6.533539474662E-02],
              [-5.009611083937E-02, 6.708857072912E-02, 1.311161468585E-01, 2.092269372375E-02,
               -1.129284389481E-01, -1.758255969435E-01, -1.209776175589E-01, 2.274775653995E-01,
               5.769388956820E-01, 6.641784598369E-01, 5.769388956871E-01, 2.274775654040E-01,
               -1.209776175557E-01, -1.758255969442E-01, -1.129284389382E-01, 2.092269373890E-02,
               1.311161468599E-01, 6.708857072939E-02, -5.009611083941E-02],
              [-4.621706540101E-02, 7.176294626322E-02, 1.341992402097E-01, 1.216654340456E-02,
               -1.297341959130E-01, -1.837715960619E-01, -1.082622304149E-01, 2.793001125562E-01,
               6.641784598370E-01, 7.607335461182E-01, 6.641784598436E-01, 2.793001125643E-01,
               -1.082622304125E-01, -1.837715960631E-01, -1.297341959078E-01, 1.216654341765E-02,
               1.341992402111E-01, 7.176294626355E-02, -4.621706540094E-02],
              [-5.009611082929E-02, 6.708857073729E-02, 1.311161468669E-01, 2.092269373071E-02,
               -1.129284389447E-01, -1.758255969403E-01, -1.209776175761E-01, 2.274775654073E-01,
               5.769388956821E-01, 6.641784598392E-01, 5.769388957902E-01, 2.274775655706E-01,
               -1.209776176134E-01, -1.758255969873E-01, -1.129284389492E-01, 2.092269373219E-02,
               1.311161468454E-01, 6.708857074029E-02, -5.009611080182E-02],
              [-6.533539473997E-02, 5.317057248932E-02, 1.255521603798E-01, 5.683811106817E-02,
               -5.255531921733E-02, -1.568720565885E-01, -1.871028118626E-01, 1.177815847917E-02,
               2.274775654026E-01, 2.793001125585E-01, 2.274775654727E-01, 1.177815853178E-02,
               -1.871028120696E-01, -1.568720567352E-01, -5.255531926419E-02, 5.683811098351E-02,
               1.255521602664E-01, 5.317057244278E-02, -6.533539471568E-02],
              [-8.112847433708E-02, 2.969356646476E-02, 1.064514000532E-01, 9.097189628181E-02,
               2.151761676876E-02, -1.122576039980E-01, -2.227007147036E-01, -1.871028118664E-01,
               -1.209776175591E-01, -1.082622304156E-01, -1.209776175661E-01, -1.871028118353E-01,
               -2.227007147055E-01, -1.122576039934E-01, 2.151761676833E-02, 9.097189625412E-02,
               1.064514002430E-01, 2.969356658068E-02, -8.112847429455E-02],
              [-9.033251397592E-02, -1.007152729493E-02, 5.642128549639E-02, 9.672805827983E-02,
               9.302969388905E-02, -3.352929970542E-03, -1.122576039919E-01, -1.568720565819E-01,
               -1.758255969480E-01, -1.837715960663E-01, -1.758255968953E-01, -1.568720565392E-01,
               -1.122576039853E-01, -3.352929960495E-03, 9.302969399592E-02, 9.672805838679E-02,
               5.642128552051E-02, -1.007152727745E-02, -9.033251396099E-02],
              [-8.555883977635E-02, -4.677849571350E-02, -1.842790624126E-03, 7.671883931907E-02,
               1.311161468705E-01, 9.302969390862E-02, 2.151761674686E-02, -5.255531925614E-02,
               -1.129284389441E-01, -1.297341959112E-01, -1.129284388699E-01, -5.255531922603E-02,
               2.151761699754E-02, 9.302969388848E-02, 1.311161468943E-01, 7.671883932526E-02,
               -1.842790500480E-03, -4.677849567124E-02, -8.555883981273E-02],
              [-5.018094386824E-02, -6.384596897954E-02, -5.935500574219E-02, 7.217721863504E-03,
               7.671883933864E-02, 9.672805830908E-02, 9.097189624100E-02, 5.683811102280E-02,
               2.092269372968E-02, 1.216654340648E-02, 2.092269371565E-02, 5.683811097553E-02,
               9.097189648010E-02, 9.672805833733E-02, 7.671883934767E-02, 7.217721867590E-03,
               -5.935500556780E-02, -6.384596884001E-02, -5.018094376223E-02],
              [-4.769492631281E-03, -5.970849728651E-02, -9.237922989701E-02, -5.935500576444E-02,
               -1.842790599765E-03, 5.642128550321E-02, 1.064514000740E-01, 1.255521603803E-01,
               1.311161468614E-01, 1.341992402127E-01, 1.311161468028E-01, 1.255521603490E-01,
               1.064513999985E-01, 5.642128534271E-02, -1.842791104900E-03, -5.935500643255E-02,
               -9.237922917914E-02, -5.970849661472E-02, -4.769491972703E-03],
              [3.345900048198E-02, -1.975584210904E-02, -5.970849730683E-02, -6.384596902583E-02,
               -4.677849570517E-02, -1.007152729181E-02, 2.969356645696E-02, 5.317057248757E-02,
               6.708857073094E-02, 7.176294626484E-02, 6.708857075698E-02, 5.317057249934E-02,
               2.969356640912E-02, -1.007152731531E-02, -4.677849614068E-02, -6.384596945303E-02,
               -5.970849648783E-02, -1.975584142900E-02, 3.345900106416E-02],
              [7.106858488811E-02, 3.345900048299E-02, -4.769492654475E-03, -5.018094392328E-02,
               -8.555883977183E-02, -9.033251397179E-02, -8.112847437000E-02, -6.533539474706E-02,
               -5.009611083778E-02, -4.621706539994E-02, -5.009611072596E-02, -6.533539468936E-02,
               -8.112847439502E-02, -9.033251386109E-02, -8.555884016392E-02, -5.018094414022E-02,
               -4.769491717269E-03, 3.345900118870E-02, 7.106858541135E-02]],

             [[7.076114327967E-02, 3.539309459055E-02, -1.191055490518E-03, -4.656675275385E-02,
               -8.287206522398E-02, -8.945863728256E-02, -8.226118588545E-02, -6.792741701812E-02,
               -5.367000806704E-02, -5.009611083919E-02, -5.367000806737E-02, -6.792741702094E-02,
               -8.226118588972E-02, -8.945863728364E-02, -8.287206517704E-02, -4.656675264442E-02,
               -1.191055423122E-03, 3.539309464836E-02, 7.076114334134E-02],
              [3.539309458953E-02, -1.715564618694E-02, -5.712280005535E-02, -6.343256199210E-02,
               -4.898599926404E-02, -1.407342396240E-02, 2.454870319866E-02, 4.805122989332E-02,
               6.236191878099E-02, 6.708857072924E-02, 6.236191878077E-02, 4.805122989092E-02,
               2.454870319327E-02, -1.407342396409E-02, -4.898599922392E-02, -6.343256189552E-02,
               -5.712279999462E-02, -1.715564613515E-02, 3.539309464617E-02],
              [-1.191055391193E-03, -5.712279996570E-02, -9.132287823463E-02, -6.181990385359E-02,
               -7.601370345190E-03, 4.969838978712E-02, 9.999127997660E-02, 1.205490947920E-01,
               1.276578283154E-01, 1.311161468583E-01, 1.276578283181E-01, 1.205490947929E-01,
               9.999127998180E-02, 4.969838979054E-02, -7.601370341319E-03, -6.181990386441E-02,
               -9.132287824058E-02, -5.712280003409E-02, -1.191055491039E-03],
              [-4.656675266097E-02, -6.343256190903E-02, -6.181990385330E-02, 2.183684690310E-03,
               7.075468372065E-02, 9.363466868121E-02, 9.204081654919E-02, 6.176314526324E-02,
               2.882470736941E-02, 2.092269372262E-02, 2.882470736501E-02, 6.176314526213E-02,
               9.204081658081E-02, 9.363466869309E-02, 7.075468371155E-02, 2.183684663622E-03,
               -6.181990385090E-02, -6.343256196830E-02, -4.656675274504E-02],
              [-8.287206520746E-02, -4.898599924998E-02, -7.601370360321E-03, 7.075468370968E-02,
               1.276578283121E-01, 9.594822035926E-02, 3.130524743288E-02, -3.882064950973E-02,
               -9.690247379281E-02, -1.129284389469E-01, -9.690247382406E-02, -3.882064954225E-02,
               3.130524738075E-02, 9.594822036331E-02, 1.276578283284E-01, 7.075468366528E-02,
               -7.601370386789E-03, -4.898599928180E-02, -8.287206520850E-02],
              [-8.945863726492E-02, -1.407342394726E-02, 4.969838976470E-02, 9.363466866457E-02,
               9.594822035352E-02, 6.256940893954E-03, -9.787657032100E-02, -1.446346315731E-01,
               -1.671645939839E-01, -1.758255969425E-01, -1.671645939401E-01, -1.446346315444E-01,
               -9.787657030364E-02, 6.256940937112E-03, 9.594822050100E-02, 9.363466885390E-02,
               4.969838983620E-02, -1.407342391962E-02, -8.945863723985E-02],
              [-8.226118586487E-02, 2.454870321251E-02, 9.999127995173E-02, 9.204081653003E-02,
               3.130524741332E-02, -9.787657036283E-02, -2.087231325123E-01, -1.847268995602E-01,
               -1.307271192216E-01, -1.209776175621E-01, -1.307271188894E-01, -1.847268993452E-01,
               -2.087231322424E-01, -9.787657033351E-02, 3.130524746824E-02, 9.204081671129E-02,
               9.999127987177E-02, 2.454870267245E-02, -8.226118682526E-02],
              [-6.792741699123E-02, 4.805122991242E-02, 1.205490947706E-01, 6.176314524771E-02,
               -3.882064954203E-02, -1.446346316051E-01, -1.847268995174E-01, -1.064405577627E-02,
               1.816770692926E-01, 2.274775653917E-01, 1.816770694867E-01, -1.064405578071E-02,
               -1.847268991738E-01, -1.446346315413E-01, -3.882064975883E-02, 6.176314495382E-02,
               1.205490934302E-01, 4.805122890517E-02, -6.792741766209E-02],
              [-5.367000805292E-02, 6.236191879248E-02, 1.276578283158E-01, 2.882470737071E-02,
               -9.690247379774E-02, -1.671645939962E-01, -1.307271192350E-01, 1.816770692999E-01,
               4.983093384638E-01, 5.769388956764E-01, 4.983093385006E-01, 1.816770693265E-01,
               -1.307271190987E-01, -1.671645939370E-01, -9.690247368600E-02, 2.882470753329E-02,
               1.276578284850E-01, 6.236191888845E-02, -5.367000801775E-02],
              [-5.009611082286E-02, 6.708857074272E-02, 1.311161468591E-01, 2.092269372422E-02,
               -1.129284389509E-01, -1.758255969458E-01, -1.209776175875E-01, 2.274775653814E-01,
               5.769388956779E-01, 6.641784598324E-01, 5.769388957282E-01, 2.274775654290E-01,
               -1.209776175879E-01, -1.758255969965E-01, -1.129284388244E-01, 2.092269388292E-02,
               1.311161469477E-01, 6.708857080694E-02, -5.009611076800E-02],
              [-5.367000796420E-02, 6.236191887842E-02, 1.276578283544E-01, 2.882470742389E-02,
               -9.690247407708E-02, -1.671645943460E-01, -1.307271192659E-01, 1.816770691477E-01,
               4.983093383813E-01, 5.769388956419E-01, 4.983093401931E-01, 1.816770711934E-01,
               -1.307271172815E-01, -1.671645934002E-01, -9.690247164471E-02, 2.882470880013E-02,
               1.276578281661E-01, 6.236191841170E-02, -5.367000858515E-02],
              [-6.792741690440E-02, 4.805123000226E-02, 1.205490948439E-01, 6.176314533317E-02,
               -3.882064990195E-02, -1.446346319719E-01, -1.847268996363E-01, -1.064405596223E-02,
               1.816770692041E-01, 2.274775653156E-01, 1.816770714082E-01, -1.064405352085E-02,
               -1.847268977002E-01, -1.446346310236E-01, -3.882064865943E-02, 6.176314581177E-02,
               1.205490931469E-01, 4.805122819033E-02, -6.792741878248E-02],
              [-8.226118589033E-02, 2.454870319094E-02, 9.999128004044E-02, 9.204081661031E-02,
               3.130524716541E-02, -9.787657051138E-02, -2.087231319942E-01, -1.847268995261E-01,
               -1.307271192524E-01, -1.209776176375E-01, -1.307271192325E-01, -1.847268988849E-01,
               -2.087231327216E-01, -9.787657036047E-02, 3.130524735867E-02, 9.204081532922E-02,
               9.999127953835E-02, 2.454870307988E-02, -8.226118568527E-02],
              [-8.945863727899E-02, -1.407342396303E-02, 4.969838985732E-02, 9.363466871527E-02,
               9.594822018729E-02, 6.256940728116E-03, -9.787657002570E-02, -1.446346312104E-01,
               -1.671645941221E-01, -1.758255970245E-01, -1.671645938294E-01, -1.446346306406E-01,
               -9.787657117684E-02, 6.256939713217E-03, 9.594822194974E-02, 9.363467210799E-02,
               4.969839016807E-02, -1.407342370556E-02, -8.945863713552E-02],
              [-8.287206516539E-02, -4.898599924774E-02, -7.601370463971E-03, 7.075468359926E-02,
               1.276578278491E-01, 9.594822016402E-02, 3.130524797259E-02, -3.882064937358E-02,
               -9.690247362411E-02, -1.129284388382E-01, -9.690247319708E-02, -3.882064968067E-02,
               3.130524760960E-02, 9.594822137458E-02, 1.276578283031E-01, 7.075468404088E-02,
               -7.601379023506E-03, -4.898600483385E-02, -8.287206783633E-02],
              [-4.656675265253E-02, -6.343256193781E-02, -6.181990400645E-02, 2.183684533694E-03,
               7.075468306033E-02, 9.363466848760E-02, 9.204081753532E-02, 6.176314551222E-02,
               2.882470753457E-02, 2.092269384214E-02, 2.882470829726E-02, 6.176314499778E-02,
               9.204081432189E-02, 9.363466939006E-02, 7.075468394321E-02, 2.183685260982E-03,
               -6.181991036938E-02, -6.343256584795E-02, -4.656675450616E-02],
              [-1.191055378347E-03, -5.712279994538E-02, -9.132287820852E-02, -6.181990384457E-02,
               -7.601370384444E-03, 4.969838982428E-02, 9.999128059486E-02, 1.205490943216E-01,
               1.276578283890E-01, 1.311161468754E-01, 1.276578292573E-01, 1.205490952561E-01,
               9.999127996652E-02, 4.969839011935E-02, -7.601375723063E-03, -6.181991346881E-02,
               -9.132287581414E-02, -5.712279927363E-02, -1.191056301131E-03],
              [3.539309468222E-02, -1.715564609454E-02, -5.712279998528E-02, -6.343256194441E-02,
               -4.898599922652E-02, -1.407342391346E-02, 2.454870345187E-02, 4.805122923712E-02,
               6.236191875919E-02, 6.708857069848E-02, 6.236192359127E-02, 4.805123288184E-02,
               2.454870353245E-02, -1.407342339253E-02, -4.898599945713E-02, -6.343256509842E-02,
               -5.712279874903E-02, -1.715564563337E-02, 3.539309440073E-02],
              [7.076114336328E-02, 3.539309467735E-02, -1.191055422377E-03, -4.656675270080E-02,
               -8.287206510042E-02, -8.945863721591E-02, -8.226118601475E-02, -6.792741789612E-02,
               -5.367000818220E-02, -5.009611091747E-02, -5.366999939629E-02, -6.792741153585E-02,
               -8.226118522084E-02, -8.945863646811E-02, -8.287206018142E-02, -4.656674954932E-02,
               -1.191055250193E-03, 3.539309498560E-02, 7.076114374065E-02]],

             [[7.107762952707E-02, 4.258073032606E-02, 1.078616355921E-02, -3.551989673851E-02,
               -7.588664280273E-02, -8.898818051476E-02, -8.866359515248E-02, -7.909406560860E-02,
               -6.792741701974E-02, -6.533539475255E-02, -6.792741702051E-02, -7.909406561193E-02,
               -8.866359515732E-02, -8.898818051619E-02, -7.588664275250E-02, -3.551989662180E-02,
               1.078616362776E-02, 4.258073039038E-02, 7.107762959552E-02],
              [4.258073032494E-02, -8.922263431163E-03, -4.992325424761E-02, -6.404296527597E-02,
               -5.841121515985E-02, -2.874753183597E-02, 7.096489199790E-03, 3.166809106764E-02,
               4.805122989192E-02, 5.317057247890E-02, 4.805122989288E-02, 3.166809106499E-02,
               7.096489194328E-03, -2.874753183816E-02, -5.841121511623E-02, -6.404296517168E-02,
               -4.992325418384E-02, -8.922263372033E-03, 4.258073038932E-02],
              [1.078616366819E-02, -4.992325414855E-02, -9.025575972236E-02, -7.238462716259E-02,
               -2.828238482815E-02, 2.749289580213E-02, 8.052457015198E-02, 1.073372671581E-01,
               1.205490947895E-01, 1.255521603698E-01, 1.205490947918E-01, 1.073372671545E-01,
               8.052457014500E-02, 2.749289581246E-02, -2.828238481473E-02, -7.238462716200E-02,
               -9.025575975183E-02, -4.992325424111E-02, 1.078616354801E-02],
              [-3.551989663612E-02, -6.404296518380E-02, -7.238462716238E-02, -1.545550682562E-02,
               5.220189526021E-02, 8.693183182014E-02, 1.011634188764E-01, 8.452337729476E-02,
               6.176314526111E-02, 5.683811105891E-02, 6.176314525804E-02, 8.452337728612E-02,
               1.011634188666E-01, 8.693183180973E-02, 5.220189527412E-02, -1.545550682051E-02,
               -7.238462725349E-02, -6.404296527345E-02, -3.551989668405E-02],
              [-7.588664278208E-02, -5.841121514237E-02, -2.828238484788E-02, 5.220189524537E-02,
               1.205490947910E-01, 1.119722038735E-01, 7.105670830959E-02, 1.302475523958E-02,
               -3.882064949879E-02, -5.255531921986E-02, -3.882064961590E-02, 1.302475508159E-02,
               7.105670827470E-02, 1.119722037888E-01, 1.205490949107E-01, 5.220189535842E-02,
               -2.828238480423E-02, -5.841121515329E-02, -7.588664279782E-02],
              [-8.898818049297E-02, -2.874753181732E-02, 2.749289577517E-02, 8.693183179974E-02,
               1.119722038720E-01, 4.401715541080E-02, -4.673523152684E-02, -1.056041462635E-01,
               -1.446346315851E-01, -1.568720565870E-01, -1.446346315370E-01, -1.056041463051E-01,
               -4.673523150286E-02, 4.401715539990E-02, 1.119722041441E-01, 8.693183223719E-02,
               2.749289582063E-02, -2.874753180422E-02, -8.898818045956E-02],
              [-8.866359513742E-02, 7.096489208817E-03, 8.052457012614E-02, 1.011634188577E-01,
               7.105670826182E-02, -4.673523157644E-02, -1.657196010013E-01, -1.900217695481E-01,
               -1.847268995056E-01, -1.871028118404E-01, -1.847268994482E-01, -1.900217696947E-01,
               -1.657196014345E-01, -4.673523208129E-02, 7.105670849838E-02, 1.011634186506E-01,
               8.052457016429E-02, 7.096488773811E-03, -8.866359604713E-02],
              [-7.909406558590E-02, 3.166809108287E-02, 1.073372671349E-01, 8.452337727935E-02,
               1.302475517962E-02, -1.056041463232E-01, -1.900217695262E-01, -1.130233203815E-01,
               -1.064405573034E-02, 1.177815851144E-02, -1.064405552084E-02, -1.130233204233E-01,
               -1.900217696439E-01, -1.056041467455E-01, 1.302475537454E-02, 8.452337672875E-02,
               1.073372659517E-01, 3.166809011550E-02, -7.909406633348E-02],
              [-6.792741700353E-02, 4.805122990509E-02, 1.205490947908E-01, 6.176314526240E-02,
               -3.882064950431E-02, -1.446346315955E-01, -1.847268995663E-01, -1.064405576823E-02,
               1.816770692895E-01, 2.274775653923E-01, 1.816770692731E-01, -1.064405571794E-02,
               -1.847268995105E-01, -1.446346315488E-01, -3.882064938471E-02, 6.176314535495E-02,
               1.205490949456E-01, 4.805122995604E-02, -6.792741704775E-02],
              [-6.533539473369E-02, 5.317057249448E-02, 1.255521603717E-01, 5.683811106188E-02,
               -5.255531922905E-02, -1.568720565946E-01, -1.871028118717E-01, 1.177815845872E-02,
               2.274775653980E-01, 2.793001125484E-01, 2.274775653737E-01, 1.177815850847E-02,
               -1.871028119293E-01, -1.568720566252E-01, -5.255531918323E-02, 5.683811112313E-02,
               1.255521603992E-01, 5.317057250279E-02, -6.533539472826E-02],
              [-6.792741691959E-02, 4.805122998793E-02, 1.205490948419E-01, 6.176314533779E-02,
               -3.882064998223E-02, -1.446346319656E-01, -1.847268994643E-01, -1.064405573908E-02,
               1.816770692931E-01, 2.274775652911E-01, 1.816770681724E-01, -1.064405421825E-02,
               -1.847269012148E-01, -1.446346317925E-01, -3.882064833708E-02, 6.176314462656E-02,
               1.205490954029E-01, 4.805122992908E-02, -6.792741751591E-02],
              [-7.909406549774E-02, 3.166809117504E-02, 1.073372672229E-01, 8.452337738405E-02,
               1.302475477149E-02, -1.056041467239E-01, -1.900217699404E-01, -1.130233208412E-01,
               -1.064405592347E-02, 1.177815826517E-02, -1.064405485507E-02, -1.130233166591E-01,
               -1.900217687938E-01, -1.056041451132E-01, 1.302475589338E-02, 8.452337743339E-02,
               1.073372664088E-01, 3.166809085984E-02, -7.909406525661E-02],
              [-8.866359515914E-02, 7.096489190597E-03, 8.052457022710E-02, 1.011634189629E-01,
               7.105670807044E-02, -4.673523179450E-02, -1.657196014950E-01, -1.900217701198E-01,
               -1.847268997063E-01, -1.871028119498E-01, -1.847268993405E-01, -1.900217697598E-01,
               -1.657195995761E-01, -4.673523054821E-02, 7.105670931917E-02, 1.011634187663E-01,
               8.052457022209E-02, 7.096489437984E-03, -8.866359473151E-02],
              [-8.898818051207E-02, -2.874753183803E-02, 2.749289588699E-02, 8.693183187050E-02,
               1.119722036190E-01, 4.401715514118E-02, -4.673523151349E-02, -1.056041459662E-01,
               -1.446346317292E-01, -1.568720566759E-01, -1.446346313549E-01, -1.056041466048E-01,
               -4.673523146657E-02, 4.401715505756E-02, 1.119722044307E-01, 8.693183205791E-02,
               2.749289610450E-02, -2.874753133494E-02, -8.898817981858E-02],
              [-7.588664267248E-02, -5.841121509036E-02, -2.828238495229E-02, 5.220189513493E-02,
               1.205490944039E-01, 1.119722038022E-01, 7.105670864908E-02, 1.302475559928E-02,
               -3.882064934503E-02, -5.255531920200E-02, -3.882064885441E-02, 1.302475611187E-02,
               7.105671053670E-02, 1.119722089642E-01, 1.205490952280E-01, 5.220189559968E-02,
               -2.828238249166E-02, -5.841121586136E-02, -7.588664641701E-02],
              [-3.551989655701E-02, -6.404296516314E-02, -7.238462732086E-02, -1.545550698230E-02,
               5.220189473309E-02, 8.693183154247E-02, 1.011634193360E-01, 8.452337754687E-02,
               6.176314541264E-02, 5.683811116427E-02, 6.176314625193E-02, 8.452337827354E-02,
               1.011634210581E-01, 8.693183558969E-02, 5.220189534067E-02, -1.545550619544E-02,
               -7.238462347684E-02, -6.404296439026E-02, -3.551989857074E-02],
              [1.078616367543E-02, -4.992325412997E-02, -9.025575971863E-02, -7.238462716050E-02,
               -2.828238481341E-02, 2.749289579781E-02, 8.052456993293E-02, 1.073372663303E-01,
               1.205490948604E-01, 1.255521604231E-01, 1.205490953390E-01, 1.073372674401E-01,
               8.052457048943E-02, 2.749289581157E-02, -2.828239329243E-02, -7.238463491852E-02,
               -9.025576045276E-02, -4.992325336572E-02, 1.078616599417E-02],
              [4.258073042625E-02, -8.922263329241E-03, -4.992325417658E-02, -6.404296523378E-02,
               -5.841121504428E-02, -2.874753178732E-02, 7.096488894100E-03, 3.166809024859E-02,
               4.805122989034E-02, 5.317057245954E-02, 4.805123136846E-02, 3.166809192507E-02,
               7.096489807257E-03, -2.874753147161E-02, -5.841122265269E-02, -6.404297218683E-02,
               -4.992325441153E-02, -8.922262575770E-03, 4.258073218394E-02],
              [7.107762961917E-02, 4.258073042082E-02, 1.078616364622E-02, -3.551989669348E-02,
               -7.588664257456E-02, -8.898818040620E-02, -8.866359551230E-02, -7.909406639700E-02,
               -6.792741709037E-02, -6.533539484262E-02, -6.792741470494E-02, -7.909406425265E-02,
               -8.866359428676E-02, -8.898817979424E-02, -7.588664966370E-02, -3.551990279207E-02,
               1.078616382979E-02, 4.258073111114E-02, 7.107763079746E-02]],

             [[6.830161081549E-02, 5.086584780000E-02, 2.743644198792E-02, -1.765322330113E-02,
               -6.137786874307E-02, -8.246762368037E-02, -9.132287823633E-02, -8.866359515807E-02,
               -8.226118588995E-02, -8.112847434679E-02, -8.226118588649E-02, -8.866359515401E-02,
               -9.132287824817E-02, -8.246762368799E-02, -6.137786865325E-02, -1.765322324576E-02,
               2.743644191464E-02, 5.086584765051E-02, 6.830161063722E-02],
              [5.086584780757E-02, 3.645921496819E-03, -3.643722187827E-02, -6.012530562324E-02,
               -6.664602762474E-02, -4.608859724759E-02, -1.661020289379E-02, 7.096489194836E-03,
               2.454870319491E-02, 2.969356645586E-02, 2.454870319729E-02, 7.096489198445E-03,
               -1.661020290275E-02, -4.608859725297E-02, -6.664602755725E-02, -6.012530558200E-02,
               -3.643722192450E-02, 3.645921378228E-03, 5.086584766311E-02],
              [2.743644197273E-02, -3.643722188970E-02, -8.287206521893E-02, -8.153908880474E-02,
               -5.367000805963E-02, -4.084775795396E-03, 4.831031939152E-02, 8.052457014975E-02,
               9.999127997785E-02, 1.064514000505E-01, 9.999127997785E-02, 8.052457014886E-02,
               4.831031938536E-02, -4.084775799809E-03, -5.367000808772E-02, -8.153908883497E-02,
               -8.287206529782E-02, -3.643722198505E-02, 2.743644186421E-02],
              [-1.765322331710E-02, -6.012530563591E-02, -8.153908880380E-02, -3.808861274759E-02,
               2.304143896083E-02, 6.888755196917E-02, 1.005921787221E-01, 1.011634188660E-01,
               9.204081655015E-02, 9.097189627835E-02, 9.204081656094E-02, 1.011634188878E-01,
               1.005921787247E-01, 6.888755199393E-02, 2.304143893488E-02, -3.808861278009E-02,
               -8.153908882121E-02, -6.012530573307E-02, -1.765322349251E-02],
              [-6.137786874012E-02, -6.664602762132E-02, -5.367000808832E-02, 2.304143894897E-02,
               9.999127997497E-02, 1.192964320985E-01, 1.096062925817E-01, 7.105670830149E-02,
               3.130524741635E-02, 2.151761677232E-02, 3.130524744180E-02, 7.105670833568E-02,
               1.096062925722E-01, 1.192964321229E-01, 9.999128002013E-02, 2.304143899060E-02,
               -5.367000815663E-02, -6.664602764618E-02, -6.137786871753E-02],
              [-8.246762366171E-02, -4.608859722999E-02, -4.084775809293E-03, 6.888755196863E-02,
               1.192964320975E-01, 8.313590780511E-02, 1.717169893236E-02, -4.673523150067E-02,
               -9.787657034315E-02, -1.122576039952E-01, -9.787657029892E-02, -4.673523153591E-02,
               1.717169892268E-02, 8.313590784422E-02, 1.192964323661E-01, 6.888755210470E-02,
               -4.084776130418E-03, -4.608859742786E-02, -8.246762372152E-02],
              [-9.132287823311E-02, -1.661020289229E-02, 4.831031933523E-02, 1.005921786789E-01,
               1.096062925302E-01, 1.717169886156E-02, -9.690247374945E-02, -1.657196009568E-01,
               -2.087231323794E-01, -2.227007147136E-01, -2.087231323497E-01, -1.657196012871E-01,
               -9.690247393149E-02, 1.717169879975E-02, 1.096062928866E-01, 1.005921789240E-01,
               4.831031943372E-02, -1.661020287737E-02, -9.132287825277E-02],
              [-8.866359515060E-02, 7.096489199433E-03, 8.052457007915E-02, 1.011634188137E-01,
               7.105670823745E-02, -4.673523158235E-02, -1.657196010360E-01, -1.900217695948E-01,
               -1.847268995428E-01, -1.871028118619E-01, -1.847268995629E-01, -1.900217699444E-01,
               -1.657196008327E-01, -4.673523148733E-02, 7.105670851046E-02, 1.011634186621E-01,
               8.052457022374E-02, 7.096489155748E-03, -8.866359531844E-02],
              [-8.226118589492E-02, 2.454870319083E-02, 9.999127999257E-02, 9.204081656480E-02,
               3.130524741657E-02, -9.787657034257E-02, -2.087231323930E-01, -1.847268995532E-01,
               -1.307271192043E-01, -1.209776175579E-01, -1.307271191983E-01, -1.847268995670E-01,
               -2.087231323778E-01, -9.787657033647E-02, 3.130524739151E-02, 9.204081650399E-02,
               9.999128005227E-02, 2.454870325807E-02, -8.226118583517E-02],
              [-8.112847435078E-02, 2.969356645250E-02, 1.064514000677E-01, 9.097189629475E-02,
               2.151761677279E-02, -1.122576039937E-01, -2.227007147264E-01, -1.871028118997E-01,
               -1.209776175578E-01, -1.082622304136E-01, -1.209776175462E-01, -1.871028118898E-01,
               -2.227007147054E-01, -1.122576039938E-01, 2.151761663527E-02, 9.097189608273E-02,
               1.064514001234E-01, 2.969356655404E-02, -8.112847422282E-02],
              [-8.226118583541E-02, 2.454870324113E-02, 9.999128035920E-02, 9.204081688289E-02,
               3.130524739965E-02, -9.787657032605E-02, -2.087231322959E-01, -1.847268997807E-01,
               -1.307271191081E-01, -1.209776175121E-01, -1.307271208063E-01, -1.847269007083E-01,
               -2.087231330233E-01, -9.787657058348E-02, 3.130524722381E-02, 9.204081519458E-02,
               9.999127987698E-02, 2.454870332739E-02, -8.226118553065E-02],
              [-8.866359509720E-02, 7.096489246526E-03, 8.052457056292E-02, 1.011634192272E-01,
               7.105670834796E-02, -4.673523151625E-02, -1.657196014781E-01, -1.900217702716E-01,
               -1.847268994978E-01, -1.871028117890E-01, -1.847269013234E-01, -1.900217699207E-01,
               -1.657196004475E-01, -4.673523045189E-02, 7.105670668996E-02, 1.011634182736E-01,
               8.052457069941E-02, 7.096489516884E-03, -8.866359507022E-02],
              [-9.132287810966E-02, -1.661020279085E-02, 4.831031935249E-02, 1.005921787215E-01,
               1.096062927182E-01, 1.717169902895E-02, -9.690247400129E-02, -1.657196011092E-01,
               -2.087231323878E-01, -2.227007147126E-01, -2.087231326654E-01, -1.657196004965E-01,
               -9.690247480764E-02, 1.717169808993E-02, 1.096062900687E-01, 1.005921768848E-01,
               4.831031975955E-02, -1.661020290024E-02, -9.132287877269E-02],
              [-8.246762354326E-02, -4.608859713863E-02, -4.084775869754E-03, 6.888755195529E-02,
               1.192964322752E-01, 8.313590794666E-02, 1.717169878956E-02, -4.673523165495E-02,
               -9.787657034486E-02, -1.122576039969E-01, -9.787657087767E-02, -4.673523167670E-02,
               1.717169810473E-02, 8.313590759347E-02, 1.192964311198E-01, 6.888755190318E-02,
               -4.084774971946E-03, -4.608859680266E-02, -8.246762354807E-02],
              [-6.137786876510E-02, -6.664602762535E-02, -5.367000829276E-02, 2.304143878235E-02,
               9.999128062830E-02, 1.192964327548E-01, 1.096062934224E-01, 7.105670919109E-02,
               3.130524744019E-02, 2.151761681342E-02, 3.130524690865E-02, 7.105670753408E-02,
               1.096062912455E-01, 1.192964310658E-01, 9.999127898200E-02, 2.304143881758E-02,
               -5.367000516612E-02, -6.664602580967E-02, -6.137786771557E-02],
              [-1.765322338911E-02, -6.012530567755E-02, -8.153908907773E-02, -3.808861296454E-02,
               2.304143971968E-02, 6.888755257237E-02, 1.005921800500E-01, 1.011634200585E-01,
               9.204081664655E-02, 9.097189638937E-02, 9.204081839108E-02, 1.011634213492E-01,
               1.005921791670E-01, 6.888755045493E-02, 2.304143795157E-02, -3.808861245539E-02,
               -8.153908567342E-02, -6.012530304330E-02, -1.765322087138E-02],
              [2.743644247553E-02, -3.643722144114E-02, -8.287206489244E-02, -8.153908845847E-02,
               -5.367000807060E-02, -4.084775614426E-03, 4.831031959875E-02, 8.052457044086E-02,
               9.999127993078E-02, 1.064513999568E-01, 9.999128013703E-02, 8.052457050380E-02,
               4.831031679515E-02, -4.084775132394E-03, -5.367000285983E-02, -8.153908412152E-02,
               -8.287207299795E-02, -3.643722642023E-02, 2.743644022599E-02],
              [5.086584820876E-02, 3.645921862665E-03, -3.643722155807E-02, -6.012530527076E-02,
               -6.664602749885E-02, -4.608859700659E-02, -1.661020234173E-02, 7.096489586233E-03,
               2.454870314417E-02, 2.969356633119E-02, 2.454870302236E-02, 7.096489544698E-03,
               -1.661020190982E-02, -4.608859716962E-02, -6.664601716914E-02, -6.012529846892E-02,
               -3.643722426414E-02, 3.645924484747E-03, 5.086585615402E-02],
              [6.830161119920E-02, 5.086584815401E-02, 2.743644231431E-02, -1.765322292982E-02,
               -6.137786849815E-02, -8.246762338822E-02, -9.132287734143E-02, -8.866359466284E-02,
               -8.226118594735E-02, -8.112847450493E-02, -8.226118641415E-02, -8.866359482038E-02,
               -9.132287314228E-02, -8.246762394929E-02, -6.137785306170E-02, -1.765321357953E-02,
               2.743644447193E-02, 5.086585812618E-02, 6.830162935898E-02]],

             [[5.936207330148E-02, 5.828995998104E-02, 4.830380230651E-02, 9.452397771605E-03,
               -3.406682304527E-02, -6.297844908817E-02, -8.246762366819E-02, -8.898818051669E-02,
               -8.945863728315E-02, -9.033251398530E-02, -8.945863727988E-02, -8.898818051232E-02,
               -8.246762368203E-02, -6.297844909684E-02, -3.406682294525E-02, 9.452397831150E-03,
               4.830380222126E-02, 5.828995982090E-02, 5.936207311212E-02],
              [5.828995998872E-02, 2.170304556191E-02, -1.282052488587E-02, -4.590027521068E-02,
               -6.742425746411E-02, -6.217010896475E-02, -4.608859723724E-02, -2.874753183769E-02,
               -1.407342396266E-02, -1.007152730367E-02, -1.407342396108E-02, -2.874753183428E-02,
               -4.608859724724E-02, -6.217010897030E-02, -6.742425738664E-02, -4.590027516346E-02,
               -1.282052494812E-02, 2.170304543272E-02, 5.828995983823E-02],
              [4.830380228315E-02, -1.282052490525E-02, -6.223172113317E-02, -8.224151799688E-02,
               -7.906219680635E-02, -4.598683888089E-02, -4.084775800923E-03, 2.749289580913E-02,
               4.969838979536E-02, 5.642128549477E-02, 4.969838979608E-02, 2.749289581055E-02,
               -4.084775815999E-03, -4.598683890269E-02, -7.906219683809E-02, -8.224151803159E-02,
               -6.223172123726E-02, -1.282052503454E-02, 4.830380212788E-02],
              [9.452397747834E-03, -4.590027523084E-02, -8.224151799565E-02, -6.344434623535E-02,
               -2.184399201316E-02, 2.574993055765E-02, 6.888755196137E-02, 8.693183182222E-02,
               9.363466868912E-02, 9.672805827840E-02, 9.363466868734E-02, 8.693183183885E-02,
               6.888755195665E-02, 2.574993057194E-02, -2.184399204970E-02, -6.344434627570E-02,
               -8.224151801432E-02, -4.590027532069E-02, 9.452397581382E-03],
              [-3.406682304192E-02, -6.742425746025E-02, -7.906219683702E-02, -2.184399202545E-02,
               4.969838979169E-02, 9.400421084337E-02, 1.192964321088E-01, 1.119722038848E-01,
               9.594822035873E-02, 9.302969389165E-02, 9.594822032548E-02, 1.119722039432E-01,
               1.192964320864E-01, 9.400421087293E-02, 4.969838983894E-02, -2.184399197736E-02,
               -7.906219697767E-02, -6.742425755468E-02, -3.406682308156E-02],
              [-6.297844906661E-02, -6.217010894450E-02, -4.598683889506E-02, 2.574993055721E-02,
               9.400421084135E-02, 1.021244353650E-01, 8.313590784826E-02, 4.401715545127E-02,
               6.256940896255E-03, -3.352929967236E-03, 6.256941014960E-03, 4.401715552747E-02,
               8.313590785672E-02, 1.021244353890E-01, 9.400421094899E-02, 2.574993068852E-02,
               -4.598683904563E-02, -6.217010904629E-02, -6.297844911074E-02],
              [-8.246762366501E-02, -4.608859723593E-02, -4.084775868034E-03, 6.888755190498E-02,
               1.192964320696E-01, 8.313590777546E-02, 1.717169883957E-02, -4.673523156601E-02,
               -9.787657034052E-02, -1.122576039951E-01, -9.787657038262E-02, -4.673523163610E-02,
               1.717169900518E-02, 8.313590785232E-02, 1.192964326214E-01, 6.888755295504E-02,
               -4.084775574666E-03, -4.608859738783E-02, -8.246762419398E-02],
              [-8.898818050847E-02, -2.874753183252E-02, 2.749289572871E-02, 8.693183175382E-02,
               1.119722038313E-01, 4.401715539091E-02, -4.673523161159E-02, -1.056041463140E-01,
               -1.446346315845E-01, -1.568720565842E-01, -1.446346312662E-01, -1.056041464726E-01,
               -4.673523142036E-02, 4.401715539779E-02, 1.119722041978E-01, 8.693183188881E-02,
               2.749289584962E-02, -2.874753203191E-02, -8.898818095707E-02],
              [-8.945863728912E-02, -1.407342396755E-02, 4.969838981216E-02, 9.363466870317E-02,
               9.594822035910E-02, 6.256940897188E-03, -9.787657034902E-02, -1.446346315940E-01,
               -1.671645939885E-01, -1.758255969424E-01, -1.671645940187E-01, -1.446346316432E-01,
               -9.787657033274E-02, 6.256940900951E-03, 9.594822045657E-02, 9.363466873942E-02,
               4.969838987085E-02, -1.407342379381E-02, -8.945863701552E-02],
              [-9.033251399012E-02, -1.007152730757E-02, 5.642128551365E-02, 9.672805829512E-02,
               9.302969389215E-02, -3.352929966261E-03, -1.122576040162E-01, -1.568720566283E-01,
               -1.758255969467E-01, -1.837715960617E-01, -1.758255969401E-01, -1.568720566178E-01,
               -1.122576039901E-01, -3.352929968586E-03, 9.302969391554E-02, 9.672805824120E-02,
               5.642128549494E-02, -1.007152710184E-02, -9.033251357911E-02],
              [-8.945863722023E-02, -1.407342390985E-02, 4.969839017871E-02, 9.363466902410E-02,
               9.594822043635E-02, 6.256940927716E-03, -9.787657094050E-02, -1.446346320840E-01,
               -1.671645938782E-01, -1.758255968814E-01, -1.671645957027E-01, -1.446346331834E-01,
               -9.787656870640E-02, 6.256940573300E-03, 9.594822065315E-02, 9.363467166077E-02,
               4.969839010308E-02, -1.407342363761E-02, -8.945863692665E-02],
              [-8.898818044544E-02, -2.874753177807E-02, 2.749289623773E-02, 8.693183219784E-02,
               1.119722039713E-01, 4.401715553160E-02, -4.673523222903E-02, -1.056041472098E-01,
               -1.446346315726E-01, -1.568720565568E-01, -1.446346348219E-01, -1.056041470581E-01,
               -4.673522957056E-02, 4.401715675000E-02, 1.119722069180E-01, 8.693183657357E-02,
               2.749289649692E-02, -2.874753111602E-02, -8.898817976496E-02],
              [-8.246762350893E-02, -4.608859710616E-02, -4.084775840814E-03, 6.888755194517E-02,
               1.192964322617E-01, 8.313590794595E-02, 1.717169858137E-02, -4.673523183195E-02,
               -9.787657035003E-02, -1.122576039996E-01, -9.787657165899E-02, -4.673523133513E-02,
               1.717169776952E-02, 8.313590700091E-02, 1.192964325729E-01, 6.888755090997E-02,
               -4.084774666669E-03, -4.608859881425E-02, -8.246762756993E-02],
              [-6.297844891506E-02, -6.217010882221E-02, -4.598683896496E-02, 2.574993052482E-02,
               9.400421099628E-02, 1.021244355060E-01, 8.313590734664E-02, 4.401715513015E-02,
               6.256940888970E-03, -3.352929972735E-03, 6.256940108248E-03, 4.401715560161E-02,
               8.313590693892E-02, 1.021244345839E-01, 9.400421161536E-02, 2.574993225773E-02,
               -4.598684170342E-02, -6.217011038194E-02, -6.297844890302E-02],
              [-3.406682305233E-02, -6.742425744758E-02, -7.906219707128E-02, -2.184399220936E-02,
               4.969839042112E-02, 9.400421125146E-02, 1.192964340561E-01, 1.119722046397E-01,
               9.594822049619E-02, 9.302969377011E-02, 9.594821924146E-02, 1.119721994615E-01,
               1.192964325178E-01, 9.400421009439E-02, 4.969839015398E-02, -2.184399270622E-02,
               -7.906219101329E-02, -6.742425623082E-02, -3.406682641371E-02],
              [9.452397692782E-03, -4.590027525083E-02, -8.224151830362E-02, -6.344434647317E-02,
               -2.184399126267E-02, 2.574993093018E-02, 6.888755366349E-02, 8.693183282044E-02,
               9.363466879364E-02, 9.672805827998E-02, 9.363466972978E-02, 8.693182947600E-02,
               6.888755537285E-02, 2.574992923294E-02, -2.184399157610E-02, -6.344434573849E-02,
               -8.224151453295E-02, -4.590027345418E-02, 9.452397880646E-03],
              [4.830380281679E-02, -1.282052443021E-02, -6.223172075225E-02, -8.224151757299E-02,
               -7.906219677648E-02, -4.598683861064E-02, -4.084775637564E-03, 2.749289600327E-02,
               4.969838973821E-02, 5.642128551036E-02, 4.969838923479E-02, 2.749289570942E-02,
               -4.084775574029E-03, -4.598684173405E-02, -7.906219400111E-02, -8.224151498124E-02,
               -6.223172922206E-02, -1.282052966935E-02, 4.830380063842E-02],
              [5.828996040645E-02, 2.170304594498E-02, -1.282052451763E-02, -4.590027478558E-02,
               -6.742425732877E-02, -6.217010872906E-02, -4.608859676091E-02, -2.874753145557E-02,
               -1.407342408121E-02, -1.007152737989E-02, -1.407342467503E-02, -2.874753205399E-02,
               -4.608859275946E-02, -6.217011081019E-02, -6.742424917439E-02, -4.590027036541E-02,
               -1.282052132177E-02, 2.170304786091E-02, 5.828996133177E-02],
              [5.936207369694E-02, 5.828996035238E-02, 4.830380268334E-02, 9.452398217996E-03,
               -3.406682282091E-02, -6.297844889354E-02, -8.246762288555E-02, -8.898817994782E-02,
               -8.945863745921E-02, -9.033251415162E-02, -8.945863816110E-02, -8.898818085115E-02,
               -8.246761457342E-02, -6.297844986249E-02, -3.406680946439E-02, 9.452404246664E-03,
               4.830381786080E-02, 5.828996939029E-02, 5.936207728870E-02]],

             [[4.356156947921E-02, 5.964570029223E-02, 6.482091611632E-02, 3.690531534624E-02,
               -1.191055458537E-03, -3.406682303796E-02, -6.137786872364E-02, -7.588664280441E-02,
               -8.287206522439E-02, -8.555883978733E-02, -8.287206522464E-02, -7.588664280220E-02,
               -6.137786872109E-02, -3.406682302198E-02, -1.191055381108E-03, 3.690531543776E-02,
               6.482091623871E-02, 5.964570020857E-02, 4.356156936429E-02],
              [5.964570030717E-02, 3.832433762356E-02, 1.385466983362E-02, -2.390080275470E-02,
               -5.712280002294E-02, -6.742425745655E-02, -6.664602760741E-02, -5.841121516077E-02,
               -4.898599926420E-02, -4.677849572149E-02, -4.898599926823E-02, -5.841121516267E-02,
               -6.664602760454E-02, -6.742425744763E-02, -5.712279995668E-02, -2.390080267880E-02,
               1.385466994423E-02, 3.832433755569E-02, 5.964570021275E-02],
              [6.482091618079E-02, 1.385466989069E-02, -3.233945739437E-02, -7.025699555783E-02,
               -9.132287823461E-02, -7.906219681437E-02, -5.367000806285E-02, -2.828238481259E-02,
               -7.601370330610E-03, -1.842790614014E-03, -7.601370335461E-03, -2.828238481870E-02,
               -5.367000806472E-02, -7.906219681195E-02, -9.132287826633E-02, -7.025699555598E-02,
               -3.233945728602E-02, 1.385466994845E-02, 6.482091625484E-02],
              [3.690531539853E-02, -2.390080270958E-02, -7.025699556461E-02, -7.803992807755E-02,
               -6.181990385465E-02, -2.184399201705E-02, 2.304143896013E-02, 5.220189526891E-02,
               7.075468373151E-02, 7.671883932792E-02, 7.075468373892E-02, 5.220189526673E-02,
               2.304143895261E-02, -2.184399201846E-02, -6.181990383499E-02, -7.803992805946E-02,
               -7.025699539485E-02, -2.390080265773E-02, 3.690531539384E-02],
              [-1.191055456349E-03, -5.712280001328E-02, -9.132287824780E-02, -6.181990386750E-02,
               -7.601370345531E-03, 4.969838977818E-02, 9.999127999359E-02, 1.205490947979E-01,
               1.276578283204E-01, 1.311161468630E-01, 1.276578284191E-01, 1.205490948279E-01,
               9.999127993051E-02, 4.969838983905E-02, -7.601370354315E-03, -6.181990386766E-02,
               -9.132287817522E-02, -5.712279995231E-02, -1.191055382820E-03],
              [-3.406682306918E-02, -6.742425747559E-02, -7.906219682729E-02, -2.184399202944E-02,
               4.969838976712E-02, 9.400421082747E-02, 1.192964321017E-01, 1.119722038734E-01,
               9.594822036310E-02, 9.302969389727E-02, 9.594822038186E-02, 1.119722039853E-01,
               1.192964320084E-01, 9.400421109082E-02, 4.969838978217E-02, -2.184399207472E-02,
               -7.906219684103E-02, -6.742425729711E-02, -3.406682271866E-02],
              [-6.137786877170E-02, -6.664602764906E-02, -5.367000805306E-02, 2.304143896605E-02,
               9.999128000940E-02, 1.192964321450E-01, 1.096062925629E-01, 7.105670825990E-02,
               3.130524743432E-02, 2.151761678899E-02, 3.130524735245E-02, 7.105670831274E-02,
               1.096062928446E-01, 1.192964321976E-01, 9.999127984208E-02, 2.304143890961E-02,
               -5.367000872054E-02, -6.664602781422E-02, -6.137786853047E-02],
              [-7.588664284592E-02, -5.841121519312E-02, -2.828238480036E-02, 5.220189527807E-02,
               1.205490948538E-01, 1.119722039399E-01, 7.105670826728E-02, 1.302475517606E-02,
               -3.882064947165E-02, -5.255531920395E-02, -3.882064966690E-02, 1.302475538685E-02,
               7.105670810873E-02, 1.119722032361E-01, 1.205490948414E-01, 5.220189527155E-02,
               -2.828238519357E-02, -5.841121474072E-02, -7.588664167337E-02],
              [-8.287206525245E-02, -4.898599928876E-02, -7.601370326263E-03, 7.075468373510E-02,
               1.276578282949E-01, 9.594822034112E-02, 3.130524741003E-02, -3.882064950361E-02,
               -9.690247379860E-02, -1.129284389518E-01, -9.690247385298E-02, -3.882064949771E-02,
               3.130524740625E-02, 9.594822031022E-02, 1.276578283302E-01, 7.075468375152E-02,
               -7.601370484105E-03, -4.898599935145E-02, -8.287206525998E-02],
              [-8.555883982043E-02, -4.677849575068E-02, -1.842790608403E-03, 7.671883933241E-02,
               1.311161468321E-01, 9.302969387336E-02, 2.151761676377E-02, -5.255531926981E-02,
               -1.129284389391E-01, -1.297341959063E-01, -1.129284389827E-01, -5.255531927927E-02,
               2.151761680393E-02, 9.302969391757E-02, 1.311161467272E-01, 7.671883915322E-02,
               -1.842791020951E-03, -4.677849599413E-02, -8.555883993984E-02],
              [-8.287206435891E-02, -4.898599855295E-02, -7.601370356260E-03, 7.075468371412E-02,
               1.276578273698E-01, 9.594821971415E-02, 3.130524750816E-02, -3.882064974598E-02,
               -9.690247365072E-02, -1.129284388487E-01, -9.690247307222E-02, -3.882064938148E-02,
               3.130524579898E-02, 9.594822003539E-02, 1.276578283030E-01, 7.075468405447E-02,
               -7.601380367056E-03, -4.898600549434E-02, -8.287206781885E-02],
              [-7.588664184552E-02, -5.841121437618E-02, -2.828238483285E-02, 5.220189526762E-02,
               1.205490939144E-01, 1.119722033177E-01, 7.105670828726E-02, 1.302475519339E-02,
               -3.882064948132E-02, -5.255531927003E-02, -3.882064640837E-02, 1.302475589500E-02,
               7.105670899228E-02, 1.119722052865E-01, 1.205490957686E-01, 5.220189601003E-02,
               -2.828239177699E-02, -5.841121990115E-02, -7.588664537391E-02],
              [-6.137786899400E-02, -6.664602782491E-02, -5.367000803271E-02, 2.304143902131E-02,
               9.999127936070E-02, 1.192964315037E-01, 1.096062924083E-01, 7.105670819910E-02,
               3.130524722506E-02, 2.151761661019E-02, 3.130524764457E-02, 7.105670786662E-02,
               1.096062931411E-01, 1.192964341401E-01, 9.999127827763E-02, 2.304143809825E-02,
               -5.367000448569E-02, -6.664602589776E-02, -6.137786859079E-02],
              [-3.406682335224E-02, -6.742425771105E-02, -7.906219675108E-02, -2.184399192170E-02,
               4.969838883936E-02, 9.400421029973E-02, 1.192964325300E-01, 1.119722041529E-01,
               9.594822035299E-02, 9.302969379053E-02, 9.594821669911E-02, 1.119721996138E-01,
               1.192964326511E-01, 9.400421491528E-02, 4.969838947112E-02, -2.184399386952E-02,
               -7.906220075823E-02, -6.742425856394E-02, -3.406682151311E-02],
              [-1.191056543525E-03, -5.712280094270E-02, -9.132287827851E-02, -6.181990387842E-02,
               -7.601370457639E-03, 4.969838951393E-02, 9.999128040271E-02, 1.205490949598E-01,
               1.276578281573E-01, 1.311161468263E-01, 1.276578279054E-01, 1.205490941243E-01,
               9.999128164240E-02, 4.969839084437E-02, -7.601365584847E-03, -6.181990361165E-02,
               -9.132286794780E-02, -5.712279453405E-02, -1.191054171399E-03],
              [3.690531409966E-02, -2.390080381490E-02, -7.025699563693E-02, -7.803992811220E-02,
               -6.181990401398E-02, -2.184399223206E-02, 2.304143920494E-02, 5.220189556893E-02,
               7.075468369309E-02, 7.671883933544E-02, 7.075468328435E-02, 5.220189490737E-02,
               2.304143799069E-02, -2.184399293202E-02, -6.181990136876E-02, -7.803992895402E-02,
               -7.025699788027E-02, -2.390080884058E-02, 3.690530649954E-02],
              [6.482091690874E-02, 1.385467052420E-02, -3.233945696121E-02, -7.025699512381E-02,
               -9.132287766799E-02, -7.906219622526E-02, -5.367000661955E-02, -2.828238395753E-02,
               -7.601370177970E-03, -1.842790484806E-03, -7.601374603131E-03, -2.828239172700E-02,
               -5.367001018729E-02, -7.906219557104E-02, -9.132287883021E-02, -7.025699519112E-02,
               -3.233945921801E-02, 1.385466831918E-02, 6.482091531714E-02],
              [5.964570095618E-02, 3.832433821040E-02, 1.385467023110E-02, -2.390080235914E-02,
               -5.712279989341E-02, -6.742425708316E-02, -6.664602619231E-02, -5.841121410229E-02,
               -4.898599891052E-02, -4.677849545042E-02, -4.898600489669E-02, -5.841122186928E-02,
               -6.664602915785E-02, -6.742425954628E-02, -5.712279540321E-02, -2.390080038694E-02,
               1.385465470834E-02, 3.832432749586E-02, 5.964569570642E-02],
              [4.356157019507E-02, 5.964570094620E-02, 6.482091649327E-02, 3.690531571911E-02,
               -1.191055739973E-03, -3.406682287209E-02, -6.137786740330E-02, -7.588664153191E-02,
               -8.287206467804E-02, -8.555883937802E-02, -8.287207217151E-02, -7.588664926515E-02,
               -6.137786986632E-02, -3.406682848978E-02, -1.191045622825E-03, 3.690531996259E-02,
               6.482088800778E-02, 5.964568187328E-02, 4.356156131450E-02]],

             [[1.607440723152E-02, 4.759923193698E-02, 6.859721738735E-02, 5.944385069905E-02,
               3.690531535485E-02, 9.452397711389E-03, -1.765322333963E-02, -3.551989669471E-02,
               -4.656675271059E-02, -5.018094388230E-02, -4.656675270993E-02, -3.551989669211E-02,
               -1.765322333643E-02, 9.452397727634E-03, 3.690531543376E-02, 5.944385079660E-02,
               6.859721752374E-02, 4.759923185274E-02, 1.607440711543E-02],
              [4.759923195514E-02, 4.831970825771E-02, 4.160199040895E-02, 1.073979768675E-02,
               -2.390080274839E-02, -4.590027525825E-02, -6.012530565255E-02, -6.404296523461E-02,
               -6.343256195249E-02, -6.384596899002E-02, -6.343256195111E-02, -6.404296523424E-02,
               -6.012530564969E-02, -4.590027524675E-02, -2.390080268239E-02, 1.073979776522E-02,
               4.160199053362E-02, 4.831970819001E-02, 4.759923185885E-02],
              [6.859721745932E-02, 4.160199047338E-02, 1.151943290776E-02, -3.223101201810E-02,
               -7.025699551580E-02, -8.224151800229E-02, -8.153908880681E-02, -7.238462716009E-02,
               -6.181990385009E-02, -5.935500573037E-02, -6.181990384626E-02, -7.238462715490E-02,
               -8.153908880641E-02, -8.224151799793E-02, -7.025699551997E-02, -3.223101202602E-02,
               1.151943299549E-02, 4.160199051486E-02, 6.859721752595E-02],
              [5.944385075744E-02, 1.073979773782E-02, -3.223101202547E-02, -6.336210617104E-02,
               -7.803992804249E-02, -6.344434624230E-02, -3.808861274919E-02, -1.545550682388E-02,
               2.183684692869E-03, 7.217721873521E-03, 2.183684701409E-03, -1.545550682473E-02,
               -3.808861274611E-02, -6.344434623452E-02, -7.803992797442E-02, -6.336210610463E-02,
               -3.223101191128E-02, 1.073979778280E-02, 5.944385080089E-02],
              [3.690531536127E-02, -2.390080273444E-02, -7.025699553170E-02, -7.803992805780E-02,
               -6.181990387164E-02, -2.184399202690E-02, 2.304143897004E-02, 5.220189528273E-02,
               7.075468373730E-02, 7.671883932705E-02, 7.075468386625E-02, 5.220189541201E-02,
               2.304143867930E-02, -2.184399193812E-02, -6.181990382900E-02, -7.803992807062E-02,
               -7.025699496131E-02, -2.390080239519E-02, 3.690531548784E-02],
              [9.452397681259E-03, -4.590027527565E-02, -8.224151801843E-02, -6.344434625754E-02,
               -2.184399206314E-02, 2.574993053482E-02, 6.888755197780E-02, 8.693183181423E-02,
               9.363466870079E-02, 9.672805828734E-02, 9.363466871225E-02, 8.693183190628E-02,
               6.888755140104E-02, 2.574993059188E-02, -2.184399207637E-02, -6.344434633845E-02,
               -8.224151782566E-02, -4.590027479190E-02, 9.452398433384E-03],
              [-1.765322339372E-02, -6.012530570064E-02, -8.153908879566E-02, -3.808861274540E-02,
               2.304143897899E-02, 6.888755201175E-02, 1.005921787995E-01, 1.011634188804E-01,
               9.204081654717E-02, 9.097189626867E-02, 9.204081652903E-02, 1.011634189583E-01,
               1.005921792885E-01, 6.888755183467E-02, 2.304143899273E-02, -3.808861291051E-02,
               -8.153908842839E-02, -6.012530450338E-02, -1.765322141608E-02],
              [-3.551989674226E-02, -6.404296527333E-02, -7.238462714697E-02, -1.545550681503E-02,
               5.220189532786E-02, 8.693183186811E-02, 1.011634188925E-01, 8.452337724864E-02,
               6.176314524614E-02, 5.683811105744E-02, 6.176314515901E-02, 8.452337749325E-02,
               1.011634196414E-01, 8.693183147879E-02, 5.220189531024E-02, -1.545550698957E-02,
               -7.238462699246E-02, -6.404296386942E-02, -3.551989417886E-02],
              [-4.656675274409E-02, -6.343256198156E-02, -6.181990384485E-02, 2.183684697211E-03,
               7.075468369637E-02, 9.363466866899E-02, 9.204081653150E-02, 6.176314523692E-02,
               2.882470737637E-02, 2.092269373141E-02, 2.882470738966E-02, 6.176314522111E-02,
               9.204081673725E-02, 9.363466868637E-02, 7.075468362358E-02, 2.183684643695E-03,
               -6.181990398677E-02, -6.343256210615E-02, -4.656675289344E-02],
              [-5.018094392162E-02, -6.384596902441E-02, -5.935500572396E-02, 7.217721878833E-03,
               7.671883929469E-02, 9.672805825739E-02, 9.097189626927E-02, 5.683811101359E-02,
               2.092269373231E-02, 1.216654341711E-02, 2.092269369416E-02, 5.683811099235E-02,
               9.097189633588E-02, 9.672805828901E-02, 7.671883909136E-02, 7.217721666961E-03,
               -5.935500614249E-02, -6.384596926075E-02, -5.018094403008E-02],
              [-4.656675175498E-02, -6.343256116876E-02, -6.181990387354E-02, 2.183684669236E-03,
               7.075468270610E-02, 9.363466808036E-02, 9.204081658501E-02, 6.176314499722E-02,
               2.882470748815E-02, 2.092269373372E-02, 2.882470741373E-02, 6.176314529433E-02,
               9.204081352995E-02, 9.363466954620E-02, 7.075468471475E-02, 2.183685806628E-03,
               -6.181990808605E-02, -6.343256577693E-02, -4.656675623028E-02],
              [-3.551989563692E-02, -6.404296437053E-02, -7.238462717901E-02, -1.545550682412E-02,
               5.220189442293E-02, 8.693183133744E-02, 1.011634187577E-01, 8.452337735671E-02,
               6.176314522297E-02, 5.683811102017E-02, 6.176314741246E-02, 8.452337750018E-02,
               1.011634213023E-01, 8.693183663726E-02, 5.220189672625E-02, -1.545550557600E-02,
               -7.238462539571E-02, -6.404296615554E-02, -3.551990002745E-02],
              [-1.765322363308E-02, -6.012530588757E-02, -8.153908877140E-02, -3.808861268257E-02,
               2.304143832914E-02, 6.888755121642E-02, 1.005921780763E-01, 1.011634190169E-01,
               9.204081627466E-02, 9.097189618305E-02, 9.204081705693E-02, 1.011634205962E-01,
               1.005921819770E-01, 6.888755510260E-02, 2.304143586398E-02, -3.808861420381E-02,
               -8.153908710576E-02, -6.012530599840E-02, -1.765322550140E-02],
              [9.452397370380E-03, -4.590027553544E-02, -8.224151792749E-02, -6.344434613079E-02,
               -2.184399278930E-02, 2.574992994187E-02, 6.888755189739E-02, 8.693183245289E-02,
               9.363466859331E-02, 9.672805819690E-02, 9.363466815764E-02, 8.693183183835E-02,
               6.888755582859E-02, 2.574993719813E-02, -2.184399359714E-02, -6.344434785499E-02,
               -8.224152161504E-02, -4.590027607223E-02, 9.452399615189E-03],
              [3.690531414060E-02, -2.390080376957E-02, -7.025699566807E-02, -7.803992815678E-02,
               -6.181990403790E-02, -2.184399226104E-02, 2.304143899477E-02, 5.220189543938E-02,
               7.075468355330E-02, 7.671883922336E-02, 7.075468362015E-02, 5.220189505327E-02,
               2.304143698674E-02, -2.184399209226E-02, -6.181990626101E-02, -7.803992972384E-02,
               -7.025698984332E-02, -2.390080098029E-02, 3.690531355910E-02],
              [5.944384930771E-02, 1.073979650878E-02, -3.223101216478E-02, -6.336210624702E-02,
               -7.803992819232E-02, -6.344434644372E-02, -3.808861258150E-02, -1.545550651007E-02,
               2.183684605920E-03, 7.217721787429E-03, 2.183685054137E-03, -1.545550688777E-02,
               -3.808861568241E-02, -6.344434880848E-02, -7.803993249843E-02, -6.336211333032E-02,
               -3.223101593500E-02, 1.073979099769E-02, 5.944384180685E-02],
              [6.859721817973E-02, 4.160199109933E-02, 1.151943339541E-02, -3.223101159294E-02,
               -7.025699504245E-02, -8.224151769342E-02, -8.153908824120E-02, -7.238462584510E-02,
               -6.181990362263E-02, -5.935500552930E-02, -6.181990658475E-02, -7.238462803497E-02,
               -8.153908855902E-02, -8.224151242717E-02, -7.025698561705E-02, -3.223100335117E-02,
               1.151944157422E-02, 4.160199429218E-02, 6.859721667592E-02],
              [4.759923258023E-02, 4.831970882380E-02, 4.160199086106E-02, 1.073979812725E-02,
               -2.390080269964E-02, -4.590027507645E-02, -6.012530494812E-02, -6.404296392485E-02,
               -6.343256172339E-02, -6.384596884096E-02, -6.343256547520E-02, -6.404296619560E-02,
               -6.012530208400E-02, -4.590027100889E-02, -2.390078981796E-02, 1.073980570771E-02,
               4.160199024988E-02, 4.831971183304E-02, 4.759923974836E-02],
              [1.607440792085E-02, 4.759923256550E-02, 6.859721781924E-02, 5.944385116682E-02,
               3.690531501172E-02, 9.452397764425E-03, -1.765322247487E-02, -3.551989528665E-02,
               -4.656675249827E-02, -5.018094379980E-02, -4.656675740298E-02, -3.551989767013E-02,
               -1.765321680648E-02, 9.452400368789E-03, 3.690533148479E-02, 5.944385889442E-02,
               6.859720926520E-02, 4.759923581686E-02, 1.607442380590E-02]],

             [[-1.378569451483E-02, 2.657147766349E-02, 5.864644230385E-02, 6.859721738448E-02,
               6.482091617367E-02, 4.830380226235E-02, 2.743644194615E-02, 1.078616361473E-02,
               -1.191055430439E-03, -4.769492603419E-03, -1.191055433654E-03, 1.078616361870E-02,
               2.743644199118E-02, 4.830380231369E-02, 6.482091612665E-02, 6.859721738214E-02,
               5.864644216931E-02, 2.657147784795E-02, -1.378569430326E-02],
              [2.657147770821E-02, 4.763924851776E-02, 5.900823043427E-02, 4.160199040118E-02,
               1.385466987954E-02, -1.282052491992E-02, -3.643722190988E-02, -4.992325419515E-02,
               -5.712280000054E-02, -5.970849726271E-02, -5.712280000417E-02, -4.992325419315E-02,
               -3.643722187786E-02, -1.282052488369E-02, 1.385466984461E-02, 4.160199041139E-02,
               5.900823028062E-02, 4.763924861570E-02, 2.657147782551E-02],
              [5.864644248034E-02, 5.900823059400E-02, 5.012671343589E-02, 1.151943298785E-02,
               -3.233945733834E-02, -6.223172113478E-02, -8.287206523020E-02, -9.025575972767E-02,
               -9.132287823568E-02, -9.237922986213E-02, -9.132287822901E-02, -9.025575971919E-02,
               -8.287206522383E-02, -6.223172111535E-02, -3.233945731238E-02, 1.151943300985E-02,
               5.012671330076E-02, 5.900823042615E-02, 5.864644234818E-02],
              [6.859721752004E-02, 4.160199052487E-02, 1.151943299205E-02, -3.223101195383E-02,
               -7.025699551017E-02, -8.224151799484E-02, -8.153908881902E-02, -7.238462717128E-02,
               -6.181990385925E-02, -5.935500573593E-02, -6.181990383523E-02, -7.238462712118E-02,
               -8.153908870891E-02, -8.224151786985E-02, -7.025699551897E-02, -3.223101194919E-02,
               1.151943295495E-02, 4.160199039812E-02, 6.859721735203E-02],
              [6.482091582718E-02, 1.385466956709E-02, -3.233945728818E-02, -7.025699548324E-02,
               -9.132287819644E-02, -7.906219678790E-02, -5.367000805132E-02, -2.828238478736E-02,
               -7.601370334799E-03, -1.842790617842E-03, -7.601370325080E-03, -2.828238479777E-02,
               -5.367000814455E-02, -7.906219678499E-02, -9.132287772992E-02, -7.025699507363E-02,
               -3.233945777427E-02, 1.385466972748E-02, 6.482091629568E-02],
              [4.830380195208E-02, -1.282052520138E-02, -6.223172111178E-02, -8.224151798333E-02,
               -7.906219677820E-02, -4.598683887427E-02, -4.084775765385E-03, 2.749289584202E-02,
               4.969838979386E-02, 5.642128549483E-02, 4.969838977760E-02, 2.749289576396E-02,
               -4.084775999088E-03, -4.598683915909E-02, -7.906219642280E-02, -8.224151782147E-02,
               -6.223172104910E-02, -1.282052484334E-02, 4.830380240319E-02],
              [2.743644192097E-02, -3.643722192924E-02, -8.287206518936E-02, -8.153908878923E-02,
               -5.367000808058E-02, -4.084775795293E-03, 4.831031943077E-02, 8.052457011237E-02,
               9.999127998384E-02, 1.064514000473E-01, 9.999127974830E-02, 8.052456985275E-02,
               4.831031928289E-02, -4.084775830465E-03, -5.367000866733E-02, -8.153908944922E-02,
               -8.287206493069E-02, -3.643722180380E-02, 2.743644187475E-02],
              [1.078616359265E-02, -4.992325421171E-02, -9.025575967140E-02, -7.238462713309E-02,
               -2.828238483947E-02, 2.749289580285E-02, 8.052457009677E-02, 1.073372671358E-01,
               1.205490948225E-01, 1.255521603829E-01, 1.205490945199E-01, 1.073372667745E-01,
               8.052456999567E-02, 2.749289563023E-02, -2.828238533905E-02, -7.238462792241E-02,
               -9.025575947552E-02, -4.992325410173E-02, 1.078616355510E-02],
              [-1.191055464323E-03, -5.712280002721E-02, -9.132287827508E-02, -6.181990388833E-02,
               -7.601370326063E-03, 4.969838979626E-02, 9.999127995571E-02, 1.205490947411E-01,
               1.276578283175E-01, 1.311161468597E-01, 1.276578283077E-01, 1.205490947596E-01,
               9.999128005987E-02, 4.969838979043E-02, -7.601370801028E-03, -6.181990430948E-02,
               -9.132287783838E-02, -5.712279985219E-02, -1.191055516624E-03],
              [-4.769492637128E-03, -5.970849728924E-02, -9.237922990655E-02, -5.935500577282E-02,
               -1.842790610115E-03, 5.642128549532E-02, 1.064514000434E-01, 1.255521603240E-01,
               1.311161468605E-01, 1.341992402116E-01, 1.311161468847E-01, 1.255521603277E-01,
               1.064514001560E-01, 5.642128549698E-02, -1.842790974948E-03, -5.935500634935E-02,
               -9.237922925780E-02, -5.970849693702E-02, -4.769492538829E-03],
              [-1.191055463615E-03, -5.712280001277E-02, -9.132287871003E-02, -6.181990423862E-02,
               -7.601370251624E-03, 4.969838993068E-02, 9.999128080230E-02, 1.205490947114E-01,
               1.276578283953E-01, 1.311161468767E-01, 1.276578275066E-01, 1.205490929487E-01,
               9.999127982633E-02, 4.969839003390E-02, -7.601373434027E-03, -6.181991128137E-02,
               -9.132287539136E-02, -5.712279881439E-02, -1.191055784788E-03],
              [1.078616361818E-02, -4.992325417397E-02, -9.025576028091E-02, -7.238462765949E-02,
               -2.828238466991E-02, 2.749289595527E-02, 8.052457072246E-02, 1.073372671880E-01,
               1.205490948018E-01, 1.255521604236E-01, 1.205490943450E-01, 1.073372669321E-01,
               8.052457050353E-02, 2.749289619115E-02, -2.828238427464E-02, -7.238462943073E-02,
               -9.025576094807E-02, -4.992325430451E-02, 1.078616464618E-02],
              [2.743644257137E-02, -3.643722123537E-02, -8.287206578685E-02, -8.153908923530E-02,
               -5.367000809498E-02, -4.084775628534E-03, 4.831031926655E-02, 8.052457018480E-02,
               9.999127997029E-02, 1.064514000084E-01, 9.999128003827E-02, 8.052457029126E-02,
               4.831031922328E-02, -4.084774043932E-03, -5.367000497374E-02, -8.153908560748E-02,
               -8.287206658803E-02, -3.643721754942E-02, 2.743645152688E-02],
              [4.830380280199E-02, -1.282052430553E-02, -6.223172176986E-02, -8.224151843406E-02,
               -7.906219667391E-02, -4.598683858316E-02, -4.084775890201E-03, 2.749289579749E-02,
               4.969838974155E-02, 5.642128539858E-02, 4.969838949061E-02, 2.749289580749E-02,
               -4.084780169398E-03, -4.598684184579E-02, -7.906219474749E-02, -8.224151608098E-02,
               -6.223171784629E-02, -1.282052173311E-02, 4.830380511728E-02],
              [6.482091652257E-02, 1.385467023824E-02, -3.233945795696E-02, -7.025699607755E-02,
               -9.132287803687E-02, -7.906219651206E-02, -5.367000660243E-02, -2.828238396160E-02,
               -7.601370366611E-03, -1.842790678071E-03, -7.601367189661E-03, -2.828238361197E-02,
               -5.367000773927E-02, -7.906219333534E-02, -9.132288045402E-02, -7.025699784778E-02,
               -3.233945652684E-02, 1.385467200634E-02, 6.482091997431E-02],
              [6.859721783290E-02, 4.160199086141E-02, 1.151943224992E-02, -3.223101262871E-02,
               -7.025699511459E-02, -8.224151801712E-02, -8.153908659595E-02, -7.238462511048E-02,
               -6.181990379712E-02, -5.935500567973E-02, -6.181989856820E-02, -7.238462156847E-02,
               -8.153908877174E-02, -8.224151885510E-02, -7.025699278295E-02, -3.223101414747E-02,
               1.151943167236E-02, 4.160198908290E-02, 6.859721604774E-02],
              [5.864644993872E-02, 5.900823718855E-02, 5.012671450535E-02, 1.151943374536E-02,
               -3.233945619425E-02, -6.223171961726E-02, -8.287206523866E-02, -9.025576007710E-02,
               -9.132287777620E-02, -9.237922943664E-02, -9.132287867654E-02, -9.025576037656E-02,
               -8.287206136335E-02, -6.223170653373E-02, -3.233944894323E-02, 1.151943930737E-02,
               5.012668046195E-02, 5.900821610394E-02, 5.864644619656E-02],
              [2.657148642163E-02, 4.763925620099E-02, 5.900823172768E-02, 4.160199140549E-02,
               1.385467136861E-02, -1.282052331108E-02, -3.643722214098E-02, -4.992325441781E-02,
               -5.712279950298E-02, -5.970849690906E-02, -5.712279832932E-02, -4.992325399153E-02,
               -3.643721845312E-02, -1.282052618459E-02, 1.385467382573E-02, 4.160199217467E-02,
               5.900821869529E-02, 4.763924792916E-02, 2.657148776788E-02],
              [-1.378568488354E-02, 2.657148611988E-02, 5.864644372265E-02, 6.859721855497E-02,
               6.482091796229E-02, 4.830380395902E-02, 2.743644149814E-02, 1.078616349929E-02,
               -1.191054880607E-03, -4.769492313418E-03, -1.191051647973E-03, 1.078616461311E-02,
               2.743644595609E-02, 4.830378623158E-02, 6.482091632133E-02, 6.859721533088E-02,
               5.864645010707E-02, 2.657148982038E-02, -1.378567855117E-02]],

             [[-3.649895255508E-02, -3.088974450723E-03, 2.657147797103E-02, 4.759923168959E-02,
               5.964570013507E-02, 5.828995985801E-02, 5.086584767967E-02, 4.258073037849E-02,
               3.539309464325E-02, 3.345900051267E-02, 3.539309464044E-02, 4.258073038392E-02,
               5.086584773084E-02, 5.828995991560E-02, 5.964570008314E-02, 4.759923168730E-02,
               2.657147783000E-02, -3.088974244952E-03, -3.649895232384E-02],
              [-3.088974395545E-03, 2.602783294470E-02, 4.763924877055E-02, 4.831970803890E-02,
               3.832433748381E-02, 2.170304545321E-02, 3.645921391712E-03, -8.922263383985E-03,
               -1.715564614035E-02, -1.975584208288E-02, -1.715564614342E-02, -8.922263380801E-03,
               3.645921433287E-03, 2.170304550121E-02, 3.832433744207E-02, 4.831970804966E-02,
               4.763924858948E-02, 2.602783304336E-02, -3.088974270671E-03],
              [2.657147816436E-02, 4.763924894535E-02, 5.900823054044E-02, 4.160199048198E-02,
               1.385466988064E-02, -1.282052491679E-02, -3.643722190638E-02, -4.992325418673E-02,
               -5.712280000433E-02, -5.970849726564E-02, -5.712280001774E-02, -4.992325422144E-02,
               -3.643722193761E-02, -1.282052491741E-02, 1.385466993000E-02, 4.160199051556E-02,
               5.900823043494E-02, 4.763924880480E-02, 2.657147806440E-02],
              [4.759923183831E-02, 4.831970817413E-02, 4.160199048710E-02, 1.073979773912E-02,
               -2.390080272085E-02, -4.590027523600E-02, -6.012530566767E-02, -6.404296524040E-02,
               -6.343256195580E-02, -6.384596899077E-02, -6.343256200766E-02, -6.404296525363E-02,
               -6.012530554487E-02, -4.590027518620E-02, -2.390080266491E-02, 1.073979775261E-02,
               4.160199047348E-02, 4.831970806061E-02, 4.759923168366E-02],
              [5.964569976431E-02, 3.832433714901E-02, 1.385466993427E-02, -2.390080268889E-02,
               -5.712279996561E-02, -6.742425741217E-02, -6.664602758775E-02, -5.841121512810E-02,
               -4.898599926615E-02, -4.677849572237E-02, -4.898599919851E-02, -5.841121513092E-02,
               -6.664602780087E-02, -6.742425748963E-02, -5.712279973421E-02, -2.390080256505E-02,
               1.385466960120E-02, 3.832433716928E-02, 5.964569976405E-02],
              [5.828995952718E-02, 2.170304515295E-02, -1.282052489214E-02, -4.590027522168E-02,
               -6.742425740446E-02, -6.217010892783E-02, -4.608859720176E-02, -2.874753180761E-02,
               -1.407342396671E-02, -1.007152730251E-02, -1.407342390543E-02, -2.874753179045E-02,
               -4.608859759356E-02, -6.217010906318E-02, -6.742425698638E-02, -4.590027517224E-02,
               -1.282052348773E-02, 2.170304584028E-02, 5.828995924898E-02],
              [5.086584765225E-02, 3.645921370811E-03, -3.643722187711E-02, -6.012530564387E-02,
               -6.664602761182E-02, -4.608859722492E-02, -1.661020281886E-02, 7.096489241857E-03,
               2.454870318898E-02, 2.969356644642E-02, 2.454870325954E-02, 7.096489071283E-03,
               -1.661020272435E-02, -4.608859710534E-02, -6.664602687524E-02, -6.012530560034E-02,
               -3.643722192286E-02, 3.645921185420E-03, 5.086584726770E-02],
              [4.258073035448E-02, -8.922263401357E-03, -4.992325414717E-02, -6.404296521448E-02,
               -5.841121516965E-02, -2.874753183067E-02, 7.096488981109E-03, 3.166809091975E-02,
               4.805122989440E-02, 5.317057248796E-02, 4.805122962978E-02, 3.166809044103E-02,
               7.096489286027E-03, -2.874753171492E-02, -5.841121489464E-02, -6.404296551539E-02,
               -4.992325405494E-02, -8.922263470574E-03, 4.258073004510E-02],
              [3.539309460772E-02, -1.715564616816E-02, -5.712280004725E-02, -6.343256199051E-02,
               -4.898599925698E-02, -1.407342395710E-02, 2.454870319857E-02, 4.805122988114E-02,
               6.236191878244E-02, 6.708857073084E-02, 6.236191878656E-02, 4.805122994512E-02,
               2.454870326458E-02, -1.407342399200E-02, -4.898599932747E-02, -6.343256212019E-02,
               -5.712279966041E-02, -1.715564595008E-02, 3.539309469589E-02],
              [3.345900047752E-02, -1.975584211028E-02, -5.970849731546E-02, -6.384596903310E-02,
               -4.677849571204E-02, -1.007152729871E-02, 2.969356645247E-02, 5.317057245645E-02,
               6.708857073031E-02, 7.176294626444E-02, 6.708857073293E-02, 5.317057244145E-02,
               2.969356657298E-02, -1.007152727044E-02, -4.677849583226E-02, -6.384596934898E-02,
               -5.970849672114E-02, -1.975584162690E-02, 3.345900089289E-02],
              [3.539309461078E-02, -1.715564614988E-02, -5.712280050066E-02, -6.343256242956E-02,
               -4.898599925386E-02, -1.407342381205E-02, 2.454870379247E-02, 4.805122964201E-02,
               6.236191887093E-02, 6.708857078867E-02, 6.236191677909E-02, 4.805122769682E-02,
               2.454870305553E-02, -1.407342366311E-02, -4.898600131700E-02, -6.343256640910E-02,
               -5.712279790266E-02, -1.715564432239E-02, 3.539309625472E-02],
              [4.258073038746E-02, -8.922263354816E-03, -4.992325480636E-02, -6.404296577765E-02,
               -5.841121501669E-02, -2.874753164323E-02, 7.096490121773E-03, 3.166809092006E-02,
               4.805122993802E-02, 5.317057249561E-02, 4.805122763696E-02, 3.166808876009E-02,
               7.096489425102E-03, -2.874753128734E-02, -5.841121727766E-02, -6.404296876244E-02,
               -4.992325512220E-02, -8.922263402228E-03, 4.258073133931E-02],
              [5.086584837902E-02, 3.645922138954E-03, -3.643722247221E-02, -6.012530614697E-02,
               -6.664602776208E-02, -4.608859714904E-02, -1.661020270091E-02, 7.096489327417E-03,
               2.454870307463E-02, 2.969356640397E-02, 2.454870304012E-02, 7.096489263582E-03,
               -1.661019821282E-02, -4.608859369801E-02, -6.664603598889E-02, -6.012530486419E-02,
               -3.643719700276E-02, 3.645941173770E-03, 5.086586245058E-02],
              [5.828996046549E-02, 2.170304613561E-02, -1.282052554398E-02, -4.590027575412E-02,
               -6.742425743181E-02, -6.217010882328E-02, -4.608859712562E-02, -2.874753179268E-02,
               -1.407342405405E-02, -1.007152735868E-02, -1.407342430207E-02, -2.874753182595E-02,
               -4.608859730638E-02, -6.217011156306E-02, -6.742426391129E-02, -4.590027572239E-02,
               -1.282049527764E-02, 2.170306223320E-02, 5.828996375699E-02],
              [5.964570052086E-02, 3.832433788730E-02, 1.385466922569E-02, -2.390080329892E-02,
               -5.712279986747E-02, -6.742425704642E-02, -6.664602795165E-02, -5.841121572180E-02,
               -4.898599906668E-02, -4.677849594232E-02, -4.898599584403E-02, -5.841121462591E-02,
               -6.664602737303E-02, -6.742424782805E-02, -5.712279719451E-02, -2.390080466283E-02,
               1.385466879239E-02, 3.832434043350E-02, 5.964570701937E-02],
              [4.759923219181E-02, 4.831970856066E-02, 4.160198970783E-02, 1.073979701410E-02,
               -2.390080242979E-02, -4.590027488698E-02, -6.012530510634E-02, -6.404296556458E-02,
               -6.343256186822E-02, -6.384596916563E-02, -6.343255769487E-02, -6.404296115526E-02,
               -6.012530648328E-02, -4.590026824717E-02, -2.390080155400E-02, 1.073979207336E-02,
               4.160199123554E-02, 4.831971192301E-02, 4.759923880419E-02],
              [2.657148620659E-02, 4.763925603055E-02, 5.900823184654E-02, 4.160199131858E-02,
               1.385467171461E-02, -1.282052317658E-02, -3.643722174257E-02, -4.992325424775E-02,
               -5.712279919854E-02, -5.970849637360E-02, -5.712280048248E-02, -4.992325510597E-02,
               -3.643720777645E-02, -1.282050270192E-02, 1.385467029691E-02, 4.160199441508E-02,
               5.900819133843E-02, 4.763922482648E-02, 2.657147021513E-02],
              [-3.088964996876E-03, 2.602784122493E-02, 4.763925033046E-02, 4.831970907286E-02,
               3.832433944786E-02, 2.170304708261E-02, 3.645921317074E-03, -8.922263412913E-03,
               -1.715564529580E-02, -1.975584136934E-02, -1.715564654133E-02, -8.922265261859E-03,
               3.645936303386E-03, 2.170305655912E-02, 3.832433924187E-02, 4.831970933954E-02,
               4.763922769613E-02, 2.602781907527E-02, -3.088981954290E-03],
              [-3.649894216064E-02, -3.088965317752E-03, 2.657147967380E-02, 4.759923283782E-02,
               5.964570218437E-02, 5.828996136596E-02, 5.086584735050E-02, 4.258073032930E-02,
               3.539309553797E-02, 3.345900106085E-02, 3.539309434009E-02, 4.258072750425E-02,
               5.086586271948E-02, 5.828995971811E-02, 5.964570383071E-02, 4.759923096674E-02,
               2.657147376084E-02, -3.088979565224E-03, -3.649896121916E-02]],

             [[-5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02,
               4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02,
               7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02,
               6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02,
               -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02],
              [-3.649895251266E-02, -3.088974410048E-03, 2.657147797163E-02, 4.759923169080E-02,
               5.964570013458E-02, 5.828995985851E-02, 5.086584768003E-02, 4.258073037812E-02,
               3.539309464253E-02, 3.345900051164E-02, 3.539309463997E-02, 4.258073038250E-02,
               5.086584773156E-02, 5.828995991989E-02, 5.964570008588E-02, 4.759923170226E-02,
               2.657147776421E-02, -3.088974310160E-03, -3.649895238078E-02],
              [-1.378569398919E-02, 2.657147816198E-02, 5.864644242385E-02, 6.859721747601E-02,
               6.482091617639E-02, 4.830380226680E-02, 2.743644195890E-02, 1.078616363559E-02,
               -1.191055435243E-03, -4.769492606911E-03, -1.191055468247E-03, 1.078616355493E-02,
               2.743644188988E-02, 4.830380225116E-02, 6.482091624807E-02, 6.859721752143E-02,
               5.864644234333E-02, 2.657147805054E-02, -1.378569404834E-02],
              [1.607440709334E-02, 4.759923183797E-02, 6.859721748211E-02, 5.944385076258E-02,
               3.690531538830E-02, 9.452397738457E-03, -1.765322336075E-02, -3.551989669594E-02,
               -4.656675271153E-02, -5.018094388164E-02, -4.656675283834E-02, -3.551989677735E-02,
               -1.765322322692E-02, 9.452397721944E-03, 3.690531550602E-02, 5.944385078247E-02,
               6.859721748235E-02, 4.759923174798E-02, 1.607440698224E-02],
              [4.356156888837E-02, 5.964569977667E-02, 6.482091623392E-02, 3.690531542572E-02,
               -1.191055397594E-03, -3.406682298584E-02, -6.137786870104E-02, -7.588664276778E-02,
               -8.287206522729E-02, -8.555883978829E-02, -8.287206510212E-02, -7.588664276484E-02,
               -6.137786904101E-02, -3.406682313806E-02, -1.191055384940E-03, 3.690531530214E-02,
               6.482091599872E-02, 5.964569958808E-02, 4.356156833446E-02],
              [5.936207280298E-02, 5.828995953883E-02, 4.830380229327E-02, 9.452397755076E-03,
               -3.406682298058E-02, -6.297844903824E-02, -8.246762363405E-02, -8.898818049045E-02,
               -8.945863729134E-02, -9.033251398458E-02, -8.945863715254E-02, -8.898818038188E-02,
               -8.246762418776E-02, -6.297844901497E-02, -3.406682249055E-02, 9.452397704854E-03,
               4.830380502371E-02, 5.828996054729E-02, 5.936207179261E-02],
              [6.830161064049E-02, 5.086584765759E-02, 2.743644197606E-02, -1.765322334310E-02,
               -6.137786872017E-02, -8.246762365112E-02, -9.132287812490E-02, -8.866359502242E-02,
               -8.226118590843E-02, -8.112847436445E-02, -8.226118553265E-02, -8.866359522795E-02,
               -9.132287779306E-02, -8.246762337901E-02, -6.137786667202E-02, -1.765322258180E-02,
               2.743644163198E-02, 5.086584718241E-02, 6.830160995493E-02],
              [7.107762956024E-02, 4.258073035995E-02, 1.078616365765E-02, -3.551989668366E-02,
               -7.588664279742E-02, -8.898818049794E-02, -8.866359553532E-02, -7.909406587553E-02,
               -6.792741704632E-02, -6.533539474735E-02, -6.792741727342E-02, -7.909406656255E-02,
               -8.866359481795E-02, -8.898818009094E-02, -7.588664168058E-02, -3.551989637476E-02,
               1.078616363163E-02, 4.258073011491E-02, 7.107762900055E-02],
              [7.076114330212E-02, 3.539309461359E-02, -1.191055481943E-03, -4.656675275215E-02,
               -8.287206521760E-02, -8.945863727463E-02, -8.226118586178E-02, -6.792741699264E-02,
               -5.367000806480E-02, -5.009611083633E-02, -5.367000804510E-02, -6.792741687764E-02,
               -8.226118583101E-02, -8.945863734646E-02, -8.287206490661E-02, -4.656675260396E-02,
               -1.191055118795E-03, 3.539309488367E-02, 7.076114352225E-02],
              [7.106858488521E-02, 3.345900048339E-02, -4.769492662390E-03, -5.018094392990E-02,
               -8.555883977530E-02, -9.033251397775E-02, -8.112847434770E-02, -6.533539475118E-02,
               -5.009611083819E-02, -4.621706539973E-02, -5.009611085778E-02, -6.533539477906E-02,
               -8.112847421729E-02, -9.033251392657E-02, -8.555883966639E-02, -5.018094399239E-02,
               -4.769492102270E-03, 3.345900110156E-02, 7.106858560417E-02],
              [7.076114330875E-02, 3.539309463675E-02, -1.191055954998E-03, -4.656675327912E-02,
               -8.287206528838E-02, -8.945863711222E-02, -8.226118548381E-02, -6.792741744182E-02,
               -5.367000796722E-02, -5.009611073526E-02, -5.367001137943E-02, -6.792741956979E-02,
               -8.226118600574E-02, -8.945863692581E-02, -8.287206615867E-02, -4.656675441066E-02,
               -1.191053975904E-03, 3.539309714428E-02, 7.076114692163E-02],
              [7.107762960220E-02, 4.258073041669E-02, 1.078616294658E-02, -3.551989728345E-02,
               -7.588664266096E-02, -8.898818027377E-02, -8.866359385549E-02, -7.909406592660E-02,
               -6.792741693994E-02, -6.533539476838E-02, -6.792742114874E-02, -7.909407003583E-02,
               -8.866359504352E-02, -8.898817980427E-02, -7.588664757437E-02, -3.551990152437E-02,
               1.078616300539E-02, 4.258073048037E-02, 7.107763051569E-02],
              [6.830161144915E-02, 5.086584850536E-02, 2.743644138385E-02, -1.765322390342E-02,
               -6.137786902090E-02, -8.246762368091E-02, -9.132287774034E-02, -8.866359493279E-02,
               -8.226118612119E-02, -8.112847441159E-02, -8.226118626526E-02, -8.866359515258E-02,
               -9.132286869337E-02, -8.246761848958E-02, -6.137788856119E-02, -1.765322526848E-02,
               2.743649294840E-02, 5.086588286773E-02, 6.830163073671E-02],
              [5.936207383592E-02, 5.828996061409E-02, 4.830380164939E-02, 9.452397146329E-03,
               -3.406682314592E-02, -6.297844913681E-02, -8.246762336583E-02, -8.898818042355E-02,
               -8.945863741340E-02, -9.033251400255E-02, -8.945863766455E-02, -8.898818049803E-02,
               -8.246761944604E-02, -6.297845133475E-02, -3.406683795400E-02, 9.452394471189E-03,
               4.830385852311E-02, 5.828999030603E-02, 5.936207796587E-02],
              [4.356156970982E-02, 5.964570058515E-02, 6.482091548318E-02, 3.690531479519E-02,
               -1.191055307466E-03, -3.406682252300E-02, -6.137787085067E-02, -7.588664476469E-02,
               -8.287206480677E-02, -8.555884016739E-02, -8.287206150668E-02, -7.588664328156E-02,
               -6.137786856535E-02, -3.406680693889E-02, -1.191047607186E-03, 3.690531411597E-02,
               6.482091306002E-02, 5.964570358467E-02, 4.356157874692E-02],
              [1.607440749088E-02, 4.759923227738E-02, 6.859721666281E-02, 5.944384998040E-02,
               3.690531561203E-02, 9.452398473741E-03, -1.765322447697E-02, -3.551989933283E-02,
               -4.656675261139E-02, -5.018094429076E-02, -4.656674952709E-02, -3.551989417196E-02,
               -1.765322484755E-02, 9.452412693733E-03, 3.690531534973E-02, 5.944384216703E-02,
               6.859722061430E-02, 4.759924067321E-02, 1.607442199727E-02],
              [-1.378568533457E-02, 2.657148576291E-02, 5.864644397361E-02, 6.859721840604E-02,
               6.482091866912E-02, 4.830380415185E-02, 2.743644229984E-02, 1.078616387454E-02,
               -1.191054274557E-03, -4.769491260276E-03, -1.191055983384E-03, 1.078616236785E-02,
               2.743646591988E-02, 4.830383191454E-02, 6.482090908749E-02, 6.859721946819E-02,
               5.864639572511E-02, 2.657144249915E-02, -1.378571623614E-02],
              [-3.649894239621E-02, -3.088965503338E-03, 2.657147980753E-02, 4.759923277444E-02,
               5.964570256308E-02, 5.828996149711E-02, 5.086584776336E-02, 4.258073056235E-02,
               3.539309583851E-02, 3.345900157706E-02, 3.539309215461E-02, 4.258072638612E-02,
               5.086587387137E-02, 5.828998283735E-02, 5.964569990800E-02, 4.759923283785E-02,
               2.657144707246E-02, -3.089003456111E-03, -3.649898102316E-02],
              [-5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02,
               4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02,
               7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02,
               6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02,
               -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02]]], dtype=np.float64
        )

        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
            [-5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02,
             5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02,
             7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02,
             5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02,
             -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02,
             4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02,
             7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02,
             6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02,
             -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02,
             1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02,
             7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02,
             7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02,
             1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02,
             -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02,
             6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02,
             7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02,
             4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -3.649895211738E-02, -3.649895211738E-02, -3.649895211738E-02, -3.649895211738E-02,
             -3.088974011050E-03, 2.657147806335E-02, 4.759923176017E-02, 5.964570012010E-02,
             5.828995984419E-02, 5.086584768151E-02, 4.258073038267E-02, 3.539309463878E-02,
             3.345900050891E-02, 3.539309463946E-02, 4.258073038068E-02, 5.086584768574E-02,
             5.828995985165E-02, 5.964570013079E-02, 4.759923175448E-02, 2.657147805303E-02,
             -3.088974030447E-03, -3.649895213245E-02, -3.649895213245E-02, -3.649895213245E-02,
             -3.649895213245E-02, -1.378569410836E-02, -1.378569410836E-02, -1.378569410836E-02,
             -1.378569410836E-02, 2.657147805303E-02, 5.864644238984E-02, 6.859721745322E-02,
             6.482091617241E-02, 4.830380225823E-02, 2.743644194715E-02, 1.078616361697E-02,
             -1.191055434163E-03, -4.769492607025E-03, -1.191055434376E-03, 1.078616361760E-02,
             2.743644194653E-02, 4.830380225562E-02, 6.482091616058E-02, 6.859721744395E-02,
             5.864644239405E-02, 2.657147806556E-02, -1.378569409104E-02, -1.378569409104E-02,
             -1.378569409104E-02, -1.378569409104E-02, 1.607440700266E-02, 1.607440700266E-02,
             1.607440700266E-02, 1.607440700266E-02, 4.759923175276E-02, 6.859721744999E-02,
             5.944385073981E-02, 3.690531538030E-02, 9.452397728552E-03, -1.765322333815E-02,
             -3.551989669485E-02, -4.656675271099E-02, -5.018094388273E-02, -4.656675271113E-02,
             -3.551989669440E-02, -1.765322333876E-02, 9.452397726093E-03, 3.690531537103E-02,
             5.944385073178E-02, 6.859721745415E-02, 4.759923176233E-02, 1.607440701568E-02,
             1.607440701568E-02, 1.607440701568E-02, 1.607440701568E-02, 4.356156929915E-02,
             4.356156929915E-02, 4.356156929915E-02, 4.356156929915E-02, 5.964570014935E-02,
             6.482091617121E-02, 3.690531537805E-02, -1.191055430596E-03, -3.406682302067E-02,
             -6.137786872165E-02, -7.588664280371E-02, -8.287206522451E-02, -8.555883978764E-02,
             -8.287206522502E-02, -7.588664280388E-02, -6.137786872209E-02, -3.406682302217E-02,
             -1.191055436745E-03, 3.690531537487E-02, 6.482091617227E-02, 5.964570014017E-02,
             4.356156929263E-02, 4.356156929263E-02, 4.356156929263E-02, 4.356156929263E-02,
             5.936207315787E-02, 5.936207315787E-02, 5.936207315787E-02, 5.936207315787E-02,
             5.828995986152E-02, 4.830380225713E-02, 9.452397726372E-03, -3.406682301951E-02,
             -6.297844907249E-02, -8.246762366680E-02, -8.898818051517E-02, -8.945863728280E-02,
             -9.033251398485E-02, -8.945863728310E-02, -8.898818051543E-02, -8.246762366704E-02,
             -6.297844907350E-02, -3.406682302335E-02, 9.452397724649E-03, 4.830380225745E-02,
             5.828995985726E-02, 5.936207315493E-02, 5.936207315493E-02, 5.936207315493E-02,
             5.936207315493E-02, 6.830161067188E-02, 6.830161067188E-02, 6.830161067188E-02,
             6.830161067188E-02, 5.086584768016E-02, 2.743644194736E-02, -1.765322333716E-02,
             -6.137786872188E-02, -8.246762366694E-02, -9.132287823500E-02, -8.866359515673E-02,
             -8.226118588935E-02, -8.112847434613E-02, -8.226118588939E-02, -8.866359515687E-02,
             -9.132287823508E-02, -8.246762366680E-02, -6.137786872115E-02, -1.765322333746E-02,
             2.743644194769E-02, 5.086584768251E-02, 6.830161067455E-02, 6.830161067455E-02,
             6.830161067455E-02, 6.830161067455E-02, 7.107762959038E-02, 7.107762959038E-02,
             7.107762959038E-02, 7.107762959038E-02, 4.258073038138E-02, 1.078616361751E-02,
             -3.551989669405E-02, -7.588664280397E-02, -8.898818051528E-02, -8.866359515675E-02,
             -7.909406561030E-02, -6.792741701972E-02, -6.533539475257E-02, -6.792741701977E-02,
             -7.909406561041E-02, -8.866359515684E-02, -8.898818051519E-02, -7.588664280371E-02,
             -3.551989669442E-02, 1.078616361837E-02, 4.258073038353E-02, 7.107762959247E-02,
             7.107762959247E-02, 7.107762959247E-02, 7.107762959247E-02, 7.076114333566E-02,
             7.076114333566E-02, 7.076114333566E-02, 7.076114333566E-02, 3.539309463963E-02,
             -1.191055434244E-03, -4.656675271103E-02, -8.287206522482E-02, -8.945863728286E-02,
             -8.226118588935E-02, -6.792741701970E-02, -5.367000806700E-02, -5.009611083921E-02,
             -5.367000806699E-02, -6.792741701970E-02, -8.226118588946E-02, -8.945863728288E-02,
             -8.287206522469E-02, -4.656675271102E-02, -1.191055434066E-03, 3.539309463997E-02,
             7.076114333521E-02, 7.076114333521E-02, 7.076114333521E-02, 7.076114333521E-02,
             7.106858491835E-02, 7.106858491835E-02, 7.106858491835E-02, 7.106858491835E-02,
             3.345900050925E-02, -4.769492607011E-03, -5.018094388270E-02, -8.555883978772E-02,
             -9.033251398488E-02, -8.112847434614E-02, -6.533539475257E-02, -5.009611083922E-02,
             -4.621706540085E-02, -5.009611083921E-02, -6.533539475257E-02, -8.112847434625E-02,
             -9.033251398490E-02, -8.555883978761E-02, -5.018094388272E-02, -4.769492606812E-03,
             3.345900050959E-02, 7.106858491790E-02, 7.106858491790E-02, 7.106858491790E-02,
             7.106858491790E-02, 7.076114333396E-02, 7.076114333396E-02, 7.076114333396E-02,
             7.076114333396E-02, 3.539309463867E-02, -1.191055434516E-03, -4.656675271126E-02,
             -8.287206522498E-02, -8.945863728301E-02, -8.226118588933E-02, -6.792741701967E-02,
             -5.367000806702E-02, -5.009611083921E-02, -5.367000806697E-02, -6.792741701964E-02,
             -8.226118588937E-02, -8.945863728288E-02, -8.287206522485E-02, -4.656675271062E-02,
             -1.191055434693E-03, 3.539309463898E-02, 7.076114333498E-02, 7.076114333498E-02,
             7.076114333498E-02, 7.076114333498E-02, 7.107762959019E-02, 7.107762959019E-02,
             7.107762959019E-02, 7.107762959019E-02, 4.258073038204E-02, 1.078616361737E-02,
             -3.551989669443E-02, -7.588664280384E-02, -8.898818051525E-02, -8.866359515677E-02,
             -7.909406561028E-02, -6.792741701971E-02, -6.533539475258E-02, -6.792741701968E-02,
             -7.909406561026E-02, -8.866359515678E-02, -8.898818051514E-02, -7.588664280389E-02,
             -3.551989669383E-02, 1.078616361796E-02, 4.258073038240E-02, 7.107762959115E-02,
             7.107762959115E-02, 7.107762959115E-02, 7.107762959115E-02, 6.830161067588E-02,
             6.830161067588E-02, 6.830161067588E-02, 6.830161067588E-02, 5.086584768557E-02,
             2.743644194734E-02, -1.765322333791E-02, -6.137786872193E-02, -8.246762366704E-02,
             -9.132287823511E-02, -8.866359515673E-02, -8.226118588927E-02, -8.112847434610E-02,
             -8.226118588891E-02, -8.866359515666E-02, -9.132287823417E-02, -8.246762366571E-02,
             -6.137786872110E-02, -1.765322333624E-02, 2.743644195026E-02, 5.086584767606E-02,
             6.830161066853E-02, 6.830161066853E-02, 6.830161066853E-02, 6.830161066853E-02,
             5.936207315005E-02, 5.936207315005E-02, 5.936207315005E-02, 5.936207315005E-02,
             5.828995985678E-02, 4.830380225714E-02, 9.452397727007E-03, -3.406682302249E-02,
             -6.297844907372E-02, -8.246762366682E-02, -8.898818051514E-02, -8.945863728277E-02,
             -9.033251398490E-02, -8.945863728260E-02, -8.898818051505E-02, -8.246762366635E-02,
             -6.297844907261E-02, -3.406682302059E-02, 9.452397730918E-03, 4.830380225822E-02,
             5.828995984248E-02, 5.936207314396E-02, 5.936207314396E-02, 5.936207314396E-02,
             5.936207314396E-02, 4.356156927876E-02, 4.356156927876E-02, 4.356156927876E-02,
             4.356156927876E-02, 5.964570012580E-02, 6.482091617804E-02, 3.690531538273E-02,
             -1.191055434191E-03, -3.406682302106E-02, -6.137786872090E-02, -7.588664280368E-02,
             -8.287206522483E-02, -8.555883978772E-02, -8.287206522455E-02, -7.588664280379E-02,
             -6.137786872198E-02, -3.406682302096E-02, -1.191055432770E-03, 3.690531537512E-02,
             6.482091616700E-02, 5.964570013270E-02, 4.356156928137E-02, 4.356156928137E-02,
             4.356156928137E-02, 4.356156928137E-02, 1.607440701081E-02, 1.607440701081E-02,
             1.607440701081E-02, 1.607440701081E-02, 4.759923175430E-02, 6.859721745905E-02,
             5.944385074129E-02, 3.690531537985E-02, 9.452397728154E-03, -1.765322333673E-02,
             -3.551989669413E-02, -4.656675271111E-02, -5.018094388280E-02, -4.656675271075E-02,
             -3.551989669466E-02, -1.765322333821E-02, 9.452397728822E-03, 3.690531537920E-02,
             5.944385072904E-02, 6.859721744377E-02, 4.759923176384E-02, 1.607440701473E-02,
             1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02, -1.378569400687E-02,
             -1.378569400687E-02, -1.378569400687E-02, -1.378569400687E-02, 2.657147812158E-02,
             5.864644239897E-02, 6.859721744841E-02, 6.482091617160E-02, 4.830380225513E-02,
             2.743644194722E-02, 1.078616361659E-02, -1.191055434532E-03, -4.769492607409E-03,
             -1.191055434106E-03, 1.078616361715E-02, 2.743644195107E-02, 4.830380225332E-02,
             6.482091617384E-02, 6.859721744401E-02, 5.864644236959E-02, 2.657147806131E-02,
             -1.378569410514E-02, -1.378569410514E-02, -1.378569410514E-02, -1.378569410514E-02,
             -3.649895199161E-02, -3.649895199161E-02, -3.649895199161E-02, -3.649895199161E-02,
             -3.088973940848E-03, 2.657147807892E-02, 4.759923176151E-02, 5.964570013216E-02,
             5.828995984835E-02, 5.086584768191E-02, 4.258073038125E-02, 3.539309463849E-02,
             3.345900050823E-02, 3.539309463952E-02, 4.258073038187E-02, 5.086584768429E-02,
             5.828995984560E-02, 5.964570013445E-02, 4.759923175558E-02, 2.657147804251E-02,
             -3.088974026880E-03, -3.649895211890E-02, -3.649895211890E-02, -3.649895211890E-02,
             -3.649895211890E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02,
             4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02,
             7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02,
             6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02,
             -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02,
             1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02,
             7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02,
             7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02,
             1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02,
             -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02,
             6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02,
             7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02,
             4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02,
             5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02,
             7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02,
             5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02,
             -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02,
             4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02,
             7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02,
             6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02,
             -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02,
             1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02,
             7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02,
             7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02,
             1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02,
             -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02,
             6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02,
             7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02,
             4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02,
             5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02,
             7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02,
             5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02,
             -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -3.649895211738E-02, -3.649895211738E-02, -3.649895211738E-02,
             -3.649895211738E-02, -3.088974011050E-03, 2.657147806335E-02, 4.759923176017E-02,
             5.964570012010E-02, 5.828995984419E-02, 5.086584768151E-02, 4.258073038267E-02,
             3.539309463878E-02, 3.345900050891E-02, 3.539309463946E-02, 4.258073038068E-02,
             5.086584768574E-02, 5.828995985165E-02, 5.964570013079E-02, 4.759923175448E-02,
             2.657147805303E-02, -3.088974030447E-03, -3.649895213245E-02, -3.649895213245E-02,
             -3.649895213245E-02, -3.649895213245E-02, -1.378569410836E-02, -1.378569410836E-02,
             -1.378569410836E-02, -1.378569410836E-02, 2.657147805303E-02, 5.864644238984E-02,
             6.859721745322E-02, 6.482091617241E-02, 4.830380225823E-02, 2.743644194715E-02,
             1.078616361697E-02, -1.191055434163E-03, -4.769492607025E-03, -1.191055434376E-03,
             1.078616361760E-02, 2.743644194653E-02, 4.830380225562E-02, 6.482091616058E-02,
             6.859721744395E-02, 5.864644239405E-02, 2.657147806556E-02, -1.378569409104E-02,
             -1.378569409104E-02, -1.378569409104E-02, -1.378569409104E-02, 1.607440700266E-02,
             1.607440700266E-02, 1.607440700266E-02, 1.607440700266E-02, 4.759923175276E-02,
             6.859721744999E-02, 5.944385073981E-02, 3.690531538030E-02, 9.452397728552E-03,
             -1.765322333815E-02, -3.551989669485E-02, -4.656675271099E-02, -5.018094388273E-02,
             -4.656675271113E-02, -3.551989669440E-02, -1.765322333876E-02, 9.452397726093E-03,
             3.690531537103E-02, 5.944385073178E-02, 6.859721745415E-02, 4.759923176233E-02,
             1.607440701568E-02, 1.607440701568E-02, 1.607440701568E-02, 1.607440701568E-02,
             4.356156929915E-02, 4.356156929915E-02, 4.356156929915E-02, 4.356156929915E-02,
             5.964570014935E-02, 6.482091617121E-02, 3.690531537805E-02, -1.191055430596E-03,
             -3.406682302067E-02, -6.137786872165E-02, -7.588664280371E-02, -8.287206522451E-02,
             -8.555883978764E-02, -8.287206522502E-02, -7.588664280388E-02, -6.137786872209E-02,
             -3.406682302217E-02, -1.191055436745E-03, 3.690531537487E-02, 6.482091617227E-02,
             5.964570014017E-02, 4.356156929263E-02, 4.356156929263E-02, 4.356156929263E-02,
             4.356156929263E-02, 5.936207315787E-02, 5.936207315787E-02, 5.936207315787E-02,
             5.936207315787E-02, 5.828995986152E-02, 4.830380225713E-02, 9.452397726372E-03,
             -3.406682301951E-02, -6.297844907249E-02, -8.246762366680E-02, -8.898818051517E-02,
             -8.945863728280E-02, -9.033251398485E-02, -8.945863728310E-02, -8.898818051543E-02,
             -8.246762366704E-02, -6.297844907350E-02, -3.406682302335E-02, 9.452397724649E-03,
             4.830380225745E-02, 5.828995985726E-02, 5.936207315493E-02, 5.936207315493E-02,
             5.936207315493E-02, 5.936207315493E-02, 6.830161067188E-02, 6.830161067188E-02,
             6.830161067188E-02, 6.830161067188E-02, 5.086584768016E-02, 2.743644194736E-02,
             -1.765322333716E-02, -6.137786872188E-02, -8.246762366694E-02, -9.132287823500E-02,
             -8.866359515673E-02, -8.226118588935E-02, -8.112847434613E-02, -8.226118588939E-02,
             -8.866359515687E-02, -9.132287823508E-02, -8.246762366680E-02, -6.137786872115E-02,
             -1.765322333746E-02, 2.743644194769E-02, 5.086584768251E-02, 6.830161067455E-02,
             6.830161067455E-02, 6.830161067455E-02, 6.830161067455E-02, 7.107762959038E-02,
             7.107762959038E-02, 7.107762959038E-02, 7.107762959038E-02, 4.258073038138E-02,
             1.078616361751E-02, -3.551989669405E-02, -7.588664280397E-02, -8.898818051528E-02,
             -8.866359515675E-02, -7.909406561030E-02, -6.792741701972E-02, -6.533539475257E-02,
             -6.792741701977E-02, -7.909406561041E-02, -8.866359515684E-02, -8.898818051519E-02,
             -7.588664280371E-02, -3.551989669442E-02, 1.078616361837E-02, 4.258073038353E-02,
             7.107762959247E-02, 7.107762959247E-02, 7.107762959247E-02, 7.107762959247E-02,
             7.076114333566E-02, 7.076114333566E-02, 7.076114333566E-02, 7.076114333566E-02,
             3.539309463963E-02, -1.191055434244E-03, -4.656675271103E-02, -8.287206522482E-02,
             -8.945863728286E-02, -8.226118588935E-02, -6.792741701970E-02, -5.367000806700E-02,
             -5.009611083921E-02, -5.367000806699E-02, -6.792741701970E-02, -8.226118588946E-02,
             -8.945863728288E-02, -8.287206522469E-02, -4.656675271102E-02, -1.191055434066E-03,
             3.539309463997E-02, 7.076114333521E-02, 7.076114333521E-02, 7.076114333521E-02,
             7.076114333521E-02, 7.106858491835E-02, 7.106858491835E-02, 7.106858491835E-02,
             7.106858491835E-02, 3.345900050925E-02, -4.769492607011E-03, -5.018094388270E-02,
             -8.555883978772E-02, -9.033251398488E-02, -8.112847434614E-02, -6.533539475257E-02,
             -5.009611083922E-02, -4.621706540085E-02, -5.009611083921E-02, -6.533539475257E-02,
             -8.112847434625E-02, -9.033251398490E-02, -8.555883978761E-02, -5.018094388272E-02,
             -4.769492606812E-03, 3.345900050959E-02, 7.106858491790E-02, 7.106858491790E-02,
             7.106858491790E-02, 7.106858491790E-02, 7.076114333396E-02, 7.076114333396E-02,
             7.076114333396E-02, 7.076114333396E-02, 3.539309463867E-02, -1.191055434516E-03,
             -4.656675271126E-02, -8.287206522498E-02, -8.945863728301E-02, -8.226118588933E-02,
             -6.792741701967E-02, -5.367000806702E-02, -5.009611083921E-02, -5.367000806697E-02,
             -6.792741701964E-02, -8.226118588937E-02, -8.945863728288E-02, -8.287206522485E-02,
             -4.656675271062E-02, -1.191055434693E-03, 3.539309463898E-02, 7.076114333498E-02,
             7.076114333498E-02, 7.076114333498E-02, 7.076114333498E-02, 7.107762959019E-02,
             7.107762959019E-02, 7.107762959019E-02, 7.107762959019E-02, 4.258073038204E-02,
             1.078616361737E-02, -3.551989669443E-02, -7.588664280384E-02, -8.898818051525E-02,
             -8.866359515677E-02, -7.909406561028E-02, -6.792741701971E-02, -6.533539475258E-02,
             -6.792741701968E-02, -7.909406561026E-02, -8.866359515678E-02, -8.898818051514E-02,
             -7.588664280389E-02, -3.551989669383E-02, 1.078616361796E-02, 4.258073038240E-02,
             7.107762959115E-02, 7.107762959115E-02, 7.107762959115E-02, 7.107762959115E-02,
             6.830161067588E-02, 6.830161067588E-02, 6.830161067588E-02, 6.830161067588E-02,
             5.086584768557E-02, 2.743644194734E-02, -1.765322333791E-02, -6.137786872193E-02,
             -8.246762366704E-02, -9.132287823511E-02, -8.866359515673E-02, -8.226118588927E-02,
             -8.112847434610E-02, -8.226118588891E-02, -8.866359515666E-02, -9.132287823417E-02,
             -8.246762366571E-02, -6.137786872110E-02, -1.765322333624E-02, 2.743644195026E-02,
             5.086584767606E-02, 6.830161066853E-02, 6.830161066853E-02, 6.830161066853E-02,
             6.830161066853E-02, 5.936207315005E-02, 5.936207315005E-02, 5.936207315005E-02,
             5.936207315005E-02, 5.828995985678E-02, 4.830380225714E-02, 9.452397727007E-03,
             -3.406682302249E-02, -6.297844907372E-02, -8.246762366682E-02, -8.898818051514E-02,
             -8.945863728277E-02, -9.033251398490E-02, -8.945863728260E-02, -8.898818051505E-02,
             -8.246762366635E-02, -6.297844907261E-02, -3.406682302059E-02, 9.452397730918E-03,
             4.830380225822E-02, 5.828995984248E-02, 5.936207314396E-02, 5.936207314396E-02,
             5.936207314396E-02, 5.936207314396E-02, 4.356156927876E-02, 4.356156927876E-02,
             4.356156927876E-02, 4.356156927876E-02, 5.964570012580E-02, 6.482091617804E-02,
             3.690531538273E-02, -1.191055434191E-03, -3.406682302106E-02, -6.137786872090E-02,
             -7.588664280368E-02, -8.287206522483E-02, -8.555883978772E-02, -8.287206522455E-02,
             -7.588664280379E-02, -6.137786872198E-02, -3.406682302096E-02, -1.191055432770E-03,
             3.690531537512E-02, 6.482091616700E-02, 5.964570013270E-02, 4.356156928137E-02,
             4.356156928137E-02, 4.356156928137E-02, 4.356156928137E-02, 1.607440701081E-02,
             1.607440701081E-02, 1.607440701081E-02, 1.607440701081E-02, 4.759923175430E-02,
             6.859721745905E-02, 5.944385074129E-02, 3.690531537985E-02, 9.452397728154E-03,
             -1.765322333673E-02, -3.551989669413E-02, -4.656675271111E-02, -5.018094388280E-02,
             -4.656675271075E-02, -3.551989669466E-02, -1.765322333821E-02, 9.452397728822E-03,
             3.690531537920E-02, 5.944385072904E-02, 6.859721744377E-02, 4.759923176384E-02,
             1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02,
             -1.378569400687E-02, -1.378569400687E-02, -1.378569400687E-02, -1.378569400687E-02,
             2.657147812158E-02, 5.864644239897E-02, 6.859721744841E-02, 6.482091617160E-02,
             4.830380225513E-02, 2.743644194722E-02, 1.078616361659E-02, -1.191055434532E-03,
             -4.769492607409E-03, -1.191055434106E-03, 1.078616361715E-02, 2.743644195107E-02,
             4.830380225332E-02, 6.482091617384E-02, 6.859721744401E-02, 5.864644236959E-02,
             2.657147806131E-02, -1.378569410514E-02, -1.378569410514E-02, -1.378569410514E-02,
             -1.378569410514E-02, -3.649895199161E-02, -3.649895199161E-02, -3.649895199161E-02,
             -3.649895199161E-02, -3.088973940848E-03, 2.657147807892E-02, 4.759923176151E-02,
             5.964570013216E-02, 5.828995984835E-02, 5.086584768191E-02, 4.258073038125E-02,
             3.539309463849E-02, 3.345900050823E-02, 3.539309463952E-02, 4.258073038187E-02,
             5.086584768429E-02, 5.828995984560E-02, 5.964570013445E-02, 4.759923175558E-02,
             2.657147804251E-02, -3.088974026880E-03, -3.649895211890E-02, -3.649895211890E-02,
             -3.649895211890E-02, -3.649895211890E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02,
             1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02,
             7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02,
             7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02,
             1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02,
             -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02,
             6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02,
             7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02,
             4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02,
             5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02,
             7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02,
             5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02,
             -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02,
             4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02,
             7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02,
             6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02,
             -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02,
             1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02,
             7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02,
             7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02,
             1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02,
             -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02,
             6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02,
             7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02,
             4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02,
             5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02,
             7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02,
             5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02,
             -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02,
             4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02,
             7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02,
             6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02,
             -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -3.649895211738E-02, -3.649895211738E-02,
             -3.649895211738E-02, -3.649895211738E-02, -3.088974011050E-03, 2.657147806335E-02,
             4.759923176017E-02, 5.964570012010E-02, 5.828995984419E-02, 5.086584768151E-02,
             4.258073038267E-02, 3.539309463878E-02, 3.345900050891E-02, 3.539309463946E-02,
             4.258073038068E-02, 5.086584768574E-02, 5.828995985165E-02, 5.964570013079E-02,
             4.759923175448E-02, 2.657147805303E-02, -3.088974030447E-03, -3.649895213245E-02,
             -3.649895213245E-02, -3.649895213245E-02, -3.649895213245E-02, -1.378569410836E-02,
             -1.378569410836E-02, -1.378569410836E-02, -1.378569410836E-02, 2.657147805303E-02,
             5.864644238984E-02, 6.859721745322E-02, 6.482091617241E-02, 4.830380225823E-02,
             2.743644194715E-02, 1.078616361697E-02, -1.191055434163E-03, -4.769492607025E-03,
             -1.191055434376E-03, 1.078616361760E-02, 2.743644194653E-02, 4.830380225562E-02,
             6.482091616058E-02, 6.859721744395E-02, 5.864644239405E-02, 2.657147806556E-02,
             -1.378569409104E-02, -1.378569409104E-02, -1.378569409104E-02, -1.378569409104E-02,
             1.607440700266E-02, 1.607440700266E-02, 1.607440700266E-02, 1.607440700266E-02,
             4.759923175276E-02, 6.859721744999E-02, 5.944385073981E-02, 3.690531538030E-02,
             9.452397728552E-03, -1.765322333815E-02, -3.551989669485E-02, -4.656675271099E-02,
             -5.018094388273E-02, -4.656675271113E-02, -3.551989669440E-02, -1.765322333876E-02,
             9.452397726093E-03, 3.690531537103E-02, 5.944385073178E-02, 6.859721745415E-02,
             4.759923176233E-02, 1.607440701568E-02, 1.607440701568E-02, 1.607440701568E-02,
             1.607440701568E-02, 4.356156929915E-02, 4.356156929915E-02, 4.356156929915E-02,
             4.356156929915E-02, 5.964570014935E-02, 6.482091617121E-02, 3.690531537805E-02,
             -1.191055430596E-03, -3.406682302067E-02, -6.137786872165E-02, -7.588664280371E-02,
             -8.287206522451E-02, -8.555883978764E-02, -8.287206522502E-02, -7.588664280388E-02,
             -6.137786872209E-02, -3.406682302217E-02, -1.191055436745E-03, 3.690531537487E-02,
             6.482091617227E-02, 5.964570014017E-02, 4.356156929263E-02, 4.356156929263E-02,
             4.356156929263E-02, 4.356156929263E-02, 5.936207315787E-02, 5.936207315787E-02,
             5.936207315787E-02, 5.936207315787E-02, 5.828995986152E-02, 4.830380225713E-02,
             9.452397726372E-03, -3.406682301951E-02, -6.297844907249E-02, -8.246762366680E-02,
             -8.898818051517E-02, -8.945863728280E-02, -9.033251398485E-02, -8.945863728310E-02,
             -8.898818051543E-02, -8.246762366704E-02, -6.297844907350E-02, -3.406682302335E-02,
             9.452397724649E-03, 4.830380225745E-02, 5.828995985726E-02, 5.936207315493E-02,
             5.936207315493E-02, 5.936207315493E-02, 5.936207315493E-02, 6.830161067188E-02,
             6.830161067188E-02, 6.830161067188E-02, 6.830161067188E-02, 5.086584768016E-02,
             2.743644194736E-02, -1.765322333716E-02, -6.137786872188E-02, -8.246762366694E-02,
             -9.132287823500E-02, -8.866359515673E-02, -8.226118588935E-02, -8.112847434613E-02,
             -8.226118588939E-02, -8.866359515687E-02, -9.132287823508E-02, -8.246762366680E-02,
             -6.137786872115E-02, -1.765322333746E-02, 2.743644194769E-02, 5.086584768251E-02,
             6.830161067455E-02, 6.830161067455E-02, 6.830161067455E-02, 6.830161067455E-02,
             7.107762959038E-02, 7.107762959038E-02, 7.107762959038E-02, 7.107762959038E-02,
             4.258073038138E-02, 1.078616361751E-02, -3.551989669405E-02, -7.588664280397E-02,
             -8.898818051528E-02, -8.866359515675E-02, -7.909406561030E-02, -6.792741701972E-02,
             -6.533539475257E-02, -6.792741701977E-02, -7.909406561041E-02, -8.866359515684E-02,
             -8.898818051519E-02, -7.588664280371E-02, -3.551989669442E-02, 1.078616361837E-02,
             4.258073038353E-02, 7.107762959247E-02, 7.107762959247E-02, 7.107762959247E-02,
             7.107762959247E-02, 7.076114333566E-02, 7.076114333566E-02, 7.076114333566E-02,
             7.076114333566E-02, 3.539309463963E-02, -1.191055434244E-03, -4.656675271103E-02,
             -8.287206522482E-02, -8.945863728286E-02, -8.226118588935E-02, -6.792741701970E-02,
             -5.367000806700E-02, -5.009611083921E-02, -5.367000806699E-02, -6.792741701970E-02,
             -8.226118588946E-02, -8.945863728288E-02, -8.287206522469E-02, -4.656675271102E-02,
             -1.191055434066E-03, 3.539309463997E-02, 7.076114333521E-02, 7.076114333521E-02,
             7.076114333521E-02, 7.076114333521E-02, 7.106858491835E-02, 7.106858491835E-02,
             7.106858491835E-02, 7.106858491835E-02, 3.345900050925E-02, -4.769492607011E-03,
             -5.018094388270E-02, -8.555883978772E-02, -9.033251398488E-02, -8.112847434614E-02,
             -6.533539475257E-02, -5.009611083922E-02, -4.621706540085E-02, -5.009611083921E-02,
             -6.533539475257E-02, -8.112847434625E-02, -9.033251398490E-02, -8.555883978761E-02,
             -5.018094388272E-02, -4.769492606812E-03, 3.345900050959E-02, 7.106858491790E-02,
             7.106858491790E-02, 7.106858491790E-02, 7.106858491790E-02, 7.076114333396E-02,
             7.076114333396E-02, 7.076114333396E-02, 7.076114333396E-02, 3.539309463867E-02,
             -1.191055434516E-03, -4.656675271126E-02, -8.287206522498E-02, -8.945863728301E-02,
             -8.226118588933E-02, -6.792741701967E-02, -5.367000806702E-02, -5.009611083921E-02,
             -5.367000806697E-02, -6.792741701964E-02, -8.226118588937E-02, -8.945863728288E-02,
             -8.287206522485E-02, -4.656675271062E-02, -1.191055434693E-03, 3.539309463898E-02,
             7.076114333498E-02, 7.076114333498E-02, 7.076114333498E-02, 7.076114333498E-02,
             7.107762959019E-02, 7.107762959019E-02, 7.107762959019E-02, 7.107762959019E-02,
             4.258073038204E-02, 1.078616361737E-02, -3.551989669443E-02, -7.588664280384E-02,
             -8.898818051525E-02, -8.866359515677E-02, -7.909406561028E-02, -6.792741701971E-02,
             -6.533539475258E-02, -6.792741701968E-02, -7.909406561026E-02, -8.866359515678E-02,
             -8.898818051514E-02, -7.588664280389E-02, -3.551989669383E-02, 1.078616361796E-02,
             4.258073038240E-02, 7.107762959115E-02, 7.107762959115E-02, 7.107762959115E-02,
             7.107762959115E-02, 6.830161067588E-02, 6.830161067588E-02, 6.830161067588E-02,
             6.830161067588E-02, 5.086584768557E-02, 2.743644194734E-02, -1.765322333791E-02,
             -6.137786872193E-02, -8.246762366704E-02, -9.132287823511E-02, -8.866359515673E-02,
             -8.226118588927E-02, -8.112847434610E-02, -8.226118588891E-02, -8.866359515666E-02,
             -9.132287823417E-02, -8.246762366571E-02, -6.137786872110E-02, -1.765322333624E-02,
             2.743644195026E-02, 5.086584767606E-02, 6.830161066853E-02, 6.830161066853E-02,
             6.830161066853E-02, 6.830161066853E-02, 5.936207315005E-02, 5.936207315005E-02,
             5.936207315005E-02, 5.936207315005E-02, 5.828995985678E-02, 4.830380225714E-02,
             9.452397727007E-03, -3.406682302249E-02, -6.297844907372E-02, -8.246762366682E-02,
             -8.898818051514E-02, -8.945863728277E-02, -9.033251398490E-02, -8.945863728260E-02,
             -8.898818051505E-02, -8.246762366635E-02, -6.297844907261E-02, -3.406682302059E-02,
             9.452397730918E-03, 4.830380225822E-02, 5.828995984248E-02, 5.936207314396E-02,
             5.936207314396E-02, 5.936207314396E-02, 5.936207314396E-02, 4.356156927876E-02,
             4.356156927876E-02, 4.356156927876E-02, 4.356156927876E-02, 5.964570012580E-02,
             6.482091617804E-02, 3.690531538273E-02, -1.191055434191E-03, -3.406682302106E-02,
             -6.137786872090E-02, -7.588664280368E-02, -8.287206522483E-02, -8.555883978772E-02,
             -8.287206522455E-02, -7.588664280379E-02, -6.137786872198E-02, -3.406682302096E-02,
             -1.191055432770E-03, 3.690531537512E-02, 6.482091616700E-02, 5.964570013270E-02,
             4.356156928137E-02, 4.356156928137E-02, 4.356156928137E-02, 4.356156928137E-02,
             1.607440701081E-02, 1.607440701081E-02, 1.607440701081E-02, 1.607440701081E-02,
             4.759923175430E-02, 6.859721745905E-02, 5.944385074129E-02, 3.690531537985E-02,
             9.452397728154E-03, -1.765322333673E-02, -3.551989669413E-02, -4.656675271111E-02,
             -5.018094388280E-02, -4.656675271075E-02, -3.551989669466E-02, -1.765322333821E-02,
             9.452397728822E-03, 3.690531537920E-02, 5.944385072904E-02, 6.859721744377E-02,
             4.759923176384E-02, 1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02,
             1.607440701473E-02, -1.378569400687E-02, -1.378569400687E-02, -1.378569400687E-02,
             -1.378569400687E-02, 2.657147812158E-02, 5.864644239897E-02, 6.859721744841E-02,
             6.482091617160E-02, 4.830380225513E-02, 2.743644194722E-02, 1.078616361659E-02,
             -1.191055434532E-03, -4.769492607409E-03, -1.191055434106E-03, 1.078616361715E-02,
             2.743644195107E-02, 4.830380225332E-02, 6.482091617384E-02, 6.859721744401E-02,
             5.864644236959E-02, 2.657147806131E-02, -1.378569410514E-02, -1.378569410514E-02,
             -1.378569410514E-02, -1.378569410514E-02, -3.649895199161E-02, -3.649895199161E-02,
             -3.649895199161E-02, -3.649895199161E-02, -3.088973940848E-03, 2.657147807892E-02,
             4.759923176151E-02, 5.964570013216E-02, 5.828995984835E-02, 5.086584768191E-02,
             4.258073038125E-02, 3.539309463849E-02, 3.345900050823E-02, 3.539309463952E-02,
             4.258073038187E-02, 5.086584768429E-02, 5.828995984560E-02, 5.964570013445E-02,
             4.759923175558E-02, 2.657147804251E-02, -3.088974026880E-03, -3.649895211890E-02,
             -3.649895211890E-02, -3.649895211890E-02, -3.649895211890E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02,
             -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02,
             6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02,
             7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02,
             4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02,
             5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02,
             7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02,
             5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02,
             -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02,
             4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02,
             7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02,
             6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02,
             -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02,
             1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02,
             7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02,
             7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02,
             1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -3.649895211129E-02,
             -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02, 5.936207313800E-02,
             6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02, 7.106858491790E-02,
             7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02, 5.936207314687E-02,
             4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02, -3.649895212129E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02, 4.356156926633E-02,
             5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02, 7.076114333415E-02,
             7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02, 6.830161067791E-02,
             5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02, -1.378569410463E-02,
             -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -3.649895211129E-02, -1.378569409712E-02, 1.607440700968E-02,
             4.356156926633E-02, 5.936207313800E-02, 6.830161067296E-02, 7.107762959174E-02,
             7.076114333415E-02, 7.106858491790E-02, 7.076114333535E-02, 7.107762958924E-02,
             6.830161067791E-02, 5.936207314687E-02, 4.356156927997E-02, 1.607440700445E-02,
             -1.378569410463E-02, -3.649895212129E-02, -5.750924623499E-02, -5.750924623499E-02,
             -5.750924623499E-02, -5.750924623499E-02, -5.750924622225E-02, -5.750924622225E-02,
             -5.750924622225E-02, -5.750924623499E-02, -5.750924623499E-02, -5.750924623499E-02,
             -3.649895211738E-02, -3.649895211738E-02, -3.649895211738E-02, -3.649895213245E-02,
             -3.649895213245E-02, -3.649895213245E-02, -1.378569410836E-02, -1.378569410836E-02,
             -1.378569410836E-02, -1.378569409104E-02, -1.378569409104E-02, -1.378569409104E-02,
             1.607440700266E-02, 1.607440700266E-02, 1.607440700266E-02, 1.607440701568E-02,
             1.607440701568E-02, 1.607440701568E-02, 4.356156929915E-02, 4.356156929915E-02,
             4.356156929915E-02, 4.356156929263E-02, 4.356156929263E-02, 4.356156929263E-02,
             5.936207315787E-02, 5.936207315787E-02, 5.936207315787E-02, 5.936207315493E-02,
             5.936207315493E-02, 5.936207315493E-02, 6.830161067188E-02, 6.830161067188E-02,
             6.830161067188E-02, 6.830161067455E-02, 6.830161067455E-02, 6.830161067455E-02,
             7.107762959038E-02, 7.107762959038E-02, 7.107762959038E-02, 7.107762959247E-02,
             7.107762959247E-02, 7.107762959247E-02, 7.076114333566E-02, 7.076114333566E-02,
             7.076114333566E-02, 7.076114333521E-02, 7.076114333521E-02, 7.076114333521E-02,
             7.106858491835E-02, 7.106858491835E-02, 7.106858491835E-02, 7.106858491790E-02,
             7.106858491790E-02, 7.106858491790E-02, 7.076114333396E-02, 7.076114333396E-02,
             7.076114333396E-02, 7.076114333498E-02, 7.076114333498E-02, 7.076114333498E-02,
             7.107762959019E-02, 7.107762959019E-02, 7.107762959019E-02, 7.107762959115E-02,
             7.107762959115E-02, 7.107762959115E-02, 6.830161067588E-02, 6.830161067588E-02,
             6.830161067588E-02, 6.830161066853E-02, 6.830161066853E-02, 6.830161066853E-02,
             5.936207315005E-02, 5.936207315005E-02, 5.936207315005E-02, 5.936207314396E-02,
             5.936207314396E-02, 5.936207314396E-02, 4.356156927876E-02, 4.356156927876E-02,
             4.356156927876E-02, 4.356156928137E-02, 4.356156928137E-02, 4.356156928137E-02,
             1.607440701081E-02, 1.607440701081E-02, 1.607440701081E-02, 1.607440701473E-02,
             1.607440701473E-02, 1.607440701473E-02, -1.378569400687E-02, -1.378569400687E-02,
             -1.378569400687E-02, -1.378569410514E-02, -1.378569410514E-02, -1.378569410514E-02,
             -3.649895199161E-02, -3.649895199161E-02, -3.649895199161E-02, -3.649895211890E-02,
             -3.649895211890E-02, -3.649895211890E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02, 4.356156928066E-02,
             5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02, 7.076114333449E-02,
             7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02, 6.830161067415E-02,
             5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02, -1.378569412327E-02,
             -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02, 1.607440701093E-02,
             4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02, 7.107762958907E-02,
             7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02, 7.107762959140E-02,
             6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02, 1.607440700366E-02,
             -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924606582E-02, -5.750924606582E-02,
             -5.750924606582E-02, -5.750924606582E-02, -3.649895201579E-02, -1.378569407871E-02,
             1.607440701093E-02, 4.356156928066E-02, 5.936207314301E-02, 6.830161067239E-02,
             7.107762958907E-02, 7.076114333449E-02, 7.106858491733E-02, 7.076114333569E-02,
             7.107762959140E-02, 6.830161067415E-02, 5.936207313869E-02, 4.356156928293E-02,
             1.607440700366E-02, -1.378569412327E-02, -3.649895213039E-02, -5.750924622771E-02,
             -5.750924622771E-02, -5.750924622771E-02, -5.750924622771E-02, -3.649895212153E-02,
             -3.649895212153E-02, -3.649895212153E-02, -3.649895212153E-02, -3.088974014866E-03,
             2.657147806395E-02, 4.759923176017E-02, 5.964570012047E-02, 5.828995984426E-02,
             5.086584768171E-02, 4.258073038296E-02, 3.539309463846E-02, 3.345900050898E-02,
             3.539309463962E-02, 4.258073038087E-02, 5.086584768556E-02, 5.828995985127E-02,
             5.964570013211E-02, 4.759923175469E-02, 2.657147805074E-02, -3.088974026225E-03,
             -3.649895213205E-02, -3.649895213205E-02, -3.649895213205E-02, -3.649895213205E-02,
             -3.649895212153E-02, -3.649895212153E-02, -3.649895212153E-02, -3.649895212153E-02,
             -3.088974014866E-03, 2.657147806395E-02, 4.759923176017E-02, 5.964570012047E-02,
             5.828995984426E-02, 5.086584768171E-02, 4.258073038296E-02, 3.539309463846E-02,
             3.345900050898E-02, 3.539309463962E-02, 4.258073038087E-02, 5.086584768556E-02,
             5.828995985127E-02, 5.964570013211E-02, 4.759923175469E-02, 2.657147805074E-02,
             -3.088974026225E-03, -3.649895213205E-02, -3.649895213205E-02, -3.649895213205E-02,
             -3.649895213205E-02, -3.649895212153E-02, -3.649895212153E-02, -3.649895212153E-02,
             -3.649895212153E-02, -3.088974014866E-03, 2.657147806395E-02, 4.759923176017E-02,
             5.964570012047E-02, 5.828995984426E-02, 5.086584768171E-02, 4.258073038296E-02,
             3.539309463846E-02, 3.345900050898E-02, 3.539309463962E-02, 4.258073038087E-02,
             5.086584768556E-02, 5.828995985127E-02, 5.964570013211E-02, 4.759923175469E-02,
             2.657147805074E-02, -3.088974026225E-03, -3.649895213205E-02, -3.649895213205E-02,
             -3.649895213205E-02, -3.649895213205E-02, -3.649895212153E-02, -3.649895212153E-02,
             -3.649895212153E-02, -3.649895213205E-02, -3.649895213205E-02, -3.649895213205E-02,
             -3.088974026933E-03, -3.088974026933E-03, -3.088974026933E-03, -3.088974039496E-03,
             -3.088974039496E-03, -3.088974039496E-03, 2.657147805463E-02, 2.657147805463E-02,
             2.657147805463E-02, 2.657147806777E-02, 2.657147806777E-02, 2.657147806777E-02,
             4.759923175432E-02, 4.759923175432E-02, 4.759923175432E-02, 4.759923176423E-02,
             4.759923176423E-02, 4.759923176423E-02, 5.964570014721E-02, 5.964570014721E-02,
             5.964570014721E-02, 5.964570014214E-02, 5.964570014214E-02, 5.964570014214E-02,
             5.828995986065E-02, 5.828995986065E-02, 5.828995986065E-02, 5.828995985828E-02,
             5.828995985828E-02, 5.828995985828E-02, 5.086584768094E-02, 5.086584768094E-02,
             5.086584768094E-02, 5.086584768228E-02, 5.086584768228E-02, 5.086584768228E-02,
             4.258073038191E-02, 4.258073038191E-02, 4.258073038191E-02, 4.258073038292E-02,
             4.258073038292E-02, 4.258073038292E-02, 3.539309463981E-02, 3.539309463981E-02,
             3.539309463981E-02, 3.539309463949E-02, 3.539309463949E-02, 3.539309463949E-02,
             3.345900050944E-02, 3.345900050944E-02, 3.345900050944E-02, 3.345900050910E-02,
             3.345900050910E-02, 3.345900050910E-02, 3.539309463820E-02, 3.539309463820E-02,
             3.539309463820E-02, 3.539309463608E-02, 3.539309463608E-02, 3.539309463608E-02,
             4.258073038148E-02, 4.258073038148E-02, 4.258073038148E-02, 4.258073038075E-02,
             4.258073038075E-02, 4.258073038075E-02, 5.086584768413E-02, 5.086584768413E-02,
             5.086584768413E-02, 5.086584766974E-02, 5.086584766974E-02, 5.086584766974E-02,
             5.828995985421E-02, 5.828995985421E-02, 5.828995985421E-02, 5.828995984365E-02,
             5.828995984365E-02, 5.828995984365E-02, 5.964570013105E-02, 5.964570013105E-02,
             5.964570013105E-02, 5.964570013002E-02, 5.964570013002E-02, 5.964570013002E-02,
             4.759923176137E-02, 4.759923176137E-02, 4.759923176137E-02, 4.759923176577E-02,
             4.759923176577E-02, 4.759923176577E-02, 2.657147814155E-02, 2.657147814155E-02,
             2.657147814155E-02, 2.657147805003E-02, 2.657147805003E-02, 2.657147805003E-02,
             -3.088973915079E-03, -3.088973915079E-03, -3.088973915079E-03, -3.088974027303E-03,
             -3.088974027303E-03, -3.088974027303E-03, -3.649895198230E-02, -3.649895198230E-02,
             -3.649895198230E-02, -3.649895212067E-02, -3.649895212067E-02, -3.649895212067E-02,
             -3.649895198230E-02, -3.649895198230E-02, -3.649895198230E-02, -3.649895198230E-02,
             -3.088973932833E-03, 2.657147807884E-02, 4.759923176142E-02, 5.964570013161E-02,
             5.828995984817E-02, 5.086584768120E-02, 4.258073038038E-02, 3.539309463881E-02,
             3.345900050856E-02, 3.539309463970E-02, 4.258073038310E-02, 5.086584768503E-02,
             5.828995984593E-02, 5.964570013567E-02, 4.759923176153E-02, 2.657147803679E-02,
             -3.088974031520E-03, -3.649895212067E-02, -3.649895212067E-02, -3.649895212067E-02,
             -3.649895212067E-02, -3.649895198230E-02, -3.649895198230E-02, -3.649895198230E-02,
             -3.649895198230E-02, -3.088973932833E-03, 2.657147807884E-02, 4.759923176142E-02,
             5.964570013161E-02, 5.828995984817E-02, 5.086584768120E-02, 4.258073038038E-02,
             3.539309463881E-02, 3.345900050856E-02, 3.539309463970E-02, 4.258073038310E-02,
             5.086584768503E-02, 5.828995984593E-02, 5.964570013567E-02, 4.759923176153E-02,
             2.657147803679E-02, -3.088974031520E-03, -3.649895212067E-02, -3.649895212067E-02,
             -3.649895212067E-02, -3.649895212067E-02, -3.649895198230E-02, -3.649895198230E-02,
             -3.649895198230E-02, -3.649895198230E-02, -3.088973932833E-03, 2.657147807884E-02,
             4.759923176142E-02, 5.964570013161E-02, 5.828995984817E-02, 5.086584768120E-02,
             4.258073038038E-02, 3.539309463881E-02, 3.345900050856E-02, 3.539309463970E-02,
             4.258073038310E-02, 5.086584768503E-02, 5.828995984593E-02, 5.964570013567E-02,
             4.759923176153E-02, 2.657147803679E-02, -3.088974031520E-03, -3.649895212067E-02,
             -3.649895212067E-02, -3.649895212067E-02, -3.649895212067E-02, -1.378569409171E-02,
             -1.378569409171E-02, -1.378569409171E-02, -1.378569409171E-02, 2.657147806499E-02,
             5.864644238892E-02, 6.859721743952E-02, 6.482091617400E-02, 4.830380225833E-02,
             2.743644194838E-02, 1.078616361776E-02, -1.191055434395E-03, -4.769492607174E-03,
             -1.191055434461E-03, 1.078616361809E-02, 2.743644194982E-02, 4.830380226187E-02,
             6.482091618142E-02, 6.859721746023E-02, 5.864644239491E-02, 2.657147806935E-02,
             -1.378569409693E-02, -1.378569409693E-02, -1.378569409693E-02, -1.378569409693E-02,
             -1.378569409171E-02, -1.378569409171E-02, -1.378569409171E-02, -1.378569409171E-02,
             2.657147806499E-02, 5.864644238892E-02, 6.859721743952E-02, 6.482091617400E-02,
             4.830380225833E-02, 2.743644194838E-02, 1.078616361776E-02, -1.191055434395E-03,
             -4.769492607174E-03, -1.191055434461E-03, 1.078616361809E-02, 2.743644194982E-02,
             4.830380226187E-02, 6.482091618142E-02, 6.859721746023E-02, 5.864644239491E-02,
             2.657147806935E-02, -1.378569409693E-02, -1.378569409693E-02, -1.378569409693E-02,
             -1.378569409693E-02, -1.378569409171E-02, -1.378569409171E-02, -1.378569409171E-02,
             -1.378569409171E-02, 2.657147806499E-02, 5.864644238892E-02, 6.859721743952E-02,
             6.482091617400E-02, 4.830380225833E-02, 2.743644194838E-02, 1.078616361776E-02,
             -1.191055434395E-03, -4.769492607174E-03, -1.191055434461E-03, 1.078616361809E-02,
             2.743644194982E-02, 4.830380226187E-02, 6.482091618142E-02, 6.859721746023E-02,
             5.864644239491E-02, 2.657147806935E-02, -1.378569409693E-02, -1.378569409693E-02,
             -1.378569409693E-02, -1.378569409693E-02, -1.378569409171E-02, -1.378569409171E-02,
             -1.378569409171E-02, -1.378569409693E-02, -1.378569409693E-02, -1.378569409693E-02,
             2.657147806897E-02, 2.657147806897E-02, 2.657147806897E-02, 2.657147806331E-02,
             2.657147806331E-02, 2.657147806331E-02, 5.864644238220E-02, 5.864644238220E-02,
             5.864644238220E-02, 5.864644238279E-02, 5.864644238279E-02, 5.864644238279E-02,
             6.859721744398E-02, 6.859721744398E-02, 6.859721744398E-02, 6.859721744240E-02,
             6.859721744240E-02, 6.859721744240E-02, 6.482091617052E-02, 6.482091617052E-02,
             6.482091617052E-02, 6.482091616928E-02, 6.482091616928E-02, 6.482091616928E-02,
             4.830380225667E-02, 4.830380225667E-02, 4.830380225667E-02, 4.830380225714E-02,
             4.830380225714E-02, 4.830380225714E-02, 2.743644194828E-02, 2.743644194828E-02,
             2.743644194828E-02, 2.743644194786E-02, 2.743644194786E-02, 2.743644194786E-02,
             1.078616361804E-02, 1.078616361804E-02, 1.078616361804E-02, 1.078616361864E-02,
             1.078616361864E-02, 1.078616361864E-02, -1.191055434311E-03, -1.191055434311E-03,
             -1.191055434311E-03, -1.191055435457E-03, -1.191055435457E-03, -1.191055435457E-03,
             -4.769492607076E-03, -4.769492607076E-03, -4.769492607076E-03, -4.769492607615E-03,
             -4.769492607615E-03, -4.769492607615E-03, -1.191055433243E-03, -1.191055433243E-03,
             -1.191055433243E-03, -1.191055444137E-03, -1.191055444137E-03, -1.191055444137E-03,
             1.078616361884E-02, 1.078616361884E-02, 1.078616361884E-02, 1.078616361705E-02,
             1.078616361705E-02, 1.078616361705E-02, 2.743644194669E-02, 2.743644194669E-02,
             2.743644194669E-02, 2.743644195000E-02, 2.743644195000E-02, 2.743644195000E-02,
             4.830380225634E-02, 4.830380225634E-02, 4.830380225634E-02, 4.830380224711E-02,
             4.830380224711E-02, 4.830380224711E-02, 6.482091617827E-02, 6.482091617827E-02,
             6.482091617827E-02, 6.482091616720E-02, 6.482091616720E-02, 6.482091616720E-02,
             6.859721745869E-02, 6.859721745869E-02, 6.859721745869E-02, 6.859721745334E-02,
             6.859721745334E-02, 6.859721745334E-02, 5.864644238720E-02, 5.864644238720E-02,
             5.864644238720E-02, 5.864644240619E-02, 5.864644240619E-02, 5.864644240619E-02,
             2.657147806632E-02, 2.657147806632E-02, 2.657147806632E-02, 2.657147808980E-02,
             2.657147808980E-02, 2.657147808980E-02, -1.378569409165E-02, -1.378569409165E-02,
             -1.378569409165E-02, -1.378569406431E-02, -1.378569406431E-02, -1.378569406431E-02,
             -1.378569409165E-02, -1.378569409165E-02, -1.378569409165E-02, -1.378569409165E-02,
             2.657147806577E-02, 5.864644239914E-02, 6.859721745113E-02, 6.482091617246E-02,
             4.830380225762E-02, 2.743644194008E-02, 1.078616361549E-02, -1.191055432767E-03,
             -4.769492606708E-03, -1.191055401971E-03, 1.078616362758E-02, 2.743644195091E-02,
             4.830380220373E-02, 6.482091616552E-02, 6.859721745172E-02, 5.864644238170E-02,
             2.657147807189E-02, -1.378569406431E-02, -1.378569406431E-02, -1.378569406431E-02,
             -1.378569406431E-02, -1.378569409165E-02, -1.378569409165E-02, -1.378569409165E-02,
             -1.378569409165E-02, 2.657147806577E-02, 5.864644239914E-02, 6.859721745113E-02,
             6.482091617246E-02, 4.830380225762E-02, 2.743644194008E-02, 1.078616361549E-02,
             -1.191055432767E-03, -4.769492606708E-03, -1.191055401971E-03, 1.078616362758E-02,
             2.743644195091E-02, 4.830380220373E-02, 6.482091616552E-02, 6.859721745172E-02,
             5.864644238170E-02, 2.657147807189E-02, -1.378569406431E-02, -1.378569406431E-02,
             -1.378569406431E-02, -1.378569406431E-02, -1.378569409165E-02, -1.378569409165E-02,
             -1.378569409165E-02, -1.378569409165E-02, 2.657147806577E-02, 5.864644239914E-02,
             6.859721745113E-02, 6.482091617246E-02, 4.830380225762E-02, 2.743644194008E-02,
             1.078616361549E-02, -1.191055432767E-03, -4.769492606708E-03, -1.191055401971E-03,
             1.078616362758E-02, 2.743644195091E-02, 4.830380220373E-02, 6.482091616552E-02,
             6.859721745172E-02, 5.864644238170E-02, 2.657147807189E-02, -1.378569406431E-02,
             -1.378569406431E-02, -1.378569406431E-02, -1.378569406431E-02, 1.607440701473E-02,
             1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02, 4.759923176159E-02,
             6.859721744875E-02, 5.944385072825E-02, 3.690531538111E-02, 9.452397727812E-03,
             -1.765322333726E-02, -3.551989669422E-02, -4.656675271119E-02, -5.018094388287E-02,
             -4.656675271125E-02, -3.551989669398E-02, -1.765322333610E-02, 9.452397730952E-03,
             3.690531538776E-02, 5.944385074483E-02, 6.859721745215E-02, 4.759923176383E-02,
             1.607440700824E-02, 1.607440700824E-02, 1.607440700824E-02, 1.607440700824E-02,
             1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02,
             4.759923176159E-02, 6.859721744875E-02, 5.944385072825E-02, 3.690531538111E-02,
             9.452397727812E-03, -1.765322333726E-02, -3.551989669422E-02, -4.656675271119E-02,
             -5.018094388287E-02, -4.656675271125E-02, -3.551989669398E-02, -1.765322333610E-02,
             9.452397730952E-03, 3.690531538776E-02, 5.944385074483E-02, 6.859721745215E-02,
             4.759923176383E-02, 1.607440700824E-02, 1.607440700824E-02, 1.607440700824E-02,
             1.607440700824E-02, 1.607440701473E-02, 1.607440701473E-02, 1.607440701473E-02,
             1.607440701473E-02, 4.759923176159E-02, 6.859721744875E-02, 5.944385072825E-02,
             3.690531538111E-02, 9.452397727812E-03, -1.765322333726E-02, -3.551989669422E-02,
             -4.656675271119E-02, -5.018094388287E-02, -4.656675271125E-02, -3.551989669398E-02,
             -1.765322333610E-02, 9.452397730952E-03, 3.690531538776E-02, 5.944385074483E-02,
             6.859721745215E-02, 4.759923176383E-02, 1.607440700824E-02, 1.607440700824E-02,
             1.607440700824E-02, 1.607440700824E-02, 1.607440701473E-02, 1.607440701473E-02,
             1.607440701473E-02, 1.607440700824E-02, 1.607440700824E-02, 1.607440700824E-02,
             4.759923176501E-02, 4.759923176501E-02, 4.759923176501E-02, 4.759923175831E-02,
             4.759923175831E-02, 4.759923175831E-02, 6.859721744378E-02, 6.859721744378E-02,
             6.859721744378E-02, 6.859721744397E-02, 6.859721744397E-02, 6.859721744397E-02,
             5.944385073225E-02, 5.944385073225E-02, 5.944385073225E-02, 5.944385073084E-02,
             5.944385073084E-02, 5.944385073084E-02, 3.690531537863E-02, 3.690531537863E-02,
             3.690531537863E-02, 3.690531537959E-02, 3.690531537959E-02, 3.690531537959E-02,
             9.452397726771E-03, 9.452397726771E-03, 9.452397726771E-03, 9.452397731244E-03,
             9.452397731244E-03, 9.452397731244E-03, -1.765322333702E-02, -1.765322333702E-02,
             -1.765322333702E-02, -1.765322335236E-02, -1.765322335236E-02, -1.765322335236E-02,
             -3.551989669374E-02, -3.551989669374E-02, -3.551989669374E-02, -3.551989670593E-02,
             -3.551989670593E-02, -3.551989670593E-02, -4.656675271117E-02, -4.656675271117E-02,
             -4.656675271117E-02, -4.656675271367E-02, -4.656675271367E-02, -4.656675271367E-02,
             -5.018094388285E-02, -5.018094388285E-02, -5.018094388285E-02, -5.018094388674E-02,
             -5.018094388674E-02, -5.018094388674E-02, -4.656675271024E-02, -4.656675271024E-02,
             -4.656675271024E-02, -4.656675266841E-02, -4.656675266841E-02, -4.656675266841E-02,
             -3.551989669340E-02, -3.551989669340E-02, -3.551989669340E-02, -3.551989665830E-02,
             -3.551989665830E-02, -3.551989665830E-02, -1.765322334055E-02, -1.765322334055E-02,
             -1.765322334055E-02, -1.765322335177E-02, -1.765322335177E-02, -1.765322335177E-02,
             9.452397724621E-03, 9.452397724621E-03, 9.452397724621E-03, 9.452397679747E-03,
             9.452397679747E-03, 9.452397679747E-03, 3.690531538472E-02, 3.690531538472E-02,
             3.690531538472E-02, 3.690531534465E-02, 3.690531534465E-02, 3.690531534465E-02,
             5.944385074365E-02, 5.944385074365E-02, 5.944385074365E-02, 5.944385071627E-02,
             5.944385071627E-02, 5.944385071627E-02, 6.859721744906E-02, 6.859721744906E-02,
             6.859721744906E-02, 6.859721743676E-02, 6.859721743676E-02, 6.859721743676E-02,
             4.759923176403E-02, 4.759923176403E-02, 4.759923176403E-02, 4.759923179396E-02,
             4.759923179396E-02, 4.759923179396E-02, 1.607440701618E-02, 1.607440701618E-02,
             1.607440701618E-02, 1.607440708711E-02, 1.607440708711E-02, 1.607440708711E-02,
             1.607440701618E-02, 1.607440701618E-02, 1.607440701618E-02, 1.607440701618E-02,
             4.759923176298E-02, 6.859721746138E-02, 5.944385074167E-02, 3.690531538286E-02,
             9.452397732280E-03, -1.765322334998E-02, -3.551989669664E-02, -4.656675270936E-02,
             -5.018094388014E-02, -4.656675266850E-02, -3.551989665375E-02, -1.765322330930E-02,
             9.452397663657E-03, 3.690531534455E-02, 5.944385072329E-02, 6.859721744358E-02,
             4.759923179470E-02, 1.607440708711E-02, 1.607440708711E-02, 1.607440708711E-02,
             1.607440708711E-02, 1.607440701618E-02, 1.607440701618E-02, 1.607440701618E-02,
             1.607440701618E-02, 4.759923176298E-02, 6.859721746138E-02, 5.944385074167E-02,
             3.690531538286E-02, 9.452397732280E-03, -1.765322334998E-02, -3.551989669664E-02,
             -4.656675270936E-02, -5.018094388014E-02, -4.656675266850E-02, -3.551989665375E-02,
             -1.765322330930E-02, 9.452397663657E-03, 3.690531534455E-02, 5.944385072329E-02,
             6.859721744358E-02, 4.759923179470E-02, 1.607440708711E-02, 1.607440708711E-02,
             1.607440708711E-02, 1.607440708711E-02, 1.607440701618E-02, 1.607440701618E-02,
             1.607440701618E-02, 1.607440701618E-02, 4.759923176298E-02, 6.859721746138E-02,
             5.944385074167E-02, 3.690531538286E-02, 9.452397732280E-03, -1.765322334998E-02,
             -3.551989669664E-02, -4.656675270936E-02, -5.018094388014E-02, -4.656675266850E-02,
             -3.551989665375E-02, -1.765322330930E-02, 9.452397663657E-03, 3.690531534455E-02,
             5.944385072329E-02, 6.859721744358E-02, 4.759923179470E-02, 1.607440708711E-02,
             1.607440708711E-02, 1.607440708711E-02, 1.607440708711E-02, 4.356156927379E-02,
             4.356156927379E-02, 4.356156927379E-02, 4.356156927379E-02, 5.964570012346E-02,
             6.482091617535E-02, 3.690531538307E-02, -1.191055438631E-03, -3.406682302464E-02,
             -6.137786872110E-02, -7.588664280318E-02, -8.287206522455E-02, -8.555883978744E-02,
             -8.287206522474E-02, -7.588664280331E-02, -6.137786872323E-02, -3.406682302612E-02,
             -1.191055431078E-03, 3.690531538558E-02, 6.482091614509E-02, 5.964570010327E-02,
             4.356156924574E-02, 4.356156924574E-02, 4.356156924574E-02, 4.356156924574E-02,
             4.356156927379E-02, 4.356156927379E-02, 4.356156927379E-02, 4.356156927379E-02,
             5.964570012346E-02, 6.482091617535E-02, 3.690531538307E-02, -1.191055438631E-03,
             -3.406682302464E-02, -6.137786872110E-02, -7.588664280318E-02, -8.287206522455E-02,
             -8.555883978744E-02, -8.287206522474E-02, -7.588664280331E-02, -6.137786872323E-02,
             -3.406682302612E-02, -1.191055431078E-03, 3.690531538558E-02, 6.482091614509E-02,
             5.964570010327E-02, 4.356156924574E-02, 4.356156924574E-02, 4.356156924574E-02,
             4.356156924574E-02, 4.356156927379E-02, 4.356156927379E-02, 4.356156927379E-02,
             4.356156927379E-02, 5.964570012346E-02, 6.482091617535E-02, 3.690531538307E-02,
             -1.191055438631E-03, -3.406682302464E-02, -6.137786872110E-02, -7.588664280318E-02,
             -8.287206522455E-02, -8.555883978744E-02, -8.287206522474E-02, -7.588664280331E-02,
             -6.137786872323E-02, -3.406682302612E-02, -1.191055431078E-03, 3.690531538558E-02,
             6.482091614509E-02, 5.964570010327E-02, 4.356156924574E-02, 4.356156924574E-02,
             4.356156924574E-02, 4.356156924574E-02, 4.356156927379E-02, 4.356156927379E-02,
             4.356156927379E-02, 4.356156924574E-02, 4.356156924574E-02, 4.356156924574E-02,
             5.964570012421E-02, 5.964570012421E-02, 5.964570012421E-02, 5.964570010279E-02,
             5.964570010279E-02, 5.964570010279E-02, 6.482091617972E-02, 6.482091617972E-02,
             6.482091617972E-02, 6.482091615177E-02, 6.482091615177E-02, 6.482091615177E-02,
             3.690531538567E-02, 3.690531538567E-02, 3.690531538567E-02, 3.690531536190E-02,
             3.690531536190E-02, 3.690531536190E-02, -1.191055433694E-03, -1.191055433694E-03,
             -1.191055433694E-03, -1.191055437527E-03, -1.191055437527E-03, -1.191055437527E-03,
             -3.406682302200E-02, -3.406682302200E-02, -3.406682302200E-02, -3.406682302215E-02,
             -3.406682302215E-02, -3.406682302215E-02, -6.137786871924E-02, -6.137786871924E-02,
             -6.137786871924E-02, -6.137786872006E-02, -6.137786872006E-02, -6.137786872006E-02,
             -7.588664280080E-02, -7.588664280080E-02, -7.588664280080E-02, -7.588664281097E-02,
             -7.588664281097E-02, -7.588664281097E-02, -8.287206522469E-02, -8.287206522469E-02,
             -8.287206522469E-02, -8.287206522583E-02, -8.287206522583E-02, -8.287206522583E-02,
             -8.555883978757E-02, -8.555883978757E-02, -8.555883978757E-02, -8.555883978745E-02,
             -8.555883978745E-02, -8.555883978745E-02, -8.287206522079E-02, -8.287206522079E-02,
             -8.287206522079E-02, -8.287206522834E-02, -8.287206522834E-02, -8.287206522834E-02,
             -7.588664279949E-02, -7.588664279949E-02, -7.588664279949E-02, -7.588664282472E-02,
             -7.588664282472E-02, -7.588664282472E-02, -6.137786872785E-02, -6.137786872785E-02,
             -6.137786872785E-02, -6.137786860979E-02, -6.137786860979E-02, -6.137786860979E-02,
             -3.406682303057E-02, -3.406682303057E-02, -3.406682303057E-02, -3.406682298535E-02,
             -3.406682298535E-02, -3.406682298535E-02, -1.191055444583E-03, -1.191055444583E-03,
             -1.191055444583E-03, -1.191055158805E-03, -1.191055158805E-03, -1.191055158805E-03,
             3.690531536620E-02, 3.690531536620E-02, 3.690531536620E-02, 3.690531546388E-02,
             3.690531546388E-02, 3.690531546388E-02, 6.482091618360E-02, 6.482091618360E-02,
             6.482091618360E-02, 6.482091617096E-02, 6.482091617096E-02, 6.482091617096E-02,
             5.964570014300E-02, 5.964570014300E-02, 5.964570014300E-02, 5.964570021521E-02,
             5.964570021521E-02, 5.964570021521E-02, 4.356156929032E-02, 4.356156929032E-02,
             4.356156929032E-02, 4.356156944608E-02, 4.356156944608E-02, 4.356156944608E-02,
             4.356156929032E-02, 4.356156929032E-02, 4.356156929032E-02, 4.356156929032E-02,
             5.964570014048E-02, 6.482091618607E-02, 3.690531539543E-02, -1.191055431360E-03,
             -3.406682302387E-02, -6.137786867720E-02, -7.588664277964E-02, -8.287206522394E-02,
             -8.555883978896E-02, -8.287206531375E-02, -7.588664283118E-02, -6.137786880848E-02,
             -3.406682336366E-02, -1.191055144756E-03, 3.690531559754E-02, 6.482091618400E-02,
             5.964570022560E-02, 4.356156944608E-02, 4.356156944608E-02, 4.356156944608E-02,
             4.356156944608E-02, 4.356156929032E-02, 4.356156929032E-02, 4.356156929032E-02,
             4.356156929032E-02, 5.964570014048E-02, 6.482091618607E-02, 3.690531539543E-02,
             -1.191055431360E-03, -3.406682302387E-02, -6.137786867720E-02, -7.588664277964E-02,
             -8.287206522394E-02, -8.555883978896E-02, -8.287206531375E-02, -7.588664283118E-02,
             -6.137786880848E-02, -3.406682336366E-02, -1.191055144756E-03, 3.690531559754E-02,
             6.482091618400E-02, 5.964570022560E-02, 4.356156944608E-02, 4.356156944608E-02,
             4.356156944608E-02, 4.356156944608E-02, 4.356156929032E-02, 4.356156929032E-02,
             4.356156929032E-02, 4.356156929032E-02, 5.964570014048E-02, 6.482091618607E-02,
             3.690531539543E-02, -1.191055431360E-03, -3.406682302387E-02, -6.137786867720E-02,
             -7.588664277964E-02, -8.287206522394E-02, -8.555883978896E-02, -8.287206531375E-02,
             -7.588664283118E-02, -6.137786880848E-02, -3.406682336366E-02, -1.191055144756E-03,
             3.690531559754E-02, 6.482091618400E-02, 5.964570022560E-02, 4.356156944608E-02,
             4.356156944608E-02, 4.356156944608E-02, 4.356156944608E-02, 5.936207313787E-02,
             5.936207313787E-02, 5.936207313787E-02, 5.936207313787E-02, 5.828995984270E-02,
             4.830380226102E-02, 9.452397730714E-03, -3.406682302576E-02, -6.297844907573E-02,
             -8.246762366632E-02, -8.898818051474E-02, -8.945863728260E-02, -9.033251398459E-02,
             -8.945863728276E-02, -8.898818051488E-02, -8.246762366854E-02, -6.297844907725E-02,
             -3.406682301648E-02, 9.452397735866E-03, 4.830380223503E-02, 5.828995982631E-02,
             5.936207311695E-02, 5.936207311695E-02, 5.936207311695E-02, 5.936207311695E-02,
             5.936207313787E-02, 5.936207313787E-02, 5.936207313787E-02, 5.936207313787E-02,
             5.828995984270E-02, 4.830380226102E-02, 9.452397730714E-03, -3.406682302576E-02,
             -6.297844907573E-02, -8.246762366632E-02, -8.898818051474E-02, -8.945863728260E-02,
             -9.033251398459E-02, -8.945863728276E-02, -8.898818051488E-02, -8.246762366854E-02,
             -6.297844907725E-02, -3.406682301648E-02, 9.452397735866E-03, 4.830380223503E-02,
             5.828995982631E-02, 5.936207311695E-02, 5.936207311695E-02, 5.936207311695E-02,
             5.936207311695E-02, 5.936207313787E-02, 5.936207313787E-02, 5.936207313787E-02,
             5.936207313787E-02, 5.828995984270E-02, 4.830380226102E-02, 9.452397730714E-03,
             -3.406682302576E-02, -6.297844907573E-02, -8.246762366632E-02, -8.898818051474E-02,
             -8.945863728260E-02, -9.033251398459E-02, -8.945863728276E-02, -8.898818051488E-02,
             -8.246762366854E-02, -6.297844907725E-02, -3.406682301648E-02, 9.452397735866E-03,
             4.830380223503E-02, 5.828995982631E-02, 5.936207311695E-02, 5.936207311695E-02,
             5.936207311695E-02, 5.936207311695E-02, 5.936207313787E-02, 5.936207313787E-02,
             5.936207313787E-02, 5.936207311695E-02, 5.936207311695E-02, 5.936207311695E-02,
             5.828995984182E-02, 5.828995984182E-02, 5.828995984182E-02, 5.828995982663E-02,
             5.828995982663E-02, 5.828995982663E-02, 4.830380226644E-02, 4.830380226644E-02,
             4.830380226644E-02, 4.830380223510E-02, 4.830380223510E-02, 4.830380223510E-02,
             9.452397734617E-03, 9.452397734617E-03, 9.452397734617E-03, 9.452397710263E-03,
             9.452397710263E-03, 9.452397710263E-03, -3.406682302120E-02, -3.406682302120E-02,
             -3.406682302120E-02, -3.406682301067E-02, -3.406682301067E-02, -3.406682301067E-02,
             -6.297844907363E-02, -6.297844907363E-02, -6.297844907363E-02, -6.297844907515E-02,
             -6.297844907515E-02, -6.297844907515E-02, -8.246762366334E-02, -8.246762366334E-02,
             -8.246762366334E-02, -8.246762366568E-02, -8.246762366568E-02, -8.246762366568E-02,
             -8.898818051105E-02, -8.898818051105E-02, -8.898818051105E-02, -8.898818052628E-02,
             -8.898818052628E-02, -8.898818052628E-02, -8.945863728284E-02, -8.945863728284E-02,
             -8.945863728284E-02, -8.945863728550E-02, -8.945863728550E-02, -8.945863728550E-02,
             -9.033251398484E-02, -9.033251398484E-02, -9.033251398484E-02, -9.033251398938E-02,
             -9.033251398938E-02, -9.033251398938E-02, -8.945863727839E-02, -8.945863727839E-02,
             -8.945863727839E-02, -8.945863735750E-02, -8.945863735750E-02, -8.945863735750E-02,
             -8.898818051067E-02, -8.898818051067E-02, -8.898818051067E-02, -8.898818065688E-02,
             -8.898818065688E-02, -8.898818065688E-02, -8.246762367447E-02, -8.246762367447E-02,
             -8.246762367447E-02, -8.246762358499E-02, -8.246762358499E-02, -8.246762358499E-02,
             -6.297844908367E-02, -6.297844908367E-02, -6.297844908367E-02, -6.297844887283E-02,
             -6.297844887283E-02, -6.297844887283E-02, -3.406682303576E-02, -3.406682303576E-02,
             -3.406682303576E-02, -3.406682258364E-02, -3.406682258364E-02, -3.406682258364E-02,
             9.452397709324E-03, 9.452397709324E-03, 9.452397709324E-03, 9.452397732179E-03,
             9.452397732179E-03, 9.452397732179E-03, 4.830380228359E-02, 4.830380228359E-02,
             4.830380228359E-02, 4.830380283486E-02, 4.830380283486E-02, 4.830380283486E-02,
             5.828995987769E-02, 5.828995987769E-02, 5.828995987769E-02, 5.828996042445E-02,
             5.828996042445E-02, 5.828996042445E-02, 5.936207317315E-02, 5.936207317315E-02,
             5.936207317315E-02, 5.936207368426E-02, 5.936207368426E-02, 5.936207368426E-02,
             5.936207317315E-02, 5.936207317315E-02, 5.936207317315E-02, 5.936207317315E-02,
             5.828995987299E-02, 4.830380226953E-02, 9.452397739812E-03, -3.406682301444E-02,
             -6.297844907272E-02, -8.246762363158E-02, -8.898818049328E-02, -8.945863728415E-02,
             -9.033251398543E-02, -8.945863737774E-02, -8.898818052876E-02, -8.246762363826E-02,
             -6.297844925954E-02, -3.406682348468E-02, 9.452397589606E-03, 4.830380218507E-02,
             5.828996005643E-02, 5.936207368426E-02, 5.936207368426E-02, 5.936207368426E-02,
             5.936207368426E-02, 5.936207317315E-02, 5.936207317315E-02, 5.936207317315E-02,
             5.936207317315E-02, 5.828995987299E-02, 4.830380226953E-02, 9.452397739812E-03,
             -3.406682301444E-02, -6.297844907272E-02, -8.246762363158E-02, -8.898818049328E-02,
             -8.945863728415E-02, -9.033251398543E-02, -8.945863737774E-02, -8.898818052876E-02,
             -8.246762363826E-02, -6.297844925954E-02, -3.406682348468E-02, 9.452397589606E-03,
             4.830380218507E-02, 5.828996005643E-02, 5.936207368426E-02, 5.936207368426E-02,
             5.936207368426E-02, 5.936207368426E-02, 5.936207317315E-02, 5.936207317315E-02,
             5.936207317315E-02, 5.936207317315E-02, 5.828995987299E-02, 4.830380226953E-02,
             9.452397739812E-03, -3.406682301444E-02, -6.297844907272E-02, -8.246762363158E-02,
             -8.898818049328E-02, -8.945863728415E-02, -9.033251398543E-02, -8.945863737774E-02,
             -8.898818052876E-02, -8.246762363826E-02, -6.297844925954E-02, -3.406682348468E-02,
             9.452397589606E-03, 4.830380218507E-02, 5.828996005643E-02, 5.936207368426E-02,
             5.936207368426E-02, 5.936207368426E-02, 5.936207368426E-02, 6.830161066213E-02,
             6.830161066213E-02, 6.830161066213E-02, 6.830161066213E-02, 5.086584767314E-02,
             2.743644194748E-02, -1.765322333770E-02, -6.137786872134E-02, -8.246762366660E-02,
             -9.132287823289E-02, -8.866359515589E-02, -8.226118588934E-02, -8.112847434614E-02,
             -8.226118588929E-02, -8.866359515663E-02, -9.132287823512E-02, -8.246762366705E-02,
             -6.137786871882E-02, -1.765322333222E-02, 2.743644194896E-02, 5.086584768387E-02,
             6.830161067476E-02, 6.830161067476E-02, 6.830161067476E-02, 6.830161067476E-02,
             6.830161066213E-02, 6.830161066213E-02, 6.830161066213E-02, 6.830161066213E-02,
             5.086584767314E-02, 2.743644194748E-02, -1.765322333770E-02, -6.137786872134E-02,
             -8.246762366660E-02, -9.132287823289E-02, -8.866359515589E-02, -8.226118588934E-02,
             -8.112847434614E-02, -8.226118588929E-02, -8.866359515663E-02, -9.132287823512E-02,
             -8.246762366705E-02, -6.137786871882E-02, -1.765322333222E-02, 2.743644194896E-02,
             5.086584768387E-02, 6.830161067476E-02, 6.830161067476E-02, 6.830161067476E-02,
             6.830161067476E-02, 6.830161066213E-02, 6.830161066213E-02, 6.830161066213E-02,
             6.830161066213E-02, 5.086584767314E-02, 2.743644194748E-02, -1.765322333770E-02,
             -6.137786872134E-02, -8.246762366660E-02, -9.132287823289E-02, -8.866359515589E-02,
             -8.226118588934E-02, -8.112847434614E-02, -8.226118588929E-02, -8.866359515663E-02,
             -9.132287823512E-02, -8.246762366705E-02, -6.137786871882E-02, -1.765322333222E-02,
             2.743644194896E-02, 5.086584768387E-02, 6.830161067476E-02, 6.830161067476E-02,
             6.830161067476E-02, 6.830161067476E-02, 6.830161066213E-02, 6.830161066213E-02,
             6.830161066213E-02, 6.830161067476E-02, 6.830161067476E-02, 6.830161067476E-02,
             5.086584767283E-02, 5.086584767283E-02, 5.086584767283E-02, 5.086584768288E-02,
             5.086584768288E-02, 5.086584768288E-02, 2.743644195038E-02, 2.743644195038E-02,
             2.743644195038E-02, 2.743644195755E-02, 2.743644195755E-02, 2.743644195755E-02,
             -1.765322333681E-02, -1.765322333681E-02, -1.765322333681E-02, -1.765322332082E-02,
             -1.765322332082E-02, -1.765322332082E-02, -6.137786871733E-02, -6.137786871733E-02,
             -6.137786871733E-02, -6.137786872614E-02, -6.137786872614E-02, -6.137786872614E-02,
             -8.246762366307E-02, -8.246762366307E-02, -8.246762366307E-02, -8.246762367722E-02,
             -8.246762367722E-02, -8.246762367722E-02, -9.132287820985E-02, -9.132287820985E-02,
             -9.132287820985E-02, -9.132287818135E-02, -9.132287818135E-02, -9.132287818135E-02,
             -8.866359512528E-02, -8.866359512528E-02, -8.866359512528E-02, -8.866359500230E-02,
             -8.866359500230E-02, -8.866359500230E-02, -8.226118588719E-02, -8.226118588719E-02,
             -8.226118588719E-02, -8.226118588565E-02, -8.226118588565E-02, -8.226118588565E-02,
             -8.112847434388E-02, -8.112847434388E-02, -8.112847434388E-02, -8.112847432667E-02,
             -8.112847432667E-02, -8.112847432667E-02, -8.226118591076E-02, -8.226118591076E-02,
             -8.226118591076E-02, -8.226118610074E-02, -8.226118610074E-02, -8.226118610074E-02,
             -8.866359518786E-02, -8.866359518786E-02, -8.866359518786E-02, -8.866359488149E-02,
             -8.866359488149E-02, -8.866359488149E-02, -9.132287826645E-02, -9.132287826645E-02,
             -9.132287826645E-02, -9.132287826749E-02, -9.132287826749E-02, -9.132287826749E-02,
             -8.246762370216E-02, -8.246762370216E-02, -8.246762370216E-02, -8.246762359012E-02,
             -8.246762359012E-02, -8.246762359012E-02, -6.137786871067E-02, -6.137786871067E-02,
             -6.137786871067E-02, -6.137786929411E-02, -6.137786929411E-02, -6.137786929411E-02,
             -1.765322332327E-02, -1.765322332327E-02, -1.765322332327E-02, -1.765322408887E-02,
             -1.765322408887E-02, -1.765322408887E-02, 2.743644197732E-02, 2.743644197732E-02,
             2.743644197732E-02, 2.743644207970E-02, 2.743644207970E-02, 2.743644207970E-02,
             5.086584771446E-02, 5.086584771446E-02, 5.086584771446E-02, 5.086584762092E-02,
             5.086584762092E-02, 5.086584762092E-02, 6.830161070891E-02, 6.830161070891E-02,
             6.830161070891E-02, 6.830161041900E-02, 6.830161041900E-02, 6.830161041900E-02,
             6.830161070891E-02, 6.830161070891E-02, 6.830161070891E-02, 6.830161070891E-02,
             5.086584770959E-02, 2.743644203252E-02, -1.765322325215E-02, -6.137786871891E-02,
             -8.246762366243E-02, -9.132287839913E-02, -8.866359510841E-02, -8.226118585715E-02,
             -8.112847433875E-02, -8.226118538192E-02, -8.866359611420E-02, -9.132287799458E-02,
             -8.246762369299E-02, -6.137787102837E-02, -1.765322453880E-02, 2.743644217104E-02,
             5.086584767308E-02, 6.830161041900E-02, 6.830161041900E-02, 6.830161041900E-02,
             6.830161041900E-02, 6.830161070891E-02, 6.830161070891E-02, 6.830161070891E-02,
             6.830161070891E-02, 5.086584770959E-02, 2.743644203252E-02, -1.765322325215E-02,
             -6.137786871891E-02, -8.246762366243E-02, -9.132287839913E-02, -8.866359510841E-02,
             -8.226118585715E-02, -8.112847433875E-02, -8.226118538192E-02, -8.866359611420E-02,
             -9.132287799458E-02, -8.246762369299E-02, -6.137787102837E-02, -1.765322453880E-02,
             2.743644217104E-02, 5.086584767308E-02, 6.830161041900E-02, 6.830161041900E-02,
             6.830161041900E-02, 6.830161041900E-02, 6.830161070891E-02, 6.830161070891E-02,
             6.830161070891E-02, 6.830161070891E-02, 5.086584770959E-02, 2.743644203252E-02,
             -1.765322325215E-02, -6.137786871891E-02, -8.246762366243E-02, -9.132287839913E-02,
             -8.866359510841E-02, -8.226118585715E-02, -8.112847433875E-02, -8.226118538192E-02,
             -8.866359611420E-02, -9.132287799458E-02, -8.246762369299E-02, -6.137787102837E-02,
             -1.765322453880E-02, 2.743644217104E-02, 5.086584767308E-02, 6.830161041900E-02,
             6.830161041900E-02, 6.830161041900E-02, 6.830161041900E-02, 7.107762957585E-02,
             7.107762957585E-02, 7.107762957585E-02, 7.107762957585E-02, 4.258073037039E-02,
             1.078616362078E-02, -3.551989669155E-02, -7.588664280340E-02, -8.898818051497E-02,
             -8.866359515457E-02, -7.909406560943E-02, -6.792741701969E-02, -6.533539475257E-02,
             -6.792741701959E-02, -7.909406561032E-02, -8.866359515683E-02, -8.898818051545E-02,
             -7.588664280120E-02, -3.551989668918E-02, 1.078616362000E-02, 4.258073038520E-02,
             7.107762959368E-02, 7.107762959368E-02, 7.107762959368E-02, 7.107762959368E-02,
             7.107762957585E-02, 7.107762957585E-02, 7.107762957585E-02, 7.107762957585E-02,
             4.258073037039E-02, 1.078616362078E-02, -3.551989669155E-02, -7.588664280340E-02,
             -8.898818051497E-02, -8.866359515457E-02, -7.909406560943E-02, -6.792741701969E-02,
             -6.533539475257E-02, -6.792741701959E-02, -7.909406561032E-02, -8.866359515683E-02,
             -8.898818051545E-02, -7.588664280120E-02, -3.551989668918E-02, 1.078616362000E-02,
             4.258073038520E-02, 7.107762959368E-02, 7.107762959368E-02, 7.107762959368E-02,
             7.107762959368E-02, 7.107762957585E-02, 7.107762957585E-02, 7.107762957585E-02,
             7.107762957585E-02, 4.258073037039E-02, 1.078616362078E-02, -3.551989669155E-02,
             -7.588664280340E-02, -8.898818051497E-02, -8.866359515457E-02, -7.909406560943E-02,
             -6.792741701969E-02, -6.533539475257E-02, -6.792741701959E-02, -7.909406561032E-02,
             -8.866359515683E-02, -8.898818051545E-02, -7.588664280120E-02, -3.551989668918E-02,
             1.078616362000E-02, 4.258073038520E-02, 7.107762959368E-02, 7.107762959368E-02,
             7.107762959368E-02, 7.107762959368E-02, 7.107762957585E-02, 7.107762957585E-02,
             7.107762957585E-02, 7.107762959368E-02, 7.107762959368E-02, 7.107762959368E-02,
             4.258073036987E-02, 4.258073036987E-02, 4.258073036987E-02, 4.258073038418E-02,
             4.258073038418E-02, 4.258073038418E-02, 1.078616362392E-02, 1.078616362392E-02,
             1.078616362392E-02, 1.078616364259E-02, 1.078616364259E-02, 1.078616364259E-02,
             -3.551989669117E-02, -3.551989669117E-02, -3.551989669117E-02, -3.551989665447E-02,
             -3.551989665447E-02, -3.551989665447E-02, -7.588664279814E-02, -7.588664279814E-02,
             -7.588664279814E-02, -7.588664282337E-02, -7.588664282337E-02, -7.588664282337E-02,
             -8.898818051028E-02, -8.898818051028E-02, -8.898818051028E-02, -8.898818053148E-02,
             -8.898818053148E-02, -8.898818053148E-02, -8.866359512567E-02, -8.866359512567E-02,
             -8.866359512567E-02, -8.866359506293E-02, -8.866359506293E-02, -8.866359506293E-02,
             -7.909406557188E-02, -7.909406557188E-02, -7.909406557188E-02, -7.909406529352E-02,
             -7.909406529352E-02, -7.909406529352E-02, -6.792741701705E-02, -6.792741701705E-02,
             -6.792741701705E-02, -6.792741696493E-02, -6.792741696493E-02, -6.792741696493E-02,
             -6.533539474976E-02, -6.533539474976E-02, -6.533539474976E-02, -6.533539469123E-02,
             -6.533539469123E-02, -6.533539469123E-02, -6.792741704415E-02, -6.792741704415E-02,
             -6.792741704415E-02, -6.792741767251E-02, -6.792741767251E-02, -6.792741767251E-02,
             -7.909406564650E-02, -7.909406564650E-02, -7.909406564650E-02, -7.909406623786E-02,
             -7.909406623786E-02, -7.909406623786E-02, -8.866359519591E-02, -8.866359519591E-02,
             -8.866359519591E-02, -8.866359533166E-02, -8.866359533166E-02, -8.866359533166E-02,
             -8.898818055922E-02, -8.898818055922E-02, -8.898818055922E-02, -8.898818043000E-02,
             -8.898818043000E-02, -8.898818043000E-02, -7.588664281381E-02, -7.588664281381E-02,
             -7.588664281381E-02, -7.588664202382E-02, -7.588664202382E-02, -7.588664202382E-02,
             -3.551989670261E-02, -3.551989670261E-02, -3.551989670261E-02, -3.551989543206E-02,
             -3.551989543206E-02, -3.551989543206E-02, 1.078616365649E-02, 1.078616365649E-02,
             1.078616365649E-02, 1.078616370475E-02, 1.078616370475E-02, 1.078616370475E-02,
             4.258073042507E-02, 4.258073042507E-02, 4.258073042507E-02, 4.258073039986E-02,
             4.258073039986E-02, 4.258073039986E-02, 7.107762963804E-02, 7.107762963804E-02,
             7.107762963804E-02, 7.107762956389E-02, 7.107762956389E-02, 7.107762956389E-02,
             7.107762963804E-02, 7.107762963804E-02, 7.107762963804E-02, 7.107762963804E-02,
             4.258073041915E-02, 1.078616372574E-02, -3.551989658162E-02, -7.588664278863E-02,
             -8.898818050755E-02, -8.866359526279E-02, -7.909406553441E-02, -6.792741699678E-02,
             -6.533539474332E-02, -6.792741636103E-02, -7.909406642640E-02, -8.866359491969E-02,
             -8.898818047505E-02, -7.588664248991E-02, -3.551989681164E-02, 1.078616357365E-02,
             4.258073036306E-02, 7.107762956389E-02, 7.107762956389E-02, 7.107762956389E-02,
             7.107762956389E-02, 7.107762963804E-02, 7.107762963804E-02, 7.107762963804E-02,
             7.107762963804E-02, 4.258073041915E-02, 1.078616372574E-02, -3.551989658162E-02,
             -7.588664278863E-02, -8.898818050755E-02, -8.866359526279E-02, -7.909406553441E-02,
             -6.792741699678E-02, -6.533539474332E-02, -6.792741636103E-02, -7.909406642640E-02,
             -8.866359491969E-02, -8.898818047505E-02, -7.588664248991E-02, -3.551989681164E-02,
             1.078616357365E-02, 4.258073036306E-02, 7.107762956389E-02, 7.107762956389E-02,
             7.107762956389E-02, 7.107762956389E-02, 7.107762963804E-02, 7.107762963804E-02,
             7.107762963804E-02, 7.107762963804E-02, 4.258073041915E-02, 1.078616372574E-02,
             -3.551989658162E-02, -7.588664278863E-02, -8.898818050755E-02, -8.866359526279E-02,
             -7.909406553441E-02, -6.792741699678E-02, -6.533539474332E-02, -6.792741636103E-02,
             -7.909406642640E-02, -8.866359491969E-02, -8.898818047505E-02, -7.588664248991E-02,
             -3.551989681164E-02, 1.078616357365E-02, 4.258073036306E-02, 7.107762956389E-02,
             7.107762956389E-02, 7.107762956389E-02, 7.107762956389E-02, 7.076114333155E-02,
             7.076114333155E-02, 7.076114333155E-02, 7.076114333155E-02, 3.539309463577E-02,
             -1.191055431739E-03, -4.656675270908E-02, -8.287206522507E-02, -8.945863728305E-02,
             -8.226118588930E-02, -6.792741701965E-02, -5.367000806701E-02, -5.009611083921E-02,
             -5.367000806697E-02, -6.792741701966E-02, -8.226118588920E-02, -8.945863728263E-02,
             -8.287206522547E-02, -4.656675271188E-02, -1.191055427695E-03, 3.539309464677E-02,
             7.076114334392E-02, 7.076114334392E-02, 7.076114334392E-02, 7.076114334392E-02,
             7.076114333155E-02, 7.076114333155E-02, 7.076114333155E-02, 7.076114333155E-02,
             3.539309463577E-02, -1.191055431739E-03, -4.656675270908E-02, -8.287206522507E-02,
             -8.945863728305E-02, -8.226118588930E-02, -6.792741701965E-02, -5.367000806701E-02,
             -5.009611083921E-02, -5.367000806697E-02, -6.792741701966E-02, -8.226118588920E-02,
             -8.945863728263E-02, -8.287206522547E-02, -4.656675271188E-02, -1.191055427695E-03,
             3.539309464677E-02, 7.076114334392E-02, 7.076114334392E-02, 7.076114334392E-02,
             7.076114334392E-02, 7.076114333155E-02, 7.076114333155E-02, 7.076114333155E-02,
             7.076114333155E-02, 3.539309463577E-02, -1.191055431739E-03, -4.656675270908E-02,
             -8.287206522507E-02, -8.945863728305E-02, -8.226118588930E-02, -6.792741701965E-02,
             -5.367000806701E-02, -5.009611083921E-02, -5.367000806697E-02, -6.792741701966E-02,
             -8.226118588920E-02, -8.945863728263E-02, -8.287206522547E-02, -4.656675271188E-02,
             -1.191055427695E-03, 3.539309464677E-02, 7.076114334392E-02, 7.076114334392E-02,
             7.076114334392E-02, 7.076114334392E-02, 7.076114333155E-02, 7.076114333155E-02,
             7.076114333155E-02, 7.076114334392E-02, 7.076114334392E-02, 7.076114334392E-02,
             3.539309463623E-02, 3.539309463623E-02, 3.539309463623E-02, 3.539309464675E-02,
             3.539309464675E-02, 3.539309464675E-02, -1.191055426103E-03, -1.191055426103E-03,
             -1.191055426103E-03, -1.191055434331E-03, -1.191055434331E-03, -1.191055434331E-03,
             -4.656675270370E-02, -4.656675270370E-02, -4.656675270370E-02, -4.656675270835E-02,
             -4.656675270835E-02, -4.656675270835E-02, -8.287206522307E-02, -8.287206522307E-02,
             -8.287206522307E-02, -8.287206522398E-02, -8.287206522398E-02, -8.287206522398E-02,
             -8.945863728122E-02, -8.945863728122E-02, -8.945863728122E-02, -8.945863728537E-02,
             -8.945863728537E-02, -8.945863728537E-02, -8.226118588935E-02, -8.226118588935E-02,
             -8.226118588935E-02, -8.226118590476E-02, -8.226118590476E-02, -8.226118590476E-02,
             -6.792741701975E-02, -6.792741701975E-02, -6.792741701975E-02, -6.792741701951E-02,
             -6.792741701951E-02, -6.792741701951E-02, -5.367000806714E-02, -5.367000806714E-02,
             -5.367000806714E-02, -5.367000806791E-02, -5.367000806791E-02, -5.367000806791E-02,
             -5.009611083936E-02, -5.009611083936E-02, -5.009611083936E-02, -5.009611083977E-02,
             -5.009611083977E-02, -5.009611083977E-02, -5.367000805850E-02, -5.367000805850E-02,
             -5.367000805850E-02, -5.367000805741E-02, -5.367000805741E-02, -5.367000805741E-02,
             -6.792741700893E-02, -6.792741700893E-02, -6.792741700893E-02, -6.792741697727E-02,
             -6.792741697727E-02, -6.792741697727E-02, -8.226118588183E-02, -8.226118588183E-02,
             -8.226118588183E-02, -8.226118585934E-02, -8.226118585934E-02, -8.226118585934E-02,
             -8.945863727548E-02, -8.945863727548E-02, -8.945863727548E-02, -8.945863718850E-02,
             -8.945863718850E-02, -8.945863718850E-02, -8.287206521480E-02, -8.287206521480E-02,
             -8.287206521480E-02, -8.287206511917E-02, -8.287206511917E-02, -8.287206511917E-02,
             -4.656675269843E-02, -4.656675269843E-02, -4.656675269843E-02, -4.656675245442E-02,
             -4.656675245442E-02, -4.656675245442E-02, -1.191055457627E-03, -1.191055457627E-03,
             -1.191055457627E-03, -1.191055359107E-03, -1.191055359107E-03, -1.191055359107E-03,
             3.539309461328E-02, 3.539309461328E-02, 3.539309461328E-02, 3.539309476082E-02,
             3.539309476082E-02, 3.539309476082E-02, 7.076114330639E-02, 7.076114330639E-02,
             7.076114330639E-02, 7.076114350452E-02, 7.076114350452E-02, 7.076114350452E-02,
             7.076114330639E-02, 7.076114330639E-02, 7.076114330639E-02, 7.076114330639E-02,
             3.539309461416E-02, -1.191055472435E-03, -4.656675274384E-02, -8.287206521028E-02,
             -8.945863727164E-02, -8.226118592860E-02, -6.792741702540E-02, -5.367000806633E-02,
             -5.009611083900E-02, -5.367000787769E-02, -6.792741693307E-02, -8.226118594013E-02,
             -8.945863713601E-02, -8.287206546646E-02, -4.656675286616E-02, -1.191054996523E-03,
             3.539309494588E-02, 7.076114350452E-02, 7.076114350452E-02, 7.076114350452E-02,
             7.076114350452E-02, 7.076114330639E-02, 7.076114330639E-02, 7.076114330639E-02,
             7.076114330639E-02, 3.539309461416E-02, -1.191055472435E-03, -4.656675274384E-02,
             -8.287206521028E-02, -8.945863727164E-02, -8.226118592860E-02, -6.792741702540E-02,
             -5.367000806633E-02, -5.009611083900E-02, -5.367000787769E-02, -6.792741693307E-02,
             -8.226118594013E-02, -8.945863713601E-02, -8.287206546646E-02, -4.656675286616E-02,
             -1.191054996523E-03, 3.539309494588E-02, 7.076114350452E-02, 7.076114350452E-02,
             7.076114350452E-02, 7.076114350452E-02, 7.076114330639E-02, 7.076114330639E-02,
             7.076114330639E-02, 7.076114330639E-02, 3.539309461416E-02, -1.191055472435E-03,
             -4.656675274384E-02, -8.287206521028E-02, -8.945863727164E-02, -8.226118592860E-02,
             -6.792741702540E-02, -5.367000806633E-02, -5.009611083900E-02, -5.367000787769E-02,
             -6.792741693307E-02, -8.226118594013E-02, -8.945863713601E-02, -8.287206546646E-02,
             -4.656675286616E-02, -1.191054996523E-03, 3.539309494588E-02, 7.076114350452E-02,
             7.076114350452E-02, 7.076114350452E-02, 7.076114350452E-02, 7.106858491369E-02,
             7.106858491369E-02, 7.106858491369E-02, 7.106858491369E-02, 3.345900050499E-02,
             -4.769492603984E-03, -5.018094388034E-02, -8.555883978796E-02, -9.033251398505E-02,
             -8.112847434608E-02, -6.533539475251E-02, -5.009611083922E-02, -4.621706540085E-02,
             -5.009611083914E-02, -6.533539475250E-02, -8.112847434598E-02, -9.033251398465E-02,
             -8.555883978861E-02, -5.018094388386E-02, -4.769492599934E-03, 3.345900051698E-02,
             7.106858492727E-02, 7.106858492727E-02, 7.106858492727E-02, 7.106858492727E-02,
             7.106858491369E-02, 7.106858491369E-02, 7.106858491369E-02, 7.106858491369E-02,
             3.345900050499E-02, -4.769492603984E-03, -5.018094388034E-02, -8.555883978796E-02,
             -9.033251398505E-02, -8.112847434608E-02, -6.533539475251E-02, -5.009611083922E-02,
             -4.621706540085E-02, -5.009611083914E-02, -6.533539475250E-02, -8.112847434598E-02,
             -9.033251398465E-02, -8.555883978861E-02, -5.018094388386E-02, -4.769492599934E-03,
             3.345900051698E-02, 7.106858492727E-02, 7.106858492727E-02, 7.106858492727E-02,
             7.106858492727E-02, 7.106858491369E-02, 7.106858491369E-02, 7.106858491369E-02,
             7.106858491369E-02, 3.345900050499E-02, -4.769492603984E-03, -5.018094388034E-02,
             -8.555883978796E-02, -9.033251398505E-02, -8.112847434608E-02, -6.533539475251E-02,
             -5.009611083922E-02, -4.621706540085E-02, -5.009611083914E-02, -6.533539475250E-02,
             -8.112847434598E-02, -9.033251398465E-02, -8.555883978861E-02, -5.018094388386E-02,
             -4.769492599934E-03, 3.345900051698E-02, 7.106858492727E-02, 7.106858492727E-02,
             7.106858492727E-02, 7.106858492727E-02, 7.106858491369E-02, 7.106858491369E-02,
             7.106858491369E-02, 7.106858492727E-02, 7.106858492727E-02, 7.106858492727E-02,
             3.345900050538E-02, 3.345900050538E-02, 3.345900050538E-02, 3.345900051687E-02,
             3.345900051687E-02, 3.345900051687E-02, -4.769492597823E-03, -4.769492597823E-03,
             -4.769492597823E-03, -4.769492607739E-03, -4.769492607739E-03, -4.769492607739E-03,
             -5.018094387443E-02, -5.018094387443E-02, -5.018094387443E-02, -5.018094388130E-02,
             -5.018094388130E-02, -5.018094388130E-02, -8.555883978569E-02, -8.555883978569E-02,
             -8.555883978569E-02, -8.555883978442E-02, -8.555883978442E-02, -8.555883978442E-02,
             -9.033251398297E-02, -9.033251398297E-02, -9.033251398297E-02, -9.033251398022E-02,
             -9.033251398022E-02, -9.033251398022E-02, -8.112847434626E-02, -8.112847434626E-02,
             -8.112847434626E-02, -8.112847434816E-02, -8.112847434816E-02, -8.112847434816E-02,
             -6.533539475276E-02, -6.533539475276E-02, -6.533539475276E-02, -6.533539474662E-02,
             -6.533539474662E-02, -6.533539474662E-02, -5.009611083937E-02, -5.009611083937E-02,
             -5.009611083937E-02, -5.009611083941E-02, -5.009611083941E-02, -5.009611083941E-02,
             -4.621706540101E-02, -4.621706540101E-02, -4.621706540101E-02, -4.621706540094E-02,
             -4.621706540094E-02, -4.621706540094E-02, -5.009611082929E-02, -5.009611082929E-02,
             -5.009611082929E-02, -5.009611080182E-02, -5.009611080182E-02, -5.009611080182E-02,
             -6.533539473997E-02, -6.533539473997E-02, -6.533539473997E-02, -6.533539471568E-02,
             -6.533539471568E-02, -6.533539471568E-02, -8.112847433708E-02, -8.112847433708E-02,
             -8.112847433708E-02, -8.112847429455E-02, -8.112847429455E-02, -8.112847429455E-02,
             -9.033251397592E-02, -9.033251397592E-02, -9.033251397592E-02, -9.033251396099E-02,
             -9.033251396099E-02, -9.033251396099E-02, -8.555883977635E-02, -8.555883977635E-02,
             -8.555883977635E-02, -8.555883981273E-02, -8.555883981273E-02, -8.555883981273E-02,
             -5.018094386824E-02, -5.018094386824E-02, -5.018094386824E-02, -5.018094376223E-02,
             -5.018094376223E-02, -5.018094376223E-02, -4.769492631281E-03, -4.769492631281E-03,
             -4.769492631281E-03, -4.769491972703E-03, -4.769491972703E-03, -4.769491972703E-03,
             3.345900048198E-02, 3.345900048198E-02, 3.345900048198E-02, 3.345900106416E-02,
             3.345900106416E-02, 3.345900106416E-02, 7.106858488811E-02, 7.106858488811E-02,
             7.106858488811E-02, 7.106858541135E-02, 7.106858541135E-02, 7.106858541135E-02,
             7.106858488811E-02, 7.106858488811E-02, 7.106858488811E-02, 7.106858488811E-02,
             3.345900048299E-02, -4.769492654475E-03, -5.018094392328E-02, -8.555883977183E-02,
             -9.033251397179E-02, -8.112847437000E-02, -6.533539474706E-02, -5.009611083778E-02,
             -4.621706539994E-02, -5.009611072596E-02, -6.533539468936E-02, -8.112847439502E-02,
             -9.033251386109E-02, -8.555884016392E-02, -5.018094414022E-02, -4.769491717269E-03,
             3.345900118870E-02, 7.106858541135E-02, 7.106858541135E-02, 7.106858541135E-02,
             7.106858541135E-02, 7.106858488811E-02, 7.106858488811E-02, 7.106858488811E-02,
             7.106858488811E-02, 3.345900048299E-02, -4.769492654475E-03, -5.018094392328E-02,
             -8.555883977183E-02, -9.033251397179E-02, -8.112847437000E-02, -6.533539474706E-02,
             -5.009611083778E-02, -4.621706539994E-02, -5.009611072596E-02, -6.533539468936E-02,
             -8.112847439502E-02, -9.033251386109E-02, -8.555884016392E-02, -5.018094414022E-02,
             -4.769491717269E-03, 3.345900118870E-02, 7.106858541135E-02, 7.106858541135E-02,
             7.106858541135E-02, 7.106858541135E-02, 7.106858488811E-02, 7.106858488811E-02,
             7.106858488811E-02, 7.106858488811E-02, 3.345900048299E-02, -4.769492654475E-03,
             -5.018094392328E-02, -8.555883977183E-02, -9.033251397179E-02, -8.112847437000E-02,
             -6.533539474706E-02, -5.009611083778E-02, -4.621706539994E-02, -5.009611072596E-02,
             -6.533539468936E-02, -8.112847439502E-02, -9.033251386109E-02, -8.555884016392E-02,
             -5.018094414022E-02, -4.769491717269E-03, 3.345900118870E-02, 7.106858541135E-02,
             7.106858541135E-02, 7.106858541135E-02, 7.106858541135E-02, 7.076114327967E-02,
             7.076114327967E-02, 7.076114327967E-02, 7.076114327967E-02, 3.539309459055E-02,
             -1.191055490518E-03, -4.656675275385E-02, -8.287206522398E-02, -8.945863728256E-02,
             -8.226118588545E-02, -6.792741701812E-02, -5.367000806704E-02, -5.009611083919E-02,
             -5.367000806737E-02, -6.792741702094E-02, -8.226118588972E-02, -8.945863728364E-02,
             -8.287206517704E-02, -4.656675264442E-02, -1.191055423122E-03, 3.539309464836E-02,
             7.076114334134E-02, 7.076114334134E-02, 7.076114334134E-02, 7.076114334134E-02,
             7.076114327967E-02, 7.076114327967E-02, 7.076114327967E-02, 7.076114327967E-02,
             3.539309459055E-02, -1.191055490518E-03, -4.656675275385E-02, -8.287206522398E-02,
             -8.945863728256E-02, -8.226118588545E-02, -6.792741701812E-02, -5.367000806704E-02,
             -5.009611083919E-02, -5.367000806737E-02, -6.792741702094E-02, -8.226118588972E-02,
             -8.945863728364E-02, -8.287206517704E-02, -4.656675264442E-02, -1.191055423122E-03,
             3.539309464836E-02, 7.076114334134E-02, 7.076114334134E-02, 7.076114334134E-02,
             7.076114334134E-02, 7.076114327967E-02, 7.076114327967E-02, 7.076114327967E-02,
             7.076114327967E-02, 3.539309459055E-02, -1.191055490518E-03, -4.656675275385E-02,
             -8.287206522398E-02, -8.945863728256E-02, -8.226118588545E-02, -6.792741701812E-02,
             -5.367000806704E-02, -5.009611083919E-02, -5.367000806737E-02, -6.792741702094E-02,
             -8.226118588972E-02, -8.945863728364E-02, -8.287206517704E-02, -4.656675264442E-02,
             -1.191055423122E-03, 3.539309464836E-02, 7.076114334134E-02, 7.076114334134E-02,
             7.076114334134E-02, 7.076114334134E-02, 7.076114327967E-02, 7.076114327967E-02,
             7.076114327967E-02, 7.076114334134E-02, 7.076114334134E-02, 7.076114334134E-02,
             3.539309458953E-02, 3.539309458953E-02, 3.539309458953E-02, 3.539309464617E-02,
             3.539309464617E-02, 3.539309464617E-02, -1.191055391193E-03, -1.191055391193E-03,
             -1.191055391193E-03, -1.191055491039E-03, -1.191055491039E-03, -1.191055491039E-03,
             -4.656675266097E-02, -4.656675266097E-02, -4.656675266097E-02, -4.656675274504E-02,
             -4.656675274504E-02, -4.656675274504E-02, -8.287206520746E-02, -8.287206520746E-02,
             -8.287206520746E-02, -8.287206520850E-02, -8.287206520850E-02, -8.287206520850E-02,
             -8.945863726492E-02, -8.945863726492E-02, -8.945863726492E-02, -8.945863723985E-02,
             -8.945863723985E-02, -8.945863723985E-02, -8.226118586487E-02, -8.226118586487E-02,
             -8.226118586487E-02, -8.226118682526E-02, -8.226118682526E-02, -8.226118682526E-02,
             -6.792741699123E-02, -6.792741699123E-02, -6.792741699123E-02, -6.792741766209E-02,
             -6.792741766209E-02, -6.792741766209E-02, -5.367000805292E-02, -5.367000805292E-02,
             -5.367000805292E-02, -5.367000801775E-02, -5.367000801775E-02, -5.367000801775E-02,
             -5.009611082286E-02, -5.009611082286E-02, -5.009611082286E-02, -5.009611076800E-02,
             -5.009611076800E-02, -5.009611076800E-02, -5.367000796420E-02, -5.367000796420E-02,
             -5.367000796420E-02, -5.367000858515E-02, -5.367000858515E-02, -5.367000858515E-02,
             -6.792741690440E-02, -6.792741690440E-02, -6.792741690440E-02, -6.792741878248E-02,
             -6.792741878248E-02, -6.792741878248E-02, -8.226118589033E-02, -8.226118589033E-02,
             -8.226118589033E-02, -8.226118568527E-02, -8.226118568527E-02, -8.226118568527E-02,
             -8.945863727899E-02, -8.945863727899E-02, -8.945863727899E-02, -8.945863713552E-02,
             -8.945863713552E-02, -8.945863713552E-02, -8.287206516539E-02, -8.287206516539E-02,
             -8.287206516539E-02, -8.287206783633E-02, -8.287206783633E-02, -8.287206783633E-02,
             -4.656675265253E-02, -4.656675265253E-02, -4.656675265253E-02, -4.656675450616E-02,
             -4.656675450616E-02, -4.656675450616E-02, -1.191055378347E-03, -1.191055378347E-03,
             -1.191055378347E-03, -1.191056301131E-03, -1.191056301131E-03, -1.191056301131E-03,
             3.539309468222E-02, 3.539309468222E-02, 3.539309468222E-02, 3.539309440073E-02,
             3.539309440073E-02, 3.539309440073E-02, 7.076114336328E-02, 7.076114336328E-02,
             7.076114336328E-02, 7.076114374065E-02, 7.076114374065E-02, 7.076114374065E-02,
             7.076114336328E-02, 7.076114336328E-02, 7.076114336328E-02, 7.076114336328E-02,
             3.539309467735E-02, -1.191055422377E-03, -4.656675270080E-02, -8.287206510042E-02,
             -8.945863721591E-02, -8.226118601475E-02, -6.792741789612E-02, -5.367000818220E-02,
             -5.009611091747E-02, -5.366999939629E-02, -6.792741153585E-02, -8.226118522084E-02,
             -8.945863646811E-02, -8.287206018142E-02, -4.656674954932E-02, -1.191055250193E-03,
             3.539309498560E-02, 7.076114374065E-02, 7.076114374065E-02, 7.076114374065E-02,
             7.076114374065E-02, 7.076114336328E-02, 7.076114336328E-02, 7.076114336328E-02,
             7.076114336328E-02, 3.539309467735E-02, -1.191055422377E-03, -4.656675270080E-02,
             -8.287206510042E-02, -8.945863721591E-02, -8.226118601475E-02, -6.792741789612E-02,
             -5.367000818220E-02, -5.009611091747E-02, -5.366999939629E-02, -6.792741153585E-02,
             -8.226118522084E-02, -8.945863646811E-02, -8.287206018142E-02, -4.656674954932E-02,
             -1.191055250193E-03, 3.539309498560E-02, 7.076114374065E-02, 7.076114374065E-02,
             7.076114374065E-02, 7.076114374065E-02, 7.076114336328E-02, 7.076114336328E-02,
             7.076114336328E-02, 7.076114336328E-02, 3.539309467735E-02, -1.191055422377E-03,
             -4.656675270080E-02, -8.287206510042E-02, -8.945863721591E-02, -8.226118601475E-02,
             -6.792741789612E-02, -5.367000818220E-02, -5.009611091747E-02, -5.366999939629E-02,
             -6.792741153585E-02, -8.226118522084E-02, -8.945863646811E-02, -8.287206018142E-02,
             -4.656674954932E-02, -1.191055250193E-03, 3.539309498560E-02, 7.076114374065E-02,
             7.076114374065E-02, 7.076114374065E-02, 7.076114374065E-02, 7.107762952707E-02,
             7.107762952707E-02, 7.107762952707E-02, 7.107762952707E-02, 4.258073032606E-02,
             1.078616355921E-02, -3.551989673851E-02, -7.588664280273E-02, -8.898818051476E-02,
             -8.866359515248E-02, -7.909406560860E-02, -6.792741701974E-02, -6.533539475255E-02,
             -6.792741702051E-02, -7.909406561193E-02, -8.866359515732E-02, -8.898818051619E-02,
             -7.588664275250E-02, -3.551989662180E-02, 1.078616362776E-02, 4.258073039038E-02,
             7.107762959552E-02, 7.107762959552E-02, 7.107762959552E-02, 7.107762959552E-02,
             7.107762952707E-02, 7.107762952707E-02, 7.107762952707E-02, 7.107762952707E-02,
             4.258073032606E-02, 1.078616355921E-02, -3.551989673851E-02, -7.588664280273E-02,
             -8.898818051476E-02, -8.866359515248E-02, -7.909406560860E-02, -6.792741701974E-02,
             -6.533539475255E-02, -6.792741702051E-02, -7.909406561193E-02, -8.866359515732E-02,
             -8.898818051619E-02, -7.588664275250E-02, -3.551989662180E-02, 1.078616362776E-02,
             4.258073039038E-02, 7.107762959552E-02, 7.107762959552E-02, 7.107762959552E-02,
             7.107762959552E-02, 7.107762952707E-02, 7.107762952707E-02, 7.107762952707E-02,
             7.107762952707E-02, 4.258073032606E-02, 1.078616355921E-02, -3.551989673851E-02,
             -7.588664280273E-02, -8.898818051476E-02, -8.866359515248E-02, -7.909406560860E-02,
             -6.792741701974E-02, -6.533539475255E-02, -6.792741702051E-02, -7.909406561193E-02,
             -8.866359515732E-02, -8.898818051619E-02, -7.588664275250E-02, -3.551989662180E-02,
             1.078616362776E-02, 4.258073039038E-02, 7.107762959552E-02, 7.107762959552E-02,
             7.107762959552E-02, 7.107762959552E-02, 7.107762952707E-02, 7.107762952707E-02,
             7.107762952707E-02, 7.107762959552E-02, 7.107762959552E-02, 7.107762959552E-02,
             4.258073032494E-02, 4.258073032494E-02, 4.258073032494E-02, 4.258073038932E-02,
             4.258073038932E-02, 4.258073038932E-02, 1.078616366819E-02, 1.078616366819E-02,
             1.078616366819E-02, 1.078616354801E-02, 1.078616354801E-02, 1.078616354801E-02,
             -3.551989663612E-02, -3.551989663612E-02, -3.551989663612E-02, -3.551989668405E-02,
             -3.551989668405E-02, -3.551989668405E-02, -7.588664278208E-02, -7.588664278208E-02,
             -7.588664278208E-02, -7.588664279782E-02, -7.588664279782E-02, -7.588664279782E-02,
             -8.898818049297E-02, -8.898818049297E-02, -8.898818049297E-02, -8.898818045956E-02,
             -8.898818045956E-02, -8.898818045956E-02, -8.866359513742E-02, -8.866359513742E-02,
             -8.866359513742E-02, -8.866359604713E-02, -8.866359604713E-02, -8.866359604713E-02,
             -7.909406558590E-02, -7.909406558590E-02, -7.909406558590E-02, -7.909406633348E-02,
             -7.909406633348E-02, -7.909406633348E-02, -6.792741700353E-02, -6.792741700353E-02,
             -6.792741700353E-02, -6.792741704775E-02, -6.792741704775E-02, -6.792741704775E-02,
             -6.533539473369E-02, -6.533539473369E-02, -6.533539473369E-02, -6.533539472826E-02,
             -6.533539472826E-02, -6.533539472826E-02, -6.792741691959E-02, -6.792741691959E-02,
             -6.792741691959E-02, -6.792741751591E-02, -6.792741751591E-02, -6.792741751591E-02,
             -7.909406549774E-02, -7.909406549774E-02, -7.909406549774E-02, -7.909406525661E-02,
             -7.909406525661E-02, -7.909406525661E-02, -8.866359515914E-02, -8.866359515914E-02,
             -8.866359515914E-02, -8.866359473151E-02, -8.866359473151E-02, -8.866359473151E-02,
             -8.898818051207E-02, -8.898818051207E-02, -8.898818051207E-02, -8.898817981858E-02,
             -8.898817981858E-02, -8.898817981858E-02, -7.588664267248E-02, -7.588664267248E-02,
             -7.588664267248E-02, -7.588664641701E-02, -7.588664641701E-02, -7.588664641701E-02,
             -3.551989655701E-02, -3.551989655701E-02, -3.551989655701E-02, -3.551989857074E-02,
             -3.551989857074E-02, -3.551989857074E-02, 1.078616367543E-02, 1.078616367543E-02,
             1.078616367543E-02, 1.078616599417E-02, 1.078616599417E-02, 1.078616599417E-02,
             4.258073042625E-02, 4.258073042625E-02, 4.258073042625E-02, 4.258073218394E-02,
             4.258073218394E-02, 4.258073218394E-02, 7.107762961917E-02, 7.107762961917E-02,
             7.107762961917E-02, 7.107763079746E-02, 7.107763079746E-02, 7.107763079746E-02,
             7.107762961917E-02, 7.107762961917E-02, 7.107762961917E-02, 7.107762961917E-02,
             4.258073042082E-02, 1.078616364622E-02, -3.551989669348E-02, -7.588664257456E-02,
             -8.898818040620E-02, -8.866359551230E-02, -7.909406639700E-02, -6.792741709037E-02,
             -6.533539484262E-02, -6.792741470494E-02, -7.909406425265E-02, -8.866359428676E-02,
             -8.898817979424E-02, -7.588664966370E-02, -3.551990279207E-02, 1.078616382979E-02,
             4.258073111114E-02, 7.107763079746E-02, 7.107763079746E-02, 7.107763079746E-02,
             7.107763079746E-02, 7.107762961917E-02, 7.107762961917E-02, 7.107762961917E-02,
             7.107762961917E-02, 4.258073042082E-02, 1.078616364622E-02, -3.551989669348E-02,
             -7.588664257456E-02, -8.898818040620E-02, -8.866359551230E-02, -7.909406639700E-02,
             -6.792741709037E-02, -6.533539484262E-02, -6.792741470494E-02, -7.909406425265E-02,
             -8.866359428676E-02, -8.898817979424E-02, -7.588664966370E-02, -3.551990279207E-02,
             1.078616382979E-02, 4.258073111114E-02, 7.107763079746E-02, 7.107763079746E-02,
             7.107763079746E-02, 7.107763079746E-02, 7.107762961917E-02, 7.107762961917E-02,
             7.107762961917E-02, 7.107762961917E-02, 4.258073042082E-02, 1.078616364622E-02,
             -3.551989669348E-02, -7.588664257456E-02, -8.898818040620E-02, -8.866359551230E-02,
             -7.909406639700E-02, -6.792741709037E-02, -6.533539484262E-02, -6.792741470494E-02,
             -7.909406425265E-02, -8.866359428676E-02, -8.898817979424E-02, -7.588664966370E-02,
             -3.551990279207E-02, 1.078616382979E-02, 4.258073111114E-02, 7.107763079746E-02,
             7.107763079746E-02, 7.107763079746E-02, 7.107763079746E-02, 6.830161081549E-02,
             6.830161081549E-02, 6.830161081549E-02, 6.830161081549E-02, 5.086584780000E-02,
             2.743644198792E-02, -1.765322330113E-02, -6.137786874307E-02, -8.246762368037E-02,
             -9.132287823633E-02, -8.866359515807E-02, -8.226118588995E-02, -8.112847434679E-02,
             -8.226118588649E-02, -8.866359515401E-02, -9.132287824817E-02, -8.246762368799E-02,
             -6.137786865325E-02, -1.765322324576E-02, 2.743644191464E-02, 5.086584765051E-02,
             6.830161063722E-02, 6.830161063722E-02, 6.830161063722E-02, 6.830161063722E-02,
             6.830161081549E-02, 6.830161081549E-02, 6.830161081549E-02, 6.830161081549E-02,
             5.086584780000E-02, 2.743644198792E-02, -1.765322330113E-02, -6.137786874307E-02,
             -8.246762368037E-02, -9.132287823633E-02, -8.866359515807E-02, -8.226118588995E-02,
             -8.112847434679E-02, -8.226118588649E-02, -8.866359515401E-02, -9.132287824817E-02,
             -8.246762368799E-02, -6.137786865325E-02, -1.765322324576E-02, 2.743644191464E-02,
             5.086584765051E-02, 6.830161063722E-02, 6.830161063722E-02, 6.830161063722E-02,
             6.830161063722E-02, 6.830161081549E-02, 6.830161081549E-02, 6.830161081549E-02,
             6.830161081549E-02, 5.086584780000E-02, 2.743644198792E-02, -1.765322330113E-02,
             -6.137786874307E-02, -8.246762368037E-02, -9.132287823633E-02, -8.866359515807E-02,
             -8.226118588995E-02, -8.112847434679E-02, -8.226118588649E-02, -8.866359515401E-02,
             -9.132287824817E-02, -8.246762368799E-02, -6.137786865325E-02, -1.765322324576E-02,
             2.743644191464E-02, 5.086584765051E-02, 6.830161063722E-02, 6.830161063722E-02,
             6.830161063722E-02, 6.830161063722E-02, 6.830161081549E-02, 6.830161081549E-02,
             6.830161081549E-02, 6.830161063722E-02, 6.830161063722E-02, 6.830161063722E-02,
             5.086584780757E-02, 5.086584780757E-02, 5.086584780757E-02, 5.086584766311E-02,
             5.086584766311E-02, 5.086584766311E-02, 2.743644197273E-02, 2.743644197273E-02,
             2.743644197273E-02, 2.743644186421E-02, 2.743644186421E-02, 2.743644186421E-02,
             -1.765322331710E-02, -1.765322331710E-02, -1.765322331710E-02, -1.765322349251E-02,
             -1.765322349251E-02, -1.765322349251E-02, -6.137786874012E-02, -6.137786874012E-02,
             -6.137786874012E-02, -6.137786871753E-02, -6.137786871753E-02, -6.137786871753E-02,
             -8.246762366171E-02, -8.246762366171E-02, -8.246762366171E-02, -8.246762372152E-02,
             -8.246762372152E-02, -8.246762372152E-02, -9.132287823311E-02, -9.132287823311E-02,
             -9.132287823311E-02, -9.132287825277E-02, -9.132287825277E-02, -9.132287825277E-02,
             -8.866359515060E-02, -8.866359515060E-02, -8.866359515060E-02, -8.866359531844E-02,
             -8.866359531844E-02, -8.866359531844E-02, -8.226118589492E-02, -8.226118589492E-02,
             -8.226118589492E-02, -8.226118583517E-02, -8.226118583517E-02, -8.226118583517E-02,
             -8.112847435078E-02, -8.112847435078E-02, -8.112847435078E-02, -8.112847422282E-02,
             -8.112847422282E-02, -8.112847422282E-02, -8.226118583541E-02, -8.226118583541E-02,
             -8.226118583541E-02, -8.226118553065E-02, -8.226118553065E-02, -8.226118553065E-02,
             -8.866359509720E-02, -8.866359509720E-02, -8.866359509720E-02, -8.866359507022E-02,
             -8.866359507022E-02, -8.866359507022E-02, -9.132287810966E-02, -9.132287810966E-02,
             -9.132287810966E-02, -9.132287877269E-02, -9.132287877269E-02, -9.132287877269E-02,
             -8.246762354326E-02, -8.246762354326E-02, -8.246762354326E-02, -8.246762354807E-02,
             -8.246762354807E-02, -8.246762354807E-02, -6.137786876510E-02, -6.137786876510E-02,
             -6.137786876510E-02, -6.137786771557E-02, -6.137786771557E-02, -6.137786771557E-02,
             -1.765322338911E-02, -1.765322338911E-02, -1.765322338911E-02, -1.765322087138E-02,
             -1.765322087138E-02, -1.765322087138E-02, 2.743644247553E-02, 2.743644247553E-02,
             2.743644247553E-02, 2.743644022599E-02, 2.743644022599E-02, 2.743644022599E-02,
             5.086584820876E-02, 5.086584820876E-02, 5.086584820876E-02, 5.086585615402E-02,
             5.086585615402E-02, 5.086585615402E-02, 6.830161119920E-02, 6.830161119920E-02,
             6.830161119920E-02, 6.830162935898E-02, 6.830162935898E-02, 6.830162935898E-02,
             6.830161119920E-02, 6.830161119920E-02, 6.830161119920E-02, 6.830161119920E-02,
             5.086584815401E-02, 2.743644231431E-02, -1.765322292982E-02, -6.137786849815E-02,
             -8.246762338822E-02, -9.132287734143E-02, -8.866359466284E-02, -8.226118594735E-02,
             -8.112847450493E-02, -8.226118641415E-02, -8.866359482038E-02, -9.132287314228E-02,
             -8.246762394929E-02, -6.137785306170E-02, -1.765321357953E-02, 2.743644447193E-02,
             5.086585812618E-02, 6.830162935898E-02, 6.830162935898E-02, 6.830162935898E-02,
             6.830162935898E-02, 6.830161119920E-02, 6.830161119920E-02, 6.830161119920E-02,
             6.830161119920E-02, 5.086584815401E-02, 2.743644231431E-02, -1.765322292982E-02,
             -6.137786849815E-02, -8.246762338822E-02, -9.132287734143E-02, -8.866359466284E-02,
             -8.226118594735E-02, -8.112847450493E-02, -8.226118641415E-02, -8.866359482038E-02,
             -9.132287314228E-02, -8.246762394929E-02, -6.137785306170E-02, -1.765321357953E-02,
             2.743644447193E-02, 5.086585812618E-02, 6.830162935898E-02, 6.830162935898E-02,
             6.830162935898E-02, 6.830162935898E-02, 6.830161119920E-02, 6.830161119920E-02,
             6.830161119920E-02, 6.830161119920E-02, 5.086584815401E-02, 2.743644231431E-02,
             -1.765322292982E-02, -6.137786849815E-02, -8.246762338822E-02, -9.132287734143E-02,
             -8.866359466284E-02, -8.226118594735E-02, -8.112847450493E-02, -8.226118641415E-02,
             -8.866359482038E-02, -9.132287314228E-02, -8.246762394929E-02, -6.137785306170E-02,
             -1.765321357953E-02, 2.743644447193E-02, 5.086585812618E-02, 6.830162935898E-02,
             6.830162935898E-02, 6.830162935898E-02, 6.830162935898E-02, 5.936207330148E-02,
             5.936207330148E-02, 5.936207330148E-02, 5.936207330148E-02, 5.828995998104E-02,
             4.830380230651E-02, 9.452397771605E-03, -3.406682304527E-02, -6.297844908817E-02,
             -8.246762366819E-02, -8.898818051669E-02, -8.945863728315E-02, -9.033251398530E-02,
             -8.945863727988E-02, -8.898818051232E-02, -8.246762368203E-02, -6.297844909684E-02,
             -3.406682294525E-02, 9.452397831150E-03, 4.830380222126E-02, 5.828995982090E-02,
             5.936207311212E-02, 5.936207311212E-02, 5.936207311212E-02, 5.936207311212E-02,
             5.936207330148E-02, 5.936207330148E-02, 5.936207330148E-02, 5.936207330148E-02,
             5.828995998104E-02, 4.830380230651E-02, 9.452397771605E-03, -3.406682304527E-02,
             -6.297844908817E-02, -8.246762366819E-02, -8.898818051669E-02, -8.945863728315E-02,
             -9.033251398530E-02, -8.945863727988E-02, -8.898818051232E-02, -8.246762368203E-02,
             -6.297844909684E-02, -3.406682294525E-02, 9.452397831150E-03, 4.830380222126E-02,
             5.828995982090E-02, 5.936207311212E-02, 5.936207311212E-02, 5.936207311212E-02,
             5.936207311212E-02, 5.936207330148E-02, 5.936207330148E-02, 5.936207330148E-02,
             5.936207330148E-02, 5.828995998104E-02, 4.830380230651E-02, 9.452397771605E-03,
             -3.406682304527E-02, -6.297844908817E-02, -8.246762366819E-02, -8.898818051669E-02,
             -8.945863728315E-02, -9.033251398530E-02, -8.945863727988E-02, -8.898818051232E-02,
             -8.246762368203E-02, -6.297844909684E-02, -3.406682294525E-02, 9.452397831150E-03,
             4.830380222126E-02, 5.828995982090E-02, 5.936207311212E-02, 5.936207311212E-02,
             5.936207311212E-02, 5.936207311212E-02, 5.936207330148E-02, 5.936207330148E-02,
             5.936207330148E-02, 5.936207311212E-02, 5.936207311212E-02, 5.936207311212E-02,
             5.828995998872E-02, 5.828995998872E-02, 5.828995998872E-02, 5.828995983823E-02,
             5.828995983823E-02, 5.828995983823E-02, 4.830380228315E-02, 4.830380228315E-02,
             4.830380228315E-02, 4.830380212788E-02, 4.830380212788E-02, 4.830380212788E-02,
             9.452397747834E-03, 9.452397747834E-03, 9.452397747834E-03, 9.452397581382E-03,
             9.452397581382E-03, 9.452397581382E-03, -3.406682304192E-02, -3.406682304192E-02,
             -3.406682304192E-02, -3.406682308156E-02, -3.406682308156E-02, -3.406682308156E-02,
             -6.297844906661E-02, -6.297844906661E-02, -6.297844906661E-02, -6.297844911074E-02,
             -6.297844911074E-02, -6.297844911074E-02, -8.246762366501E-02, -8.246762366501E-02,
             -8.246762366501E-02, -8.246762419398E-02, -8.246762419398E-02, -8.246762419398E-02,
             -8.898818050847E-02, -8.898818050847E-02, -8.898818050847E-02, -8.898818095707E-02,
             -8.898818095707E-02, -8.898818095707E-02, -8.945863728912E-02, -8.945863728912E-02,
             -8.945863728912E-02, -8.945863701552E-02, -8.945863701552E-02, -8.945863701552E-02,
             -9.033251399012E-02, -9.033251399012E-02, -9.033251399012E-02, -9.033251357911E-02,
             -9.033251357911E-02, -9.033251357911E-02, -8.945863722023E-02, -8.945863722023E-02,
             -8.945863722023E-02, -8.945863692665E-02, -8.945863692665E-02, -8.945863692665E-02,
             -8.898818044544E-02, -8.898818044544E-02, -8.898818044544E-02, -8.898817976496E-02,
             -8.898817976496E-02, -8.898817976496E-02, -8.246762350893E-02, -8.246762350893E-02,
             -8.246762350893E-02, -8.246762756993E-02, -8.246762756993E-02, -8.246762756993E-02,
             -6.297844891506E-02, -6.297844891506E-02, -6.297844891506E-02, -6.297844890302E-02,
             -6.297844890302E-02, -6.297844890302E-02, -3.406682305233E-02, -3.406682305233E-02,
             -3.406682305233E-02, -3.406682641371E-02, -3.406682641371E-02, -3.406682641371E-02,
             9.452397692782E-03, 9.452397692782E-03, 9.452397692782E-03, 9.452397880646E-03,
             9.452397880646E-03, 9.452397880646E-03, 4.830380281679E-02, 4.830380281679E-02,
             4.830380281679E-02, 4.830380063842E-02, 4.830380063842E-02, 4.830380063842E-02,
             5.828996040645E-02, 5.828996040645E-02, 5.828996040645E-02, 5.828996133177E-02,
             5.828996133177E-02, 5.828996133177E-02, 5.936207369694E-02, 5.936207369694E-02,
             5.936207369694E-02, 5.936207728870E-02, 5.936207728870E-02, 5.936207728870E-02,
             5.936207369694E-02, 5.936207369694E-02, 5.936207369694E-02, 5.936207369694E-02,
             5.828996035238E-02, 4.830380268334E-02, 9.452398217996E-03, -3.406682282091E-02,
             -6.297844889354E-02, -8.246762288555E-02, -8.898817994782E-02, -8.945863745921E-02,
             -9.033251415162E-02, -8.945863816110E-02, -8.898818085115E-02, -8.246761457342E-02,
             -6.297844986249E-02, -3.406680946439E-02, 9.452404246664E-03, 4.830381786080E-02,
             5.828996939029E-02, 5.936207728870E-02, 5.936207728870E-02, 5.936207728870E-02,
             5.936207728870E-02, 5.936207369694E-02, 5.936207369694E-02, 5.936207369694E-02,
             5.936207369694E-02, 5.828996035238E-02, 4.830380268334E-02, 9.452398217996E-03,
             -3.406682282091E-02, -6.297844889354E-02, -8.246762288555E-02, -8.898817994782E-02,
             -8.945863745921E-02, -9.033251415162E-02, -8.945863816110E-02, -8.898818085115E-02,
             -8.246761457342E-02, -6.297844986249E-02, -3.406680946439E-02, 9.452404246664E-03,
             4.830381786080E-02, 5.828996939029E-02, 5.936207728870E-02, 5.936207728870E-02,
             5.936207728870E-02, 5.936207728870E-02, 5.936207369694E-02, 5.936207369694E-02,
             5.936207369694E-02, 5.936207369694E-02, 5.828996035238E-02, 4.830380268334E-02,
             9.452398217996E-03, -3.406682282091E-02, -6.297844889354E-02, -8.246762288555E-02,
             -8.898817994782E-02, -8.945863745921E-02, -9.033251415162E-02, -8.945863816110E-02,
             -8.898818085115E-02, -8.246761457342E-02, -6.297844986249E-02, -3.406680946439E-02,
             9.452404246664E-03, 4.830381786080E-02, 5.828996939029E-02, 5.936207728870E-02,
             5.936207728870E-02, 5.936207728870E-02, 5.936207728870E-02, 4.356156947921E-02,
             4.356156947921E-02, 4.356156947921E-02, 4.356156947921E-02, 5.964570029223E-02,
             6.482091611632E-02, 3.690531534624E-02, -1.191055458537E-03, -3.406682303796E-02,
             -6.137786872364E-02, -7.588664280441E-02, -8.287206522439E-02, -8.555883978733E-02,
             -8.287206522464E-02, -7.588664280220E-02, -6.137786872109E-02, -3.406682302198E-02,
             -1.191055381108E-03, 3.690531543776E-02, 6.482091623871E-02, 5.964570020857E-02,
             4.356156936429E-02, 4.356156936429E-02, 4.356156936429E-02, 4.356156936429E-02,
             4.356156947921E-02, 4.356156947921E-02, 4.356156947921E-02, 4.356156947921E-02,
             5.964570029223E-02, 6.482091611632E-02, 3.690531534624E-02, -1.191055458537E-03,
             -3.406682303796E-02, -6.137786872364E-02, -7.588664280441E-02, -8.287206522439E-02,
             -8.555883978733E-02, -8.287206522464E-02, -7.588664280220E-02, -6.137786872109E-02,
             -3.406682302198E-02, -1.191055381108E-03, 3.690531543776E-02, 6.482091623871E-02,
             5.964570020857E-02, 4.356156936429E-02, 4.356156936429E-02, 4.356156936429E-02,
             4.356156936429E-02, 4.356156947921E-02, 4.356156947921E-02, 4.356156947921E-02,
             4.356156947921E-02, 5.964570029223E-02, 6.482091611632E-02, 3.690531534624E-02,
             -1.191055458537E-03, -3.406682303796E-02, -6.137786872364E-02, -7.588664280441E-02,
             -8.287206522439E-02, -8.555883978733E-02, -8.287206522464E-02, -7.588664280220E-02,
             -6.137786872109E-02, -3.406682302198E-02, -1.191055381108E-03, 3.690531543776E-02,
             6.482091623871E-02, 5.964570020857E-02, 4.356156936429E-02, 4.356156936429E-02,
             4.356156936429E-02, 4.356156936429E-02, 4.356156947921E-02, 4.356156947921E-02,
             4.356156947921E-02, 4.356156936429E-02, 4.356156936429E-02, 4.356156936429E-02,
             5.964570030717E-02, 5.964570030717E-02, 5.964570030717E-02, 5.964570021275E-02,
             5.964570021275E-02, 5.964570021275E-02, 6.482091618079E-02, 6.482091618079E-02,
             6.482091618079E-02, 6.482091625484E-02, 6.482091625484E-02, 6.482091625484E-02,
             3.690531539853E-02, 3.690531539853E-02, 3.690531539853E-02, 3.690531539384E-02,
             3.690531539384E-02, 3.690531539384E-02, -1.191055456349E-03, -1.191055456349E-03,
             -1.191055456349E-03, -1.191055382820E-03, -1.191055382820E-03, -1.191055382820E-03,
             -3.406682306918E-02, -3.406682306918E-02, -3.406682306918E-02, -3.406682271866E-02,
             -3.406682271866E-02, -3.406682271866E-02, -6.137786877170E-02, -6.137786877170E-02,
             -6.137786877170E-02, -6.137786853047E-02, -6.137786853047E-02, -6.137786853047E-02,
             -7.588664284592E-02, -7.588664284592E-02, -7.588664284592E-02, -7.588664167337E-02,
             -7.588664167337E-02, -7.588664167337E-02, -8.287206525245E-02, -8.287206525245E-02,
             -8.287206525245E-02, -8.287206525998E-02, -8.287206525998E-02, -8.287206525998E-02,
             -8.555883982043E-02, -8.555883982043E-02, -8.555883982043E-02, -8.555883993984E-02,
             -8.555883993984E-02, -8.555883993984E-02, -8.287206435891E-02, -8.287206435891E-02,
             -8.287206435891E-02, -8.287206781885E-02, -8.287206781885E-02, -8.287206781885E-02,
             -7.588664184552E-02, -7.588664184552E-02, -7.588664184552E-02, -7.588664537391E-02,
             -7.588664537391E-02, -7.588664537391E-02, -6.137786899400E-02, -6.137786899400E-02,
             -6.137786899400E-02, -6.137786859079E-02, -6.137786859079E-02, -6.137786859079E-02,
             -3.406682335224E-02, -3.406682335224E-02, -3.406682335224E-02, -3.406682151311E-02,
             -3.406682151311E-02, -3.406682151311E-02, -1.191056543525E-03, -1.191056543525E-03,
             -1.191056543525E-03, -1.191054171399E-03, -1.191054171399E-03, -1.191054171399E-03,
             3.690531409966E-02, 3.690531409966E-02, 3.690531409966E-02, 3.690530649954E-02,
             3.690530649954E-02, 3.690530649954E-02, 6.482091690874E-02, 6.482091690874E-02,
             6.482091690874E-02, 6.482091531714E-02, 6.482091531714E-02, 6.482091531714E-02,
             5.964570095618E-02, 5.964570095618E-02, 5.964570095618E-02, 5.964569570642E-02,
             5.964569570642E-02, 5.964569570642E-02, 4.356157019507E-02, 4.356157019507E-02,
             4.356157019507E-02, 4.356156131450E-02, 4.356156131450E-02, 4.356156131450E-02,
             4.356157019507E-02, 4.356157019507E-02, 4.356157019507E-02, 4.356157019507E-02,
             5.964570094620E-02, 6.482091649327E-02, 3.690531571911E-02, -1.191055739973E-03,
             -3.406682287209E-02, -6.137786740330E-02, -7.588664153191E-02, -8.287206467804E-02,
             -8.555883937802E-02, -8.287207217151E-02, -7.588664926515E-02, -6.137786986632E-02,
             -3.406682848978E-02, -1.191045622825E-03, 3.690531996259E-02, 6.482088800778E-02,
             5.964568187328E-02, 4.356156131450E-02, 4.356156131450E-02, 4.356156131450E-02,
             4.356156131450E-02, 4.356157019507E-02, 4.356157019507E-02, 4.356157019507E-02,
             4.356157019507E-02, 5.964570094620E-02, 6.482091649327E-02, 3.690531571911E-02,
             -1.191055739973E-03, -3.406682287209E-02, -6.137786740330E-02, -7.588664153191E-02,
             -8.287206467804E-02, -8.555883937802E-02, -8.287207217151E-02, -7.588664926515E-02,
             -6.137786986632E-02, -3.406682848978E-02, -1.191045622825E-03, 3.690531996259E-02,
             6.482088800778E-02, 5.964568187328E-02, 4.356156131450E-02, 4.356156131450E-02,
             4.356156131450E-02, 4.356156131450E-02, 4.356157019507E-02, 4.356157019507E-02,
             4.356157019507E-02, 4.356157019507E-02, 5.964570094620E-02, 6.482091649327E-02,
             3.690531571911E-02, -1.191055739973E-03, -3.406682287209E-02, -6.137786740330E-02,
             -7.588664153191E-02, -8.287206467804E-02, -8.555883937802E-02, -8.287207217151E-02,
             -7.588664926515E-02, -6.137786986632E-02, -3.406682848978E-02, -1.191045622825E-03,
             3.690531996259E-02, 6.482088800778E-02, 5.964568187328E-02, 4.356156131450E-02,
             4.356156131450E-02, 4.356156131450E-02, 4.356156131450E-02, 1.607440723152E-02,
             1.607440723152E-02, 1.607440723152E-02, 1.607440723152E-02, 4.759923193698E-02,
             6.859721738735E-02, 5.944385069905E-02, 3.690531535485E-02, 9.452397711389E-03,
             -1.765322333963E-02, -3.551989669471E-02, -4.656675271059E-02, -5.018094388230E-02,
             -4.656675270993E-02, -3.551989669211E-02, -1.765322333643E-02, 9.452397727634E-03,
             3.690531543376E-02, 5.944385079660E-02, 6.859721752374E-02, 4.759923185274E-02,
             1.607440711543E-02, 1.607440711543E-02, 1.607440711543E-02, 1.607440711543E-02,
             1.607440723152E-02, 1.607440723152E-02, 1.607440723152E-02, 1.607440723152E-02,
             4.759923193698E-02, 6.859721738735E-02, 5.944385069905E-02, 3.690531535485E-02,
             9.452397711389E-03, -1.765322333963E-02, -3.551989669471E-02, -4.656675271059E-02,
             -5.018094388230E-02, -4.656675270993E-02, -3.551989669211E-02, -1.765322333643E-02,
             9.452397727634E-03, 3.690531543376E-02, 5.944385079660E-02, 6.859721752374E-02,
             4.759923185274E-02, 1.607440711543E-02, 1.607440711543E-02, 1.607440711543E-02,
             1.607440711543E-02, 1.607440723152E-02, 1.607440723152E-02, 1.607440723152E-02,
             1.607440723152E-02, 4.759923193698E-02, 6.859721738735E-02, 5.944385069905E-02,
             3.690531535485E-02, 9.452397711389E-03, -1.765322333963E-02, -3.551989669471E-02,
             -4.656675271059E-02, -5.018094388230E-02, -4.656675270993E-02, -3.551989669211E-02,
             -1.765322333643E-02, 9.452397727634E-03, 3.690531543376E-02, 5.944385079660E-02,
             6.859721752374E-02, 4.759923185274E-02, 1.607440711543E-02, 1.607440711543E-02,
             1.607440711543E-02, 1.607440711543E-02, 1.607440723152E-02, 1.607440723152E-02,
             1.607440723152E-02, 1.607440711543E-02, 1.607440711543E-02, 1.607440711543E-02,
             4.759923195514E-02, 4.759923195514E-02, 4.759923195514E-02, 4.759923185885E-02,
             4.759923185885E-02, 4.759923185885E-02, 6.859721745932E-02, 6.859721745932E-02,
             6.859721745932E-02, 6.859721752595E-02, 6.859721752595E-02, 6.859721752595E-02,
             5.944385075744E-02, 5.944385075744E-02, 5.944385075744E-02, 5.944385080089E-02,
             5.944385080089E-02, 5.944385080089E-02, 3.690531536127E-02, 3.690531536127E-02,
             3.690531536127E-02, 3.690531548784E-02, 3.690531548784E-02, 3.690531548784E-02,
             9.452397681259E-03, 9.452397681259E-03, 9.452397681259E-03, 9.452398433384E-03,
             9.452398433384E-03, 9.452398433384E-03, -1.765322339372E-02, -1.765322339372E-02,
             -1.765322339372E-02, -1.765322141608E-02, -1.765322141608E-02, -1.765322141608E-02,
             -3.551989674226E-02, -3.551989674226E-02, -3.551989674226E-02, -3.551989417886E-02,
             -3.551989417886E-02, -3.551989417886E-02, -4.656675274409E-02, -4.656675274409E-02,
             -4.656675274409E-02, -4.656675289344E-02, -4.656675289344E-02, -4.656675289344E-02,
             -5.018094392162E-02, -5.018094392162E-02, -5.018094392162E-02, -5.018094403008E-02,
             -5.018094403008E-02, -5.018094403008E-02, -4.656675175498E-02, -4.656675175498E-02,
             -4.656675175498E-02, -4.656675623028E-02, -4.656675623028E-02, -4.656675623028E-02,
             -3.551989563692E-02, -3.551989563692E-02, -3.551989563692E-02, -3.551990002745E-02,
             -3.551990002745E-02, -3.551990002745E-02, -1.765322363308E-02, -1.765322363308E-02,
             -1.765322363308E-02, -1.765322550140E-02, -1.765322550140E-02, -1.765322550140E-02,
             9.452397370380E-03, 9.452397370380E-03, 9.452397370380E-03, 9.452399615189E-03,
             9.452399615189E-03, 9.452399615189E-03, 3.690531414060E-02, 3.690531414060E-02,
             3.690531414060E-02, 3.690531355910E-02, 3.690531355910E-02, 3.690531355910E-02,
             5.944384930771E-02, 5.944384930771E-02, 5.944384930771E-02, 5.944384180685E-02,
             5.944384180685E-02, 5.944384180685E-02, 6.859721817973E-02, 6.859721817973E-02,
             6.859721817973E-02, 6.859721667592E-02, 6.859721667592E-02, 6.859721667592E-02,
             4.759923258023E-02, 4.759923258023E-02, 4.759923258023E-02, 4.759923974836E-02,
             4.759923974836E-02, 4.759923974836E-02, 1.607440792085E-02, 1.607440792085E-02,
             1.607440792085E-02, 1.607442380590E-02, 1.607442380590E-02, 1.607442380590E-02,
             1.607440792085E-02, 1.607440792085E-02, 1.607440792085E-02, 1.607440792085E-02,
             4.759923256550E-02, 6.859721781924E-02, 5.944385116682E-02, 3.690531501172E-02,
             9.452397764425E-03, -1.765322247487E-02, -3.551989528665E-02, -4.656675249827E-02,
             -5.018094379980E-02, -4.656675740298E-02, -3.551989767013E-02, -1.765321680648E-02,
             9.452400368789E-03, 3.690533148479E-02, 5.944385889442E-02, 6.859720926520E-02,
             4.759923581686E-02, 1.607442380590E-02, 1.607442380590E-02, 1.607442380590E-02,
             1.607442380590E-02, 1.607440792085E-02, 1.607440792085E-02, 1.607440792085E-02,
             1.607440792085E-02, 4.759923256550E-02, 6.859721781924E-02, 5.944385116682E-02,
             3.690531501172E-02, 9.452397764425E-03, -1.765322247487E-02, -3.551989528665E-02,
             -4.656675249827E-02, -5.018094379980E-02, -4.656675740298E-02, -3.551989767013E-02,
             -1.765321680648E-02, 9.452400368789E-03, 3.690533148479E-02, 5.944385889442E-02,
             6.859720926520E-02, 4.759923581686E-02, 1.607442380590E-02, 1.607442380590E-02,
             1.607442380590E-02, 1.607442380590E-02, 1.607440792085E-02, 1.607440792085E-02,
             1.607440792085E-02, 1.607440792085E-02, 4.759923256550E-02, 6.859721781924E-02,
             5.944385116682E-02, 3.690531501172E-02, 9.452397764425E-03, -1.765322247487E-02,
             -3.551989528665E-02, -4.656675249827E-02, -5.018094379980E-02, -4.656675740298E-02,
             -3.551989767013E-02, -1.765321680648E-02, 9.452400368789E-03, 3.690533148479E-02,
             5.944385889442E-02, 6.859720926520E-02, 4.759923581686E-02, 1.607442380590E-02,
             1.607442380590E-02, 1.607442380590E-02, 1.607442380590E-02, -1.378569451483E-02,
             -1.378569451483E-02, -1.378569451483E-02, -1.378569451483E-02, 2.657147766349E-02,
             5.864644230385E-02, 6.859721738448E-02, 6.482091617367E-02, 4.830380226235E-02,
             2.743644194615E-02, 1.078616361473E-02, -1.191055430439E-03, -4.769492603419E-03,
             -1.191055433654E-03, 1.078616361870E-02, 2.743644199118E-02, 4.830380231369E-02,
             6.482091612665E-02, 6.859721738214E-02, 5.864644216931E-02, 2.657147784795E-02,
             -1.378569430326E-02, -1.378569430326E-02, -1.378569430326E-02, -1.378569430326E-02,
             -1.378569451483E-02, -1.378569451483E-02, -1.378569451483E-02, -1.378569451483E-02,
             2.657147766349E-02, 5.864644230385E-02, 6.859721738448E-02, 6.482091617367E-02,
             4.830380226235E-02, 2.743644194615E-02, 1.078616361473E-02, -1.191055430439E-03,
             -4.769492603419E-03, -1.191055433654E-03, 1.078616361870E-02, 2.743644199118E-02,
             4.830380231369E-02, 6.482091612665E-02, 6.859721738214E-02, 5.864644216931E-02,
             2.657147784795E-02, -1.378569430326E-02, -1.378569430326E-02, -1.378569430326E-02,
             -1.378569430326E-02, -1.378569451483E-02, -1.378569451483E-02, -1.378569451483E-02,
             -1.378569451483E-02, 2.657147766349E-02, 5.864644230385E-02, 6.859721738448E-02,
             6.482091617367E-02, 4.830380226235E-02, 2.743644194615E-02, 1.078616361473E-02,
             -1.191055430439E-03, -4.769492603419E-03, -1.191055433654E-03, 1.078616361870E-02,
             2.743644199118E-02, 4.830380231369E-02, 6.482091612665E-02, 6.859721738214E-02,
             5.864644216931E-02, 2.657147784795E-02, -1.378569430326E-02, -1.378569430326E-02,
             -1.378569430326E-02, -1.378569430326E-02, -1.378569451483E-02, -1.378569451483E-02,
             -1.378569451483E-02, -1.378569430326E-02, -1.378569430326E-02, -1.378569430326E-02,
             2.657147770821E-02, 2.657147770821E-02, 2.657147770821E-02, 2.657147782551E-02,
             2.657147782551E-02, 2.657147782551E-02, 5.864644248034E-02, 5.864644248034E-02,
             5.864644248034E-02, 5.864644234818E-02, 5.864644234818E-02, 5.864644234818E-02,
             6.859721752004E-02, 6.859721752004E-02, 6.859721752004E-02, 6.859721735203E-02,
             6.859721735203E-02, 6.859721735203E-02, 6.482091582718E-02, 6.482091582718E-02,
             6.482091582718E-02, 6.482091629568E-02, 6.482091629568E-02, 6.482091629568E-02,
             4.830380195208E-02, 4.830380195208E-02, 4.830380195208E-02, 4.830380240319E-02,
             4.830380240319E-02, 4.830380240319E-02, 2.743644192097E-02, 2.743644192097E-02,
             2.743644192097E-02, 2.743644187475E-02, 2.743644187475E-02, 2.743644187475E-02,
             1.078616359265E-02, 1.078616359265E-02, 1.078616359265E-02, 1.078616355510E-02,
             1.078616355510E-02, 1.078616355510E-02, -1.191055464323E-03, -1.191055464323E-03,
             -1.191055464323E-03, -1.191055516624E-03, -1.191055516624E-03, -1.191055516624E-03,
             -4.769492637128E-03, -4.769492637128E-03, -4.769492637128E-03, -4.769492538829E-03,
             -4.769492538829E-03, -4.769492538829E-03, -1.191055463615E-03, -1.191055463615E-03,
             -1.191055463615E-03, -1.191055784788E-03, -1.191055784788E-03, -1.191055784788E-03,
             1.078616361818E-02, 1.078616361818E-02, 1.078616361818E-02, 1.078616464618E-02,
             1.078616464618E-02, 1.078616464618E-02, 2.743644257137E-02, 2.743644257137E-02,
             2.743644257137E-02, 2.743645152688E-02, 2.743645152688E-02, 2.743645152688E-02,
             4.830380280199E-02, 4.830380280199E-02, 4.830380280199E-02, 4.830380511728E-02,
             4.830380511728E-02, 4.830380511728E-02, 6.482091652257E-02, 6.482091652257E-02,
             6.482091652257E-02, 6.482091997431E-02, 6.482091997431E-02, 6.482091997431E-02,
             6.859721783290E-02, 6.859721783290E-02, 6.859721783290E-02, 6.859721604774E-02,
             6.859721604774E-02, 6.859721604774E-02, 5.864644993872E-02, 5.864644993872E-02,
             5.864644993872E-02, 5.864644619656E-02, 5.864644619656E-02, 5.864644619656E-02,
             2.657148642163E-02, 2.657148642163E-02, 2.657148642163E-02, 2.657148776788E-02,
             2.657148776788E-02, 2.657148776788E-02, -1.378568488354E-02, -1.378568488354E-02,
             -1.378568488354E-02, -1.378567855117E-02, -1.378567855117E-02, -1.378567855117E-02,
             -1.378568488354E-02, -1.378568488354E-02, -1.378568488354E-02, -1.378568488354E-02,
             2.657148611988E-02, 5.864644372265E-02, 6.859721855497E-02, 6.482091796229E-02,
             4.830380395902E-02, 2.743644149814E-02, 1.078616349929E-02, -1.191054880607E-03,
             -4.769492313418E-03, -1.191051647973E-03, 1.078616461311E-02, 2.743644595609E-02,
             4.830378623158E-02, 6.482091632133E-02, 6.859721533088E-02, 5.864645010707E-02,
             2.657148982038E-02, -1.378567855117E-02, -1.378567855117E-02, -1.378567855117E-02,
             -1.378567855117E-02, -1.378568488354E-02, -1.378568488354E-02, -1.378568488354E-02,
             -1.378568488354E-02, 2.657148611988E-02, 5.864644372265E-02, 6.859721855497E-02,
             6.482091796229E-02, 4.830380395902E-02, 2.743644149814E-02, 1.078616349929E-02,
             -1.191054880607E-03, -4.769492313418E-03, -1.191051647973E-03, 1.078616461311E-02,
             2.743644595609E-02, 4.830378623158E-02, 6.482091632133E-02, 6.859721533088E-02,
             5.864645010707E-02, 2.657148982038E-02, -1.378567855117E-02, -1.378567855117E-02,
             -1.378567855117E-02, -1.378567855117E-02, -1.378568488354E-02, -1.378568488354E-02,
             -1.378568488354E-02, -1.378568488354E-02, 2.657148611988E-02, 5.864644372265E-02,
             6.859721855497E-02, 6.482091796229E-02, 4.830380395902E-02, 2.743644149814E-02,
             1.078616349929E-02, -1.191054880607E-03, -4.769492313418E-03, -1.191051647973E-03,
             1.078616461311E-02, 2.743644595609E-02, 4.830378623158E-02, 6.482091632133E-02,
             6.859721533088E-02, 5.864645010707E-02, 2.657148982038E-02, -1.378567855117E-02,
             -1.378567855117E-02, -1.378567855117E-02, -1.378567855117E-02, -3.649895255508E-02,
             -3.649895255508E-02, -3.649895255508E-02, -3.649895255508E-02, -3.088974450723E-03,
             2.657147797103E-02, 4.759923168959E-02, 5.964570013507E-02, 5.828995985801E-02,
             5.086584767967E-02, 4.258073037849E-02, 3.539309464325E-02, 3.345900051267E-02,
             3.539309464044E-02, 4.258073038392E-02, 5.086584773084E-02, 5.828995991560E-02,
             5.964570008314E-02, 4.759923168730E-02, 2.657147783000E-02, -3.088974244952E-03,
             -3.649895232384E-02, -3.649895232384E-02, -3.649895232384E-02, -3.649895232384E-02,
             -3.649895255508E-02, -3.649895255508E-02, -3.649895255508E-02, -3.649895255508E-02,
             -3.088974450723E-03, 2.657147797103E-02, 4.759923168959E-02, 5.964570013507E-02,
             5.828995985801E-02, 5.086584767967E-02, 4.258073037849E-02, 3.539309464325E-02,
             3.345900051267E-02, 3.539309464044E-02, 4.258073038392E-02, 5.086584773084E-02,
             5.828995991560E-02, 5.964570008314E-02, 4.759923168730E-02, 2.657147783000E-02,
             -3.088974244952E-03, -3.649895232384E-02, -3.649895232384E-02, -3.649895232384E-02,
             -3.649895232384E-02, -3.649895255508E-02, -3.649895255508E-02, -3.649895255508E-02,
             -3.649895255508E-02, -3.088974450723E-03, 2.657147797103E-02, 4.759923168959E-02,
             5.964570013507E-02, 5.828995985801E-02, 5.086584767967E-02, 4.258073037849E-02,
             3.539309464325E-02, 3.345900051267E-02, 3.539309464044E-02, 4.258073038392E-02,
             5.086584773084E-02, 5.828995991560E-02, 5.964570008314E-02, 4.759923168730E-02,
             2.657147783000E-02, -3.088974244952E-03, -3.649895232384E-02, -3.649895232384E-02,
             -3.649895232384E-02, -3.649895232384E-02, -3.649895255508E-02, -3.649895255508E-02,
             -3.649895255508E-02, -3.649895232384E-02, -3.649895232384E-02, -3.649895232384E-02,
             -3.088974395545E-03, -3.088974395545E-03, -3.088974395545E-03, -3.088974270671E-03,
             -3.088974270671E-03, -3.088974270671E-03, 2.657147816436E-02, 2.657147816436E-02,
             2.657147816436E-02, 2.657147806440E-02, 2.657147806440E-02, 2.657147806440E-02,
             4.759923183831E-02, 4.759923183831E-02, 4.759923183831E-02, 4.759923168366E-02,
             4.759923168366E-02, 4.759923168366E-02, 5.964569976431E-02, 5.964569976431E-02,
             5.964569976431E-02, 5.964569976405E-02, 5.964569976405E-02, 5.964569976405E-02,
             5.828995952718E-02, 5.828995952718E-02, 5.828995952718E-02, 5.828995924898E-02,
             5.828995924898E-02, 5.828995924898E-02, 5.086584765225E-02, 5.086584765225E-02,
             5.086584765225E-02, 5.086584726770E-02, 5.086584726770E-02, 5.086584726770E-02,
             4.258073035448E-02, 4.258073035448E-02, 4.258073035448E-02, 4.258073004510E-02,
             4.258073004510E-02, 4.258073004510E-02, 3.539309460772E-02, 3.539309460772E-02,
             3.539309460772E-02, 3.539309469589E-02, 3.539309469589E-02, 3.539309469589E-02,
             3.345900047752E-02, 3.345900047752E-02, 3.345900047752E-02, 3.345900089289E-02,
             3.345900089289E-02, 3.345900089289E-02, 3.539309461078E-02, 3.539309461078E-02,
             3.539309461078E-02, 3.539309625472E-02, 3.539309625472E-02, 3.539309625472E-02,
             4.258073038746E-02, 4.258073038746E-02, 4.258073038746E-02, 4.258073133931E-02,
             4.258073133931E-02, 4.258073133931E-02, 5.086584837902E-02, 5.086584837902E-02,
             5.086584837902E-02, 5.086586245058E-02, 5.086586245058E-02, 5.086586245058E-02,
             5.828996046549E-02, 5.828996046549E-02, 5.828996046549E-02, 5.828996375699E-02,
             5.828996375699E-02, 5.828996375699E-02, 5.964570052086E-02, 5.964570052086E-02,
             5.964570052086E-02, 5.964570701937E-02, 5.964570701937E-02, 5.964570701937E-02,
             4.759923219181E-02, 4.759923219181E-02, 4.759923219181E-02, 4.759923880419E-02,
             4.759923880419E-02, 4.759923880419E-02, 2.657148620659E-02, 2.657148620659E-02,
             2.657148620659E-02, 2.657147021513E-02, 2.657147021513E-02, 2.657147021513E-02,
             -3.088964996876E-03, -3.088964996876E-03, -3.088964996876E-03, -3.088981954290E-03,
             -3.088981954290E-03, -3.088981954290E-03, -3.649894216064E-02, -3.649894216064E-02,
             -3.649894216064E-02, -3.649896121916E-02, -3.649896121916E-02, -3.649896121916E-02,
             -3.649894216064E-02, -3.649894216064E-02, -3.649894216064E-02, -3.649894216064E-02,
             -3.088965317752E-03, 2.657147967380E-02, 4.759923283782E-02, 5.964570218437E-02,
             5.828996136596E-02, 5.086584735050E-02, 4.258073032930E-02, 3.539309553797E-02,
             3.345900106085E-02, 3.539309434009E-02, 4.258072750425E-02, 5.086586271948E-02,
             5.828995971811E-02, 5.964570383071E-02, 4.759923096674E-02, 2.657147376084E-02,
             -3.088979565224E-03, -3.649896121916E-02, -3.649896121916E-02, -3.649896121916E-02,
             -3.649896121916E-02, -3.649894216064E-02, -3.649894216064E-02, -3.649894216064E-02,
             -3.649894216064E-02, -3.088965317752E-03, 2.657147967380E-02, 4.759923283782E-02,
             5.964570218437E-02, 5.828996136596E-02, 5.086584735050E-02, 4.258073032930E-02,
             3.539309553797E-02, 3.345900106085E-02, 3.539309434009E-02, 4.258072750425E-02,
             5.086586271948E-02, 5.828995971811E-02, 5.964570383071E-02, 4.759923096674E-02,
             2.657147376084E-02, -3.088979565224E-03, -3.649896121916E-02, -3.649896121916E-02,
             -3.649896121916E-02, -3.649896121916E-02, -3.649894216064E-02, -3.649894216064E-02,
             -3.649894216064E-02, -3.649894216064E-02, -3.088965317752E-03, 2.657147967380E-02,
             4.759923283782E-02, 5.964570218437E-02, 5.828996136596E-02, 5.086584735050E-02,
             4.258073032930E-02, 3.539309553797E-02, 3.345900106085E-02, 3.539309434009E-02,
             4.258072750425E-02, 5.086586271948E-02, 5.828995971811E-02, 5.964570383071E-02,
             4.759923096674E-02, 2.657147376084E-02, -3.088979565224E-03, -3.649896121916E-02,
             -3.649896121916E-02, -3.649896121916E-02, -3.649896121916E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02,
             -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02,
             6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02,
             7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02,
             4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02,
             5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02,
             7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02,
             5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02,
             -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02,
             4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02,
             7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02,
             6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02,
             -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -3.649895251266E-02, -3.649895251266E-02, -3.649895251266E-02, -3.649895238078E-02,
             -3.649895238078E-02, -3.649895238078E-02, -1.378569398919E-02, -1.378569398919E-02,
             -1.378569398919E-02, -1.378569404834E-02, -1.378569404834E-02, -1.378569404834E-02,
             1.607440709334E-02, 1.607440709334E-02, 1.607440709334E-02, 1.607440698224E-02,
             1.607440698224E-02, 1.607440698224E-02, 4.356156888837E-02, 4.356156888837E-02,
             4.356156888837E-02, 4.356156833446E-02, 4.356156833446E-02, 4.356156833446E-02,
             5.936207280298E-02, 5.936207280298E-02, 5.936207280298E-02, 5.936207179261E-02,
             5.936207179261E-02, 5.936207179261E-02, 6.830161064049E-02, 6.830161064049E-02,
             6.830161064049E-02, 6.830160995493E-02, 6.830160995493E-02, 6.830160995493E-02,
             7.107762956024E-02, 7.107762956024E-02, 7.107762956024E-02, 7.107762900055E-02,
             7.107762900055E-02, 7.107762900055E-02, 7.076114330212E-02, 7.076114330212E-02,
             7.076114330212E-02, 7.076114352225E-02, 7.076114352225E-02, 7.076114352225E-02,
             7.106858488521E-02, 7.106858488521E-02, 7.106858488521E-02, 7.106858560417E-02,
             7.106858560417E-02, 7.106858560417E-02, 7.076114330875E-02, 7.076114330875E-02,
             7.076114330875E-02, 7.076114692163E-02, 7.076114692163E-02, 7.076114692163E-02,
             7.107762960220E-02, 7.107762960220E-02, 7.107762960220E-02, 7.107763051569E-02,
             7.107763051569E-02, 7.107763051569E-02, 6.830161144915E-02, 6.830161144915E-02,
             6.830161144915E-02, 6.830163073671E-02, 6.830163073671E-02, 6.830163073671E-02,
             5.936207383592E-02, 5.936207383592E-02, 5.936207383592E-02, 5.936207796587E-02,
             5.936207796587E-02, 5.936207796587E-02, 4.356156970982E-02, 4.356156970982E-02,
             4.356156970982E-02, 4.356157874692E-02, 4.356157874692E-02, 4.356157874692E-02,
             1.607440749088E-02, 1.607440749088E-02, 1.607440749088E-02, 1.607442199727E-02,
             1.607442199727E-02, 1.607442199727E-02, -1.378568533457E-02, -1.378568533457E-02,
             -1.378568533457E-02, -1.378571623614E-02, -1.378571623614E-02, -1.378571623614E-02,
             -3.649894239621E-02, -3.649894239621E-02, -3.649894239621E-02, -3.649898102316E-02,
             -3.649898102316E-02, -3.649898102316E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02,
             5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02,
             7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02,
             5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02,
             -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02,
             4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02,
             7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02,
             6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02,
             -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02,
             1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02,
             7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02,
             7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02,
             1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02,
             -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02,
             6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02,
             7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02,
             4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02,
             5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02,
             7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02,
             5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02,
             -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02,
             4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02,
             7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02,
             6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02,
             -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02,
             1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02,
             7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02,
             7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02,
             1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -3.649895251266E-02,
             -3.649895251266E-02, -3.649895251266E-02, -3.649895251266E-02, -3.088974410048E-03,
             2.657147797163E-02, 4.759923169080E-02, 5.964570013458E-02, 5.828995985851E-02,
             5.086584768003E-02, 4.258073037812E-02, 3.539309464253E-02, 3.345900051164E-02,
             3.539309463997E-02, 4.258073038250E-02, 5.086584773156E-02, 5.828995991989E-02,
             5.964570008588E-02, 4.759923170226E-02, 2.657147776421E-02, -3.088974310160E-03,
             -3.649895238078E-02, -3.649895238078E-02, -3.649895238078E-02, -3.649895238078E-02,
             -1.378569398919E-02, -1.378569398919E-02, -1.378569398919E-02, -1.378569398919E-02,
             2.657147816198E-02, 5.864644242385E-02, 6.859721747601E-02, 6.482091617639E-02,
             4.830380226680E-02, 2.743644195890E-02, 1.078616363559E-02, -1.191055435243E-03,
             -4.769492606911E-03, -1.191055468247E-03, 1.078616355493E-02, 2.743644188988E-02,
             4.830380225116E-02, 6.482091624807E-02, 6.859721752143E-02, 5.864644234333E-02,
             2.657147805054E-02, -1.378569404834E-02, -1.378569404834E-02, -1.378569404834E-02,
             -1.378569404834E-02, 1.607440709334E-02, 1.607440709334E-02, 1.607440709334E-02,
             1.607440709334E-02, 4.759923183797E-02, 6.859721748211E-02, 5.944385076258E-02,
             3.690531538830E-02, 9.452397738457E-03, -1.765322336075E-02, -3.551989669594E-02,
             -4.656675271153E-02, -5.018094388164E-02, -4.656675283834E-02, -3.551989677735E-02,
             -1.765322322692E-02, 9.452397721944E-03, 3.690531550602E-02, 5.944385078247E-02,
             6.859721748235E-02, 4.759923174798E-02, 1.607440698224E-02, 1.607440698224E-02,
             1.607440698224E-02, 1.607440698224E-02, 4.356156888837E-02, 4.356156888837E-02,
             4.356156888837E-02, 4.356156888837E-02, 5.964569977667E-02, 6.482091623392E-02,
             3.690531542572E-02, -1.191055397594E-03, -3.406682298584E-02, -6.137786870104E-02,
             -7.588664276778E-02, -8.287206522729E-02, -8.555883978829E-02, -8.287206510212E-02,
             -7.588664276484E-02, -6.137786904101E-02, -3.406682313806E-02, -1.191055384940E-03,
             3.690531530214E-02, 6.482091599872E-02, 5.964569958808E-02, 4.356156833446E-02,
             4.356156833446E-02, 4.356156833446E-02, 4.356156833446E-02, 5.936207280298E-02,
             5.936207280298E-02, 5.936207280298E-02, 5.936207280298E-02, 5.828995953883E-02,
             4.830380229327E-02, 9.452397755076E-03, -3.406682298058E-02, -6.297844903824E-02,
             -8.246762363405E-02, -8.898818049045E-02, -8.945863729134E-02, -9.033251398458E-02,
             -8.945863715254E-02, -8.898818038188E-02, -8.246762418776E-02, -6.297844901497E-02,
             -3.406682249055E-02, 9.452397704854E-03, 4.830380502371E-02, 5.828996054729E-02,
             5.936207179261E-02, 5.936207179261E-02, 5.936207179261E-02, 5.936207179261E-02,
             6.830161064049E-02, 6.830161064049E-02, 6.830161064049E-02, 6.830161064049E-02,
             5.086584765759E-02, 2.743644197606E-02, -1.765322334310E-02, -6.137786872017E-02,
             -8.246762365112E-02, -9.132287812490E-02, -8.866359502242E-02, -8.226118590843E-02,
             -8.112847436445E-02, -8.226118553265E-02, -8.866359522795E-02, -9.132287779306E-02,
             -8.246762337901E-02, -6.137786667202E-02, -1.765322258180E-02, 2.743644163198E-02,
             5.086584718241E-02, 6.830160995493E-02, 6.830160995493E-02, 6.830160995493E-02,
             6.830160995493E-02, 7.107762956024E-02, 7.107762956024E-02, 7.107762956024E-02,
             7.107762956024E-02, 4.258073035995E-02, 1.078616365765E-02, -3.551989668366E-02,
             -7.588664279742E-02, -8.898818049794E-02, -8.866359553532E-02, -7.909406587553E-02,
             -6.792741704632E-02, -6.533539474735E-02, -6.792741727342E-02, -7.909406656255E-02,
             -8.866359481795E-02, -8.898818009094E-02, -7.588664168058E-02, -3.551989637476E-02,
             1.078616363163E-02, 4.258073011491E-02, 7.107762900055E-02, 7.107762900055E-02,
             7.107762900055E-02, 7.107762900055E-02, 7.076114330212E-02, 7.076114330212E-02,
             7.076114330212E-02, 7.076114330212E-02, 3.539309461359E-02, -1.191055481943E-03,
             -4.656675275215E-02, -8.287206521760E-02, -8.945863727463E-02, -8.226118586178E-02,
             -6.792741699264E-02, -5.367000806480E-02, -5.009611083633E-02, -5.367000804510E-02,
             -6.792741687764E-02, -8.226118583101E-02, -8.945863734646E-02, -8.287206490661E-02,
             -4.656675260396E-02, -1.191055118795E-03, 3.539309488367E-02, 7.076114352225E-02,
             7.076114352225E-02, 7.076114352225E-02, 7.076114352225E-02, 7.106858488521E-02,
             7.106858488521E-02, 7.106858488521E-02, 7.106858488521E-02, 3.345900048339E-02,
             -4.769492662390E-03, -5.018094392990E-02, -8.555883977530E-02, -9.033251397775E-02,
             -8.112847434770E-02, -6.533539475118E-02, -5.009611083819E-02, -4.621706539973E-02,
             -5.009611085778E-02, -6.533539477906E-02, -8.112847421729E-02, -9.033251392657E-02,
             -8.555883966639E-02, -5.018094399239E-02, -4.769492102270E-03, 3.345900110156E-02,
             7.106858560417E-02, 7.106858560417E-02, 7.106858560417E-02, 7.106858560417E-02,
             7.076114330875E-02, 7.076114330875E-02, 7.076114330875E-02, 7.076114330875E-02,
             3.539309463675E-02, -1.191055954998E-03, -4.656675327912E-02, -8.287206528838E-02,
             -8.945863711222E-02, -8.226118548381E-02, -6.792741744182E-02, -5.367000796722E-02,
             -5.009611073526E-02, -5.367001137943E-02, -6.792741956979E-02, -8.226118600574E-02,
             -8.945863692581E-02, -8.287206615867E-02, -4.656675441066E-02, -1.191053975904E-03,
             3.539309714428E-02, 7.076114692163E-02, 7.076114692163E-02, 7.076114692163E-02,
             7.076114692163E-02, 7.107762960220E-02, 7.107762960220E-02, 7.107762960220E-02,
             7.107762960220E-02, 4.258073041669E-02, 1.078616294658E-02, -3.551989728345E-02,
             -7.588664266096E-02, -8.898818027377E-02, -8.866359385549E-02, -7.909406592660E-02,
             -6.792741693994E-02, -6.533539476838E-02, -6.792742114874E-02, -7.909407003583E-02,
             -8.866359504352E-02, -8.898817980427E-02, -7.588664757437E-02, -3.551990152437E-02,
             1.078616300539E-02, 4.258073048037E-02, 7.107763051569E-02, 7.107763051569E-02,
             7.107763051569E-02, 7.107763051569E-02, 6.830161144915E-02, 6.830161144915E-02,
             6.830161144915E-02, 6.830161144915E-02, 5.086584850536E-02, 2.743644138385E-02,
             -1.765322390342E-02, -6.137786902090E-02, -8.246762368091E-02, -9.132287774034E-02,
             -8.866359493279E-02, -8.226118612119E-02, -8.112847441159E-02, -8.226118626526E-02,
             -8.866359515258E-02, -9.132286869337E-02, -8.246761848958E-02, -6.137788856119E-02,
             -1.765322526848E-02, 2.743649294840E-02, 5.086588286773E-02, 6.830163073671E-02,
             6.830163073671E-02, 6.830163073671E-02, 6.830163073671E-02, 5.936207383592E-02,
             5.936207383592E-02, 5.936207383592E-02, 5.936207383592E-02, 5.828996061409E-02,
             4.830380164939E-02, 9.452397146329E-03, -3.406682314592E-02, -6.297844913681E-02,
             -8.246762336583E-02, -8.898818042355E-02, -8.945863741340E-02, -9.033251400255E-02,
             -8.945863766455E-02, -8.898818049803E-02, -8.246761944604E-02, -6.297845133475E-02,
             -3.406683795400E-02, 9.452394471189E-03, 4.830385852311E-02, 5.828999030603E-02,
             5.936207796587E-02, 5.936207796587E-02, 5.936207796587E-02, 5.936207796587E-02,
             4.356156970982E-02, 4.356156970982E-02, 4.356156970982E-02, 4.356156970982E-02,
             5.964570058515E-02, 6.482091548318E-02, 3.690531479519E-02, -1.191055307466E-03,
             -3.406682252300E-02, -6.137787085067E-02, -7.588664476469E-02, -8.287206480677E-02,
             -8.555884016739E-02, -8.287206150668E-02, -7.588664328156E-02, -6.137786856535E-02,
             -3.406680693889E-02, -1.191047607186E-03, 3.690531411597E-02, 6.482091306002E-02,
             5.964570358467E-02, 4.356157874692E-02, 4.356157874692E-02, 4.356157874692E-02,
             4.356157874692E-02, 1.607440749088E-02, 1.607440749088E-02, 1.607440749088E-02,
             1.607440749088E-02, 4.759923227738E-02, 6.859721666281E-02, 5.944384998040E-02,
             3.690531561203E-02, 9.452398473741E-03, -1.765322447697E-02, -3.551989933283E-02,
             -4.656675261139E-02, -5.018094429076E-02, -4.656674952709E-02, -3.551989417196E-02,
             -1.765322484755E-02, 9.452412693733E-03, 3.690531534973E-02, 5.944384216703E-02,
             6.859722061430E-02, 4.759924067321E-02, 1.607442199727E-02, 1.607442199727E-02,
             1.607442199727E-02, 1.607442199727E-02, -1.378568533457E-02, -1.378568533457E-02,
             -1.378568533457E-02, -1.378568533457E-02, 2.657148576291E-02, 5.864644397361E-02,
             6.859721840604E-02, 6.482091866912E-02, 4.830380415185E-02, 2.743644229984E-02,
             1.078616387454E-02, -1.191054274557E-03, -4.769491260276E-03, -1.191055983384E-03,
             1.078616236785E-02, 2.743646591988E-02, 4.830383191454E-02, 6.482090908749E-02,
             6.859721946819E-02, 5.864639572511E-02, 2.657144249915E-02, -1.378571623614E-02,
             -1.378571623614E-02, -1.378571623614E-02, -1.378571623614E-02, -3.649894239621E-02,
             -3.649894239621E-02, -3.649894239621E-02, -3.649894239621E-02, -3.088965503338E-03,
             2.657147980753E-02, 4.759923277444E-02, 5.964570256308E-02, 5.828996149711E-02,
             5.086584776336E-02, 4.258073056235E-02, 3.539309583851E-02, 3.345900157706E-02,
             3.539309215461E-02, 4.258072638612E-02, 5.086587387137E-02, 5.828998283735E-02,
             5.964569990800E-02, 4.759923283785E-02, 2.657144707246E-02, -3.089003456111E-03,
             -3.649898102316E-02, -3.649898102316E-02, -3.649898102316E-02, -3.649898102316E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02,
             5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02,
             7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02,
             5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02,
             -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02,
             4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02,
             7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02,
             6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02,
             -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02,
             1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02,
             7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02,
             7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02,
             1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02,
             -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02,
             6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02,
             7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02,
             4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02,
             5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02,
             7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02,
             5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02,
             -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02,
             4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02,
             7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02,
             6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02,
             -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02,
             1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02,
             7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02,
             7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02,
             1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02,
             -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02,
             6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02,
             7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02,
             4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -3.649895251266E-02, -3.649895251266E-02, -3.649895251266E-02, -3.649895251266E-02,
             -3.088974410048E-03, 2.657147797163E-02, 4.759923169080E-02, 5.964570013458E-02,
             5.828995985851E-02, 5.086584768003E-02, 4.258073037812E-02, 3.539309464253E-02,
             3.345900051164E-02, 3.539309463997E-02, 4.258073038250E-02, 5.086584773156E-02,
             5.828995991989E-02, 5.964570008588E-02, 4.759923170226E-02, 2.657147776421E-02,
             -3.088974310160E-03, -3.649895238078E-02, -3.649895238078E-02, -3.649895238078E-02,
             -3.649895238078E-02, -1.378569398919E-02, -1.378569398919E-02, -1.378569398919E-02,
             -1.378569398919E-02, 2.657147816198E-02, 5.864644242385E-02, 6.859721747601E-02,
             6.482091617639E-02, 4.830380226680E-02, 2.743644195890E-02, 1.078616363559E-02,
             -1.191055435243E-03, -4.769492606911E-03, -1.191055468247E-03, 1.078616355493E-02,
             2.743644188988E-02, 4.830380225116E-02, 6.482091624807E-02, 6.859721752143E-02,
             5.864644234333E-02, 2.657147805054E-02, -1.378569404834E-02, -1.378569404834E-02,
             -1.378569404834E-02, -1.378569404834E-02, 1.607440709334E-02, 1.607440709334E-02,
             1.607440709334E-02, 1.607440709334E-02, 4.759923183797E-02, 6.859721748211E-02,
             5.944385076258E-02, 3.690531538830E-02, 9.452397738457E-03, -1.765322336075E-02,
             -3.551989669594E-02, -4.656675271153E-02, -5.018094388164E-02, -4.656675283834E-02,
             -3.551989677735E-02, -1.765322322692E-02, 9.452397721944E-03, 3.690531550602E-02,
             5.944385078247E-02, 6.859721748235E-02, 4.759923174798E-02, 1.607440698224E-02,
             1.607440698224E-02, 1.607440698224E-02, 1.607440698224E-02, 4.356156888837E-02,
             4.356156888837E-02, 4.356156888837E-02, 4.356156888837E-02, 5.964569977667E-02,
             6.482091623392E-02, 3.690531542572E-02, -1.191055397594E-03, -3.406682298584E-02,
             -6.137786870104E-02, -7.588664276778E-02, -8.287206522729E-02, -8.555883978829E-02,
             -8.287206510212E-02, -7.588664276484E-02, -6.137786904101E-02, -3.406682313806E-02,
             -1.191055384940E-03, 3.690531530214E-02, 6.482091599872E-02, 5.964569958808E-02,
             4.356156833446E-02, 4.356156833446E-02, 4.356156833446E-02, 4.356156833446E-02,
             5.936207280298E-02, 5.936207280298E-02, 5.936207280298E-02, 5.936207280298E-02,
             5.828995953883E-02, 4.830380229327E-02, 9.452397755076E-03, -3.406682298058E-02,
             -6.297844903824E-02, -8.246762363405E-02, -8.898818049045E-02, -8.945863729134E-02,
             -9.033251398458E-02, -8.945863715254E-02, -8.898818038188E-02, -8.246762418776E-02,
             -6.297844901497E-02, -3.406682249055E-02, 9.452397704854E-03, 4.830380502371E-02,
             5.828996054729E-02, 5.936207179261E-02, 5.936207179261E-02, 5.936207179261E-02,
             5.936207179261E-02, 6.830161064049E-02, 6.830161064049E-02, 6.830161064049E-02,
             6.830161064049E-02, 5.086584765759E-02, 2.743644197606E-02, -1.765322334310E-02,
             -6.137786872017E-02, -8.246762365112E-02, -9.132287812490E-02, -8.866359502242E-02,
             -8.226118590843E-02, -8.112847436445E-02, -8.226118553265E-02, -8.866359522795E-02,
             -9.132287779306E-02, -8.246762337901E-02, -6.137786667202E-02, -1.765322258180E-02,
             2.743644163198E-02, 5.086584718241E-02, 6.830160995493E-02, 6.830160995493E-02,
             6.830160995493E-02, 6.830160995493E-02, 7.107762956024E-02, 7.107762956024E-02,
             7.107762956024E-02, 7.107762956024E-02, 4.258073035995E-02, 1.078616365765E-02,
             -3.551989668366E-02, -7.588664279742E-02, -8.898818049794E-02, -8.866359553532E-02,
             -7.909406587553E-02, -6.792741704632E-02, -6.533539474735E-02, -6.792741727342E-02,
             -7.909406656255E-02, -8.866359481795E-02, -8.898818009094E-02, -7.588664168058E-02,
             -3.551989637476E-02, 1.078616363163E-02, 4.258073011491E-02, 7.107762900055E-02,
             7.107762900055E-02, 7.107762900055E-02, 7.107762900055E-02, 7.076114330212E-02,
             7.076114330212E-02, 7.076114330212E-02, 7.076114330212E-02, 3.539309461359E-02,
             -1.191055481943E-03, -4.656675275215E-02, -8.287206521760E-02, -8.945863727463E-02,
             -8.226118586178E-02, -6.792741699264E-02, -5.367000806480E-02, -5.009611083633E-02,
             -5.367000804510E-02, -6.792741687764E-02, -8.226118583101E-02, -8.945863734646E-02,
             -8.287206490661E-02, -4.656675260396E-02, -1.191055118795E-03, 3.539309488367E-02,
             7.076114352225E-02, 7.076114352225E-02, 7.076114352225E-02, 7.076114352225E-02,
             7.106858488521E-02, 7.106858488521E-02, 7.106858488521E-02, 7.106858488521E-02,
             3.345900048339E-02, -4.769492662390E-03, -5.018094392990E-02, -8.555883977530E-02,
             -9.033251397775E-02, -8.112847434770E-02, -6.533539475118E-02, -5.009611083819E-02,
             -4.621706539973E-02, -5.009611085778E-02, -6.533539477906E-02, -8.112847421729E-02,
             -9.033251392657E-02, -8.555883966639E-02, -5.018094399239E-02, -4.769492102270E-03,
             3.345900110156E-02, 7.106858560417E-02, 7.106858560417E-02, 7.106858560417E-02,
             7.106858560417E-02, 7.076114330875E-02, 7.076114330875E-02, 7.076114330875E-02,
             7.076114330875E-02, 3.539309463675E-02, -1.191055954998E-03, -4.656675327912E-02,
             -8.287206528838E-02, -8.945863711222E-02, -8.226118548381E-02, -6.792741744182E-02,
             -5.367000796722E-02, -5.009611073526E-02, -5.367001137943E-02, -6.792741956979E-02,
             -8.226118600574E-02, -8.945863692581E-02, -8.287206615867E-02, -4.656675441066E-02,
             -1.191053975904E-03, 3.539309714428E-02, 7.076114692163E-02, 7.076114692163E-02,
             7.076114692163E-02, 7.076114692163E-02, 7.107762960220E-02, 7.107762960220E-02,
             7.107762960220E-02, 7.107762960220E-02, 4.258073041669E-02, 1.078616294658E-02,
             -3.551989728345E-02, -7.588664266096E-02, -8.898818027377E-02, -8.866359385549E-02,
             -7.909406592660E-02, -6.792741693994E-02, -6.533539476838E-02, -6.792742114874E-02,
             -7.909407003583E-02, -8.866359504352E-02, -8.898817980427E-02, -7.588664757437E-02,
             -3.551990152437E-02, 1.078616300539E-02, 4.258073048037E-02, 7.107763051569E-02,
             7.107763051569E-02, 7.107763051569E-02, 7.107763051569E-02, 6.830161144915E-02,
             6.830161144915E-02, 6.830161144915E-02, 6.830161144915E-02, 5.086584850536E-02,
             2.743644138385E-02, -1.765322390342E-02, -6.137786902090E-02, -8.246762368091E-02,
             -9.132287774034E-02, -8.866359493279E-02, -8.226118612119E-02, -8.112847441159E-02,
             -8.226118626526E-02, -8.866359515258E-02, -9.132286869337E-02, -8.246761848958E-02,
             -6.137788856119E-02, -1.765322526848E-02, 2.743649294840E-02, 5.086588286773E-02,
             6.830163073671E-02, 6.830163073671E-02, 6.830163073671E-02, 6.830163073671E-02,
             5.936207383592E-02, 5.936207383592E-02, 5.936207383592E-02, 5.936207383592E-02,
             5.828996061409E-02, 4.830380164939E-02, 9.452397146329E-03, -3.406682314592E-02,
             -6.297844913681E-02, -8.246762336583E-02, -8.898818042355E-02, -8.945863741340E-02,
             -9.033251400255E-02, -8.945863766455E-02, -8.898818049803E-02, -8.246761944604E-02,
             -6.297845133475E-02, -3.406683795400E-02, 9.452394471189E-03, 4.830385852311E-02,
             5.828999030603E-02, 5.936207796587E-02, 5.936207796587E-02, 5.936207796587E-02,
             5.936207796587E-02, 4.356156970982E-02, 4.356156970982E-02, 4.356156970982E-02,
             4.356156970982E-02, 5.964570058515E-02, 6.482091548318E-02, 3.690531479519E-02,
             -1.191055307466E-03, -3.406682252300E-02, -6.137787085067E-02, -7.588664476469E-02,
             -8.287206480677E-02, -8.555884016739E-02, -8.287206150668E-02, -7.588664328156E-02,
             -6.137786856535E-02, -3.406680693889E-02, -1.191047607186E-03, 3.690531411597E-02,
             6.482091306002E-02, 5.964570358467E-02, 4.356157874692E-02, 4.356157874692E-02,
             4.356157874692E-02, 4.356157874692E-02, 1.607440749088E-02, 1.607440749088E-02,
             1.607440749088E-02, 1.607440749088E-02, 4.759923227738E-02, 6.859721666281E-02,
             5.944384998040E-02, 3.690531561203E-02, 9.452398473741E-03, -1.765322447697E-02,
             -3.551989933283E-02, -4.656675261139E-02, -5.018094429076E-02, -4.656674952709E-02,
             -3.551989417196E-02, -1.765322484755E-02, 9.452412693733E-03, 3.690531534973E-02,
             5.944384216703E-02, 6.859722061430E-02, 4.759924067321E-02, 1.607442199727E-02,
             1.607442199727E-02, 1.607442199727E-02, 1.607442199727E-02, -1.378568533457E-02,
             -1.378568533457E-02, -1.378568533457E-02, -1.378568533457E-02, 2.657148576291E-02,
             5.864644397361E-02, 6.859721840604E-02, 6.482091866912E-02, 4.830380415185E-02,
             2.743644229984E-02, 1.078616387454E-02, -1.191054274557E-03, -4.769491260276E-03,
             -1.191055983384E-03, 1.078616236785E-02, 2.743646591988E-02, 4.830383191454E-02,
             6.482090908749E-02, 6.859721946819E-02, 5.864639572511E-02, 2.657144249915E-02,
             -1.378571623614E-02, -1.378571623614E-02, -1.378571623614E-02, -1.378571623614E-02,
             -3.649894239621E-02, -3.649894239621E-02, -3.649894239621E-02, -3.649894239621E-02,
             -3.088965503338E-03, 2.657147980753E-02, 4.759923277444E-02, 5.964570256308E-02,
             5.828996149711E-02, 5.086584776336E-02, 4.258073056235E-02, 3.539309583851E-02,
             3.345900157706E-02, 3.539309215461E-02, 4.258072638612E-02, 5.086587387137E-02,
             5.828998283735E-02, 5.964569990800E-02, 4.759923283785E-02, 2.657144707246E-02,
             -3.089003456111E-03, -3.649898102316E-02, -3.649898102316E-02, -3.649898102316E-02,
             -3.649898102316E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02,
             4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02,
             7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02,
             6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02,
             -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02,
             1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02,
             7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02,
             7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02,
             1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02,
             -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02,
             6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02,
             7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02,
             4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02,
             5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02,
             7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02,
             5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02,
             -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02,
             4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02,
             7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02,
             6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02,
             -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02, -1.378569419990E-02,
             1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02, 6.830161067012E-02,
             7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02, 7.076114333682E-02,
             7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02, 4.356156922722E-02,
             1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02, -5.750924643689E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924668791E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -3.649895257968E-02,
             -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02, 5.936207315506E-02,
             6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02, 7.106858492176E-02,
             7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02, 5.936207321929E-02,
             4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02, -3.649895235139E-02,
             -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02, -5.750924668791E-02,
             -3.649895257968E-02, -1.378569419990E-02, 1.607440693101E-02, 4.356156928429E-02,
             5.936207315506E-02, 6.830161067012E-02, 7.107762958617E-02, 7.076114333927E-02,
             7.106858492176E-02, 7.076114333682E-02, 7.107762959305E-02, 6.830161072793E-02,
             5.936207321929E-02, 4.356156922722E-02, 1.607440692874E-02, -1.378569434610E-02,
             -3.649895235139E-02, -5.750924643689E-02, -5.750924643689E-02, -5.750924643689E-02,
             -5.750924643689E-02, -3.649895251266E-02, -3.649895251266E-02, -3.649895251266E-02,
             -3.649895251266E-02, -3.088974410048E-03, 2.657147797163E-02, 4.759923169080E-02,
             5.964570013458E-02, 5.828995985851E-02, 5.086584768003E-02, 4.258073037812E-02,
             3.539309464253E-02, 3.345900051164E-02, 3.539309463997E-02, 4.258073038250E-02,
             5.086584773156E-02, 5.828995991989E-02, 5.964570008588E-02, 4.759923170226E-02,
             2.657147776421E-02, -3.088974310160E-03, -3.649895238078E-02, -3.649895238078E-02,
             -3.649895238078E-02, -3.649895238078E-02, -1.378569398919E-02, -1.378569398919E-02,
             -1.378569398919E-02, -1.378569398919E-02, 2.657147816198E-02, 5.864644242385E-02,
             6.859721747601E-02, 6.482091617639E-02, 4.830380226680E-02, 2.743644195890E-02,
             1.078616363559E-02, -1.191055435243E-03, -4.769492606911E-03, -1.191055468247E-03,
             1.078616355493E-02, 2.743644188988E-02, 4.830380225116E-02, 6.482091624807E-02,
             6.859721752143E-02, 5.864644234333E-02, 2.657147805054E-02, -1.378569404834E-02,
             -1.378569404834E-02, -1.378569404834E-02, -1.378569404834E-02, 1.607440709334E-02,
             1.607440709334E-02, 1.607440709334E-02, 1.607440709334E-02, 4.759923183797E-02,
             6.859721748211E-02, 5.944385076258E-02, 3.690531538830E-02, 9.452397738457E-03,
             -1.765322336075E-02, -3.551989669594E-02, -4.656675271153E-02, -5.018094388164E-02,
             -4.656675283834E-02, -3.551989677735E-02, -1.765322322692E-02, 9.452397721944E-03,
             3.690531550602E-02, 5.944385078247E-02, 6.859721748235E-02, 4.759923174798E-02,
             1.607440698224E-02, 1.607440698224E-02, 1.607440698224E-02, 1.607440698224E-02,
             4.356156888837E-02, 4.356156888837E-02, 4.356156888837E-02, 4.356156888837E-02,
             5.964569977667E-02, 6.482091623392E-02, 3.690531542572E-02, -1.191055397594E-03,
             -3.406682298584E-02, -6.137786870104E-02, -7.588664276778E-02, -8.287206522729E-02,
             -8.555883978829E-02, -8.287206510212E-02, -7.588664276484E-02, -6.137786904101E-02,
             -3.406682313806E-02, -1.191055384940E-03, 3.690531530214E-02, 6.482091599872E-02,
             5.964569958808E-02, 4.356156833446E-02, 4.356156833446E-02, 4.356156833446E-02,
             4.356156833446E-02, 5.936207280298E-02, 5.936207280298E-02, 5.936207280298E-02,
             5.936207280298E-02, 5.828995953883E-02, 4.830380229327E-02, 9.452397755076E-03,
             -3.406682298058E-02, -6.297844903824E-02, -8.246762363405E-02, -8.898818049045E-02,
             -8.945863729134E-02, -9.033251398458E-02, -8.945863715254E-02, -8.898818038188E-02,
             -8.246762418776E-02, -6.297844901497E-02, -3.406682249055E-02, 9.452397704854E-03,
             4.830380502371E-02, 5.828996054729E-02, 5.936207179261E-02, 5.936207179261E-02,
             5.936207179261E-02, 5.936207179261E-02, 6.830161064049E-02, 6.830161064049E-02,
             6.830161064049E-02, 6.830161064049E-02, 5.086584765759E-02, 2.743644197606E-02,
             -1.765322334310E-02, -6.137786872017E-02, -8.246762365112E-02, -9.132287812490E-02,
             -8.866359502242E-02, -8.226118590843E-02, -8.112847436445E-02, -8.226118553265E-02,
             -8.866359522795E-02, -9.132287779306E-02, -8.246762337901E-02, -6.137786667202E-02,
             -1.765322258180E-02, 2.743644163198E-02, 5.086584718241E-02, 6.830160995493E-02,
             6.830160995493E-02, 6.830160995493E-02, 6.830160995493E-02, 7.107762956024E-02,
             7.107762956024E-02, 7.107762956024E-02, 7.107762956024E-02, 4.258073035995E-02,
             1.078616365765E-02, -3.551989668366E-02, -7.588664279742E-02, -8.898818049794E-02,
             -8.866359553532E-02, -7.909406587553E-02, -6.792741704632E-02, -6.533539474735E-02,
             -6.792741727342E-02, -7.909406656255E-02, -8.866359481795E-02, -8.898818009094E-02,
             -7.588664168058E-02, -3.551989637476E-02, 1.078616363163E-02, 4.258073011491E-02,
             7.107762900055E-02, 7.107762900055E-02, 7.107762900055E-02, 7.107762900055E-02,
             7.076114330212E-02, 7.076114330212E-02, 7.076114330212E-02, 7.076114330212E-02,
             3.539309461359E-02, -1.191055481943E-03, -4.656675275215E-02, -8.287206521760E-02,
             -8.945863727463E-02, -8.226118586178E-02, -6.792741699264E-02, -5.367000806480E-02,
             -5.009611083633E-02, -5.367000804510E-02, -6.792741687764E-02, -8.226118583101E-02,
             -8.945863734646E-02, -8.287206490661E-02, -4.656675260396E-02, -1.191055118795E-03,
             3.539309488367E-02, 7.076114352225E-02, 7.076114352225E-02, 7.076114352225E-02,
             7.076114352225E-02, 7.106858488521E-02, 7.106858488521E-02, 7.106858488521E-02,
             7.106858488521E-02, 3.345900048339E-02, -4.769492662390E-03, -5.018094392990E-02,
             -8.555883977530E-02, -9.033251397775E-02, -8.112847434770E-02, -6.533539475118E-02,
             -5.009611083819E-02, -4.621706539973E-02, -5.009611085778E-02, -6.533539477906E-02,
             -8.112847421729E-02, -9.033251392657E-02, -8.555883966639E-02, -5.018094399239E-02,
             -4.769492102270E-03, 3.345900110156E-02, 7.106858560417E-02, 7.106858560417E-02,
             7.106858560417E-02, 7.106858560417E-02, 7.076114330875E-02, 7.076114330875E-02,
             7.076114330875E-02, 7.076114330875E-02, 3.539309463675E-02, -1.191055954998E-03,
             -4.656675327912E-02, -8.287206528838E-02, -8.945863711222E-02, -8.226118548381E-02,
             -6.792741744182E-02, -5.367000796722E-02, -5.009611073526E-02, -5.367001137943E-02,
             -6.792741956979E-02, -8.226118600574E-02, -8.945863692581E-02, -8.287206615867E-02,
             -4.656675441066E-02, -1.191053975904E-03, 3.539309714428E-02, 7.076114692163E-02,
             7.076114692163E-02, 7.076114692163E-02, 7.076114692163E-02, 7.107762960220E-02,
             7.107762960220E-02, 7.107762960220E-02, 7.107762960220E-02, 4.258073041669E-02,
             1.078616294658E-02, -3.551989728345E-02, -7.588664266096E-02, -8.898818027377E-02,
             -8.866359385549E-02, -7.909406592660E-02, -6.792741693994E-02, -6.533539476838E-02,
             -6.792742114874E-02, -7.909407003583E-02, -8.866359504352E-02, -8.898817980427E-02,
             -7.588664757437E-02, -3.551990152437E-02, 1.078616300539E-02, 4.258073048037E-02,
             7.107763051569E-02, 7.107763051569E-02, 7.107763051569E-02, 7.107763051569E-02,
             6.830161144915E-02, 6.830161144915E-02, 6.830161144915E-02, 6.830161144915E-02,
             5.086584850536E-02, 2.743644138385E-02, -1.765322390342E-02, -6.137786902090E-02,
             -8.246762368091E-02, -9.132287774034E-02, -8.866359493279E-02, -8.226118612119E-02,
             -8.112847441159E-02, -8.226118626526E-02, -8.866359515258E-02, -9.132286869337E-02,
             -8.246761848958E-02, -6.137788856119E-02, -1.765322526848E-02, 2.743649294840E-02,
             5.086588286773E-02, 6.830163073671E-02, 6.830163073671E-02, 6.830163073671E-02,
             6.830163073671E-02, 5.936207383592E-02, 5.936207383592E-02, 5.936207383592E-02,
             5.936207383592E-02, 5.828996061409E-02, 4.830380164939E-02, 9.452397146329E-03,
             -3.406682314592E-02, -6.297844913681E-02, -8.246762336583E-02, -8.898818042355E-02,
             -8.945863741340E-02, -9.033251400255E-02, -8.945863766455E-02, -8.898818049803E-02,
             -8.246761944604E-02, -6.297845133475E-02, -3.406683795400E-02, 9.452394471189E-03,
             4.830385852311E-02, 5.828999030603E-02, 5.936207796587E-02, 5.936207796587E-02,
             5.936207796587E-02, 5.936207796587E-02, 4.356156970982E-02, 4.356156970982E-02,
             4.356156970982E-02, 4.356156970982E-02, 5.964570058515E-02, 6.482091548318E-02,
             3.690531479519E-02, -1.191055307466E-03, -3.406682252300E-02, -6.137787085067E-02,
             -7.588664476469E-02, -8.287206480677E-02, -8.555884016739E-02, -8.287206150668E-02,
             -7.588664328156E-02, -6.137786856535E-02, -3.406680693889E-02, -1.191047607186E-03,
             3.690531411597E-02, 6.482091306002E-02, 5.964570358467E-02, 4.356157874692E-02,
             4.356157874692E-02, 4.356157874692E-02, 4.356157874692E-02, 1.607440749088E-02,
             1.607440749088E-02, 1.607440749088E-02, 1.607440749088E-02, 4.759923227738E-02,
             6.859721666281E-02, 5.944384998040E-02, 3.690531561203E-02, 9.452398473741E-03,
             -1.765322447697E-02, -3.551989933283E-02, -4.656675261139E-02, -5.018094429076E-02,
             -4.656674952709E-02, -3.551989417196E-02, -1.765322484755E-02, 9.452412693733E-03,
             3.690531534973E-02, 5.944384216703E-02, 6.859722061430E-02, 4.759924067321E-02,
             1.607442199727E-02, 1.607442199727E-02, 1.607442199727E-02, 1.607442199727E-02,
             -1.378568533457E-02, -1.378568533457E-02, -1.378568533457E-02, -1.378568533457E-02,
             2.657148576291E-02, 5.864644397361E-02, 6.859721840604E-02, 6.482091866912E-02,
             4.830380415185E-02, 2.743644229984E-02, 1.078616387454E-02, -1.191054274557E-03,
             -4.769491260276E-03, -1.191055983384E-03, 1.078616236785E-02, 2.743646591988E-02,
             4.830383191454E-02, 6.482090908749E-02, 6.859721946819E-02, 5.864639572511E-02,
             2.657144249915E-02, -1.378571623614E-02, -1.378571623614E-02, -1.378571623614E-02,
             -1.378571623614E-02, -3.649894239621E-02, -3.649894239621E-02, -3.649894239621E-02,
             -3.649894239621E-02, -3.088965503338E-03, 2.657147980753E-02, 4.759923277444E-02,
             5.964570256308E-02, 5.828996149711E-02, 5.086584776336E-02, 4.258073056235E-02,
             3.539309583851E-02, 3.345900157706E-02, 3.539309215461E-02, 4.258072638612E-02,
             5.086587387137E-02, 5.828998283735E-02, 5.964569990800E-02, 4.759923283785E-02,
             2.657144707246E-02, -3.089003456111E-03, -3.649898102316E-02, -3.649898102316E-02,
             -3.649898102316E-02, -3.649898102316E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02,
             1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02,
             7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02,
             7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02,
             1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750923549385E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -3.649894273713E-02,
             -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02, 5.936207452839E-02,
             6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02, 7.106858572433E-02,
             7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02, 5.936208814239E-02,
             4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02, -3.649897692594E-02,
             -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02, 4.356157160396E-02,
             5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02, 7.076114457914E-02,
             7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02, 6.830163680087E-02,
             5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02, -1.378571022724E-02,
             -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750923549385E-02, -5.750923549385E-02, -5.750923549385E-02,
             -5.750923549385E-02, -3.649894273713E-02, -1.378569220219E-02, 1.607440808175E-02,
             4.356157160396E-02, 5.936207452839E-02, 6.830161045752E-02, 7.107762962909E-02,
             7.076114457914E-02, 7.106858572433E-02, 7.076113896017E-02, 7.107762285500E-02,
             6.830163680087E-02, 5.936208814239E-02, 4.356157649408E-02, 1.607440775774E-02,
             -1.378571022724E-02, -3.649897692594E-02, -5.750928408452E-02, -5.750928408452E-02,
             -5.750928408452E-02, -5.750928408452E-02], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
            [8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01, 1.480862149879E+00,
       -3.653094590856E+00, -7.874665253011E+00, -9.806984369113E+00, -9.931104027256E+00,
       -7.799246504079E+00, -4.821035766484E+00, -2.144463029636E+00, -1.066488253839E-01,
       4.826495422536E-01, -1.066488253839E-01, -2.144463029636E+00, -4.821035766484E+00,
       -7.799246504079E+00, -9.931104027256E+00, -9.806984369113E+00, -7.874665253011E+00,
       -3.653094590856E+00, 1.480862149879E+00, 2.954744435855E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00,
       -6.669993535749E+00, -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00,
       1.683444271630E-01, 5.729701223819E-01, 1.683444271630E-01, -1.225075078640E+00,
       -3.068661134588E+00, -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00,
       -5.471668476150E+00, -2.636774001778E+00, 8.369010281869E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00, -3.068671699289E+00,
       -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00, -1.316286502692E+00,
       -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01, 4.433376797099E-01,
       -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00, -3.408883044242E+00,
       -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00, 1.929399064948E-01,
       9.977774723846E+00, 1.976260954120E+01, 2.954744435855E+01, 1.480862149879E+00,
       8.369010281869E-01, 1.929399064948E-01, -4.510212151972E-01, -6.041328236203E-01,
       -6.656749224279E-01, -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01,
       4.360881292036E-01, 6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01,
       7.183309322567E-01, 6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01,
       -1.477725527353E-01, -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01,
       -4.510212151972E-01, 1.929399064948E-01, 8.369010281869E-01, 1.480862149879E+00,
       -3.653094590856E+00, -2.636774001778E+00, -1.620453412699E+00, -6.041328236203E-01,
       -4.596429798148E-01, -2.653592939274E-01, 1.015744035042E-01, 4.717649724219E-01,
       6.982843228335E-01, 8.365934740037E-01, 8.751165915620E-01, 8.680361464225E-01,
       8.719664804748E-01, 8.680361464225E-01, 8.751165915620E-01, 8.365934740037E-01,
       6.982843228335E-01, 4.717649724219E-01, 1.015744035042E-01, -2.653592939274E-01,
       -4.596429798148E-01, -6.041328236203E-01, -1.620453412699E+00, -2.636774001778E+00,
       -3.653094590856E+00, -7.874665253011E+00, -5.471668476150E+00, -3.068671699289E+00,
       -6.656749224279E-01, -2.653592939274E-01, 1.353240031925E-01, 5.822972777850E-01,
       9.392642777670E-01, 1.043123512841E+00, 1.020213006502E+00, 9.201634736832E-01,
       8.099953497724E-01, 7.837181426811E-01, 8.099953497724E-01, 9.201634736832E-01,
       1.020213006502E+00, 1.043123512841E+00, 9.392642777670E-01, 5.822972777850E-01,
       1.353240031925E-01, -2.653592939274E-01, -6.656749224279E-01, -3.068671699289E+00,
       -5.471668476150E+00, -7.874665253011E+00, -9.806984369113E+00, -6.690199875043E+00,
       -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01, 5.822972777850E-01,
       8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01, 5.573195659183E-01,
       2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02, 9.071160757681E-02,
       2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01, 1.001366113450E+00,
       8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01, -4.566308869047E-01,
       -3.573415380974E+00, -6.690199875043E+00, -9.806984369113E+00, -9.931104027256E+00,
       -6.669993535749E+00, -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01,
       9.392642777670E-01, 1.001366113450E+00, 8.099953497724E-01, 3.708915411190E-01,
       -1.307486146132E-01, -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01,
       -7.603083192388E-01, -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01,
       8.099953497724E-01, 1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01,
       -1.477725527353E-01, -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00,
       -7.799246504079E+00, -5.146535499003E+00, -2.493824493927E+00, 1.588865111491E-01,
       6.982843228335E-01, 1.043123512841E+00, 8.346976393096E-01, 3.708915411190E-01,
       -2.159029407473E-01, -7.879132544621E-01, -1.137317867441E+00, -1.341871880972E+00,
       -1.411116709286E+00, -1.341871880972E+00, -1.137317867441E+00, -7.879132544621E-01,
       -2.159029407473E-01, 3.708915411190E-01, 8.346976393096E-01, 1.043123512841E+00,
       6.982843228335E-01, 1.588865111491E-01, -2.493824493927E+00, -5.146535499003E+00,
       -7.799246504079E+00, -4.821035766484E+00, -3.068661134588E+00, -1.316286502692E+00,
       4.360881292036E-01, 8.365934740037E-01, 1.020213006502E+00, 5.573195659183E-01,
       -1.307486146132E-01, -7.879132544621E-01, -1.348021656895E+00, -1.611357082932E+00,
       -1.722533378684E+00, -1.769347343907E+00, -1.722533378684E+00, -1.611357082932E+00,
       -1.348021656895E+00, -7.879132544621E-01, -1.307486146132E-01, 5.573195659183E-01,
       1.020213006502E+00, 8.365934740037E-01, 4.360881292036E-01, -1.316286502692E+00,
       -3.068661134588E+00, -4.821035766484E+00, -2.144463029636E+00, -1.225075078640E+00,
       -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01, 9.201634736832E-01,
       2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00, -5.043249647001E-01,
       2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01, 6.137008233513E-01,
       -3.056871276443E-01, -1.225075078640E+00, -2.144463029636E+00, -1.066488253839E-01,
       1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01,
       8.099953497724E-01, 9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00,
       -1.722533378684E+00, -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00,
       -1.685620535521E+00, -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00,
       -7.603083192388E-01, 9.071160757681E-02, 8.099953497724E-01, 8.680361464225E-01,
       7.183309322567E-01, 4.433376797099E-01, 1.683444271630E-01, -1.066488253839E-01,
       4.826495422536E-01, 5.729701223819E-01, 6.632907025101E-01, 7.536112826384E-01,
       8.719664804748E-01, 7.837181426811E-01, 3.238561278413E-02, -8.390032823170E-01,
       -1.411116709286E+00, -1.769347343907E+00, -1.783323390866E+00, -1.681006430126E+00,
       -1.669963815903E+00, -1.681006430126E+00, -1.783323390866E+00, -1.769347343907E+00,
       -1.411116709286E+00, -8.390032823170E-01, 3.238561278413E-02, 7.837181426811E-01,
       8.719664804748E-01, 7.536112826384E-01, 6.632907025101E-01, 5.729701223819E-01,
       4.826495422536E-01, -1.066488253839E-01, 1.683444271630E-01, 4.433376797099E-01,
       7.183309322567E-01, 8.680361464225E-01, 8.099953497724E-01, 9.071160757681E-02,
       -7.603083192388E-01, -1.341871880972E+00, -1.722533378684E+00, -1.764216023318E+00,
       -1.685620535521E+00, -1.681006430126E+00, -1.685620535521E+00, -1.764216023318E+00,
       -1.722533378684E+00, -1.341871880972E+00, -7.603083192388E-01, 9.071160757681E-02,
       8.099953497724E-01, 8.680361464225E-01, 7.183309322567E-01, 4.433376797099E-01,
       1.683444271630E-01, -1.066488253839E-01, -2.144463029636E+00, -1.225075078640E+00,
       -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01, 9.201634736832E-01,
       2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00, -5.043249647001E-01,
       2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01, 6.137008233513E-01,
       -3.056871276443E-01, -1.225075078640E+00, -2.144463029636E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01,
       1.020213006502E+00, 5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01,
       -1.348021656895E+00, -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00,
       -1.722533378684E+00, -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01,
       -1.307486146132E-01, 5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01,
       4.360881292036E-01, -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00,
       -7.799246504079E+00, -5.146535499003E+00, -2.493824493927E+00, 1.588865111491E-01,
       6.982843228335E-01, 1.043123512841E+00, 8.346976393096E-01, 3.708915411190E-01,
       -2.159029407473E-01, -7.879132544621E-01, -1.137317867441E+00, -1.341871880972E+00,
       -1.411116709286E+00, -1.341871880972E+00, -1.137317867441E+00, -7.879132544621E-01,
       -2.159029407473E-01, 3.708915411190E-01, 8.346976393096E-01, 1.043123512841E+00,
       6.982843228335E-01, 1.588865111491E-01, -2.493824493927E+00, -5.146535499003E+00,
       -7.799246504079E+00, -9.931104027256E+00, -6.669993535749E+00, -3.408883044242E+00,
       -1.477725527353E-01, 4.717649724219E-01, 9.392642777670E-01, 1.001366113450E+00,
       8.099953497724E-01, 3.708915411190E-01, -1.307486146132E-01, -5.043249647001E-01,
       -7.603083192388E-01, -8.390032823170E-01, -7.603083192388E-01, -5.043249647001E-01,
       -1.307486146132E-01, 3.708915411190E-01, 8.099953497724E-01, 1.001366113450E+00,
       9.392642777670E-01, 4.717649724219E-01, -1.477725527353E-01, -3.408883044242E+00,
       -6.669993535749E+00, -9.931104027256E+00, -9.806984369113E+00, -6.690199875043E+00,
       -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01, 5.822972777850E-01,
       8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01, 5.573195659183E-01,
       2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02, 9.071160757681E-02,
       2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01, 1.001366113450E+00,
       8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01, -4.566308869047E-01,
       -3.573415380974E+00, -6.690199875043E+00, -9.806984369113E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01,
       1.353240031925E-01, 5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00,
       1.020213006502E+00, 9.201634736832E-01, 8.099953497724E-01, 7.837181426811E-01,
       8.099953497724E-01, 9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00,
       9.392642777670E-01, 5.822972777850E-01, 1.353240031925E-01, -2.653592939274E-01,
       -6.656749224279E-01, -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00,
       -3.653094590856E+00, -2.636774001778E+00, -1.620453412699E+00, -6.041328236203E-01,
       -4.596429798148E-01, -2.653592939274E-01, 1.015744035042E-01, 4.717649724219E-01,
       6.982843228335E-01, 8.365934740037E-01, 8.751165915620E-01, 8.680361464225E-01,
       8.719664804748E-01, 8.680361464225E-01, 8.751165915620E-01, 8.365934740037E-01,
       6.982843228335E-01, 4.717649724219E-01, 1.015744035042E-01, -2.653592939274E-01,
       -4.596429798148E-01, -6.041328236203E-01, -1.620453412699E+00, -2.636774001778E+00,
       -3.653094590856E+00, 1.480862149879E+00, 8.369010281869E-01, 1.929399064948E-01,
       -4.510212151972E-01, -6.041328236203E-01, -6.656749224279E-01, -4.566308869047E-01,
       -1.477725527353E-01, 1.588865111491E-01, 4.360881292036E-01, 6.137008233513E-01,
       7.183309322567E-01, 7.536112826384E-01, 7.183309322567E-01, 6.137008233513E-01,
       4.360881292036E-01, 1.588865111491E-01, -1.477725527353E-01, -4.566308869047E-01,
       -6.656749224279E-01, -6.041328236203E-01, -4.510212151972E-01, 1.929399064949E-01,
       8.369010281870E-01, 1.480862149879E+00, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00, -3.068671699289E+00,
       -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00, -1.316286502692E+00,
       -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01, 4.433376797099E-01,
       -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00, -3.408883044242E+00,
       -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00, 1.929399064949E-01,
       9.977774723847E+00, 1.976260954120E+01, 2.954744435855E+01, 5.761402656722E+01,
       3.868831805421E+01, 1.976260954120E+01, 8.369010281869E-01, -2.636774001778E+00,
       -5.471668476150E+00, -6.690199875043E+00, -6.669993535749E+00, -5.146535499003E+00,
       -3.068661134588E+00, -1.225075078640E+00, 1.683444271630E-01, 5.729701223819E-01,
       1.683444271630E-01, -1.225075078640E+00, -3.068661134588E+00, -5.146535499003E+00,
       -6.669993535749E+00, -6.690199875043E+00, -5.471668476150E+00, -2.636774001778E+00,
       8.369010281870E-01, 1.976260954120E+01, 3.868831805421E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01, 1.480862149879E+00,
       -3.653094590856E+00, -7.874665253011E+00, -9.806984369113E+00, -9.931104027256E+00,
       -7.799246504079E+00, -4.821035766484E+00, -2.144463029636E+00, -1.066488253839E-01,
       4.826495422536E-01, -1.066488253839E-01, -2.144463029636E+00, -4.821035766484E+00,
       -7.799246504079E+00, -9.931104027256E+00, -9.806984369113E+00, -7.874665253011E+00,
       -3.653094590856E+00, 1.480862149879E+00, 2.954744435855E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00,
       -6.669993535749E+00, -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00,
       1.683444271630E-01, 5.729701223819E-01, 1.683444271630E-01, -1.225075078640E+00,
       -3.068661134588E+00, -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00,
       -5.471668476150E+00, -2.636774001778E+00, 8.369010281869E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 3.868831805421E+01, 2.594265109142E+01,
       1.319698412863E+01, 4.513171658361E-01, -1.896156512223E+00, -3.797238488428E+00,
       -4.559820735228E+00, -4.474660527854E+00, -3.389119766508E+00, -1.941276548716E+00,
       -6.724412439495E-01, 2.797210634251E-01, 5.573457649428E-01, 2.797210634251E-01,
       -6.724412439495E-01, -1.941276548716E+00, -3.389119766508E+00, -4.474660527854E+00,
       -4.559820735228E+00, -3.797238488428E+00, -1.896156512223E+00, 4.513171658361E-01,
       1.319698412863E+01, 2.594265109142E+01, 3.868831805421E+01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.155539022668E+00,
       -2.122808500705E+00, -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00,
       -8.138919628448E-01, -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01,
       3.910976996873E-01, -1.198074092591E-01, -8.138919628448E-01, -1.631704034013E+00,
       -2.279327519959E+00, -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00,
       6.573330348527E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       8.369010281869E-01, 4.513171658361E-01, 6.573330348527E-02, -3.198505588656E-01,
       -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01, -8.399451206323E-02,
       1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01, 5.024743359494E-01,
       5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01, 3.134926230269E-01,
       1.257116984816E-01, -8.399451206323E-02, -2.990624555983E-01, -4.483785129825E-01,
       -4.149215331138E-01, -3.198505588656E-01, 6.573330348527E-02, 4.513171658361E-01,
       8.369010281869E-01, -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00,
       -4.149215331138E-01, -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02,
       3.343918816587E-01, 4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01,
       5.904884624947E-01, 5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01,
       5.746842652299E-01, 4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02,
       -1.680490365956E-01, -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00,
       -1.896156512223E+00, -2.636774001778E+00, -5.471668476150E+00, -3.797238488428E+00,
       -2.122808500705E+00, -4.483785129825E-01, -1.680490365956E-01, 1.097648162584E-01,
       4.110639243400E-01, 6.477831572352E-01, 7.115169426466E-01, 6.892874849841E-01,
       6.170377036614E-01, 5.395998813701E-01, 5.208889309184E-01, 5.395998813701E-01,
       6.170377036614E-01, 6.892874849841E-01, 7.115169426466E-01, 6.477831572352E-01,
       4.110639243400E-01, 1.097648162584E-01, -1.680490365956E-01, -4.483785129825E-01,
       -2.122808500705E+00, -3.797238488428E+00, -5.471668476150E+00, -6.690199875043E+00,
       -4.559820735228E+00, -2.429441595413E+00, -2.990624555983E-01, 8.358267959076E-02,
       4.110639243400E-01, 6.094930270000E-01, 6.798791807600E-01, 5.596158921156E-01,
       3.656619694996E-01, 1.856684861198E-01, 4.495215414751E-02, 4.863427228491E-03,
       4.495215414751E-02, 1.856684861198E-01, 3.656619694996E-01, 5.596158921156E-01,
       6.798791807600E-01, 6.094930270000E-01, 4.110639243400E-01, 8.358267959076E-02,
       -2.990624555983E-01, -2.429441595413E+00, -4.559820735228E+00, -6.690199875043E+00,
       -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00, -8.399451206323E-02,
       3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01, 5.395998813701E-01,
       2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01, -5.344962345675E-01,
       -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01, -1.076250326493E-01,
       2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01, 6.477831572352E-01,
       3.343918816587E-01, -8.399451206323E-02, -2.279327519959E+00, -4.474660527854E+00,
       -6.669993535749E+00, -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00,
       1.257116984816E-01, 4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01,
       2.359054197387E-01, -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01,
       -9.244007997424E-01, -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01,
       -5.527647108636E-01, -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01,
       7.115169426466E-01, 4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00,
       -3.389119766508E+00, -5.146535499003E+00, -3.068661134588E+00, -1.941276548716E+00,
       -8.138919628448E-01, 3.134926230269E-01, 5.746842652299E-01, 6.892874849841E-01,
       3.656619694996E-01, -1.076250326493E-01, -5.527647108636E-01, -9.291220640084E-01,
       -1.103792587007E+00, -1.175775981086E+00, -1.206607720720E+00, -1.175775981086E+00,
       -1.103792587007E+00, -9.291220640084E-01, -5.527647108636E-01, -1.076250326493E-01,
       3.656619694996E-01, 6.892874849841E-01, 5.746842652299E-01, 3.134926230269E-01,
       -8.138919628448E-01, -1.941276548716E+00, -3.068661134588E+00, -1.225075078640E+00,
       -6.724412439495E-01, -1.198074092591E-01, 4.328264254313E-01, 5.976046378355E-01,
       6.170377036614E-01, 1.856684861198E-01, -3.615121907346E-01, -7.878746384657E-01,
       -1.103792587007E+00, -1.197682062213E+00, -1.198786487885E+00, -1.210660725495E+00,
       -1.198786487885E+00, -1.197682062213E+00, -1.103792587007E+00, -7.878746384657E-01,
       -3.615121907346E-01, 1.856684861198E-01, 6.170377036614E-01, 5.976046378355E-01,
       4.328264254313E-01, -1.198074092591E-01, -6.724412439495E-01, -1.225075078640E+00,
       1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01, 5.024743359494E-01,
       5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02, -5.344962345675E-01,
       -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00, -1.141637026370E+00,
       -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00, -1.175775981086E+00,
       -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02, 5.395998813701E-01,
       5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01, 2.797210634251E-01,
       1.683444271630E-01, 5.729701223819E-01, 5.573457649428E-01, 5.417214075038E-01,
       5.260970500647E-01, 5.924639871528E-01, 5.208889309184E-01, 4.863427228491E-03,
       -5.878551348073E-01, -9.708553108523E-01, -1.206607720720E+00, -1.210660725495E+00,
       -1.137369657030E+00, -1.128714899069E+00, -1.137369657030E+00, -1.210660725495E+00,
       -1.206607720720E+00, -9.708553108523E-01, -5.878551348073E-01, 4.863427228491E-03,
       5.208889309184E-01, 5.924639871528E-01, 5.260970500647E-01, 5.417214075038E-01,
       5.573457649428E-01, 5.729701223819E-01, 1.683444271630E-01, 2.797210634251E-01,
       3.910976996873E-01, 5.024743359494E-01, 5.904884624947E-01, 5.395998813701E-01,
       4.495215414751E-02, -5.344962345675E-01, -9.244007997424E-01, -1.175775981086E+00,
       -1.198786487885E+00, -1.141637026370E+00, -1.137369657030E+00, -1.141637026370E+00,
       -1.198786487885E+00, -1.175775981086E+00, -9.244007997424E-01, -5.344962345675E-01,
       4.495215414751E-02, 5.395998813701E-01, 5.904884624947E-01, 5.024743359494E-01,
       3.910976996873E-01, 2.797210634251E-01, 1.683444271630E-01, -1.225075078640E+00,
       -6.724412439495E-01, -1.198074092591E-01, 4.328264254313E-01, 5.976046378355E-01,
       6.170377036614E-01, 1.856684861198E-01, -3.615121907346E-01, -7.878746384657E-01,
       -1.103792587007E+00, -1.197682062213E+00, -1.198786487885E+00, -1.210660725495E+00,
       -1.198786487885E+00, -1.197682062213E+00, -1.103792587007E+00, -7.878746384657E-01,
       -3.615121907346E-01, 1.856684861198E-01, 6.170377036614E-01, 5.976046378355E-01,
       4.328264254313E-01, -1.198074092591E-01, -6.724412439495E-01, -1.225075078640E+00,
       -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01, 3.134926230269E-01,
       5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01, -1.076250326493E-01,
       -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00, -1.175775981086E+00,
       -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00, -9.291220640084E-01,
       -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01, 6.892874849841E-01,
       5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01, -1.941276548716E+00,
       -3.068661134588E+00, -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00,
       1.257116984816E-01, 4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01,
       2.359054197387E-01, -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01,
       -9.244007997424E-01, -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01,
       -5.527647108636E-01, -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01,
       7.115169426466E-01, 4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00,
       -3.389119766508E+00, -5.146535499003E+00, -6.669993535749E+00, -4.474660527854E+00,
       -2.279327519959E+00, -8.399451206323E-02, 3.343918816587E-01, 6.477831572352E-01,
       6.798791807600E-01, 5.395998813701E-01, 2.359054197387E-01, -1.076250326493E-01,
       -3.615121907346E-01, -5.344962345675E-01, -5.878551348073E-01, -5.344962345675E-01,
       -3.615121907346E-01, -1.076250326493E-01, 2.359054197387E-01, 5.395998813701E-01,
       6.798791807600E-01, 6.477831572352E-01, 3.343918816587E-01, -8.399451206323E-02,
       -2.279327519959E+00, -4.474660527854E+00, -6.669993535749E+00, -6.690199875043E+00,
       -4.559820735228E+00, -2.429441595413E+00, -2.990624555983E-01, 8.358267959076E-02,
       4.110639243400E-01, 6.094930270000E-01, 6.798791807600E-01, 5.596158921156E-01,
       3.656619694996E-01, 1.856684861198E-01, 4.495215414751E-02, 4.863427228491E-03,
       4.495215414751E-02, 1.856684861198E-01, 3.656619694996E-01, 5.596158921156E-01,
       6.798791807600E-01, 6.094930270000E-01, 4.110639243400E-01, 8.358267959076E-02,
       -2.990624555983E-01, -2.429441595413E+00, -4.559820735228E+00, -6.690199875043E+00,
       -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00, -4.483785129825E-01,
       -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01, 6.477831572352E-01,
       7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01, 5.395998813701E-01,
       5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01, 6.892874849841E-01,
       7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01, 1.097648162584E-01,
       -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00, -3.797238488428E+00,
       -5.471668476150E+00, -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00,
       -4.149215331138E-01, -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02,
       3.343918816587E-01, 4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01,
       5.904884624947E-01, 5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01,
       5.746842652299E-01, 4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02,
       -1.680490365956E-01, -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00,
       -1.896156512223E+00, -2.636774001778E+00, 8.369010281869E-01, 4.513171658361E-01,
       6.573330348527E-02, -3.198505588656E-01, -4.149215331138E-01, -4.483785129825E-01,
       -2.990624555983E-01, -8.399451206323E-02, 1.257116984816E-01, 3.134926230269E-01,
       4.328264254313E-01, 5.024743359494E-01, 5.260970500647E-01, 5.024743359494E-01,
       4.328264254313E-01, 3.134926230269E-01, 1.257116984816E-01, -8.399451206323E-02,
       -2.990624555983E-01, -4.483785129825E-01, -4.149215331138E-01, -3.198505588655E-01,
       6.573330348529E-02, 4.513171658361E-01, 8.369010281870E-01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.155539022668E+00,
       -2.122808500705E+00, -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00,
       -8.138919628448E-01, -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01,
       3.910976996873E-01, -1.198074092591E-01, -8.138919628448E-01, -1.631704034013E+00,
       -2.279327519959E+00, -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00,
       6.573330348529E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       3.868831805421E+01, 2.594265109142E+01, 1.319698412863E+01, 4.513171658361E-01,
       -1.896156512223E+00, -3.797238488428E+00, -4.559820735228E+00, -4.474660527854E+00,
       -3.389119766508E+00, -1.941276548716E+00, -6.724412439495E-01, 2.797210634251E-01,
       5.573457649428E-01, 2.797210634251E-01, -6.724412439495E-01, -1.941276548716E+00,
       -3.389119766508E+00, -4.474660527854E+00, -4.559820735228E+00, -3.797238488428E+00,
       -1.896156512223E+00, 4.513171658361E-01, 1.319698412863E+01, 2.594265109142E+01,
       3.868831805421E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00,
       -6.669993535749E+00, -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00,
       1.683444271630E-01, 5.729701223819E-01, 1.683444271630E-01, -1.225075078640E+00,
       -3.068661134588E+00, -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00,
       -5.471668476150E+00, -2.636774001778E+00, 8.369010281870E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00, -3.068671699289E+00,
       -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00, -1.316286502692E+00,
       -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01, 4.433376797099E-01,
       -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00, -3.408883044242E+00,
       -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00, 1.929399064948E-01,
       9.977774723846E+00, 1.976260954120E+01, 2.954744435855E+01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.155539022668E+00,
       -2.122808500705E+00, -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00,
       -8.138919628448E-01, -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01,
       3.910976996873E-01, -1.198074092591E-01, -8.138919628448E-01, -1.631704034013E+00,
       -2.279327519959E+00, -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00,
       6.573330348527E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       9.977774723846E+00, 6.631358716056E+00, 3.284942708266E+00, -6.147329952430E-02,
       -6.906246326379E-01, -1.176945302121E+00, -1.285467809853E+00, -1.149771995675E+00,
       -7.695835740996E-01, -3.114974229973E-01, 6.607230912604E-02, 3.388577196647E-01,
       4.201521124974E-01, 3.388577196647E-01, 6.607230912604E-02, -3.114974229973E-01,
       -7.695835740996E-01, -1.149771995675E+00, -1.285467809853E+00, -1.176945302121E+00,
       -6.906246326379E-01, -6.147329952430E-02, 3.284942708266E+00, 6.631358716056E+00,
       9.977774723846E+00, 1.929399064948E-01, 6.573330348527E-02, -6.147329952430E-02,
       -1.886799025339E-01, -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01,
       -2.021647139113E-02, 9.253688581413E-02, 1.908971168502E-01, 2.519520275112E-01,
       2.866177396421E-01, 2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01,
       1.908971168502E-01, 9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01,
       -2.310821035371E-01, -2.257102426073E-01, -1.886799025339E-01, -6.147329952430E-02,
       6.573330348527E-02, 1.929399064948E-01, -1.620453412699E+00, -1.155539022668E+00,
       -6.906246326379E-01, -2.257102426073E-01, -1.552736426115E-01, -7.073877926376E-02,
       6.559095567733E-02, 1.970187908955E-01, 2.716214141788E-01, 3.127750564560E-01,
       3.200926841090E-01, 3.129407785669E-01, 3.129614938307E-01, 3.129407785669E-01,
       3.200926841090E-01, 3.127750564560E-01, 2.716214141788E-01, 1.970187908955E-01,
       6.559095567733E-02, -7.073877926376E-02, -1.552736426115E-01, -2.257102426073E-01,
       -6.906246326379E-01, -1.155539022668E+00, -1.620453412699E+00, -3.068671699289E+00,
       -2.122808500705E+00, -1.176945302121E+00, -2.310821035371E-01, -7.073877926376E-02,
       8.420562932427E-02, 2.398305708950E-01, 3.563020367034E-01, 3.799103724520E-01,
       3.583619634660E-01, 3.139119336396E-01, 2.692044129679E-01, 2.580597191557E-01,
       2.692044129679E-01, 3.139119336396E-01, 3.583619634660E-01, 3.799103724520E-01,
       3.563020367034E-01, 2.398305708950E-01, 8.420562932427E-02, -7.073877926376E-02,
       -2.310821035371E-01, -1.176945302121E+00, -2.122808500705E+00, -3.068671699289E+00,
       -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00, -1.414940242919E-01,
       6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01, 3.583922480699E-01,
       2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02, -8.072992817954E-04,
       -2.265875832714E-02, -8.072992817954E-04, 7.507429471279E-02, 1.740043730810E-01,
       2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01, 2.398305708950E-01,
       6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00, -2.429441595413E+00,
       -3.573415380974E+00, -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00,
       -2.021647139113E-02, 1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01,
       2.692044129679E-01, 1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01,
       -3.086841498962E-01, -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01,
       -8.450145068548E-02, 1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01,
       3.563020367034E-01, 1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00,
       -2.279327519959E+00, -3.408883044242E+00, -2.493824493927E+00, -1.631704034013E+00,
       -7.695835740996E-01, 9.253688581413E-02, 2.716214141788E-01, 3.799103724520E-01,
       2.845341449216E-01, 1.009192983584E-01, -1.139532796313E-01, -3.176161672652E-01,
       -4.384314094904E-01, -5.069297185127E-01, -5.305939124186E-01, -5.069297185127E-01,
       -4.384314094904E-01, -3.176161672652E-01, -1.139532796313E-01, 1.009192983584E-01,
       2.845341449216E-01, 3.799103724520E-01, 2.716214141788E-01, 9.253688581413E-02,
       -7.695835740996E-01, -1.631704034013E+00, -2.493824493927E+00, -1.316286502692E+00,
       -8.138919628448E-01, -3.114974229973E-01, 1.908971168502E-01, 3.127750564560E-01,
       3.583619634660E-01, 1.740043730810E-01, -8.450145068548E-02, -3.176161672652E-01,
       -5.102224711217E-01, -5.962280910819E-01, -6.290185834875E-01, -6.438680975332E-01,
       -6.290185834875E-01, -5.962280910819E-01, -5.102224711217E-01, -3.176161672652E-01,
       -8.450145068548E-02, 1.740043730810E-01, 3.583619634660E-01, 3.127750564560E-01,
       1.908971168502E-01, -3.114974229973E-01, -8.138919628448E-01, -1.316286502692E+00,
       -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02, 2.519520275112E-01,
       3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02, -2.186994167692E-01,
       -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01, -6.333569524524E-01,
       -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01, -5.962280910819E-01,
       -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02, 3.139119336396E-01,
       3.200926841090E-01, 2.519520275112E-01, 6.607230912604E-02, -1.198074092591E-01,
       -3.056871276443E-01, 4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01,
       2.866177396421E-01, 3.129407785669E-01, 2.692044129679E-01, -8.072992817954E-04,
       -3.086841498962E-01, -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01,
       -5.976535172184E-01, -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01,
       -6.290185834875E-01, -5.069297185127E-01, -3.086841498962E-01, -8.072992817955E-04,
       2.692044129679E-01, 3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01,
       3.910976996873E-01, 4.433376797099E-01, 6.632907025101E-01, 5.417214075038E-01,
       4.201521124974E-01, 2.985828174911E-01, 3.129614938307E-01, 2.580597191557E-01,
       -2.265875832714E-02, -3.367069872975E-01, -5.305939124186E-01, -6.438680975332E-01,
       -6.379980601237E-01, -5.937328839348E-01, -5.874659822349E-01, -5.937328839348E-01,
       -6.379980601237E-01, -6.438680975332E-01, -5.305939124186E-01, -3.367069872975E-01,
       -2.265875832714E-02, 2.580597191557E-01, 3.129614938307E-01, 2.985828174911E-01,
       4.201521124974E-01, 5.417214075038E-01, 6.632907025101E-01, 4.433376797099E-01,
       3.910976996873E-01, 3.388577196647E-01, 2.866177396421E-01, 3.129407785669E-01,
       2.692044129679E-01, -8.072992817954E-04, -3.086841498962E-01, -5.069297185127E-01,
       -6.290185834875E-01, -6.333569524524E-01, -5.976535172184E-01, -5.937328839348E-01,
       -5.976535172184E-01, -6.333569524524E-01, -6.290185834875E-01, -5.069297185127E-01,
       -3.086841498962E-01, -8.072992817955E-04, 2.692044129679E-01, 3.129407785669E-01,
       2.866177396421E-01, 3.388577196647E-01, 3.910976996873E-01, 4.433376797099E-01,
       -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02, 2.519520275112E-01,
       3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02, -2.186994167692E-01,
       -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01, -6.333569524524E-01,
       -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01, -5.962280910819E-01,
       -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02, 3.139119336396E-01,
       3.200926841090E-01, 2.519520275112E-01, 6.607230912604E-02, -1.198074092591E-01,
       -3.056871276443E-01, -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01,
       1.908971168502E-01, 3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01,
       -8.450145068548E-02, -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01,
       -6.290185834875E-01, -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01,
       -5.102224711217E-01, -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01,
       3.583619634660E-01, 3.127750564560E-01, 1.908971168502E-01, -3.114974229973E-01,
       -8.138919628448E-01, -1.316286502692E+00, -2.493824493927E+00, -1.631704034013E+00,
       -7.695835740996E-01, 9.253688581413E-02, 2.716214141788E-01, 3.799103724520E-01,
       2.845341449216E-01, 1.009192983584E-01, -1.139532796313E-01, -3.176161672652E-01,
       -4.384314094904E-01, -5.069297185127E-01, -5.305939124186E-01, -5.069297185127E-01,
       -4.384314094904E-01, -3.176161672652E-01, -1.139532796313E-01, 1.009192983584E-01,
       2.845341449216E-01, 3.799103724520E-01, 2.716214141788E-01, 9.253688581413E-02,
       -7.695835740996E-01, -1.631704034013E+00, -2.493824493927E+00, -3.408883044242E+00,
       -2.279327519959E+00, -1.149771995675E+00, -2.021647139113E-02, 1.970187908955E-01,
       3.563020367034E-01, 3.583922480699E-01, 2.692044129679E-01, 1.009192983584E-01,
       -8.450145068548E-02, -2.186994167692E-01, -3.086841498962E-01, -3.367069872975E-01,
       -3.086841498962E-01, -2.186994167692E-01, -8.450145068548E-02, 1.009192983584E-01,
       2.692044129679E-01, 3.583922480699E-01, 3.563020367034E-01, 1.970187908955E-01,
       -2.021647139113E-02, -1.149771995675E+00, -2.279327519959E+00, -3.408883044242E+00,
       -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00, -1.414940242919E-01,
       6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01, 3.583922480699E-01,
       2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02, -8.072992817955E-04,
       -2.265875832714E-02, -8.072992817955E-04, 7.507429471279E-02, 1.740043730810E-01,
       2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01, 2.398305708950E-01,
       6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00, -2.429441595413E+00,
       -3.573415380974E+00, -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00,
       -2.310821035371E-01, -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01,
       3.563020367034E-01, 3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01,
       2.692044129679E-01, 2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01,
       3.583619634660E-01, 3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01,
       8.420562932427E-02, -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00,
       -2.122808500705E+00, -3.068671699289E+00, -1.620453412699E+00, -1.155539022668E+00,
       -6.906246326379E-01, -2.257102426073E-01, -1.552736426115E-01, -7.073877926376E-02,
       6.559095567733E-02, 1.970187908955E-01, 2.716214141788E-01, 3.127750564560E-01,
       3.200926841090E-01, 3.129407785669E-01, 3.129614938307E-01, 3.129407785669E-01,
       3.200926841090E-01, 3.127750564560E-01, 2.716214141788E-01, 1.970187908955E-01,
       6.559095567733E-02, -7.073877926376E-02, -1.552736426115E-01, -2.257102426073E-01,
       -6.906246326379E-01, -1.155539022668E+00, -1.620453412699E+00, 1.929399064948E-01,
       6.573330348527E-02, -6.147329952430E-02, -1.886799025339E-01, -2.257102426073E-01,
       -2.310821035371E-01, -1.414940242919E-01, -2.021647139113E-02, 9.253688581413E-02,
       1.908971168502E-01, 2.519520275112E-01, 2.866177396421E-01, 2.985828174911E-01,
       2.866177396421E-01, 2.519520275112E-01, 1.908971168502E-01, 9.253688581413E-02,
       -2.021647139113E-02, -1.414940242919E-01, -2.310821035371E-01, -2.257102426073E-01,
       -1.886799025339E-01, -6.147329952429E-02, 6.573330348528E-02, 1.929399064949E-01,
       9.977774723846E+00, 6.631358716056E+00, 3.284942708266E+00, -6.147329952430E-02,
       -6.906246326379E-01, -1.176945302121E+00, -1.285467809853E+00, -1.149771995675E+00,
       -7.695835740996E-01, -3.114974229973E-01, 6.607230912604E-02, 3.388577196647E-01,
       4.201521124974E-01, 3.388577196647E-01, 6.607230912604E-02, -3.114974229973E-01,
       -7.695835740996E-01, -1.149771995675E+00, -1.285467809853E+00, -1.176945302121E+00,
       -6.906246326379E-01, -6.147329952429E-02, 3.284942708266E+00, 6.631358716056E+00,
       9.977774723847E+00, 1.976260954120E+01, 1.319698412863E+01, 6.631358716056E+00,
       6.573330348527E-02, -1.155539022668E+00, -2.122808500705E+00, -2.429441595413E+00,
       -2.279327519959E+00, -1.631704034013E+00, -8.138919628448E-01, -1.198074092591E-01,
       3.910976996873E-01, 5.417214075038E-01, 3.910976996873E-01, -1.198074092591E-01,
       -8.138919628448E-01, -1.631704034013E+00, -2.279327519959E+00, -2.429441595413E+00,
       -2.122808500705E+00, -1.155539022668E+00, 6.573330348528E-02, 6.631358716056E+00,
       1.319698412863E+01, 1.976260954120E+01, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00, -3.068671699289E+00,
       -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00, -1.316286502692E+00,
       -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01, 4.433376797099E-01,
       -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00, -3.408883044242E+00,
       -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00, 1.929399064949E-01,
       9.977774723847E+00, 1.976260954120E+01, 2.954744435855E+01, 1.480862149879E+00,
       8.369010281869E-01, 1.929399064948E-01, -4.510212151972E-01, -6.041328236203E-01,
       -6.656749224279E-01, -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01,
       4.360881292036E-01, 6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01,
       7.183309322567E-01, 6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01,
       -1.477725527353E-01, -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01,
       -4.510212151972E-01, 1.929399064948E-01, 8.369010281869E-01, 1.480862149879E+00,
       8.369010281869E-01, 4.513171658361E-01, 6.573330348527E-02, -3.198505588656E-01,
       -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01, -8.399451206323E-02,
       1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01, 5.024743359494E-01,
       5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01, 3.134926230269E-01,
       1.257116984816E-01, -8.399451206323E-02, -2.990624555983E-01, -4.483785129825E-01,
       -4.149215331138E-01, -3.198505588656E-01, 6.573330348527E-02, 4.513171658361E-01,
       8.369010281869E-01, 1.929399064948E-01, 6.573330348527E-02, -6.147329952430E-02,
       -1.886799025339E-01, -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01,
       -2.021647139113E-02, 9.253688581413E-02, 1.908971168502E-01, 2.519520275112E-01,
       2.866177396421E-01, 2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01,
       1.908971168502E-01, 9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01,
       -2.310821035371E-01, -2.257102426073E-01, -1.886799025339E-01, -6.147329952430E-02,
       6.573330348527E-02, 1.929399064948E-01, -4.510212151972E-01, -3.198505588656E-01,
       -1.886799025339E-01, -1.886799025339E-01, -3.198505588655E-01, -4.510212151972E-01,
       -6.041328236203E-01, -4.149215331138E-01, -2.257102426073E-01, -2.257102426073E-01,
       -4.149215331138E-01, -6.041328236203E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -4.566308869047E-01, -2.990624555983E-01, -1.414940242919E-01, -1.414940242919E-01,
       -2.990624555983E-01, -4.566308869047E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206323E-02, -1.477725527353E-01,
       1.588865111491E-01, 1.257116984816E-01, 9.253688581413E-02, 9.253688581413E-02,
       1.257116984816E-01, 1.588865111491E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       6.137008233513E-01, 4.328264254313E-01, 2.519520275112E-01, 2.519520275112E-01,
       4.328264254313E-01, 6.137008233513E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       7.536112826384E-01, 5.260970500647E-01, 2.985828174911E-01, 2.985828174911E-01,
       5.260970500647E-01, 7.536112826384E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       6.137008233513E-01, 4.328264254313E-01, 2.519520275112E-01, 2.519520275112E-01,
       4.328264254313E-01, 6.137008233513E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       1.588865111491E-01, 1.257116984816E-01, 9.253688581413E-02, 9.253688581413E-02,
       1.257116984816E-01, 1.588865111491E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206323E-02, -1.477725527353E-01,
       -4.566308869047E-01, -2.990624555983E-01, -1.414940242919E-01, -1.414940242919E-01,
       -2.990624555983E-01, -4.566308869047E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -6.041328236203E-01, -4.149215331138E-01, -2.257102426073E-01, -2.257102426073E-01,
       -4.149215331138E-01, -6.041328236203E-01, -4.510212151972E-01, -3.198505588656E-01,
       -1.886799025339E-01, -1.886799025339E-01, -3.198505588656E-01, -4.510212151972E-01,
       1.929399064948E-01, 6.573330348527E-02, -6.147329952430E-02, -1.886799025339E-01,
       -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01, -2.021647139113E-02,
       9.253688581413E-02, 1.908971168501E-01, 2.519520275112E-01, 2.866177396421E-01,
       2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01, 1.908971168501E-01,
       9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01, -2.310821035371E-01,
       -2.257102426073E-01, -1.886799025339E-01, -6.147329952430E-02, 6.573330348526E-02,
       1.929399064948E-01, 8.369010281869E-01, 4.513171658361E-01, 6.573330348527E-02,
       -3.198505588655E-01, -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01,
       -8.399451206323E-02, 1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01,
       5.024743359494E-01, 5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01,
       3.134926230269E-01, 1.257116984816E-01, -8.399451206323E-02, -2.990624555983E-01,
       -4.483785129825E-01, -4.149215331138E-01, -3.198505588656E-01, 6.573330348526E-02,
       4.513171658361E-01, 8.369010281869E-01, 1.480862149879E+00, 8.369010281869E-01,
       1.929399064948E-01, -4.510212151972E-01, -6.041328236203E-01, -6.656749224279E-01,
       -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01, 4.360881292036E-01,
       6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01, 7.183309322567E-01,
       6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01, -1.477725527353E-01,
       -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01, -4.510212151972E-01,
       1.929399064948E-01, 8.369010281869E-01, 1.480862149879E+00, -3.653094590856E+00,
       -2.636774001778E+00, -1.620453412699E+00, -6.041328236203E-01, -4.596429798148E-01,
       -2.653592939274E-01, 1.015744035042E-01, 4.717649724219E-01, 6.982843228335E-01,
       8.365934740037E-01, 8.751165915620E-01, 8.680361464225E-01, 8.719664804748E-01,
       8.680361464225E-01, 8.751165915620E-01, 8.365934740037E-01, 6.982843228335E-01,
       4.717649724219E-01, 1.015744035042E-01, -2.653592939274E-01, -4.596429798148E-01,
       -6.041328236203E-01, -1.620453412699E+00, -2.636774001778E+00, -3.653094590856E+00,
       -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00, -4.149215331138E-01,
       -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02, 3.343918816587E-01,
       4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01, 5.904884624947E-01,
       5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01, 5.746842652299E-01,
       4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02, -1.680490365956E-01,
       -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00, -1.896156512223E+00,
       -2.636774001778E+00, -1.620453412699E+00, -1.155539022668E+00, -6.906246326379E-01,
       -2.257102426073E-01, -1.552736426115E-01, -7.073877926376E-02, 6.559095567733E-02,
       1.970187908955E-01, 2.716214141788E-01, 3.127750564560E-01, 3.200926841090E-01,
       3.129407785669E-01, 3.129614938307E-01, 3.129407785669E-01, 3.200926841090E-01,
       3.127750564560E-01, 2.716214141788E-01, 1.970187908955E-01, 6.559095567733E-02,
       -7.073877926376E-02, -1.552736426115E-01, -2.257102426073E-01, -6.906246326379E-01,
       -1.155539022668E+00, -1.620453412699E+00, -6.041328236203E-01, -4.149215331138E-01,
       -2.257102426073E-01, -2.257102426073E-01, -4.149215331138E-01, -6.041328236203E-01,
       -4.596429798148E-01, -3.074583112132E-01, -1.552736426115E-01, -1.552736426115E-01,
       -3.074583112132E-01, -4.596429798148E-01, -2.653592939274E-01, -1.680490365956E-01,
       -7.073877926376E-02, -7.073877926376E-02, -1.680490365956E-01, -2.653592939274E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, 4.717649724219E-01, 3.343918816587E-01,
       1.970187908955E-01, 1.970187908955E-01, 3.343918816587E-01, 4.717649724219E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 8.365934740037E-01, 5.746842652299E-01,
       3.127750564560E-01, 3.127750564560E-01, 5.746842652299E-01, 8.365934740037E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 8.680361464225E-01, 5.904884624947E-01,
       3.129407785669E-01, 3.129407785669E-01, 5.904884624947E-01, 8.680361464225E-01,
       8.719664804748E-01, 5.924639871528E-01, 3.129614938307E-01, 3.129614938307E-01,
       5.924639871528E-01, 8.719664804748E-01, 8.680361464225E-01, 5.904884624947E-01,
       3.129407785669E-01, 3.129407785669E-01, 5.904884624947E-01, 8.680361464225E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 8.365934740037E-01, 5.746842652299E-01,
       3.127750564560E-01, 3.127750564560E-01, 5.746842652299E-01, 8.365934740037E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 4.717649724219E-01, 3.343918816587E-01,
       1.970187908955E-01, 1.970187908955E-01, 3.343918816587E-01, 4.717649724219E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, -2.653592939274E-01, -1.680490365956E-01,
       -7.073877926376E-02, -7.073877926376E-02, -1.680490365956E-01, -2.653592939274E-01,
       -4.596429798148E-01, -3.074583112132E-01, -1.552736426115E-01, -1.552736426115E-01,
       -3.074583112132E-01, -4.596429798148E-01, -6.041328236203E-01, -4.149215331138E-01,
       -2.257102426073E-01, -2.257102426073E-01, -4.149215331138E-01, -6.041328236203E-01,
       -1.620453412699E+00, -1.155539022668E+00, -6.906246326379E-01, -2.257102426073E-01,
       -1.552736426115E-01, -7.073877926376E-02, 6.559095567733E-02, 1.970187908955E-01,
       2.716214141788E-01, 3.127750564560E-01, 3.200926841090E-01, 3.129407785669E-01,
       3.129614938307E-01, 3.129407785669E-01, 3.200926841090E-01, 3.127750564560E-01,
       2.716214141788E-01, 1.970187908955E-01, 6.559095567733E-02, -7.073877926376E-02,
       -1.552736426115E-01, -2.257102426073E-01, -6.906246326379E-01, -1.155539022669E+00,
       -1.620453412699E+00, -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00,
       -4.149215331138E-01, -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02,
       3.343918816587E-01, 4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01,
       5.904884624947E-01, 5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01,
       5.746842652299E-01, 4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02,
       -1.680490365956E-01, -3.074583112132E-01, -4.149215331138E-01, -1.155539022669E+00,
       -1.896156512223E+00, -2.636774001778E+00, -3.653094590856E+00, -2.636774001778E+00,
       -1.620453412699E+00, -6.041328236203E-01, -4.596429798148E-01, -2.653592939274E-01,
       1.015744035042E-01, 4.717649724219E-01, 6.982843228335E-01, 8.365934740037E-01,
       8.751165915620E-01, 8.680361464225E-01, 8.719664804748E-01, 8.680361464225E-01,
       8.751165915620E-01, 8.365934740037E-01, 6.982843228335E-01, 4.717649724219E-01,
       1.015744035042E-01, -2.653592939274E-01, -4.596429798148E-01, -6.041328236203E-01,
       -1.620453412699E+00, -2.636774001778E+00, -3.653094590857E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01,
       1.353240031925E-01, 5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00,
       1.020213006502E+00, 9.201634736832E-01, 8.099953497724E-01, 7.837181426811E-01,
       8.099953497724E-01, 9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00,
       9.392642777670E-01, 5.822972777850E-01, 1.353240031925E-01, -2.653592939274E-01,
       -6.656749224279E-01, -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00, -4.483785129825E-01,
       -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01, 6.477831572352E-01,
       7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01, 5.395998813701E-01,
       5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01, 6.892874849841E-01,
       7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01, 1.097648162584E-01,
       -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00, -3.797238488428E+00,
       -5.471668476150E+00, -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00,
       -2.310821035371E-01, -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01,
       3.563020367034E-01, 3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01,
       2.692044129679E-01, 2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01,
       3.583619634660E-01, 3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01,
       8.420562932427E-02, -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00,
       -2.122808500705E+00, -3.068671699289E+00, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -2.653592939274E-01, -1.680490365956E-01, -7.073877926376E-02, -7.073877926376E-02,
       -1.680490365956E-01, -2.653592939274E-01, 1.353240031925E-01, 1.097648162584E-01,
       8.420562932427E-02, 8.420562932427E-02, 1.097648162584E-01, 1.353240031924E-01,
       5.822972777850E-01, 4.110639243400E-01, 2.398305708950E-01, 2.398305708950E-01,
       4.110639243400E-01, 5.822972777850E-01, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       1.043123512841E+00, 7.115169426466E-01, 3.799103724520E-01, 3.799103724520E-01,
       7.115169426466E-01, 1.043123512841E+00, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       9.201634736832E-01, 6.170377036614E-01, 3.139119336396E-01, 3.139119336396E-01,
       6.170377036614E-01, 9.201634736832E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       7.837181426811E-01, 5.208889309184E-01, 2.580597191557E-01, 2.580597191557E-01,
       5.208889309184E-01, 7.837181426811E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       9.201634736832E-01, 6.170377036614E-01, 3.139119336396E-01, 3.139119336396E-01,
       6.170377036614E-01, 9.201634736832E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       1.043123512841E+00, 7.115169426466E-01, 3.799103724520E-01, 3.799103724520E-01,
       7.115169426466E-01, 1.043123512841E+00, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       5.822972777850E-01, 4.110639243400E-01, 2.398305708950E-01, 2.398305708950E-01,
       4.110639243400E-01, 5.822972777850E-01, 1.353240031925E-01, 1.097648162584E-01,
       8.420562932427E-02, 8.420562932427E-02, 1.097648162584E-01, 1.353240031924E-01,
       -2.653592939274E-01, -1.680490365956E-01, -7.073877926376E-02, -7.073877926376E-02,
       -1.680490365956E-01, -2.653592939274E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00, -2.310821035371E-01,
       -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01, 3.563020367034E-01,
       3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01, 2.692044129679E-01,
       2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01, 3.583619634660E-01,
       3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01, 8.420562932427E-02,
       -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00, -2.122808500705E+00,
       -3.068671699289E+00, -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00,
       -4.483785129825E-01, -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01,
       6.477831572352E-01, 7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01,
       5.395998813701E-01, 5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01,
       6.892874849841E-01, 7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01,
       1.097648162584E-01, -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00,
       -3.797238488428E+00, -5.471668476150E+00, -7.874665253011E+00, -5.471668476150E+00,
       -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01, 1.353240031924E-01,
       5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00, 1.020213006502E+00,
       9.201634736832E-01, 8.099953497723E-01, 7.837181426811E-01, 8.099953497723E-01,
       9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00, 9.392642777670E-01,
       5.822972777850E-01, 1.353240031924E-01, -2.653592939274E-01, -6.656749224279E-01,
       -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00, -9.806984369113E+00,
       -6.690199875043E+00, -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01,
       5.822972777850E-01, 8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01,
       5.573195659183E-01, 2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02,
       9.071160757681E-02, 2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01,
       1.001366113450E+00, 8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01,
       -4.566308869047E-01, -3.573415380974E+00, -6.690199875043E+00, -9.806984369113E+00,
       -6.690199875043E+00, -4.559820735228E+00, -2.429441595413E+00, -2.990624555983E-01,
       8.358267959076E-02, 4.110639243400E-01, 6.094930270000E-01, 6.798791807600E-01,
       5.596158921156E-01, 3.656619694996E-01, 1.856684861198E-01, 4.495215414751E-02,
       4.863427228491E-03, 4.495215414751E-02, 1.856684861198E-01, 3.656619694996E-01,
       5.596158921156E-01, 6.798791807600E-01, 6.094930270000E-01, 4.110639243400E-01,
       8.358267959076E-02, -2.990624555983E-01, -2.429441595413E+00, -4.559820735228E+00,
       -6.690199875043E+00, -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00,
       -1.414940242919E-01, 6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01,
       3.583922480699E-01, 2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02,
       -8.072992817954E-04, -2.265875832714E-02, -8.072992817954E-04, 7.507429471279E-02,
       1.740043730810E-01, 2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01,
       2.398305708950E-01, 6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00,
       -2.429441595413E+00, -3.573415380974E+00, -4.566308869047E-01, -2.990624555983E-01,
       -1.414940242919E-01, -1.414940242919E-01, -2.990624555983E-01, -4.566308869047E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, 5.822972777850E-01, 4.110639243400E-01,
       2.398305708950E-01, 2.398305708950E-01, 4.110639243400E-01, 5.822972777850E-01,
       8.845176151315E-01, 6.094930270000E-01, 3.344684388685E-01, 3.344684388685E-01,
       6.094930270000E-01, 8.845176151314E-01, 1.001366113450E+00, 6.798791807600E-01,
       3.583922480699E-01, 3.583922480699E-01, 6.798791807600E-01, 1.001366113450E+00,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 5.573195659183E-01, 3.656619694996E-01,
       1.740043730810E-01, 1.740043730810E-01, 3.656619694996E-01, 5.573195659183E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 9.071160757681E-02, 4.495215414751E-02,
       -8.072992817954E-04, -8.072992817951E-04, 4.495215414751E-02, 9.071160757681E-02,
       3.238561278413E-02, 4.863427228491E-03, -2.265875832714E-02, -2.265875832714E-02,
       4.863427228492E-03, 3.238561278413E-02, 9.071160757681E-02, 4.495215414751E-02,
       -8.072992817954E-04, -8.072992817951E-04, 4.495215414751E-02, 9.071160757681E-02,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 5.573195659183E-01, 3.656619694996E-01,
       1.740043730810E-01, 1.740043730810E-01, 3.656619694996E-01, 5.573195659183E-01,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 1.001366113450E+00, 6.798791807600E-01,
       3.583922480699E-01, 3.583922480699E-01, 6.798791807600E-01, 1.001366113450E+00,
       8.845176151315E-01, 6.094930270000E-01, 3.344684388685E-01, 3.344684388685E-01,
       6.094930270000E-01, 8.845176151314E-01, 5.822972777850E-01, 4.110639243400E-01,
       2.398305708950E-01, 2.398305708950E-01, 4.110639243400E-01, 5.822972777850E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, -4.566308869047E-01, -2.990624555983E-01,
       -1.414940242919E-01, -1.414940242919E-01, -2.990624555983E-01, -4.566308869047E-01,
       -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00, -1.414940242919E-01,
       6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01, 3.583922480699E-01,
       2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02, -8.072992817951E-04,
       -2.265875832714E-02, -8.072992817951E-04, 7.507429471279E-02, 1.740043730810E-01,
       2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01, 2.398305708950E-01,
       6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00, -2.429441595413E+00,
       -3.573415380974E+00, -6.690199875043E+00, -4.559820735228E+00, -2.429441595413E+00,
       -2.990624555983E-01, 8.358267959076E-02, 4.110639243400E-01, 6.094930270000E-01,
       6.798791807600E-01, 5.596158921156E-01, 3.656619694996E-01, 1.856684861198E-01,
       4.495215414751E-02, 4.863427228492E-03, 4.495215414751E-02, 1.856684861198E-01,
       3.656619694996E-01, 5.596158921156E-01, 6.798791807600E-01, 6.094930270000E-01,
       4.110639243400E-01, 8.358267959076E-02, -2.990624555983E-01, -2.429441595413E+00,
       -4.559820735228E+00, -6.690199875043E+00, -9.806984369113E+00, -6.690199875043E+00,
       -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01, 5.822972777850E-01,
       8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01, 5.573195659183E-01,
       2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02, 9.071160757681E-02,
       2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01, 1.001366113450E+00,
       8.845176151314E-01, 5.822972777850E-01, 1.015744035042E-01, -4.566308869047E-01,
       -3.573415380974E+00, -6.690199875043E+00, -9.806984369112E+00, -9.931104027256E+00,
       -6.669993535749E+00, -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01,
       9.392642777670E-01, 1.001366113450E+00, 8.099953497724E-01, 3.708915411190E-01,
       -1.307486146132E-01, -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01,
       -7.603083192388E-01, -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01,
       8.099953497724E-01, 1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01,
       -1.477725527353E-01, -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00,
       -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00, -8.399451206323E-02,
       3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01, 5.395998813701E-01,
       2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01, -5.344962345675E-01,
       -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01, -1.076250326493E-01,
       2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01, 6.477831572352E-01,
       3.343918816587E-01, -8.399451206323E-02, -2.279327519959E+00, -4.474660527854E+00,
       -6.669993535749E+00, -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00,
       -2.021647139113E-02, 1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01,
       2.692044129679E-01, 1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01,
       -3.086841498962E-01, -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01,
       -8.450145068548E-02, 1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01,
       3.563020367034E-01, 1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00,
       -2.279327519959E+00, -3.408883044242E+00, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206323E-02, -1.477725527353E-01,
       4.717649724219E-01, 3.343918816587E-01, 1.970187908955E-01, 1.970187908955E-01,
       3.343918816587E-01, 4.717649724219E-01, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       1.001366113450E+00, 6.798791807600E-01, 3.583922480699E-01, 3.583922480699E-01,
       6.798791807600E-01, 1.001366113450E+00, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       3.708915411190E-01, 2.359054197387E-01, 1.009192983584E-01, 1.009192983584E-01,
       2.359054197387E-01, 3.708915411190E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       -5.043249647001E-01, -3.615121907346E-01, -2.186994167692E-01, -2.186994167692E-01,
       -3.615121907346E-01, -5.043249647001E-01, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -8.390032823170E-01, -5.878551348073E-01, -3.367069872975E-01, -3.367069872975E-01,
       -5.878551348073E-01, -8.390032823170E-01, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -5.043249647001E-01, -3.615121907346E-01, -2.186994167692E-01, -2.186994167692E-01,
       -3.615121907346E-01, -5.043249647001E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       3.708915411190E-01, 2.359054197387E-01, 1.009192983584E-01, 1.009192983584E-01,
       2.359054197387E-01, 3.708915411190E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       1.001366113450E+00, 6.798791807600E-01, 3.583922480699E-01, 3.583922480699E-01,
       6.798791807600E-01, 1.001366113450E+00, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       4.717649724219E-01, 3.343918816587E-01, 1.970187908955E-01, 1.970187908955E-01,
       3.343918816587E-01, 4.717649724219E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206322E-02, -1.477725527353E-01,
       -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00, -2.021647139113E-02,
       1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01, 2.692044129679E-01,
       1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01, -3.086841498962E-01,
       -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01, -8.450145068548E-02,
       1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01, 3.563020367034E-01,
       1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00, -2.279327519959E+00,
       -3.408883044242E+00, -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00,
       -8.399451206323E-02, 3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01,
       5.395998813701E-01, 2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01,
       -5.344962345675E-01, -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01,
       -1.076250326493E-01, 2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01,
       6.477831572352E-01, 3.343918816587E-01, -8.399451206322E-02, -2.279327519959E+00,
       -4.474660527854E+00, -6.669993535749E+00, -9.931104027256E+00, -6.669993535749E+00,
       -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01, 9.392642777670E-01,
       1.001366113450E+00, 8.099953497723E-01, 3.708915411190E-01, -1.307486146132E-01,
       -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01, -7.603083192388E-01,
       -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01, 8.099953497723E-01,
       1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01, -1.477725527353E-01,
       -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00, -7.799246504079E+00,
       -5.146535499003E+00, -2.493824493927E+00, 1.588865111491E-01, 6.982843228335E-01,
       1.043123512841E+00, 8.346976393096E-01, 3.708915411190E-01, -2.159029407473E-01,
       -7.879132544621E-01, -1.137317867441E+00, -1.341871880972E+00, -1.411116709286E+00,
       -1.341871880972E+00, -1.137317867441E+00, -7.879132544621E-01, -2.159029407473E-01,
       3.708915411190E-01, 8.346976393096E-01, 1.043123512841E+00, 6.982843228335E-01,
       1.588865111491E-01, -2.493824493927E+00, -5.146535499003E+00, -7.799246504079E+00,
       -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00, 1.257116984816E-01,
       4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01, 2.359054197387E-01,
       -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01, -9.244007997424E-01,
       -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01, -5.527647108636E-01,
       -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01, 7.115169426466E-01,
       4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00, -3.389119766508E+00,
       -5.146535499003E+00, -2.493824493927E+00, -1.631704034013E+00, -7.695835740996E-01,
       9.253688581413E-02, 2.716214141788E-01, 3.799103724520E-01, 2.845341449216E-01,
       1.009192983584E-01, -1.139532796313E-01, -3.176161672652E-01, -4.384314094904E-01,
       -5.069297185127E-01, -5.305939124186E-01, -5.069297185127E-01, -4.384314094904E-01,
       -3.176161672652E-01, -1.139532796313E-01, 1.009192983584E-01, 2.845341449216E-01,
       3.799103724520E-01, 2.716214141788E-01, 9.253688581413E-02, -7.695835740996E-01,
       -1.631704034013E+00, -2.493824493927E+00, 1.588865111491E-01, 1.257116984816E-01,
       9.253688581413E-02, 9.253688581413E-02, 1.257116984816E-01, 1.588865111491E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 1.043123512841E+00, 7.115169426466E-01,
       3.799103724520E-01, 3.799103724520E-01, 7.115169426466E-01, 1.043123512841E+00,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 3.708915411190E-01, 2.359054197387E-01,
       1.009192983584E-01, 1.009192983584E-01, 2.359054197387E-01, 3.708915411190E-01,
       -2.159029407473E-01, -1.649281101893E-01, -1.139532796313E-01, -1.139532796313E-01,
       -1.649281101893E-01, -2.159029407473E-01, -7.879132544621E-01, -5.527647108636E-01,
       -3.176161672652E-01, -3.176161672652E-01, -5.527647108636E-01, -7.879132544621E-01,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -1.341871880972E+00, -9.244007997424E-01,
       -5.069297185127E-01, -5.069297185127E-01, -9.244007997424E-01, -1.341871880972E+00,
       -1.411116709286E+00, -9.708553108523E-01, -5.305939124186E-01, -5.305939124186E-01,
       -9.708553108523E-01, -1.411116709286E+00, -1.341871880972E+00, -9.244007997424E-01,
       -5.069297185127E-01, -5.069297185127E-01, -9.244007997424E-01, -1.341871880972E+00,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -7.879132544621E-01, -5.527647108636E-01,
       -3.176161672652E-01, -3.176161672652E-01, -5.527647108636E-01, -7.879132544621E-01,
       -2.159029407473E-01, -1.649281101893E-01, -1.139532796313E-01, -1.139532796313E-01,
       -1.649281101893E-01, -2.159029407473E-01, 3.708915411190E-01, 2.359054197387E-01,
       1.009192983584E-01, 1.009192983584E-01, 2.359054197387E-01, 3.708915411190E-01,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 1.043123512841E+00, 7.115169426466E-01,
       3.799103724520E-01, 3.799103724520E-01, 7.115169426466E-01, 1.043123512841E+00,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 1.588865111491E-01, 1.257116984816E-01,
       9.253688581413E-02, 9.253688581414E-02, 1.257116984816E-01, 1.588865111491E-01,
       -2.493824493927E+00, -1.631704034013E+00, -7.695835740996E-01, 9.253688581413E-02,
       2.716214141788E-01, 3.799103724520E-01, 2.845341449216E-01, 1.009192983584E-01,
       -1.139532796313E-01, -3.176161672652E-01, -4.384314094904E-01, -5.069297185127E-01,
       -5.305939124186E-01, -5.069297185127E-01, -4.384314094904E-01, -3.176161672652E-01,
       -1.139532796313E-01, 1.009192983584E-01, 2.845341449216E-01, 3.799103724520E-01,
       2.716214141788E-01, 9.253688581414E-02, -7.695835740995E-01, -1.631704034013E+00,
       -2.493824493927E+00, -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00,
       1.257116984816E-01, 4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01,
       2.359054197387E-01, -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01,
       -9.244007997424E-01, -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01,
       -5.527647108636E-01, -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01,
       7.115169426466E-01, 4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00,
       -3.389119766508E+00, -5.146535499003E+00, -7.799246504079E+00, -5.146535499003E+00,
       -2.493824493927E+00, 1.588865111491E-01, 6.982843228335E-01, 1.043123512841E+00,
       8.346976393096E-01, 3.708915411190E-01, -2.159029407473E-01, -7.879132544621E-01,
       -1.137317867441E+00, -1.341871880972E+00, -1.411116709286E+00, -1.341871880972E+00,
       -1.137317867441E+00, -7.879132544621E-01, -2.159029407473E-01, 3.708915411190E-01,
       8.346976393096E-01, 1.043123512841E+00, 6.982843228335E-01, 1.588865111491E-01,
       -2.493824493927E+00, -5.146535499003E+00, -7.799246504079E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01,
       1.020213006502E+00, 5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01,
       -1.348021656895E+00, -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00,
       -1.722533378684E+00, -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01,
       -1.307486146132E-01, 5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01,
       4.360881292036E-01, -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01, 3.134926230269E-01,
       5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01, -1.076250326493E-01,
       -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00, -1.175775981086E+00,
       -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00, -9.291220640084E-01,
       -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01, 6.892874849841E-01,
       5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01, -1.941276548716E+00,
       -3.068661134588E+00, -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01,
       1.908971168502E-01, 3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01,
       -8.450145068548E-02, -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01,
       -6.290185834875E-01, -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01,
       -5.102224711217E-01, -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01,
       3.583619634660E-01, 3.127750564560E-01, 1.908971168502E-01, -3.114974229973E-01,
       -8.138919628448E-01, -1.316286502692E+00, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       8.365934740037E-01, 5.746842652299E-01, 3.127750564560E-01, 3.127750564560E-01,
       5.746842652299E-01, 8.365934740037E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       5.573195659183E-01, 3.656619694996E-01, 1.740043730810E-01, 1.740043730810E-01,
       3.656619694996E-01, 5.573195659183E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       -7.879132544621E-01, -5.527647108636E-01, -3.176161672652E-01, -3.176161672652E-01,
       -5.527647108636E-01, -7.879132544621E-01, -1.348021656895E+00, -9.291220640084E-01,
       -5.102224711217E-01, -5.102224711217E-01, -9.291220640084E-01, -1.348021656895E+00,
       -1.611357082932E+00, -1.103792587007E+00, -5.962280910819E-01, -5.962280910819E-01,
       -1.103792587007E+00, -1.611357082932E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.769347343907E+00, -1.206607720720E+00, -6.438680975332E-01, -6.438680975332E-01,
       -1.206607720720E+00, -1.769347343907E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.611357082932E+00, -1.103792587007E+00, -5.962280910819E-01, -5.962280910819E-01,
       -1.103792587007E+00, -1.611357082932E+00, -1.348021656895E+00, -9.291220640084E-01,
       -5.102224711217E-01, -5.102224711217E-01, -9.291220640084E-01, -1.348021656895E+00,
       -7.879132544621E-01, -5.527647108636E-01, -3.176161672652E-01, -3.176161672652E-01,
       -5.527647108636E-01, -7.879132544621E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       5.573195659183E-01, 3.656619694996E-01, 1.740043730810E-01, 1.740043730810E-01,
       3.656619694996E-01, 5.573195659183E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       8.365934740037E-01, 5.746842652299E-01, 3.127750564560E-01, 3.127750564560E-01,
       5.746842652299E-01, 8.365934740037E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01, 1.908971168501E-01,
       3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01, -8.450145068548E-02,
       -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01, -6.290185834875E-01,
       -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01, -5.102224711217E-01,
       -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01, 3.583619634660E-01,
       3.127750564560E-01, 1.908971168501E-01, -3.114974229973E-01, -8.138919628448E-01,
       -1.316286502692E+00, -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01,
       3.134926230269E-01, 5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01,
       -1.076250326493E-01, -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00,
       -1.175775981086E+00, -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00,
       -9.291220640084E-01, -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01,
       6.892874849841E-01, 5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01,
       -1.941276548716E+00, -3.068661134588E+00, -4.821035766484E+00, -3.068661134588E+00,
       -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01, 1.020213006502E+00,
       5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01, -1.348021656895E+00,
       -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00, -1.722533378684E+00,
       -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01, -1.307486146132E-01,
       5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01, 4.360881292036E-01,
       -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00, -2.144463029636E+00,
       -1.225075078640E+00, -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01,
       9.201634736832E-01, 2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00,
       -1.611357082932E+00, -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00,
       -1.764216023318E+00, -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00,
       -5.043249647001E-01, 2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01,
       6.137008233513E-01, -3.056871276443E-01, -1.225075078640E+00, -2.144463029636E+00,
       -1.225075078640E+00, -6.724412439495E-01, -1.198074092591E-01, 4.328264254313E-01,
       5.976046378355E-01, 6.170377036614E-01, 1.856684861198E-01, -3.615121907346E-01,
       -7.878746384657E-01, -1.103792587007E+00, -1.197682062213E+00, -1.198786487885E+00,
       -1.210660725495E+00, -1.198786487885E+00, -1.197682062213E+00, -1.103792587007E+00,
       -7.878746384657E-01, -3.615121907346E-01, 1.856684861198E-01, 6.170377036614E-01,
       5.976046378355E-01, 4.328264254313E-01, -1.198074092591E-01, -6.724412439495E-01,
       -1.225075078640E+00, -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02,
       2.519520275112E-01, 3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02,
       -2.186994167692E-01, -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01,
       -6.333569524524E-01, -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01,
       -5.962280910819E-01, -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02,
       3.139119336396E-01, 3.200926841090E-01, 2.519520275112E-01, 6.607230912604E-02,
       -1.198074092591E-01, -3.056871276443E-01, 6.137008233513E-01, 4.328264254313E-01,
       2.519520275112E-01, 2.519520275112E-01, 4.328264254313E-01, 6.137008233513E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 9.201634736832E-01, 6.170377036614E-01,
       3.139119336396E-01, 3.139119336396E-01, 6.170377036614E-01, 9.201634736832E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, -5.043249647001E-01, -3.615121907346E-01,
       -2.186994167692E-01, -2.186994167692E-01, -3.615121907346E-01, -5.043249647001E-01,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -1.611357082932E+00, -1.103792587007E+00,
       -5.962280910819E-01, -5.962280910819E-01, -1.103792587007E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.197682062213E+00, -6.383880639117E-01, -6.383880639117E-01,
       -1.197682062213E+00, -1.756976060514E+00, -1.764216023318E+00, -1.198786487885E+00,
       -6.333569524524E-01, -6.333569524524E-01, -1.198786487885E+00, -1.764216023318E+00,
       -1.783323390866E+00, -1.210660725495E+00, -6.379980601237E-01, -6.379980601237E-01,
       -1.210660725495E+00, -1.783323390866E+00, -1.764216023318E+00, -1.198786487885E+00,
       -6.333569524524E-01, -6.333569524524E-01, -1.198786487885E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.197682062213E+00, -6.383880639117E-01, -6.383880639117E-01,
       -1.197682062213E+00, -1.756976060514E+00, -1.611357082932E+00, -1.103792587007E+00,
       -5.962280910819E-01, -5.962280910819E-01, -1.103792587007E+00, -1.611357082932E+00,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -5.043249647001E-01, -3.615121907346E-01,
       -2.186994167692E-01, -2.186994167692E-01, -3.615121907346E-01, -5.043249647001E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 9.201634736832E-01, 6.170377036614E-01,
       3.139119336396E-01, 3.139119336396E-01, 6.170377036614E-01, 9.201634736832E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 6.137008233513E-01, 4.328264254313E-01,
       2.519520275112E-01, 2.519520275112E-01, 4.328264254313E-01, 6.137008233513E-01,
       -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02, 2.519520275112E-01,
       3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02, -2.186994167692E-01,
       -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01, -6.333569524524E-01,
       -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01, -5.962280910819E-01,
       -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02, 3.139119336396E-01,
       3.200926841090E-01, 2.519520275112E-01, 6.607230912603E-02, -1.198074092592E-01,
       -3.056871276443E-01, -1.225075078640E+00, -6.724412439495E-01, -1.198074092591E-01,
       4.328264254313E-01, 5.976046378355E-01, 6.170377036614E-01, 1.856684861198E-01,
       -3.615121907346E-01, -7.878746384657E-01, -1.103792587007E+00, -1.197682062213E+00,
       -1.198786487885E+00, -1.210660725495E+00, -1.198786487885E+00, -1.197682062213E+00,
       -1.103792587007E+00, -7.878746384657E-01, -3.615121907346E-01, 1.856684861198E-01,
       6.170377036614E-01, 5.976046378355E-01, 4.328264254313E-01, -1.198074092592E-01,
       -6.724412439496E-01, -1.225075078640E+00, -2.144463029636E+00, -1.225075078640E+00,
       -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01, 9.201634736832E-01,
       2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00, -5.043249647001E-01,
       2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01, 6.137008233513E-01,
       -3.056871276443E-01, -1.225075078640E+00, -2.144463029636E+00, -1.066488253839E-01,
       1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01,
       8.099953497724E-01, 9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00,
       -1.722533378684E+00, -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00,
       -1.685620535521E+00, -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00,
       -7.603083192388E-01, 9.071160757681E-02, 8.099953497724E-01, 8.680361464225E-01,
       7.183309322567E-01, 4.433376797099E-01, 1.683444271630E-01, -1.066488253839E-01,
       1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01, 5.024743359494E-01,
       5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02, -5.344962345675E-01,
       -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00, -1.141637026370E+00,
       -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00, -1.175775981086E+00,
       -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02, 5.395998813701E-01,
       5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01, 2.797210634251E-01,
       1.683444271630E-01, 4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01,
       2.866177396421E-01, 3.129407785669E-01, 2.692044129679E-01, -8.072992817954E-04,
       -3.086841498962E-01, -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01,
       -5.976535172184E-01, -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01,
       -6.290185834875E-01, -5.069297185127E-01, -3.086841498962E-01, -8.072992817955E-04,
       2.692044129679E-01, 3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01,
       3.910976996873E-01, 4.433376797099E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       8.680361464225E-01, 5.904884624947E-01, 3.129407785669E-01, 3.129407785669E-01,
       5.904884624947E-01, 8.680361464225E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       9.071160757681E-02, 4.495215414751E-02, -8.072992817954E-04, -8.072992817951E-04,
       4.495215414751E-02, 9.071160757681E-02, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -1.341871880972E+00, -9.244007997424E-01, -5.069297185127E-01, -5.069297185127E-01,
       -9.244007997424E-01, -1.341871880972E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.764216023318E+00, -1.198786487885E+00, -6.333569524524E-01, -6.333569524524E-01,
       -1.198786487885E+00, -1.764216023318E+00, -1.685620535521E+00, -1.141637026370E+00,
       -5.976535172184E-01, -5.976535172184E-01, -1.141637026370E+00, -1.685620535521E+00,
       -1.681006430126E+00, -1.137369657030E+00, -5.937328839348E-01, -5.937328839348E-01,
       -1.137369657030E+00, -1.681006430126E+00, -1.685620535521E+00, -1.141637026370E+00,
       -5.976535172184E-01, -5.976535172184E-01, -1.141637026370E+00, -1.685620535521E+00,
       -1.764216023318E+00, -1.198786487885E+00, -6.333569524524E-01, -6.333569524524E-01,
       -1.198786487885E+00, -1.764216023318E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.341871880972E+00, -9.244007997424E-01, -5.069297185127E-01, -5.069297185127E-01,
       -9.244007997424E-01, -1.341871880972E+00, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       9.071160757681E-02, 4.495215414751E-02, -8.072992817955E-04, -8.072992817957E-04,
       4.495215414751E-02, 9.071160757681E-02, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       8.680361464225E-01, 5.904884624947E-01, 3.129407785669E-01, 3.129407785669E-01,
       5.904884624947E-01, 8.680361464225E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01, 2.866177396421E-01,
       3.129407785669E-01, 2.692044129679E-01, -8.072992817951E-04, -3.086841498962E-01,
       -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01, -5.976535172184E-01,
       -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01, -6.290185834875E-01,
       -5.069297185127E-01, -3.086841498962E-01, -8.072992817954E-04, 2.692044129679E-01,
       3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01, 3.910976996873E-01,
       4.433376797098E-01, 1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01,
       5.024743359494E-01, 5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02,
       -5.344962345675E-01, -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00,
       -1.141637026370E+00, -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00,
       -1.175775981086E+00, -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02,
       5.395998813701E-01, 5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01,
       2.797210634251E-01, 1.683444271630E-01, -1.066488253839E-01, 1.683444271630E-01,
       4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01, 8.099953497723E-01,
       9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00, -1.722533378684E+00,
       -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00, -1.685620535521E+00,
       -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00, -7.603083192388E-01,
       9.071160757681E-02, 8.099953497723E-01, 8.680361464225E-01, 7.183309322567E-01,
       4.433376797098E-01, 1.683444271630E-01, -1.066488253839E-01, 4.826495422536E-01,
       5.729701223819E-01, 6.632907025101E-01, 7.536112826384E-01, 8.719664804748E-01,
       7.837181426811E-01, 3.238561278413E-02, -8.390032823170E-01, -1.411116709286E+00,
       -1.769347343907E+00, -1.783323390866E+00, -1.681006430126E+00, -1.669963815903E+00,
       -1.681006430126E+00, -1.783323390866E+00, -1.769347343907E+00, -1.411116709286E+00,
       -8.390032823170E-01, 3.238561278413E-02, 7.837181426811E-01, 8.719664804748E-01,
       7.536112826384E-01, 6.632907025101E-01, 5.729701223819E-01, 4.826495422536E-01,
       5.729701223819E-01, 5.573457649428E-01, 5.417214075038E-01, 5.260970500647E-01,
       5.924639871528E-01, 5.208889309184E-01, 4.863427228491E-03, -5.878551348073E-01,
       -9.708553108523E-01, -1.206607720720E+00, -1.210660725495E+00, -1.137369657030E+00,
       -1.128714899069E+00, -1.137369657030E+00, -1.210660725495E+00, -1.206607720720E+00,
       -9.708553108523E-01, -5.878551348073E-01, 4.863427228491E-03, 5.208889309184E-01,
       5.924639871528E-01, 5.260970500647E-01, 5.417214075038E-01, 5.573457649428E-01,
       5.729701223819E-01, 6.632907025101E-01, 5.417214075038E-01, 4.201521124974E-01,
       2.985828174911E-01, 3.129614938307E-01, 2.580597191557E-01, -2.265875832714E-02,
       -3.367069872975E-01, -5.305939124186E-01, -6.438680975332E-01, -6.379980601237E-01,
       -5.937328839348E-01, -5.874659822349E-01, -5.937328839348E-01, -6.379980601237E-01,
       -6.438680975332E-01, -5.305939124186E-01, -3.367069872975E-01, -2.265875832714E-02,
       2.580597191557E-01, 3.129614938307E-01, 2.985828174911E-01, 4.201521124974E-01,
       5.417214075038E-01, 6.632907025101E-01, 7.536112826384E-01, 5.260970500647E-01,
       2.985828174911E-01, 2.985828174911E-01, 5.260970500647E-01, 7.536112826384E-01,
       8.719664804748E-01, 5.924639871528E-01, 3.129614938307E-01, 3.129614938307E-01,
       5.924639871528E-01, 8.719664804748E-01, 7.837181426811E-01, 5.208889309184E-01,
       2.580597191557E-01, 2.580597191557E-01, 5.208889309184E-01, 7.837181426811E-01,
       3.238561278413E-02, 4.863427228491E-03, -2.265875832714E-02, -2.265875832714E-02,
       4.863427228491E-03, 3.238561278413E-02, -8.390032823170E-01, -5.878551348073E-01,
       -3.367069872975E-01, -3.367069872975E-01, -5.878551348073E-01, -8.390032823170E-01,
       -1.411116709286E+00, -9.708553108523E-01, -5.305939124186E-01, -5.305939124186E-01,
       -9.708553108523E-01, -1.411116709286E+00, -1.769347343907E+00, -1.206607720720E+00,
       -6.438680975332E-01, -6.438680975332E-01, -1.206607720720E+00, -1.769347343907E+00,
       -1.783323390866E+00, -1.210660725495E+00, -6.379980601237E-01, -6.379980601237E-01,
       -1.210660725495E+00, -1.783323390866E+00, -1.681006430126E+00, -1.137369657030E+00,
       -5.937328839348E-01, -5.937328839348E-01, -1.137369657030E+00, -1.681006430126E+00,
       -1.669963815903E+00, -1.128714899069E+00, -5.874659822349E-01, -5.874659822349E-01,
       -1.128714899069E+00, -1.669963815903E+00, -1.681006430126E+00, -1.137369657030E+00,
       -5.937328839348E-01, -5.937328839348E-01, -1.137369657030E+00, -1.681006430126E+00,
       -1.783323390866E+00, -1.210660725495E+00, -6.379980601237E-01, -6.379980601237E-01,
       -1.210660725495E+00, -1.783323390866E+00, -1.769347343907E+00, -1.206607720720E+00,
       -6.438680975332E-01, -6.438680975332E-01, -1.206607720720E+00, -1.769347343907E+00,
       -1.411116709286E+00, -9.708553108523E-01, -5.305939124186E-01, -5.305939124186E-01,
       -9.708553108523E-01, -1.411116709286E+00, -8.390032823170E-01, -5.878551348073E-01,
       -3.367069872975E-01, -3.367069872975E-01, -5.878551348073E-01, -8.390032823170E-01,
       3.238561278413E-02, 4.863427228491E-03, -2.265875832714E-02, -2.265875832714E-02,
       4.863427228491E-03, 3.238561278413E-02, 7.837181426811E-01, 5.208889309184E-01,
       2.580597191557E-01, 2.580597191557E-01, 5.208889309184E-01, 7.837181426811E-01,
       8.719664804748E-01, 5.924639871528E-01, 3.129614938307E-01, 3.129614938307E-01,
       5.924639871528E-01, 8.719664804748E-01, 7.536112826384E-01, 5.260970500647E-01,
       2.985828174911E-01, 2.985828174911E-01, 5.260970500647E-01, 7.536112826384E-01,
       6.632907025101E-01, 5.417214075038E-01, 4.201521124974E-01, 2.985828174911E-01,
       3.129614938307E-01, 2.580597191557E-01, -2.265875832714E-02, -3.367069872975E-01,
       -5.305939124186E-01, -6.438680975332E-01, -6.379980601237E-01, -5.937328839348E-01,
       -5.874659822349E-01, -5.937328839348E-01, -6.379980601237E-01, -6.438680975332E-01,
       -5.305939124186E-01, -3.367069872975E-01, -2.265875832714E-02, 2.580597191557E-01,
       3.129614938307E-01, 2.985828174911E-01, 4.201521124974E-01, 5.417214075038E-01,
       6.632907025101E-01, 5.729701223819E-01, 5.573457649428E-01, 5.417214075038E-01,
       5.260970500647E-01, 5.924639871528E-01, 5.208889309184E-01, 4.863427228491E-03,
       -5.878551348073E-01, -9.708553108523E-01, -1.206607720720E+00, -1.210660725495E+00,
       -1.137369657030E+00, -1.128714899069E+00, -1.137369657030E+00, -1.210660725495E+00,
       -1.206607720720E+00, -9.708553108523E-01, -5.878551348073E-01, 4.863427228492E-03,
       5.208889309184E-01, 5.924639871528E-01, 5.260970500647E-01, 5.417214075038E-01,
       5.573457649428E-01, 5.729701223818E-01, 4.826495422536E-01, 5.729701223819E-01,
       6.632907025101E-01, 7.536112826384E-01, 8.719664804748E-01, 7.837181426811E-01,
       3.238561278413E-02, -8.390032823170E-01, -1.411116709286E+00, -1.769347343907E+00,
       -1.783323390866E+00, -1.681006430126E+00, -1.669963815903E+00, -1.681006430126E+00,
       -1.783323390866E+00, -1.769347343907E+00, -1.411116709286E+00, -8.390032823170E-01,
       3.238561278413E-02, 7.837181426811E-01, 8.719664804748E-01, 7.536112826384E-01,
       6.632907025101E-01, 5.729701223818E-01, 4.826495422535E-01, -1.066488253839E-01,
       1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01,
       8.099953497724E-01, 9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00,
       -1.722533378684E+00, -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00,
       -1.685620535521E+00, -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00,
       -7.603083192388E-01, 9.071160757681E-02, 8.099953497724E-01, 8.680361464225E-01,
       7.183309322567E-01, 4.433376797099E-01, 1.683444271630E-01, -1.066488253839E-01,
       1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01, 5.024743359494E-01,
       5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02, -5.344962345675E-01,
       -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00, -1.141637026370E+00,
       -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00, -1.175775981086E+00,
       -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02, 5.395998813701E-01,
       5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01, 2.797210634251E-01,
       1.683444271630E-01, 4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01,
       2.866177396421E-01, 3.129407785669E-01, 2.692044129679E-01, -8.072992817954E-04,
       -3.086841498962E-01, -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01,
       -5.976535172184E-01, -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01,
       -6.290185834875E-01, -5.069297185127E-01, -3.086841498962E-01, -8.072992817955E-04,
       2.692044129679E-01, 3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01,
       3.910976996873E-01, 4.433376797099E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       8.680361464225E-01, 5.904884624947E-01, 3.129407785669E-01, 3.129407785669E-01,
       5.904884624947E-01, 8.680361464225E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       9.071160757681E-02, 4.495215414751E-02, -8.072992817954E-04, -8.072992817951E-04,
       4.495215414751E-02, 9.071160757681E-02, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -1.341871880972E+00, -9.244007997424E-01, -5.069297185127E-01, -5.069297185127E-01,
       -9.244007997424E-01, -1.341871880972E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.764216023318E+00, -1.198786487885E+00, -6.333569524524E-01, -6.333569524524E-01,
       -1.198786487885E+00, -1.764216023318E+00, -1.685620535521E+00, -1.141637026370E+00,
       -5.976535172184E-01, -5.976535172184E-01, -1.141637026370E+00, -1.685620535521E+00,
       -1.681006430126E+00, -1.137369657030E+00, -5.937328839348E-01, -5.937328839348E-01,
       -1.137369657030E+00, -1.681006430126E+00, -1.685620535521E+00, -1.141637026370E+00,
       -5.976535172184E-01, -5.976535172184E-01, -1.141637026370E+00, -1.685620535521E+00,
       -1.764216023318E+00, -1.198786487885E+00, -6.333569524524E-01, -6.333569524524E-01,
       -1.198786487885E+00, -1.764216023318E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.341871880972E+00, -9.244007997424E-01, -5.069297185127E-01, -5.069297185127E-01,
       -9.244007997424E-01, -1.341871880972E+00, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       9.071160757681E-02, 4.495215414751E-02, -8.072992817955E-04, -8.072992817957E-04,
       4.495215414751E-02, 9.071160757681E-02, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       8.680361464225E-01, 5.904884624947E-01, 3.129407785669E-01, 3.129407785669E-01,
       5.904884624947E-01, 8.680361464225E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01, 2.866177396421E-01,
       3.129407785669E-01, 2.692044129679E-01, -8.072992817951E-04, -3.086841498962E-01,
       -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01, -5.976535172184E-01,
       -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01, -6.290185834875E-01,
       -5.069297185127E-01, -3.086841498962E-01, -8.072992817954E-04, 2.692044129679E-01,
       3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01, 3.910976996873E-01,
       4.433376797098E-01, 1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01,
       5.024743359494E-01, 5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02,
       -5.344962345675E-01, -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00,
       -1.141637026370E+00, -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00,
       -1.175775981086E+00, -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02,
       5.395998813701E-01, 5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01,
       2.797210634251E-01, 1.683444271630E-01, -1.066488253839E-01, 1.683444271630E-01,
       4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01, 8.099953497723E-01,
       9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00, -1.722533378684E+00,
       -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00, -1.685620535521E+00,
       -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00, -7.603083192388E-01,
       9.071160757681E-02, 8.099953497723E-01, 8.680361464225E-01, 7.183309322567E-01,
       4.433376797098E-01, 1.683444271630E-01, -1.066488253839E-01, -2.144463029636E+00,
       -1.225075078640E+00, -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01,
       9.201634736832E-01, 2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00,
       -1.611357082932E+00, -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00,
       -1.764216023318E+00, -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00,
       -5.043249647001E-01, 2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01,
       6.137008233513E-01, -3.056871276443E-01, -1.225075078640E+00, -2.144463029636E+00,
       -1.225075078640E+00, -6.724412439495E-01, -1.198074092591E-01, 4.328264254313E-01,
       5.976046378355E-01, 6.170377036614E-01, 1.856684861198E-01, -3.615121907346E-01,
       -7.878746384657E-01, -1.103792587007E+00, -1.197682062213E+00, -1.198786487885E+00,
       -1.210660725495E+00, -1.198786487885E+00, -1.197682062213E+00, -1.103792587007E+00,
       -7.878746384657E-01, -3.615121907346E-01, 1.856684861198E-01, 6.170377036614E-01,
       5.976046378355E-01, 4.328264254313E-01, -1.198074092591E-01, -6.724412439495E-01,
       -1.225075078640E+00, -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02,
       2.519520275112E-01, 3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02,
       -2.186994167692E-01, -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01,
       -6.333569524524E-01, -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01,
       -5.962280910819E-01, -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02,
       3.139119336396E-01, 3.200926841090E-01, 2.519520275112E-01, 6.607230912604E-02,
       -1.198074092591E-01, -3.056871276443E-01, 6.137008233513E-01, 4.328264254313E-01,
       2.519520275112E-01, 2.519520275112E-01, 4.328264254313E-01, 6.137008233513E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 9.201634736832E-01, 6.170377036614E-01,
       3.139119336396E-01, 3.139119336396E-01, 6.170377036614E-01, 9.201634736832E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, -5.043249647001E-01, -3.615121907346E-01,
       -2.186994167692E-01, -2.186994167692E-01, -3.615121907346E-01, -5.043249647001E-01,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -1.611357082932E+00, -1.103792587007E+00,
       -5.962280910819E-01, -5.962280910819E-01, -1.103792587007E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.197682062213E+00, -6.383880639117E-01, -6.383880639117E-01,
       -1.197682062213E+00, -1.756976060514E+00, -1.764216023318E+00, -1.198786487885E+00,
       -6.333569524524E-01, -6.333569524524E-01, -1.198786487885E+00, -1.764216023318E+00,
       -1.783323390866E+00, -1.210660725495E+00, -6.379980601237E-01, -6.379980601237E-01,
       -1.210660725495E+00, -1.783323390866E+00, -1.764216023318E+00, -1.198786487885E+00,
       -6.333569524524E-01, -6.333569524524E-01, -1.198786487885E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.197682062213E+00, -6.383880639117E-01, -6.383880639117E-01,
       -1.197682062213E+00, -1.756976060514E+00, -1.611357082932E+00, -1.103792587007E+00,
       -5.962280910819E-01, -5.962280910819E-01, -1.103792587007E+00, -1.611357082932E+00,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -5.043249647001E-01, -3.615121907346E-01,
       -2.186994167692E-01, -2.186994167692E-01, -3.615121907346E-01, -5.043249647001E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 9.201634736832E-01, 6.170377036614E-01,
       3.139119336396E-01, 3.139119336396E-01, 6.170377036614E-01, 9.201634736832E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 6.137008233513E-01, 4.328264254313E-01,
       2.519520275112E-01, 2.519520275112E-01, 4.328264254313E-01, 6.137008233513E-01,
       -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02, 2.519520275112E-01,
       3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02, -2.186994167692E-01,
       -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01, -6.333569524524E-01,
       -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01, -5.962280910819E-01,
       -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02, 3.139119336396E-01,
       3.200926841090E-01, 2.519520275112E-01, 6.607230912602E-02, -1.198074092592E-01,
       -3.056871276444E-01, -1.225075078640E+00, -6.724412439495E-01, -1.198074092591E-01,
       4.328264254313E-01, 5.976046378355E-01, 6.170377036614E-01, 1.856684861198E-01,
       -3.615121907346E-01, -7.878746384657E-01, -1.103792587007E+00, -1.197682062213E+00,
       -1.198786487885E+00, -1.210660725495E+00, -1.198786487885E+00, -1.197682062213E+00,
       -1.103792587007E+00, -7.878746384657E-01, -3.615121907346E-01, 1.856684861198E-01,
       6.170377036614E-01, 5.976046378355E-01, 4.328264254313E-01, -1.198074092592E-01,
       -6.724412439496E-01, -1.225075078640E+00, -2.144463029636E+00, -1.225075078640E+00,
       -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01, 9.201634736832E-01,
       2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00, -1.611357082932E+00,
       -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00, -1.764216023318E+00,
       -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00, -5.043249647001E-01,
       2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01, 6.137008233513E-01,
       -3.056871276444E-01, -1.225075078640E+00, -2.144463029636E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01,
       1.020213006502E+00, 5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01,
       -1.348021656895E+00, -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00,
       -1.722533378684E+00, -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01,
       -1.307486146132E-01, 5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01,
       4.360881292036E-01, -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01, 3.134926230269E-01,
       5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01, -1.076250326493E-01,
       -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00, -1.175775981086E+00,
       -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00, -9.291220640084E-01,
       -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01, 6.892874849841E-01,
       5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01, -1.941276548716E+00,
       -3.068661134588E+00, -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01,
       1.908971168502E-01, 3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01,
       -8.450145068548E-02, -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01,
       -6.290185834875E-01, -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01,
       -5.102224711217E-01, -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01,
       3.583619634660E-01, 3.127750564560E-01, 1.908971168502E-01, -3.114974229973E-01,
       -8.138919628448E-01, -1.316286502692E+00, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       8.365934740037E-01, 5.746842652299E-01, 3.127750564560E-01, 3.127750564560E-01,
       5.746842652299E-01, 8.365934740037E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       5.573195659183E-01, 3.656619694996E-01, 1.740043730810E-01, 1.740043730810E-01,
       3.656619694996E-01, 5.573195659183E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       -7.879132544621E-01, -5.527647108636E-01, -3.176161672652E-01, -3.176161672652E-01,
       -5.527647108636E-01, -7.879132544621E-01, -1.348021656895E+00, -9.291220640084E-01,
       -5.102224711217E-01, -5.102224711217E-01, -9.291220640084E-01, -1.348021656895E+00,
       -1.611357082932E+00, -1.103792587007E+00, -5.962280910819E-01, -5.962280910819E-01,
       -1.103792587007E+00, -1.611357082932E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.769347343907E+00, -1.206607720720E+00, -6.438680975332E-01, -6.438680975332E-01,
       -1.206607720720E+00, -1.769347343907E+00, -1.722533378684E+00, -1.175775981086E+00,
       -6.290185834875E-01, -6.290185834875E-01, -1.175775981086E+00, -1.722533378684E+00,
       -1.611357082932E+00, -1.103792587007E+00, -5.962280910819E-01, -5.962280910819E-01,
       -1.103792587007E+00, -1.611357082932E+00, -1.348021656895E+00, -9.291220640084E-01,
       -5.102224711217E-01, -5.102224711217E-01, -9.291220640084E-01, -1.348021656895E+00,
       -7.879132544621E-01, -5.527647108636E-01, -3.176161672652E-01, -3.176161672652E-01,
       -5.527647108636E-01, -7.879132544621E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       5.573195659183E-01, 3.656619694996E-01, 1.740043730810E-01, 1.740043730810E-01,
       3.656619694996E-01, 5.573195659183E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       8.365934740037E-01, 5.746842652299E-01, 3.127750564560E-01, 3.127750564560E-01,
       5.746842652299E-01, 8.365934740037E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01, 1.908971168501E-01,
       3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01, -8.450145068548E-02,
       -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01, -6.290185834875E-01,
       -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01, -5.102224711217E-01,
       -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01, 3.583619634660E-01,
       3.127750564560E-01, 1.908971168501E-01, -3.114974229973E-01, -8.138919628448E-01,
       -1.316286502692E+00, -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01,
       3.134926230269E-01, 5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01,
       -1.076250326493E-01, -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00,
       -1.175775981086E+00, -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00,
       -9.291220640084E-01, -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01,
       6.892874849841E-01, 5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01,
       -1.941276548716E+00, -3.068661134588E+00, -4.821035766484E+00, -3.068661134588E+00,
       -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01, 1.020213006502E+00,
       5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01, -1.348021656895E+00,
       -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00, -1.722533378684E+00,
       -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01, -1.307486146132E-01,
       5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01, 4.360881292036E-01,
       -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00, -7.799246504079E+00,
       -5.146535499003E+00, -2.493824493927E+00, 1.588865111491E-01, 6.982843228335E-01,
       1.043123512841E+00, 8.346976393096E-01, 3.708915411190E-01, -2.159029407473E-01,
       -7.879132544621E-01, -1.137317867441E+00, -1.341871880972E+00, -1.411116709286E+00,
       -1.341871880972E+00, -1.137317867441E+00, -7.879132544621E-01, -2.159029407473E-01,
       3.708915411190E-01, 8.346976393096E-01, 1.043123512841E+00, 6.982843228335E-01,
       1.588865111491E-01, -2.493824493927E+00, -5.146535499003E+00, -7.799246504079E+00,
       -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00, 1.257116984816E-01,
       4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01, 2.359054197387E-01,
       -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01, -9.244007997424E-01,
       -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01, -5.527647108636E-01,
       -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01, 7.115169426466E-01,
       4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00, -3.389119766508E+00,
       -5.146535499003E+00, -2.493824493927E+00, -1.631704034013E+00, -7.695835740996E-01,
       9.253688581413E-02, 2.716214141788E-01, 3.799103724520E-01, 2.845341449216E-01,
       1.009192983584E-01, -1.139532796313E-01, -3.176161672652E-01, -4.384314094904E-01,
       -5.069297185127E-01, -5.305939124186E-01, -5.069297185127E-01, -4.384314094904E-01,
       -3.176161672652E-01, -1.139532796313E-01, 1.009192983584E-01, 2.845341449216E-01,
       3.799103724520E-01, 2.716214141788E-01, 9.253688581413E-02, -7.695835740996E-01,
       -1.631704034013E+00, -2.493824493927E+00, 1.588865111491E-01, 1.257116984816E-01,
       9.253688581413E-02, 9.253688581413E-02, 1.257116984816E-01, 1.588865111491E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 1.043123512841E+00, 7.115169426466E-01,
       3.799103724520E-01, 3.799103724520E-01, 7.115169426466E-01, 1.043123512841E+00,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 3.708915411190E-01, 2.359054197387E-01,
       1.009192983584E-01, 1.009192983584E-01, 2.359054197387E-01, 3.708915411190E-01,
       -2.159029407473E-01, -1.649281101893E-01, -1.139532796313E-01, -1.139532796313E-01,
       -1.649281101893E-01, -2.159029407473E-01, -7.879132544621E-01, -5.527647108636E-01,
       -3.176161672652E-01, -3.176161672652E-01, -5.527647108636E-01, -7.879132544621E-01,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -1.341871880972E+00, -9.244007997424E-01,
       -5.069297185127E-01, -5.069297185127E-01, -9.244007997424E-01, -1.341871880972E+00,
       -1.411116709286E+00, -9.708553108523E-01, -5.305939124186E-01, -5.305939124186E-01,
       -9.708553108523E-01, -1.411116709286E+00, -1.341871880972E+00, -9.244007997424E-01,
       -5.069297185127E-01, -5.069297185127E-01, -9.244007997424E-01, -1.341871880972E+00,
       -1.137317867441E+00, -7.878746384657E-01, -4.384314094904E-01, -4.384314094904E-01,
       -7.878746384657E-01, -1.137317867441E+00, -7.879132544621E-01, -5.527647108636E-01,
       -3.176161672652E-01, -3.176161672652E-01, -5.527647108636E-01, -7.879132544621E-01,
       -2.159029407473E-01, -1.649281101893E-01, -1.139532796313E-01, -1.139532796313E-01,
       -1.649281101893E-01, -2.159029407473E-01, 3.708915411190E-01, 2.359054197387E-01,
       1.009192983584E-01, 1.009192983584E-01, 2.359054197387E-01, 3.708915411190E-01,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 1.043123512841E+00, 7.115169426466E-01,
       3.799103724520E-01, 3.799103724520E-01, 7.115169426466E-01, 1.043123512841E+00,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 1.588865111491E-01, 1.257116984816E-01,
       9.253688581413E-02, 9.253688581413E-02, 1.257116984816E-01, 1.588865111491E-01,
       -2.493824493927E+00, -1.631704034013E+00, -7.695835740996E-01, 9.253688581413E-02,
       2.716214141788E-01, 3.799103724520E-01, 2.845341449216E-01, 1.009192983584E-01,
       -1.139532796313E-01, -3.176161672652E-01, -4.384314094904E-01, -5.069297185127E-01,
       -5.305939124186E-01, -5.069297185127E-01, -4.384314094904E-01, -3.176161672652E-01,
       -1.139532796313E-01, 1.009192983584E-01, 2.845341449216E-01, 3.799103724520E-01,
       2.716214141788E-01, 9.253688581413E-02, -7.695835740996E-01, -1.631704034013E+00,
       -2.493824493927E+00, -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00,
       1.257116984816E-01, 4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01,
       2.359054197387E-01, -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01,
       -9.244007997424E-01, -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01,
       -5.527647108636E-01, -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01,
       7.115169426466E-01, 4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00,
       -3.389119766508E+00, -5.146535499003E+00, -7.799246504079E+00, -5.146535499003E+00,
       -2.493824493927E+00, 1.588865111491E-01, 6.982843228335E-01, 1.043123512841E+00,
       8.346976393096E-01, 3.708915411190E-01, -2.159029407473E-01, -7.879132544621E-01,
       -1.137317867441E+00, -1.341871880972E+00, -1.411116709286E+00, -1.341871880972E+00,
       -1.137317867441E+00, -7.879132544621E-01, -2.159029407473E-01, 3.708915411190E-01,
       8.346976393096E-01, 1.043123512841E+00, 6.982843228335E-01, 1.588865111491E-01,
       -2.493824493927E+00, -5.146535499003E+00, -7.799246504079E+00, -9.931104027256E+00,
       -6.669993535749E+00, -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01,
       9.392642777670E-01, 1.001366113450E+00, 8.099953497724E-01, 3.708915411190E-01,
       -1.307486146132E-01, -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01,
       -7.603083192388E-01, -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01,
       8.099953497724E-01, 1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01,
       -1.477725527353E-01, -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00,
       -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00, -8.399451206323E-02,
       3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01, 5.395998813701E-01,
       2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01, -5.344962345675E-01,
       -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01, -1.076250326493E-01,
       2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01, 6.477831572352E-01,
       3.343918816587E-01, -8.399451206323E-02, -2.279327519959E+00, -4.474660527854E+00,
       -6.669993535749E+00, -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00,
       -2.021647139113E-02, 1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01,
       2.692044129679E-01, 1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01,
       -3.086841498962E-01, -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01,
       -8.450145068548E-02, 1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01,
       3.563020367034E-01, 1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00,
       -2.279327519959E+00, -3.408883044242E+00, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206323E-02, -1.477725527353E-01,
       4.717649724219E-01, 3.343918816587E-01, 1.970187908955E-01, 1.970187908955E-01,
       3.343918816587E-01, 4.717649724219E-01, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       1.001366113450E+00, 6.798791807600E-01, 3.583922480699E-01, 3.583922480699E-01,
       6.798791807600E-01, 1.001366113450E+00, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       3.708915411190E-01, 2.359054197387E-01, 1.009192983584E-01, 1.009192983584E-01,
       2.359054197387E-01, 3.708915411190E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       -5.043249647001E-01, -3.615121907346E-01, -2.186994167692E-01, -2.186994167692E-01,
       -3.615121907346E-01, -5.043249647001E-01, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -8.390032823170E-01, -5.878551348073E-01, -3.367069872975E-01, -3.367069872975E-01,
       -5.878551348073E-01, -8.390032823170E-01, -7.603083192388E-01, -5.344962345675E-01,
       -3.086841498962E-01, -3.086841498962E-01, -5.344962345675E-01, -7.603083192388E-01,
       -5.043249647001E-01, -3.615121907346E-01, -2.186994167692E-01, -2.186994167692E-01,
       -3.615121907346E-01, -5.043249647001E-01, -1.307486146132E-01, -1.076250326493E-01,
       -8.450145068548E-02, -8.450145068548E-02, -1.076250326493E-01, -1.307486146132E-01,
       3.708915411190E-01, 2.359054197387E-01, 1.009192983584E-01, 1.009192983584E-01,
       2.359054197387E-01, 3.708915411190E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       1.001366113450E+00, 6.798791807600E-01, 3.583922480699E-01, 3.583922480699E-01,
       6.798791807600E-01, 1.001366113450E+00, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       4.717649724219E-01, 3.343918816587E-01, 1.970187908955E-01, 1.970187908955E-01,
       3.343918816587E-01, 4.717649724219E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206322E-02, -1.477725527353E-01,
       -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00, -2.021647139113E-02,
       1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01, 2.692044129679E-01,
       1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01, -3.086841498962E-01,
       -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01, -8.450145068548E-02,
       1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01, 3.563020367034E-01,
       1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00, -2.279327519959E+00,
       -3.408883044242E+00, -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00,
       -8.399451206323E-02, 3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01,
       5.395998813701E-01, 2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01,
       -5.344962345675E-01, -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01,
       -1.076250326493E-01, 2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01,
       6.477831572352E-01, 3.343918816587E-01, -8.399451206322E-02, -2.279327519959E+00,
       -4.474660527854E+00, -6.669993535749E+00, -9.931104027256E+00, -6.669993535749E+00,
       -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01, 9.392642777670E-01,
       1.001366113450E+00, 8.099953497723E-01, 3.708915411190E-01, -1.307486146132E-01,
       -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01, -7.603083192388E-01,
       -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01, 8.099953497723E-01,
       1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01, -1.477725527353E-01,
       -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00, -9.806984369113E+00,
       -6.690199875043E+00, -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01,
       5.822972777850E-01, 8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01,
       5.573195659183E-01, 2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02,
       9.071160757681E-02, 2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01,
       1.001366113450E+00, 8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01,
       -4.566308869047E-01, -3.573415380974E+00, -6.690199875043E+00, -9.806984369113E+00,
       -6.690199875043E+00, -4.559820735228E+00, -2.429441595413E+00, -2.990624555983E-01,
       8.358267959076E-02, 4.110639243400E-01, 6.094930270000E-01, 6.798791807600E-01,
       5.596158921156E-01, 3.656619694996E-01, 1.856684861198E-01, 4.495215414751E-02,
       4.863427228491E-03, 4.495215414751E-02, 1.856684861198E-01, 3.656619694996E-01,
       5.596158921156E-01, 6.798791807600E-01, 6.094930270000E-01, 4.110639243400E-01,
       8.358267959076E-02, -2.990624555983E-01, -2.429441595413E+00, -4.559820735228E+00,
       -6.690199875043E+00, -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00,
       -1.414940242919E-01, 6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01,
       3.583922480699E-01, 2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02,
       -8.072992817955E-04, -2.265875832714E-02, -8.072992817955E-04, 7.507429471279E-02,
       1.740043730810E-01, 2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01,
       2.398305708950E-01, 6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00,
       -2.429441595413E+00, -3.573415380974E+00, -4.566308869047E-01, -2.990624555983E-01,
       -1.414940242919E-01, -1.414940242919E-01, -2.990624555983E-01, -4.566308869047E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, 5.822972777850E-01, 4.110639243400E-01,
       2.398305708950E-01, 2.398305708950E-01, 4.110639243400E-01, 5.822972777850E-01,
       8.845176151315E-01, 6.094930270000E-01, 3.344684388685E-01, 3.344684388685E-01,
       6.094930270000E-01, 8.845176151314E-01, 1.001366113450E+00, 6.798791807600E-01,
       3.583922480699E-01, 3.583922480699E-01, 6.798791807600E-01, 1.001366113450E+00,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 5.573195659183E-01, 3.656619694996E-01,
       1.740043730810E-01, 1.740043730810E-01, 3.656619694996E-01, 5.573195659183E-01,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 9.071160757681E-02, 4.495215414751E-02,
       -8.072992817955E-04, -8.072992817957E-04, 4.495215414751E-02, 9.071160757681E-02,
       3.238561278413E-02, 4.863427228491E-03, -2.265875832714E-02, -2.265875832714E-02,
       4.863427228491E-03, 3.238561278413E-02, 9.071160757681E-02, 4.495215414751E-02,
       -8.072992817955E-04, -8.072992817957E-04, 4.495215414751E-02, 9.071160757681E-02,
       2.962626775268E-01, 1.856684861198E-01, 7.507429471279E-02, 7.507429471279E-02,
       1.856684861198E-01, 2.962626775268E-01, 5.573195659183E-01, 3.656619694996E-01,
       1.740043730810E-01, 1.740043730810E-01, 3.656619694996E-01, 5.573195659183E-01,
       8.346976393096E-01, 5.596158921156E-01, 2.845341449216E-01, 2.845341449216E-01,
       5.596158921156E-01, 8.346976393096E-01, 1.001366113450E+00, 6.798791807600E-01,
       3.583922480699E-01, 3.583922480699E-01, 6.798791807600E-01, 1.001366113450E+00,
       8.845176151315E-01, 6.094930270000E-01, 3.344684388685E-01, 3.344684388685E-01,
       6.094930270000E-01, 8.845176151315E-01, 5.822972777850E-01, 4.110639243400E-01,
       2.398305708950E-01, 2.398305708950E-01, 4.110639243400E-01, 5.822972777850E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, -4.566308869047E-01, -2.990624555983E-01,
       -1.414940242919E-01, -1.414940242919E-01, -2.990624555983E-01, -4.566308869047E-01,
       -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00, -1.414940242919E-01,
       6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01, 3.583922480699E-01,
       2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02, -8.072992817957E-04,
       -2.265875832715E-02, -8.072992817957E-04, 7.507429471279E-02, 1.740043730810E-01,
       2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01, 2.398305708950E-01,
       6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00, -2.429441595413E+00,
       -3.573415380974E+00, -6.690199875043E+00, -4.559820735228E+00, -2.429441595413E+00,
       -2.990624555983E-01, 8.358267959076E-02, 4.110639243400E-01, 6.094930270000E-01,
       6.798791807600E-01, 5.596158921156E-01, 3.656619694996E-01, 1.856684861198E-01,
       4.495215414751E-02, 4.863427228491E-03, 4.495215414751E-02, 1.856684861198E-01,
       3.656619694996E-01, 5.596158921156E-01, 6.798791807600E-01, 6.094930270000E-01,
       4.110639243400E-01, 8.358267959076E-02, -2.990624555983E-01, -2.429441595413E+00,
       -4.559820735228E+00, -6.690199875043E+00, -9.806984369113E+00, -6.690199875043E+00,
       -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01, 5.822972777850E-01,
       8.845176151314E-01, 1.001366113450E+00, 8.346976393096E-01, 5.573195659183E-01,
       2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02, 9.071160757681E-02,
       2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01, 1.001366113450E+00,
       8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01, -4.566308869047E-01,
       -3.573415380974E+00, -6.690199875043E+00, -9.806984369112E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01,
       1.353240031925E-01, 5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00,
       1.020213006502E+00, 9.201634736832E-01, 8.099953497724E-01, 7.837181426811E-01,
       8.099953497724E-01, 9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00,
       9.392642777670E-01, 5.822972777850E-01, 1.353240031925E-01, -2.653592939274E-01,
       -6.656749224279E-01, -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00, -4.483785129825E-01,
       -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01, 6.477831572352E-01,
       7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01, 5.395998813701E-01,
       5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01, 6.892874849841E-01,
       7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01, 1.097648162584E-01,
       -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00, -3.797238488428E+00,
       -5.471668476150E+00, -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00,
       -2.310821035371E-01, -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01,
       3.563020367034E-01, 3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01,
       2.692044129679E-01, 2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01,
       3.583619634660E-01, 3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01,
       8.420562932427E-02, -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00,
       -2.122808500705E+00, -3.068671699289E+00, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -2.653592939274E-01, -1.680490365956E-01, -7.073877926376E-02, -7.073877926376E-02,
       -1.680490365956E-01, -2.653592939274E-01, 1.353240031925E-01, 1.097648162584E-01,
       8.420562932427E-02, 8.420562932427E-02, 1.097648162584E-01, 1.353240031924E-01,
       5.822972777850E-01, 4.110639243400E-01, 2.398305708950E-01, 2.398305708950E-01,
       4.110639243400E-01, 5.822972777850E-01, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       1.043123512841E+00, 7.115169426466E-01, 3.799103724520E-01, 3.799103724520E-01,
       7.115169426466E-01, 1.043123512841E+00, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       9.201634736832E-01, 6.170377036614E-01, 3.139119336396E-01, 3.139119336396E-01,
       6.170377036614E-01, 9.201634736832E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       7.837181426811E-01, 5.208889309184E-01, 2.580597191557E-01, 2.580597191557E-01,
       5.208889309184E-01, 7.837181426811E-01, 8.099953497724E-01, 5.395998813701E-01,
       2.692044129679E-01, 2.692044129679E-01, 5.395998813701E-01, 8.099953497723E-01,
       9.201634736832E-01, 6.170377036614E-01, 3.139119336396E-01, 3.139119336396E-01,
       6.170377036614E-01, 9.201634736832E-01, 1.020213006502E+00, 6.892874849841E-01,
       3.583619634660E-01, 3.583619634660E-01, 6.892874849841E-01, 1.020213006502E+00,
       1.043123512841E+00, 7.115169426466E-01, 3.799103724520E-01, 3.799103724520E-01,
       7.115169426466E-01, 1.043123512841E+00, 9.392642777670E-01, 6.477831572352E-01,
       3.563020367034E-01, 3.563020367034E-01, 6.477831572352E-01, 9.392642777670E-01,
       5.822972777850E-01, 4.110639243400E-01, 2.398305708950E-01, 2.398305708950E-01,
       4.110639243400E-01, 5.822972777850E-01, 1.353240031925E-01, 1.097648162584E-01,
       8.420562932427E-02, 8.420562932427E-02, 1.097648162584E-01, 1.353240031924E-01,
       -2.653592939274E-01, -1.680490365956E-01, -7.073877926376E-02, -7.073877926376E-02,
       -1.680490365956E-01, -2.653592939274E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00, -2.310821035371E-01,
       -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01, 3.563020367034E-01,
       3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01, 2.692044129679E-01,
       2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01, 3.583619634660E-01,
       3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01, 8.420562932427E-02,
       -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00, -2.122808500705E+00,
       -3.068671699289E+00, -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00,
       -4.483785129825E-01, -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01,
       6.477831572352E-01, 7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01,
       5.395998813701E-01, 5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01,
       6.892874849841E-01, 7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01,
       1.097648162584E-01, -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00,
       -3.797238488428E+00, -5.471668476150E+00, -7.874665253011E+00, -5.471668476150E+00,
       -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01, 1.353240031924E-01,
       5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00, 1.020213006502E+00,
       9.201634736832E-01, 8.099953497723E-01, 7.837181426811E-01, 8.099953497723E-01,
       9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00, 9.392642777670E-01,
       5.822972777850E-01, 1.353240031924E-01, -2.653592939274E-01, -6.656749224279E-01,
       -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00, -3.653094590856E+00,
       -2.636774001778E+00, -1.620453412699E+00, -6.041328236203E-01, -4.596429798148E-01,
       -2.653592939274E-01, 1.015744035042E-01, 4.717649724219E-01, 6.982843228335E-01,
       8.365934740037E-01, 8.751165915620E-01, 8.680361464225E-01, 8.719664804748E-01,
       8.680361464225E-01, 8.751165915620E-01, 8.365934740037E-01, 6.982843228335E-01,
       4.717649724219E-01, 1.015744035042E-01, -2.653592939274E-01, -4.596429798148E-01,
       -6.041328236203E-01, -1.620453412699E+00, -2.636774001778E+00, -3.653094590856E+00,
       -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00, -4.149215331138E-01,
       -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02, 3.343918816587E-01,
       4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01, 5.904884624947E-01,
       5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01, 5.746842652299E-01,
       4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02, -1.680490365956E-01,
       -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00, -1.896156512223E+00,
       -2.636774001778E+00, -1.620453412699E+00, -1.155539022668E+00, -6.906246326379E-01,
       -2.257102426073E-01, -1.552736426115E-01, -7.073877926376E-02, 6.559095567733E-02,
       1.970187908955E-01, 2.716214141788E-01, 3.127750564560E-01, 3.200926841090E-01,
       3.129407785669E-01, 3.129614938307E-01, 3.129407785669E-01, 3.200926841090E-01,
       3.127750564560E-01, 2.716214141788E-01, 1.970187908955E-01, 6.559095567733E-02,
       -7.073877926376E-02, -1.552736426115E-01, -2.257102426073E-01, -6.906246326379E-01,
       -1.155539022668E+00, -1.620453412699E+00, -6.041328236203E-01, -4.149215331138E-01,
       -2.257102426073E-01, -2.257102426073E-01, -4.149215331138E-01, -6.041328236203E-01,
       -4.596429798148E-01, -3.074583112132E-01, -1.552736426115E-01, -1.552736426115E-01,
       -3.074583112132E-01, -4.596429798149E-01, -2.653592939274E-01, -1.680490365956E-01,
       -7.073877926376E-02, -7.073877926376E-02, -1.680490365956E-01, -2.653592939274E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, 4.717649724219E-01, 3.343918816587E-01,
       1.970187908955E-01, 1.970187908955E-01, 3.343918816587E-01, 4.717649724219E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 8.365934740037E-01, 5.746842652299E-01,
       3.127750564560E-01, 3.127750564560E-01, 5.746842652299E-01, 8.365934740037E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 8.680361464225E-01, 5.904884624947E-01,
       3.129407785669E-01, 3.129407785669E-01, 5.904884624947E-01, 8.680361464225E-01,
       8.719664804748E-01, 5.924639871528E-01, 3.129614938307E-01, 3.129614938307E-01,
       5.924639871528E-01, 8.719664804748E-01, 8.680361464225E-01, 5.904884624947E-01,
       3.129407785669E-01, 3.129407785669E-01, 5.904884624947E-01, 8.680361464225E-01,
       8.751165915620E-01, 5.976046378355E-01, 3.200926841090E-01, 3.200926841090E-01,
       5.976046378355E-01, 8.751165915620E-01, 8.365934740037E-01, 5.746842652299E-01,
       3.127750564560E-01, 3.127750564560E-01, 5.746842652299E-01, 8.365934740037E-01,
       6.982843228335E-01, 4.849528685061E-01, 2.716214141788E-01, 2.716214141788E-01,
       4.849528685061E-01, 6.982843228335E-01, 4.717649724219E-01, 3.343918816587E-01,
       1.970187908955E-01, 1.970187908955E-01, 3.343918816587E-01, 4.717649724219E-01,
       1.015744035042E-01, 8.358267959076E-02, 6.559095567733E-02, 6.559095567733E-02,
       8.358267959076E-02, 1.015744035042E-01, -2.653592939274E-01, -1.680490365956E-01,
       -7.073877926376E-02, -7.073877926376E-02, -1.680490365956E-01, -2.653592939274E-01,
       -4.596429798148E-01, -3.074583112132E-01, -1.552736426115E-01, -1.552736426115E-01,
       -3.074583112132E-01, -4.596429798148E-01, -6.041328236203E-01, -4.149215331138E-01,
       -2.257102426073E-01, -2.257102426073E-01, -4.149215331138E-01, -6.041328236203E-01,
       -1.620453412699E+00, -1.155539022668E+00, -6.906246326379E-01, -2.257102426073E-01,
       -1.552736426115E-01, -7.073877926376E-02, 6.559095567733E-02, 1.970187908955E-01,
       2.716214141788E-01, 3.127750564560E-01, 3.200926841090E-01, 3.129407785669E-01,
       3.129614938307E-01, 3.129407785669E-01, 3.200926841090E-01, 3.127750564560E-01,
       2.716214141788E-01, 1.970187908955E-01, 6.559095567733E-02, -7.073877926376E-02,
       -1.552736426115E-01, -2.257102426073E-01, -6.906246326379E-01, -1.155539022668E+00,
       -1.620453412699E+00, -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00,
       -4.149215331138E-01, -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02,
       3.343918816587E-01, 4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01,
       5.904884624947E-01, 5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01,
       5.746842652299E-01, 4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02,
       -1.680490365956E-01, -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00,
       -1.896156512223E+00, -2.636774001778E+00, -3.653094590856E+00, -2.636774001778E+00,
       -1.620453412699E+00, -6.041328236203E-01, -4.596429798149E-01, -2.653592939274E-01,
       1.015744035042E-01, 4.717649724219E-01, 6.982843228335E-01, 8.365934740037E-01,
       8.751165915620E-01, 8.680361464225E-01, 8.719664804748E-01, 8.680361464225E-01,
       8.751165915620E-01, 8.365934740037E-01, 6.982843228335E-01, 4.717649724219E-01,
       1.015744035042E-01, -2.653592939274E-01, -4.596429798148E-01, -6.041328236203E-01,
       -1.620453412699E+00, -2.636774001778E+00, -3.653094590856E+00, 1.480862149879E+00,
       8.369010281869E-01, 1.929399064948E-01, -4.510212151972E-01, -6.041328236203E-01,
       -6.656749224279E-01, -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01,
       4.360881292036E-01, 6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01,
       7.183309322567E-01, 6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01,
       -1.477725527353E-01, -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01,
       -4.510212151972E-01, 1.929399064949E-01, 8.369010281870E-01, 1.480862149879E+00,
       8.369010281869E-01, 4.513171658361E-01, 6.573330348527E-02, -3.198505588656E-01,
       -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01, -8.399451206323E-02,
       1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01, 5.024743359494E-01,
       5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01, 3.134926230269E-01,
       1.257116984816E-01, -8.399451206323E-02, -2.990624555983E-01, -4.483785129825E-01,
       -4.149215331138E-01, -3.198505588655E-01, 6.573330348529E-02, 4.513171658361E-01,
       8.369010281870E-01, 1.929399064948E-01, 6.573330348527E-02, -6.147329952430E-02,
       -1.886799025339E-01, -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01,
       -2.021647139113E-02, 9.253688581413E-02, 1.908971168502E-01, 2.519520275112E-01,
       2.866177396421E-01, 2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01,
       1.908971168502E-01, 9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01,
       -2.310821035371E-01, -2.257102426073E-01, -1.886799025339E-01, -6.147329952429E-02,
       6.573330348528E-02, 1.929399064949E-01, -4.510212151972E-01, -3.198505588656E-01,
       -1.886799025339E-01, -1.886799025339E-01, -3.198505588656E-01, -4.510212151972E-01,
       -6.041328236203E-01, -4.149215331138E-01, -2.257102426073E-01, -2.257102426073E-01,
       -4.149215331138E-01, -6.041328236203E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -4.566308869047E-01, -2.990624555983E-01, -1.414940242919E-01, -1.414940242919E-01,
       -2.990624555983E-01, -4.566308869047E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206322E-02, -1.477725527353E-01,
       1.588865111491E-01, 1.257116984816E-01, 9.253688581413E-02, 9.253688581414E-02,
       1.257116984816E-01, 1.588865111491E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       6.137008233513E-01, 4.328264254313E-01, 2.519520275112E-01, 2.519520275112E-01,
       4.328264254313E-01, 6.137008233513E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       7.536112826384E-01, 5.260970500647E-01, 2.985828174911E-01, 2.985828174911E-01,
       5.260970500647E-01, 7.536112826384E-01, 7.183309322567E-01, 5.024743359494E-01,
       2.866177396421E-01, 2.866177396421E-01, 5.024743359494E-01, 7.183309322567E-01,
       6.137008233513E-01, 4.328264254313E-01, 2.519520275112E-01, 2.519520275112E-01,
       4.328264254313E-01, 6.137008233513E-01, 4.360881292036E-01, 3.134926230269E-01,
       1.908971168502E-01, 1.908971168501E-01, 3.134926230269E-01, 4.360881292036E-01,
       1.588865111491E-01, 1.257116984816E-01, 9.253688581413E-02, 9.253688581413E-02,
       1.257116984816E-01, 1.588865111491E-01, -1.477725527353E-01, -8.399451206323E-02,
       -2.021647139113E-02, -2.021647139113E-02, -8.399451206322E-02, -1.477725527353E-01,
       -4.566308869047E-01, -2.990624555983E-01, -1.414940242919E-01, -1.414940242919E-01,
       -2.990624555983E-01, -4.566308869047E-01, -6.656749224279E-01, -4.483785129825E-01,
       -2.310821035371E-01, -2.310821035371E-01, -4.483785129825E-01, -6.656749224279E-01,
       -6.041328236203E-01, -4.149215331138E-01, -2.257102426073E-01, -2.257102426073E-01,
       -4.149215331138E-01, -6.041328236203E-01, -4.510212151972E-01, -3.198505588655E-01,
       -1.886799025339E-01, -1.886799025339E-01, -3.198505588656E-01, -4.510212151972E-01,
       1.929399064949E-01, 6.573330348529E-02, -6.147329952429E-02, -1.886799025339E-01,
       -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01, -2.021647139113E-02,
       9.253688581414E-02, 1.908971168501E-01, 2.519520275112E-01, 2.866177396421E-01,
       2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01, 1.908971168501E-01,
       9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01, -2.310821035371E-01,
       -2.257102426073E-01, -1.886799025339E-01, -6.147329952434E-02, 6.573330348520E-02,
       1.929399064947E-01, 8.369010281870E-01, 4.513171658361E-01, 6.573330348529E-02,
       -3.198505588656E-01, -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01,
       -8.399451206322E-02, 1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01,
       5.024743359494E-01, 5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01,
       3.134926230269E-01, 1.257116984816E-01, -8.399451206322E-02, -2.990624555983E-01,
       -4.483785129825E-01, -4.149215331138E-01, -3.198505588656E-01, 6.573330348520E-02,
       4.513171658360E-01, 8.369010281867E-01, 1.480862149879E+00, 8.369010281870E-01,
       1.929399064949E-01, -4.510212151972E-01, -6.041328236203E-01, -6.656749224279E-01,
       -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01, 4.360881292036E-01,
       6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01, 7.183309322567E-01,
       6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01, -1.477725527353E-01,
       -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01, -4.510212151972E-01,
       1.929399064947E-01, 8.369010281867E-01, 1.480862149879E+00, 2.954744435855E+01,
       1.976260954120E+01, 9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00,
       -3.068671699289E+00, -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00,
       -1.316286502692E+00, -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01,
       4.433376797099E-01, -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00,
       -3.408883044242E+00, -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00,
       1.929399064949E-01, 9.977774723847E+00, 1.976260954120E+01, 2.954744435855E+01,
       1.976260954120E+01, 1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02,
       -1.155539022668E+00, -2.122808500705E+00, -2.429441595413E+00, -2.279327519959E+00,
       -1.631704034013E+00, -8.138919628448E-01, -1.198074092591E-01, 3.910976996873E-01,
       5.417214075038E-01, 3.910976996873E-01, -1.198074092591E-01, -8.138919628448E-01,
       -1.631704034013E+00, -2.279327519959E+00, -2.429441595413E+00, -2.122808500705E+00,
       -1.155539022668E+00, 6.573330348529E-02, 6.631358716056E+00, 1.319698412863E+01,
       1.976260954120E+01, 9.977774723846E+00, 6.631358716056E+00, 3.284942708266E+00,
       -6.147329952430E-02, -6.906246326379E-01, -1.176945302121E+00, -1.285467809853E+00,
       -1.149771995675E+00, -7.695835740996E-01, -3.114974229973E-01, 6.607230912604E-02,
       3.388577196647E-01, 4.201521124974E-01, 3.388577196647E-01, 6.607230912604E-02,
       -3.114974229973E-01, -7.695835740996E-01, -1.149771995675E+00, -1.285467809853E+00,
       -1.176945302121E+00, -6.906246326379E-01, -6.147329952429E-02, 3.284942708266E+00,
       6.631358716056E+00, 9.977774723847E+00, 1.929399064948E-01, 6.573330348527E-02,
       -6.147329952430E-02, -1.886799025339E-01, -2.257102426073E-01, -2.310821035371E-01,
       -1.414940242919E-01, -2.021647139113E-02, 9.253688581413E-02, 1.908971168501E-01,
       2.519520275112E-01, 2.866177396421E-01, 2.985828174911E-01, 2.866177396421E-01,
       2.519520275112E-01, 1.908971168501E-01, 9.253688581413E-02, -2.021647139113E-02,
       -1.414940242919E-01, -2.310821035371E-01, -2.257102426073E-01, -1.886799025339E-01,
       -6.147329952430E-02, 6.573330348526E-02, 1.929399064948E-01, -1.620453412699E+00,
       -1.155539022668E+00, -6.906246326379E-01, -2.257102426073E-01, -1.552736426115E-01,
       -7.073877926376E-02, 6.559095567733E-02, 1.970187908955E-01, 2.716214141788E-01,
       3.127750564560E-01, 3.200926841090E-01, 3.129407785669E-01, 3.129614938307E-01,
       3.129407785669E-01, 3.200926841090E-01, 3.127750564560E-01, 2.716214141788E-01,
       1.970187908955E-01, 6.559095567733E-02, -7.073877926376E-02, -1.552736426115E-01,
       -2.257102426073E-01, -6.906246326379E-01, -1.155539022669E+00, -1.620453412699E+00,
       -3.068671699289E+00, -2.122808500705E+00, -1.176945302121E+00, -2.310821035371E-01,
       -7.073877926376E-02, 8.420562932427E-02, 2.398305708950E-01, 3.563020367034E-01,
       3.799103724520E-01, 3.583619634660E-01, 3.139119336396E-01, 2.692044129679E-01,
       2.580597191557E-01, 2.692044129679E-01, 3.139119336396E-01, 3.583619634660E-01,
       3.799103724520E-01, 3.563020367034E-01, 2.398305708950E-01, 8.420562932427E-02,
       -7.073877926376E-02, -2.310821035371E-01, -1.176945302121E+00, -2.122808500705E+00,
       -3.068671699289E+00, -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00,
       -1.414940242919E-01, 6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01,
       3.583922480699E-01, 2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02,
       -8.072992817951E-04, -2.265875832714E-02, -8.072992817951E-04, 7.507429471279E-02,
       1.740043730810E-01, 2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01,
       2.398305708950E-01, 6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00,
       -2.429441595413E+00, -3.573415380974E+00, -3.408883044242E+00, -2.279327519959E+00,
       -1.149771995675E+00, -2.021647139113E-02, 1.970187908955E-01, 3.563020367034E-01,
       3.583922480699E-01, 2.692044129679E-01, 1.009192983584E-01, -8.450145068548E-02,
       -2.186994167692E-01, -3.086841498962E-01, -3.367069872975E-01, -3.086841498962E-01,
       -2.186994167692E-01, -8.450145068548E-02, 1.009192983584E-01, 2.692044129679E-01,
       3.583922480699E-01, 3.563020367034E-01, 1.970187908955E-01, -2.021647139113E-02,
       -1.149771995675E+00, -2.279327519959E+00, -3.408883044242E+00, -2.493824493927E+00,
       -1.631704034013E+00, -7.695835740996E-01, 9.253688581413E-02, 2.716214141788E-01,
       3.799103724520E-01, 2.845341449216E-01, 1.009192983584E-01, -1.139532796313E-01,
       -3.176161672652E-01, -4.384314094904E-01, -5.069297185127E-01, -5.305939124186E-01,
       -5.069297185127E-01, -4.384314094904E-01, -3.176161672652E-01, -1.139532796313E-01,
       1.009192983584E-01, 2.845341449216E-01, 3.799103724520E-01, 2.716214141788E-01,
       9.253688581414E-02, -7.695835740995E-01, -1.631704034013E+00, -2.493824493927E+00,
       -1.316286502692E+00, -8.138919628448E-01, -3.114974229973E-01, 1.908971168501E-01,
       3.127750564560E-01, 3.583619634660E-01, 1.740043730810E-01, -8.450145068548E-02,
       -3.176161672652E-01, -5.102224711217E-01, -5.962280910819E-01, -6.290185834875E-01,
       -6.438680975332E-01, -6.290185834875E-01, -5.962280910819E-01, -5.102224711217E-01,
       -3.176161672652E-01, -8.450145068548E-02, 1.740043730810E-01, 3.583619634660E-01,
       3.127750564560E-01, 1.908971168501E-01, -3.114974229973E-01, -8.138919628448E-01,
       -1.316286502692E+00, -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02,
       2.519520275112E-01, 3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02,
       -2.186994167692E-01, -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01,
       -6.333569524524E-01, -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01,
       -5.962280910819E-01, -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02,
       3.139119336396E-01, 3.200926841090E-01, 2.519520275112E-01, 6.607230912602E-02,
       -1.198074092592E-01, -3.056871276444E-01, 4.433376797099E-01, 3.910976996873E-01,
       3.388577196647E-01, 2.866177396421E-01, 3.129407785669E-01, 2.692044129679E-01,
       -8.072992817951E-04, -3.086841498962E-01, -5.069297185127E-01, -6.290185834875E-01,
       -6.333569524524E-01, -5.976535172184E-01, -5.937328839348E-01, -5.976535172184E-01,
       -6.333569524524E-01, -6.290185834875E-01, -5.069297185127E-01, -3.086841498962E-01,
       -8.072992817954E-04, 2.692044129679E-01, 3.129407785669E-01, 2.866177396421E-01,
       3.388577196647E-01, 3.910976996873E-01, 4.433376797098E-01, 6.632907025101E-01,
       5.417214075038E-01, 4.201521124974E-01, 2.985828174911E-01, 3.129614938307E-01,
       2.580597191557E-01, -2.265875832714E-02, -3.367069872975E-01, -5.305939124186E-01,
       -6.438680975332E-01, -6.379980601237E-01, -5.937328839348E-01, -5.874659822349E-01,
       -5.937328839348E-01, -6.379980601237E-01, -6.438680975332E-01, -5.305939124186E-01,
       -3.367069872975E-01, -2.265875832715E-02, 2.580597191557E-01, 3.129614938307E-01,
       2.985828174911E-01, 4.201521124974E-01, 5.417214075038E-01, 6.632907025101E-01,
       4.433376797099E-01, 3.910976996873E-01, 3.388577196647E-01, 2.866177396421E-01,
       3.129407785669E-01, 2.692044129679E-01, -8.072992817951E-04, -3.086841498962E-01,
       -5.069297185127E-01, -6.290185834875E-01, -6.333569524524E-01, -5.976535172184E-01,
       -5.937328839348E-01, -5.976535172184E-01, -6.333569524524E-01, -6.290185834875E-01,
       -5.069297185127E-01, -3.086841498962E-01, -8.072992817954E-04, 2.692044129679E-01,
       3.129407785669E-01, 2.866177396421E-01, 3.388577196647E-01, 3.910976996873E-01,
       4.433376797098E-01, -3.056871276443E-01, -1.198074092591E-01, 6.607230912604E-02,
       2.519520275112E-01, 3.200926841090E-01, 3.139119336396E-01, 7.507429471279E-02,
       -2.186994167692E-01, -4.384314094904E-01, -5.962280910819E-01, -6.383880639117E-01,
       -6.333569524524E-01, -6.379980601237E-01, -6.333569524524E-01, -6.383880639117E-01,
       -5.962280910819E-01, -4.384314094904E-01, -2.186994167692E-01, 7.507429471279E-02,
       3.139119336396E-01, 3.200926841090E-01, 2.519520275112E-01, 6.607230912602E-02,
       -1.198074092592E-01, -3.056871276444E-01, -1.316286502692E+00, -8.138919628448E-01,
       -3.114974229973E-01, 1.908971168501E-01, 3.127750564560E-01, 3.583619634660E-01,
       1.740043730810E-01, -8.450145068548E-02, -3.176161672652E-01, -5.102224711217E-01,
       -5.962280910819E-01, -6.290185834875E-01, -6.438680975332E-01, -6.290185834875E-01,
       -5.962280910819E-01, -5.102224711217E-01, -3.176161672652E-01, -8.450145068548E-02,
       1.740043730810E-01, 3.583619634660E-01, 3.127750564560E-01, 1.908971168501E-01,
       -3.114974229973E-01, -8.138919628448E-01, -1.316286502692E+00, -2.493824493927E+00,
       -1.631704034013E+00, -7.695835740996E-01, 9.253688581413E-02, 2.716214141788E-01,
       3.799103724520E-01, 2.845341449216E-01, 1.009192983584E-01, -1.139532796313E-01,
       -3.176161672652E-01, -4.384314094904E-01, -5.069297185127E-01, -5.305939124186E-01,
       -5.069297185127E-01, -4.384314094904E-01, -3.176161672652E-01, -1.139532796313E-01,
       1.009192983584E-01, 2.845341449216E-01, 3.799103724520E-01, 2.716214141788E-01,
       9.253688581413E-02, -7.695835740996E-01, -1.631704034013E+00, -2.493824493927E+00,
       -3.408883044242E+00, -2.279327519959E+00, -1.149771995675E+00, -2.021647139113E-02,
       1.970187908955E-01, 3.563020367034E-01, 3.583922480699E-01, 2.692044129679E-01,
       1.009192983584E-01, -8.450145068548E-02, -2.186994167692E-01, -3.086841498962E-01,
       -3.367069872975E-01, -3.086841498962E-01, -2.186994167692E-01, -8.450145068548E-02,
       1.009192983584E-01, 2.692044129679E-01, 3.583922480699E-01, 3.563020367034E-01,
       1.970187908955E-01, -2.021647139113E-02, -1.149771995675E+00, -2.279327519959E+00,
       -3.408883044242E+00, -3.573415380974E+00, -2.429441595413E+00, -1.285467809853E+00,
       -1.414940242919E-01, 6.559095567733E-02, 2.398305708950E-01, 3.344684388685E-01,
       3.583922480699E-01, 2.845341449216E-01, 1.740043730810E-01, 7.507429471279E-02,
       -8.072992817954E-04, -2.265875832714E-02, -8.072992817954E-04, 7.507429471279E-02,
       1.740043730810E-01, 2.845341449216E-01, 3.583922480699E-01, 3.344684388685E-01,
       2.398305708950E-01, 6.559095567733E-02, -1.414940242919E-01, -1.285467809853E+00,
       -2.429441595413E+00, -3.573415380974E+00, -3.068671699289E+00, -2.122808500705E+00,
       -1.176945302121E+00, -2.310821035371E-01, -7.073877926376E-02, 8.420562932427E-02,
       2.398305708950E-01, 3.563020367034E-01, 3.799103724520E-01, 3.583619634660E-01,
       3.139119336396E-01, 2.692044129679E-01, 2.580597191557E-01, 2.692044129679E-01,
       3.139119336396E-01, 3.583619634660E-01, 3.799103724520E-01, 3.563020367034E-01,
       2.398305708950E-01, 8.420562932427E-02, -7.073877926376E-02, -2.310821035371E-01,
       -1.176945302121E+00, -2.122808500705E+00, -3.068671699289E+00, -1.620453412699E+00,
       -1.155539022668E+00, -6.906246326379E-01, -2.257102426073E-01, -1.552736426115E-01,
       -7.073877926376E-02, 6.559095567733E-02, 1.970187908955E-01, 2.716214141788E-01,
       3.127750564560E-01, 3.200926841090E-01, 3.129407785669E-01, 3.129614938307E-01,
       3.129407785669E-01, 3.200926841090E-01, 3.127750564560E-01, 2.716214141788E-01,
       1.970187908955E-01, 6.559095567733E-02, -7.073877926376E-02, -1.552736426115E-01,
       -2.257102426073E-01, -6.906246326379E-01, -1.155539022668E+00, -1.620453412699E+00,
       1.929399064949E-01, 6.573330348529E-02, -6.147329952429E-02, -1.886799025339E-01,
       -2.257102426073E-01, -2.310821035371E-01, -1.414940242919E-01, -2.021647139113E-02,
       9.253688581414E-02, 1.908971168501E-01, 2.519520275112E-01, 2.866177396421E-01,
       2.985828174911E-01, 2.866177396421E-01, 2.519520275112E-01, 1.908971168501E-01,
       9.253688581413E-02, -2.021647139113E-02, -1.414940242919E-01, -2.310821035371E-01,
       -2.257102426073E-01, -1.886799025339E-01, -6.147329952434E-02, 6.573330348519E-02,
       1.929399064947E-01, 9.977774723847E+00, 6.631358716056E+00, 3.284942708266E+00,
       -6.147329952430E-02, -6.906246326379E-01, -1.176945302121E+00, -1.285467809853E+00,
       -1.149771995675E+00, -7.695835740995E-01, -3.114974229973E-01, 6.607230912604E-02,
       3.388577196647E-01, 4.201521124974E-01, 3.388577196647E-01, 6.607230912602E-02,
       -3.114974229973E-01, -7.695835740996E-01, -1.149771995675E+00, -1.285467809853E+00,
       -1.176945302121E+00, -6.906246326379E-01, -6.147329952435E-02, 3.284942708265E+00,
       6.631358716055E+00, 9.977774723845E+00, 1.976260954120E+01, 1.319698412863E+01,
       6.631358716056E+00, 6.573330348526E-02, -1.155539022669E+00, -2.122808500705E+00,
       -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00, -8.138919628448E-01,
       -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01, 3.910976996873E-01,
       -1.198074092592E-01, -8.138919628448E-01, -1.631704034013E+00, -2.279327519959E+00,
       -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00, 6.573330348518E-02,
       6.631358716055E+00, 1.319698412863E+01, 1.976260954120E+01, 2.954744435855E+01,
       1.976260954120E+01, 9.977774723847E+00, 1.929399064948E-01, -1.620453412699E+00,
       -3.068671699289E+00, -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00,
       -1.316286502692E+00, -3.056871276443E-01, 4.433376797098E-01, 6.632907025101E-01,
       4.433376797098E-01, -3.056871276444E-01, -1.316286502692E+00, -2.493824493927E+00,
       -3.408883044242E+00, -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00,
       1.929399064947E-01, 9.977774723845E+00, 1.976260954120E+01, 2.954744435855E+01,
       5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01, 8.369010281869E-01,
       -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00, -6.669993535749E+00,
       -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00, 1.683444271630E-01,
       5.729701223819E-01, 1.683444271630E-01, -1.225075078640E+00, -3.068661134588E+00,
       -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00, -5.471668476150E+00,
       -2.636774001778E+00, 8.369010281870E-01, 1.976260954120E+01, 3.868831805421E+01,
       5.761402656722E+01, 3.868831805421E+01, 2.594265109142E+01, 1.319698412863E+01,
       4.513171658361E-01, -1.896156512223E+00, -3.797238488428E+00, -4.559820735228E+00,
       -4.474660527854E+00, -3.389119766508E+00, -1.941276548716E+00, -6.724412439495E-01,
       2.797210634251E-01, 5.573457649428E-01, 2.797210634251E-01, -6.724412439495E-01,
       -1.941276548716E+00, -3.389119766508E+00, -4.474660527854E+00, -4.559820735228E+00,
       -3.797238488428E+00, -1.896156512223E+00, 4.513171658361E-01, 1.319698412863E+01,
       2.594265109142E+01, 3.868831805421E+01, 1.976260954120E+01, 1.319698412863E+01,
       6.631358716056E+00, 6.573330348527E-02, -1.155539022668E+00, -2.122808500705E+00,
       -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00, -8.138919628448E-01,
       -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01, 3.910976996873E-01,
       -1.198074092591E-01, -8.138919628448E-01, -1.631704034013E+00, -2.279327519959E+00,
       -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00, 6.573330348528E-02,
       6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01, 8.369010281869E-01,
       4.513171658361E-01, 6.573330348527E-02, -3.198505588655E-01, -4.149215331138E-01,
       -4.483785129825E-01, -2.990624555983E-01, -8.399451206323E-02, 1.257116984816E-01,
       3.134926230269E-01, 4.328264254313E-01, 5.024743359494E-01, 5.260970500647E-01,
       5.024743359494E-01, 4.328264254313E-01, 3.134926230269E-01, 1.257116984816E-01,
       -8.399451206323E-02, -2.990624555983E-01, -4.483785129825E-01, -4.149215331138E-01,
       -3.198505588656E-01, 6.573330348526E-02, 4.513171658361E-01, 8.369010281869E-01,
       -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00, -4.149215331138E-01,
       -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02, 3.343918816587E-01,
       4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01, 5.904884624947E-01,
       5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01, 5.746842652299E-01,
       4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02, -1.680490365956E-01,
       -3.074583112132E-01, -4.149215331138E-01, -1.155539022669E+00, -1.896156512223E+00,
       -2.636774001778E+00, -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00,
       -4.483785129825E-01, -1.680490365956E-01, 1.097648162584E-01, 4.110639243400E-01,
       6.477831572352E-01, 7.115169426466E-01, 6.892874849841E-01, 6.170377036614E-01,
       5.395998813701E-01, 5.208889309184E-01, 5.395998813701E-01, 6.170377036614E-01,
       6.892874849841E-01, 7.115169426466E-01, 6.477831572352E-01, 4.110639243400E-01,
       1.097648162584E-01, -1.680490365956E-01, -4.483785129825E-01, -2.122808500705E+00,
       -3.797238488428E+00, -5.471668476150E+00, -6.690199875043E+00, -4.559820735228E+00,
       -2.429441595413E+00, -2.990624555983E-01, 8.358267959076E-02, 4.110639243400E-01,
       6.094930270000E-01, 6.798791807600E-01, 5.596158921156E-01, 3.656619694996E-01,
       1.856684861198E-01, 4.495215414751E-02, 4.863427228491E-03, 4.495215414751E-02,
       1.856684861198E-01, 3.656619694996E-01, 5.596158921156E-01, 6.798791807600E-01,
       6.094930270000E-01, 4.110639243400E-01, 8.358267959076E-02, -2.990624555983E-01,
       -2.429441595413E+00, -4.559820735228E+00, -6.690199875043E+00, -6.669993535749E+00,
       -4.474660527854E+00, -2.279327519959E+00, -8.399451206323E-02, 3.343918816587E-01,
       6.477831572352E-01, 6.798791807600E-01, 5.395998813701E-01, 2.359054197387E-01,
       -1.076250326493E-01, -3.615121907346E-01, -5.344962345675E-01, -5.878551348073E-01,
       -5.344962345675E-01, -3.615121907346E-01, -1.076250326493E-01, 2.359054197387E-01,
       5.395998813701E-01, 6.798791807600E-01, 6.477831572352E-01, 3.343918816587E-01,
       -8.399451206322E-02, -2.279327519959E+00, -4.474660527854E+00, -6.669993535749E+00,
       -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00, 1.257116984816E-01,
       4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01, 2.359054197387E-01,
       -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01, -9.244007997424E-01,
       -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01, -5.527647108636E-01,
       -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01, 7.115169426466E-01,
       4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00, -3.389119766508E+00,
       -5.146535499003E+00, -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01,
       3.134926230269E-01, 5.746842652299E-01, 6.892874849841E-01, 3.656619694996E-01,
       -1.076250326493E-01, -5.527647108636E-01, -9.291220640084E-01, -1.103792587007E+00,
       -1.175775981086E+00, -1.206607720720E+00, -1.175775981086E+00, -1.103792587007E+00,
       -9.291220640084E-01, -5.527647108636E-01, -1.076250326493E-01, 3.656619694996E-01,
       6.892874849841E-01, 5.746842652299E-01, 3.134926230269E-01, -8.138919628448E-01,
       -1.941276548716E+00, -3.068661134588E+00, -1.225075078640E+00, -6.724412439495E-01,
       -1.198074092591E-01, 4.328264254313E-01, 5.976046378355E-01, 6.170377036614E-01,
       1.856684861198E-01, -3.615121907346E-01, -7.878746384657E-01, -1.103792587007E+00,
       -1.197682062213E+00, -1.198786487885E+00, -1.210660725495E+00, -1.198786487885E+00,
       -1.197682062213E+00, -1.103792587007E+00, -7.878746384657E-01, -3.615121907346E-01,
       1.856684861198E-01, 6.170377036614E-01, 5.976046378355E-01, 4.328264254313E-01,
       -1.198074092592E-01, -6.724412439496E-01, -1.225075078640E+00, 1.683444271630E-01,
       2.797210634251E-01, 3.910976996873E-01, 5.024743359494E-01, 5.904884624947E-01,
       5.395998813701E-01, 4.495215414751E-02, -5.344962345675E-01, -9.244007997424E-01,
       -1.175775981086E+00, -1.198786487885E+00, -1.141637026370E+00, -1.137369657030E+00,
       -1.141637026370E+00, -1.198786487885E+00, -1.175775981086E+00, -9.244007997424E-01,
       -5.344962345675E-01, 4.495215414751E-02, 5.395998813701E-01, 5.904884624947E-01,
       5.024743359494E-01, 3.910976996873E-01, 2.797210634251E-01, 1.683444271630E-01,
       5.729701223819E-01, 5.573457649428E-01, 5.417214075038E-01, 5.260970500647E-01,
       5.924639871528E-01, 5.208889309184E-01, 4.863427228492E-03, -5.878551348073E-01,
       -9.708553108523E-01, -1.206607720720E+00, -1.210660725495E+00, -1.137369657030E+00,
       -1.128714899069E+00, -1.137369657030E+00, -1.210660725495E+00, -1.206607720720E+00,
       -9.708553108523E-01, -5.878551348073E-01, 4.863427228490E-03, 5.208889309184E-01,
       5.924639871528E-01, 5.260970500647E-01, 5.417214075038E-01, 5.573457649428E-01,
       5.729701223818E-01, 1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01,
       5.024743359494E-01, 5.904884624947E-01, 5.395998813701E-01, 4.495215414751E-02,
       -5.344962345675E-01, -9.244007997424E-01, -1.175775981086E+00, -1.198786487885E+00,
       -1.141637026370E+00, -1.137369657030E+00, -1.141637026370E+00, -1.198786487885E+00,
       -1.175775981086E+00, -9.244007997424E-01, -5.344962345675E-01, 4.495215414751E-02,
       5.395998813701E-01, 5.904884624947E-01, 5.024743359494E-01, 3.910976996873E-01,
       2.797210634251E-01, 1.683444271630E-01, -1.225075078640E+00, -6.724412439495E-01,
       -1.198074092591E-01, 4.328264254313E-01, 5.976046378355E-01, 6.170377036614E-01,
       1.856684861198E-01, -3.615121907346E-01, -7.878746384657E-01, -1.103792587007E+00,
       -1.197682062213E+00, -1.198786487885E+00, -1.210660725495E+00, -1.198786487885E+00,
       -1.197682062213E+00, -1.103792587007E+00, -7.878746384657E-01, -3.615121907346E-01,
       1.856684861198E-01, 6.170377036614E-01, 5.976046378355E-01, 4.328264254313E-01,
       -1.198074092592E-01, -6.724412439496E-01, -1.225075078640E+00, -3.068661134588E+00,
       -1.941276548716E+00, -8.138919628448E-01, 3.134926230269E-01, 5.746842652299E-01,
       6.892874849841E-01, 3.656619694996E-01, -1.076250326493E-01, -5.527647108636E-01,
       -9.291220640084E-01, -1.103792587007E+00, -1.175775981086E+00, -1.206607720720E+00,
       -1.175775981086E+00, -1.103792587007E+00, -9.291220640084E-01, -5.527647108636E-01,
       -1.076250326493E-01, 3.656619694996E-01, 6.892874849841E-01, 5.746842652299E-01,
       3.134926230269E-01, -8.138919628448E-01, -1.941276548716E+00, -3.068661134588E+00,
       -5.146535499003E+00, -3.389119766508E+00, -1.631704034013E+00, 1.257116984816E-01,
       4.849528685061E-01, 7.115169426466E-01, 5.596158921156E-01, 2.359054197387E-01,
       -1.649281101893E-01, -5.527647108636E-01, -7.878746384657E-01, -9.244007997424E-01,
       -9.708553108523E-01, -9.244007997424E-01, -7.878746384657E-01, -5.527647108636E-01,
       -1.649281101893E-01, 2.359054197387E-01, 5.596158921156E-01, 7.115169426466E-01,
       4.849528685061E-01, 1.257116984816E-01, -1.631704034013E+00, -3.389119766508E+00,
       -5.146535499003E+00, -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00,
       -8.399451206323E-02, 3.343918816587E-01, 6.477831572352E-01, 6.798791807600E-01,
       5.395998813701E-01, 2.359054197387E-01, -1.076250326493E-01, -3.615121907346E-01,
       -5.344962345675E-01, -5.878551348073E-01, -5.344962345675E-01, -3.615121907346E-01,
       -1.076250326493E-01, 2.359054197387E-01, 5.395998813701E-01, 6.798791807600E-01,
       6.477831572352E-01, 3.343918816587E-01, -8.399451206322E-02, -2.279327519959E+00,
       -4.474660527854E+00, -6.669993535749E+00, -6.690199875043E+00, -4.559820735228E+00,
       -2.429441595413E+00, -2.990624555983E-01, 8.358267959076E-02, 4.110639243400E-01,
       6.094930270000E-01, 6.798791807600E-01, 5.596158921156E-01, 3.656619694996E-01,
       1.856684861198E-01, 4.495215414751E-02, 4.863427228491E-03, 4.495215414751E-02,
       1.856684861198E-01, 3.656619694996E-01, 5.596158921156E-01, 6.798791807600E-01,
       6.094930270000E-01, 4.110639243400E-01, 8.358267959076E-02, -2.990624555983E-01,
       -2.429441595413E+00, -4.559820735228E+00, -6.690199875043E+00, -5.471668476150E+00,
       -3.797238488428E+00, -2.122808500705E+00, -4.483785129825E-01, -1.680490365956E-01,
       1.097648162584E-01, 4.110639243400E-01, 6.477831572352E-01, 7.115169426466E-01,
       6.892874849841E-01, 6.170377036614E-01, 5.395998813701E-01, 5.208889309184E-01,
       5.395998813701E-01, 6.170377036614E-01, 6.892874849841E-01, 7.115169426466E-01,
       6.477831572352E-01, 4.110639243400E-01, 1.097648162584E-01, -1.680490365956E-01,
       -4.483785129825E-01, -2.122808500705E+00, -3.797238488428E+00, -5.471668476150E+00,
       -2.636774001778E+00, -1.896156512223E+00, -1.155539022668E+00, -4.149215331138E-01,
       -3.074583112132E-01, -1.680490365956E-01, 8.358267959076E-02, 3.343918816587E-01,
       4.849528685061E-01, 5.746842652299E-01, 5.976046378355E-01, 5.904884624947E-01,
       5.924639871528E-01, 5.904884624947E-01, 5.976046378355E-01, 5.746842652299E-01,
       4.849528685061E-01, 3.343918816587E-01, 8.358267959076E-02, -1.680490365956E-01,
       -3.074583112132E-01, -4.149215331138E-01, -1.155539022668E+00, -1.896156512223E+00,
       -2.636774001778E+00, 8.369010281870E-01, 4.513171658361E-01, 6.573330348529E-02,
       -3.198505588656E-01, -4.149215331138E-01, -4.483785129825E-01, -2.990624555983E-01,
       -8.399451206322E-02, 1.257116984816E-01, 3.134926230269E-01, 4.328264254313E-01,
       5.024743359494E-01, 5.260970500647E-01, 5.024743359494E-01, 4.328264254313E-01,
       3.134926230269E-01, 1.257116984816E-01, -8.399451206322E-02, -2.990624555983E-01,
       -4.483785129825E-01, -4.149215331138E-01, -3.198505588656E-01, 6.573330348519E-02,
       4.513171658359E-01, 8.369010281867E-01, 1.976260954120E+01, 1.319698412863E+01,
       6.631358716056E+00, 6.573330348526E-02, -1.155539022669E+00, -2.122808500705E+00,
       -2.429441595413E+00, -2.279327519959E+00, -1.631704034013E+00, -8.138919628448E-01,
       -1.198074092591E-01, 3.910976996873E-01, 5.417214075038E-01, 3.910976996873E-01,
       -1.198074092592E-01, -8.138919628448E-01, -1.631704034013E+00, -2.279327519959E+00,
       -2.429441595413E+00, -2.122808500705E+00, -1.155539022668E+00, 6.573330348518E-02,
       6.631358716055E+00, 1.319698412863E+01, 1.976260954120E+01, 3.868831805421E+01,
       2.594265109142E+01, 1.319698412863E+01, 4.513171658361E-01, -1.896156512223E+00,
       -3.797238488428E+00, -4.559820735228E+00, -4.474660527854E+00, -3.389119766508E+00,
       -1.941276548716E+00, -6.724412439495E-01, 2.797210634251E-01, 5.573457649428E-01,
       2.797210634251E-01, -6.724412439496E-01, -1.941276548716E+00, -3.389119766508E+00,
       -4.474660527854E+00, -4.559820735228E+00, -3.797238488428E+00, -1.896156512223E+00,
       4.513171658359E-01, 1.319698412863E+01, 2.594265109142E+01, 3.868831805420E+01,
       5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01, 8.369010281869E-01,
       -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00, -6.669993535749E+00,
       -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00, 1.683444271630E-01,
       5.729701223818E-01, 1.683444271630E-01, -1.225075078640E+00, -3.068661134588E+00,
       -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00, -5.471668476150E+00,
       -2.636774001778E+00, 8.369010281867E-01, 1.976260954120E+01, 3.868831805420E+01,
       5.761402656721E+01, 8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01,
       1.480862149879E+00, -3.653094590856E+00, -7.874665253011E+00, -9.806984369113E+00,
       -9.931104027256E+00, -7.799246504079E+00, -4.821035766484E+00, -2.144463029636E+00,
       -1.066488253839E-01, 4.826495422536E-01, -1.066488253839E-01, -2.144463029636E+00,
       -4.821035766484E+00, -7.799246504079E+00, -9.931104027256E+00, -9.806984369113E+00,
       -7.874665253011E+00, -3.653094590856E+00, 1.480862149879E+00, 2.954744435855E+01,
       5.761402656722E+01, 8.568060877589E+01, 5.761402656722E+01, 3.868831805421E+01,
       1.976260954120E+01, 8.369010281869E-01, -2.636774001778E+00, -5.471668476150E+00,
       -6.690199875043E+00, -6.669993535749E+00, -5.146535499003E+00, -3.068661134588E+00,
       -1.225075078640E+00, 1.683444271630E-01, 5.729701223819E-01, 1.683444271630E-01,
       -1.225075078640E+00, -3.068661134588E+00, -5.146535499003E+00, -6.669993535749E+00,
       -6.690199875043E+00, -5.471668476150E+00, -2.636774001778E+00, 8.369010281870E-01,
       1.976260954120E+01, 3.868831805421E+01, 5.761402656722E+01, 2.954744435855E+01,
       1.976260954120E+01, 9.977774723846E+00, 1.929399064948E-01, -1.620453412699E+00,
       -3.068671699289E+00, -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00,
       -1.316286502692E+00, -3.056871276443E-01, 4.433376797099E-01, 6.632907025101E-01,
       4.433376797099E-01, -3.056871276443E-01, -1.316286502692E+00, -2.493824493927E+00,
       -3.408883044242E+00, -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00,
       1.929399064949E-01, 9.977774723847E+00, 1.976260954120E+01, 2.954744435855E+01,
       1.480862149879E+00, 8.369010281869E-01, 1.929399064948E-01, -4.510212151972E-01,
       -6.041328236203E-01, -6.656749224279E-01, -4.566308869047E-01, -1.477725527353E-01,
       1.588865111491E-01, 4.360881292036E-01, 6.137008233513E-01, 7.183309322567E-01,
       7.536112826384E-01, 7.183309322567E-01, 6.137008233513E-01, 4.360881292036E-01,
       1.588865111491E-01, -1.477725527353E-01, -4.566308869047E-01, -6.656749224279E-01,
       -6.041328236203E-01, -4.510212151972E-01, 1.929399064948E-01, 8.369010281869E-01,
       1.480862149879E+00, -3.653094590856E+00, -2.636774001778E+00, -1.620453412699E+00,
       -6.041328236203E-01, -4.596429798148E-01, -2.653592939274E-01, 1.015744035042E-01,
       4.717649724219E-01, 6.982843228335E-01, 8.365934740037E-01, 8.751165915620E-01,
       8.680361464225E-01, 8.719664804748E-01, 8.680361464225E-01, 8.751165915620E-01,
       8.365934740037E-01, 6.982843228335E-01, 4.717649724219E-01, 1.015744035042E-01,
       -2.653592939274E-01, -4.596429798149E-01, -6.041328236203E-01, -1.620453412699E+00,
       -2.636774001778E+00, -3.653094590857E+00, -7.874665253011E+00, -5.471668476150E+00,
       -3.068671699289E+00, -6.656749224279E-01, -2.653592939274E-01, 1.353240031924E-01,
       5.822972777850E-01, 9.392642777670E-01, 1.043123512841E+00, 1.020213006502E+00,
       9.201634736832E-01, 8.099953497723E-01, 7.837181426811E-01, 8.099953497723E-01,
       9.201634736832E-01, 1.020213006502E+00, 1.043123512841E+00, 9.392642777670E-01,
       5.822972777850E-01, 1.353240031924E-01, -2.653592939274E-01, -6.656749224279E-01,
       -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00, -9.806984369113E+00,
       -6.690199875043E+00, -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01,
       5.822972777850E-01, 8.845176151315E-01, 1.001366113450E+00, 8.346976393096E-01,
       5.573195659183E-01, 2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02,
       9.071160757681E-02, 2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01,
       1.001366113450E+00, 8.845176151314E-01, 5.822972777850E-01, 1.015744035042E-01,
       -4.566308869047E-01, -3.573415380974E+00, -6.690199875043E+00, -9.806984369112E+00,
       -9.931104027256E+00, -6.669993535749E+00, -3.408883044242E+00, -1.477725527353E-01,
       4.717649724219E-01, 9.392642777670E-01, 1.001366113450E+00, 8.099953497723E-01,
       3.708915411190E-01, -1.307486146132E-01, -5.043249647001E-01, -7.603083192388E-01,
       -8.390032823170E-01, -7.603083192388E-01, -5.043249647001E-01, -1.307486146132E-01,
       3.708915411190E-01, 8.099953497723E-01, 1.001366113450E+00, 9.392642777670E-01,
       4.717649724219E-01, -1.477725527353E-01, -3.408883044242E+00, -6.669993535749E+00,
       -9.931104027256E+00, -7.799246504079E+00, -5.146535499003E+00, -2.493824493927E+00,
       1.588865111491E-01, 6.982843228335E-01, 1.043123512841E+00, 8.346976393096E-01,
       3.708915411190E-01, -2.159029407473E-01, -7.879132544621E-01, -1.137317867441E+00,
       -1.341871880972E+00, -1.411116709286E+00, -1.341871880972E+00, -1.137317867441E+00,
       -7.879132544621E-01, -2.159029407473E-01, 3.708915411190E-01, 8.346976393096E-01,
       1.043123512841E+00, 6.982843228335E-01, 1.588865111491E-01, -2.493824493927E+00,
       -5.146535499003E+00, -7.799246504079E+00, -4.821035766484E+00, -3.068661134588E+00,
       -1.316286502692E+00, 4.360881292036E-01, 8.365934740037E-01, 1.020213006502E+00,
       5.573195659183E-01, -1.307486146132E-01, -7.879132544621E-01, -1.348021656895E+00,
       -1.611357082932E+00, -1.722533378684E+00, -1.769347343907E+00, -1.722533378684E+00,
       -1.611357082932E+00, -1.348021656895E+00, -7.879132544621E-01, -1.307486146132E-01,
       5.573195659183E-01, 1.020213006502E+00, 8.365934740037E-01, 4.360881292036E-01,
       -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00, -2.144463029636E+00,
       -1.225075078640E+00, -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01,
       9.201634736832E-01, 2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00,
       -1.611357082932E+00, -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00,
       -1.764216023318E+00, -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00,
       -5.043249647001E-01, 2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01,
       6.137008233513E-01, -3.056871276444E-01, -1.225075078640E+00, -2.144463029636E+00,
       -1.066488253839E-01, 1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01,
       8.680361464225E-01, 8.099953497723E-01, 9.071160757681E-02, -7.603083192388E-01,
       -1.341871880972E+00, -1.722533378684E+00, -1.764216023318E+00, -1.685620535521E+00,
       -1.681006430126E+00, -1.685620535521E+00, -1.764216023318E+00, -1.722533378684E+00,
       -1.341871880972E+00, -7.603083192388E-01, 9.071160757681E-02, 8.099953497723E-01,
       8.680361464225E-01, 7.183309322567E-01, 4.433376797098E-01, 1.683444271630E-01,
       -1.066488253839E-01, 4.826495422536E-01, 5.729701223819E-01, 6.632907025101E-01,
       7.536112826384E-01, 8.719664804748E-01, 7.837181426811E-01, 3.238561278413E-02,
       -8.390032823170E-01, -1.411116709286E+00, -1.769347343907E+00, -1.783323390866E+00,
       -1.681006430126E+00, -1.669963815903E+00, -1.681006430126E+00, -1.783323390866E+00,
       -1.769347343907E+00, -1.411116709286E+00, -8.390032823170E-01, 3.238561278413E-02,
       7.837181426811E-01, 8.719664804748E-01, 7.536112826384E-01, 6.632907025101E-01,
       5.729701223818E-01, 4.826495422535E-01, -1.066488253839E-01, 1.683444271630E-01,
       4.433376797099E-01, 7.183309322567E-01, 8.680361464225E-01, 8.099953497723E-01,
       9.071160757681E-02, -7.603083192388E-01, -1.341871880972E+00, -1.722533378684E+00,
       -1.764216023318E+00, -1.685620535521E+00, -1.681006430126E+00, -1.685620535521E+00,
       -1.764216023318E+00, -1.722533378684E+00, -1.341871880972E+00, -7.603083192388E-01,
       9.071160757681E-02, 8.099953497723E-01, 8.680361464225E-01, 7.183309322567E-01,
       4.433376797098E-01, 1.683444271630E-01, -1.066488253839E-01, -2.144463029636E+00,
       -1.225075078640E+00, -3.056871276443E-01, 6.137008233513E-01, 8.751165915620E-01,
       9.201634736832E-01, 2.962626775268E-01, -5.043249647001E-01, -1.137317867441E+00,
       -1.611357082932E+00, -1.756976060514E+00, -1.764216023318E+00, -1.783323390866E+00,
       -1.764216023318E+00, -1.756976060514E+00, -1.611357082932E+00, -1.137317867441E+00,
       -5.043249647001E-01, 2.962626775268E-01, 9.201634736832E-01, 8.751165915620E-01,
       6.137008233513E-01, -3.056871276444E-01, -1.225075078640E+00, -2.144463029636E+00,
       -4.821035766484E+00, -3.068661134588E+00, -1.316286502692E+00, 4.360881292036E-01,
       8.365934740037E-01, 1.020213006502E+00, 5.573195659183E-01, -1.307486146132E-01,
       -7.879132544621E-01, -1.348021656895E+00, -1.611357082932E+00, -1.722533378684E+00,
       -1.769347343907E+00, -1.722533378684E+00, -1.611357082932E+00, -1.348021656895E+00,
       -7.879132544621E-01, -1.307486146132E-01, 5.573195659183E-01, 1.020213006502E+00,
       8.365934740037E-01, 4.360881292036E-01, -1.316286502692E+00, -3.068661134588E+00,
       -4.821035766484E+00, -7.799246504079E+00, -5.146535499003E+00, -2.493824493927E+00,
       1.588865111491E-01, 6.982843228335E-01, 1.043123512841E+00, 8.346976393096E-01,
       3.708915411190E-01, -2.159029407473E-01, -7.879132544621E-01, -1.137317867441E+00,
       -1.341871880972E+00, -1.411116709286E+00, -1.341871880972E+00, -1.137317867441E+00,
       -7.879132544621E-01, -2.159029407473E-01, 3.708915411190E-01, 8.346976393096E-01,
       1.043123512841E+00, 6.982843228335E-01, 1.588865111491E-01, -2.493824493927E+00,
       -5.146535499003E+00, -7.799246504079E+00, -9.931104027256E+00, -6.669993535749E+00,
       -3.408883044242E+00, -1.477725527353E-01, 4.717649724219E-01, 9.392642777670E-01,
       1.001366113450E+00, 8.099953497723E-01, 3.708915411190E-01, -1.307486146132E-01,
       -5.043249647001E-01, -7.603083192388E-01, -8.390032823170E-01, -7.603083192388E-01,
       -5.043249647001E-01, -1.307486146132E-01, 3.708915411190E-01, 8.099953497723E-01,
       1.001366113450E+00, 9.392642777670E-01, 4.717649724219E-01, -1.477725527353E-01,
       -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00, -9.806984369113E+00,
       -6.690199875043E+00, -3.573415380974E+00, -4.566308869047E-01, 1.015744035042E-01,
       5.822972777850E-01, 8.845176151314E-01, 1.001366113450E+00, 8.346976393096E-01,
       5.573195659183E-01, 2.962626775268E-01, 9.071160757681E-02, 3.238561278413E-02,
       9.071160757681E-02, 2.962626775268E-01, 5.573195659183E-01, 8.346976393096E-01,
       1.001366113450E+00, 8.845176151315E-01, 5.822972777850E-01, 1.015744035042E-01,
       -4.566308869047E-01, -3.573415380974E+00, -6.690199875043E+00, -9.806984369113E+00,
       -7.874665253011E+00, -5.471668476150E+00, -3.068671699289E+00, -6.656749224279E-01,
       -2.653592939274E-01, 1.353240031924E-01, 5.822972777850E-01, 9.392642777670E-01,
       1.043123512841E+00, 1.020213006502E+00, 9.201634736832E-01, 8.099953497723E-01,
       7.837181426811E-01, 8.099953497723E-01, 9.201634736832E-01, 1.020213006502E+00,
       1.043123512841E+00, 9.392642777670E-01, 5.822972777850E-01, 1.353240031924E-01,
       -2.653592939274E-01, -6.656749224279E-01, -3.068671699289E+00, -5.471668476150E+00,
       -7.874665253011E+00, -3.653094590856E+00, -2.636774001778E+00, -1.620453412699E+00,
       -6.041328236203E-01, -4.596429798149E-01, -2.653592939274E-01, 1.015744035042E-01,
       4.717649724219E-01, 6.982843228335E-01, 8.365934740037E-01, 8.751165915620E-01,
       8.680361464225E-01, 8.719664804748E-01, 8.680361464225E-01, 8.751165915620E-01,
       8.365934740037E-01, 6.982843228335E-01, 4.717649724219E-01, 1.015744035042E-01,
       -2.653592939274E-01, -4.596429798148E-01, -6.041328236203E-01, -1.620453412699E+00,
       -2.636774001778E+00, -3.653094590856E+00, 1.480862149879E+00, 8.369010281870E-01,
       1.929399064949E-01, -4.510212151972E-01, -6.041328236203E-01, -6.656749224279E-01,
       -4.566308869047E-01, -1.477725527353E-01, 1.588865111491E-01, 4.360881292036E-01,
       6.137008233513E-01, 7.183309322567E-01, 7.536112826384E-01, 7.183309322567E-01,
       6.137008233513E-01, 4.360881292036E-01, 1.588865111491E-01, -1.477725527353E-01,
       -4.566308869047E-01, -6.656749224279E-01, -6.041328236203E-01, -4.510212151972E-01,
       1.929399064947E-01, 8.369010281867E-01, 1.480862149879E+00, 2.954744435855E+01,
       1.976260954120E+01, 9.977774723847E+00, 1.929399064948E-01, -1.620453412699E+00,
       -3.068671699289E+00, -3.573415380974E+00, -3.408883044242E+00, -2.493824493927E+00,
       -1.316286502692E+00, -3.056871276443E-01, 4.433376797098E-01, 6.632907025101E-01,
       4.433376797098E-01, -3.056871276444E-01, -1.316286502692E+00, -2.493824493927E+00,
       -3.408883044242E+00, -3.573415380974E+00, -3.068671699289E+00, -1.620453412699E+00,
       1.929399064947E-01, 9.977774723845E+00, 1.976260954120E+01, 2.954744435855E+01,
       5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01, 8.369010281869E-01,
       -2.636774001778E+00, -5.471668476150E+00, -6.690199875043E+00, -6.669993535749E+00,
       -5.146535499003E+00, -3.068661134588E+00, -1.225075078640E+00, 1.683444271630E-01,
       5.729701223818E-01, 1.683444271630E-01, -1.225075078640E+00, -3.068661134588E+00,
       -5.146535499003E+00, -6.669993535749E+00, -6.690199875043E+00, -5.471668476150E+00,
       -2.636774001778E+00, 8.369010281867E-01, 1.976260954120E+01, 3.868831805420E+01,
       5.761402656721E+01, 8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01,
       1.480862149879E+00, -3.653094590857E+00, -7.874665253011E+00, -9.806984369112E+00,
       -9.931104027256E+00, -7.799246504079E+00, -4.821035766484E+00, -2.144463029636E+00,
       -1.066488253839E-01, 4.826495422536E-01, -1.066488253839E-01, -2.144463029636E+00,
       -4.821035766484E+00, -7.799246504079E+00, -9.931104027256E+00, -9.806984369113E+00,
       -7.874665253011E+00, -3.653094590856E+00, 1.480862149879E+00, 2.954744435855E+01,
       5.761402656721E+01, 8.568060877588E+01], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
         [[[-5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
            4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
            7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
            6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
            -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
            4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
            7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
            6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
            -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02]],

          [[-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-6.608547998817E-03, 1.589996401320E-02, 3.840847602522E-02, 3.681223104828E-02,
            3.521598607135E-02, 2.264579670372E-02, 1.007560733609E-02, 2.169822946944E-03,
            -5.735961442200E-03, -5.735961442200E-03, -5.735961442200E-03, 2.169822946944E-03,
            1.007560733609E-02, 2.264579670372E-02, 3.521598607135E-02, 3.681223104828E-02,
            3.840847602522E-02, 1.589996401320E-02, -6.608547998817E-03],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043689E-03, 8.865047043689E-03, 8.865047043689E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946944E-03, 4.132703513046E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946943E-03, 4.132703513046E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043690E-03, 8.865047043690E-03, 8.865047043690E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [-6.608547998817E-03, 1.589996401320E-02, 3.840847602522E-02, 3.681223104828E-02,
            3.521598607135E-02, 2.264579670372E-02, 1.007560733609E-02, 2.169822946944E-03,
            -5.735961442199E-03, -5.735961442199E-03, -5.735961442199E-03, 2.169822946944E-03,
            1.007560733609E-02, 2.264579670372E-02, 3.521598607135E-02, 3.681223104828E-02,
            3.840847602522E-02, 1.589996401320E-02, -6.608547998817E-03],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02]],

          [[-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [5.864644239018E-02, 5.438657790117E-02, 5.012671341215E-02, 8.893628036615E-03,
            -3.233945733892E-02, -5.760576128188E-02, -8.287206522483E-02, -8.709747172993E-02,
            -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -8.709747172993E-02,
            -8.287206522483E-02, -5.760576128188E-02, -3.233945733892E-02, 8.893628036615E-03,
            5.012671341215E-02, 5.438657790117E-02, 5.864644239018E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [5.864644239018E-02, 5.438657790117E-02, 5.012671341215E-02, 8.893628036615E-03,
            -3.233945733892E-02, -5.760576128188E-02, -8.287206522483E-02, -8.709747172993E-02,
            -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -8.709747172993E-02,
            -8.287206522483E-02, -5.760576128188E-02, -3.233945733892E-02, 8.893628036615E-03,
            5.012671341215E-02, 5.438657790117E-02, 5.864644239018E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02]],

          [[1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [4.677430482481E-02, 1.015276747481E-02, -2.646876987518E-02, -4.105770795551E-02,
            -5.564664603584E-02, -3.910092319068E-02, -2.255520034553E-02, -8.636073996132E-03,
            5.283052353262E-03, 5.283052353262E-03, 5.283052353262E-03, -8.636073996132E-03,
            -2.255520034553E-02, -3.910092319068E-02, -5.564664603584E-02, -4.105770795551E-02,
            -2.646876987518E-02, 1.015276747481E-02, 4.677430482481E-02],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [4.677430482481E-02, 1.015276747481E-02, -2.646876987518E-02, -4.105770795551E-02,
            -5.564664603584E-02, -3.910092319068E-02, -2.255520034553E-02, -8.636073996132E-03,
            5.283052353262E-03, 5.283052353262E-03, 5.283052353262E-03, -8.636073996132E-03,
            -2.255520034553E-02, -3.910092319068E-02, -5.564664603584E-02, -4.105770795551E-02,
            -2.646876987518E-02, 1.015276747481E-02, 4.677430482481E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02]],

          [[4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02]],

          [[5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-5.381741777814E-02, -4.570278076096E-02, -3.758814374379E-02, 1.895436340182E-02,
            7.549687054743E-02, 6.553610918883E-02, 5.557534783023E-02, 2.548082276045E-02,
            -4.613702309316E-03, -4.613702309316E-03, -4.613702309316E-03, 2.548082276045E-02,
            5.557534783023E-02, 6.553610918883E-02, 7.549687054743E-02, 1.895436340182E-02,
            -3.758814374379E-02, -4.570278076096E-02, -5.381741777814E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-5.381741777814E-02, -4.570278076096E-02, -3.758814374379E-02, 1.895436340182E-02,
            7.549687054743E-02, 6.553610918883E-02, 5.557534783023E-02, 2.548082276045E-02,
            -4.613702309316E-03, -4.613702309316E-03, -4.613702309316E-03, 2.548082276045E-02,
            5.557534783023E-02, 6.553610918883E-02, 7.549687054743E-02, 1.895436340182E-02,
            -3.758814374379E-02, -4.570278076096E-02, -5.381741777814E-02],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02]],

          [[6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043689E-03, 8.865047043689E-03, 8.865047043689E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-9.132287823503E-02, -2.150627942058E-02, 4.831031939387E-02, 7.895830598227E-02,
            1.096062925707E-01, 6.351909386754E-03, -9.690247379716E-02, -1.528128030916E-01,
            -2.087231323860E-01, -2.087231323860E-01, -2.087231323860E-01, -1.528128030916E-01,
            -9.690247379716E-02, 6.351909386754E-03, 1.096062925707E-01, 7.895830598227E-02,
            4.831031939387E-02, -2.150627942058E-02, -9.132287823503E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-9.132287823503E-02, -2.150627942058E-02, 4.831031939387E-02, 7.895830598227E-02,
            1.096062925707E-01, 6.351909386754E-03, -9.690247379716E-02, -1.528128030916E-01,
            -2.087231323860E-01, -2.087231323860E-01, -2.087231323860E-01, -1.528128030916E-01,
            -9.690247379716E-02, 6.351909386754E-03, 1.096062925707E-01, 7.895830598227E-02,
            4.831031939387E-02, -2.150627942058E-02, -9.132287823503E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043689E-03, 8.865047043689E-03, 8.865047043689E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02]],

          [[6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946944E-03, 4.132703513046E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-7.737881452018E-02, 8.304431197932E-03, 9.398767691604E-02, 5.640812765869E-02,
            1.882857840134E-02, -7.122019302118E-02, -1.612689644437E-01, -7.711798626249E-02,
            7.032991918724E-03, 7.032991918724E-03, 7.032991918724E-03, -7.711798626249E-02,
            -1.612689644437E-01, -7.122019302118E-02, 1.882857840134E-02, 5.640812765869E-02,
            9.398767691604E-02, 8.304431197932E-03, -7.737881452018E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-7.737881452018E-02, 8.304431197932E-03, 9.398767691604E-02, 5.640812765869E-02,
            1.882857840134E-02, -7.122019302118E-02, -1.612689644437E-01, -7.711798626249E-02,
            7.032991918724E-03, 7.032991918724E-03, 7.032991918724E-03, -7.711798626249E-02,
            -1.612689644437E-01, -7.122019302118E-02, 1.882857840134E-02, 5.640812765869E-02,
            9.398767691604E-02, 8.304431197932E-03, -7.737881452018E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946944E-03, 4.132703513046E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02]],

          [[7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442200E-03, 3.478504395026E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [3.478504395026E-02, -5.735961442199E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02]],

          [[7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442200E-03, 3.478504395026E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [3.478504395026E-02, -5.735961442199E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02]],

          [[7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442200E-03, 3.478504395026E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353262E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918724E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-5.367000806700E-02, 3.699391012493E-02, 1.276578283168E-01, 1.537767725984E-02,
            -9.690247379716E-02, -1.138147965015E-01, -1.307271192057E-01, 1.837911096333E-01,
            4.983093384723E-01, 4.983093384723E-01, 4.983093384723E-01, 1.837911096333E-01,
            -1.307271192057E-01, -1.138147965015E-01, -9.690247379716E-02, 1.537767725984E-02,
            1.276578283168E-01, 3.699391012493E-02, -5.367000806700E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [3.478504395026E-02, -5.735961442199E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02]],

          [[6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946944E-03, 4.132703513046E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-7.737881452018E-02, 8.304431197932E-03, 9.398767691604E-02, 5.640812765869E-02,
            1.882857840134E-02, -7.122019302118E-02, -1.612689644437E-01, -7.711798626249E-02,
            7.032991918724E-03, 7.032991918724E-03, 7.032991918724E-03, -7.711798626249E-02,
            -1.612689644437E-01, -7.122019302118E-02, 1.882857840134E-02, 5.640812765869E-02,
            9.398767691604E-02, 8.304431197932E-03, -7.737881452018E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-6.796559697817E-02, 2.292947858431E-02, 1.138245541468E-01, 4.051297047808E-02,
            -3.279861319062E-02, -1.012618694932E-01, -1.697251257959E-01, 7.032991918724E-03,
            1.837911096333E-01, 1.837911096333E-01, 1.837911096333E-01, 7.032991918725E-03,
            -1.697251257959E-01, -1.012618694932E-01, -3.279861319062E-02, 4.051297047808E-02,
            1.138245541468E-01, 2.292947858431E-02, -6.796559697817E-02],
           [-7.737881452018E-02, 8.304431197932E-03, 9.398767691604E-02, 5.640812765869E-02,
            1.882857840134E-02, -7.122019302118E-02, -1.612689644437E-01, -7.711798626249E-02,
            7.032991918725E-03, 7.032991918725E-03, 7.032991918725E-03, -7.711798626249E-02,
            -1.612689644437E-01, -7.122019302118E-02, 1.882857840134E-02, 5.640812765869E-02,
            9.398767691604E-02, 8.304431197932E-03, -7.737881452018E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946943E-03, 4.132703513046E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02]],

          [[6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043690E-03, 8.865047043690E-03, 8.865047043690E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-9.132287823503E-02, -2.150627942058E-02, 4.831031939387E-02, 7.895830598227E-02,
            1.096062925707E-01, 6.351909386754E-03, -9.690247379716E-02, -1.528128030916E-01,
            -2.087231323860E-01, -2.087231323860E-01, -2.087231323860E-01, -1.528128030916E-01,
            -9.690247379716E-02, 6.351909386754E-03, 1.096062925707E-01, 7.895830598227E-02,
            4.831031939387E-02, -2.150627942058E-02, -9.132287823503E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.226118588934E-02, 8.865047043690E-03, 9.999127997672E-02, 6.564826369632E-02,
            3.130524741593E-02, -8.870894248502E-02, -2.087231323860E-01, -1.697251257959E-01,
            -1.307271192057E-01, -1.307271192057E-01, -1.307271192057E-01, -1.697251257959E-01,
            -2.087231323860E-01, -8.870894248502E-02, 3.130524741593E-02, 6.564826369632E-02,
            9.999127997672E-02, 8.865047043690E-03, -8.226118588934E-02],
           [-8.679203206218E-02, -6.320616188443E-03, 7.415079968530E-02, 7.230328483930E-02,
            7.045576999330E-02, -4.117851654913E-02, -1.528128030916E-01, -1.612689644437E-01,
            -1.697251257959E-01, -1.697251257959E-01, -1.697251257959E-01, -1.612689644437E-01,
            -1.528128030916E-01, -4.117851654913E-02, 7.045576999330E-02, 7.230328483930E-02,
            7.415079968530E-02, -6.320616188443E-03, -8.679203206218E-02],
           [-9.132287823503E-02, -2.150627942058E-02, 4.831031939387E-02, 7.895830598227E-02,
            1.096062925707E-01, 6.351909386754E-03, -9.690247379716E-02, -1.528128030916E-01,
            -2.087231323860E-01, -2.087231323860E-01, -2.087231323860E-01, -1.528128030916E-01,
            -9.690247379716E-02, 6.351909386754E-03, 1.096062925707E-01, 7.895830598227E-02,
            4.831031939387E-02, -2.150627942058E-02, -9.132287823503E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043690E-03, 8.865047043690E-03, 8.865047043690E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02]],

          [[5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-5.381741777814E-02, -4.570278076096E-02, -3.758814374379E-02, 1.895436340182E-02,
            7.549687054743E-02, 6.553610918883E-02, 5.557534783023E-02, 2.548082276045E-02,
            -4.613702309316E-03, -4.613702309316E-03, -4.613702309316E-03, 2.548082276045E-02,
            5.557534783023E-02, 6.553610918883E-02, 7.549687054743E-02, 1.895436340182E-02,
            -3.758814374379E-02, -4.570278076096E-02, -5.381741777814E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-8.256662555709E-02, -1.818583536796E-02, 4.619495482117E-02, 6.283824634378E-02,
            7.948153786639E-02, -4.613702309316E-03, -8.870894248502E-02, -1.012618694932E-01,
            -1.138147965015E-01, -1.138147965015E-01, -1.138147965015E-01, -1.012618694932E-01,
            -8.870894248502E-02, -4.613702309316E-03, 7.948153786639E-02, 6.283824634378E-02,
            4.619495482117E-02, -1.818583536796E-02, -8.256662555709E-02],
           [-7.945849951771E-02, -2.885047213770E-02, 2.175755524230E-02, 5.694885865617E-02,
            9.214016207004E-02, 2.548082276045E-02, -4.117851654913E-02, -7.122019302118E-02,
            -1.012618694932E-01, -1.012618694932E-01, -1.012618694932E-01, -7.122019302118E-02,
            -4.117851654913E-02, 2.548082276045E-02, 9.214016207004E-02, 5.694885865617E-02,
            2.175755524230E-02, -2.885047213770E-02, -7.945849951771E-02],
           [-7.635037347832E-02, -3.951510890744E-02, -2.679844336562E-03, 5.105947096857E-02,
            1.047987862737E-01, 5.557534783023E-02, 6.351909386754E-03, -4.117851654913E-02,
            -8.870894248502E-02, -8.870894248502E-02, -8.870894248502E-02, -4.117851654913E-02,
            6.351909386754E-03, 5.557534783023E-02, 1.047987862737E-01, 5.105947096857E-02,
            -2.679844336562E-03, -3.951510890744E-02, -7.635037347832E-02],
           [-5.381741777814E-02, -4.570278076096E-02, -3.758814374379E-02, 1.895436340182E-02,
            7.549687054743E-02, 6.553610918883E-02, 5.557534783023E-02, 2.548082276045E-02,
            -4.613702309316E-03, -4.613702309316E-03, -4.613702309316E-03, 2.548082276045E-02,
            5.557534783023E-02, 6.553610918883E-02, 7.549687054743E-02, 1.895436340182E-02,
            -3.758814374379E-02, -4.570278076096E-02, -5.381741777814E-02],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02]],

          [[4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-8.287206522483E-02, -4.523671777961E-02, -7.601370334387E-03, 6.002822899123E-02,
            1.276578283168E-01, 7.948153786639E-02, 3.130524741593E-02, -3.279861319062E-02,
            -9.690247379716E-02, -9.690247379716E-02, -9.690247379716E-02, -3.279861319062E-02,
            3.130524741593E-02, 7.948153786639E-02, 1.276578283168E-01, 6.002822899123E-02,
            -7.601370334387E-03, -4.523671777961E-02, -8.287206522483E-02],
           [-7.212496697323E-02, -5.138032808696E-02, -3.063568920069E-02, 4.159443247305E-02,
            1.138245541468E-01, 9.214016207004E-02, 7.045576999330E-02, 1.882857840134E-02,
            -3.279861319062E-02, -3.279861319062E-02, -3.279861319062E-02, 1.882857840134E-02,
            7.045576999330E-02, 9.214016207004E-02, 1.138245541468E-01, 4.159443247305E-02,
            -3.063568920069E-02, -5.138032808696E-02, -7.212496697323E-02],
           [-6.137786872162E-02, -5.752393839431E-02, -5.367000806700E-02, 2.316063595486E-02,
            9.999127997672E-02, 1.047987862737E-01, 1.096062925707E-01, 7.045576999330E-02,
            3.130524741593E-02, 3.130524741593E-02, 3.130524741593E-02, 7.045576999330E-02,
            1.096062925707E-01, 1.047987862737E-01, 9.999127997672E-02, 2.316063595486E-02,
            -5.367000806700E-02, -5.752393839431E-02, -6.137786872162E-02],
           [-3.128446207796E-02, -5.189045261448E-02, -7.249644315101E-02, -1.315074416492E-02,
            4.619495482117E-02, 7.549687054743E-02, 1.047987862737E-01, 9.214016207004E-02,
            7.948153786639E-02, 7.948153786639E-02, 7.948153786639E-02, 9.214016207004E-02,
            1.047987862737E-01, 7.549687054743E-02, 4.619495482117E-02, -1.315074416492E-02,
            -7.249644315101E-02, -5.189045261448E-02, -3.128446207796E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334386E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02]],

          [[1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [4.677430482481E-02, 1.015276747481E-02, -2.646876987518E-02, -4.105770795551E-02,
            -5.564664603584E-02, -3.910092319068E-02, -2.255520034553E-02, -8.636073996132E-03,
            5.283052353262E-03, 5.283052353262E-03, 5.283052353262E-03, -8.636073996132E-03,
            -2.255520034553E-02, -3.910092319068E-02, -5.564664603584E-02, -4.105770795551E-02,
            -2.646876987518E-02, 1.015276747481E-02, 4.677430482481E-02],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-4.203156032956E-02, -4.574684230713E-02, -4.946212428471E-02, 5.283052353262E-03,
            6.002822899123E-02, 6.283824634378E-02, 6.564826369632E-02, 4.051297047808E-02,
            1.537767725984E-02, 1.537767725984E-02, 1.537767725984E-02, 4.051297047808E-02,
            6.564826369632E-02, 6.283824634378E-02, 6.002822899123E-02, 5.283052353263E-03,
            -4.946212428471E-02, -4.574684230713E-02, -4.203156032956E-02],
           [-2.950113685822E-02, -4.418385866176E-02, -5.886658046531E-02, -8.636073996132E-03,
            4.159443247305E-02, 5.694885865617E-02, 7.230328483930E-02, 5.640812765869E-02,
            4.051297047808E-02, 4.051297047808E-02, 4.051297047808E-02, 5.640812765869E-02,
            7.230328483930E-02, 5.694885865617E-02, 4.159443247305E-02, -8.636073996132E-03,
            -5.886658046531E-02, -4.418385866176E-02, -2.950113685822E-02],
           [-1.697071338687E-02, -4.262087501639E-02, -6.827103664592E-02, -2.255520034553E-02,
            2.316063595486E-02, 5.105947096857E-02, 7.895830598227E-02, 7.230328483930E-02,
            6.564826369632E-02, 6.564826369632E-02, 6.564826369632E-02, 7.230328483930E-02,
            7.895830598227E-02, 5.105947096857E-02, 2.316063595486E-02, -2.255520034553E-02,
            -6.827103664592E-02, -4.262087501639E-02, -1.697071338687E-02],
           [7.422108490909E-03, -2.881449686277E-02, -6.505110221644E-02, -3.910092319068E-02,
            -1.315074416492E-02, 1.895436340182E-02, 5.105947096857E-02, 5.694885865617E-02,
            6.283824634378E-02, 6.283824634378E-02, 6.283824634378E-02, 5.694885865617E-02,
            5.105947096857E-02, 1.895436340182E-02, -1.315074416492E-02, -3.910092319068E-02,
            -6.505110221644E-02, -2.881449686277E-02, 7.422108490909E-03],
           [3.181493036869E-02, -1.500811870914E-02, -6.183116778697E-02, -5.564664603584E-02,
            -4.946212428471E-02, -1.315074416492E-02, 2.316063595486E-02, 4.159443247305E-02,
            6.002822899123E-02, 6.002822899123E-02, 6.002822899123E-02, 4.159443247305E-02,
            2.316063595486E-02, -1.315074416492E-02, -4.946212428471E-02, -5.564664603584E-02,
            -6.183116778697E-02, -1.500811870914E-02, 3.181493036869E-02],
           [4.677430482481E-02, 1.015276747481E-02, -2.646876987518E-02, -4.105770795551E-02,
            -5.564664603584E-02, -3.910092319068E-02, -2.255520034553E-02, -8.636073996132E-03,
            5.283052353263E-03, 5.283052353263E-03, 5.283052353263E-03, -8.636073996132E-03,
            -2.255520034553E-02, -3.910092319068E-02, -5.564664603584E-02, -4.105770795551E-02,
            -2.646876987518E-02, 1.015276747481E-02, 4.677430482481E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02]],

          [[-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [5.864644239018E-02, 5.438657790117E-02, 5.012671341215E-02, 8.893628036615E-03,
            -3.233945733892E-02, -5.760576128188E-02, -8.287206522483E-02, -8.709747172993E-02,
            -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -8.709747172993E-02,
            -8.287206522483E-02, -5.760576128188E-02, -3.233945733892E-02, 8.893628036615E-03,
            5.012671341215E-02, 5.438657790117E-02, 5.864644239018E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [-1.191055434292E-03, -4.625696683466E-02, -9.132287823503E-02, -4.946212428471E-02,
            -7.601370334387E-03, 4.619495482117E-02, 9.999127997672E-02, 1.138245541468E-01,
            1.276578283168E-01, 1.276578283168E-01, 1.276578283168E-01, 1.138245541468E-01,
            9.999127997672E-02, 4.619495482117E-02, -7.601370334387E-03, -4.946212428471E-02,
            -9.132287823503E-02, -4.625696683466E-02, -1.191055434292E-03],
           [1.312269325679E-02, -3.698738923657E-02, -8.709747172993E-02, -5.886658046531E-02,
            -3.063568920069E-02, 2.175755524230E-02, 7.415079968530E-02, 9.398767691604E-02,
            1.138245541468E-01, 1.138245541468E-01, 1.138245541468E-01, 9.398767691604E-02,
            7.415079968530E-02, 2.175755524230E-02, -3.063568920069E-02, -5.886658046531E-02,
            -8.709747172993E-02, -3.698738923657E-02, 1.312269325679E-02],
           [2.743644194787E-02, -2.771781163848E-02, -8.287206522483E-02, -6.827103664592E-02,
            -5.367000806700E-02, -2.679844336562E-03, 4.831031939387E-02, 7.415079968530E-02,
            9.999127997672E-02, 9.999127997672E-02, 9.999127997672E-02, 7.415079968530E-02,
            4.831031939387E-02, -2.679844336562E-03, -5.367000806700E-02, -6.827103664592E-02,
            -8.287206522483E-02, -2.771781163848E-02, 2.743644194787E-02],
           [4.612867905977E-02, -5.738541111052E-03, -5.760576128188E-02, -6.505110221644E-02,
            -7.249644315101E-02, -3.758814374379E-02, -2.679844336562E-03, 2.175755524230E-02,
            4.619495482117E-02, 4.619495482117E-02, 4.619495482117E-02, 2.175755524230E-02,
            -2.679844336562E-03, -3.758814374379E-02, -7.249644315101E-02, -6.505110221644E-02,
            -5.760576128188E-02, -5.738541111052E-03, 4.612867905977E-02],
           [6.482091617167E-02, 1.624072941638E-02, -3.233945733892E-02, -6.183116778697E-02,
            -9.132287823503E-02, -7.249644315101E-02, -5.367000806700E-02, -3.063568920069E-02,
            -7.601370334387E-03, -7.601370334387E-03, -7.601370334387E-03, -3.063568920069E-02,
            -5.367000806700E-02, -7.249644315101E-02, -9.132287823503E-02, -6.183116778697E-02,
            -3.233945733892E-02, 1.624072941638E-02, 6.482091617167E-02],
           [6.173367928093E-02, 3.531365365877E-02, 8.893628036615E-03, -2.646876987518E-02,
            -6.183116778697E-02, -6.505110221644E-02, -6.827103664592E-02, -5.886658046531E-02,
            -4.946212428471E-02, -4.946212428471E-02, -4.946212428471E-02, -5.886658046531E-02,
            -6.827103664592E-02, -6.505110221644E-02, -6.183116778697E-02, -2.646876987518E-02,
            8.893628036615E-03, 3.531365365877E-02, 6.173367928093E-02],
           [5.864644239018E-02, 5.438657790117E-02, 5.012671341215E-02, 8.893628036615E-03,
            -3.233945733892E-02, -5.760576128188E-02, -8.287206522483E-02, -8.709747172993E-02,
            -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -8.709747172993E-02,
            -8.287206522483E-02, -5.760576128188E-02, -3.233945733892E-02, 8.893628036615E-03,
            5.012671341215E-02, 5.438657790117E-02, 5.864644239018E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02]],

          [[-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-6.608547998817E-03, 1.589996401320E-02, 3.840847602522E-02, 3.681223104828E-02,
            3.521598607135E-02, 2.264579670372E-02, 1.007560733609E-02, 2.169822946944E-03,
            -5.735961442200E-03, -5.735961442200E-03, -5.735961442200E-03, 2.169822946944E-03,
            1.007560733609E-02, 2.264579670372E-02, 3.521598607135E-02, 3.681223104828E-02,
            3.840847602522E-02, 1.589996401320E-02, -6.608547998817E-03],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043689E-03, 8.865047043689E-03, 8.865047043689E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946943E-03, 4.132703513046E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [3.478504395026E-02, -5.735961442200E-03, -4.625696683466E-02, -4.574684230713E-02,
            -4.523671777961E-02, -1.818583536796E-02, 8.865047043689E-03, 2.292947858431E-02,
            3.699391012493E-02, 3.699391012493E-02, 3.699391012493E-02, 2.292947858431E-02,
            8.865047043690E-03, -1.818583536796E-02, -4.523671777961E-02, -4.574684230713E-02,
            -4.625696683466E-02, -5.735961442199E-03, 3.478504395026E-02],
           [4.132703513046E-02, 2.169822946944E-03, -3.698738923657E-02, -4.418385866176E-02,
            -5.138032808696E-02, -2.885047213770E-02, -6.320616188443E-03, 8.304431197932E-03,
            2.292947858431E-02, 2.292947858431E-02, 2.292947858431E-02, 8.304431197932E-03,
            -6.320616188443E-03, -2.885047213770E-02, -5.138032808696E-02, -4.418385866176E-02,
            -3.698738923657E-02, 2.169822946943E-03, 4.132703513046E-02],
           [4.786902631065E-02, 1.007560733609E-02, -2.771781163848E-02, -4.262087501639E-02,
            -5.752393839431E-02, -3.951510890744E-02, -2.150627942058E-02, -6.320616188443E-03,
            8.865047043690E-03, 8.865047043690E-03, 8.865047043690E-03, -6.320616188443E-03,
            -2.150627942058E-02, -3.951510890744E-02, -5.752393839431E-02, -4.262087501639E-02,
            -2.771781163848E-02, 1.007560733609E-02, 4.786902631065E-02],
           [5.103013451849E-02, 2.264579670372E-02, -5.738541111052E-03, -2.881449686277E-02,
            -5.189045261448E-02, -4.570278076096E-02, -3.951510890744E-02, -2.885047213770E-02,
            -1.818583536796E-02, -1.818583536796E-02, -1.818583536796E-02, -2.885047213770E-02,
            -3.951510890744E-02, -4.570278076096E-02, -5.189045261448E-02, -2.881449686277E-02,
            -5.738541111052E-03, 2.264579670372E-02, 5.103013451849E-02],
           [5.419124272632E-02, 3.521598607135E-02, 1.624072941638E-02, -1.500811870914E-02,
            -4.625696683466E-02, -5.189045261448E-02, -5.752393839431E-02, -5.138032808696E-02,
            -4.523671777961E-02, -4.523671777961E-02, -4.523671777961E-02, -5.138032808696E-02,
            -5.752393839431E-02, -5.189045261448E-02, -4.625696683466E-02, -1.500811870914E-02,
            1.624072941638E-02, 3.521598607135E-02, 5.419124272632E-02],
           [3.831080843780E-02, 3.681223104828E-02, 3.531365365877E-02, 1.015276747481E-02,
            -1.500811870914E-02, -2.881449686277E-02, -4.262087501639E-02, -4.418385866176E-02,
            -4.574684230713E-02, -4.574684230713E-02, -4.574684230713E-02, -4.418385866176E-02,
            -4.262087501639E-02, -2.881449686277E-02, -1.500811870914E-02, 1.015276747481E-02,
            3.531365365877E-02, 3.681223104828E-02, 3.831080843780E-02],
           [2.243037414927E-02, 3.840847602522E-02, 5.438657790117E-02, 3.531365365877E-02,
            1.624072941638E-02, -5.738541111052E-03, -2.771781163848E-02, -3.698738923657E-02,
            -4.625696683466E-02, -4.625696683466E-02, -4.625696683466E-02, -3.698738923657E-02,
            -2.771781163848E-02, -5.738541111052E-03, 1.624072941638E-02, 3.531365365877E-02,
            5.438657790117E-02, 3.840847602522E-02, 2.243037414927E-02],
           [-6.608547998817E-03, 1.589996401320E-02, 3.840847602522E-02, 3.681223104828E-02,
            3.521598607135E-02, 2.264579670372E-02, 1.007560733609E-02, 2.169822946944E-03,
            -5.735961442199E-03, -5.735961442199E-03, -5.735961442199E-03, 2.169822946943E-03,
            1.007560733609E-02, 2.264579670372E-02, 3.521598607135E-02, 3.681223104828E-02,
            3.840847602522E-02, 1.589996401320E-02, -6.608547998817E-03],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02]],

          [[-5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
            4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
            7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
            6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
            -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
            -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
            -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
            -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
            -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02],
           [6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
            -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
            -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
            -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
            1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02],
           [6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
            -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
            -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
            -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
            2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02],
           [5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
            -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
            -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
            -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
            4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02],
           [4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
            -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
            -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
            -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
            6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02],
           [1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
            3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
            -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
            -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
            6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02],
           [-1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
            6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
            -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
            2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
            5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02],
           [-3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
            5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
            3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
            4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
            2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02],
           [-5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
            4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
            7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
            6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
            -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02]]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
            [-5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02,
             6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02,
             -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02,
             -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02,
             2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02,
             -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02,
             -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02,
             5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02,
             2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02,
             6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02,
             1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02,
             6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02,
             -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02,
             -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02,
             3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02,
             -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03,
             3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02,
             3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02,
             4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02,
             3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02,
             5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02,
             2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02,
             6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02,
             7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02,
             -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02,
             7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02,
             2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02,
             -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02,
             -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02,
             7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02,
             -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02,
             7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02,
             6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02,
             -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02,
             -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02,
             3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02,
             4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02,
             3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03,
             2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02,
             4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02,
             5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03,
             -6.608547998817E-03, -6.608547998817E-03, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03,
             -6.608547998817E-03, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02,
             3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02,
             4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02,
             3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02,
             5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02,
             2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02,
             6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 5.864644239018E-02, 5.864644239018E-02,
             5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 5.864644239018E-02, 5.864644239018E-02,
             5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02,
             1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02,
             6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02,
             4.677430482481E-02, 4.677430482481E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02,
             4.677430482481E-02, 4.677430482481E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02,
             6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02,
             -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02,
             -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02,
             3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02,
             -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03,
             3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02,
             4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02,
             -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02,
             -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02,
             -5.381741777814E-02, -5.381741777814E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02,
             -5.381741777814E-02, -5.381741777814E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02,
             7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02,
             -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02,
             7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02,
             2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02,
             -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02,
             -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -9.132287823503E-02, -9.132287823503E-02,
             -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -9.132287823503E-02, -9.132287823503E-02,
             -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02,
             -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02,
             -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02,
             -7.737881452018E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02,
             -7.737881452018E-02, -7.737881452018E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02,
             -7.737881452018E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02, -7.737881452018E-02,
             -7.737881452018E-02, -7.737881452018E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02,
             -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02,
             -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02,
             -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02,
             4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02,
             2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02,
             -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02,
             -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -9.132287823503E-02, -9.132287823503E-02,
             -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02, -8.679203206218E-02,
             -8.679203206218E-02, -8.679203206218E-02, -9.132287823503E-02, -9.132287823503E-02,
             -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02, -9.132287823503E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02,
             -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02,
             -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02,
             4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02,
             -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02,
             -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02,
             -5.381741777814E-02, -5.381741777814E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02, -7.945849951771E-02,
             -7.945849951771E-02, -7.945849951771E-02, -7.635037347832E-02, -7.635037347832E-02,
             -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02, -7.635037347832E-02,
             -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02, -5.381741777814E-02,
             -5.381741777814E-02, -5.381741777814E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02,
             7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02,
             -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02,
             -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02,
             7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02,
             6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02,
             -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02,
             -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02, -7.212496697323E-02,
             -7.212496697323E-02, -7.212496697323E-02, -6.137786872162E-02, -6.137786872162E-02,
             -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02, -6.137786872162E-02,
             -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02, -3.128446207796E-02,
             -3.128446207796E-02, -3.128446207796E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02,
             3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02,
             -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03,
             3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02,
             4.677430482481E-02, 4.677430482481E-02, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02, -2.950113685822E-02,
             -2.950113685822E-02, -2.950113685822E-02, -1.697071338687E-02, -1.697071338687E-02,
             -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02, -1.697071338687E-02,
             7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03, 7.422108490909E-03,
             7.422108490909E-03, 7.422108490909E-03, 3.181493036869E-02, 3.181493036869E-02,
             3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02, 3.181493036869E-02,
             4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02, 4.677430482481E-02,
             4.677430482481E-02, 4.677430482481E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02,
             7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02,
             -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02,
             7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02,
             3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02,
             5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02,
             2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02,
             6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 5.864644239018E-02, 5.864644239018E-02,
             5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02, 1.312269325679E-02,
             1.312269325679E-02, 1.312269325679E-02, 2.743644194787E-02, 2.743644194787E-02,
             2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02, 2.743644194787E-02,
             4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02, 4.612867905977E-02,
             4.612867905977E-02, 4.612867905977E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02, 6.482091617167E-02,
             6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02, 6.173367928093E-02,
             6.173367928093E-02, 6.173367928093E-02, 5.864644239018E-02, 5.864644239018E-02,
             5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02, 5.864644239018E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02,
             1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02,
             6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03,
             2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02,
             4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02,
             5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03,
             -6.608547998817E-03, -6.608547998817E-03, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02, 4.132703513046E-02,
             4.132703513046E-02, 4.132703513046E-02, 4.786902631065E-02, 4.786902631065E-02,
             4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02, 5.103013451849E-02,
             5.103013451849E-02, 5.103013451849E-02, 5.419124272632E-02, 5.419124272632E-02,
             5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02, 5.419124272632E-02,
             3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02, 3.831080843780E-02,
             3.831080843780E-02, 3.831080843780E-02, 2.243037414927E-02, 2.243037414927E-02,
             2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02, 2.243037414927E-02,
             -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03, -6.608547998817E-03,
             -6.608547998817E-03, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02,
             3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02,
             4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02,
             4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02,
             3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03,
             2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02,
             4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02,
             5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02,
             3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02,
             -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03,
             3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02,
             4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02,
             -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02,
             -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02, -2.950113685822E-02,
             -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02, -7.737881452018E-02,
             -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02, -7.737881452018E-02,
             -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02, -2.950113685822E-02,
             1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02,
             -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02,
             -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.103013451849E-02,
             4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02, -5.381741777814E-02,
             -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02, -8.256662555709E-02,
             -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02, -5.381741777814E-02,
             -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02, 5.103013451849E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02, 4.677430482481E-02,
             3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02, -2.950113685822E-02,
             -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02, -2.950113685822E-02,
             -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02, 4.677430482481E-02,
             6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02,
             1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02,
             6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -6.608547998817E-03,
             2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02, 5.103013451849E-02,
             4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02, 3.478504395026E-02,
             3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02, 5.103013451849E-02,
             5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02, -6.608547998817E-03,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02,
             6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02,
             -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02,
             2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02,
             5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02,
             6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02,
             -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02,
             -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02,
             -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02,
             -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02,
             -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02,
             2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02,
             -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02,
             -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02,
             -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02,
             -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02, 1.312269325679E-02,
             -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02, -8.679203206218E-02,
             -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02, -6.796559697817E-02,
             -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02, -7.212496697323E-02,
             -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02,
             2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02,
             -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02,
             -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03, -3.128446207796E-02,
             -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02, -8.256662555709E-02,
             -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02, -7.635037347832E-02,
             -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03, 4.612867905977E-02,
             5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02,
             -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02,
             -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02,
             -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02,
             6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 1.488793759467E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02, 6.173367928093E-02,
             4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03, -1.697071338687E-02,
             -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02, -4.203156032956E-02,
             -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03, 3.181493036869E-02,
             4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02,
             5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02,
             2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03,
             -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02,
             6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02, 5.419124272632E-02,
             5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02, 3.478504395026E-02,
             3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02, 4.786902631065E-02,
             5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02, 2.243037414927E-02,
             -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, 2.243037414927E-02, 5.864644239018E-02,
             6.173367928093E-02, 6.482091617167E-02, 4.612867905977E-02, 2.743644194787E-02,
             1.312269325679E-02, -1.191055434292E-03, -1.191055434292E-03, -1.191055434292E-03,
             1.312269325679E-02, 2.743644194787E-02, 4.612867905977E-02, 6.482091617167E-02,
             6.173367928093E-02, 5.864644239018E-02, 2.243037414927E-02, -1.378569409163E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             5.419124272632E-02, 6.482091617167E-02, 3.181493036869E-02, -1.191055434292E-03,
             -3.128446207796E-02, -6.137786872162E-02, -7.212496697323E-02, -8.287206522483E-02,
             -8.287206522483E-02, -8.287206522483E-02, -7.212496697323E-02, -6.137786872162E-02,
             -3.128446207796E-02, -1.191055434292E-03, 3.181493036869E-02, 6.482091617167E-02,
             5.419124272632E-02, 4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 4.786902631065E-02, 2.743644194787E-02,
             -1.697071338687E-02, -6.137786872162E-02, -7.635037347832E-02, -9.132287823503E-02,
             -8.679203206218E-02, -8.226118588934E-02, -8.226118588934E-02, -8.226118588934E-02,
             -8.679203206218E-02, -9.132287823503E-02, -7.635037347832E-02, -6.137786872162E-02,
             -1.697071338687E-02, 2.743644194787E-02, 4.786902631065E-02, 6.830161067344E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02, -8.287206522483E-02,
             -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02, -5.367000806700E-02,
             -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02, -8.226118588934E-02,
             -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02, -1.191055434292E-03,
             3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03, -4.203156032956E-02,
             -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02, -6.796559697817E-02,
             -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02, -6.796559697817E-02,
             -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02, -4.203156032956E-02,
             -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 3.478504395026E-02, -1.191055434292E-03,
             -4.203156032956E-02, -8.287206522483E-02, -8.256662555709E-02, -8.226118588934E-02,
             -6.796559697817E-02, -5.367000806700E-02, -5.367000806700E-02, -5.367000806700E-02,
             -6.796559697817E-02, -8.226118588934E-02, -8.256662555709E-02, -8.287206522483E-02,
             -4.203156032956E-02, -1.191055434292E-03, 3.478504395026E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 4.132703513046E-02,
             1.312269325679E-02, -2.950113685822E-02, -7.212496697323E-02, -7.945849951771E-02,
             -8.679203206218E-02, -7.737881452018E-02, -6.796559697817E-02, -6.796559697817E-02,
             -6.796559697817E-02, -7.737881452018E-02, -8.679203206218E-02, -7.945849951771E-02,
             -7.212496697323E-02, -2.950113685822E-02, 1.312269325679E-02, 4.132703513046E-02,
             6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02, 6.953137700412E-02,
             6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             4.786902631065E-02, 2.743644194787E-02, -1.697071338687E-02, -6.137786872162E-02,
             -7.635037347832E-02, -9.132287823503E-02, -8.679203206218E-02, -8.226118588934E-02,
             -8.226118588934E-02, -8.226118588934E-02, -8.679203206218E-02, -9.132287823503E-02,
             -7.635037347832E-02, -6.137786872162E-02, -1.697071338687E-02, 2.743644194787E-02,
             4.786902631065E-02, 6.830161067344E-02, 6.830161067344E-02, 6.830161067344E-02,
             6.830161067344E-02, 5.593158997720E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.103013451849E-02, 4.612867905977E-02, 7.422108490909E-03,
             -3.128446207796E-02, -5.381741777814E-02, -7.635037347832E-02, -7.945849951771E-02,
             -8.256662555709E-02, -8.256662555709E-02, -8.256662555709E-02, -7.945849951771E-02,
             -7.635037347832E-02, -5.381741777814E-02, -3.128446207796E-02, 7.422108490909E-03,
             4.612867905977E-02, 5.103013451849E-02, 5.593158997720E-02, 5.593158997720E-02,
             5.593158997720E-02, 5.593158997720E-02, 4.356156928097E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 5.419124272632E-02, 6.482091617167E-02,
             3.181493036869E-02, -1.191055434292E-03, -3.128446207796E-02, -6.137786872162E-02,
             -7.212496697323E-02, -8.287206522483E-02, -8.287206522483E-02, -8.287206522483E-02,
             -7.212496697323E-02, -6.137786872162E-02, -3.128446207796E-02, -1.191055434292E-03,
             3.181493036869E-02, 6.482091617167E-02, 5.419124272632E-02, 4.356156928097E-02,
             4.356156928097E-02, 4.356156928097E-02, 4.356156928097E-02, 1.488793759467E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 3.831080843780E-02,
             6.173367928093E-02, 4.677430482481E-02, 3.181493036869E-02, 7.422108490909E-03,
             -1.697071338687E-02, -2.950113685822E-02, -4.203156032956E-02, -4.203156032956E-02,
             -4.203156032956E-02, -2.950113685822E-02, -1.697071338687E-02, 7.422108490909E-03,
             3.181493036869E-02, 4.677430482481E-02, 6.173367928093E-02, 3.831080843780E-02,
             1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02, 1.488793759467E-02,
             -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             2.243037414927E-02, 5.864644239018E-02, 6.173367928093E-02, 6.482091617167E-02,
             4.612867905977E-02, 2.743644194787E-02, 1.312269325679E-02, -1.191055434292E-03,
             -1.191055434292E-03, -1.191055434292E-03, 1.312269325679E-02, 2.743644194787E-02,
             4.612867905977E-02, 6.482091617167E-02, 6.173367928093E-02, 5.864644239018E-02,
             2.243037414927E-02, -1.378569409163E-02, -1.378569409163E-02, -1.378569409163E-02,
             -1.378569409163E-02, -3.564747014691E-02, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -6.608547998817E-03, 2.243037414927E-02, 3.831080843780E-02,
             5.419124272632E-02, 5.103013451849E-02, 4.786902631065E-02, 4.132703513046E-02,
             3.478504395026E-02, 3.478504395026E-02, 3.478504395026E-02, 4.132703513046E-02,
             4.786902631065E-02, 5.103013451849E-02, 5.419124272632E-02, 3.831080843780E-02,
             2.243037414927E-02, -6.608547998817E-03, -3.564747014691E-02, -3.564747014691E-02,
             -3.564747014691E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02,
             1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02,
             6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02,
             6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02,
             1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -3.564747014691E-02,
             -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02, 5.593158997720E-02,
             6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02, 7.076114333481E-02,
             7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02, 5.593158997720E-02,
             4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02, -3.564747014691E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02, 4.356156928097E-02,
             5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02, 7.076114333481E-02,
             7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02, 6.830161067344E-02,
             5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02, -1.378569409163E-02,
             -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -3.564747014691E-02, -1.378569409163E-02, 1.488793759467E-02,
             4.356156928097E-02, 5.593158997720E-02, 6.830161067344E-02, 6.953137700412E-02,
             7.076114333481E-02, 7.076114333481E-02, 7.076114333481E-02, 6.953137700412E-02,
             6.830161067344E-02, 5.593158997720E-02, 4.356156928097E-02, 1.488793759467E-02,
             -1.378569409163E-02, -3.564747014691E-02, -5.750924620219E-02, -5.750924620219E-02,
             -5.750924620219E-02, -5.750924620219E-02], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
            [8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01, 1.480862149879E+00,
       -3.196901551566E+00, -7.874665253011E+00, -8.902884640134E+00, -9.931104027256E+00,
       -7.376069896870E+00, -4.821035766484E+00, -2.463842295934E+00, -1.066488253839E-01,
       -1.066488253839E-01, -1.066488253839E-01, -2.463842295934E+00, -4.821035766484E+00,
       -7.376069896870E+00, -9.931104027256E+00, -8.902884640134E+00, -7.874665253011E+00,
       -3.196901551566E+00, 1.480862149879E+00, 2.954744435855E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.317383723982E+00, -5.471668476150E+00, -6.070831005950E+00,
       -6.669993535749E+00, -4.869327335169E+00, -3.068661134588E+00, -1.450158353713E+00,
       1.683444271630E-01, 1.683444271630E-01, 1.683444271630E-01, -1.450158353713E+00,
       -3.068661134588E+00, -4.869327335169E+00, -6.669993535749E+00, -6.070831005950E+00,
       -5.471668476150E+00, -2.317383723982E+00, 8.369010281869E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.437865896397E+00, -3.068671699289E+00,
       -3.238777371766E+00, -3.408883044242E+00, -2.362584773467E+00, -1.316286502692E+00,
       -4.364744114912E-01, 4.433376797099E-01, 4.433376797099E-01, 4.433376797099E-01,
       -4.364744114912E-01, -1.316286502692E+00, -2.362584773467E+00, -3.408883044242E+00,
       -3.238777371766E+00, -3.068671699289E+00, -1.437865896397E+00, 1.929399064948E-01,
       9.977774723846E+00, 1.976260954120E+01, 2.954744435855E+01, 1.480862149879E+00,
       8.369010281869E-01, 1.929399064948E-01, -4.510212151972E-01, -5.583480688126E-01,
       -6.656749224279E-01, -4.067237375816E-01, -1.477725527353E-01, 1.441577882341E-01,
       4.360881292036E-01, 5.772095307301E-01, 7.183309322567E-01, 7.183309322567E-01,
       7.183309322567E-01, 5.772095307301E-01, 4.360881292036E-01, 1.441577882341E-01,
       -1.477725527353E-01, -4.067237375816E-01, -6.656749224279E-01, -5.583480688126E-01,
       -4.510212151972E-01, 1.929399064948E-01, 8.369010281869E-01, 1.480862149879E+00,
       -3.196901551566E+00, -2.317383723982E+00, -1.437865896397E+00, -5.583480688126E-01,
       -4.117617642152E-01, -2.651754596177E-01, 6.528520144905E-02, 3.957458625158E-01,
       5.619482151843E-01, 7.281505678529E-01, 7.461568544337E-01, 7.641631410145E-01,
       7.641631410145E-01, 7.641631410145E-01, 7.461568544337E-01, 7.281505678529E-01,
       5.619482151843E-01, 3.957458625158E-01, 6.528520144905E-02, -2.651754596177E-01,
       -4.117617642152E-01, -5.583480688126E-01, -1.437865896397E+00, -2.317383723982E+00,
       -3.196901551566E+00, -7.874665253011E+00, -5.471668476150E+00, -3.068671699289E+00,
       -6.656749224279E-01, -2.651754596177E-01, 1.353240031925E-01, 5.372941404797E-01,
       9.392642777670E-01, 9.797386421346E-01, 1.020213006502E+00, 9.151041781373E-01,
       8.099953497724E-01, 8.099953497724E-01, 8.099953497724E-01, 9.151041781373E-01,
       1.020213006502E+00, 9.797386421346E-01, 9.392642777670E-01, 5.372941404797E-01,
       1.353240031925E-01, -2.651754596177E-01, -6.656749224279E-01, -3.068671699289E+00,
       -5.471668476150E+00, -7.874665253011E+00, -8.902884640134E+00, -6.070831005950E+00,
       -3.238777371766E+00, -4.067237375816E-01, 6.528520144905E-02, 5.372941404797E-01,
       7.059619771247E-01, 8.746298137697E-01, 6.596810048571E-01, 4.447321959445E-01,
       2.347878556056E-01, 2.484351526677E-02, 2.484351526677E-02, 2.484351526677E-02,
       2.347878556056E-01, 4.447321959445E-01, 6.596810048571E-01, 8.746298137697E-01,
       7.059619771247E-01, 5.372941404797E-01, 6.528520144904E-02, -4.067237375816E-01,
       -3.238777371766E+00, -6.070831005950E+00, -8.902884640134E+00, -9.931104027256E+00,
       -6.669993535749E+00, -3.408883044242E+00, -1.477725527353E-01, 3.957458625158E-01,
       9.392642777670E-01, 8.746298137697E-01, 8.099953497724E-01, 3.396233675796E-01,
       -1.307486146132E-01, -4.455284669260E-01, -7.603083192388E-01, -7.603083192388E-01,
       -7.603083192388E-01, -4.455284669260E-01, -1.307486146132E-01, 3.396233675796E-01,
       8.099953497724E-01, 8.746298137697E-01, 9.392642777670E-01, 3.957458625158E-01,
       -1.477725527353E-01, -3.408883044242E+00, -6.669993535749E+00, -9.931104027256E+00,
       -7.376069896870E+00, -4.869327335169E+00, -2.362584773467E+00, 1.441577882341E-01,
       5.619482151843E-01, 9.797386421346E-01, 6.596810048571E-01, 3.396233675796E-01,
       -1.998808840873E-01, -7.393851357542E-01, -9.904029923578E-01, -1.241420848961E+00,
       -1.241420848961E+00, -1.241420848961E+00, -9.904029923578E-01, -7.393851357542E-01,
       -1.998808840873E-01, 3.396233675796E-01, 6.596810048571E-01, 9.797386421346E-01,
       5.619482151843E-01, 1.441577882341E-01, -2.362584773467E+00, -4.869327335169E+00,
       -7.376069896870E+00, -4.821035766484E+00, -3.068661134588E+00, -1.316286502692E+00,
       4.360881292036E-01, 7.281505678529E-01, 1.020213006502E+00, 4.447321959445E-01,
       -1.307486146132E-01, -7.393851357542E-01, -1.348021656895E+00, -1.535277517790E+00,
       -1.722533378684E+00, -1.722533378684E+00, -1.722533378684E+00, -1.535277517790E+00,
       -1.348021656895E+00, -7.393851357542E-01, -1.307486146132E-01, 4.447321959445E-01,
       1.020213006502E+00, 7.281505678529E-01, 4.360881292036E-01, -1.316286502692E+00,
       -3.068661134588E+00, -4.821035766484E+00, -2.463842295934E+00, -1.450158353713E+00,
       -4.364744114912E-01, 5.772095307301E-01, 7.461568544337E-01, 9.151041781373E-01,
       2.347878556056E-01, -4.455284669260E-01, -9.904029923578E-01, -1.535277517790E+00,
       -1.619677237446E+00, -1.704076957103E+00, -1.704076957103E+00, -1.704076957103E+00,
       -1.619677237446E+00, -1.535277517790E+00, -9.904029923578E-01, -4.455284669260E-01,
       2.347878556056E-01, 9.151041781373E-01, 7.461568544337E-01, 5.772095307301E-01,
       -4.364744114912E-01, -1.450158353713E+00, -2.463842295934E+00, -1.066488253839E-01,
       1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01, 7.641631410145E-01,
       8.099953497724E-01, 2.484351526677E-02, -7.603083192388E-01, -1.241420848961E+00,
       -1.722533378684E+00, -1.704076957103E+00, -1.685620535521E+00, -1.685620535521E+00,
       -1.685620535521E+00, -1.704076957103E+00, -1.722533378684E+00, -1.241420848961E+00,
       -7.603083192388E-01, 2.484351526677E-02, 8.099953497724E-01, 7.641631410145E-01,
       7.183309322567E-01, 4.433376797099E-01, 1.683444271630E-01, -1.066488253839E-01,
       -1.066488253839E-01, 1.683444271630E-01, 4.433376797099E-01, 7.183309322567E-01,
       7.641631410145E-01, 8.099953497724E-01, 2.484351526677E-02, -7.603083192388E-01,
       -1.241420848961E+00, -1.722533378684E+00, -1.704076957103E+00, -1.685620535521E+00,
       -1.685620535521E+00, -1.685620535521E+00, -1.704076957103E+00, -1.722533378684E+00,
       -1.241420848961E+00, -7.603083192388E-01, 2.484351526677E-02, 8.099953497724E-01,
       7.641631410145E-01, 7.183309322567E-01, 4.433376797099E-01, 1.683444271630E-01,
       -1.066488253839E-01, -1.066488253839E-01, 1.683444271630E-01, 4.433376797099E-01,
       7.183309322567E-01, 7.641631410145E-01, 8.099953497724E-01, 2.484351526677E-02,
       -7.603083192388E-01, -1.241420848961E+00, -1.722533378684E+00, -1.704076957103E+00,
       -1.685620535521E+00, -1.685620535521E+00, -1.685620535521E+00, -1.704076957103E+00,
       -1.722533378684E+00, -1.241420848961E+00, -7.603083192388E-01, 2.484351526677E-02,
       8.099953497724E-01, 7.641631410145E-01, 7.183309322567E-01, 4.433376797099E-01,
       1.683444271630E-01, -1.066488253839E-01, -2.463842295934E+00, -1.450158353713E+00,
       -4.364744114912E-01, 5.772095307301E-01, 7.461568544337E-01, 9.151041781373E-01,
       2.347878556056E-01, -4.455284669260E-01, -9.904029923578E-01, -1.535277517790E+00,
       -1.619677237446E+00, -1.704076957103E+00, -1.704076957103E+00, -1.704076957103E+00,
       -1.619677237446E+00, -1.535277517790E+00, -9.904029923578E-01, -4.455284669260E-01,
       2.347878556056E-01, 9.151041781373E-01, 7.461568544337E-01, 5.772095307301E-01,
       -4.364744114912E-01, -1.450158353713E+00, -2.463842295934E+00, -4.821035766484E+00,
       -3.068661134588E+00, -1.316286502692E+00, 4.360881292036E-01, 7.281505678529E-01,
       1.020213006502E+00, 4.447321959445E-01, -1.307486146132E-01, -7.393851357542E-01,
       -1.348021656895E+00, -1.535277517790E+00, -1.722533378684E+00, -1.722533378684E+00,
       -1.722533378684E+00, -1.535277517790E+00, -1.348021656895E+00, -7.393851357542E-01,
       -1.307486146132E-01, 4.447321959445E-01, 1.020213006502E+00, 7.281505678529E-01,
       4.360881292036E-01, -1.316286502692E+00, -3.068661134588E+00, -4.821035766484E+00,
       -7.376069896870E+00, -4.869327335169E+00, -2.362584773467E+00, 1.441577882341E-01,
       5.619482151843E-01, 9.797386421346E-01, 6.596810048571E-01, 3.396233675796E-01,
       -1.998808840873E-01, -7.393851357542E-01, -9.904029923578E-01, -1.241420848961E+00,
       -1.241420848961E+00, -1.241420848961E+00, -9.904029923578E-01, -7.393851357542E-01,
       -1.998808840873E-01, 3.396233675796E-01, 6.596810048571E-01, 9.797386421346E-01,
       5.619482151843E-01, 1.441577882341E-01, -2.362584773467E+00, -4.869327335169E+00,
       -7.376069896870E+00, -9.931104027256E+00, -6.669993535749E+00, -3.408883044242E+00,
       -1.477725527353E-01, 3.957458625158E-01, 9.392642777670E-01, 8.746298137697E-01,
       8.099953497724E-01, 3.396233675796E-01, -1.307486146132E-01, -4.455284669260E-01,
       -7.603083192388E-01, -7.603083192388E-01, -7.603083192388E-01, -4.455284669260E-01,
       -1.307486146132E-01, 3.396233675796E-01, 8.099953497724E-01, 8.746298137697E-01,
       9.392642777670E-01, 3.957458625158E-01, -1.477725527353E-01, -3.408883044242E+00,
       -6.669993535749E+00, -9.931104027256E+00, -8.902884640134E+00, -6.070831005950E+00,
       -3.238777371766E+00, -4.067237375816E-01, 6.528520144905E-02, 5.372941404797E-01,
       7.059619771247E-01, 8.746298137697E-01, 6.596810048571E-01, 4.447321959445E-01,
       2.347878556056E-01, 2.484351526677E-02, 2.484351526677E-02, 2.484351526677E-02,
       2.347878556056E-01, 4.447321959445E-01, 6.596810048571E-01, 8.746298137697E-01,
       7.059619771247E-01, 5.372941404797E-01, 6.528520144905E-02, -4.067237375816E-01,
       -3.238777371766E+00, -6.070831005950E+00, -8.902884640134E+00, -7.874665253011E+00,
       -5.471668476150E+00, -3.068671699289E+00, -6.656749224279E-01, -2.651754596177E-01,
       1.353240031925E-01, 5.372941404797E-01, 9.392642777670E-01, 9.797386421346E-01,
       1.020213006502E+00, 9.151041781373E-01, 8.099953497724E-01, 8.099953497724E-01,
       8.099953497724E-01, 9.151041781373E-01, 1.020213006502E+00, 9.797386421346E-01,
       9.392642777670E-01, 5.372941404797E-01, 1.353240031925E-01, -2.651754596177E-01,
       -6.656749224279E-01, -3.068671699289E+00, -5.471668476150E+00, -7.874665253011E+00,
       -3.196901551566E+00, -2.317383723982E+00, -1.437865896397E+00, -5.583480688126E-01,
       -4.117617642152E-01, -2.651754596177E-01, 6.528520144904E-02, 3.957458625158E-01,
       5.619482151843E-01, 7.281505678529E-01, 7.461568544337E-01, 7.641631410145E-01,
       7.641631410145E-01, 7.641631410145E-01, 7.461568544337E-01, 7.281505678529E-01,
       5.619482151843E-01, 3.957458625158E-01, 6.528520144905E-02, -2.651754596177E-01,
       -4.117617642152E-01, -5.583480688126E-01, -1.437865896397E+00, -2.317383723982E+00,
       -3.196901551566E+00, 1.480862149879E+00, 8.369010281869E-01, 1.929399064948E-01,
       -4.510212151972E-01, -5.583480688126E-01, -6.656749224279E-01, -4.067237375816E-01,
       -1.477725527353E-01, 1.441577882341E-01, 4.360881292036E-01, 5.772095307301E-01,
       7.183309322567E-01, 7.183309322567E-01, 7.183309322567E-01, 5.772095307301E-01,
       4.360881292036E-01, 1.441577882341E-01, -1.477725527353E-01, -4.067237375816E-01,
       -6.656749224279E-01, -5.583480688126E-01, -4.510212151972E-01, 1.929399064948E-01,
       8.369010281869E-01, 1.480862149879E+00, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.437865896397E+00, -3.068671699289E+00,
       -3.238777371766E+00, -3.408883044242E+00, -2.362584773467E+00, -1.316286502692E+00,
       -4.364744114912E-01, 4.433376797099E-01, 4.433376797099E-01, 4.433376797099E-01,
       -4.364744114912E-01, -1.316286502692E+00, -2.362584773467E+00, -3.408883044242E+00,
       -3.238777371766E+00, -3.068671699289E+00, -1.437865896397E+00, 1.929399064948E-01,
       9.977774723846E+00, 1.976260954120E+01, 2.954744435855E+01, 5.761402656722E+01,
       3.868831805421E+01, 1.976260954120E+01, 8.369010281869E-01, -2.317383723982E+00,
       -5.471668476150E+00, -6.070831005950E+00, -6.669993535749E+00, -4.869327335169E+00,
       -3.068661134588E+00, -1.450158353713E+00, 1.683444271630E-01, 1.683444271630E-01,
       1.683444271630E-01, -1.450158353713E+00, -3.068661134588E+00, -4.869327335169E+00,
       -6.669993535749E+00, -6.070831005950E+00, -5.471668476150E+00, -2.317383723982E+00,
       8.369010281869E-01, 1.976260954120E+01, 3.868831805421E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 2.954744435855E+01, 1.480862149879E+00,
       -3.196901551566E+00, -7.874665253011E+00, -8.902884640134E+00, -9.931104027256E+00,
       -7.376069896870E+00, -4.821035766484E+00, -2.463842295934E+00, -1.066488253839E-01,
       -1.066488253839E-01, -1.066488253839E-01, -2.463842295934E+00, -4.821035766484E+00,
       -7.376069896870E+00, -9.931104027256E+00, -8.902884640134E+00, -7.874665253011E+00,
       -3.196901551566E+00, 1.480862149879E+00, 2.954744435855E+01, 5.761402656722E+01,
       8.568060877589E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.317383723982E+00, -5.471668476150E+00, -6.070831005950E+00,
       -6.669993535749E+00, -4.869327335169E+00, -3.068661134588E+00, -1.450158353713E+00,
       1.683444271630E-01, 1.683444271630E-01, 1.683444271630E-01, -1.450158353713E+00,
       -3.068661134588E+00, -4.869327335169E+00, -6.669993535749E+00, -6.070831005950E+00,
       -5.471668476150E+00, -2.317383723982E+00, 8.369010281869E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 3.868831805421E+01, 2.594265109142E+01,
       1.319698412863E+01, 4.513171658361E-01, -1.672960661296E+00, -3.797238488428E+00,
       -4.135949508141E+00, -4.474660527854E+00, -3.207968538285E+00, -1.941276548716E+00,
       -8.307777426456E-01, 2.797210634251E-01, 2.797210634251E-01, 2.797210634251E-01,
       -8.307777426456E-01, -1.941276548716E+00, -3.207968538285E+00, -4.474660527854E+00,
       -4.135949508141E+00, -3.797238488428E+00, -1.672960661296E+00, 4.513171658361E-01,
       1.319698412863E+01, 2.594265109142E+01, 3.868831805421E+01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.028537598610E+00,
       -2.122808500705E+00, -2.201068010332E+00, -2.279327519959E+00, -1.546609741402E+00,
       -8.138919628448E-01, -2.113971315787E-01, 3.910976996873E-01, 3.910976996873E-01,
       3.910976996873E-01, -2.113971315787E-01, -8.138919628448E-01, -1.546609741402E+00,
       -2.279327519959E+00, -2.201068010332E+00, -2.122808500705E+00, -1.028537598610E+00,
       6.573330348527E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       8.369010281869E-01, 4.513171658361E-01, 6.573330348527E-02, -3.198505588656E-01,
       -3.841145359240E-01, -4.483785129825E-01, -2.661865125229E-01, -8.399451206323E-02,
       1.147490554818E-01, 3.134926230269E-01, 4.079834794881E-01, 5.024743359494E-01,
       5.024743359494E-01, 5.024743359494E-01, 4.079834794881E-01, 3.134926230269E-01,
       1.147490554818E-01, -8.399451206323E-02, -2.661865125229E-01, -4.483785129825E-01,
       -3.841145359240E-01, -3.198505588656E-01, 6.573330348527E-02, 4.513171658361E-01,
       8.369010281869E-01, -2.317383723982E+00, -1.672960661296E+00, -1.028537598610E+00,
       -3.841145359240E-01, -2.767106921430E-01, -1.693068483621E-01, 5.629373711196E-02,
       2.818943225860E-01, 3.916421882957E-01, 5.013900540055E-01, 5.112135813326E-01,
       5.210371086598E-01, 5.210371086598E-01, 5.210371086598E-01, 5.112135813326E-01,
       5.013900540055E-01, 3.916421882957E-01, 2.818943225860E-01, 5.629373711196E-02,
       -1.693068483621E-01, -2.767106921430E-01, -3.841145359240E-01, -1.028537598610E+00,
       -1.672960661296E+00, -2.317383723982E+00, -5.471668476150E+00, -3.797238488428E+00,
       -2.122808500705E+00, -4.483785129825E-01, -1.693068483621E-01, 1.097648162584E-01,
       3.787739867468E-01, 6.477831572352E-01, 6.685353211096E-01, 6.892874849841E-01,
       6.144436831771E-01, 5.395998813701E-01, 5.395998813701E-01, 5.395998813701E-01,
       6.144436831771E-01, 6.892874849841E-01, 6.685353211096E-01, 6.477831572352E-01,
       3.787739867468E-01, 1.097648162584E-01, -1.693068483621E-01, -4.483785129825E-01,
       -2.122808500705E+00, -3.797238488428E+00, -5.471668476150E+00, -6.070831005950E+00,
       -4.135949508141E+00, -2.201068010332E+00, -2.661865125229E-01, 5.629373711196E-02,
       3.787739867468E-01, 4.862327530247E-01, 5.936915193027E-01, 4.422613727350E-01,
       2.908312261674E-01, 1.466915247843E-01, 2.551823401324E-03, 2.551823401324E-03,
       2.551823401324E-03, 1.466915247843E-01, 2.908312261674E-01, 4.422613727350E-01,
       5.936915193027E-01, 4.862327530247E-01, 3.787739867468E-01, 5.629373711196E-02,
       -2.661865125229E-01, -2.201068010332E+00, -4.135949508141E+00, -6.070831005950E+00,
       -6.669993535749E+00, -4.474660527854E+00, -2.279327519959E+00, -8.399451206323E-02,
       2.818943225860E-01, 6.477831572352E-01, 5.936915193027E-01, 5.395998813701E-01,
       2.159874243604E-01, -1.076250326493E-01, -3.210606336084E-01, -5.344962345675E-01,
       -5.344962345675E-01, -5.344962345675E-01, -3.210606336084E-01, -1.076250326493E-01,
       2.159874243604E-01, 5.395998813701E-01, 5.936915193027E-01, 6.477831572352E-01,
       2.818943225860E-01, -8.399451206323E-02, -2.279327519959E+00, -4.474660527854E+00,
       -6.669993535749E+00, -4.869327335169E+00, -3.207968538285E+00, -1.546609741402E+00,
       1.147490554818E-01, 3.916421882957E-01, 6.685353211096E-01, 4.422613727350E-01,
       2.159874243604E-01, -1.511930619842E-01, -5.183735483289E-01, -6.867548280777E-01,
       -8.551361078266E-01, -8.551361078266E-01, -8.551361078266E-01, -6.867548280777E-01,
       -5.183735483289E-01, -1.511930619842E-01, 2.159874243604E-01, 4.422613727350E-01,
       6.685353211096E-01, 3.916421882957E-01, 1.147490554818E-01, -1.546609741402E+00,
       -3.207968538285E+00, -4.869327335169E+00, -3.068661134588E+00, -1.941276548716E+00,
       -8.138919628448E-01, 3.134926230269E-01, 5.013900540055E-01, 6.892874849841E-01,
       2.908312261674E-01, -1.076250326493E-01, -5.183735483289E-01, -9.291220640084E-01,
       -1.052449022547E+00, -1.175775981086E+00, -1.175775981086E+00, -1.175775981086E+00,
       -1.052449022547E+00, -9.291220640084E-01, -5.183735483289E-01, -1.076250326493E-01,
       2.908312261674E-01, 6.892874849841E-01, 5.013900540055E-01, 3.134926230269E-01,
       -8.138919628448E-01, -1.941276548716E+00, -3.068661134588E+00, -1.450158353713E+00,
       -8.307777426456E-01, -2.113971315787E-01, 4.079834794881E-01, 5.112135813326E-01,
       6.144436831771E-01, 1.466915247843E-01, -3.210606336084E-01, -6.867548280777E-01,
       -1.052449022547E+00, -1.105577763137E+00, -1.158706503728E+00, -1.158706503728E+00,
       -1.158706503728E+00, -1.105577763137E+00, -1.052449022547E+00, -6.867548280777E-01,
       -3.210606336084E-01, 1.466915247843E-01, 6.144436831771E-01, 5.112135813326E-01,
       4.079834794881E-01, -2.113971315787E-01, -8.307777426456E-01, -1.450158353713E+00,
       1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01, 5.024743359494E-01,
       5.210371086598E-01, 5.395998813701E-01, 2.551823401324E-03, -5.344962345675E-01,
       -8.551361078266E-01, -1.175775981086E+00, -1.158706503728E+00, -1.141637026370E+00,
       -1.141637026370E+00, -1.141637026370E+00, -1.158706503728E+00, -1.175775981086E+00,
       -8.551361078266E-01, -5.344962345675E-01, 2.551823401324E-03, 5.395998813701E-01,
       5.210371086598E-01, 5.024743359494E-01, 3.910976996873E-01, 2.797210634251E-01,
       1.683444271630E-01, 1.683444271630E-01, 2.797210634251E-01, 3.910976996873E-01,
       5.024743359494E-01, 5.210371086598E-01, 5.395998813701E-01, 2.551823401324E-03,
       -5.344962345675E-01, -8.551361078266E-01, -1.175775981086E+00, -1.158706503728E+00,
       -1.141637026370E+00, -1.141637026370E+00, -1.141637026370E+00, -1.158706503728E+00,
       -1.175775981086E+00, -8.551361078266E-01, -5.344962345675E-01, 2.551823401324E-03,
       5.395998813701E-01, 5.210371086598E-01, 5.024743359494E-01, 3.910976996873E-01,
       2.797210634251E-01, 1.683444271630E-01, 1.683444271630E-01, 2.797210634251E-01,
       3.910976996873E-01, 5.024743359494E-01, 5.210371086598E-01, 5.395998813701E-01,
       2.551823401324E-03, -5.344962345675E-01, -8.551361078266E-01, -1.175775981086E+00,
       -1.158706503728E+00, -1.141637026370E+00, -1.141637026370E+00, -1.141637026370E+00,
       -1.158706503728E+00, -1.175775981086E+00, -8.551361078266E-01, -5.344962345675E-01,
       2.551823401324E-03, 5.395998813701E-01, 5.210371086598E-01, 5.024743359494E-01,
       3.910976996873E-01, 2.797210634251E-01, 1.683444271630E-01, -1.450158353713E+00,
       -8.307777426456E-01, -2.113971315787E-01, 4.079834794881E-01, 5.112135813326E-01,
       6.144436831771E-01, 1.466915247843E-01, -3.210606336084E-01, -6.867548280777E-01,
       -1.052449022547E+00, -1.105577763137E+00, -1.158706503728E+00, -1.158706503728E+00,
       -1.158706503728E+00, -1.105577763137E+00, -1.052449022547E+00, -6.867548280777E-01,
       -3.210606336084E-01, 1.466915247843E-01, 6.144436831771E-01, 5.112135813326E-01,
       4.079834794881E-01, -2.113971315787E-01, -8.307777426456E-01, -1.450158353713E+00,
       -3.068661134588E+00, -1.941276548716E+00, -8.138919628448E-01, 3.134926230269E-01,
       5.013900540055E-01, 6.892874849841E-01, 2.908312261674E-01, -1.076250326493E-01,
       -5.183735483289E-01, -9.291220640084E-01, -1.052449022547E+00, -1.175775981086E+00,
       -1.175775981086E+00, -1.175775981086E+00, -1.052449022547E+00, -9.291220640084E-01,
       -5.183735483289E-01, -1.076250326493E-01, 2.908312261674E-01, 6.892874849841E-01,
       5.013900540055E-01, 3.134926230269E-01, -8.138919628448E-01, -1.941276548716E+00,
       -3.068661134588E+00, -4.869327335169E+00, -3.207968538285E+00, -1.546609741402E+00,
       1.147490554818E-01, 3.916421882957E-01, 6.685353211096E-01, 4.422613727350E-01,
       2.159874243604E-01, -1.511930619842E-01, -5.183735483289E-01, -6.867548280777E-01,
       -8.551361078266E-01, -8.551361078266E-01, -8.551361078266E-01, -6.867548280777E-01,
       -5.183735483289E-01, -1.511930619842E-01, 2.159874243604E-01, 4.422613727350E-01,
       6.685353211096E-01, 3.916421882957E-01, 1.147490554818E-01, -1.546609741402E+00,
       -3.207968538285E+00, -4.869327335169E+00, -6.669993535749E+00, -4.474660527854E+00,
       -2.279327519959E+00, -8.399451206323E-02, 2.818943225860E-01, 6.477831572352E-01,
       5.936915193027E-01, 5.395998813701E-01, 2.159874243604E-01, -1.076250326493E-01,
       -3.210606336084E-01, -5.344962345675E-01, -5.344962345675E-01, -5.344962345675E-01,
       -3.210606336084E-01, -1.076250326493E-01, 2.159874243604E-01, 5.395998813701E-01,
       5.936915193027E-01, 6.477831572352E-01, 2.818943225860E-01, -8.399451206323E-02,
       -2.279327519959E+00, -4.474660527854E+00, -6.669993535749E+00, -6.070831005950E+00,
       -4.135949508141E+00, -2.201068010332E+00, -2.661865125229E-01, 5.629373711196E-02,
       3.787739867468E-01, 4.862327530247E-01, 5.936915193027E-01, 4.422613727350E-01,
       2.908312261674E-01, 1.466915247843E-01, 2.551823401324E-03, 2.551823401324E-03,
       2.551823401324E-03, 1.466915247843E-01, 2.908312261674E-01, 4.422613727350E-01,
       5.936915193027E-01, 4.862327530247E-01, 3.787739867468E-01, 5.629373711196E-02,
       -2.661865125229E-01, -2.201068010332E+00, -4.135949508141E+00, -6.070831005950E+00,
       -5.471668476150E+00, -3.797238488428E+00, -2.122808500705E+00, -4.483785129825E-01,
       -1.693068483621E-01, 1.097648162584E-01, 3.787739867468E-01, 6.477831572352E-01,
       6.685353211096E-01, 6.892874849841E-01, 6.144436831771E-01, 5.395998813701E-01,
       5.395998813701E-01, 5.395998813701E-01, 6.144436831771E-01, 6.892874849841E-01,
       6.685353211096E-01, 6.477831572352E-01, 3.787739867468E-01, 1.097648162584E-01,
       -1.693068483621E-01, -4.483785129825E-01, -2.122808500705E+00, -3.797238488428E+00,
       -5.471668476150E+00, -2.317383723982E+00, -1.672960661296E+00, -1.028537598610E+00,
       -3.841145359240E-01, -2.767106921430E-01, -1.693068483621E-01, 5.629373711196E-02,
       2.818943225860E-01, 3.916421882957E-01, 5.013900540055E-01, 5.112135813326E-01,
       5.210371086598E-01, 5.210371086598E-01, 5.210371086598E-01, 5.112135813326E-01,
       5.013900540055E-01, 3.916421882957E-01, 2.818943225860E-01, 5.629373711196E-02,
       -1.693068483621E-01, -2.767106921430E-01, -3.841145359240E-01, -1.028537598610E+00,
       -1.672960661296E+00, -2.317383723982E+00, 8.369010281869E-01, 4.513171658361E-01,
       6.573330348527E-02, -3.198505588656E-01, -3.841145359240E-01, -4.483785129825E-01,
       -2.661865125229E-01, -8.399451206323E-02, 1.147490554818E-01, 3.134926230269E-01,
       4.079834794881E-01, 5.024743359494E-01, 5.024743359494E-01, 5.024743359494E-01,
       4.079834794881E-01, 3.134926230269E-01, 1.147490554818E-01, -8.399451206323E-02,
       -2.661865125229E-01, -4.483785129825E-01, -3.841145359240E-01, -3.198505588656E-01,
       6.573330348527E-02, 4.513171658361E-01, 8.369010281869E-01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.028537598610E+00,
       -2.122808500705E+00, -2.201068010332E+00, -2.279327519959E+00, -1.546609741402E+00,
       -8.138919628448E-01, -2.113971315787E-01, 3.910976996873E-01, 3.910976996873E-01,
       3.910976996873E-01, -2.113971315787E-01, -8.138919628448E-01, -1.546609741402E+00,
       -2.279327519959E+00, -2.201068010332E+00, -2.122808500705E+00, -1.028537598610E+00,
       6.573330348527E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       3.868831805421E+01, 2.594265109142E+01, 1.319698412863E+01, 4.513171658361E-01,
       -1.672960661296E+00, -3.797238488428E+00, -4.135949508141E+00, -4.474660527854E+00,
       -3.207968538285E+00, -1.941276548716E+00, -8.307777426456E-01, 2.797210634251E-01,
       2.797210634251E-01, 2.797210634251E-01, -8.307777426456E-01, -1.941276548716E+00,
       -3.207968538285E+00, -4.474660527854E+00, -4.135949508141E+00, -3.797238488428E+00,
       -1.672960661296E+00, 4.513171658361E-01, 1.319698412863E+01, 2.594265109142E+01,
       3.868831805421E+01, 5.761402656722E+01, 3.868831805421E+01, 1.976260954120E+01,
       8.369010281869E-01, -2.317383723982E+00, -5.471668476150E+00, -6.070831005950E+00,
       -6.669993535749E+00, -4.869327335169E+00, -3.068661134588E+00, -1.450158353713E+00,
       1.683444271630E-01, 1.683444271630E-01, 1.683444271630E-01, -1.450158353713E+00,
       -3.068661134588E+00, -4.869327335169E+00, -6.669993535749E+00, -6.070831005950E+00,
       -5.471668476150E+00, -2.317383723982E+00, 8.369010281869E-01, 1.976260954120E+01,
       3.868831805421E+01, 5.761402656722E+01, 2.954744435855E+01, 1.976260954120E+01,
       9.977774723846E+00, 1.929399064948E-01, -1.437865896397E+00, -3.068671699289E+00,
       -3.238777371766E+00, -3.408883044242E+00, -2.362584773467E+00, -1.316286502692E+00,
       -4.364744114912E-01, 4.433376797099E-01, 4.433376797099E-01, 4.433376797099E-01,
       -4.364744114912E-01, -1.316286502692E+00, -2.362584773467E+00, -3.408883044242E+00,
       -3.238777371766E+00, -3.068671699289E+00, -1.437865896397E+00, 1.929399064948E-01,
       9.977774723846E+00, 1.976260954120E+01, 2.954744435855E+01, 1.976260954120E+01,
       1.319698412863E+01, 6.631358716056E+00, 6.573330348527E-02, -1.028537598610E+00,
       -2.122808500705E+00, -2.201068010332E+00, -2.279327519959E+00, -1.546609741402E+00,
       -8.138919628448E-01, -2.113971315787E-01, 3.910976996873E-01, 3.910976996873E-01,
       3.910976996873E-01, -2.113971315787E-01, -8.138919628448E-01, -1.546609741402E+00,
       -2.279327519959E+00, -2.201068010332E+00, -2.122808500705E+00, -1.028537598610E+00,
       6.573330348527E-02, 6.631358716056E+00, 1.319698412863E+01, 1.976260954120E+01,
       9.977774723846E+00, 6.631358716056E+00, 3.284942708266E+00, -6.147329952430E-02,
       -6.192093008227E-01, -1.176945302121E+00, -1.163358648898E+00, -1.149771995675E+00,
       -7.306347093361E-01, -3.114974229973E-01, 1.368014833369E-02, 3.388577196647E-01,
       3.388577196647E-01, 3.388577196647E-01, 1.368014833369E-02, -3.114974229973E-01,
       -7.306347093361E-01, -1.149771995675E+00, -1.163358648898E+00, -1.176945302121E+00,
       -6.192093008227E-01, -6.147329952430E-02, 3.284942708266E+00, 6.631358716056E+00,
       9.977774723846E+00, 1.929399064948E-01, 6.573330348527E-02, -6.147329952430E-02,
       -1.886799025339E-01, -2.098810030355E-01, -2.310821035371E-01, -1.256492874641E-01,
       -2.021647139113E-02, 8.534032272951E-02, 1.908971168502E-01, 2.387574282461E-01,
       2.866177396421E-01, 2.866177396421E-01, 2.866177396421E-01, 2.387574282461E-01,
       1.908971168502E-01, 8.534032272951E-02, -2.021647139113E-02, -1.256492874641E-01,
       -2.310821035371E-01, -2.098810030355E-01, -1.886799025339E-01, -6.147329952430E-02,
       6.573330348527E-02, 1.929399064948E-01, -1.437865896397E+00, -1.028537598610E+00,
       -6.192093008227E-01, -2.098810030355E-01, -1.416596200709E-01, -7.343823710639E-02,
       4.730227277488E-02, 1.680427826562E-01, 2.213361614071E-01, 2.746295401581E-01,
       2.762703082315E-01, 2.779110763050E-01, 2.779110763050E-01, 2.779110763050E-01,
       2.762703082315E-01, 2.746295401581E-01, 2.213361614071E-01, 1.680427826562E-01,
       4.730227277488E-02, -7.343823710639E-02, -1.416596200709E-01, -2.098810030355E-01,
       -6.192093008227E-01, -1.028537598610E+00, -1.437865896397E+00, -3.068671699289E+00,
       -2.122808500705E+00, -1.176945302121E+00, -2.310821035371E-01, -7.343823710639E-02,
       8.420562932427E-02, 2.202538330139E-01, 3.563020367034E-01, 3.573320000847E-01,
       3.583619634660E-01, 3.137831882169E-01, 2.692044129679E-01, 2.692044129679E-01,
       2.692044129679E-01, 3.137831882169E-01, 3.583619634660E-01, 3.573320000847E-01,
       3.563020367034E-01, 2.202538330139E-01, 8.420562932427E-02, -7.343823710640E-02,
       -2.310821035371E-01, -1.176945302121E+00, -2.122808500705E+00, -3.068671699289E+00,
       -3.238777371766E+00, -2.201068010332E+00, -1.163358648898E+00, -1.256492874641E-01,
       4.730227277488E-02, 2.202538330139E-01, 2.665035289248E-01, 3.127532248