import numpy as np


class TestInterpolatorLoadValues:
    """
    Base class for loading values for a 2D sinc function test.

    Storage for interpolation and extrapolation data to be test against.
    These data are saved to 12 significant figures. self.data is generated by applying the sinc function to an
    (NB_X = 10, NB_Y = 10) dimension xy data, which is used as the spline knots. The precalc_interpolation
    is setup for cubic interpolation separately using the generate_2d_splines script moved into Cherab.
    The data for nearest extrapolation is also generated through Cherab, but linear extrapolation is handled
    differently (Cherab duplicates the edge border to calculate derivatives, these interpolators handle each edge
    as a special case). Linear interpolation is calculated from interp2d(kind=linear), with nearest neighbour
    extrapolation having fill_value=None. Interpolation data is generated on a 13X13 grid. The extrapolation positions
    are 1 point outside from the middle of each edge of the 2D grid in every direction. Linear extrapolation from both
    interpolator types were generated from these interpolators on 12/07/2021, so are not mathematical tests and
    just highlight if something changes unexpectedly.
    """
    def __init__(self):
        # Define in setup_cubic or setup_linear
        self.precalc_interpolation = None

        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = None

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = None

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = None


class TestInterpolatorLoadNormalValues(TestInterpolatorLoadValues):
    """
    Loading normal sized values for a 2D sinc function test.

    For description of data storage, see TestInterpolatorLoadValues.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        self.data: np.array = np.array(
            [[7.049456954407E-02, -5.031133752816E-03, -8.474851229653E-02, -7.975908114097E-02,
              -4.876973734940E-02, -4.876973734940E-02, -7.975908114097E-02, -8.474851229653E-02,
              -5.031133752816E-03, 7.049456954407E-02],
             [-5.031133752816E-03, -9.121921863446E-02, -9.251264987298E-04, 1.052139178127E-01,
              1.283205555674E-01, 1.283205555674E-01, 1.052139178127E-01, -9.251264987298E-04,
              -9.121921863446E-02, -5.031133752816E-03],
             [-8.474851229653E-02, -9.251264987298E-04, 1.283205555674E-01, 1.734970013481E-02,
              -1.140407180451E-01, -1.140407180451E-01, 1.734970013481E-02, 1.283205555674E-01,
              -9.251264987298E-04, -8.474851229653E-02],
             [-7.975908114097E-02, 1.052139178127E-01, 1.734970013481E-02, -2.145503300375E-01,
              -9.241435356589E-02, -9.241435356589E-02, -2.145503300375E-01, 1.734970013480E-02,
              1.052139178127E-01, -7.975908114097E-02],
             [-4.876973734940E-02, 1.283205555674E-01, -1.140407180451E-01, -9.241435356589E-02,
              6.446759109720E-01, 6.446759109720E-01, -9.241435356589E-02, -1.140407180451E-01,
              1.283205555674E-01, -4.876973734940E-02],
             [-4.876973734940E-02, 1.283205555674E-01, -1.140407180451E-01, -9.241435356589E-02,
              6.446759109720E-01, 6.446759109720E-01, -9.241435356589E-02, -1.140407180451E-01,
              1.283205555674E-01, -4.876973734940E-02],
             [-7.975908114097E-02, 1.052139178127E-01, 1.734970013481E-02, -2.145503300375E-01,
              -9.241435356589E-02, -9.241435356589E-02, -2.145503300375E-01, 1.734970013480E-02,
              1.052139178127E-01, -7.975908114097E-02],
             [-8.474851229653E-02, -9.251264987298E-04, 1.283205555674E-01, 1.734970013480E-02,
              -1.140407180451E-01, -1.140407180451E-01, 1.734970013480E-02, 1.283205555674E-01,
              -9.251264987296E-04, -8.474851229653E-02],
             [-5.031133752816E-03, -9.121921863446E-02, -9.251264987298E-04, 1.052139178127E-01,
              1.283205555674E-01, 1.283205555674E-01, 1.052139178127E-01, -9.251264987296E-04,
              -9.121921863446E-02, -5.031133752816E-03],
             [7.049456954407E-02, -5.031133752816E-03, -8.474851229653E-02, -7.975908114097E-02,
              -4.876973734940E-02, -4.876973734940E-02, -7.975908114097E-02, -8.474851229653E-02,
              -5.031133752816E-03, 7.049456954407E-02]],
            dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
            [[7.049456954263E-02, 1.414501923625E-02, -4.992201892798E-02, -9.006647501680E-02,
              -7.975908114097E-02, -5.494750801437E-02, -4.489606937545E-02, -5.494750801435E-02,
              -7.975908114097E-02, -9.006647501693E-02, -4.992201892798E-02, 1.414501923767E-02,
              7.049456954405E-02],
             [1.414501923605E-02, -6.879873015310E-02, -6.604387574170E-02, -9.273861046803E-04,
              7.815453486867E-02, 1.087722976395E-01, 1.179639375775E-01, 1.087722976395E-01,
              7.815453486865E-02, -9.273861047998E-04, -6.604387574172E-02, -6.879873015206E-02,
              1.414501923718E-02],
             [-4.992201892793E-02, -6.604387574169E-02, 7.662704616627E-03, 8.927226959204E-02,
              8.733637329408E-02, 3.144134389937E-02, 8.046419412819E-03, 3.144134389941E-02,
              8.733637329409E-02, 8.927226959202E-02, 7.662704616474E-03, -6.604387574166E-02,
              -4.992201892771E-02],
             [-9.006647501696E-02, -9.273861048698E-04, 8.927226959221E-02, 9.500724468553E-02,
              -3.879550824800E-02, -1.262756473835E-01, -1.571105597160E-01, -1.262756473836E-01,
              -3.879550824809E-02, 9.500724468562E-02, 8.927226959157E-02, -9.273861044683E-04,
              -9.006647501561E-02],
             [-7.975908114100E-02, 7.815453486864E-02, 8.733637329436E-02, -3.879550824802E-02,
              -2.145503300371E-01, -1.226583807440E-01, -7.714735650690E-02, -1.226583807460E-01,
              -2.145503300375E-01, -3.879550824901E-02, 8.733637329762E-02, 7.815453486872E-02,
              -7.975908114039E-02],
             [-5.494750801440E-02, 1.087722976394E-01, 3.144134389962E-02, -1.262756473835E-01,
              -1.226583807429E-01, 3.691392206500E-01, 5.727262813613E-01, 3.691392206475E-01,
              -1.226583807439E-01, -1.262756473872E-01, 3.144134391206E-02, 1.087722976398E-01,
              -5.494750801420E-02],
             [-4.489606937547E-02, 1.179639375774E-01, 8.046419412844E-03, -1.571105597160E-01,
              -7.714735650701E-02, 5.727262813612E-01, 8.385571378575E-01, 5.727262813612E-01,
              -7.714735650690E-02, -1.571105597160E-01, 8.046419413157E-03, 1.179639375770E-01,
              -4.489606937618E-02],
             [-5.494750801430E-02, 1.087722976395E-01, 3.144134390001E-02, -1.262756473835E-01,
              -1.226583807419E-01, 3.691392206490E-01, 5.727262813611E-01, 3.691392206364E-01,
              -1.226583807442E-01, -1.262756473844E-01, 3.144134390361E-02, 1.087722976398E-01,
              -5.494750801408E-02],
             [-7.975908114226E-02, 7.815453486758E-02, 8.733637329316E-02, -3.879550824867E-02,
              -2.145503300372E-01, -1.226583807424E-01, -7.714735650698E-02, -1.226583807482E-01,
              -2.145503300413E-01, -3.879550825780E-02, 8.733637330092E-02, 7.815453489359E-02,
              -7.975908108233E-02],
             [-9.006647501836E-02, -9.273861060066E-04, 8.927226959113E-02, 9.500724468497E-02,
              -3.879550824717E-02, -1.262756473835E-01, -1.571105597161E-01, -1.262756473903E-01,
              -3.879550824831E-02, 9.500724467413E-02, 8.927226957006E-02, -9.273860602121E-04,
              -9.006647496468E-02],
             [-4.992201892777E-02, -6.604387574152E-02, 7.662704615258E-03, 8.927226959362E-02,
              8.733637329806E-02, 3.144134391022E-02, 8.046419412349E-03, 3.144134390902E-02,
              8.733637328552E-02, 8.927226957102E-02, 7.662704628725E-03, -6.604387573864E-02,
              -4.992201892713E-02],
             [1.414501923638E-02, -6.879873015296E-02, -6.604387574257E-02, -9.273861067053E-04,
              7.815453486814E-02, 1.087722976396E-01, 1.179639375771E-01, 1.087722976389E-01,
              7.815453487338E-02, -9.273860952677E-04, -6.604387573003E-02, -6.879873013669E-02,
              1.414501929255E-02],
             [7.049456954365E-02, 1.414501923699E-02, -4.992201892895E-02, -9.006647501957E-02,
              -7.975908114108E-02, -5.494750801466E-02, -4.489606937580E-02, -5.494750801505E-02,
              -7.975908112863E-02, -9.006647503033E-02, -4.992201892673E-02, 1.414501928696E-02,
              7.049456960033E-02]], dtype=np.float64
        )

        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954263E-02, 7.049456954263E-02, 7.049456954263E-02, 7.049456954263E-02,
          1.414501923625E-02, -4.992201892798E-02, -9.006647501680E-02, -7.975908114097E-02,
          -5.494750801437E-02, -4.489606937545E-02, -5.494750801435E-02, -7.975908114097E-02,
          -9.006647501693E-02, -4.992201892798E-02, 1.414501923767E-02, 7.049456954405E-02,
          7.049456954405E-02, 7.049456954405E-02, 7.049456954405E-02, 7.049456954263E-02,
          7.049456954263E-02, 7.049456954263E-02, 7.049456954263E-02, 1.414501923625E-02,
          -4.992201892798E-02, -9.006647501680E-02, -7.975908114097E-02, -5.494750801437E-02,
          -4.489606937545E-02, -5.494750801435E-02, -7.975908114097E-02, -9.006647501693E-02,
          -4.992201892798E-02, 1.414501923767E-02, 7.049456954405E-02, 7.049456954405E-02,
          7.049456954405E-02, 7.049456954405E-02, 7.049456954263E-02, 7.049456954263E-02,
          7.049456954263E-02, 7.049456954263E-02, 1.414501923625E-02, -4.992201892798E-02,
          -9.006647501680E-02, -7.975908114097E-02, -5.494750801437E-02, -4.489606937545E-02,
          -5.494750801435E-02, -7.975908114097E-02, -9.006647501693E-02, -4.992201892798E-02,
          1.414501923767E-02, 7.049456954405E-02, 7.049456954405E-02, 7.049456954405E-02,
          7.049456954405E-02, 7.049456954263E-02, 7.049456954263E-02, 7.049456954263E-02,
          7.049456954405E-02, 7.049456954405E-02, 7.049456954405E-02, 1.414501923605E-02,
          1.414501923605E-02, 1.414501923605E-02, 1.414501923718E-02, 1.414501923718E-02,
          1.414501923718E-02, -4.992201892793E-02, -4.992201892793E-02, -4.992201892793E-02,
          -4.992201892771E-02, -4.992201892771E-02, -4.992201892771E-02, -9.006647501696E-02,
          -9.006647501696E-02, -9.006647501696E-02, -9.006647501561E-02, -9.006647501561E-02,
          -9.006647501561E-02, -7.975908114100E-02, -7.975908114100E-02, -7.975908114100E-02,
          -7.975908114039E-02, -7.975908114039E-02, -7.975908114039E-02, -5.494750801440E-02,
          -5.494750801440E-02, -5.494750801440E-02, -5.494750801420E-02, -5.494750801420E-02,
          -5.494750801420E-02, -4.489606937547E-02, -4.489606937547E-02, -4.489606937547E-02,
          -4.489606937618E-02, -4.489606937618E-02, -4.489606937618E-02, -5.494750801430E-02,
          -5.494750801430E-02, -5.494750801430E-02, -5.494750801408E-02, -5.494750801408E-02,
          -5.494750801408E-02, -7.975908114226E-02, -7.975908114226E-02, -7.975908114226E-02,
          -7.975908108233E-02, -7.975908108233E-02, -7.975908108233E-02, -9.006647501836E-02,
          -9.006647501836E-02, -9.006647501836E-02, -9.006647496468E-02, -9.006647496468E-02,
          -9.006647496468E-02, -4.992201892777E-02, -4.992201892777E-02, -4.992201892777E-02,
          -4.992201892713E-02, -4.992201892713E-02, -4.992201892713E-02, 1.414501923638E-02,
          1.414501923638E-02, 1.414501923638E-02, 1.414501929255E-02, 1.414501929255E-02,
          1.414501929255E-02, 7.049456954365E-02, 7.049456954365E-02, 7.049456954365E-02,
          7.049456960033E-02, 7.049456960033E-02, 7.049456960033E-02, 7.049456954365E-02,
          7.049456954365E-02, 7.049456954365E-02, 7.049456954365E-02, 1.414501923699E-02,
          -4.992201892895E-02, -9.006647501957E-02, -7.975908114108E-02, -5.494750801466E-02,
          -4.489606937580E-02, -5.494750801505E-02, -7.975908112863E-02, -9.006647503033E-02,
          -4.992201892673E-02, 1.414501928696E-02, 7.049456960033E-02, 7.049456960033E-02,
          7.049456960033E-02, 7.049456960033E-02, 7.049456954365E-02, 7.049456954365E-02,
          7.049456954365E-02, 7.049456954365E-02, 1.414501923699E-02, -4.992201892895E-02,
          -9.006647501957E-02, -7.975908114108E-02, -5.494750801466E-02, -4.489606937580E-02,
          -5.494750801505E-02, -7.975908112863E-02, -9.006647503033E-02, -4.992201892673E-02,
          1.414501928696E-02, 7.049456960033E-02, 7.049456960033E-02, 7.049456960033E-02,
          7.049456960033E-02, 7.049456954365E-02, 7.049456954365E-02, 7.049456954365E-02,
          7.049456954365E-02, 1.414501923699E-02, -4.992201892895E-02, -9.006647501957E-02,
          -7.975908114108E-02, -5.494750801466E-02, -4.489606937580E-02, -5.494750801505E-02,
          -7.975908112863E-02, -9.006647503033E-02, -4.992201892673E-02, 1.414501928696E-02,
          7.049456960033E-02, 7.049456960033E-02, 7.049456960033E-02, 7.049456960033E-02], dtype=np.float64
        )
        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E-01, 6.276115134204E-01, 6.889187063182E-01, 7.502258992160E-01,
          8.801800967422E-01, 2.361337399095E-02, -1.138639225842E+00, -1.744516071724E+00,
          -1.684506962764E+00, -1.629840661335E+00, -1.684506962764E+00, -1.744516071724E+00,
          -1.138639225842E+00, 2.361337399095E-02, 8.801800967422E-01, 7.502258992160E-01,
          6.889187063182E-01, 6.276115134204E-01, 5.663043205226E-01, 6.276115134204E-01,
          5.929572720554E-01, 5.583030306904E-01, 5.236487893254E-01, 5.915017375739E-01,
          -8.984236486837E-04, -7.891149755669E-01, -1.189597074863E+00, -1.141320477848E+00,
          -1.101525797349E+00, -1.141320477848E+00, -1.189597074863E+00, -7.891149755669E-01,
          -8.984236486838E-04, 5.915017375739E-01, 5.236487893254E-01, 5.583030306904E-01,
          5.929572720554E-01, 6.276115134204E-01, 6.889187063182E-01, 5.583030306904E-01,
          4.276873550626E-01, 2.970716794347E-01, 3.028233784055E-01, -2.541022128832E-02,
          -4.395907252919E-01, -6.346780780020E-01, -5.981339929310E-01, -5.732109333620E-01,
          -5.981339929310E-01, -6.346780780020E-01, -4.395907252919E-01, -2.541022128832E-02,
          3.028233784055E-01, 2.970716794347E-01, 4.276873550626E-01, 5.583030306904E-01,
          6.889187063182E-01, 7.502258992160E-01, 5.236487893254E-01, 2.970716794347E-01,
          2.970716794347E-01, 5.236487893254E-01, 7.502258992160E-01, 8.801800967422E-01,
          5.915017375739E-01, 3.028233784055E-01, 3.028233784055E-01, 5.915017375739E-01,
          8.801800967422E-01, 2.361337399095E-02, -8.984236486837E-04, -2.541022128832E-02,
          -2.541022128832E-02, -8.984236486847E-04, 2.361337399095E-02, -1.138639225842E+00,
          -7.891149755669E-01, -4.395907252919E-01, -4.395907252919E-01, -7.891149755669E-01,
          -1.138639225842E+00, -1.744516071724E+00, -1.189597074863E+00, -6.346780780020E-01,
          -6.346780780020E-01, -1.189597074863E+00, -1.744516071724E+00, -1.684506962764E+00,
          -1.141320477848E+00, -5.981339929310E-01, -5.981339929310E-01, -1.141320477848E+00,
          -1.684506962764E+00, -1.629840661335E+00, -1.101525797349E+00, -5.732109333620E-01,
          -5.732109333620E-01, -1.101525797349E+00, -1.629840661335E+00, -1.684506962764E+00,
          -1.141320477848E+00, -5.981339929310E-01, -5.981339929310E-01, -1.141320477848E+00,
          -1.684506962764E+00, -1.744516071724E+00, -1.189597074863E+00, -6.346780780020E-01,
          -6.346780780020E-01, -1.189597074863E+00, -1.744516071724E+00, -1.138639225842E+00,
          -7.891149755669E-01, -4.395907252919E-01, -4.395907252919E-01, -7.891149755669E-01,
          -1.138639225842E+00, 2.361337399095E-02, -8.984236486838E-04, -2.541022128832E-02,
          -2.541022128832E-02, -8.984236486829E-04, 2.361337399095E-02, 8.801800967422E-01,
          5.915017375739E-01, 3.028233784055E-01, 3.028233784055E-01, 5.915017375739E-01,
          8.801800967422E-01, 7.502258992160E-01, 5.236487893254E-01, 2.970716794347E-01,
          2.970716794347E-01, 5.236487893254E-01, 7.502258992160E-01, 6.889187063182E-01,
          5.583030306904E-01, 4.276873550626E-01, 2.970716794347E-01, 3.028233784055E-01,
          -2.541022128832E-02, -4.395907252919E-01, -6.346780780020E-01, -5.981339929310E-01,
          -5.732109333620E-01, -5.981339929310E-01, -6.346780780020E-01, -4.395907252919E-01,
          -2.541022128832E-02, 3.028233784055E-01, 2.970716794347E-01, 4.276873550626E-01,
          5.583030306904E-01, 6.889187063182E-01, 6.276115134204E-01, 5.929572720554E-01,
          5.583030306904E-01, 5.236487893254E-01, 5.915017375739E-01, -8.984236486841E-04,
          -7.891149755669E-01, -1.189597074863E+00, -1.141320477848E+00, -1.101525797349E+00,
          -1.141320477848E+00, -1.189597074863E+00, -7.891149755669E-01, -8.984236486832E-04,
          5.915017375739E-01, 5.236487893254E-01, 5.583030306904E-01, 5.929572720554E-01,
          6.276115134204E-01, 5.663043205226E-01, 6.276115134204E-01, 6.889187063182E-01,
          7.502258992160E-01, 8.801800967422E-01, 2.361337399095E-02, -1.138639225842E+00,
          -1.744516071724E+00, -1.684506962764E+00, -1.629840661335E+00, -1.684506962764E+00,
          -1.744516071724E+00, -1.138639225842E+00, 2.361337399095E-02, 8.801800967422E-01,
          7.502258992160E-01, 6.889187063182E-01, 6.276115134204E-01, 5.663043205226E-01], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
            [[7.049456954407E-02, 1.385029207141E-02, -4.488982302467E-02, -8.350115450764E-02,
              -7.975908114097E-02, -5.651707329729E-02, -4.876973734940E-02, -5.651707329729E-02,
              -7.975908114097E-02, -8.350115450764E-02, -4.488982302467E-02, 1.385029207141E-02,
              7.049456954407E-02],
             [1.385029207141E-02, -4.879157504269E-02, -4.577658518112E-02, -1.668062692568E-03,
              5.897066807427E-02, 7.777865377223E-02, 8.404798233821E-02, 7.777865377223E-02,
              5.897066807427E-02, -1.668062692568E-03, -4.577658518112E-02, -4.879157504269E-02,
              1.385029207141E-02],
             [-4.488982302467E-02, -4.577658518112E-02, 8.812770983874E-03, 6.309373814419E-02,
              6.128180897374E-02, 2.067539131431E-02, 7.139918761166E-03, 2.067539131431E-02,
              6.128180897374E-02, 6.309373814419E-02, 8.812770983874E-03, -4.577658518112E-02,
              -4.488982302467E-02],
             [-8.350115450764E-02, -1.668062692568E-03, 6.309373814419E-02, 6.527705442988E-02,
              -4.062530740828E-02, -9.163192204604E-02, -1.086341269253E-01, -9.163192204604E-02,
              -4.062530740828E-02, 6.527705442988E-02, 6.309373814419E-02, -1.668062692568E-03,
              -8.350115450764E-02],
             [-7.975908114097E-02, 5.897066807427E-02, 6.128180897374E-02, -4.062530740828E-02,
              -2.145503300375E-01, -1.229483476838E-01, -9.241435356589E-02, -1.229483476838E-01,
              -2.145503300375E-01, -4.062530740828E-02, 6.128180897374E-02, 5.897066807427E-02,
              -7.975908114097E-02],
             [-5.651707329729E-02, 7.777865377223E-02, 2.067539131431E-02, -9.163192204604E-02,
              -1.229483476838E-01, 3.145654217072E-01, 4.604033448375E-01, 3.145654217072E-01,
              -1.229483476838E-01, -9.163192204604E-02, 2.067539131431E-02, 7.777865377223E-02,
              -5.651707329729E-02],
             [-4.876973734940E-02, 8.404798233821E-02, 7.139918761166E-03, -1.086341269253E-01,
              -9.241435356589E-02, 4.604033448375E-01, 6.446759109720E-01, 4.604033448375E-01,
              -9.241435356589E-02, -1.086341269253E-01, 7.139918761166E-03, 8.404798233821E-02,
              -4.876973734940E-02],
             [-5.651707329729E-02, 7.777865377223E-02, 2.067539131431E-02, -9.163192204604E-02,
              -1.229483476838E-01, 3.145654217072E-01, 4.604033448375E-01, 3.145654217072E-01,
              -1.229483476838E-01, -9.163192204604E-02, 2.067539131431E-02, 7.777865377223E-02,
              -5.651707329729E-02],
             [-7.975908114097E-02, 5.897066807427E-02, 6.128180897374E-02, -4.062530740828E-02,
              -2.145503300375E-01, -1.229483476838E-01, -9.241435356589E-02, -1.229483476838E-01,
              -2.145503300375E-01, -4.062530740828E-02, 6.128180897374E-02, 5.897066807427E-02,
              -7.975908114097E-02],
             [-8.350115450764E-02, -1.668062692568E-03, 6.309373814419E-02, 6.527705442988E-02,
              -4.062530740828E-02, -9.163192204604E-02, -1.086341269253E-01, -9.163192204604E-02,
              -4.062530740828E-02, 6.527705442988E-02, 6.309373814419E-02, -1.668062692568E-03,
              -8.350115450764E-02],
             [-4.488982302467E-02, -4.577658518112E-02, 8.812770983874E-03, 6.309373814419E-02,
              6.128180897374E-02, 2.067539131431E-02, 7.139918761166E-03, 2.067539131431E-02,
              6.128180897374E-02, 6.309373814419E-02, 8.812770983874E-03, -4.577658518112E-02,
              -4.488982302467E-02],
             [1.385029207141E-02, -4.879157504269E-02, -4.577658518112E-02, -1.668062692568E-03,
              5.897066807427E-02, 7.777865377223E-02, 8.404798233821E-02, 7.777865377223E-02,
              5.897066807427E-02, -1.668062692568E-03, -4.577658518112E-02, -4.879157504269E-02,
              1.385029207141E-02],
             [7.049456954407E-02, 1.385029207141E-02, -4.488982302467E-02, -8.350115450764E-02,
              -7.975908114097E-02, -5.651707329729E-02, -4.876973734940E-02, -5.651707329729E-02,
              -7.975908114097E-02, -8.350115450764E-02, -4.488982302467E-02, 1.385029207141E-02,
              7.049456954407E-02]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          1.385029207141E-02, -4.488982302467E-02, -8.350115450764E-02, -7.975908114097E-02,
          -5.651707329729E-02, -4.876973734940E-02, -5.651707329729E-02, -7.975908114097E-02,
          -8.350115450764E-02, -4.488982302467E-02, 1.385029207141E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.385029207141E-02,
          -4.488982302467E-02, -8.350115450764E-02, -7.975908114097E-02, -5.651707329729E-02,
          -4.876973734940E-02, -5.651707329729E-02, -7.975908114097E-02, -8.350115450764E-02,
          -4.488982302467E-02, 1.385029207141E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 1.385029207141E-02, -4.488982302467E-02,
          -8.350115450764E-02, -7.975908114097E-02, -5.651707329729E-02, -4.876973734940E-02,
          -5.651707329729E-02, -7.975908114097E-02, -8.350115450764E-02, -4.488982302467E-02,
          1.385029207141E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.385029207141E-02,
          1.385029207141E-02, 1.385029207141E-02, 1.385029207141E-02, 1.385029207141E-02,
          1.385029207141E-02, -4.488982302467E-02, -4.488982302467E-02, -4.488982302467E-02,
          -4.488982302467E-02, -4.488982302467E-02, -4.488982302467E-02, -8.350115450764E-02,
          -8.350115450764E-02, -8.350115450764E-02, -8.350115450764E-02, -8.350115450764E-02,
          -8.350115450764E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -5.651707329729E-02,
          -5.651707329729E-02, -5.651707329729E-02, -5.651707329729E-02, -5.651707329729E-02,
          -5.651707329729E-02, -4.876973734940E-02, -4.876973734940E-02, -4.876973734940E-02,
          -4.876973734940E-02, -4.876973734940E-02, -4.876973734940E-02, -5.651707329729E-02,
          -5.651707329729E-02, -5.651707329729E-02, -5.651707329729E-02, -5.651707329729E-02,
          -5.651707329729E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -8.350115450764E-02,
          -8.350115450764E-02, -8.350115450764E-02, -8.350115450764E-02, -8.350115450764E-02,
          -8.350115450764E-02, -4.488982302467E-02, -4.488982302467E-02, -4.488982302467E-02,
          -4.488982302467E-02, -4.488982302467E-02, -4.488982302467E-02, 1.385029207141E-02,
          1.385029207141E-02, 1.385029207141E-02, 1.385029207141E-02, 1.385029207141E-02,
          1.385029207141E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.385029207141E-02,
          -4.488982302467E-02, -8.350115450764E-02, -7.975908114097E-02, -5.651707329729E-02,
          -4.876973734940E-02, -5.651707329729E-02, -7.975908114097E-02, -8.350115450764E-02,
          -4.488982302467E-02, 1.385029207141E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 1.385029207141E-02, -4.488982302467E-02,
          -8.350115450764E-02, -7.975908114097E-02, -5.651707329729E-02, -4.876973734940E-02,
          -5.651707329729E-02, -7.975908114097E-02, -8.350115450764E-02, -4.488982302467E-02,
          1.385029207141E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 1.385029207141E-02, -4.488982302467E-02, -8.350115450764E-02,
          -7.975908114097E-02, -5.651707329729E-02, -4.876973734940E-02, -5.651707329729E-02,
          -7.975908114097E-02, -8.350115450764E-02, -4.488982302467E-02, 1.385029207141E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E-01, 6.276115134204E-01, 6.889187063182E-01, 7.502258992160E-01,
          7.655526974405E-01, -3.424867714738E-02, -1.065498256289E+00, -1.744516071724E+00,
          -1.668065798131E+00, -1.642582373601E+00, -1.668065798131E+00, -1.744516071724E+00,
          -1.065498256289E+00, -3.424867714738E-02, 7.655526974405E-01, 7.502258992160E-01,
          6.889187063182E-01, 6.276115134204E-01, 5.663043205226E-01, 6.276115134204E-01,
          5.929572720554E-01, 5.583030306904E-01, 5.236487893254E-01, 5.149852289841E-01,
          -3.779572577314E-02, -7.381658890282E-01, -1.189597074863E+00, -1.130882889853E+00,
          -1.111311494850E+00, -1.130882889853E+00, -1.189597074863E+00, -7.381658890282E-01,
          -3.779572577314E-02, 5.149852289841E-01, 5.236487893254E-01, 5.583030306904E-01,
          5.929572720554E-01, 6.276115134204E-01, 6.889187063182E-01, 5.583030306904E-01,
          4.276873550626E-01, 2.970716794347E-01, 2.644177605278E-01, -4.134277439891E-02,
          -4.108335217679E-01, -6.346780780020E-01, -5.936999815753E-01, -5.800406160998E-01,
          -5.936999815753E-01, -6.346780780020E-01, -4.108335217679E-01, -4.134277439891E-02,
          2.644177605278E-01, 2.970716794347E-01, 4.276873550626E-01, 5.583030306904E-01,
          6.889187063182E-01, 7.502258992160E-01, 5.236487893254E-01, 2.970716794347E-01,
          2.970716794347E-01, 5.236487893254E-01, 7.502258992160E-01, 7.655526974405E-01,
          5.149852289841E-01, 2.644177605278E-01, 2.644177605278E-01, 5.149852289841E-01,
          7.655526974405E-01, -3.424867714738E-02, -3.779572577314E-02, -4.134277439891E-02,
          -4.134277439891E-02, -3.779572577314E-02, -3.424867714738E-02, -1.065498256289E+00,
          -7.381658890282E-01, -4.108335217679E-01, -4.108335217679E-01, -7.381658890282E-01,
          -1.065498256289E+00, -1.744516071724E+00, -1.189597074863E+00, -6.346780780020E-01,
          -6.346780780020E-01, -1.189597074863E+00, -1.744516071724E+00, -1.668065798131E+00,
          -1.130882889853E+00, -5.936999815753E-01, -5.936999815753E-01, -1.130882889853E+00,
          -1.668065798131E+00, -1.642582373601E+00, -1.111311494850E+00, -5.800406160998E-01,
          -5.800406160998E-01, -1.111311494850E+00, -1.642582373601E+00, -1.668065798131E+00,
          -1.130882889853E+00, -5.936999815753E-01, -5.936999815753E-01, -1.130882889853E+00,
          -1.668065798131E+00, -1.744516071724E+00, -1.189597074863E+00, -6.346780780020E-01,
          -6.346780780020E-01, -1.189597074863E+00, -1.744516071724E+00, -1.065498256289E+00,
          -7.381658890282E-01, -4.108335217679E-01, -4.108335217679E-01, -7.381658890282E-01,
          -1.065498256289E+00, -3.424867714738E-02, -3.779572577314E-02, -4.134277439891E-02,
          -4.134277439891E-02, -3.779572577314E-02, -3.424867714738E-02, 7.655526974405E-01,
          5.149852289841E-01, 2.644177605278E-01, 2.644177605278E-01, 5.149852289841E-01,
          7.655526974405E-01, 7.502258992160E-01, 5.236487893254E-01, 2.970716794347E-01,
          2.970716794347E-01, 5.236487893254E-01, 7.502258992160E-01, 6.889187063182E-01,
          5.583030306904E-01, 4.276873550626E-01, 2.970716794347E-01, 2.644177605278E-01,
          -4.134277439891E-02, -4.108335217679E-01, -6.346780780020E-01, -5.936999815753E-01,
          -5.800406160998E-01, -5.936999815753E-01, -6.346780780020E-01, -4.108335217679E-01,
          -4.134277439891E-02, 2.644177605278E-01, 2.970716794347E-01, 4.276873550626E-01,
          5.583030306904E-01, 6.889187063182E-01, 6.276115134204E-01, 5.929572720554E-01,
          5.583030306904E-01, 5.236487893254E-01, 5.149852289841E-01, -3.779572577314E-02,
          -7.381658890282E-01, -1.189597074863E+00, -1.130882889853E+00, -1.111311494850E+00,
          -1.130882889853E+00, -1.189597074863E+00, -7.381658890282E-01, -3.779572577314E-02,
          5.149852289841E-01, 5.236487893254E-01, 5.583030306904E-01, 5.929572720554E-01,
          6.276115134204E-01, 5.663043205226E-01, 6.276115134204E-01, 6.889187063182E-01,
          7.502258992160E-01, 7.655526974405E-01, -3.424867714738E-02, -1.065498256289E+00,
          -1.744516071724E+00, -1.668065798131E+00, -1.642582373601E+00, -1.668065798131E+00,
          -1.744516071724E+00, -1.065498256289E+00, -3.424867714738E-02, 7.655526974405E-01,
          7.502258992160E-01, 6.889187063182E-01, 6.276115134204E-01, 5.663043205226E-01], dtype=np.float64
        )


class TestInterpolatorLoadBigValues(TestInterpolatorLoadValues):
    """
    Loading small values (10^-20 times the original) instead of the original 2D sinc function test.

    For description of data storage, see TestInterpolatorLoadValues.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        # self.data: np.array = np.sin(self.x)
        self.data: np.array = np.array(
            [[7.049456954407E+18, -5.031133752816E+17, -8.474851229653E+18, -7.975908114097E+18,
              -4.876973734940E+18, -4.876973734940E+18, -7.975908114097E+18, -8.474851229653E+18,
              -5.031133752816E+17, 7.049456954407E+18],
             [-5.031133752816E+17, -9.121921863446E+18, -9.251264987298E+16, 1.052139178127E+19,
              1.283205555674E+19, 1.283205555674E+19, 1.052139178127E+19, -9.251264987298E+16,
              -9.121921863446E+18, -5.031133752816E+17],
             [-8.474851229653E+18, -9.251264987298E+16, 1.283205555674E+19, 1.734970013481E+18,
              -1.140407180451E+19, -1.140407180451E+19, 1.734970013481E+18, 1.283205555674E+19,
              -9.251264987298E+16, -8.474851229653E+18],
             [-7.975908114097E+18, 1.052139178127E+19, 1.734970013481E+18, -2.145503300375E+19,
              -9.241435356589E+18, -9.241435356589E+18, -2.145503300375E+19, 1.734970013480E+18,
              1.052139178127E+19, -7.975908114097E+18],
             [-4.876973734940E+18, 1.283205555674E+19, -1.140407180451E+19, -9.241435356589E+18,
              6.446759109720E+19, 6.446759109720E+19, -9.241435356589E+18, -1.140407180451E+19,
              1.283205555674E+19, -4.876973734940E+18],
             [-4.876973734940E+18, 1.283205555674E+19, -1.140407180451E+19, -9.241435356589E+18,
              6.446759109720E+19, 6.446759109720E+19, -9.241435356589E+18, -1.140407180451E+19,
              1.283205555674E+19, -4.876973734940E+18],
             [-7.975908114097E+18, 1.052139178127E+19, 1.734970013481E+18, -2.145503300375E+19,
              -9.241435356589E+18, -9.241435356589E+18, -2.145503300375E+19, 1.734970013480E+18,
              1.052139178127E+19, -7.975908114097E+18],
             [-8.474851229653E+18, -9.251264987298E+16, 1.283205555674E+19, 1.734970013480E+18,
              -1.140407180451E+19, -1.140407180451E+19, 1.734970013480E+18, 1.283205555674E+19,
              -9.251264987296E+16, -8.474851229653E+18],
             [-5.031133752816E+17, -9.121921863446E+18, -9.251264987298E+16, 1.052139178127E+19,
              1.283205555674E+19, 1.283205555674E+19, 1.052139178127E+19, -9.251264987296E+16,
              -9.121921863446E+18, -5.031133752816E+17],
             [7.049456954407E+18, -5.031133752816E+17, -8.474851229653E+18, -7.975908114097E+18,
              -4.876973734940E+18, -4.876973734940E+18, -7.975908114097E+18, -8.474851229653E+18,
              -5.031133752816E+17, 7.049456954407E+18]], dtype=np.float64
        )

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = np.array(
            [], dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
            [[7.049456954460E+18, 1.414501923721E+18, -4.992201892784E+18, -9.006647501706E+18,
              -7.975908114096E+18, -5.494750801436E+18, -4.489606937545E+18, -5.494750801434E+18,
              -7.975908114100E+18, -9.006647501698E+18, -4.992201892788E+18, 1.414501923683E+18,
              7.049456954376E+18],
             [1.414501923764E+18, -6.879873015236E+18, -6.604387574162E+18, -9.273861048722E+16,
              7.815453486867E+18, 1.087722976395E+19, 1.179639375775E+19, 1.087722976395E+19,
              7.815453486864E+18, -9.273861048516E+16, -6.604387574163E+18, -6.879873015260E+18,
              1.414501923709E+18],
             [-4.992201892782E+18, -6.604387574159E+18, 7.662704616480E+17, 8.927226959209E+18,
              8.733637329413E+18, 3.144134389938E+18, 8.046419412828E+17, 3.144134389934E+18,
              8.733637329421E+18, 8.927226959216E+18, 7.662704616522E+17, -6.604387574177E+18,
              -4.992201892789E+18],
             [-9.006647501668E+18, -9.273861045643E+16, 8.927226959223E+18, 9.500724468548E+18,
              -3.879550824799E+18, -1.262756473835E+19, -1.571105597160E+19, -1.262756473838E+19,
              -3.879550824803E+18, 9.500724468565E+18, 8.927226959332E+18, -9.273861058776E+16,
              -9.006647501760E+18],
             [-7.975908114094E+18, 7.815453486869E+18, 8.733637329383E+18, -3.879550824786E+18,
              -2.145503300374E+19, -1.226583807440E+19, -7.714735650690E+18, -1.226583807425E+19,
              -2.145503300392E+19, -3.879550825037E+18, 8.733637328991E+18, 7.815453486779E+18,
              -7.975908114134E+18],
             [-5.494750801433E+18, 1.087722976395E+19, 3.144134389889E+18, -1.262756473834E+19,
              -1.226583807434E+19, 3.691392206497E+19, 5.727262813613E+19, 3.691392206533E+19,
              -1.226583807459E+19, -1.262756473889E+19, 3.144134390513E+18, 1.087722976389E+19,
              -5.494750801509E+18],
             [-4.489606937542E+18, 1.179639375775E+19, 8.046419412881E+17, -1.571105597160E+19,
              -7.714735650701E+18, 5.727262813612E+19, 8.385571378575E+19, 5.727262813618E+19,
              -7.714735650696E+18, -1.571105597161E+19, 8.046419412838E+17, 1.179639375773E+19,
              -4.489606937591E+18],
             [-5.494750801434E+18, 1.087722976395E+19, 3.144134390093E+18, -1.262756473833E+19,
              -1.226583807460E+19, 3.691392206464E+19, 5.727262813614E+19, 3.691392206388E+19,
              -1.226583807413E+19, -1.262756473809E+19, 3.144134391132E+18, 1.087722976399E+19,
              -5.494750801389E+18],
             [-7.975908114416E+18, 7.815453486610E+18, 8.733637329526E+18, -3.879550824930E+18,
              -2.145503300385E+19, -1.226583807444E+19, -7.714735650699E+18, -1.226583807452E+19,
              -2.145503300371E+19, -3.879550825350E+18, 8.733637329826E+18, 7.815453486864E+18,
              -7.975908110886E+18],
             [-9.006647502091E+18, -9.273861080508E+16, 8.927226959362E+18, 9.500724468432E+18,
              -3.879550824923E+18, -1.262756473862E+19, -1.571105597161E+19, -1.262756473885E+19,
              -3.879550824244E+18, 9.500724467988E+18, 8.927226956139E+18, -9.273860862352E+16,
              -9.006647499242E+18],
             [-4.992201892748E+18, -6.604387574117E+18, 7.662704614151E+17, 8.927226959268E+18,
              8.733637329008E+18, 3.144134390448E+18, 8.046419412263E+17, 3.144134389968E+18,
              8.733637329917E+18, 8.927226958675E+18, 7.662704629822E+17, -6.604387573416E+18,
              -4.992201892348E+18],
             [1.414501923439E+18, -6.879873015445E+18, -6.604387573988E+18, -9.273861059517E+16,
              7.815453486837E+18, 1.087722976397E+19, 1.179639375772E+19, 1.087722976397E+19,
              7.815453486894E+18, -9.273860926067E+16, -6.604387573672E+18, -6.879873017304E+18,
              1.414501925358E+18],
             [7.049456954108E+18, 1.414501923499E+18, -4.992201892571E+18, -9.006647501872E+18,
              -7.975908114100E+18, -5.494750801478E+18, -4.489606937572E+18, -5.494750801431E+18,
              -7.975908113487E+18, -9.006647502955E+18, -4.992201893463E+18, 1.414501925348E+18,
              7.049456956398E+18]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954460E+18, 7.049456954460E+18, 7.049456954460E+18, 7.049456954460E+18,
          1.414501923721E+18, -4.992201892784E+18, -9.006647501706E+18, -7.975908114096E+18,
          -5.494750801436E+18, -4.489606937545E+18, -5.494750801434E+18, -7.975908114100E+18,
          -9.006647501698E+18, -4.992201892788E+18, 1.414501923683E+18, 7.049456954376E+18,
          7.049456954376E+18, 7.049456954376E+18, 7.049456954376E+18, 7.049456954460E+18,
          7.049456954460E+18, 7.049456954460E+18, 7.049456954460E+18, 1.414501923721E+18,
          -4.992201892784E+18, -9.006647501706E+18, -7.975908114096E+18, -5.494750801436E+18,
          -4.489606937545E+18, -5.494750801434E+18, -7.975908114100E+18, -9.006647501698E+18,
          -4.992201892788E+18, 1.414501923683E+18, 7.049456954376E+18, 7.049456954376E+18,
          7.049456954376E+18, 7.049456954376E+18, 7.049456954460E+18, 7.049456954460E+18,
          7.049456954460E+18, 7.049456954460E+18, 1.414501923721E+18, -4.992201892784E+18,
          -9.006647501706E+18, -7.975908114096E+18, -5.494750801436E+18, -4.489606937545E+18,
          -5.494750801434E+18, -7.975908114100E+18, -9.006647501698E+18, -4.992201892788E+18,
          1.414501923683E+18, 7.049456954376E+18, 7.049456954376E+18, 7.049456954376E+18,
          7.049456954376E+18, 7.049456954460E+18, 7.049456954460E+18, 7.049456954460E+18,
          7.049456954376E+18, 7.049456954376E+18, 7.049456954376E+18, 1.414501923764E+18,
          1.414501923764E+18, 1.414501923764E+18, 1.414501923709E+18, 1.414501923709E+18,
          1.414501923709E+18, -4.992201892782E+18, -4.992201892782E+18, -4.992201892782E+18,
          -4.992201892789E+18, -4.992201892789E+18, -4.992201892789E+18, -9.006647501668E+18,
          -9.006647501668E+18, -9.006647501668E+18, -9.006647501760E+18, -9.006647501760E+18,
          -9.006647501760E+18, -7.975908114094E+18, -7.975908114094E+18, -7.975908114094E+18,
          -7.975908114134E+18, -7.975908114134E+18, -7.975908114134E+18, -5.494750801433E+18,
          -5.494750801433E+18, -5.494750801433E+18, -5.494750801509E+18, -5.494750801509E+18,
          -5.494750801509E+18, -4.489606937542E+18, -4.489606937542E+18, -4.489606937542E+18,
          -4.489606937591E+18, -4.489606937591E+18, -4.489606937591E+18, -5.494750801434E+18,
          -5.494750801434E+18, -5.494750801434E+18, -5.494750801389E+18, -5.494750801389E+18,
          -5.494750801389E+18, -7.975908114416E+18, -7.975908114416E+18, -7.975908114416E+18,
          -7.975908110886E+18, -7.975908110886E+18, -7.975908110886E+18, -9.006647502091E+18,
          -9.006647502091E+18, -9.006647502091E+18, -9.006647499242E+18, -9.006647499242E+18,
          -9.006647499242E+18, -4.992201892748E+18, -4.992201892748E+18, -4.992201892748E+18,
          -4.992201892348E+18, -4.992201892348E+18, -4.992201892348E+18, 1.414501923439E+18,
          1.414501923439E+18, 1.414501923439E+18, 1.414501925358E+18, 1.414501925358E+18,
          1.414501925358E+18, 7.049456954108E+18, 7.049456954108E+18, 7.049456954108E+18,
          7.049456956398E+18, 7.049456956398E+18, 7.049456956398E+18, 7.049456954108E+18,
          7.049456954108E+18, 7.049456954108E+18, 7.049456954108E+18, 1.414501923499E+18,
          -4.992201892571E+18, -9.006647501872E+18, -7.975908114100E+18, -5.494750801478E+18,
          -4.489606937572E+18, -5.494750801431E+18, -7.975908113487E+18, -9.006647502955E+18,
          -4.992201893463E+18, 1.414501925348E+18, 7.049456956398E+18, 7.049456956398E+18,
          7.049456956398E+18, 7.049456956398E+18, 7.049456954108E+18, 7.049456954108E+18,
          7.049456954108E+18, 7.049456954108E+18, 1.414501923499E+18, -4.992201892571E+18,
          -9.006647501872E+18, -7.975908114100E+18, -5.494750801478E+18, -4.489606937572E+18,
          -5.494750801431E+18, -7.975908113487E+18, -9.006647502955E+18, -4.992201893463E+18,
          1.414501925348E+18, 7.049456956398E+18, 7.049456956398E+18, 7.049456956398E+18,
          7.049456956398E+18, 7.049456954108E+18, 7.049456954108E+18, 7.049456954108E+18,
          7.049456954108E+18, 1.414501923499E+18, -4.992201892571E+18, -9.006647501872E+18,
          -7.975908114100E+18, -5.494750801478E+18, -4.489606937572E+18, -5.494750801431E+18,
          -7.975908113487E+18, -9.006647502955E+18, -4.992201893463E+18, 1.414501925348E+18,
          7.049456956398E+18, 7.049456956398E+18, 7.049456956398E+18, 7.049456956398E+18], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E+19, 6.276115134204E+19, 6.889187063182E+19, 7.502258992160E+19,
          8.801800967422E+19, 2.361337399095E+18, -1.138639225842E+20, -1.744516071724E+20,
          -1.684506962764E+20, -1.629840661335E+20, -1.684506962764E+20, -1.744516071724E+20,
          -1.138639225842E+20, 2.361337399095E+18, 8.801800967422E+19, 7.502258992160E+19,
          6.889187063182E+19, 6.276115134204E+19, 5.663043205226E+19, 6.276115134204E+19,
          5.929572720554E+19, 5.583030306904E+19, 5.236487893254E+19, 5.915017375739E+19,
          -8.984236486837E+16, -7.891149755669E+19, -1.189597074863E+20, -1.141320477848E+20,
          -1.101525797349E+20, -1.141320477848E+20, -1.189597074863E+20, -7.891149755669E+19,
          -8.984236486841E+16, 5.915017375739E+19, 5.236487893254E+19, 5.583030306904E+19,
          5.929572720554E+19, 6.276115134204E+19, 6.889187063182E+19, 5.583030306904E+19,
          4.276873550626E+19, 2.970716794347E+19, 3.028233784055E+19, -2.541022128832E+18,
          -4.395907252919E+19, -6.346780780020E+19, -5.981339929310E+19, -5.732109333620E+19,
          -5.981339929310E+19, -6.346780780020E+19, -4.395907252919E+19, -2.541022128832E+18,
          3.028233784055E+19, 2.970716794347E+19, 4.276873550626E+19, 5.583030306904E+19,
          6.889187063182E+19, 7.502258992160E+19, 5.236487893254E+19, 2.970716794347E+19,
          2.970716794347E+19, 5.236487893254E+19, 7.502258992160E+19, 8.801800967422E+19,
          5.915017375739E+19, 3.028233784055E+19, 3.028233784055E+19, 5.915017375739E+19,
          8.801800967422E+19, 2.361337399095E+18, -8.984236486837E+16, -2.541022128832E+18,
          -2.541022128832E+18, -8.984236486841E+16, 2.361337399095E+18, -1.138639225842E+20,
          -7.891149755669E+19, -4.395907252919E+19, -4.395907252919E+19, -7.891149755669E+19,
          -1.138639225842E+20, -1.744516071724E+20, -1.189597074863E+20, -6.346780780020E+19,
          -6.346780780020E+19, -1.189597074863E+20, -1.744516071724E+20, -1.684506962764E+20,
          -1.141320477848E+20, -5.981339929310E+19, -5.981339929310E+19, -1.141320477848E+20,
          -1.684506962764E+20, -1.629840661335E+20, -1.101525797349E+20, -5.732109333620E+19,
          -5.732109333620E+19, -1.101525797349E+20, -1.629840661335E+20, -1.684506962764E+20,
          -1.141320477848E+20, -5.981339929310E+19, -5.981339929310E+19, -1.141320477848E+20,
          -1.684506962764E+20, -1.744516071724E+20, -1.189597074863E+20, -6.346780780020E+19,
          -6.346780780020E+19, -1.189597074863E+20, -1.744516071724E+20, -1.138639225842E+20,
          -7.891149755669E+19, -4.395907252919E+19, -4.395907252920E+19, -7.891149755669E+19,
          -1.138639225842E+20, 2.361337399095E+18, -8.984236486840E+16, -2.541022128832E+18,
          -2.541022128832E+18, -8.984236486853E+16, 2.361337399095E+18, 8.801800967422E+19,
          5.915017375739E+19, 3.028233784055E+19, 3.028233784055E+19, 5.915017375739E+19,
          8.801800967422E+19, 7.502258992160E+19, 5.236487893254E+19, 2.970716794347E+19,
          2.970716794347E+19, 5.236487893254E+19, 7.502258992161E+19, 6.889187063182E+19,
          5.583030306904E+19, 4.276873550626E+19, 2.970716794347E+19, 3.028233784055E+19,
          -2.541022128832E+18, -4.395907252919E+19, -6.346780780020E+19, -5.981339929310E+19,
          -5.732109333620E+19, -5.981339929310E+19, -6.346780780020E+19, -4.395907252919E+19,
          -2.541022128832E+18, 3.028233784055E+19, 2.970716794347E+19, 4.276873550626E+19,
          5.583030306905E+19, 6.889187063183E+19, 6.276115134204E+19, 5.929572720554E+19,
          5.583030306904E+19, 5.236487893254E+19, 5.915017375739E+19, -8.984236486838E+16,
          -7.891149755669E+19, -1.189597074863E+20, -1.141320477848E+20, -1.101525797349E+20,
          -1.141320477848E+20, -1.189597074863E+20, -7.891149755669E+19, -8.984236486857E+16,
          5.915017375739E+19, 5.236487893254E+19, 5.583030306905E+19, 5.929572720555E+19,
          6.276115134206E+19, 5.663043205226E+19, 6.276115134204E+19, 6.889187063182E+19,
          7.502258992160E+19, 8.801800967422E+19, 2.361337399095E+18, -1.138639225842E+20,
          -1.744516071724E+20, -1.684506962764E+20, -1.629840661335E+20, -1.684506962764E+20,
          -1.744516071724E+20, -1.138639225842E+20, 2.361337399095E+18, 8.801800967422E+19,
          7.502258992161E+19, 6.889187063183E+19, 6.276115134206E+19, 5.663043205228E+19], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
            [[7.049456954407E+18, 1.385029207141E+18, -4.488982302467E+18, -8.350115450764E+18,
              -7.975908114097E+18, -5.651707329729E+18, -4.876973734940E+18, -5.651707329729E+18,
              -7.975908114097E+18, -8.350115450764E+18, -4.488982302467E+18, 1.385029207141E+18,
              7.049456954407E+18],
             [1.385029207141E+18, -4.879157504269E+18, -4.577658518112E+18, -1.668062692568E+17,
              5.897066807427E+18, 7.777865377223E+18, 8.404798233821E+18, 7.777865377223E+18,
              5.897066807427E+18, -1.668062692568E+17, -4.577658518112E+18, -4.879157504269E+18,
              1.385029207141E+18],
             [-4.488982302467E+18, -4.577658518112E+18, 8.812770983874E+17, 6.309373814419E+18,
              6.128180897374E+18, 2.067539131431E+18, 7.139918761166E+17, 2.067539131431E+18,
              6.128180897374E+18, 6.309373814419E+18, 8.812770983874E+17, -4.577658518112E+18,
              -4.488982302467E+18],
             [-8.350115450764E+18, -1.668062692568E+17, 6.309373814419E+18, 6.527705442988E+18,
              -4.062530740828E+18, -9.163192204604E+18, -1.086341269253E+19, -9.163192204604E+18,
              -4.062530740828E+18, 6.527705442988E+18, 6.309373814419E+18, -1.668062692568E+17,
              -8.350115450764E+18],
             [-7.975908114097E+18, 5.897066807427E+18, 6.128180897374E+18, -4.062530740828E+18,
              -2.145503300375E+19, -1.229483476838E+19, -9.241435356589E+18, -1.229483476838E+19,
              -2.145503300375E+19, -4.062530740828E+18, 6.128180897374E+18, 5.897066807427E+18,
              -7.975908114097E+18],
             [-5.651707329729E+18, 7.777865377223E+18, 2.067539131431E+18, -9.163192204604E+18,
              -1.229483476838E+19, 3.145654217072E+19, 4.604033448375E+19, 3.145654217072E+19,
              -1.229483476838E+19, -9.163192204604E+18, 2.067539131431E+18, 7.777865377223E+18,
              -5.651707329729E+18],
             [-4.876973734940E+18, 8.404798233821E+18, 7.139918761166E+17, -1.086341269253E+19,
              -9.241435356589E+18, 4.604033448375E+19, 6.446759109720E+19, 4.604033448375E+19,
              -9.241435356589E+18, -1.086341269253E+19, 7.139918761166E+17, 8.404798233821E+18,
              -4.876973734940E+18],
             [-5.651707329729E+18, 7.777865377223E+18, 2.067539131431E+18, -9.163192204604E+18,
              -1.229483476838E+19, 3.145654217072E+19, 4.604033448375E+19, 3.145654217072E+19,
              -1.229483476838E+19, -9.163192204604E+18, 2.067539131431E+18, 7.777865377223E+18,
              -5.651707329729E+18],
             [-7.975908114097E+18, 5.897066807427E+18, 6.128180897374E+18, -4.062530740828E+18,
              -2.145503300375E+19, -1.229483476838E+19, -9.241435356589E+18, -1.229483476838E+19,
              -2.145503300375E+19, -4.062530740828E+18, 6.128180897374E+18, 5.897066807427E+18,
              -7.975908114097E+18],
             [-8.350115450764E+18, -1.668062692568E+17, 6.309373814419E+18, 6.527705442988E+18,
              -4.062530740828E+18, -9.163192204604E+18, -1.086341269253E+19, -9.163192204604E+18,
              -4.062530740828E+18, 6.527705442988E+18, 6.309373814419E+18, -1.668062692568E+17,
              -8.350115450764E+18],
             [-4.488982302467E+18, -4.577658518112E+18, 8.812770983874E+17, 6.309373814419E+18,
              6.128180897374E+18, 2.067539131431E+18, 7.139918761166E+17, 2.067539131431E+18,
              6.128180897374E+18, 6.309373814419E+18, 8.812770983874E+17, -4.577658518112E+18,
              -4.488982302467E+18],
             [1.385029207141E+18, -4.879157504269E+18, -4.577658518112E+18, -1.668062692568E+17,
              5.897066807427E+18, 7.777865377223E+18, 8.404798233821E+18, 7.777865377223E+18,
              5.897066807427E+18, -1.668062692568E+17, -4.577658518112E+18, -4.879157504269E+18,
              1.385029207141E+18],
             [7.049456954407E+18, 1.385029207141E+18, -4.488982302467E+18, -8.350115450764E+18,
              -7.975908114097E+18, -5.651707329729E+18, -4.876973734940E+18, -5.651707329729E+18,
              -7.975908114097E+18, -8.350115450764E+18, -4.488982302467E+18, 1.385029207141E+18,
              7.049456954407E+18]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          1.385029207141E+18, -4.488982302467E+18, -8.350115450764E+18, -7.975908114097E+18,
          -5.651707329729E+18, -4.876973734940E+18, -5.651707329729E+18, -7.975908114097E+18,
          -8.350115450764E+18, -4.488982302467E+18, 1.385029207141E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.385029207141E+18,
          -4.488982302467E+18, -8.350115450764E+18, -7.975908114097E+18, -5.651707329729E+18,
          -4.876973734940E+18, -5.651707329729E+18, -7.975908114097E+18, -8.350115450764E+18,
          -4.488982302467E+18, 1.385029207141E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 1.385029207141E+18, -4.488982302467E+18,
          -8.350115450764E+18, -7.975908114097E+18, -5.651707329729E+18, -4.876973734940E+18,
          -5.651707329729E+18, -7.975908114097E+18, -8.350115450764E+18, -4.488982302467E+18,
          1.385029207141E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.385029207141E+18,
          1.385029207141E+18, 1.385029207141E+18, 1.385029207141E+18, 1.385029207141E+18,
          1.385029207141E+18, -4.488982302467E+18, -4.488982302467E+18, -4.488982302467E+18,
          -4.488982302467E+18, -4.488982302467E+18, -4.488982302467E+18, -8.350115450764E+18,
          -8.350115450764E+18, -8.350115450764E+18, -8.350115450764E+18, -8.350115450764E+18,
          -8.350115450764E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -5.651707329729E+18,
          -5.651707329729E+18, -5.651707329729E+18, -5.651707329729E+18, -5.651707329729E+18,
          -5.651707329729E+18, -4.876973734940E+18, -4.876973734940E+18, -4.876973734940E+18,
          -4.876973734940E+18, -4.876973734940E+18, -4.876973734940E+18, -5.651707329729E+18,
          -5.651707329729E+18, -5.651707329729E+18, -5.651707329729E+18, -5.651707329729E+18,
          -5.651707329729E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -8.350115450764E+18,
          -8.350115450764E+18, -8.350115450764E+18, -8.350115450764E+18, -8.350115450764E+18,
          -8.350115450764E+18, -4.488982302467E+18, -4.488982302467E+18, -4.488982302467E+18,
          -4.488982302467E+18, -4.488982302467E+18, -4.488982302467E+18, 1.385029207141E+18,
          1.385029207141E+18, 1.385029207141E+18, 1.385029207141E+18, 1.385029207141E+18,
          1.385029207141E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.385029207141E+18,
          -4.488982302467E+18, -8.350115450764E+18, -7.975908114097E+18, -5.651707329729E+18,
          -4.876973734940E+18, -5.651707329729E+18, -7.975908114097E+18, -8.350115450764E+18,
          -4.488982302467E+18, 1.385029207141E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 1.385029207141E+18, -4.488982302467E+18,
          -8.350115450764E+18, -7.975908114097E+18, -5.651707329729E+18, -4.876973734940E+18,
          -5.651707329729E+18, -7.975908114097E+18, -8.350115450764E+18, -4.488982302467E+18,
          1.385029207141E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 1.385029207141E+18, -4.488982302467E+18, -8.350115450764E+18,
          -7.975908114097E+18, -5.651707329729E+18, -4.876973734940E+18, -5.651707329729E+18,
          -7.975908114097E+18, -8.350115450764E+18, -4.488982302467E+18, 1.385029207141E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E+19, 6.276115134204E+19, 6.889187063182E+19, 7.502258992160E+19,
          7.655526974405E+19, -3.424867714738E+18, -1.065498256289E+20, -1.744516071724E+20,
          -1.668065798131E+20, -1.642582373601E+20, -1.668065798131E+20, -1.744516071724E+20,
          -1.065498256289E+20, -3.424867714738E+18, 7.655526974405E+19, 7.502258992160E+19,
          6.889187063182E+19, 6.276115134204E+19, 5.663043205226E+19, 6.276115134204E+19,
          5.929572720554E+19, 5.583030306904E+19, 5.236487893254E+19, 5.149852289841E+19,
          -3.779572577314E+18, -7.381658890282E+19, -1.189597074863E+20, -1.130882889853E+20,
          -1.111311494850E+20, -1.130882889853E+20, -1.189597074863E+20, -7.381658890282E+19,
          -3.779572577314E+18, 5.149852289841E+19, 5.236487893254E+19, 5.583030306904E+19,
          5.929572720554E+19, 6.276115134204E+19, 6.889187063182E+19, 5.583030306904E+19,
          4.276873550626E+19, 2.970716794347E+19, 2.644177605278E+19, -4.134277439891E+18,
          -4.108335217679E+19, -6.346780780020E+19, -5.936999815753E+19, -5.800406160998E+19,
          -5.936999815753E+19, -6.346780780020E+19, -4.108335217679E+19, -4.134277439891E+18,
          2.644177605278E+19, 2.970716794347E+19, 4.276873550626E+19, 5.583030306904E+19,
          6.889187063182E+19, 7.502258992160E+19, 5.236487893254E+19, 2.970716794347E+19,
          2.970716794347E+19, 5.236487893254E+19, 7.502258992160E+19, 7.655526974405E+19,
          5.149852289841E+19, 2.644177605278E+19, 2.644177605278E+19, 5.149852289841E+19,
          7.655526974405E+19, -3.424867714738E+18, -3.779572577314E+18, -4.134277439891E+18,
          -4.134277439891E+18, -3.779572577314E+18, -3.424867714738E+18, -1.065498256289E+20,
          -7.381658890282E+19, -4.108335217679E+19, -4.108335217679E+19, -7.381658890282E+19,
          -1.065498256289E+20, -1.744516071724E+20, -1.189597074863E+20, -6.346780780020E+19,
          -6.346780780020E+19, -1.189597074863E+20, -1.744516071724E+20, -1.668065798131E+20,
          -1.130882889853E+20, -5.936999815753E+19, -5.936999815753E+19, -1.130882889853E+20,
          -1.668065798131E+20, -1.642582373601E+20, -1.111311494850E+20, -5.800406160998E+19,
          -5.800406160998E+19, -1.111311494850E+20, -1.642582373601E+20, -1.668065798131E+20,
          -1.130882889853E+20, -5.936999815753E+19, -5.936999815753E+19, -1.130882889853E+20,
          -1.668065798131E+20, -1.744516071724E+20, -1.189597074863E+20, -6.346780780020E+19,
          -6.346780780020E+19, -1.189597074863E+20, -1.744516071724E+20, -1.065498256289E+20,
          -7.381658890282E+19, -4.108335217679E+19, -4.108335217679E+19, -7.381658890282E+19,
          -1.065498256289E+20, -3.424867714738E+18, -3.779572577314E+18, -4.134277439891E+18,
          -4.134277439891E+18, -3.779572577314E+18, -3.424867714738E+18, 7.655526974405E+19,
          5.149852289841E+19, 2.644177605278E+19, 2.644177605278E+19, 5.149852289841E+19,
          7.655526974405E+19, 7.502258992160E+19, 5.236487893254E+19, 2.970716794347E+19,
          2.970716794347E+19, 5.236487893254E+19, 7.502258992160E+19, 6.889187063182E+19,
          5.583030306904E+19, 4.276873550626E+19, 2.970716794347E+19, 2.644177605278E+19,
          -4.134277439891E+18, -4.108335217679E+19, -6.346780780020E+19, -5.936999815753E+19,
          -5.800406160998E+19, -5.936999815753E+19, -6.346780780020E+19, -4.108335217679E+19,
          -4.134277439891E+18, 2.644177605278E+19, 2.970716794347E+19, 4.276873550626E+19,
          5.583030306904E+19, 6.889187063182E+19, 6.276115134204E+19, 5.929572720554E+19,
          5.583030306904E+19, 5.236487893254E+19, 5.149852289841E+19, -3.779572577314E+18,
          -7.381658890282E+19, -1.189597074863E+20, -1.130882889853E+20, -1.111311494850E+20,
          -1.130882889853E+20, -1.189597074863E+20, -7.381658890282E+19, -3.779572577314E+18,
          5.149852289841E+19, 5.236487893254E+19, 5.583030306904E+19, 5.929572720554E+19,
          6.276115134204E+19, 5.663043205226E+19, 6.276115134204E+19, 6.889187063182E+19,
          7.502258992160E+19, 7.655526974405E+19, -3.424867714738E+18, -1.065498256289E+20,
          -1.744516071724E+20, -1.668065798131E+20, -1.642582373601E+20, -1.668065798131E+20,
          -1.744516071724E+20, -1.065498256289E+20, -3.424867714738E+18, 7.655526974405E+19,
          7.502258992160E+19, 6.889187063182E+19, 6.276115134204E+19, 5.663043205226E+19], dtype=np.float64
        )


class TestInterpolatorLoadSmallValues(TestInterpolatorLoadValues):
    """
    Loading small values (10^-20 times the original) instead of the original np.sin(x)

    These data are saved to 12 significant figures.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        # self.data: np.array = np.sin(self.x)
        self.data: np.array = np.array(
            [[7.049456954407E-22, -5.031133752816E-23, -8.474851229653E-22, -7.975908114097E-22,
              -4.876973734940E-22, -4.876973734940E-22, -7.975908114097E-22, -8.474851229653E-22,
              -5.031133752816E-23, 7.049456954407E-22],
             [-5.031133752816E-23, -9.121921863446E-22, -9.251264987298E-24, 1.052139178127E-21,
              1.283205555674E-21, 1.283205555674E-21, 1.052139178127E-21, -9.251264987298E-24,
              -9.121921863446E-22, -5.031133752816E-23],
             [-8.474851229653E-22, -9.251264987298E-24, 1.283205555674E-21, 1.734970013481E-22,
              -1.140407180451E-21, -1.140407180451E-21, 1.734970013481E-22, 1.283205555674E-21,
              -9.251264987298E-24, -8.474851229653E-22],
             [-7.975908114097E-22, 1.052139178127E-21, 1.734970013481E-22, -2.145503300375E-21,
              -9.241435356589E-22, -9.241435356589E-22, -2.145503300375E-21, 1.734970013480E-22,
              1.052139178127E-21, -7.975908114097E-22],
             [-4.876973734940E-22, 1.283205555674E-21, -1.140407180451E-21, -9.241435356589E-22,
              6.446759109720E-21, 6.446759109720E-21, -9.241435356589E-22, -1.140407180451E-21,
              1.283205555674E-21, -4.876973734940E-22],
             [-4.876973734940E-22, 1.283205555674E-21, -1.140407180451E-21, -9.241435356589E-22,
              6.446759109720E-21, 6.446759109720E-21, -9.241435356589E-22, -1.140407180451E-21,
              1.283205555674E-21, -4.876973734940E-22],
             [-7.975908114097E-22, 1.052139178127E-21, 1.734970013481E-22, -2.145503300375E-21,
              -9.241435356589E-22, -9.241435356589E-22, -2.145503300375E-21, 1.734970013480E-22,
              1.052139178127E-21, -7.975908114097E-22],
             [-8.474851229653E-22, -9.251264987298E-24, 1.283205555674E-21, 1.734970013480E-22,
              -1.140407180451E-21, -1.140407180451E-21, 1.734970013480E-22, 1.283205555674E-21,
              -9.251264987296E-24, -8.474851229653E-22],
             [-5.031133752816E-23, -9.121921863446E-22, -9.251264987298E-24, 1.052139178127E-21,
              1.283205555674E-21, 1.283205555674E-21, 1.052139178127E-21, -9.251264987296E-24,
              -9.121921863446E-22, -5.031133752816E-23],
             [7.049456954407E-22, -5.031133752816E-23, -8.474851229653E-22, -7.975908114097E-22,
              -4.876973734940E-22, -4.876973734940E-22, -7.975908114097E-22, -8.474851229653E-22,
              -5.031133752816E-23, 7.049456954407E-22]], dtype=np.float64
        )

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = np.array(
            [], dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
            [[7.049456954264E-22, 1.414501923508E-22, -4.992201892787E-22, -9.006647501692E-22,
              -7.975908114097E-22, -5.494750801435E-22, -4.489606937545E-22, -5.494750801436E-22,
              -7.975908114100E-22, -9.006647501711E-22, -4.992201892810E-22, 1.414501923847E-22,
              7.049456954556E-22],
             [1.414501923618E-22, -6.879873015399E-22, -6.604387574163E-22, -9.273861047815E-24,
              7.815453486868E-22, 1.087722976395E-21, 1.179639375774E-21, 1.087722976395E-21,
              7.815453486864E-22, -9.273861049481E-24, -6.604387574180E-22, -6.879873015137E-22,
              1.414501923837E-22],
             [-4.992201892795E-22, -6.604387574171E-22, 7.662704616648E-23, 8.927226959196E-22,
              8.733637329416E-22, 3.144134389939E-22, 8.046419412826E-23, 3.144134389938E-22,
              8.733637329419E-22, 8.927226959212E-22, 7.662704616608E-23, -6.604387574155E-22,
              -4.992201892762E-22],
             [-9.006647501697E-22, -9.273861047635E-24, 8.927226959229E-22, 9.500724468556E-22,
              -3.879550824799E-22, -1.262756473835E-21, -1.571105597160E-21, -1.262756473839E-21,
              -3.879550824817E-22, 9.500724468550E-22, 8.927226959408E-22, -9.273861038222E-24,
              -9.006647501493E-22],
             [-7.975908114093E-22, 7.815453486870E-22, 8.733637329381E-22, -3.879550824787E-22,
              -2.145503300374E-21, -1.226583807440E-21, -7.714735650690E-22, -1.226583807454E-21,
              -2.145503300370E-21, -3.879550824826E-22, 8.733637328991E-22, 7.815453486854E-22,
              -7.975908114063E-22],
             [-5.494750801431E-22, 1.087722976395E-21, 3.144134389893E-22, -1.262756473834E-21,
              -1.226583807434E-21, 3.691392206497E-21, 5.727262813613E-21, 3.691392206474E-21,
              -1.226583807439E-21, -1.262756473868E-21, 3.144134390512E-22, 1.087722976396E-21,
              -5.494750801446E-22],
             [-4.489606937538E-22, 1.179639375775E-21, 8.046419412696E-23, -1.571105597160E-21,
              -7.714735650701E-22, 5.727262813612E-21, 8.385571378575E-21, 5.727262813618E-21,
              -7.714735650696E-22, -1.571105597161E-21, 8.046419412846E-23, 1.179639375773E-21,
              -4.489606937591E-22],
             [-5.494750801434E-22, 1.087722976395E-21, 3.144134390008E-22, -1.262756473844E-21,
              -1.226583807433E-21, 3.691392206488E-21, 5.727262813614E-21, 3.691392206419E-21,
              -1.226583807457E-21, -1.262756473861E-21, 3.144134391132E-22, 1.087722976390E-21,
              -5.494750801485E-22],
             [-7.975908114306E-22, 7.815453486712E-22, 8.733637329342E-22, -3.879550824792E-22,
              -2.145503300377E-21, -1.226583807429E-21, -7.714735650699E-22, -1.226583807452E-21,
              -2.145503300310E-21, -3.879550824614E-22, 8.733637328587E-22, 7.815453486866E-22,
              -7.975908110894E-22],
             [-9.006647501952E-22, -9.273861067304E-24, 8.927226959138E-22, 9.500724468601E-22,
              -3.879550824810E-22, -1.262756473846E-21, -1.571105597161E-21, -1.262756473885E-21,
              -3.879550824035E-22, 9.500724468340E-22, 8.927226955189E-22, -9.273860862048E-24,
              -9.006647499257E-22],
             [-4.992201892821E-22, -6.604387574194E-22, 7.662704617160E-23, 8.927226959387E-22,
              8.733637329469E-22, 3.144134390697E-22, 8.046419413383E-23, 3.144134392460E-22,
              8.733637329685E-22, 8.927226958359E-22, 7.662704629676E-23, -6.604387573915E-22,
              -4.992201893027E-22],
             [1.414501923332E-22, -6.879873015551E-22, -6.604387574050E-22, -9.273861079956E-24,
              7.815453486878E-22, 1.087722976402E-21, 1.179639375773E-21, 1.087722976395E-21,
              7.815453486844E-22, -9.273860941952E-24, -6.604387571973E-22, -6.879873019460E-22,
              1.414501922245E-22],
             [7.049456953963E-22, 1.414501923357E-22, -4.992201892648E-22, -9.006647502111E-22,
              -7.975908114054E-22, -5.494750801416E-22, -4.489606937555E-22, -5.494750801446E-22,
              -7.975908113966E-22, -9.006647503577E-22, -4.992201891716E-22, 1.414501923407E-22,
              7.049456953362E-22]], dtype=np.float64
        )
        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954264E-22, 7.049456954264E-22, 7.049456954264E-22, 7.049456954264E-22,
          1.414501923508E-22, -4.992201892787E-22, -9.006647501692E-22, -7.975908114097E-22,
          -5.494750801435E-22, -4.489606937545E-22, -5.494750801436E-22, -7.975908114100E-22,
          -9.006647501711E-22, -4.992201892810E-22, 1.414501923847E-22, 7.049456954556E-22,
          7.049456954556E-22, 7.049456954556E-22, 7.049456954556E-22, 7.049456954264E-22,
          7.049456954264E-22, 7.049456954264E-22, 7.049456954264E-22, 1.414501923508E-22,
          -4.992201892787E-22, -9.006647501692E-22, -7.975908114097E-22, -5.494750801435E-22,
          -4.489606937545E-22, -5.494750801436E-22, -7.975908114100E-22, -9.006647501711E-22,
          -4.992201892810E-22, 1.414501923847E-22, 7.049456954556E-22, 7.049456954556E-22,
          7.049456954556E-22, 7.049456954556E-22, 7.049456954264E-22, 7.049456954264E-22,
          7.049456954264E-22, 7.049456954264E-22, 1.414501923508E-22, -4.992201892787E-22,
          -9.006647501692E-22, -7.975908114097E-22, -5.494750801435E-22, -4.489606937545E-22,
          -5.494750801436E-22, -7.975908114100E-22, -9.006647501711E-22, -4.992201892810E-22,
          1.414501923847E-22, 7.049456954556E-22, 7.049456954556E-22, 7.049456954556E-22,
          7.049456954556E-22, 7.049456954264E-22, 7.049456954264E-22, 7.049456954264E-22,
          7.049456954556E-22, 7.049456954556E-22, 7.049456954556E-22, 1.414501923618E-22,
          1.414501923618E-22, 1.414501923618E-22, 1.414501923837E-22, 1.414501923837E-22,
          1.414501923837E-22, -4.992201892795E-22, -4.992201892795E-22, -4.992201892795E-22,
          -4.992201892762E-22, -4.992201892762E-22, -4.992201892762E-22, -9.006647501697E-22,
          -9.006647501697E-22, -9.006647501697E-22, -9.006647501493E-22, -9.006647501493E-22,
          -9.006647501493E-22, -7.975908114093E-22, -7.975908114093E-22, -7.975908114093E-22,
          -7.975908114063E-22, -7.975908114063E-22, -7.975908114063E-22, -5.494750801431E-22,
          -5.494750801431E-22, -5.494750801431E-22, -5.494750801446E-22, -5.494750801446E-22,
          -5.494750801446E-22, -4.489606937538E-22, -4.489606937538E-22, -4.489606937538E-22,
          -4.489606937591E-22, -4.489606937591E-22, -4.489606937591E-22, -5.494750801434E-22,
          -5.494750801434E-22, -5.494750801434E-22, -5.494750801485E-22, -5.494750801485E-22,
          -5.494750801485E-22, -7.975908114306E-22, -7.975908114306E-22, -7.975908114306E-22,
          -7.975908110894E-22, -7.975908110894E-22, -7.975908110894E-22, -9.006647501952E-22,
          -9.006647501952E-22, -9.006647501952E-22, -9.006647499257E-22, -9.006647499257E-22,
          -9.006647499257E-22, -4.992201892821E-22, -4.992201892821E-22, -4.992201892821E-22,
          -4.992201893027E-22, -4.992201893027E-22, -4.992201893027E-22, 1.414501923332E-22,
          1.414501923332E-22, 1.414501923332E-22, 1.414501922245E-22, 1.414501922245E-22,
          1.414501922245E-22, 7.049456953963E-22, 7.049456953963E-22, 7.049456953963E-22,
          7.049456953362E-22, 7.049456953362E-22, 7.049456953362E-22, 7.049456953963E-22,
          7.049456953963E-22, 7.049456953963E-22, 7.049456953963E-22, 1.414501923357E-22,
          -4.992201892648E-22, -9.006647502111E-22, -7.975908114054E-22, -5.494750801416E-22,
          -4.489606937555E-22, -5.494750801446E-22, -7.975908113966E-22, -9.006647503577E-22,
          -4.992201891716E-22, 1.414501923407E-22, 7.049456953362E-22, 7.049456953362E-22,
          7.049456953362E-22, 7.049456953362E-22, 7.049456953963E-22, 7.049456953963E-22,
          7.049456953963E-22, 7.049456953963E-22, 1.414501923357E-22, -4.992201892648E-22,
          -9.006647502111E-22, -7.975908114054E-22, -5.494750801416E-22, -4.489606937555E-22,
          -5.494750801446E-22, -7.975908113966E-22, -9.006647503577E-22, -4.992201891716E-22,
          1.414501923407E-22, 7.049456953362E-22, 7.049456953362E-22, 7.049456953362E-22,
          7.049456953362E-22, 7.049456953963E-22, 7.049456953963E-22, 7.049456953963E-22,
          7.049456953963E-22, 1.414501923357E-22, -4.992201892648E-22, -9.006647502111E-22,
          -7.975908114054E-22, -5.494750801416E-22, -4.489606937555E-22, -5.494750801446E-22,
          -7.975908113966E-22, -9.006647503577E-22, -4.992201891716E-22, 1.414501923407E-22,
          7.049456953362E-22, 7.049456953362E-22, 7.049456953362E-22, 7.049456953362E-22], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E-21, 6.276115134204E-21, 6.889187063182E-21, 7.502258992160E-21,
          8.801800967422E-21, 2.361337399095E-22, -1.138639225842E-20, -1.744516071724E-20,
          -1.684506962764E-20, -1.629840661335E-20, -1.684506962764E-20, -1.744516071724E-20,
          -1.138639225842E-20, 2.361337399095E-22, 8.801800967422E-21, 7.502258992160E-21,
          6.889187063182E-21, 6.276115134204E-21, 5.663043205226E-21, 6.276115134204E-21,
          5.929572720554E-21, 5.583030306904E-21, 5.236487893254E-21, 5.915017375739E-21,
          -8.984236486835E-24, -7.891149755669E-21, -1.189597074863E-20, -1.141320477848E-20,
          -1.101525797349E-20, -1.141320477848E-20, -1.189597074863E-20, -7.891149755669E-21,
          -8.984236486840E-24, 5.915017375739E-21, 5.236487893254E-21, 5.583030306904E-21,
          5.929572720554E-21, 6.276115134204E-21, 6.889187063182E-21, 5.583030306904E-21,
          4.276873550626E-21, 2.970716794347E-21, 3.028233784055E-21, -2.541022128832E-22,
          -4.395907252919E-21, -6.346780780020E-21, -5.981339929310E-21, -5.732109333620E-21,
          -5.981339929310E-21, -6.346780780020E-21, -4.395907252919E-21, -2.541022128832E-22,
          3.028233784055E-21, 2.970716794347E-21, 4.276873550626E-21, 5.583030306904E-21,
          6.889187063182E-21, 7.502258992160E-21, 5.236487893254E-21, 2.970716794347E-21,
          2.970716794347E-21, 5.236487893254E-21, 7.502258992160E-21, 8.801800967422E-21,
          5.915017375739E-21, 3.028233784055E-21, 3.028233784055E-21, 5.915017375739E-21,
          8.801800967422E-21, 2.361337399095E-22, -8.984236486836E-24, -2.541022128832E-22,
          -2.541022128832E-22, -8.984236486841E-24, 2.361337399095E-22, -1.138639225842E-20,
          -7.891149755669E-21, -4.395907252919E-21, -4.395907252919E-21, -7.891149755669E-21,
          -1.138639225842E-20, -1.744516071724E-20, -1.189597074863E-20, -6.346780780020E-21,
          -6.346780780020E-21, -1.189597074863E-20, -1.744516071724E-20, -1.684506962764E-20,
          -1.141320477848E-20, -5.981339929310E-21, -5.981339929310E-21, -1.141320477848E-20,
          -1.684506962764E-20, -1.629840661335E-20, -1.101525797349E-20, -5.732109333620E-21,
          -5.732109333620E-21, -1.101525797349E-20, -1.629840661335E-20, -1.684506962764E-20,
          -1.141320477848E-20, -5.981339929310E-21, -5.981339929310E-21, -1.141320477848E-20,
          -1.684506962764E-20, -1.744516071724E-20, -1.189597074863E-20, -6.346780780020E-21,
          -6.346780780020E-21, -1.189597074863E-20, -1.744516071724E-20, -1.138639225842E-20,
          -7.891149755669E-21, -4.395907252919E-21, -4.395907252919E-21, -7.891149755669E-21,
          -1.138639225842E-20, 2.361337399095E-22, -8.984236486840E-24, -2.541022128832E-22,
          -2.541022128832E-22, -8.984236486847E-24, 2.361337399095E-22, 8.801800967422E-21,
          5.915017375739E-21, 3.028233784055E-21, 3.028233784055E-21, 5.915017375739E-21,
          8.801800967422E-21, 7.502258992160E-21, 5.236487893254E-21, 2.970716794347E-21,
          2.970716794347E-21, 5.236487893254E-21, 7.502258992160E-21, 6.889187063182E-21,
          5.583030306904E-21, 4.276873550626E-21, 2.970716794347E-21, 3.028233784055E-21,
          -2.541022128832E-22, -4.395907252919E-21, -6.346780780020E-21, -5.981339929310E-21,
          -5.732109333620E-21, -5.981339929310E-21, -6.346780780020E-21, -4.395907252919E-21,
          -2.541022128832E-22, 3.028233784055E-21, 2.970716794347E-21, 4.276873550626E-21,
          5.583030306904E-21, 6.889187063182E-21, 6.276115134204E-21, 5.929572720554E-21,
          5.583030306904E-21, 5.236487893254E-21, 5.915017375739E-21, -8.984236486850E-24,
          -7.891149755669E-21, -1.189597074863E-20, -1.141320477848E-20, -1.101525797349E-20,
          -1.141320477848E-20, -1.189597074863E-20, -7.891149755669E-21, -8.984236486845E-24,
          5.915017375739E-21, 5.236487893254E-21, 5.583030306904E-21, 5.929572720554E-21,
          6.276115134204E-21, 5.663043205226E-21, 6.276115134204E-21, 6.889187063182E-21,
          7.502258992160E-21, 8.801800967422E-21, 2.361337399095E-22, -1.138639225842E-20,
          -1.744516071724E-20, -1.684506962764E-20, -1.629840661335E-20, -1.684506962764E-20,
          -1.744516071724E-20, -1.138639225842E-20, 2.361337399095E-22, 8.801800967422E-21,
          7.502258992160E-21, 6.889187063182E-21, 6.276115134204E-21, 5.663043205226E-21], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
            [[7.049456954407E-22, 1.385029207141E-22, -4.488982302467E-22, -8.350115450764E-22,
              -7.975908114097E-22, -5.651707329729E-22, -4.876973734940E-22, -5.651707329729E-22,
              -7.975908114097E-22, -8.350115450764E-22, -4.488982302467E-22, 1.385029207141E-22,
              7.049456954407E-22],
             [1.385029207141E-22, -4.879157504269E-22, -4.577658518112E-22, -1.668062692568E-23,
              5.897066807427E-22, 7.777865377223E-22, 8.404798233821E-22, 7.777865377223E-22,
              5.897066807427E-22, -1.668062692568E-23, -4.577658518112E-22, -4.879157504269E-22,
              1.385029207141E-22],
             [-4.488982302467E-22, -4.577658518112E-22, 8.812770983874E-23, 6.309373814419E-22,
              6.128180897374E-22, 2.067539131431E-22, 7.139918761166E-23, 2.067539131431E-22,
              6.128180897374E-22, 6.309373814419E-22, 8.812770983874E-23, -4.577658518112E-22,
              -4.488982302467E-22],
             [-8.350115450764E-22, -1.668062692568E-23, 6.309373814419E-22, 6.527705442988E-22,
              -4.062530740828E-22, -9.163192204604E-22, -1.086341269253E-21, -9.163192204604E-22,
              -4.062530740828E-22, 6.527705442988E-22, 6.309373814419E-22, -1.668062692568E-23,
              -8.350115450764E-22],
             [-7.975908114097E-22, 5.897066807427E-22, 6.128180897374E-22, -4.062530740828E-22,
              -2.145503300375E-21, -1.229483476838E-21, -9.241435356589E-22, -1.229483476838E-21,
              -2.145503300375E-21, -4.062530740828E-22, 6.128180897374E-22, 5.897066807427E-22,
              -7.975908114097E-22],
             [-5.651707329729E-22, 7.777865377223E-22, 2.067539131431E-22, -9.163192204604E-22,
              -1.229483476838E-21, 3.145654217072E-21, 4.604033448375E-21, 3.145654217072E-21,
              -1.229483476838E-21, -9.163192204604E-22, 2.067539131431E-22, 7.777865377223E-22,
              -5.651707329729E-22],
             [-4.876973734940E-22, 8.404798233821E-22, 7.139918761166E-23, -1.086341269253E-21,
              -9.241435356589E-22, 4.604033448375E-21, 6.446759109720E-21, 4.604033448375E-21,
              -9.241435356589E-22, -1.086341269253E-21, 7.139918761166E-23, 8.404798233821E-22,
              -4.876973734940E-22],
             [-5.651707329729E-22, 7.777865377223E-22, 2.067539131431E-22, -9.163192204604E-22,
              -1.229483476838E-21, 3.145654217072E-21, 4.604033448375E-21, 3.145654217072E-21,
              -1.229483476838E-21, -9.163192204604E-22, 2.067539131431E-22, 7.777865377223E-22,
              -5.651707329729E-22],
             [-7.975908114097E-22, 5.897066807427E-22, 6.128180897374E-22, -4.062530740828E-22,
              -2.145503300375E-21, -1.229483476838E-21, -9.241435356589E-22, -1.229483476838E-21,
              -2.145503300375E-21, -4.062530740828E-22, 6.128180897374E-22, 5.897066807427E-22,
              -7.975908114097E-22],
             [-8.350115450764E-22, -1.668062692568E-23, 6.309373814419E-22, 6.527705442988E-22,
              -4.062530740828E-22, -9.163192204604E-22, -1.086341269253E-21, -9.163192204604E-22,
              -4.062530740828E-22, 6.527705442988E-22, 6.309373814419E-22, -1.668062692568E-23,
              -8.350115450764E-22],
             [-4.488982302467E-22, -4.577658518112E-22, 8.812770983874E-23, 6.309373814419E-22,
              6.128180897374E-22, 2.067539131431E-22, 7.139918761166E-23, 2.067539131431E-22,
              6.128180897374E-22, 6.309373814419E-22, 8.812770983874E-23, -4.577658518112E-22,
              -4.488982302467E-22],
             [1.385029207141E-22, -4.879157504269E-22, -4.577658518112E-22, -1.668062692568E-23,
              5.897066807427E-22, 7.777865377223E-22, 8.404798233821E-22, 7.777865377223E-22,
              5.897066807427E-22, -1.668062692568E-23, -4.577658518112E-22, -4.879157504269E-22,
              1.385029207141E-22],
             [7.049456954407E-22, 1.385029207141E-22, -4.488982302467E-22, -8.350115450764E-22,
              -7.975908114097E-22, -5.651707329729E-22, -4.876973734940E-22, -5.651707329729E-22,
              -7.975908114097E-22, -8.350115450764E-22, -4.488982302467E-22, 1.385029207141E-22,
              7.049456954407E-22]], dtype=np.float64
        )
        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          1.385029207141E-22, -4.488982302467E-22, -8.350115450764E-22, -7.975908114097E-22,
          -5.651707329729E-22, -4.876973734940E-22, -5.651707329729E-22, -7.975908114097E-22,
          -8.350115450764E-22, -4.488982302467E-22, 1.385029207141E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.385029207141E-22,
          -4.488982302467E-22, -8.350115450764E-22, -7.975908114097E-22, -5.651707329729E-22,
          -4.876973734940E-22, -5.651707329729E-22, -7.975908114097E-22, -8.350115450764E-22,
          -4.488982302467E-22, 1.385029207141E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 1.385029207141E-22, -4.488982302467E-22,
          -8.350115450764E-22, -7.975908114097E-22, -5.651707329729E-22, -4.876973734940E-22,
          -5.651707329729E-22, -7.975908114097E-22, -8.350115450764E-22, -4.488982302467E-22,
          1.385029207141E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.385029207141E-22,
          1.385029207141E-22, 1.385029207141E-22, 1.385029207141E-22, 1.385029207141E-22,
          1.385029207141E-22, -4.488982302467E-22, -4.488982302467E-22, -4.488982302467E-22,
          -4.488982302467E-22, -4.488982302467E-22, -4.488982302467E-22, -8.350115450764E-22,
          -8.350115450764E-22, -8.350115450764E-22, -8.350115450764E-22, -8.350115450764E-22,
          -8.350115450764E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -5.651707329729E-22,
          -5.651707329729E-22, -5.651707329729E-22, -5.651707329729E-22, -5.651707329729E-22,
          -5.651707329729E-22, -4.876973734940E-22, -4.876973734940E-22, -4.876973734940E-22,
          -4.876973734940E-22, -4.876973734940E-22, -4.876973734940E-22, -5.651707329729E-22,
          -5.651707329729E-22, -5.651707329729E-22, -5.651707329729E-22, -5.651707329729E-22,
          -5.651707329729E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -8.350115450764E-22,
          -8.350115450764E-22, -8.350115450764E-22, -8.350115450764E-22, -8.350115450764E-22,
          -8.350115450764E-22, -4.488982302467E-22, -4.488982302467E-22, -4.488982302467E-22,
          -4.488982302467E-22, -4.488982302467E-22, -4.488982302467E-22, 1.385029207141E-22,
          1.385029207141E-22, 1.385029207141E-22, 1.385029207141E-22, 1.385029207141E-22,
          1.385029207141E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.385029207141E-22,
          -4.488982302467E-22, -8.350115450764E-22, -7.975908114097E-22, -5.651707329729E-22,
          -4.876973734940E-22, -5.651707329729E-22, -7.975908114097E-22, -8.350115450764E-22,
          -4.488982302467E-22, 1.385029207141E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 1.385029207141E-22, -4.488982302467E-22,
          -8.350115450764E-22, -7.975908114097E-22, -5.651707329729E-22, -4.876973734940E-22,
          -5.651707329729E-22, -7.975908114097E-22, -8.350115450764E-22, -4.488982302467E-22,
          1.385029207141E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 1.385029207141E-22, -4.488982302467E-22, -8.350115450764E-22,
          -7.975908114097E-22, -5.651707329729E-22, -4.876973734940E-22, -5.651707329729E-22,
          -7.975908114097E-22, -8.350115450764E-22, -4.488982302467E-22, 1.385029207141E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22], dtype=np.float64
        )
        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [5.663043205226E-21, 6.276115134204E-21, 6.889187063182E-21, 7.502258992160E-21,
          7.655526974405E-21, -3.424867714738E-22, -1.065498256289E-20, -1.744516071724E-20,
          -1.668065798131E-20, -1.642582373601E-20, -1.668065798131E-20, -1.744516071724E-20,
          -1.065498256289E-20, -3.424867714738E-22, 7.655526974405E-21, 7.502258992160E-21,
          6.889187063182E-21, 6.276115134204E-21, 5.663043205226E-21, 6.276115134204E-21,
          5.929572720554E-21, 5.583030306904E-21, 5.236487893254E-21, 5.149852289841E-21,
          -3.779572577314E-22, -7.381658890282E-21, -1.189597074863E-20, -1.130882889853E-20,
          -1.111311494850E-20, -1.130882889853E-20, -1.189597074863E-20, -7.381658890282E-21,
          -3.779572577314E-22, 5.149852289841E-21, 5.236487893254E-21, 5.583030306904E-21,
          5.929572720554E-21, 6.276115134204E-21, 6.889187063182E-21, 5.583030306904E-21,
          4.276873550626E-21, 2.970716794347E-21, 2.644177605278E-21, -4.134277439891E-22,
          -4.108335217679E-21, -6.346780780020E-21, -5.936999815753E-21, -5.800406160998E-21,
          -5.936999815753E-21, -6.346780780020E-21, -4.108335217679E-21, -4.134277439891E-22,
          2.644177605278E-21, 2.970716794347E-21, 4.276873550626E-21, 5.583030306904E-21,
          6.889187063182E-21, 7.502258992160E-21, 5.236487893254E-21, 2.970716794347E-21,
          2.970716794347E-21, 5.236487893254E-21, 7.502258992160E-21, 7.655526974405E-21,
          5.149852289841E-21, 2.644177605278E-21, 2.644177605278E-21, 5.149852289841E-21,
          7.655526974405E-21, -3.424867714738E-22, -3.779572577314E-22, -4.134277439891E-22,
          -4.134277439891E-22, -3.779572577314E-22, -3.424867714738E-22, -1.065498256289E-20,
          -7.381658890282E-21, -4.108335217679E-21, -4.108335217679E-21, -7.381658890282E-21,
          -1.065498256289E-20, -1.744516071724E-20, -1.189597074863E-20, -6.346780780020E-21,
          -6.346780780020E-21, -1.189597074863E-20, -1.744516071724E-20, -1.668065798131E-20,
          -1.130882889853E-20, -5.936999815753E-21, -5.936999815753E-21, -1.130882889853E-20,
          -1.668065798131E-20, -1.642582373601E-20, -1.111311494850E-20, -5.800406160998E-21,
          -5.800406160998E-21, -1.111311494850E-20, -1.642582373601E-20, -1.668065798131E-20,
          -1.130882889853E-20, -5.936999815753E-21, -5.936999815753E-21, -1.130882889853E-20,
          -1.668065798131E-20, -1.744516071724E-20, -1.189597074863E-20, -6.346780780020E-21,
          -6.346780780020E-21, -1.189597074863E-20, -1.744516071724E-20, -1.065498256289E-20,
          -7.381658890282E-21, -4.108335217679E-21, -4.108335217679E-21, -7.381658890282E-21,
          -1.065498256289E-20, -3.424867714738E-22, -3.779572577314E-22, -4.134277439891E-22,
          -4.134277439891E-22, -3.779572577314E-22, -3.424867714738E-22, 7.655526974405E-21,
          5.149852289841E-21, 2.644177605278E-21, 2.644177605278E-21, 5.149852289841E-21,
          7.655526974405E-21, 7.502258992160E-21, 5.236487893254E-21, 2.970716794347E-21,
          2.970716794347E-21, 5.236487893254E-21, 7.502258992160E-21, 6.889187063182E-21,
          5.583030306904E-21, 4.276873550626E-21, 2.970716794347E-21, 2.644177605278E-21,
          -4.134277439891E-22, -4.108335217679E-21, -6.346780780020E-21, -5.936999815753E-21,
          -5.800406160998E-21, -5.936999815753E-21, -6.346780780020E-21, -4.108335217679E-21,
          -4.134277439891E-22, 2.644177605278E-21, 2.970716794347E-21, 4.276873550626E-21,
          5.583030306904E-21, 6.889187063182E-21, 6.276115134204E-21, 5.929572720554E-21,
          5.583030306904E-21, 5.236487893254E-21, 5.149852289841E-21, -3.779572577314E-22,
          -7.381658890282E-21, -1.189597074863E-20, -1.130882889853E-20, -1.111311494850E-20,
          -1.130882889853E-20, -1.189597074863E-20, -7.381658890282E-21, -3.779572577314E-22,
          5.149852289841E-21, 5.236487893254E-21, 5.583030306904E-21, 5.929572720554E-21,
          6.276115134204E-21, 5.663043205226E-21, 6.276115134204E-21, 6.889187063182E-21,
          7.502258992160E-21, 7.655526974405E-21, -3.424867714738E-22, -1.065498256289E-20,
          -1.744516071724E-20, -1.668065798131E-20, -1.642582373601E-20, -1.668065798131E-20,
          -1.744516071724E-20, -1.065498256289E-20, -3.424867714738E-22, 7.655526974405E-21,
          7.502258992160E-21, 6.889187063182E-21, 6.276115134204E-21, 5.663043205226E-21], dtype=np.float64
        )


class TestInterpolatorLoadNormalValuesUneven(TestInterpolatorLoadValues):
    """
    Loading normal sized values for a 2D sinc function test.

    For description of data storage, see TestInterpolatorLoadValues.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        self.data: np.array = np.array(
         [[7.049456954407E-02, 5.750613283457E-02, -5.031133752816E-03, -3.957902354963E-02,
           -8.474851229653E-02, -9.121180789640E-02, -7.975908114097E-02, -6.835527608142E-02,
           -4.876973734940E-02, -4.418624359019E-02, -4.876973734940E-02, -5.728794999748E-02,
           -7.975908114097E-02, -8.851381966774E-02, -8.474851229653E-02, -6.734331948907E-02,
           -5.031133752816E-03, 2.994648599639E-02, 7.049456954407E-02],
          [5.750613283457E-02, 2.707313596327E-02, -4.947098908050E-02, -7.716977009236E-02,
           -8.851381966774E-02, -7.454434661366E-02, -2.923247487940E-02, -6.867195016855E-03,
           2.378357578524E-02, 3.010595454905E-02, 2.378357578524E-02, 1.127556015714E-02,
           -2.923247487940E-02, -5.313921590225E-02, -8.851381966774E-02, -9.034807299146E-02,
           -4.947098908050E-02, -1.193366977476E-02, 5.750613283457E-02],
          [-5.031133752816E-03, -4.947098908050E-02, -9.121921863446E-02, -7.727418937786E-02,
           -9.251264987298E-04, 4.278478892878E-02, 1.052139178127E-01, 1.201126954831E-01,
           1.283205555674E-01, 1.282705054889E-01, 1.283205555674E-01, 1.266409993718E-01,
           1.052139178127E-01, 7.942173990430E-02, -9.251264987298E-04, -4.418624359019E-02,
           -9.121921863446E-02, -8.127030064783E-02, -5.031133752816E-03],
          [-3.957902354963E-02, -7.716977009236E-02, -7.727418937786E-02, -3.939635588637E-02,
           6.155045982467E-02, 1.005907311298E-01, 1.282705054889E-01, 1.215171020957E-01,
           9.674890958243E-02, 8.947967120127E-02, 9.674890958243E-02, 1.089467585825E-01,
           1.282705054889E-01, 1.227830098222E-01, 6.155045982467E-02, 1.127556015714E-02,
           -7.727418937786E-02, -9.132515608025E-02, -3.957902354963E-02],
          [-8.474851229653E-02, -8.851381966774E-02, -9.251264987298E-04, 6.155045982467E-02,
           1.283205555674E-01, 1.138884597353E-01, 1.734970013481E-02, -3.806455343631E-02,
           -1.140407180451E-01, -1.289967784984E-01, -1.140407180451E-01, -8.347707818621E-02,
           1.734970013481E-02, 7.259972187812E-02, 1.283205555674E-01, 1.095087852132E-01,
           -9.251264987298E-04, -5.728794999748E-02, -8.474851229653E-02],
          [-9.121180789640E-02, -7.454434661366E-02, 4.278478892878E-02, 1.005907311298E-01,
           1.138884597353E-01, 6.300331483592E-02, -8.347707818621E-02, -1.435151388671E-01,
           -2.022780434934E-01, -2.095142217650E-01, -2.022780434934E-01, -1.824711128590E-01,
           -8.347707818621E-02, -9.351916346701E-03, 1.138884597353E-01, 1.277678821031E-01,
           4.278478892878E-02, -2.387837583277E-02, -9.121180789640E-02],
          [-7.975908114097E-02, -2.923247487940E-02, 1.052139178127E-01, 1.282705054889E-01,
           1.734970013481E-02, -8.347707818621E-02, -2.145503300375E-01, -2.052264171648E-01,
           -9.241435356589E-02, -5.067031374727E-02, -9.241435356589E-02, -1.547945550760E-01,
           -2.145503300375E-01, -1.704692811750E-01, 1.734970013480E-02, 9.674890958243E-02,
           1.052139178127E-01, 4.278478892878E-02, -7.975908114097E-02],
          [-6.835527608142E-02, -6.867195016855E-03, 1.201126954831E-01, 1.215171020957E-01,
           -3.806455343631E-02, -1.435151388671E-01, -2.052264171648E-01, -1.268933271435E-01,
           1.257727427008E-01, 2.044658799484E-01, 1.257727427008E-01, -9.251369377679E-04,
           -2.052264171648E-01, -2.095142217650E-01, -3.806455343631E-02, 6.300331483592E-02,
           1.201126954831E-01, 6.764157650466E-02, -6.835527608142E-02],
          [-4.876973734940E-02, 2.378357578524E-02, 1.283205555674E-01, 9.674890958243E-02,
           -1.140407180451E-01, -2.022780434934E-01, -9.241435356589E-02, 1.257727427008E-01,
           6.446759109720E-01, 7.917136926885E-01, 6.446759109720E-01, 3.973381423908E-01,
           -9.241435356589E-02, -2.052264171648E-01, -1.140407180451E-01, 4.282861041084E-03,
           1.283205555674E-01, 9.566990623455E-02, -4.876973734940E-02],
          [-4.418624359019E-02, 3.010595454905E-02, 1.282705054889E-01, 8.947967120127E-02,
           -1.289967784984E-01, -2.095142217650E-01, -5.067031374727E-02, 2.044658799484E-01,
           7.917136926885E-01, 9.560513212622E-01, 7.917136926885E-01, 5.136281978234E-01,
           -5.067031374727E-02, -1.933844713753E-01, -1.289967784984E-01, -9.351916346701E-03,
           1.282705054889E-01, 1.005907311298E-01, -4.418624359019E-02],
          [-4.876973734940E-02, 2.378357578524E-02, 1.283205555674E-01, 9.674890958243E-02,
           -1.140407180451E-01, -2.022780434934E-01, -9.241435356589E-02, 1.257727427008E-01,
           6.446759109720E-01, 7.917136926885E-01, 6.446759109720E-01, 3.973381423908E-01,
           -9.241435356589E-02, -2.052264171648E-01, -1.140407180451E-01, 4.282861041084E-03,
           1.283205555674E-01, 9.566990623455E-02, -4.876973734940E-02],
          [-5.728794999748E-02, 1.127556015714E-02, 1.266409993718E-01, 1.089467585825E-01,
           -8.347707818621E-02, -1.824711128590E-01, -1.547945550760E-01, -9.251369377679E-04,
           3.973381423908E-01, 5.136281978234E-01, 3.973381423908E-01, 2.044658799484E-01,
           -1.547945550760E-01, -2.165899038868E-01, -8.347707818621E-02, 2.979548967146E-02,
           1.266409993718E-01, 8.505592925358E-02, -5.728794999748E-02],
          [-7.975908114097E-02, -2.923247487940E-02, 1.052139178127E-01, 1.282705054889E-01,
           1.734970013481E-02, -8.347707818621E-02, -2.145503300375E-01, -2.052264171648E-01,
           -9.241435356589E-02, -5.067031374727E-02, -9.241435356589E-02, -1.547945550760E-01,
           -2.145503300375E-01, -1.704692811750E-01, 1.734970013480E-02, 9.674890958243E-02,
           1.052139178127E-01, 4.278478892878E-02, -7.975908114097E-02],
          [-8.851381966774E-02, -5.313921590225E-02, 7.942173990430E-02, 1.227830098222E-01,
           7.259972187812E-02, -9.351916346701E-03, -1.704692811750E-01, -2.095142217650E-01,
           -2.052264171648E-01, -1.933844713753E-01, -2.052264171648E-01, -2.165899038868E-01,
           -1.704692811750E-01, -9.881655954565E-02, 7.259972187812E-02, 1.215171020957E-01,
           7.942173990430E-02, 1.127556015714E-02, -8.851381966774E-02],
          [-8.474851229653E-02, -8.851381966774E-02, -9.251264987298E-04, 6.155045982467E-02,
           1.283205555674E-01, 1.138884597353E-01, 1.734970013480E-02, -3.806455343631E-02,
           -1.140407180451E-01, -1.289967784984E-01, -1.140407180451E-01, -8.347707818621E-02,
           1.734970013480E-02, 7.259972187812E-02, 1.283205555674E-01, 1.095087852132E-01,
           -9.251264987296E-04, -5.728794999748E-02, -8.474851229653E-02],
          [-6.734331948907E-02, -9.034807299146E-02, -4.418624359019E-02, 1.127556015714E-02,
           1.095087852132E-01, 1.277678821031E-01, 9.674890958243E-02, 6.300331483592E-02,
           4.282861041084E-03, -9.351916346701E-03, 4.282861041084E-03, 2.979548967146E-02,
           9.674890958243E-02, 1.215171020957E-01, 1.095087852132E-01, 6.764157650466E-02,
           -4.418624359019E-02, -8.199851501756E-02, -6.734331948907E-02],
          [-5.031133752816E-03, -4.947098908050E-02, -9.121921863446E-02, -7.727418937786E-02,
           -9.251264987298E-04, 4.278478892878E-02, 1.052139178127E-01, 1.201126954831E-01,
           1.283205555674E-01, 1.282705054889E-01, 1.283205555674E-01, 1.266409993718E-01,
           1.052139178127E-01, 7.942173990430E-02, -9.251264987296E-04, -4.418624359019E-02,
           -9.121921863446E-02, -8.127030064783E-02, -5.031133752816E-03],
          [2.994648599639E-02, -1.193366977476E-02, -8.127030064783E-02, -9.132515608025E-02,
           -5.728794999748E-02, -2.387837583277E-02, 4.278478892878E-02, 6.764157650466E-02,
           9.566990623455E-02, 1.005907311298E-01, 9.566990623455E-02, 8.505592925358E-02,
           4.278478892878E-02, 1.127556015714E-02, -5.728794999748E-02, -8.199851501756E-02,
           -8.127030064783E-02, -5.265231727816E-02, 2.994648599639E-02],
          [7.049456954407E-02, 5.750613283457E-02, -5.031133752816E-03, -3.957902354963E-02,
           -8.474851229653E-02, -9.121180789640E-02, -7.975908114097E-02, -6.835527608142E-02,
           -4.876973734940E-02, -4.418624359019E-02, -4.876973734940E-02, -5.728794999748E-02,
           -7.975908114097E-02, -8.851381966774E-02, -8.474851229653E-02, -6.734331948907E-02,
           -5.031133752816E-03, 2.994648599639E-02, 7.049456954407E-02]],
            dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E-02, 2.120303154043E-02, -5.412207605158E-02, -9.041927189809E-02,
           -7.975908114097E-02, -5.490808448662E-02, -4.365385529969E-02, -5.493373238953E-02,
           -7.975908114097E-02, -9.043881788286E-02, -5.458241224645E-02, 2.183159991917E-02,
           7.049456954407E-02],
          [2.120303154043E-02, -6.840828181977E-02, -8.291758075355E-02, -1.514915773345E-02,
           6.146617965847E-02, 1.016705772106E-01, 1.117191819513E-01, 1.019683374662E-01,
           6.146617965847E-02, -1.567520827568E-02, -8.211550352574E-02, -6.795064436020E-02,
           2.120303154043E-02],
          [-5.412207605158E-02, -8.291758075355E-02, 9.666098759062E-03, 1.115403574145E-01,
           1.180976383194E-01, 7.071918180842E-02, 4.552636512940E-02, 7.066851884792E-02,
           1.180976383194E-01, 1.094887240678E-01, 1.271424487456E-02, -8.489926255180E-02,
           -5.412207605158E-02],
          [-9.041927189809E-02, -1.514915773345E-02, 1.115403574145E-01, 9.058021229999E-02,
           -5.921575988005E-02, -1.697171170804E-01, -1.986325220649E-01, -1.708674849009E-01,
           -5.921575988005E-02, 9.071011939526E-02, 1.130155601111E-01, -1.774015779353E-02,
           -9.041927189809E-02],
          [-7.975908114097E-02, 6.146617965847E-02, 1.180976383194E-01, -5.921575988005E-02,
           -2.145503300375E-01, -1.373726429185E-01, -4.677155115289E-02, -1.400744853311E-01,
           -2.145503300375E-01, -5.608764908063E-02, 1.171252709523E-01, 5.937941570370E-02,
           -7.975908114097E-02],
          [-5.490808448662E-02, 1.016705772106E-01, 7.071918180842E-02, -1.697171170804E-01,
           -1.373726429185E-01, 3.180255899274E-01, 6.051776838087E-01, 3.123175894536E-01,
           -1.373726429185E-01, -1.640188016429E-01, 6.586737199675E-02, 1.009003074435E-01,
           -5.490808448662E-02],
          [-4.365385529969E-02, 1.117191819513E-01, 4.552636512940E-02, -1.986325220649E-01,
           -4.677155115289E-02, 6.051776838087E-01, 9.899192445544E-01, 5.987956201910E-01,
           -4.677155115289E-02, -1.913782291956E-01, 3.931492909576E-02, 1.117308113990E-01,
           -4.365385529969E-02],
          [-5.493373238953E-02, 1.019683374662E-01, 7.066851884792E-02, -1.708674849009E-01,
           -1.400744853311E-01, 3.123175894536E-01, 5.987956201910E-01, 3.081035634172E-01,
           -1.400744853311E-01, -1.646011917477E-01, 6.660176847751E-02, 1.011817145737E-01,
           -5.493373238953E-02],
          [-7.975908114097E-02, 6.146617965847E-02, 1.180976383194E-01, -5.921575988005E-02,
           -2.145503300375E-01, -1.373726429185E-01, -4.677155115289E-02, -1.400744853311E-01,
           -2.145503300375E-01, -5.608764908063E-02, 1.171252709523E-01, 5.937941570370E-02,
           -7.975908114097E-02],
          [-9.043881788286E-02, -1.567520827568E-02, 1.094887240678E-01, 9.071011939526E-02,
           -5.608764908063E-02, -1.640188016429E-01, -1.913782291956E-01, -1.646011917477E-01,
           -5.608764908063E-02, 9.061889238735E-02, 1.119056766335E-01, -1.844151449703E-02,
           -9.043881788286E-02],
          [-5.458241224645E-02, -8.211550352574E-02, 1.271424487456E-02, 1.130155601111E-01,
           1.171252709523E-01, 6.586737199675E-02, 3.931492909576E-02, 6.660176847751E-02,
           1.171252709523E-01, 1.119056766335E-01, 1.568973617012E-02, -8.405046373540E-02,
           -5.458241224645E-02],
          [2.183159991917E-02, -6.795064436020E-02, -8.489926255180E-02, -1.774015779353E-02,
           5.937941570370E-02, 1.009003074435E-01, 1.117308113990E-01, 1.011817145737E-01,
           5.937941570370E-02, -1.844151449703E-02, -8.405046373540E-02, -6.845929971507E-02,
           2.183159991917E-02],
          [7.049456954407E-02, 2.120303154043E-02, -5.412207605158E-02, -9.041927189809E-02,
           -7.975908114097E-02, -5.490808448662E-02, -4.365385529969E-02, -5.493373238953E-02,
           -7.975908114097E-02, -9.043881788286E-02, -5.458241224645E-02, 2.183159991917E-02,
           7.049456954407E-02]], dtype=np.float64
        )

        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          2.120303154043E-02, -5.412207605158E-02, -9.041927189809E-02, -7.975908114097E-02,
          -5.490808448662E-02, -4.365385529969E-02, -5.493373238953E-02, -7.975908114097E-02,
          -9.043881788286E-02, -5.458241224645E-02, 2.183159991917E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 2.120303154043E-02,
          -5.412207605158E-02, -9.041927189809E-02, -7.975908114097E-02, -5.490808448662E-02,
          -4.365385529969E-02, -5.493373238953E-02, -7.975908114097E-02, -9.043881788286E-02,
          -5.458241224645E-02, 2.183159991917E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 2.120303154043E-02, -5.412207605158E-02,
          -9.041927189809E-02, -7.975908114097E-02, -5.490808448662E-02, -4.365385529969E-02,
          -5.493373238953E-02, -7.975908114097E-02, -9.043881788286E-02, -5.458241224645E-02,
          2.183159991917E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 2.120303154043E-02,
          2.120303154043E-02, 2.120303154043E-02, 2.120303154043E-02, 2.120303154043E-02,
          2.120303154043E-02, -5.412207605158E-02, -5.412207605158E-02, -5.412207605158E-02,
          -5.412207605158E-02, -5.412207605158E-02, -5.412207605158E-02, -9.041927189809E-02,
          -9.041927189809E-02, -9.041927189809E-02, -9.041927189809E-02, -9.041927189809E-02,
          -9.041927189809E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -5.490808448662E-02,
          -5.490808448662E-02, -5.490808448662E-02, -5.490808448662E-02, -5.490808448662E-02,
          -5.490808448662E-02, -4.365385529969E-02, -4.365385529969E-02, -4.365385529969E-02,
          -4.365385529969E-02, -4.365385529969E-02, -4.365385529969E-02, -5.493373238953E-02,
          -5.493373238953E-02, -5.493373238953E-02, -5.493373238953E-02, -5.493373238953E-02,
          -5.493373238953E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -9.043881788286E-02,
          -9.043881788286E-02, -9.043881788286E-02, -9.043881788286E-02, -9.043881788286E-02,
          -9.043881788286E-02, -5.458241224645E-02, -5.458241224645E-02, -5.458241224645E-02,
          -5.458241224645E-02, -5.458241224645E-02, -5.458241224645E-02, 2.183159991917E-02,
          2.183159991917E-02, 2.183159991917E-02, 2.183159991917E-02, 2.183159991917E-02,
          2.183159991917E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 2.120303154043E-02,
          -5.412207605158E-02, -9.041927189809E-02, -7.975908114097E-02, -5.490808448662E-02,
          -4.365385529969E-02, -5.493373238953E-02, -7.975908114097E-02, -9.043881788286E-02,
          -5.458241224645E-02, 2.183159991917E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 2.120303154043E-02, -5.412207605158E-02,
          -9.041927189809E-02, -7.975908114097E-02, -5.490808448662E-02, -4.365385529969E-02,
          -5.493373238953E-02, -7.975908114097E-02, -9.043881788286E-02, -5.458241224645E-02,
          2.183159991917E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 2.120303154043E-02, -5.412207605158E-02, -9.041927189809E-02,
          -7.975908114097E-02, -5.490808448662E-02, -4.365385529969E-02, -5.493373238953E-02,
          -7.975908114097E-02, -9.043881788286E-02, -5.458241224645E-02, 2.183159991917E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02], dtype=np.float64
        )
        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E+01, -7.823082017163E+00, -3.700949828231E+00, 4.211823607006E-01,
          1.132303125654E+00, 7.609499224352E-01, -4.084169620081E-01, -1.443977450203E+00,
          -1.920403690376E+00, -2.047040864433E+00, -1.948441248529E+00, -1.443977450203E+00,
          -3.911849985106E-01, 7.440065969941E-01, 1.211163300265E+00, 4.211823607006E-01,
          -2.906695129325E+00, -6.234572619351E+00, -9.562450109377E+00, -7.823082017163E+00,
          -5.113959201337E+00, -2.404836385511E+00, 3.042864303151E-01, 7.619364276159E-01,
          4.892592562729E-01, -3.024177319714E-01, -9.892379938492E-01, -1.298571821746E+00,
          -1.379245194722E+00, -1.317272076482E+00, -9.892379938492E-01, -2.909362716347E-01,
          4.778102605806E-01, 8.147194001498E-01, 3.042864303151E-01, -1.853476437714E+00,
          -4.011239305743E+00, -6.169002173772E+00, -3.700949828231E+00, -2.404836385511E+00,
          -1.108722942791E+00, 1.873904999296E-01, 3.915697295782E-01, 2.175685901107E-01,
          -1.964185019348E-01, -5.344985374951E-01, -6.767399531163E-01, -7.114495250108E-01,
          -6.861029044360E-01, -5.344985374951E-01, -1.906875447588E-01, 2.116139241671E-01,
          4.182755000345E-01, 1.873904999296E-01, -8.002577461027E-01, -1.787905992135E+00,
          -2.775554238167E+00, 4.211823607006E-01, 3.042864303151E-01, 1.873904999296E-01,
          2.529609455086E-01, 4.354273214732E-01, 6.178936974377E-01, 1.132303125654E+00,
          7.619364276159E-01, 3.915697295782E-01, 3.717844299417E-01, 7.223658283430E-01,
          1.072947226744E+00, 7.609499224352E-01, 4.892592562729E-01, 2.175685901107E-01,
          9.604055355141E-02, 2.462031831544E-01, 3.963658127574E-01, -4.084169620081E-01,
          -3.024177319714E-01, -1.964185019348E-01, -3.537002512839E-01, -6.169812306696E-01,
          -8.802622100554E-01, -1.443977450203E+00, -9.892379938492E-01, -5.344985374951E-01,
          -6.312064964548E-01, -1.182653911769E+00, -1.734101327083E+00, -1.920403690376E+00,
          -1.298571821746E+00, -6.767399531163E-01, -6.947802448856E-01, -1.334652405285E+00,
          -1.974524565683E+00, -2.047040864433E+00, -1.379245194722E+00, -7.114495250108E-01,
          -6.952552096349E-01, -1.346856563970E+00, -1.998457918305E+00, -1.948441248529E+00,
          -1.317272076482E+00, -6.861029044360E-01, -7.014023360164E-01, -1.347870939643E+00,
          -1.994339543270E+00, -1.443977450203E+00, -9.892379938492E-01, -5.344985374951E-01,
          -6.312064964548E-01, -1.182653911769E+00, -1.734101327083E+00, -3.911849985106E-01,
          -2.909362716347E-01, -1.906875447588E-01, -3.422173851071E-01, -5.939959523313E-01,
          -8.457745195555E-01, 7.440065969941E-01, 4.778102605806E-01, 2.116139241671E-01,
          8.886095091330E-02, 2.323043140730E-01, 3.757476772328E-01, 1.211163300265E+00,
          8.147194001498E-01, 4.182755000345E-01, 4.013604681385E-01, 7.808893363578E-01,
          1.160418204577E+00, 4.211823607006E-01, 3.042864303151E-01, 1.873904999296E-01,
          2.529609455086E-01, 4.354273214732E-01, 6.178936974378E-01, -2.906695129325E+00,
          -1.853476437714E+00, -8.002577461027E-01, 2.529609455086E-01, 3.717844299417E-01,
          9.604055355141E-02, -3.537002512839E-01, -6.312064964548E-01, -6.947802448856E-01,
          -6.952552096349E-01, -7.014023360164E-01, -6.312064964548E-01, -3.422173851071E-01,
          8.886095091330E-02, 4.013604681385E-01, 2.529609455086E-01, -4.160997529959E-01,
          -1.085160451500E+00, -1.754221150005E+00, -6.234572619351E+00, -4.011239305743E+00,
          -1.787905992135E+00, 4.354273214732E-01, 7.223658283430E-01, 2.462031831544E-01,
          -6.169812306696E-01, -1.182653911769E+00, -1.334652405285E+00, -1.346856563970E+00,
          -1.347870939643E+00, -1.182653911769E+00, -5.939959523313E-01, 2.323043140730E-01,
          7.808893363578E-01, 4.354273214732E-01, -1.085160451500E+00, -2.605748224474E+00,
          -4.126335997448E+00, -9.562450109377E+00, -6.169002173772E+00, -2.775554238167E+00,
          6.178936974377E-01, 1.072947226744E+00, 3.963658127574E-01, -8.802622100554E-01,
          -1.734101327083E+00, -1.974524565683E+00, -1.998457918305E+00, -1.994339543270E+00,
          -1.734101327083E+00, -8.457745195555E-01, 3.757476772328E-01, 1.160418204577E+00,
          6.178936974378E-01, -1.754221150005E+00, -4.126335997448E+00, -6.498450844890E+00], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E-02, 1.842034121745E-02, -5.087139573635E-02, -8.959598399643E-02,
           -7.975908114097E-02, -5.611431437391E-02, -4.533211702999E-02, -5.515839683546E-02,
           -7.975908114097E-02, -8.616050256073E-02, -5.176527305501E-02, 2.120208105909E-02,
           7.049456954407E-02],
          [1.842034121745E-02, -5.501487366574E-02, -6.636899571863E-02, -9.352949284003E-03,
           5.479652055316E-02, 8.288520629779E-02, 9.087389620211E-02, 8.481401678700E-02,
           5.479652055316E-02, -9.965035969377E-03, -6.501360534968E-02, -6.034270619075E-02,
           1.842034121745E-02],
          [-5.087139573635E-02, -6.636899571863E-02, 8.941006971131E-03, 9.749711840097E-02,
           1.005403041504E-01, 5.814032225198E-02, 3.715829475115E-02, 5.664347521163E-02,
           1.005403041504E-01, 9.024081028879E-02, 1.232866890135E-02, -7.665862183419E-02,
           -5.087139573635E-02],
          [-8.959598399643E-02, -9.352949284003E-03, 9.749711840097E-02, 8.616757171890E-02,
           -5.827038360596E-02, -1.565688797731E-01, -1.870933237441E-01, -1.633466311759E-01,
           -5.827038360596E-02, 7.761129976189E-02, 1.003666584284E-01, -1.620874951248E-02,
           -8.959598399643E-02],
          [-7.975908114097E-02, 5.479652055316E-02, 1.005403041504E-01, -5.827038360596E-02,
           -2.145503300375E-01, -1.347188774155E-01, -6.110632370193E-02, -1.391995046985E-01,
           -2.145503300375E-01, -5.308241785637E-02, 9.886516164000E-02, 5.839207114976E-02,
           -7.975908114097E-02],
          [-5.611431437391E-02, 8.288520629779E-02, 5.814032225198E-02, -1.565688797731E-01,
           -1.347188774155E-01, 2.929381267349E-01, 5.411436279008E-01, 2.985138651995E-01,
           -1.347188774155E-01, -1.310314141567E-01, 5.103792541956E-02, 9.518011394833E-02,
           -5.611431437391E-02],
          [-4.533211702999E-02, 9.087389620211E-02, 3.715829475115E-02, -1.870933237441E-01,
           -6.110632370193E-02, 5.411436279008E-01, 8.749637474039E-01, 5.521553247888E-01,
           -6.110632370193E-02, -1.519154612992E-01, 2.761333800232E-02, 1.065911481816E-01,
           -4.533211702999E-02],
          [-5.515839683546E-02, 8.481401678700E-02, 5.664347521163E-02, -1.633466311759E-01,
           -1.391995046985E-01, 2.985138651995E-01, 5.521553247888E-01, 3.043061053033E-01,
           -1.391995046985E-01, -1.371046296717E-01, 4.932822149057E-02, 9.754728972929E-02,
           -5.515839683546E-02],
          [-7.975908114097E-02, 5.479652055316E-02, 1.005403041504E-01, -5.827038360596E-02,
           -2.145503300375E-01, -1.347188774155E-01, -6.110632370193E-02, -1.391995046985E-01,
           -2.145503300375E-01, -5.308241785637E-02, 9.886516164000E-02, 5.839207114976E-02,
           -7.975908114097E-02],
          [-8.616050256073E-02, -9.965035969377E-03, 9.024081028879E-02, 7.761129976189E-02,
           -5.308241785637E-02, -1.310314141567E-01, -1.519154612992E-01, -1.371046296717E-01,
           -5.308241785637E-02, 7.026025796278E-02, 9.281016513370E-02, -1.638123816652E-02,
           -8.616050256073E-02],
          [-5.176527305501E-02, -6.501360534968E-02, 1.232866890135E-02, 1.003666584284E-01,
           9.886516164000E-02, 5.103792541956E-02, 2.761333800232E-02, 4.932822149057E-02,
           9.886516164000E-02, 9.281016513370E-02, 1.577734427290E-02, -7.534846790666E-02,
           -5.176527305501E-02],
          [2.120208105909E-02, -6.034270619075E-02, -7.665862183419E-02, -1.620874951248E-02,
           5.839207114976E-02, 9.518011394833E-02, 1.065911481816E-01, 9.754728972929E-02,
           5.839207114976E-02, -1.638123816652E-02, -7.534846790666E-02, -6.579449237655E-02,
           2.120208105909E-02],
          [7.049456954407E-02, 1.842034121745E-02, -5.087139573635E-02, -8.959598399643E-02,
           -7.975908114097E-02, -5.611431437391E-02, -4.533211702999E-02, -5.515839683546E-02,
           -7.975908114097E-02, -8.616050256073E-02, -5.176527305501E-02, 2.120208105909E-02,
           7.049456954407E-02]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          1.842034121745E-02, -5.087139573635E-02, -8.959598399643E-02, -7.975908114097E-02,
          -5.611431437391E-02, -4.533211702999E-02, -5.515839683546E-02, -7.975908114097E-02,
          -8.616050256073E-02, -5.176527305501E-02, 2.120208105909E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.842034121745E-02,
          -5.087139573635E-02, -8.959598399643E-02, -7.975908114097E-02, -5.611431437391E-02,
          -4.533211702999E-02, -5.515839683546E-02, -7.975908114097E-02, -8.616050256073E-02,
          -5.176527305501E-02, 2.120208105909E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 1.842034121745E-02, -5.087139573635E-02,
          -8.959598399643E-02, -7.975908114097E-02, -5.611431437391E-02, -4.533211702999E-02,
          -5.515839683546E-02, -7.975908114097E-02, -8.616050256073E-02, -5.176527305501E-02,
          2.120208105909E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.842034121745E-02,
          1.842034121745E-02, 1.842034121745E-02, 1.842034121745E-02, 1.842034121745E-02,
          1.842034121745E-02, -5.087139573635E-02, -5.087139573635E-02, -5.087139573635E-02,
          -5.087139573635E-02, -5.087139573635E-02, -5.087139573635E-02, -8.959598399643E-02,
          -8.959598399643E-02, -8.959598399643E-02, -8.959598399643E-02, -8.959598399643E-02,
          -8.959598399643E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -5.611431437391E-02,
          -5.611431437391E-02, -5.611431437391E-02, -5.611431437391E-02, -5.611431437391E-02,
          -5.611431437391E-02, -4.533211702999E-02, -4.533211702999E-02, -4.533211702999E-02,
          -4.533211702999E-02, -4.533211702999E-02, -4.533211702999E-02, -5.515839683546E-02,
          -5.515839683546E-02, -5.515839683546E-02, -5.515839683546E-02, -5.515839683546E-02,
          -5.515839683546E-02, -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02,
          -7.975908114097E-02, -7.975908114097E-02, -7.975908114097E-02, -8.616050256073E-02,
          -8.616050256073E-02, -8.616050256073E-02, -8.616050256073E-02, -8.616050256073E-02,
          -8.616050256073E-02, -5.176527305501E-02, -5.176527305501E-02, -5.176527305501E-02,
          -5.176527305501E-02, -5.176527305501E-02, -5.176527305501E-02, 2.120208105909E-02,
          2.120208105909E-02, 2.120208105909E-02, 2.120208105909E-02, 2.120208105909E-02,
          2.120208105909E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 1.842034121745E-02,
          -5.087139573635E-02, -8.959598399643E-02, -7.975908114097E-02, -5.611431437391E-02,
          -4.533211702999E-02, -5.515839683546E-02, -7.975908114097E-02, -8.616050256073E-02,
          -5.176527305501E-02, 2.120208105909E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 1.842034121745E-02, -5.087139573635E-02,
          -8.959598399643E-02, -7.975908114097E-02, -5.611431437391E-02, -4.533211702999E-02,
          -5.515839683546E-02, -7.975908114097E-02, -8.616050256073E-02, -5.176527305501E-02,
          2.120208105909E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02,
          7.049456954407E-02, 1.842034121745E-02, -5.087139573635E-02, -8.959598399643E-02,
          -7.975908114097E-02, -5.611431437391E-02, -4.533211702999E-02, -5.515839683546E-02,
          -7.975908114097E-02, -8.616050256073E-02, -5.176527305501E-02, 2.120208105909E-02,
          7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02, 7.049456954407E-02], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E+01, -7.823082017163E+00, -3.700949828231E+00, 4.211823607006E-01,
          1.076476993194E+00, 7.357570465096E-01, -4.016962502163E-01, -1.443977450203E+00,
          -1.903018294300E+00, -2.039483993008E+00, -1.933304341125E+00, -1.443977450203E+00,
          -3.807888037972E-01, 7.140500088303E-01, 1.169244258887E+00, 4.211823607006E-01,
          -2.906695129325E+00, -6.234572619351E+00, -9.562450109377E+00, -7.823082017163E+00,
          -5.113959201337E+00, -2.404836385511E+00, 3.042864303151E-01, 7.237914425352E-01,
          4.735475657609E-01, -2.976628281430E-01, -9.892379938492E-01, -1.287383634324E+00,
          -1.374766701016E+00, -1.307255693029E+00, -9.892379938492E-01, -2.825793700517E-01,
          4.587782482018E-01, 7.865635329442E-01, 3.042864303151E-01, -1.853476437714E+00,
          -4.011239305743E+00, -6.169002173772E+00, -3.700949828231E+00, -2.404836385511E+00,
          -1.108722942791E+00, 1.873904999296E-01, 3.711058918763E-01, 2.113380850123E-01,
          -1.936294060697E-01, -5.344985374951E-01, -6.717489743492E-01, -7.100494090228E-01,
          -6.812070449321E-01, -5.344985374951E-01, -1.843699363062E-01, 2.035064875734E-01,
          4.038828070016E-01, 1.873904999296E-01, -8.002577461027E-01, -1.787905992135E+00,
          -2.775554238167E+00, 4.211823607006E-01, 3.042864303151E-01, 1.873904999296E-01,
          2.529609455086E-01, 4.354273214732E-01, 6.178936974377E-01, 1.076476993194E+00,
          7.237914425352E-01, 3.711058918763E-01, 3.500226650129E-01, 6.816249888084E-01,
          1.013227312604E+00, 7.357570465096E-01, 4.735475657609E-01, 2.113380850123E-01,
          9.287866896806E-02, 2.366287336725E-01, 3.803787983769E-01, -4.016962502163E-01,
          -2.976628281430E-01, -1.936294060697E-01, -3.477394497976E-01, -6.058829155988E-01,
          -8.640263814000E-01, -1.443977450203E+00, -9.892379938492E-01, -5.344985374951E-01,
          -6.312064964548E-01, -1.182653911769E+00, -1.734101327083E+00, -1.903018294300E+00,
          -1.287383634324E+00, -6.717489743492E-01, -6.918455006928E-01, -1.327576687012E+00,
          -1.963307873331E+00, -2.039483993008E+00, -1.374766701016E+00, -7.100494090228E-01,
          -6.964490057419E-01, -1.347565894454E+00, -1.998682783166E+00, -1.933304341125E+00,
          -1.307255693029E+00, -6.812070449321E-01, -6.980635883397E-01, -1.340968779844E+00,
          -1.983873971348E+00, -1.443977450203E+00, -9.892379938492E-01, -5.344985374951E-01,
          -6.312064964548E-01, -1.182653911769E+00, -1.734101327083E+00, -3.807888037972E-01,
          -2.825793700517E-01, -1.843699363062E-01, -3.317879124813E-01, -5.774153224019E-01,
          -8.230427323224E-01, 7.140500088303E-01, 4.587782482018E-01, 2.035064875734E-01,
          8.346507461054E-02, 2.186954222761E-01, 3.539257699416E-01, 1.169244258887E+00,
          7.865635329442E-01, 4.038828070016E-01, 3.857421048676E-01, 7.502821286761E-01,
          1.114822152485E+00, 4.211823607006E-01, 3.042864303151E-01, 1.873904999296E-01,
          2.529609455086E-01, 4.354273214732E-01, 6.178936974377E-01, -2.906695129325E+00,
          -1.853476437714E+00, -8.002577461027E-01, 2.529609455086E-01, 3.500226650129E-01,
          9.287866896806E-02, -3.477394497976E-01, -6.312064964548E-01, -6.918455006928E-01,
          -6.964490057419E-01, -6.980635883397E-01, -6.312064964548E-01, -3.317879124813E-01,
          8.346507461054E-02, 3.857421048676E-01, 2.529609455086E-01, -4.160997529959E-01,
          -1.085160451500E+00, -1.754221150005E+00, -6.234572619351E+00, -4.011239305743E+00,
          -1.787905992135E+00, 4.354273214732E-01, 6.816249888084E-01, 2.366287336725E-01,
          -6.058829155988E-01, -1.182653911769E+00, -1.327576687012E+00, -1.347565894454E+00,
          -1.340968779844E+00, -1.182653911769E+00, -5.774153224019E-01, 2.186954222761E-01,
          7.502821286761E-01, 4.354273214732E-01, -1.085160451500E+00, -2.605748224474E+00,
          -4.126335997448E+00, -9.562450109377E+00, -6.169002173772E+00, -2.775554238167E+00,
          6.178936974377E-01, 1.013227312604E+00, 3.803787983769E-01, -8.640263814000E-01,
          -1.734101327083E+00, -1.963307873331E+00, -1.998682783166E+00, -1.983873971348E+00,
          -1.734101327083E+00, -8.230427323224E-01, 3.539257699416E-01, 1.114822152485E+00,
          6.178936974377E-01, -1.754221150005E+00, -4.126335997448E+00, -6.498450844891E+00], dtype=np.float64
        )


class TestInterpolatorLoadBigValuesUneven(TestInterpolatorLoadValues):
    """
    Loading small values (10^-20 times the original) instead of the original 2D sinc function test.

    For description of data storage, see TestInterpolatorLoadValues.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        # self.data: np.array = np.sin(self.x)
        self.data: np.array = np.array(
         [[7.049456954407E+18, 5.750613283457E+18, -5.031133752816E+17, -3.957902354963E+18,
           -8.474851229653E+18, -9.121180789640E+18, -7.975908114097E+18, -6.835527608142E+18,
           -4.876973734940E+18, -4.418624359019E+18, -4.876973734940E+18, -5.728794999748E+18,
           -7.975908114097E+18, -8.851381966774E+18, -8.474851229653E+18, -6.734331948907E+18,
           -5.031133752816E+17, 2.994648599639E+18, 7.049456954407E+18],
          [5.750613283457E+18, 2.707313596327E+18, -4.947098908050E+18, -7.716977009236E+18,
           -8.851381966774E+18, -7.454434661366E+18, -2.923247487940E+18, -6.867195016855E+17,
           2.378357578524E+18, 3.010595454905E+18, 2.378357578524E+18, 1.127556015714E+18,
           -2.923247487940E+18, -5.313921590225E+18, -8.851381966774E+18, -9.034807299146E+18,
           -4.947098908050E+18, -1.193366977476E+18, 5.750613283457E+18],
          [-5.031133752816E+17, -4.947098908050E+18, -9.121921863446E+18, -7.727418937786E+18,
           -9.251264987298E+16, 4.278478892878E+18, 1.052139178127E+19, 1.201126954831E+19,
           1.283205555674E+19, 1.282705054889E+19, 1.283205555674E+19, 1.266409993718E+19,
           1.052139178127E+19, 7.942173990430E+18, -9.251264987298E+16, -4.418624359019E+18,
           -9.121921863446E+18, -8.127030064783E+18, -5.031133752816E+17],
          [-3.957902354963E+18, -7.716977009236E+18, -7.727418937786E+18, -3.939635588637E+18,
           6.155045982467E+18, 1.005907311298E+19, 1.282705054889E+19, 1.215171020957E+19,
           9.674890958243E+18, 8.947967120127E+18, 9.674890958243E+18, 1.089467585825E+19,
           1.282705054889E+19, 1.227830098222E+19, 6.155045982467E+18, 1.127556015714E+18,
           -7.727418937786E+18, -9.132515608025E+18, -3.957902354963E+18],
          [-8.474851229653E+18, -8.851381966774E+18, -9.251264987298E+16, 6.155045982467E+18,
           1.283205555674E+19, 1.138884597353E+19, 1.734970013481E+18, -3.806455343631E+18,
           -1.140407180451E+19, -1.289967784984E+19, -1.140407180451E+19, -8.347707818621E+18,
           1.734970013481E+18, 7.259972187812E+18, 1.283205555674E+19, 1.095087852132E+19,
           -9.251264987298E+16, -5.728794999748E+18, -8.474851229653E+18],
          [-9.121180789640E+18, -7.454434661366E+18, 4.278478892878E+18, 1.005907311298E+19,
           1.138884597353E+19, 6.300331483592E+18, -8.347707818621E+18, -1.435151388671E+19,
           -2.022780434934E+19, -2.095142217650E+19, -2.022780434934E+19, -1.824711128590E+19,
           -8.347707818621E+18, -9.351916346701E+17, 1.138884597353E+19, 1.277678821031E+19,
           4.278478892878E+18, -2.387837583277E+18, -9.121180789640E+18],
          [-7.975908114097E+18, -2.923247487940E+18, 1.052139178127E+19, 1.282705054889E+19,
           1.734970013481E+18, -8.347707818621E+18, -2.145503300375E+19, -2.052264171648E+19,
           -9.241435356589E+18, -5.067031374727E+18, -9.241435356589E+18, -1.547945550760E+19,
           -2.145503300375E+19, -1.704692811750E+19, 1.734970013480E+18, 9.674890958243E+18,
           1.052139178127E+19, 4.278478892878E+18, -7.975908114097E+18],
          [-6.835527608142E+18, -6.867195016855E+17, 1.201126954831E+19, 1.215171020957E+19,
           -3.806455343631E+18, -1.435151388671E+19, -2.052264171648E+19, -1.268933271435E+19,
           1.257727427008E+19, 2.044658799484E+19, 1.257727427008E+19, -9.251369377679E+16,
           -2.052264171648E+19, -2.095142217650E+19, -3.806455343631E+18, 6.300331483592E+18,
           1.201126954831E+19, 6.764157650466E+18, -6.835527608142E+18],
          [-4.876973734940E+18, 2.378357578524E+18, 1.283205555674E+19, 9.674890958243E+18,
           -1.140407180451E+19, -2.022780434934E+19, -9.241435356589E+18, 1.257727427008E+19,
           6.446759109720E+19, 7.917136926885E+19, 6.446759109720E+19, 3.973381423908E+19,
           -9.241435356589E+18, -2.052264171648E+19, -1.140407180451E+19, 4.282861041084E+17,
           1.283205555674E+19, 9.566990623455E+18, -4.876973734940E+18],
          [-4.418624359019E+18, 3.010595454905E+18, 1.282705054889E+19, 8.947967120127E+18,
           -1.289967784984E+19, -2.095142217650E+19, -5.067031374727E+18, 2.044658799484E+19,
           7.917136926885E+19, 9.560513212622E+19, 7.917136926885E+19, 5.136281978234E+19,
           -5.067031374727E+18, -1.933844713753E+19, -1.289967784984E+19, -9.351916346701E+17,
           1.282705054889E+19, 1.005907311298E+19, -4.418624359019E+18],
          [-4.876973734940E+18, 2.378357578524E+18, 1.283205555674E+19, 9.674890958243E+18,
           -1.140407180451E+19, -2.022780434934E+19, -9.241435356589E+18, 1.257727427008E+19,
           6.446759109720E+19, 7.917136926885E+19, 6.446759109720E+19, 3.973381423908E+19,
           -9.241435356589E+18, -2.052264171648E+19, -1.140407180451E+19, 4.282861041084E+17,
           1.283205555674E+19, 9.566990623455E+18, -4.876973734940E+18],
          [-5.728794999748E+18, 1.127556015714E+18, 1.266409993718E+19, 1.089467585825E+19,
           -8.347707818621E+18, -1.824711128590E+19, -1.547945550760E+19, -9.251369377679E+16,
           3.973381423908E+19, 5.136281978234E+19, 3.973381423908E+19, 2.044658799484E+19,
           -1.547945550760E+19, -2.165899038868E+19, -8.347707818621E+18, 2.979548967146E+18,
           1.266409993718E+19, 8.505592925358E+18, -5.728794999748E+18],
          [-7.975908114097E+18, -2.923247487940E+18, 1.052139178127E+19, 1.282705054889E+19,
           1.734970013481E+18, -8.347707818621E+18, -2.145503300375E+19, -2.052264171648E+19,
           -9.241435356589E+18, -5.067031374727E+18, -9.241435356589E+18, -1.547945550760E+19,
           -2.145503300375E+19, -1.704692811750E+19, 1.734970013480E+18, 9.674890958243E+18,
           1.052139178127E+19, 4.278478892878E+18, -7.975908114097E+18],
          [-8.851381966774E+18, -5.313921590225E+18, 7.942173990430E+18, 1.227830098222E+19,
           7.259972187812E+18, -9.351916346701E+17, -1.704692811750E+19, -2.095142217650E+19,
           -2.052264171648E+19, -1.933844713753E+19, -2.052264171648E+19, -2.165899038868E+19,
           -1.704692811750E+19, -9.881655954565E+18, 7.259972187812E+18, 1.215171020957E+19,
           7.942173990430E+18, 1.127556015714E+18, -8.851381966774E+18],
          [-8.474851229653E+18, -8.851381966774E+18, -9.251264987298E+16, 6.155045982467E+18,
           1.283205555674E+19, 1.138884597353E+19, 1.734970013480E+18, -3.806455343631E+18,
           -1.140407180451E+19, -1.289967784984E+19, -1.140407180451E+19, -8.347707818621E+18,
           1.734970013480E+18, 7.259972187812E+18, 1.283205555674E+19, 1.095087852132E+19,
           -9.251264987296E+16, -5.728794999748E+18, -8.474851229653E+18],
          [-6.734331948907E+18, -9.034807299146E+18, -4.418624359019E+18, 1.127556015714E+18,
           1.095087852132E+19, 1.277678821031E+19, 9.674890958243E+18, 6.300331483592E+18,
           4.282861041084E+17, -9.351916346701E+17, 4.282861041084E+17, 2.979548967146E+18,
           9.674890958243E+18, 1.215171020957E+19, 1.095087852132E+19, 6.764157650466E+18,
           -4.418624359019E+18, -8.199851501756E+18, -6.734331948907E+18],
          [-5.031133752816E+17, -4.947098908050E+18, -9.121921863446E+18, -7.727418937786E+18,
           -9.251264987298E+16, 4.278478892878E+18, 1.052139178127E+19, 1.201126954831E+19,
           1.283205555674E+19, 1.282705054889E+19, 1.283205555674E+19, 1.266409993718E+19,
           1.052139178127E+19, 7.942173990430E+18, -9.251264987296E+16, -4.418624359019E+18,
           -9.121921863446E+18, -8.127030064783E+18, -5.031133752816E+17],
          [2.994648599639E+18, -1.193366977476E+18, -8.127030064783E+18, -9.132515608025E+18,
           -5.728794999748E+18, -2.387837583277E+18, 4.278478892878E+18, 6.764157650466E+18,
           9.566990623455E+18, 1.005907311298E+19, 9.566990623455E+18, 8.505592925358E+18,
           4.278478892878E+18, 1.127556015714E+18, -5.728794999748E+18, -8.199851501756E+18,
           -8.127030064783E+18, -5.265231727816E+18, 2.994648599639E+18],
          [7.049456954407E+18, 5.750613283457E+18, -5.031133752816E+17, -3.957902354963E+18,
           -8.474851229653E+18, -9.121180789640E+18, -7.975908114097E+18, -6.835527608142E+18,
           -4.876973734940E+18, -4.418624359019E+18, -4.876973734940E+18, -5.728794999748E+18,
           -7.975908114097E+18, -8.851381966774E+18, -8.474851229653E+18, -6.734331948907E+18,
           -5.031133752816E+17, 2.994648599639E+18, 7.049456954407E+18]], dtype=np.float64
        )

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = np.array(
            [], dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E+18, 2.120303154043E+18, -5.412207605158E+18, -9.041927189809E+18,
           -7.975908114097E+18, -5.490808448662E+18, -4.365385529969E+18, -5.493373238953E+18,
           -7.975908114097E+18, -9.043881788286E+18, -5.458241224645E+18, 2.183159991917E+18,
           7.049456954407E+18],
          [2.120303154043E+18, -6.840828181977E+18, -8.291758075355E+18, -1.514915773345E+18,
           6.146617965847E+18, 1.016705772106E+19, 1.117191819513E+19, 1.019683374662E+19,
           6.146617965847E+18, -1.567520827568E+18, -8.211550352574E+18, -6.795064436020E+18,
           2.120303154043E+18],
          [-5.412207605158E+18, -8.291758075355E+18, 9.666098759062E+17, 1.115403574145E+19,
           1.180976383194E+19, 7.071918180842E+18, 4.552636512940E+18, 7.066851884792E+18,
           1.180976383194E+19, 1.094887240678E+19, 1.271424487456E+18, -8.489926255180E+18,
           -5.412207605158E+18],
          [-9.041927189809E+18, -1.514915773345E+18, 1.115403574145E+19, 9.058021229999E+18,
           -5.921575988005E+18, -1.697171170804E+19, -1.986325220649E+19, -1.708674849009E+19,
           -5.921575988005E+18, 9.071011939526E+18, 1.130155601111E+19, -1.774015779353E+18,
           -9.041927189809E+18],
          [-7.975908114097E+18, 6.146617965847E+18, 1.180976383194E+19, -5.921575988005E+18,
           -2.145503300375E+19, -1.373726429185E+19, -4.677155115289E+18, -1.400744853311E+19,
           -2.145503300375E+19, -5.608764908063E+18, 1.171252709523E+19, 5.937941570370E+18,
           -7.975908114097E+18],
          [-5.490808448662E+18, 1.016705772106E+19, 7.071918180842E+18, -1.697171170804E+19,
           -1.373726429185E+19, 3.180255899274E+19, 6.051776838087E+19, 3.123175894536E+19,
           -1.373726429185E+19, -1.640188016429E+19, 6.586737199675E+18, 1.009003074435E+19,
           -5.490808448662E+18],
          [-4.365385529969E+18, 1.117191819513E+19, 4.552636512940E+18, -1.986325220649E+19,
           -4.677155115289E+18, 6.051776838087E+19, 9.899192445544E+19, 5.987956201910E+19,
           -4.677155115289E+18, -1.913782291956E+19, 3.931492909576E+18, 1.117308113990E+19,
           -4.365385529968E+18],
          [-5.493373238953E+18, 1.019683374662E+19, 7.066851884792E+18, -1.708674849009E+19,
           -1.400744853311E+19, 3.123175894536E+19, 5.987956201910E+19, 3.081035634172E+19,
           -1.400744853311E+19, -1.646011917477E+19, 6.660176847751E+18, 1.011817145737E+19,
           -5.493373238953E+18],
          [-7.975908114097E+18, 6.146617965847E+18, 1.180976383194E+19, -5.921575988005E+18,
           -2.145503300375E+19, -1.373726429185E+19, -4.677155115289E+18, -1.400744853311E+19,
           -2.145503300375E+19, -5.608764908063E+18, 1.171252709523E+19, 5.937941570370E+18,
           -7.975908114097E+18],
          [-9.043881788286E+18, -1.567520827568E+18, 1.094887240678E+19, 9.071011939526E+18,
           -5.608764908063E+18, -1.640188016429E+19, -1.913782291956E+19, -1.646011917477E+19,
           -5.608764908063E+18, 9.061889238735E+18, 1.119056766335E+19, -1.844151449703E+18,
           -9.043881788286E+18],
          [-5.458241224645E+18, -8.211550352574E+18, 1.271424487456E+18, 1.130155601111E+19,
           1.171252709523E+19, 6.586737199675E+18, 3.931492909576E+18, 6.660176847751E+18,
           1.171252709523E+19, 1.119056766335E+19, 1.568973617012E+18, -8.405046373540E+18,
           -5.458241224645E+18],
          [2.183159991917E+18, -6.795064436020E+18, -8.489926255180E+18, -1.774015779353E+18,
           5.937941570370E+18, 1.009003074435E+19, 1.117308113990E+19, 1.011817145737E+19,
           5.937941570370E+18, -1.844151449703E+18, -8.405046373540E+18, -6.845929971507E+18,
           2.183159991917E+18],
          [7.049456954407E+18, 2.120303154043E+18, -5.412207605158E+18, -9.041927189809E+18,
           -7.975908114097E+18, -5.490808448662E+18, -4.365385529968E+18, -5.493373238953E+18,
           -7.975908114097E+18, -9.043881788286E+18, -5.458241224645E+18, 2.183159991917E+18,
           7.049456954407E+18]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          2.120303154043E+18, -5.412207605158E+18, -9.041927189809E+18, -7.975908114097E+18,
          -5.490808448662E+18, -4.365385529969E+18, -5.493373238953E+18, -7.975908114097E+18,
          -9.043881788286E+18, -5.458241224645E+18, 2.183159991917E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 2.120303154043E+18,
          -5.412207605158E+18, -9.041927189809E+18, -7.975908114097E+18, -5.490808448662E+18,
          -4.365385529969E+18, -5.493373238953E+18, -7.975908114097E+18, -9.043881788286E+18,
          -5.458241224645E+18, 2.183159991917E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 2.120303154043E+18, -5.412207605158E+18,
          -9.041927189809E+18, -7.975908114097E+18, -5.490808448662E+18, -4.365385529969E+18,
          -5.493373238953E+18, -7.975908114097E+18, -9.043881788286E+18, -5.458241224645E+18,
          2.183159991917E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 2.120303154043E+18,
          2.120303154043E+18, 2.120303154043E+18, 2.120303154043E+18, 2.120303154043E+18,
          2.120303154043E+18, -5.412207605158E+18, -5.412207605158E+18, -5.412207605158E+18,
          -5.412207605158E+18, -5.412207605158E+18, -5.412207605158E+18, -9.041927189809E+18,
          -9.041927189809E+18, -9.041927189809E+18, -9.041927189809E+18, -9.041927189809E+18,
          -9.041927189809E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -5.490808448662E+18,
          -5.490808448662E+18, -5.490808448662E+18, -5.490808448662E+18, -5.490808448662E+18,
          -5.490808448662E+18, -4.365385529969E+18, -4.365385529969E+18, -4.365385529969E+18,
          -4.365385529968E+18, -4.365385529968E+18, -4.365385529968E+18, -5.493373238953E+18,
          -5.493373238953E+18, -5.493373238953E+18, -5.493373238953E+18, -5.493373238953E+18,
          -5.493373238953E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -9.043881788286E+18,
          -9.043881788286E+18, -9.043881788286E+18, -9.043881788286E+18, -9.043881788286E+18,
          -9.043881788286E+18, -5.458241224645E+18, -5.458241224645E+18, -5.458241224645E+18,
          -5.458241224645E+18, -5.458241224645E+18, -5.458241224645E+18, 2.183159991917E+18,
          2.183159991917E+18, 2.183159991917E+18, 2.183159991917E+18, 2.183159991917E+18,
          2.183159991917E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 2.120303154043E+18,
          -5.412207605158E+18, -9.041927189809E+18, -7.975908114097E+18, -5.490808448662E+18,
          -4.365385529968E+18, -5.493373238953E+18, -7.975908114097E+18, -9.043881788286E+18,
          -5.458241224645E+18, 2.183159991917E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 2.120303154043E+18, -5.412207605158E+18,
          -9.041927189809E+18, -7.975908114097E+18, -5.490808448662E+18, -4.365385529968E+18,
          -5.493373238953E+18, -7.975908114097E+18, -9.043881788286E+18, -5.458241224645E+18,
          2.183159991917E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 2.120303154043E+18, -5.412207605158E+18, -9.041927189809E+18,
          -7.975908114097E+18, -5.490808448662E+18, -4.365385529968E+18, -5.493373238953E+18,
          -7.975908114097E+18, -9.043881788286E+18, -5.458241224645E+18, 2.183159991917E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E+21, -7.823082017163E+20, -3.700949828231E+20, 4.211823607006E+19,
          1.132303125654E+20, 7.609499224352E+19, -4.084169620081E+19, -1.443977450203E+20,
          -1.920403690376E+20, -2.047040864433E+20, -1.948441248529E+20, -1.443977450203E+20,
          -3.911849985106E+19, 7.440065969941E+19, 1.211163300265E+20, 4.211823607006E+19,
          -2.906695129325E+20, -6.234572619351E+20, -9.562450109377E+20, -7.823082017163E+20,
          -5.113959201337E+20, -2.404836385511E+20, 3.042864303151E+19, 7.619364276159E+19,
          4.892592562729E+19, -3.024177319714E+19, -9.892379938492E+19, -1.298571821746E+20,
          -1.379245194722E+20, -1.317272076482E+20, -9.892379938492E+19, -2.909362716347E+19,
          4.778102605806E+19, 8.147194001498E+19, 3.042864303151E+19, -1.853476437714E+20,
          -4.011239305743E+20, -6.169002173772E+20, -3.700949828231E+20, -2.404836385511E+20,
          -1.108722942791E+20, 1.873904999296E+19, 3.915697295782E+19, 2.175685901107E+19,
          -1.964185019348E+19, -5.344985374951E+19, -6.767399531163E+19, -7.114495250108E+19,
          -6.861029044360E+19, -5.344985374951E+19, -1.906875447588E+19, 2.116139241671E+19,
          4.182755000345E+19, 1.873904999296E+19, -8.002577461027E+19, -1.787905992135E+20,
          -2.775554238167E+20, 4.211823607006E+19, 3.042864303151E+19, 1.873904999296E+19,
          2.529609455086E+19, 4.354273214732E+19, 6.178936974377E+19, 1.132303125654E+20,
          7.619364276159E+19, 3.915697295782E+19, 3.717844299417E+19, 7.223658283430E+19,
          1.072947226744E+20, 7.609499224352E+19, 4.892592562729E+19, 2.175685901107E+19,
          9.604055355141E+18, 2.462031831544E+19, 3.963658127574E+19, -4.084169620081E+19,
          -3.024177319714E+19, -1.964185019348E+19, -3.537002512839E+19, -6.169812306696E+19,
          -8.802622100554E+19, -1.443977450203E+20, -9.892379938492E+19, -5.344985374951E+19,
          -6.312064964548E+19, -1.182653911769E+20, -1.734101327083E+20, -1.920403690376E+20,
          -1.298571821746E+20, -6.767399531163E+19, -6.947802448856E+19, -1.334652405285E+20,
          -1.974524565683E+20, -2.047040864433E+20, -1.379245194722E+20, -7.114495250108E+19,
          -6.952552096349E+19, -1.346856563970E+20, -1.998457918305E+20, -1.948441248529E+20,
          -1.317272076482E+20, -6.861029044360E+19, -7.014023360164E+19, -1.347870939643E+20,
          -1.994339543270E+20, -1.443977450203E+20, -9.892379938492E+19, -5.344985374951E+19,
          -6.312064964548E+19, -1.182653911769E+20, -1.734101327083E+20, -3.911849985106E+19,
          -2.909362716347E+19, -1.906875447588E+19, -3.422173851071E+19, -5.939959523313E+19,
          -8.457745195555E+19, 7.440065969941E+19, 4.778102605806E+19, 2.116139241671E+19,
          8.886095091330E+18, 2.323043140730E+19, 3.757476772328E+19, 1.211163300265E+20,
          8.147194001498E+19, 4.182755000345E+19, 4.013604681385E+19, 7.808893363578E+19,
          1.160418204577E+20, 4.211823607006E+19, 3.042864303151E+19, 1.873904999296E+19,
          2.529609455086E+19, 4.354273214732E+19, 6.178936974378E+19, -2.906695129325E+20,
          -1.853476437714E+20, -8.002577461027E+19, 2.529609455086E+19, 3.717844299417E+19,
          9.604055355141E+18, -3.537002512839E+19, -6.312064964548E+19, -6.947802448856E+19,
          -6.952552096349E+19, -7.014023360164E+19, -6.312064964548E+19, -3.422173851071E+19,
          8.886095091330E+18, 4.013604681385E+19, 2.529609455086E+19, -4.160997529959E+19,
          -1.085160451500E+20, -1.754221150005E+20, -6.234572619351E+20, -4.011239305743E+20,
          -1.787905992135E+20, 4.354273214732E+19, 7.223658283430E+19, 2.462031831544E+19,
          -6.169812306696E+19, -1.182653911769E+20, -1.334652405285E+20, -1.346856563970E+20,
          -1.347870939643E+20, -1.182653911769E+20, -5.939959523313E+19, 2.323043140730E+19,
          7.808893363578E+19, 4.354273214732E+19, -1.085160451500E+20, -2.605748224474E+20,
          -4.126335997448E+20, -9.562450109377E+20, -6.169002173772E+20, -2.775554238167E+20,
          6.178936974377E+19, 1.072947226744E+20, 3.963658127574E+19, -8.802622100554E+19,
          -1.734101327083E+20, -1.974524565683E+20, -1.998457918305E+20, -1.994339543270E+20,
          -1.734101327083E+20, -8.457745195555E+19, 3.757476772328E+19, 1.160418204577E+20,
          6.178936974378E+19, -1.754221150005E+20, -4.126335997448E+20, -6.498450844890E+20], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E+18, 1.842034121745E+18, -5.087139573635E+18, -8.959598399643E+18,
           -7.975908114097E+18, -5.611431437391E+18, -4.533211702999E+18, -5.515839683546E+18,
           -7.975908114097E+18, -8.616050256073E+18, -5.176527305501E+18, 2.120208105909E+18,
           7.049456954407E+18],
          [1.842034121745E+18, -5.501487366574E+18, -6.636899571863E+18, -9.352949284003E+17,
           5.479652055316E+18, 8.288520629779E+18, 9.087389620211E+18, 8.481401678700E+18,
           5.479652055316E+18, -9.965035969377E+17, -6.501360534968E+18, -6.034270619075E+18,
           1.842034121745E+18],
          [-5.087139573635E+18, -6.636899571863E+18, 8.941006971131E+17, 9.749711840097E+18,
           1.005403041504E+19, 5.814032225198E+18, 3.715829475115E+18, 5.664347521163E+18,
           1.005403041504E+19, 9.024081028879E+18, 1.232866890135E+18, -7.665862183419E+18,
           -5.087139573635E+18],
          [-8.959598399643E+18, -9.352949284003E+17, 9.749711840097E+18, 8.616757171890E+18,
           -5.827038360596E+18, -1.565688797731E+19, -1.870933237441E+19, -1.633466311759E+19,
           -5.827038360596E+18, 7.761129976189E+18, 1.003666584284E+19, -1.620874951248E+18,
           -8.959598399643E+18],
          [-7.975908114097E+18, 5.479652055316E+18, 1.005403041504E+19, -5.827038360596E+18,
           -2.145503300375E+19, -1.347188774155E+19, -6.110632370193E+18, -1.391995046985E+19,
           -2.145503300375E+19, -5.308241785637E+18, 9.886516164000E+18, 5.839207114976E+18,
           -7.975908114097E+18],
          [-5.611431437391E+18, 8.288520629779E+18, 5.814032225198E+18, -1.565688797731E+19,
           -1.347188774155E+19, 2.929381267349E+19, 5.411436279008E+19, 2.985138651995E+19,
           -1.347188774155E+19, -1.310314141567E+19, 5.103792541956E+18, 9.518011394833E+18,
           -5.611431437391E+18],
          [-4.533211702999E+18, 9.087389620211E+18, 3.715829475115E+18, -1.870933237441E+19,
           -6.110632370193E+18, 5.411436279008E+19, 8.749637474039E+19, 5.521553247888E+19,
           -6.110632370193E+18, -1.519154612992E+19, 2.761333800232E+18, 1.065911481816E+19,
           -4.533211702999E+18],
          [-5.515839683546E+18, 8.481401678700E+18, 5.664347521163E+18, -1.633466311759E+19,
           -1.391995046985E+19, 2.985138651995E+19, 5.521553247888E+19, 3.043061053033E+19,
           -1.391995046985E+19, -1.371046296717E+19, 4.932822149057E+18, 9.754728972929E+18,
           -5.515839683546E+18],
          [-7.975908114097E+18, 5.479652055316E+18, 1.005403041504E+19, -5.827038360596E+18,
           -2.145503300375E+19, -1.347188774155E+19, -6.110632370193E+18, -1.391995046985E+19,
           -2.145503300375E+19, -5.308241785637E+18, 9.886516164000E+18, 5.839207114976E+18,
           -7.975908114097E+18],
          [-8.616050256073E+18, -9.965035969377E+17, 9.024081028879E+18, 7.761129976189E+18,
           -5.308241785637E+18, -1.310314141567E+19, -1.519154612992E+19, -1.371046296717E+19,
           -5.308241785637E+18, 7.026025796278E+18, 9.281016513370E+18, -1.638123816652E+18,
           -8.616050256073E+18],
          [-5.176527305501E+18, -6.501360534968E+18, 1.232866890135E+18, 1.003666584284E+19,
           9.886516164000E+18, 5.103792541956E+18, 2.761333800232E+18, 4.932822149057E+18,
           9.886516164000E+18, 9.281016513370E+18, 1.577734427290E+18, -7.534846790666E+18,
           -5.176527305501E+18],
          [2.120208105909E+18, -6.034270619075E+18, -7.665862183419E+18, -1.620874951248E+18,
           5.839207114976E+18, 9.518011394833E+18, 1.065911481816E+19, 9.754728972929E+18,
           5.839207114976E+18, -1.638123816652E+18, -7.534846790666E+18, -6.579449237655E+18,
           2.120208105909E+18],
          [7.049456954407E+18, 1.842034121745E+18, -5.087139573635E+18, -8.959598399643E+18,
           -7.975908114097E+18, -5.611431437391E+18, -4.533211702999E+18, -5.515839683546E+18,
           -7.975908114097E+18, -8.616050256073E+18, -5.176527305501E+18, 2.120208105909E+18,
           7.049456954407E+18]], dtype=np.float64
        )
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          1.842034121745E+18, -5.087139573635E+18, -8.959598399643E+18, -7.975908114097E+18,
          -5.611431437391E+18, -4.533211702999E+18, -5.515839683546E+18, -7.975908114097E+18,
          -8.616050256073E+18, -5.176527305501E+18, 2.120208105909E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.842034121745E+18,
          -5.087139573635E+18, -8.959598399643E+18, -7.975908114097E+18, -5.611431437391E+18,
          -4.533211702999E+18, -5.515839683546E+18, -7.975908114097E+18, -8.616050256073E+18,
          -5.176527305501E+18, 2.120208105909E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 1.842034121745E+18, -5.087139573635E+18,
          -8.959598399643E+18, -7.975908114097E+18, -5.611431437391E+18, -4.533211702999E+18,
          -5.515839683546E+18, -7.975908114097E+18, -8.616050256073E+18, -5.176527305501E+18,
          2.120208105909E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.842034121745E+18,
          1.842034121745E+18, 1.842034121745E+18, 1.842034121745E+18, 1.842034121745E+18,
          1.842034121745E+18, -5.087139573635E+18, -5.087139573635E+18, -5.087139573635E+18,
          -5.087139573635E+18, -5.087139573635E+18, -5.087139573635E+18, -8.959598399643E+18,
          -8.959598399643E+18, -8.959598399643E+18, -8.959598399643E+18, -8.959598399643E+18,
          -8.959598399643E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -5.611431437391E+18,
          -5.611431437391E+18, -5.611431437391E+18, -5.611431437391E+18, -5.611431437391E+18,
          -5.611431437391E+18, -4.533211702999E+18, -4.533211702999E+18, -4.533211702999E+18,
          -4.533211702999E+18, -4.533211702999E+18, -4.533211702999E+18, -5.515839683546E+18,
          -5.515839683546E+18, -5.515839683546E+18, -5.515839683546E+18, -5.515839683546E+18,
          -5.515839683546E+18, -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18,
          -7.975908114097E+18, -7.975908114097E+18, -7.975908114097E+18, -8.616050256073E+18,
          -8.616050256073E+18, -8.616050256073E+18, -8.616050256073E+18, -8.616050256073E+18,
          -8.616050256073E+18, -5.176527305501E+18, -5.176527305501E+18, -5.176527305501E+18,
          -5.176527305501E+18, -5.176527305501E+18, -5.176527305501E+18, 2.120208105909E+18,
          2.120208105909E+18, 2.120208105909E+18, 2.120208105909E+18, 2.120208105909E+18,
          2.120208105909E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 1.842034121745E+18,
          -5.087139573635E+18, -8.959598399643E+18, -7.975908114097E+18, -5.611431437391E+18,
          -4.533211702999E+18, -5.515839683546E+18, -7.975908114097E+18, -8.616050256073E+18,
          -5.176527305501E+18, 2.120208105909E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 1.842034121745E+18, -5.087139573635E+18,
          -8.959598399643E+18, -7.975908114097E+18, -5.611431437391E+18, -4.533211702999E+18,
          -5.515839683546E+18, -7.975908114097E+18, -8.616050256073E+18, -5.176527305501E+18,
          2.120208105909E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18,
          7.049456954407E+18, 1.842034121745E+18, -5.087139573635E+18, -8.959598399643E+18,
          -7.975908114097E+18, -5.611431437391E+18, -4.533211702999E+18, -5.515839683546E+18,
          -7.975908114097E+18, -8.616050256073E+18, -5.176527305501E+18, 2.120208105909E+18,
          7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18, 7.049456954407E+18], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E+21, -7.823082017163E+20, -3.700949828231E+20, 4.211823607006E+19,
          1.076476993194E+20, 7.357570465096E+19, -4.016962502163E+19, -1.443977450203E+20,
          -1.903018294300E+20, -2.039483993008E+20, -1.933304341125E+20, -1.443977450203E+20,
          -3.807888037972E+19, 7.140500088303E+19, 1.169244258887E+20, 4.211823607006E+19,
          -2.906695129325E+20, -6.234572619351E+20, -9.562450109377E+20, -7.823082017163E+20,
          -5.113959201337E+20, -2.404836385511E+20, 3.042864303151E+19, 7.237914425352E+19,
          4.735475657609E+19, -2.976628281430E+19, -9.892379938492E+19, -1.287383634324E+20,
          -1.374766701016E+20, -1.307255693029E+20, -9.892379938492E+19, -2.825793700517E+19,
          4.587782482018E+19, 7.865635329442E+19, 3.042864303151E+19, -1.853476437714E+20,
          -4.011239305743E+20, -6.169002173772E+20, -3.700949828231E+20, -2.404836385511E+20,
          -1.108722942791E+20, 1.873904999296E+19, 3.711058918763E+19, 2.113380850123E+19,
          -1.936294060697E+19, -5.344985374951E+19, -6.717489743492E+19, -7.100494090228E+19,
          -6.812070449321E+19, -5.344985374951E+19, -1.843699363062E+19, 2.035064875734E+19,
          4.038828070016E+19, 1.873904999296E+19, -8.002577461027E+19, -1.787905992135E+20,
          -2.775554238167E+20, 4.211823607006E+19, 3.042864303151E+19, 1.873904999296E+19,
          2.529609455086E+19, 4.354273214732E+19, 6.178936974377E+19, 1.076476993194E+20,
          7.237914425352E+19, 3.711058918763E+19, 3.500226650129E+19, 6.816249888084E+19,
          1.013227312604E+20, 7.357570465096E+19, 4.735475657609E+19, 2.113380850123E+19,
          9.287866896806E+18, 2.366287336725E+19, 3.803787983769E+19, -4.016962502163E+19,
          -2.976628281430E+19, -1.936294060697E+19, -3.477394497976E+19, -6.058829155988E+19,
          -8.640263814000E+19, -1.443977450203E+20, -9.892379938492E+19, -5.344985374951E+19,
          -6.312064964548E+19, -1.182653911769E+20, -1.734101327083E+20, -1.903018294300E+20,
          -1.287383634324E+20, -6.717489743492E+19, -6.918455006928E+19, -1.327576687012E+20,
          -1.963307873331E+20, -2.039483993008E+20, -1.374766701016E+20, -7.100494090228E+19,
          -6.964490057419E+19, -1.347565894454E+20, -1.998682783166E+20, -1.933304341125E+20,
          -1.307255693029E+20, -6.812070449321E+19, -6.980635883397E+19, -1.340968779844E+20,
          -1.983873971348E+20, -1.443977450203E+20, -9.892379938492E+19, -5.344985374951E+19,
          -6.312064964548E+19, -1.182653911769E+20, -1.734101327083E+20, -3.807888037972E+19,
          -2.825793700517E+19, -1.843699363062E+19, -3.317879124813E+19, -5.774153224019E+19,
          -8.230427323224E+19, 7.140500088303E+19, 4.587782482018E+19, 2.035064875734E+19,
          8.346507461054E+18, 2.186954222761E+19, 3.539257699416E+19, 1.169244258887E+20,
          7.865635329442E+19, 4.038828070016E+19, 3.857421048676E+19, 7.502821286761E+19,
          1.114822152485E+20, 4.211823607006E+19, 3.042864303151E+19, 1.873904999296E+19,
          2.529609455086E+19, 4.354273214732E+19, 6.178936974377E+19, -2.906695129325E+20,
          -1.853476437714E+20, -8.002577461027E+19, 2.529609455086E+19, 3.500226650129E+19,
          9.287866896806E+18, -3.477394497976E+19, -6.312064964548E+19, -6.918455006928E+19,
          -6.964490057419E+19, -6.980635883397E+19, -6.312064964548E+19, -3.317879124813E+19,
          8.346507461054E+18, 3.857421048676E+19, 2.529609455086E+19, -4.160997529959E+19,
          -1.085160451500E+20, -1.754221150005E+20, -6.234572619351E+20, -4.011239305743E+20,
          -1.787905992135E+20, 4.354273214732E+19, 6.816249888084E+19, 2.366287336725E+19,
          -6.058829155988E+19, -1.182653911769E+20, -1.327576687012E+20, -1.347565894454E+20,
          -1.340968779844E+20, -1.182653911769E+20, -5.774153224019E+19, 2.186954222761E+19,
          7.502821286761E+19, 4.354273214732E+19, -1.085160451500E+20, -2.605748224474E+20,
          -4.126335997448E+20, -9.562450109377E+20, -6.169002173772E+20, -2.775554238167E+20,
          6.178936974377E+19, 1.013227312604E+20, 3.803787983769E+19, -8.640263814000E+19,
          -1.734101327083E+20, -1.963307873331E+20, -1.998682783166E+20, -1.983873971348E+20,
          -1.734101327083E+20, -8.230427323224E+19, 3.539257699416E+19, 1.114822152485E+20,
          6.178936974377E+19, -1.754221150005E+20, -4.126335997448E+20, -6.498450844891E+20], dtype=np.float64
        )


class TestInterpolatorLoadSmallValuesUneven(TestInterpolatorLoadValues):
    """
    Loading small values (10^-20 times the original) instead of the original np.sin(x)

    These data are saved to 12 significant figures.
    """
    def __init__(self):
        super().__init__()
        #: data array from a function sampled on self.x. dtype should be np.float64
        # self.data: np.array = np.sin(self.x)
        self.data: np.array = np.array(
         [[7.049456954407E-22, 5.750613283457E-22, -5.031133752816E-23, -3.957902354963E-22,
           -8.474851229653E-22, -9.121180789640E-22, -7.975908114097E-22, -6.835527608142E-22,
           -4.876973734940E-22, -4.418624359019E-22, -4.876973734940E-22, -5.728794999748E-22,
           -7.975908114097E-22, -8.851381966774E-22, -8.474851229653E-22, -6.734331948907E-22,
           -5.031133752816E-23, 2.994648599639E-22, 7.049456954407E-22],
          [5.750613283457E-22, 2.707313596327E-22, -4.947098908050E-22, -7.716977009236E-22,
           -8.851381966774E-22, -7.454434661366E-22, -2.923247487940E-22, -6.867195016855E-23,
           2.378357578524E-22, 3.010595454905E-22, 2.378357578524E-22, 1.127556015714E-22,
           -2.923247487940E-22, -5.313921590225E-22, -8.851381966774E-22, -9.034807299146E-22,
           -4.947098908050E-22, -1.193366977476E-22, 5.750613283457E-22],
          [-5.031133752816E-23, -4.947098908050E-22, -9.121921863446E-22, -7.727418937786E-22,
           -9.251264987298E-24, 4.278478892878E-22, 1.052139178127E-21, 1.201126954831E-21,
           1.283205555674E-21, 1.282705054889E-21, 1.283205555674E-21, 1.266409993718E-21,
           1.052139178127E-21, 7.942173990430E-22, -9.251264987298E-24, -4.418624359019E-22,
           -9.121921863446E-22, -8.127030064783E-22, -5.031133752816E-23],
          [-3.957902354963E-22, -7.716977009236E-22, -7.727418937786E-22, -3.939635588637E-22,
           6.155045982467E-22, 1.005907311298E-21, 1.282705054889E-21, 1.215171020957E-21,
           9.674890958243E-22, 8.947967120127E-22, 9.674890958243E-22, 1.089467585825E-21,
           1.282705054889E-21, 1.227830098222E-21, 6.155045982467E-22, 1.127556015714E-22,
           -7.727418937786E-22, -9.132515608025E-22, -3.957902354963E-22],
          [-8.474851229653E-22, -8.851381966774E-22, -9.251264987298E-24, 6.155045982467E-22,
           1.283205555674E-21, 1.138884597353E-21, 1.734970013481E-22, -3.806455343631E-22,
           -1.140407180451E-21, -1.289967784984E-21, -1.140407180451E-21, -8.347707818621E-22,
           1.734970013481E-22, 7.259972187812E-22, 1.283205555674E-21, 1.095087852132E-21,
           -9.251264987298E-24, -5.728794999748E-22, -8.474851229653E-22],
          [-9.121180789640E-22, -7.454434661366E-22, 4.278478892878E-22, 1.005907311298E-21,
           1.138884597353E-21, 6.300331483592E-22, -8.347707818621E-22, -1.435151388671E-21,
           -2.022780434934E-21, -2.095142217650E-21, -2.022780434934E-21, -1.824711128590E-21,
           -8.347707818621E-22, -9.351916346701E-23, 1.138884597353E-21, 1.277678821031E-21,
           4.278478892878E-22, -2.387837583277E-22, -9.121180789640E-22],
          [-7.975908114097E-22, -2.923247487940E-22, 1.052139178127E-21, 1.282705054889E-21,
           1.734970013481E-22, -8.347707818621E-22, -2.145503300375E-21, -2.052264171648E-21,
           -9.241435356589E-22, -5.067031374727E-22, -9.241435356589E-22, -1.547945550760E-21,
           -2.145503300375E-21, -1.704692811750E-21, 1.734970013480E-22, 9.674890958243E-22,
           1.052139178127E-21, 4.278478892878E-22, -7.975908114097E-22],
          [-6.835527608142E-22, -6.867195016855E-23, 1.201126954831E-21, 1.215171020957E-21,
           -3.806455343631E-22, -1.435151388671E-21, -2.052264171648E-21, -1.268933271435E-21,
           1.257727427008E-21, 2.044658799484E-21, 1.257727427008E-21, -9.251369377679E-24,
           -2.052264171648E-21, -2.095142217650E-21, -3.806455343631E-22, 6.300331483592E-22,
           1.201126954831E-21, 6.764157650466E-22, -6.835527608142E-22],
          [-4.876973734940E-22, 2.378357578524E-22, 1.283205555674E-21, 9.674890958243E-22,
           -1.140407180451E-21, -2.022780434934E-21, -9.241435356589E-22, 1.257727427008E-21,
           6.446759109720E-21, 7.917136926885E-21, 6.446759109720E-21, 3.973381423908E-21,
           -9.241435356589E-22, -2.052264171648E-21, -1.140407180451E-21, 4.282861041084E-23,
           1.283205555674E-21, 9.566990623455E-22, -4.876973734940E-22],
          [-4.418624359019E-22, 3.010595454905E-22, 1.282705054889E-21, 8.947967120127E-22,
           -1.289967784984E-21, -2.095142217650E-21, -5.067031374727E-22, 2.044658799484E-21,
           7.917136926885E-21, 9.560513212622E-21, 7.917136926885E-21, 5.136281978234E-21,
           -5.067031374727E-22, -1.933844713753E-21, -1.289967784984E-21, -9.351916346701E-23,
           1.282705054889E-21, 1.005907311298E-21, -4.418624359019E-22],
          [-4.876973734940E-22, 2.378357578524E-22, 1.283205555674E-21, 9.674890958243E-22,
           -1.140407180451E-21, -2.022780434934E-21, -9.241435356589E-22, 1.257727427008E-21,
           6.446759109720E-21, 7.917136926885E-21, 6.446759109720E-21, 3.973381423908E-21,
           -9.241435356589E-22, -2.052264171648E-21, -1.140407180451E-21, 4.282861041084E-23,
           1.283205555674E-21, 9.566990623455E-22, -4.876973734940E-22],
          [-5.728794999748E-22, 1.127556015714E-22, 1.266409993718E-21, 1.089467585825E-21,
           -8.347707818621E-22, -1.824711128590E-21, -1.547945550760E-21, -9.251369377679E-24,
           3.973381423908E-21, 5.136281978234E-21, 3.973381423908E-21, 2.044658799484E-21,
           -1.547945550760E-21, -2.165899038868E-21, -8.347707818621E-22, 2.979548967146E-22,
           1.266409993718E-21, 8.505592925358E-22, -5.728794999748E-22],
          [-7.975908114097E-22, -2.923247487940E-22, 1.052139178127E-21, 1.282705054889E-21,
           1.734970013481E-22, -8.347707818621E-22, -2.145503300375E-21, -2.052264171648E-21,
           -9.241435356589E-22, -5.067031374727E-22, -9.241435356589E-22, -1.547945550760E-21,
           -2.145503300375E-21, -1.704692811750E-21, 1.734970013480E-22, 9.674890958243E-22,
           1.052139178127E-21, 4.278478892878E-22, -7.975908114097E-22],
          [-8.851381966774E-22, -5.313921590225E-22, 7.942173990430E-22, 1.227830098222E-21,
           7.259972187812E-22, -9.351916346701E-23, -1.704692811750E-21, -2.095142217650E-21,
           -2.052264171648E-21, -1.933844713753E-21, -2.052264171648E-21, -2.165899038868E-21,
           -1.704692811750E-21, -9.881655954565E-22, 7.259972187812E-22, 1.215171020957E-21,
           7.942173990430E-22, 1.127556015714E-22, -8.851381966774E-22],
          [-8.474851229653E-22, -8.851381966774E-22, -9.251264987298E-24, 6.155045982467E-22,
           1.283205555674E-21, 1.138884597353E-21, 1.734970013480E-22, -3.806455343631E-22,
           -1.140407180451E-21, -1.289967784984E-21, -1.140407180451E-21, -8.347707818621E-22,
           1.734970013480E-22, 7.259972187812E-22, 1.283205555674E-21, 1.095087852132E-21,
           -9.251264987296E-24, -5.728794999748E-22, -8.474851229653E-22],
          [-6.734331948907E-22, -9.034807299146E-22, -4.418624359019E-22, 1.127556015714E-22,
           1.095087852132E-21, 1.277678821031E-21, 9.674890958243E-22, 6.300331483592E-22,
           4.282861041084E-23, -9.351916346701E-23, 4.282861041084E-23, 2.979548967146E-22,
           9.674890958243E-22, 1.215171020957E-21, 1.095087852132E-21, 6.764157650466E-22,
           -4.418624359019E-22, -8.199851501756E-22, -6.734331948907E-22],
          [-5.031133752816E-23, -4.947098908050E-22, -9.121921863446E-22, -7.727418937786E-22,
           -9.251264987298E-24, 4.278478892878E-22, 1.052139178127E-21, 1.201126954831E-21,
           1.283205555674E-21, 1.282705054889E-21, 1.283205555674E-21, 1.266409993718E-21,
           1.052139178127E-21, 7.942173990430E-22, -9.251264987296E-24, -4.418624359019E-22,
           -9.121921863446E-22, -8.127030064783E-22, -5.031133752816E-23],
          [2.994648599639E-22, -1.193366977476E-22, -8.127030064783E-22, -9.132515608025E-22,
           -5.728794999748E-22, -2.387837583277E-22, 4.278478892878E-22, 6.764157650466E-22,
           9.566990623455E-22, 1.005907311298E-21, 9.566990623455E-22, 8.505592925358E-22,
           4.278478892878E-22, 1.127556015714E-22, -5.728794999748E-22, -8.199851501756E-22,
           -8.127030064783E-22, -5.265231727816E-22, 2.994648599639E-22],
          [7.049456954407E-22, 5.750613283457E-22, -5.031133752816E-23, -3.957902354963E-22,
           -8.474851229653E-22, -9.121180789640E-22, -7.975908114097E-22, -6.835527608142E-22,
           -4.876973734940E-22, -4.418624359019E-22, -4.876973734940E-22, -5.728794999748E-22,
           -7.975908114097E-22, -8.851381966774E-22, -8.474851229653E-22, -6.734331948907E-22,
           -5.031133752816E-23, 2.994648599639E-22, 7.049456954407E-22]], dtype=np.float64
        )

        #: array holding precalculated quadratic extrapolation data
        self.precalc_extrapolation_quadratic: np.array = np.array(
            [], dtype=np.float64
        )

    def setup_cubic(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E-22, 2.120303154043E-22, -5.412207605158E-22, -9.041927189809E-22,
           -7.975908114097E-22, -5.490808448662E-22, -4.365385529968E-22, -5.493373238953E-22,
           -7.975908114097E-22, -9.043881788286E-22, -5.458241224645E-22, 2.183159991917E-22,
           7.049456954407E-22],
          [2.120303154043E-22, -6.840828181977E-22, -8.291758075355E-22, -1.514915773345E-22,
           6.146617965847E-22, 1.016705772106E-21, 1.117191819513E-21, 1.019683374662E-21,
           6.146617965847E-22, -1.567520827568E-22, -8.211550352574E-22, -6.795064436020E-22,
           2.120303154043E-22],
          [-5.412207605158E-22, -8.291758075355E-22, 9.666098759062E-23, 1.115403574145E-21,
           1.180976383194E-21, 7.071918180842E-22, 4.552636512940E-22, 7.066851884792E-22,
           1.180976383194E-21, 1.094887240678E-21, 1.271424487456E-22, -8.489926255180E-22,
           -5.412207605158E-22],
          [-9.041927189809E-22, -1.514915773345E-22, 1.115403574145E-21, 9.058021229999E-22,
           -5.921575988005E-22, -1.697171170804E-21, -1.986325220649E-21, -1.708674849009E-21,
           -5.921575988005E-22, 9.071011939526E-22, 1.130155601111E-21, -1.774015779353E-22,
           -9.041927189809E-22],
          [-7.975908114097E-22, 6.146617965847E-22, 1.180976383194E-21, -5.921575988005E-22,
           -2.145503300375E-21, -1.373726429185E-21, -4.677155115289E-22, -1.400744853311E-21,
           -2.145503300375E-21, -5.608764908063E-22, 1.171252709523E-21, 5.937941570370E-22,
           -7.975908114097E-22],
          [-5.490808448662E-22, 1.016705772106E-21, 7.071918180842E-22, -1.697171170804E-21,
           -1.373726429185E-21, 3.180255899274E-21, 6.051776838087E-21, 3.123175894536E-21,
           -1.373726429185E-21, -1.640188016429E-21, 6.586737199675E-22, 1.009003074435E-21,
           -5.490808448662E-22],
          [-4.365385529968E-22, 1.117191819513E-21, 4.552636512940E-22, -1.986325220649E-21,
           -4.677155115289E-22, 6.051776838087E-21, 9.899192445544E-21, 5.987956201910E-21,
           -4.677155115289E-22, -1.913782291956E-21, 3.931492909576E-22, 1.117308113990E-21,
           -4.365385529968E-22],
          [-5.493373238953E-22, 1.019683374662E-21, 7.066851884792E-22, -1.708674849009E-21,
           -1.400744853311E-21, 3.123175894536E-21, 5.987956201910E-21, 3.081035634172E-21,
           -1.400744853311E-21, -1.646011917477E-21, 6.660176847751E-22, 1.011817145737E-21,
           -5.493373238953E-22],
          [-7.975908114097E-22, 6.146617965847E-22, 1.180976383194E-21, -5.921575988005E-22,
           -2.145503300375E-21, -1.373726429185E-21, -4.677155115289E-22, -1.400744853311E-21,
           -2.145503300375E-21, -5.608764908063E-22, 1.171252709523E-21, 5.937941570370E-22,
           -7.975908114097E-22],
          [-9.043881788286E-22, -1.567520827568E-22, 1.094887240678E-21, 9.071011939526E-22,
           -5.608764908063E-22, -1.640188016429E-21, -1.913782291956E-21, -1.646011917477E-21,
           -5.608764908063E-22, 9.061889238735E-22, 1.119056766335E-21, -1.844151449703E-22,
           -9.043881788286E-22],
          [-5.458241224645E-22, -8.211550352574E-22, 1.271424487456E-22, 1.130155601111E-21,
           1.171252709523E-21, 6.586737199675E-22, 3.931492909576E-22, 6.660176847751E-22,
           1.171252709523E-21, 1.119056766335E-21, 1.568973617012E-22, -8.405046373540E-22,
           -5.458241224645E-22],
          [2.183159991917E-22, -6.795064436020E-22, -8.489926255180E-22, -1.774015779353E-22,
           5.937941570370E-22, 1.009003074435E-21, 1.117308113990E-21, 1.011817145737E-21,
           5.937941570370E-22, -1.844151449703E-22, -8.405046373540E-22, -6.845929971507E-22,
           2.183159991917E-22],
          [7.049456954407E-22, 2.120303154043E-22, -5.412207605158E-22, -9.041927189809E-22,
           -7.975908114097E-22, -5.490808448662E-22, -4.365385529968E-22, -5.493373238953E-22,
           -7.975908114097E-22, -9.043881788286E-22, -5.458241224645E-22, 2.183159991917E-22,
           7.049456954407E-22]], dtype=np.float64
        )
        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          2.120303154043E-22, -5.412207605158E-22, -9.041927189809E-22, -7.975908114097E-22,
          -5.490808448662E-22, -4.365385529968E-22, -5.493373238953E-22, -7.975908114097E-22,
          -9.043881788286E-22, -5.458241224645E-22, 2.183159991917E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 2.120303154043E-22,
          -5.412207605158E-22, -9.041927189809E-22, -7.975908114097E-22, -5.490808448662E-22,
          -4.365385529968E-22, -5.493373238953E-22, -7.975908114097E-22, -9.043881788286E-22,
          -5.458241224645E-22, 2.183159991917E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 2.120303154043E-22, -5.412207605158E-22,
          -9.041927189809E-22, -7.975908114097E-22, -5.490808448662E-22, -4.365385529968E-22,
          -5.493373238953E-22, -7.975908114097E-22, -9.043881788286E-22, -5.458241224645E-22,
          2.183159991917E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 2.120303154043E-22,
          2.120303154043E-22, 2.120303154043E-22, 2.120303154043E-22, 2.120303154043E-22,
          2.120303154043E-22, -5.412207605158E-22, -5.412207605158E-22, -5.412207605158E-22,
          -5.412207605158E-22, -5.412207605158E-22, -5.412207605158E-22, -9.041927189809E-22,
          -9.041927189809E-22, -9.041927189809E-22, -9.041927189809E-22, -9.041927189809E-22,
          -9.041927189809E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -5.490808448662E-22,
          -5.490808448662E-22, -5.490808448662E-22, -5.490808448662E-22, -5.490808448662E-22,
          -5.490808448662E-22, -4.365385529968E-22, -4.365385529968E-22, -4.365385529968E-22,
          -4.365385529968E-22, -4.365385529968E-22, -4.365385529968E-22, -5.493373238953E-22,
          -5.493373238953E-22, -5.493373238953E-22, -5.493373238953E-22, -5.493373238953E-22,
          -5.493373238953E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -9.043881788286E-22,
          -9.043881788286E-22, -9.043881788286E-22, -9.043881788286E-22, -9.043881788286E-22,
          -9.043881788286E-22, -5.458241224645E-22, -5.458241224645E-22, -5.458241224645E-22,
          -5.458241224645E-22, -5.458241224645E-22, -5.458241224645E-22, 2.183159991917E-22,
          2.183159991917E-22, 2.183159991917E-22, 2.183159991917E-22, 2.183159991917E-22,
          2.183159991917E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 2.120303154043E-22,
          -5.412207605158E-22, -9.041927189809E-22, -7.975908114097E-22, -5.490808448662E-22,
          -4.365385529968E-22, -5.493373238953E-22, -7.975908114097E-22, -9.043881788286E-22,
          -5.458241224645E-22, 2.183159991917E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 2.120303154043E-22, -5.412207605158E-22,
          -9.041927189809E-22, -7.975908114097E-22, -5.490808448662E-22, -4.365385529968E-22,
          -5.493373238953E-22, -7.975908114097E-22, -9.043881788286E-22, -5.458241224645E-22,
          2.183159991917E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 2.120303154043E-22, -5.412207605158E-22, -9.041927189809E-22,
          -7.975908114097E-22, -5.490808448662E-22, -4.365385529968E-22, -5.493373238953E-22,
          -7.975908114097E-22, -9.043881788286E-22, -5.458241224645E-22, 2.183159991917E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22], dtype=np.float64
        )

        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E-19, -7.823082017163E-20, -3.700949828231E-20, 4.211823607006E-21,
          1.132303125654E-20, 7.609499224352E-21, -4.084169620081E-21, -1.443977450203E-20,
          -1.920403690376E-20, -2.047040864433E-20, -1.948441248529E-20, -1.443977450203E-20,
          -3.911849985106E-21, 7.440065969941E-21, 1.211163300265E-20, 4.211823607006E-21,
          -2.906695129325E-20, -6.234572619351E-20, -9.562450109377E-20, -7.823082017163E-20,
          -5.113959201337E-20, -2.404836385511E-20, 3.042864303151E-21, 7.619364276159E-21,
          4.892592562729E-21, -3.024177319714E-21, -9.892379938492E-21, -1.298571821746E-20,
          -1.379245194722E-20, -1.317272076482E-20, -9.892379938492E-21, -2.909362716347E-21,
          4.778102605806E-21, 8.147194001498E-21, 3.042864303151E-21, -1.853476437714E-20,
          -4.011239305743E-20, -6.169002173772E-20, -3.700949828231E-20, -2.404836385511E-20,
          -1.108722942791E-20, 1.873904999296E-21, 3.915697295782E-21, 2.175685901107E-21,
          -1.964185019348E-21, -5.344985374951E-21, -6.767399531163E-21, -7.114495250108E-21,
          -6.861029044360E-21, -5.344985374951E-21, -1.906875447588E-21, 2.116139241671E-21,
          4.182755000345E-21, 1.873904999296E-21, -8.002577461027E-21, -1.787905992135E-20,
          -2.775554238167E-20, 4.211823607006E-21, 3.042864303151E-21, 1.873904999296E-21,
          2.529609455086E-21, 4.354273214732E-21, 6.178936974377E-21, 1.132303125654E-20,
          7.619364276159E-21, 3.915697295782E-21, 3.717844299417E-21, 7.223658283430E-21,
          1.072947226744E-20, 7.609499224352E-21, 4.892592562729E-21, 2.175685901107E-21,
          9.604055355141E-22, 2.462031831544E-21, 3.963658127574E-21, -4.084169620081E-21,
          -3.024177319714E-21, -1.964185019348E-21, -3.537002512839E-21, -6.169812306696E-21,
          -8.802622100554E-21, -1.443977450203E-20, -9.892379938492E-21, -5.344985374951E-21,
          -6.312064964548E-21, -1.182653911769E-20, -1.734101327083E-20, -1.920403690376E-20,
          -1.298571821746E-20, -6.767399531163E-21, -6.947802448856E-21, -1.334652405285E-20,
          -1.974524565683E-20, -2.047040864433E-20, -1.379245194722E-20, -7.114495250108E-21,
          -6.952552096349E-21, -1.346856563970E-20, -1.998457918305E-20, -1.948441248529E-20,
          -1.317272076482E-20, -6.861029044360E-21, -7.014023360164E-21, -1.347870939643E-20,
          -1.994339543270E-20, -1.443977450203E-20, -9.892379938492E-21, -5.344985374951E-21,
          -6.312064964548E-21, -1.182653911769E-20, -1.734101327083E-20, -3.911849985106E-21,
          -2.909362716347E-21, -1.906875447588E-21, -3.422173851071E-21, -5.939959523313E-21,
          -8.457745195555E-21, 7.440065969941E-21, 4.778102605806E-21, 2.116139241671E-21,
          8.886095091330E-22, 2.323043140730E-21, 3.757476772328E-21, 1.211163300265E-20,
          8.147194001498E-21, 4.182755000345E-21, 4.013604681385E-21, 7.808893363578E-21,
          1.160418204577E-20, 4.211823607006E-21, 3.042864303151E-21, 1.873904999296E-21,
          2.529609455086E-21, 4.354273214732E-21, 6.178936974378E-21, -2.906695129325E-20,
          -1.853476437714E-20, -8.002577461027E-21, 2.529609455086E-21, 3.717844299417E-21,
          9.604055355141E-22, -3.537002512839E-21, -6.312064964548E-21, -6.947802448856E-21,
          -6.952552096349E-21, -7.014023360164E-21, -6.312064964548E-21, -3.422173851071E-21,
          8.886095091330E-22, 4.013604681385E-21, 2.529609455086E-21, -4.160997529959E-21,
          -1.085160451500E-20, -1.754221150005E-20, -6.234572619351E-20, -4.011239305743E-20,
          -1.787905992135E-20, 4.354273214732E-21, 7.223658283430E-21, 2.462031831544E-21,
          -6.169812306696E-21, -1.182653911769E-20, -1.334652405285E-20, -1.346856563970E-20,
          -1.347870939643E-20, -1.182653911769E-20, -5.939959523313E-21, 2.323043140730E-21,
          7.808893363578E-21, 4.354273214732E-21, -1.085160451500E-20, -2.605748224474E-20,
          -4.126335997448E-20, -9.562450109377E-20, -6.169002173772E-20, -2.775554238167E-20,
          6.178936974377E-21, 1.072947226744E-20, 3.963658127574E-21, -8.802622100554E-21,
          -1.734101327083E-20, -1.974524565683E-20, -1.998457918305E-20, -1.994339543270E-20,
          -1.734101327083E-20, -8.457745195555E-21, 3.757476772328E-21, 1.160418204577E-20,
          6.178936974378E-21, -1.754221150005E-20, -4.126335997448E-20, -6.498450844891E-20], dtype=np.float64
        )

    def setup_linear(self):
        self.precalc_interpolation = np.array(
         [[7.049456954407E-22, 1.842034121745E-22, -5.087139573635E-22, -8.959598399643E-22,
           -7.975908114097E-22, -5.611431437391E-22, -4.533211702999E-22, -5.515839683546E-22,
           -7.975908114097E-22, -8.616050256073E-22, -5.176527305501E-22, 2.120208105909E-22,
           7.049456954407E-22],
          [1.842034121745E-22, -5.501487366574E-22, -6.636899571863E-22, -9.352949284003E-23,
           5.479652055316E-22, 8.288520629779E-22, 9.087389620211E-22, 8.481401678700E-22,
           5.479652055316E-22, -9.965035969377E-23, -6.501360534968E-22, -6.034270619075E-22,
           1.842034121745E-22],
          [-5.087139573635E-22, -6.636899571863E-22, 8.941006971131E-23, 9.749711840097E-22,
           1.005403041504E-21, 5.814032225198E-22, 3.715829475115E-22, 5.664347521163E-22,
           1.005403041504E-21, 9.024081028879E-22, 1.232866890135E-22, -7.665862183419E-22,
           -5.087139573635E-22],
          [-8.959598399643E-22, -9.352949284003E-23, 9.749711840097E-22, 8.616757171890E-22,
           -5.827038360596E-22, -1.565688797731E-21, -1.870933237441E-21, -1.633466311759E-21,
           -5.827038360596E-22, 7.761129976189E-22, 1.003666584284E-21, -1.620874951248E-22,
           -8.959598399643E-22],
          [-7.975908114097E-22, 5.479652055316E-22, 1.005403041504E-21, -5.827038360596E-22,
           -2.145503300375E-21, -1.347188774155E-21, -6.110632370193E-22, -1.391995046985E-21,
           -2.145503300375E-21, -5.308241785637E-22, 9.886516164000E-22, 5.839207114976E-22,
           -7.975908114097E-22],
          [-5.611431437391E-22, 8.288520629779E-22, 5.814032225198E-22, -1.565688797731E-21,
           -1.347188774155E-21, 2.929381267349E-21, 5.411436279008E-21, 2.985138651995E-21,
           -1.347188774155E-21, -1.310314141567E-21, 5.103792541956E-22, 9.518011394833E-22,
           -5.611431437391E-22],
          [-4.533211702999E-22, 9.087389620211E-22, 3.715829475115E-22, -1.870933237441E-21,
           -6.110632370193E-22, 5.411436279008E-21, 8.749637474039E-21, 5.521553247888E-21,
           -6.110632370193E-22, -1.519154612992E-21, 2.761333800232E-22, 1.065911481816E-21,
           -4.533211702999E-22],
          [-5.515839683546E-22, 8.481401678700E-22, 5.664347521163E-22, -1.633466311759E-21,
           -1.391995046985E-21, 2.985138651995E-21, 5.521553247888E-21, 3.043061053033E-21,
           -1.391995046985E-21, -1.371046296717E-21, 4.932822149057E-22, 9.754728972929E-22,
           -5.515839683546E-22],
          [-7.975908114097E-22, 5.479652055316E-22, 1.005403041504E-21, -5.827038360596E-22,
           -2.145503300375E-21, -1.347188774155E-21, -6.110632370193E-22, -1.391995046985E-21,
           -2.145503300375E-21, -5.308241785637E-22, 9.886516164000E-22, 5.839207114976E-22,
           -7.975908114097E-22],
          [-8.616050256073E-22, -9.965035969377E-23, 9.024081028879E-22, 7.761129976189E-22,
           -5.308241785637E-22, -1.310314141567E-21, -1.519154612992E-21, -1.371046296717E-21,
           -5.308241785637E-22, 7.026025796278E-22, 9.281016513370E-22, -1.638123816652E-22,
           -8.616050256073E-22],
          [-5.176527305501E-22, -6.501360534968E-22, 1.232866890135E-22, 1.003666584284E-21,
           9.886516164000E-22, 5.103792541956E-22, 2.761333800232E-22, 4.932822149057E-22,
           9.886516164000E-22, 9.281016513370E-22, 1.577734427290E-22, -7.534846790666E-22,
           -5.176527305501E-22],
          [2.120208105909E-22, -6.034270619075E-22, -7.665862183419E-22, -1.620874951248E-22,
           5.839207114976E-22, 9.518011394833E-22, 1.065911481816E-21, 9.754728972929E-22,
           5.839207114976E-22, -1.638123816652E-22, -7.534846790666E-22, -6.579449237655E-22,
           2.120208105909E-22],
          [7.049456954407E-22, 1.842034121745E-22, -5.087139573635E-22, -8.959598399643E-22,
           -7.975908114097E-22, -5.611431437391E-22, -4.533211702999E-22, -5.515839683546E-22,
           -7.975908114097E-22, -8.616050256073E-22, -5.176527305501E-22, 2.120208105909E-22,
           7.049456954407E-22]], dtype=np.float64
        )
        #: array holding precalculated nearest neighbour extrapolation data
        self.precalc_extrapolation_nearest: np.array = np.array(
         [7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          1.842034121745E-22, -5.087139573635E-22, -8.959598399643E-22, -7.975908114097E-22,
          -5.611431437391E-22, -4.533211702999E-22, -5.515839683546E-22, -7.975908114097E-22,
          -8.616050256073E-22, -5.176527305501E-22, 2.120208105909E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.842034121745E-22,
          -5.087139573635E-22, -8.959598399643E-22, -7.975908114097E-22, -5.611431437391E-22,
          -4.533211702999E-22, -5.515839683546E-22, -7.975908114097E-22, -8.616050256073E-22,
          -5.176527305501E-22, 2.120208105909E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 1.842034121745E-22, -5.087139573635E-22,
          -8.959598399643E-22, -7.975908114097E-22, -5.611431437391E-22, -4.533211702999E-22,
          -5.515839683546E-22, -7.975908114097E-22, -8.616050256073E-22, -5.176527305501E-22,
          2.120208105909E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.842034121745E-22,
          1.842034121745E-22, 1.842034121745E-22, 1.842034121745E-22, 1.842034121745E-22,
          1.842034121745E-22, -5.087139573635E-22, -5.087139573635E-22, -5.087139573635E-22,
          -5.087139573635E-22, -5.087139573635E-22, -5.087139573635E-22, -8.959598399643E-22,
          -8.959598399643E-22, -8.959598399643E-22, -8.959598399643E-22, -8.959598399643E-22,
          -8.959598399643E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -5.611431437391E-22,
          -5.611431437391E-22, -5.611431437391E-22, -5.611431437391E-22, -5.611431437391E-22,
          -5.611431437391E-22, -4.533211702999E-22, -4.533211702999E-22, -4.533211702999E-22,
          -4.533211702999E-22, -4.533211702999E-22, -4.533211702999E-22, -5.515839683546E-22,
          -5.515839683546E-22, -5.515839683546E-22, -5.515839683546E-22, -5.515839683546E-22,
          -5.515839683546E-22, -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22,
          -7.975908114097E-22, -7.975908114097E-22, -7.975908114097E-22, -8.616050256073E-22,
          -8.616050256073E-22, -8.616050256073E-22, -8.616050256073E-22, -8.616050256073E-22,
          -8.616050256073E-22, -5.176527305501E-22, -5.176527305501E-22, -5.176527305501E-22,
          -5.176527305501E-22, -5.176527305501E-22, -5.176527305501E-22, 2.120208105909E-22,
          2.120208105909E-22, 2.120208105909E-22, 2.120208105909E-22, 2.120208105909E-22,
          2.120208105909E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 1.842034121745E-22,
          -5.087139573635E-22, -8.959598399643E-22, -7.975908114097E-22, -5.611431437391E-22,
          -4.533211702999E-22, -5.515839683546E-22, -7.975908114097E-22, -8.616050256073E-22,
          -5.176527305501E-22, 2.120208105909E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 1.842034121745E-22, -5.087139573635E-22,
          -8.959598399643E-22, -7.975908114097E-22, -5.611431437391E-22, -4.533211702999E-22,
          -5.515839683546E-22, -7.975908114097E-22, -8.616050256073E-22, -5.176527305501E-22,
          2.120208105909E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22,
          7.049456954407E-22, 1.842034121745E-22, -5.087139573635E-22, -8.959598399643E-22,
          -7.975908114097E-22, -5.611431437391E-22, -4.533211702999E-22, -5.515839683546E-22,
          -7.975908114097E-22, -8.616050256073E-22, -5.176527305501E-22, 2.120208105909E-22,
          7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22, 7.049456954407E-22], dtype=np.float64
        )
        #: array holding precalculated linear extrapolation data
        self.precalc_extrapolation_linear: np.array = np.array(
         [-1.194521420610E-19, -7.823082017163E-20, -3.700949828231E-20, 4.211823607006E-21,
          1.076476993194E-20, 7.357570465096E-21, -4.016962502163E-21, -1.443977450203E-20,
          -1.903018294300E-20, -2.039483993008E-20, -1.933304341125E-20, -1.443977450203E-20,
          -3.807888037972E-21, 7.140500088303E-21, 1.169244258887E-20, 4.211823607006E-21,
          -2.906695129325E-20, -6.234572619351E-20, -9.562450109377E-20, -7.823082017163E-20,
          -5.113959201337E-20, -2.404836385511E-20, 3.042864303151E-21, 7.237914425352E-21,
          4.735475657609E-21, -2.976628281430E-21, -9.892379938492E-21, -1.287383634324E-20,
          -1.374766701016E-20, -1.307255693029E-20, -9.892379938492E-21, -2.825793700517E-21,
          4.587782482018E-21, 7.865635329442E-21, 3.042864303151E-21, -1.853476437714E-20,
          -4.011239305743E-20, -6.169002173772E-20, -3.700949828231E-20, -2.404836385511E-20,
          -1.108722942791E-20, 1.873904999296E-21, 3.711058918763E-21, 2.113380850123E-21,
          -1.936294060697E-21, -5.344985374951E-21, -6.717489743492E-21, -7.100494090228E-21,
          -6.812070449321E-21, -5.344985374951E-21, -1.843699363062E-21, 2.035064875734E-21,
          4.038828070016E-21, 1.873904999296E-21, -8.002577461027E-21, -1.787905992135E-20,
          -2.775554238167E-20, 4.211823607006E-21, 3.042864303151E-21, 1.873904999296E-21,
          2.529609455086E-21, 4.354273214732E-21, 6.178936974377E-21, 1.076476993194E-20,
          7.237914425352E-21, 3.711058918763E-21, 3.500226650129E-21, 6.816249888084E-21,
          1.013227312604E-20, 7.357570465096E-21, 4.735475657609E-21, 2.113380850123E-21,
          9.287866896806E-22, 2.366287336725E-21, 3.803787983769E-21, -4.016962502163E-21,
          -2.976628281430E-21, -1.936294060697E-21, -3.477394497976E-21, -6.058829155988E-21,
          -8.640263814000E-21, -1.443977450203E-20, -9.892379938492E-21, -5.344985374951E-21,
          -6.312064964548E-21, -1.182653911769E-20, -1.734101327083E-20, -1.903018294300E-20,
          -1.287383634324E-20, -6.717489743492E-21, -6.918455006928E-21, -1.327576687012E-20,
          -1.963307873331E-20, -2.039483993008E-20, -1.374766701016E-20, -7.100494090228E-21,
          -6.964490057419E-21, -1.347565894454E-20, -1.998682783166E-20, -1.933304341125E-20,
          -1.307255693029E-20, -6.812070449321E-21, -6.980635883397E-21, -1.340968779844E-20,
          -1.983873971348E-20, -1.443977450203E-20, -9.892379938492E-21, -5.344985374951E-21,
          -6.312064964548E-21, -1.182653911769E-20, -1.734101327083E-20, -3.807888037972E-21,
          -2.825793700517E-21, -1.843699363062E-21, -3.317879124813E-21, -5.774153224019E-21,
          -8.230427323224E-21, 7.140500088303E-21, 4.587782482018E-21, 2.035064875734E-21,
          8.346507461054E-22, 2.186954222761E-21, 3.539257699416E-21, 1.169244258887E-20,
          7.865635329442E-21, 4.038828070016E-21, 3.857421048676E-21, 7.502821286761E-21,
          1.114822152485E-20, 4.211823607006E-21, 3.042864303151E-21, 1.873904999296E-21,
          2.529609455086E-21, 4.354273214732E-21, 6.178936974377E-21, -2.906695129325E-20,
          -1.853476437714E-20, -8.002577461027E-21, 2.529609455086E-21, 3.500226650129E-21,
          9.287866896806E-22, -3.477394497976E-21, -6.312064964548E-21, -6.918455006928E-21,
          -6.964490057419E-21, -6.980635883397E-21, -6.312064964548E-21, -3.317879124813E-21,
          8.346507461054E-22, 3.857421048676E-21, 2.529609455086E-21, -4.160997529959E-21,
          -1.085160451500E-20, -1.754221150005E-20, -6.234572619351E-20, -4.011239305743E-20,
          -1.787905992135E-20, 4.354273214732E-21, 6.816249888084E-21, 2.366287336725E-21,
          -6.058829155988E-21, -1.182653911769E-20, -1.327576687012E-20, -1.347565894454E-20,
          -1.340968779844E-20, -1.182653911769E-20, -5.774153224019E-21, 2.186954222761E-21,
          7.502821286761E-21, 4.354273214732E-21, -1.085160451500E-20, -2.605748224474E-20,
          -4.126335997448E-20, -9.562450109377E-20, -6.169002173772E-20, -2.775554238167E-20,
          6.178936974377E-21, 1.013227312604E-20, 3.803787983769E-21, -8.640263814000E-21,
          -1.734101327083E-20, -1.963307873331E-20, -1.998682783166E-20, -1.983873971348E-20,
          -1.734101327083E-20, -8.230427323224E-21, 3.539257699416E-21, 1.114822152485E-20,
          6.178936974377E-21, -1.754221150005E-20, -4.126335997448E-20, -6.498450844891E-20], dtype=np.float64
        )
