"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const cleaner_1 = require("./cleaner");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, slackConfig) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = slackConfig;
        this.componentBuilder = new component_builder_1.ComponentBuilder(this, componentDepsConfig, this.id, this.pipelineConfig.parent_image);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        this.createCleanerTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            if (slackConfig && slackConfig.channel && slackConfig.slackWebhookUrl && slackConfig.username) {
                new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, slackConfig, `${this.id}Recipe`);
            }
        }
    }
    /**
     * @stability stable
     */
    createCleanerTask() {
        if (this.pipelineConfig.cleaning_schedule) {
            new cleaner_1.Cleaner(this, 'Cleaner', this.pipelineConfig.cleaning_schedule, this.id);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let compMap = this.componentBuilder.createComponentDependenciesMap();
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.pipelineConfig.recipe.components.map((c) => ({
                componentArn: compMap[c.name].ref,
            })),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.createDistributionConfig();
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    createDistributionConfig() {
        // Get current region
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        const amiTags = {
            Name: this.id,
            CreatedBy: 'EC2 Image Builder',
            Ec2ImageBuilderArn: '{{imagebuilder:buildVersion}}',
            AutoClean: 'True',
        };
        let sharedWithDistributions = [];
        if (this.pipelineConfig.shared_with) {
            sharedWithDistributions = this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: amiTags,
                    TargetAccountIds: s.account_ids,
                },
            }));
            // current region might already be included
            if (sharedWithDistributions.some(s => s.region === region)) {
                const distribution = sharedWithDistributions.find(s => s.region === region);
                //Push current account ID as distribute
                distribution === null || distribution === void 0 ? void 0 : distribution.amiDistributionConfiguration.TargetAccountIds.push(accountId);
            }
        }
        if (sharedWithDistributions.length == 0) {
            sharedWithDistributions = [{
                    region: region,
                    amiDistributionConfiguration: {
                        Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                        Description: `${this.id} Shared image`,
                        AmiTags: amiTags,
                        TargetAccountIds: [accountId],
                    },
                }];
        }
        return new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: sharedWithDistributions,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : undefined;
        this.ebsEncryptionKey = key;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let envVariables = {
                IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                PIPELINE_NAME: { value: this.imagePipeline.name },
            };
            if (this.slackConfig && this.slackConfig.slackWebhookUrl && this.slackConfig.channel && this.slackConfig.username) {
                envVariables.SLACK_WEBHOOK_URL = { value: this.slackConfig.slackWebhookUrl };
                envVariables.SLACK_CHANNEL = { value: this.slackConfig.channel };
                envVariables.SLACK_USERNAME = { value: this.slackConfig.username };
            }
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'if [ -n "$SLACK_WEBHOOK_URL" ]; then curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL; fi',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: envVariables,
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.25" };
//# sourceMappingURL=data:application/json;base64,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