"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirewallDomainList = exports.FirewallDomains = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const route53resolver_generated_1 = require("./route53resolver.generated");
/**
 * A list of domains
 */
class FirewallDomains {
    /**
     * Firewall domains created from a list of domains
     *
     * @param list the list of domains
     */
    static fromList(list) {
        for (const domain of list) {
            if (!/^([\w-.]{1,255}|\*[\w-.]{1,254})$/.test(domain)) {
                throw new Error(`Invalid domain: ${domain}. Domain can optionally start with *. Max length of 255. Valid characters: A-Z, a-z, 0-9, _, -, .`);
            }
        }
        return {
            bind(_scope) {
                return { domains: list };
            },
        };
    }
    /**
     * Firewall domains created from the URL of a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param url S3 bucket url (s3://bucket/prefix/objet).
     */
    static fromS3Url(url) {
        if (!core_1.Token.isUnresolved(url) && !url.startsWith('s3://')) {
            throw new Error(`The S3 URL must start with s3://, got ${url}`);
        }
        return {
            bind(_scope) {
                return { domainFileUrl: url };
            },
        };
    }
    /**
     * Firewall domains created from a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param bucket S3 bucket
     * @param key S3 key
     */
    static fromS3(bucket, key) {
        return this.fromS3Url(bucket.s3UrlForObject(key));
    }
    /**
     * Firewall domains created from a local disk path to a text file.
     * The file must be a text file (`.txt` extension) and must contain a single
     * domain per line. It will be uploaded to S3.
     *
     * @param assetPath path to the text file
     */
    static fromAsset(assetPath) {
        // cdk-assets will correctly set the content type for the S3 object
        // if the file has the correct extension
        if (path.extname(assetPath) !== '.txt') {
            throw new Error(`FirewallDomains.fromAsset() expects a file with the .txt extension, got ${assetPath}`);
        }
        return {
            bind(scope) {
                const asset = new aws_s3_assets_1.Asset(scope, 'Domains', { path: assetPath });
                if (!asset.isFile) {
                    throw new Error('FirewallDomains.fromAsset() expects a file');
                }
                return { domainFileUrl: asset.s3ObjectUrl };
            },
        };
    }
}
exports.FirewallDomains = FirewallDomains;
_a = JSII_RTTI_SYMBOL_1;
FirewallDomains[_a] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomains", version: "1.146.0" };
/**
 * A Firewall Domain List
 */
class FirewallDomainList extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_route53resolver_FirewallDomainListProps(props);
        if (props.name && !core_1.Token.isUnresolved(props.name) && !/^[\w-.]{1,128}$/.test(props.name)) {
            throw new Error(`Invalid domain list name: ${props.name}. The name must have 1-128 characters. Valid characters: A-Z, a-z, 0-9, _, -, .`);
        }
        const domainsConfig = props.domains.bind(this);
        const domainList = new route53resolver_generated_1.CfnFirewallDomainList(this, 'Resource', {
            name: props.name,
            domainFileUrl: domainsConfig.domainFileUrl,
            domains: domainsConfig.domains,
        });
        this.firewallDomainListId = domainList.attrId;
        this.firewallDomainListArn = domainList.attrArn;
        this.firewallDomainListCreationTime = domainList.attrCreationTime;
        this.firewallDomainListCreatorRequestId = domainList.attrCreatorRequestId;
        this.firewallDomainListDomainCount = domainList.attrDomainCount;
        this.firewallDomainListManagedOwnerName = domainList.attrManagedOwnerName;
        this.firewallDomainListModificationTime = domainList.attrModificationTime;
        this.firewallDomainListStatus = domainList.attrStatus;
        this.firewallDomainListStatusMessage = domainList.attrStatusMessage;
    }
    /**
     * Import an existing Firewall Rule Group
     */
    static fromFirewallDomainListId(scope, id, firewallDomainListId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.firewallDomainListId = firewallDomainListId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FirewallDomainList = FirewallDomainList;
_b = JSII_RTTI_SYMBOL_1;
FirewallDomainList[_b] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomainList", version: "1.146.0" };
//# sourceMappingURL=data:application/json;base64,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