#!/bin/bash

# Changelog 생성 스크립트
# Git 히스토리로부터 자동으로 changelog entry를 생성합니다

set -e

if [ -z "$1" ]; then
  echo "❌ Usage: generate-changelog.sh <VERSION>"
  echo "   Example: generate-changelog.sh 0.31.0"
  exit 1
fi

VERSION="$1"
PREVIOUS_TAG=$(git describe --tags --abbrev=0 HEAD^ 2>/dev/null || echo "")

echo "📝 Generating Changelog for $VERSION"
echo "   Previous tag: ${PREVIOUS_TAG:-'<initial release>'}"
echo ""

# Get commits since last tag
if [ -z "$PREVIOUS_TAG" ]; then
  COMMITS=$(git log --format="%h %s" --reverse)
else
  COMMITS=$(git log $PREVIOUS_TAG...HEAD --format="%h %s" --reverse)
fi

# Categorize commits
FEATURES=""
FIXES=""
DOCS=""
REFACTOR=""
OTHER=""

while IFS= read -r COMMIT; do
  if [[ $COMMIT == *"feat:"* ]]; then
    FEATURES+="- $(echo "$COMMIT" | sed 's/^[a-f0-9]* //')\n"
  elif [[ $COMMIT == *"fix:"* ]]; then
    FIXES+="- $(echo "$COMMIT" | sed 's/^[a-f0-9]* //')\n"
  elif [[ $COMMIT == *"docs:"* ]]; then
    DOCS+="- $(echo "$COMMIT" | sed 's/^[a-f0-9]* //')\n"
  elif [[ $COMMIT == *"refactor:"* ]]; then
    REFACTOR+="- $(echo "$COMMIT" | sed 's/^[a-f0-9]* //')\n"
  else
    OTHER+="- $(echo "$COMMIT" | sed 's/^[a-f0-9]* //')\n"
  fi
done <<< "$COMMITS"

# Create changelog entry
RELEASE_DATE=$(date +"%Y-%m-%d")

cat > CHANGELOG.entry.md << EOF
## [$VERSION] - $RELEASE_DATE

### 🇰🇷 한국어

#### ✨ 주요 기능
EOF

if [ -n "$FEATURES" ]; then
  echo -e "$FEATURES" >> CHANGELOG.entry.md
else
  echo "- 새로운 기능이 추가되었습니다" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 🐛 버그 수정
EOF

if [ -n "$FIXES" ]; then
  echo -e "$FIXES" >> CHANGELOG.entry.md
else
  echo "- 버그 수정이 포함되었습니다" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 📚 문서
EOF

if [ -n "$DOCS" ]; then
  echo -e "$DOCS" >> CHANGELOG.entry.md
else
  echo "- 문서가 업데이트되었습니다" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 🔧 기타
- 리팩토링 및 개선사항이 포함되었습니다

#### 설치 방법
\`\`\`bash
# UV 사용 (권장)
uv add moai-adk==$VERSION

# pip 사용
pip install moai-adk==$VERSION
\`\`\`

---

### 🇺🇸 English

#### ✨ Features
EOF

if [ -n "$FEATURES" ]; then
  echo -e "$FEATURES" >> CHANGELOG.entry.md
else
  echo "- New features have been added" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 🐛 Bug Fixes
EOF

if [ -n "$FIXES" ]; then
  echo -e "$FIXES" >> CHANGELOG.entry.md
else
  echo "- Bug fixes have been included" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 📚 Documentation
EOF

if [ -n "$DOCS" ]; then
  echo -e "$DOCS" >> CHANGELOG.entry.md
else
  echo "- Documentation has been updated" >> CHANGELOG.entry.md
fi

cat >> CHANGELOG.entry.md << EOF

#### 🔧 Other Changes
- Refactoring and improvements have been made

#### Installation
\`\`\`bash
# UV (Recommended)
uv add moai-adk==$VERSION

# pip
pip install moai-adk==$VERSION
\`\`\`

---

🤖 Generated by [Claude Code](https://claude.com/claude-code) - Automated Release Pipeline

EOF

echo "✅ Changelog entry generated: CHANGELOG.entry.md"
echo ""
echo "📋 Entry preview:"
head -20 CHANGELOG.entry.md

exit 0
