import abc
import typing

from .exception import *
from .token import *
from .lexer import *
from .node import *


__all__ = [
    "Parser",
    "ParamParser",
    "ParamOptionsParser",
    "ParamSequenceParser",
    "parse",
]


class Parser(abc.ABC):
    """
    Base class for all parser implementations, such as a :class:`ParamSequenceParser`. A parser takes the stream of
    tokens produced by the lexer and returns a meaningful representation of the data.
    """

    def __init__(self, lexer: Lexer, offset: int):
        """
        Init method for the class.

        :param lexer: The lexer that produces the tokens for the parser.
        :param offset: The offset to calculate absolute token position.
        """
        self._lexer = lexer
        self._offset = offset

    @abc.abstractmethod
    def parse(self) -> list:
        """
        Parses the stream of tokens generated by the lexer and returns a meaningful representation of the data as a
        list. Depending on the type of parser, this list might contain strings or might contain nodes.

        :raise ParserSyntaxException: The exception raised if the syntax of the expression is invalid.
        :return: The parsed strings or nodes.
        """
        pass

    def eat(self, token_types: typing.Optional[typing.List[str]]) -> Token:
        """
        Consumes the next token produced by the lexer. If a list of valid token types is given, this method will throw
        a :class:`.exception.ParserSyntaxException` if the produced token's type is not within the list.

        :param token_types: The list of valid token types.
        :raise ParserSyntaxException: The exception raised if the produced token is not within the allowed types.
        :return: The next token produced by the lexer.
        """
        token = self._lexer.get_next_token()

        if token_types and token.token_type not in token_types:
            self.raise_unexpected_token(token)

        return token

    def raise_unexpected_token(self, token: Token):
        """
        Raises a :class:`.exception.ParserSyntaxException` for the given token.

        :param token: The token to raise an exception for.
        :raise ParserSyntaxException: The exception for the given token.
        :return:
        """
        raise ParserSyntaxException(
            "Unexpected token '{}' at position '{}'".format(
                token.token_type,
                token.token_position + self._offset,
            )
        )


class ParamParser(Parser):
    """
    Parses tokens of a param expression. That is an expression that looks as follows:
    `"name:type:option1,option2,option3"`
    """

    def parse(self) -> typing.List[typing.Union[str, list]]:
        """
        Parses the tokens generated by the lexer and returns a list containing the param configuration of the
        expression. The first element of the list is the name of the param as a string. The second element is the type
        of the param as a string. The type may be :data:`None`. The third param is a list of strings, where each element
        represents the option's value.

        :raise ParserSyntaxException: The exception raised if the syntax of the expression is invalid.
        :return: The parsed strings and lists representing the expression.
        """
        param_name_token = self.eat([Token.TYPE_SEQUENCE])
        param_name = param_name_token.token_value
        self.eat([Token.TYPE_EOF, Token.TYPE_COLON])

        param_type_token = self.eat([Token.TYPE_EOF, Token.TYPE_SEQUENCE])
        param_type = param_type_token.token_value
        self.eat([Token.TYPE_EOF, Token.TYPE_COLON])

        param_options_token = self.eat([Token.TYPE_EOF, Token.TYPE_SEQUENCE])
        param_options = param_options_token.token_value
        self.eat([Token.TYPE_EOF])

        if param_options_token.token_type == Token.TYPE_SEQUENCE:
            param_options_parser = ParamOptionsParser(
                ParamOptionsLexer(param_options_token.token_raw_value),
                param_options_token.token_position + self._offset,
            )
            param_options = param_options_parser.parse()

        return [
            param_name,
            param_type,
            param_options,
        ]


class ParamOptionsParser(Parser):
    """
    Parses tokens of a param options expression. That is an expression that looks as follows:
    `"option1,option2,option3"`
    """

    def parse(self) -> typing.List[str]:
        """
        Parses the tokens generated by the lexer and returns a list of strings, where each string is an option value.
        The order of the list equals the order of option values within the expression.

        :raise ParserSyntaxException: The exception raised if the syntax of the expression is invalid.
        :return: The parsed strings representing the expression.
        """
        options_list = []
        current_token = self.eat([Token.TYPE_EOF, Token.TYPE_SEQUENCE])

        while current_token.token_type != Token.TYPE_EOF:
            options_list.append(current_token.token_value)

            self.eat([Token.TYPE_EOF, Token.TYPE_COMMA])
            current_token = self.eat([Token.TYPE_EOF, Token.TYPE_SEQUENCE])

        return options_list


class ParamSequenceParser(Parser):
    """
    Parses tokens of a param sequence expression. That is an expression that looks as follows:
    `"random-sequence{name:type:option1,option2,option3}other-random-sequence"`
    """

    def parse(self) -> typing.List[Node]:
        """
        Parses the tokens generated by the lexer and returns a list of nodes representing the structure
        of the expression. A node may be a :class:`.node.SequenceNode`, which represents a sequence of text without any
        meaning, or may be a :class:`.node.ParamNode`, which represent a parameter with a name, optional type and some
        options.

        :raise ParserSyntaxException: The exception raised if the syntax of the expression is invalid.
        :return: The parsed nodes representing the expression.
        """
        node_list = []
        current_token = self.eat(
            [Token.TYPE_EOF, Token.TYPE_SEQUENCE, Token.TYPE_LC_BRACKET]
        )

        while current_token.token_type != Token.TYPE_EOF:
            current_token_type = current_token.token_type

            if current_token_type == Token.TYPE_SEQUENCE:
                node_list.append(SequenceNode(current_token.token_value))

            elif current_token_type == Token.TYPE_LC_BRACKET:
                param_sequence_token = self.eat([Token.TYPE_SEQUENCE])
                param_sequence_parser = ParamParser(
                    ParamLexer(param_sequence_token.token_raw_value),
                    param_sequence_token.token_position + self._offset,
                )
                param_sequence = param_sequence_parser.parse()

                node_list.append(
                    ParamNode(param_sequence[0], param_sequence[1], param_sequence[2])
                )

                self.eat([Token.TYPE_RC_BRACKET])

            current_token = self.eat(
                [Token.TYPE_EOF, Token.TYPE_SEQUENCE, Token.TYPE_LC_BRACKET]
            )

        return node_list


def parse(expression: str) -> typing.List[Node]:
    """
    Parses the given expression and returns a list of :class:`.node.SequenceNode` and :class:`.node.ParamNode`
    instances. The expected expression looks as follows:
    `"random-sequence{name:type:option1,option2,option3}other-random-sequence"`

    :raise ParserSyntaxException: The exception raised if the syntax of the expression is invalid.
    :return: The parsed nodes representing the expression.
    """
    parser = ParamSequenceParser(ParamSequenceLexer(expression), 0)
    return parser.parse()
