"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type.
 *
 * @stability stable
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_CustomLambdaDeploymentConfigProps(props);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = (_b = props.deploymentConfigName) !== null && _b !== void 0 ? _b : `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
            ? 'Every'
            : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
_a = JSII_RTTI_SYMBOL_1;
CustomLambdaDeploymentConfig[_a] = { fqn: "@aws-cdk/aws-codedeploy.CustomLambdaDeploymentConfig", version: "1.137.0" };
//# sourceMappingURL=data:application/json;base64,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