from typing import Any, Dict, Optional, Union

import httpx

from ...client import AuthenticatedClient
from ...models.folder_create import FolderCreate
from ...models.folder_out import FolderOut
from ...models.http_validation_error import HTTPValidationError
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    client: AuthenticatedClient,
    json_body: FolderCreate,
    ignore_conflicts: Union[Unset, bool] = False,
) -> Dict[str, Any]:
    url = "{}/api/v1/folders/".format(client.base_url)

    headers: Dict[str, Any] = client.get_headers()
    cookies: Dict[str, Any] = client.get_cookies()

    params: Dict[str, Any] = {
        "ignore_conflicts": ignore_conflicts,
    }
    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    json_json_body = json_body.to_dict()

    return {
        "url": url,
        "headers": headers,
        "cookies": cookies,
        "timeout": client.get_timeout(),
        "json": json_json_body,
        "params": params,
    }


def _parse_response(
    *, response: httpx.Response
) -> Optional[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    if response.status_code == 201:
        response_201 = FolderOut.from_dict(response.json())

        return response_201
    if response.status_code == 401:
        response_401 = None

        return response_401
    if response.status_code == 404:
        response_404 = None

        return response_404
    if response.status_code == 403:
        response_403 = None

        return response_403
    if response.status_code == 409:
        response_409 = None

        return response_409
    if response.status_code == 422:
        response_422 = HTTPValidationError.from_dict(response.json())

        return response_422
    return None


def _build_response(
    *, response: httpx.Response
) -> Response[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    return Response(
        status_code=response.status_code,
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(response=response),
    )


def sync_detailed(
    *,
    client: AuthenticatedClient,
    json_body: FolderCreate,
    ignore_conflicts: Union[Unset, bool] = False,
) -> Response[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    kwargs = _get_kwargs(
        client=client,
        json_body=json_body,
        ignore_conflicts=ignore_conflicts,
    )

    response = httpx.post(
        **kwargs,
    )

    return _build_response(response=response)


def sync(
    *,
    client: AuthenticatedClient,
    json_body: FolderCreate,
    ignore_conflicts: Union[Unset, bool] = False,
) -> Optional[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    """Creates a new folder.

    To be allowed to create a folder, a user must either:
    - be orgadmin: they can create a folder in their organization file system.
    - have write rights on the parent folder if they are basic users.

    Additionally, services can also create folders.

    The constraints of folder creation are:

    - Two folders with same parent cannot have the same name."""

    return sync_detailed(
        client=client,
        json_body=json_body,
        ignore_conflicts=ignore_conflicts,
    ).parsed


async def asyncio_detailed(
    *,
    client: AuthenticatedClient,
    json_body: FolderCreate,
    ignore_conflicts: Union[Unset, bool] = False,
) -> Response[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    kwargs = _get_kwargs(
        client=client,
        json_body=json_body,
        ignore_conflicts=ignore_conflicts,
    )

    async with httpx.AsyncClient() as _client:
        response = await _client.post(**kwargs)

    return _build_response(response=response)


async def asyncio(
    *,
    client: AuthenticatedClient,
    json_body: FolderCreate,
    ignore_conflicts: Union[Unset, bool] = False,
) -> Optional[Union[FolderOut, None, None, None, None, HTTPValidationError]]:
    """Creates a new folder.

    To be allowed to create a folder, a user must either:
    - be orgadmin: they can create a folder in their organization file system.
    - have write rights on the parent folder if they are basic users.

    Additionally, services can also create folders.

    The constraints of folder creation are:

    - Two folders with same parent cannot have the same name."""

    return (
        await asyncio_detailed(
            client=client,
            json_body=json_body,
            ignore_conflicts=ignore_conflicts,
        )
    ).parsed
