from typing import Any, Dict, List, Tuple, Union

import logging

from .dataclasses import (
    BuiltinCallable,
    ValidationError,
    ValidationResult,
    ValidationStep,
    ValidatorResult,
    load_ValidationStep,
    path_to_ValidationStep,
)
from .exceptions import ParsingError
from .path_matching import path_matches_pattern

logger = logging.getLogger(__name__)


def run_BuiltinCallable(target: Any, builtin: BuiltinCallable) -> Any:
    """Runs a BuiltinCallable on an arbitrary value."""
    return builtin.resolved_function(
        target,
        **builtin.args,
    )


def run_ValidationStep(filepath: str, step: ValidationStep) -> ValidationResult:
    """Runs a single validation step.

    Args:
        filepath (str): filepath of data to validate.
        step (ValidationStep): validation step to run.

    Returns:
        result (ValidationResult): result of the validation
    """
    if step.processor:
        try:
            processed = run_BuiltinCallable(
                filepath,
                step.processor.builtin,
            )  # type: Union[Any, str]
        except Exception as e:
            return ValidationResult(
                filepath=filepath,
                errors=[
                    ValidationError(
                        str(e),
                        # processor=step.processor,
                    )
                ],
            )
    else:
        processed = filepath

    results = []
    errors = []
    for validator in step.validators:
        try:
            value, passed = run_BuiltinCallable(
                processed,
                validator.builtin,
            )
            results.append(
                ValidatorResult(
                    value=value,
                    passed=passed,
                    validator=validator,
                )
            )
        except Exception as e:
            errors.append(
                ValidationError(
                    str(e),
                    # message=str(e),
                    validator=validator,
                )
            )

    if step.processor:
        return ValidationResult(
            filepath=filepath,
            processor=step.processor,
            results=results,
            errors=errors,
        )
    else:
        return ValidationResult(
            filepath=filepath,
            results=results,
            errors=errors,
        )


def run_validation(
    file_list: List[str],
    validation_steps: List[ValidationStep] = [],
    autogen_steps: bool = True,
) -> List[ValidationResult]:
    """Validates a list of files.

    Args:
        file_list (:obj:`list` of :obj:`str`): List of filepaths to validate.
        validation_steps: List of validation steps to run.
        autogen_steps: Add additional autogenerated ValidationSteps based on file types in `file_list`.
    Returns:
        results: List of validation results.
    """
    if autogen_steps:
        logger.debug("Autogenerating validation steps")
        autogenerated = autogenerate_validation_steps(file_list)
        logger.debug("Autogenerated: %s" % autogenerated)
        validation_steps = list(validation_steps)  # cast in case of map
        validation_steps.extend(autogenerated)

    results = []
    for validation_step in validation_steps:
        for filepath in file_list:
            if path_matches_pattern(filepath, validation_step.path_suffix):
                results.append(
                    run_ValidationStep(
                        filepath,
                        validation_step,
                    )
                )

    return results


def autogenerate_validation_steps(
    file_list: List[str],
) -> List[ValidationStep]:
    return list(map(path_to_ValidationStep, file_list))
