"""Module containing helper parts for constructing reusable SciKit models."""
from pipelinehelper import PipelineHelper
from sklearn.pipeline import Pipeline, FeatureUnion


class Part:
    """Represents a combination of a SciKit transformer and its parameters."""

    def __init__(self, name, pipeline, params=None):
        """
        Initializes a part.

        Args:
            name: string
                The name of this part. This will be included in the name of the
                parameters generated by a parent part. Names must be unique
                among siblings of parts.
            pipeline:
                a SciKit-compatible transformer.
            params:
                parameters to the transformer.

        """
        self.name = name
        self.pipeline = pipeline
        self.params = params if params is not None else {}

    def replace_params(self, params):
        """
        Replaces all parameters of the transformer with a specified dict.

        Args:
            params: dict
                new parameters for the transformer of this part

        Returns: Part
            this part for chaining

        """
        self.params = params
        return self

    def add_params(self, params):
        """
        Updates parameters of this part with values in a specified dict.

        Args:
            params: dict
                parameter values that are added/updated to the existing
                parameters of this part.

        Returns: Part
            this part for chaining

        """
        self.params.update(params)
        return self

    def alias(self, name):
        """
        Assigns a new name to this part.
        Args:
            name: string
                the new name of this part

        Returns: Part
            this part for chaining

        """
        self.name = name
        return self

    def __iter__(self):
        # convenience method for unpacking in plans.
        # this way, the final (root) part can be unpacked like so:
        # model, parameters = sequential(...
        yield self.pipeline
        yield self.params


def prefix_parameters(prefix, params):
    """
    Returns a copy of a dict with all keys prefixed.

    Args:
        prefix: string
            Which prefix to prepend to each key. This should not include the
            double underscore used to separate them.
        params: dict
            Dictionary of parameters. Only the keys are modified.
    Returns: dict
        A dictionary containing the same entries as the original, but each key
        is prepended with "<prefix>__".

    """
    return {f'{prefix}__{key}': value for key, value in params.items()}


def pick_one(name, parts, optional=False):
    """
    Produces a part with a PipelineHelper, its steps and their parameters.

    Args:
        name: string
            name of this part
        parts: list of parts
            all parts to choose from
        optional: boolean
            whether to include a "passthrough" option

    Returns: Part
        a part containing a PipelineHelper and its parameters.

    """
    models = []
    params = {}
    for part in parts:
        models.append((part.name, part.pipeline))
        params.update(prefix_parameters(part.name, part.params))
    result = PipelineHelper(models, optional=optional)
    final_params = {'selected_model': result.generate(params)}
    return Part(name, result, final_params)


def sequential(name, parts):
    """
    Produces a part with a Pipeline, its steps and their parameters.

    Args:
        name: string
            the name of this part
        parts: list of parts
            a list of parts that are included in the pipeline

    Returns: Part
        a part with a Pipeline, its steps and their parameters.

    """
    models = []
    params = {}
    for part in parts:
        models.append((part.name, part.pipeline))
        params.update(prefix_parameters(part.name, part.params))
    result = Pipeline(models)
    return Part(name, result, params)


def union(name, parts):
    """
    Produces a part with a FeatureUnion, its steps and their parameters.

    Args:
        name: string
            the name of this part
        parts: list of parts
            a list of parts that are included in the feature union

    Returns: Part
        a part with a FeatureUnion, its steps and their parameters.

    """
    models = []
    params = {}
    for part in parts:
        models.append((part.name, part.pipeline))
        params.update(prefix_parameters(part.name, part.params))
    result = FeatureUnion(models)
    return Part(name, result, params)

