# ========================================================================================
#                                                                                        =
#              Copyright (c) 2010 D.H.J. Takken (d.h.j.takken@xs4all.nl)                 =
#                      Copyright (c) 2020 the EDXML Foundation                           =
#                                                                                        =
#                                   http://edxml.org                                     =
#                                                                                        =
#             This file is part of the EDXML Software Development Kit (SDK)              =
#                       and is released under the MIT License:                           =
#                         https://opensource.org/licenses/MIT                            =
#                                                                                        =
# ========================================================================================

from lxml import etree
from edxml.transcode import TranscoderTestHarness


class XmlTranscoderTestHarness(TranscoderTestHarness):

    def __init__(self, fixtures_path, transcoder, transcoder_root, base_ontology=None, register=True):
        """
        Creates a new test harness for testing specified record transcoder
        using XML fixtures stored at the indicated path. When provided,
        the harness will try to upgrade the base ontology to the
        ontology generated by the record transcoder, raising an exception
        in case of backward incompatibilities.

        The transcoder_root is the XPath expression that the record transcoder
        will be registered at. It will be used to extract the input
        elements for the record transcoder from XML fixtures, exactly
        as XmlTranscoderMediator would do on real data.

        By default the record transcoder will be automatically registered at
        the specified xpath. In case you wish to do the record registration
        on your own you must set the register parameter to False.

        Args:
            fixtures_path (str): Path to the fixtures set
            transcoder (edxml.transcode.RecordTranscoder): The record transcoder under test
            transcoder_root (str): XPath for record transcoder registration
            base_ontology (edxml.Ontology): Base ontology
            register (bool): Register the transcoder yes or no
        """
        super().__init__(transcoder, transcoder_root, base_ontology=base_ontology, register=register)
        self._transcoder_root = transcoder_root
        self._fixtures_path = fixtures_path

    def process_xml(self, filename, element_root=None):
        """
        Parses specified XML file and transcodes it to produce output
        events. The output events are added to the event set. The filename
        argument must be a path relative to the fixtures path.

        The XML file is expected to be structured like real input data
        stripped down to contain the XML elements that are relevant to
        the record transcoder under test.

        The element_root is only relevant when the record transcoder can output
        multiple types of events. It must be set to the XPath expression
        of the sub-element inside the element being transcoded that corresponds
        with one specific output event type. When unspecified and the transcoder
        produces a single type of output events it will be fetched from
        the TYPE_MAP constant of the record transcoder.

        Args:
            filename (str): The XML file to use as input record fixture
            element_root:
        """
        if element_root is None:
            if len(self.transcoder.TYPE_MAP.keys()) > 1:
                raise Exception(
                    'No element root was specified for transcoding XML element using %s. '
                    'You must specify one because this record transcoder has multiple element roots in its TYPE_MAP.'
                    % type(self.transcoder).__name__
                )
            element_root = list(self.transcoder.TYPE_MAP.keys())[0]

        element = etree.fromstring(open(self._fixtures_path.rstrip('/') + '/' + filename, 'rb').read())

        for sub_element in element.xpath(self._transcoder_root):
            self.process(sub_element, selector=element_root)
        self.close()
