# ========================================================================================
#                                                                                        =
#              Copyright (c) 2010 D.H.J. Takken (d.h.j.takken@xs4all.nl)                 =
#                      Copyright (c) 2020 the EDXML Foundation                           =
#                                                                                        =
#                                   http://edxml.org                                     =
#                                                                                        =
#             This file is part of the EDXML Software Development Kit (SDK)              =
#                       and is released under the MIT License:                           =
#                         https://opensource.org/licenses/MIT                            =
#                                                                                        =
# ========================================================================================

from edxml.transcode import TranscoderTestHarness


class ObjectTranscoderTestHarness(TranscoderTestHarness):

    def __init__(self, transcoder, record_selector, base_ontology=None, register=True):
        """
        Creates a new test harness for testing specified record transcoder.
        When provided, the harness will try to upgrade the base ontology to the
        ontology generated by the record transcoder, raising an exception
        in case of backward incompatibilities.

        The record_selector is the selector that the record transcoder
        will be registered at.

        By default the record transcoder will be automatically registered at
        the specified xpath. In case you wish to do the record registration
        on your own you must set the register parameter to False.

        Args:
            transcoder (edxml.transcode.RecordTranscoder): The record transcoder under test
            base_ontology (edxml.Ontology): Base ontology
            register (bool): Register the transcoder yes or no
        """
        super().__init__(transcoder, record_selector, base_ontology=base_ontology, register=register)

    def process_object(self, fixture_object, selector=None, close=True):
        """
        Parses specified object and transcodes it to produce output
        events. The output events are added to the event set.

        The selector is only relevant when the record transcoder can output
        multiple types of events. It must be set to the selector of the sub-object
        inside the object being transcoded that corresponds with one specific output
        event type. When unspecified and the transcoder produces a single type of
        output events it will be fetched from the TYPE_MAP constant of the record
        transcoder.

        By default, the test harness is automatically closed after processing the
        record. When processing multiple input records this can be prevented.
        Note that the harness must be closed before using it in assertions.

        Args:
            fixture_object: The object to use as input record fixture
            selector:
            close (bool): Close test harness after processing yes / no
        """
        if selector is None:
            if len(self.transcoder.TYPE_MAP.keys()) > 1:
                raise Exception(
                    'No selector was specified for transcoding the object using %s. '
                    'You must specify one because this record transcoder has multiple selectors in its TYPE_MAP.'
                    % type(self.transcoder).__name__
                )
            selector = list(self.transcoder.TYPE_MAP.keys())[0]

        self.process(fixture_object, selector)

        if close:
            self.close()
