"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const state_graph_1 = require("./state-graph");
const stepfunctions_canned_metrics_generated_1 = require("./stepfunctions-canned-metrics.generated");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 *
 * @default STANDARD
 */
var StateMachineType;
(function (StateMachineType) {
    /**
     * Express Workflows are ideal for high-volume, event processing workloads.
     */
    StateMachineType["EXPRESS"] = "EXPRESS";
    /**
     * Standard Workflows are ideal for long-running, durable, and auditable workflows.
     */
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 *
 * @default ERROR
 */
var LogLevel;
(function (LogLevel) {
    /**
     * No Logging
     */
    LogLevel["OFF"] = "OFF";
    /**
     * Log everything
     */
    LogLevel["ALL"] = "ALL";
    /**
     * Log all errors
     */
    LogLevel["ERROR"] = "ERROR";
    /**
     * Log fatal errors
     */
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id, {
            environmentFromArn: stateMachineArn,
        });
    }
    /**
     * Grant the given identity permissions to start an execution of this state
     * machine.
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to start a synchronous execution of
     * this state machine.
     */
    grantStartSyncExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartSyncExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to read results from state
     * machine.
     */
    grantRead(identity) {
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListExecutions',
                'states:ListStateMachines',
            ],
            resourceArns: [this.stateMachineArn],
        });
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:DescribeExecution',
                'states:DescribeStateMachineForExecution',
                'states:GetExecutionHistory',
            ],
            resourceArns: [`${this.executionArn()}:*`],
        });
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListActivities',
                'states:DescribeStateMachine',
                'states:DescribeActivity',
            ],
            resourceArns: ['*'],
        });
    }
    /**
     * Grant the given identity task response permissions on a state machine
     */
    grantTaskResponse(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:SendTaskSuccess',
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
            ],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions on all executions of the state machine
     */
    grantExecution(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [`${this.executionArn()}:*`],
        });
    }
    /**
     * Grant the given identity custom permissions
     */
    grant(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Return the given named metric for this State Machine's executions
     *
     * @default - sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensionsMap: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed
     *
     * @default - sum over 5 minutes
     */
    metricFailed(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsFailedSum, props);
    }
    /**
     * Metric for the number of executions that were throttled
     *
     * @default - sum over 5 minutes
     */
    metricThrottled(props) {
        // There's a typo in the "canned" version of this
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted
     *
     * @default - sum over 5 minutes
     */
    metricAborted(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsAbortedSum, props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default - sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsSucceededSum, props);
    }
    /**
     * Metric for the number of executions that timed out
     *
     * @default - sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionsTimedOutSum, props);
    }
    /**
     * Metric for the number of executions that were started
     *
     * @default - sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes
     *
     * @default - average over 5 minutes
     */
    metricTime(props) {
        return this.cannedMetric(stepfunctions_canned_metrics_generated_1.StatesMetrics.executionTimeAverage, props);
    }
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    executionArn() {
        return core_1.Stack.of(this).formatArn({
            resource: 'execution',
            service: 'states',
            resourceName: core_1.Arn.split(this.stateMachineArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ StateMachineArn: this.stateMachineArn }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Define a StepFunctions State Machine
 */
class StateMachine extends StateMachineBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateMachineProps(props);
        if (props.stateMachineName !== undefined) {
            this.validateStateMachineName(props.stateMachineName);
        }
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = (_b = props.stateMachineType) !== null && _b !== void 0 ? _b : StateMachineType.STANDARD;
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: (_c = props.stateMachineType) !== null && _c !== void 0 ? _c : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration: props.logs ? this.buildLoggingConfiguration(props.logs) : undefined,
            tracingConfiguration: props.tracingEnabled ? this.buildTracingConfiguration() : undefined,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * The principal this state machine is running as
     */
    get grantPrincipal() {
        return this.role.grantPrincipal;
    }
    /**
     * Add the given statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    validateStateMachineName(stateMachineName) {
        if (!core_1.Token.isUnresolved(stateMachineName)) {
            if (stateMachineName.length < 1 || stateMachineName.length > 80) {
                throw new Error(`State Machine name must be between 1 and 80 characters. Received: ${stateMachineName}`);
            }
            if (!stateMachineName.match(/^[a-z0-9\+\!\@\.\(\)\-\=\_\']+$/i)) {
                throw new Error(`State Machine name must match "^[a-z0-9+!@.()-=_']+$/i". Received: ${stateMachineName}`);
            }
        }
    }
    buildLoggingConfiguration(logOptions) {
        // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
        this.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogDelivery',
                'logs:GetLogDelivery',
                'logs:UpdateLogDelivery',
                'logs:DeleteLogDelivery',
                'logs:ListLogDeliveries',
                'logs:PutResourcePolicy',
                'logs:DescribeResourcePolicies',
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        return {
            destinations: [{
                    cloudWatchLogsLogGroup: { logGroupArn: logOptions.destination.logGroupArn },
                }],
            includeExecutionData: logOptions.includeExecutionData,
            level: logOptions.level || 'ERROR',
        };
    }
    buildTracingConfiguration() {
        this.addToRolePolicy(new iam.PolicyStatement({
            // https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#xray-permissions-resources
            // https://docs.aws.amazon.com/step-functions/latest/dg/xray-iam.html
            actions: [
                'xray:PutTraceSegments',
                'xray:PutTelemetryRecords',
                'xray:GetSamplingRules',
                'xray:GetSamplingTargets',
            ],
            resources: ['*'],
        }));
        return {
            enabled: true,
        };
    }
}
exports.StateMachine = StateMachine;
_a = JSII_RTTI_SYMBOL_1;
StateMachine[_a] = { fqn: "@aws-cdk/aws-stepfunctions.StateMachine", version: "1.147.0" };
//# sourceMappingURL=data:application/json;base64,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