import numpy as np
import pytest
import sympy

from copul.family.archimedean.nelsen18 import Nelsen18


@pytest.fixture
def nelsen18_copula():
    """Fixture providing a Nelsen18 copula with theta=3."""
    return Nelsen18(3)


def test_parameter_validation():
    """Test parameter validation for the Nelsen18 copula."""
    # Valid values should not raise errors
    Nelsen18(2)  # Lower bound
    Nelsen18(3)  # Interior point
    Nelsen18(10)  # Larger value

    # Values below lower bound should raise ValueError
    with pytest.raises(ValueError, match="must be >= 2"):
        Nelsen18(1.9)

    # Negative values should also raise ValueError
    with pytest.raises(ValueError, match="must be >= 2"):
        Nelsen18(-1)


def test_is_absolutely_continuous(nelsen18_copula):
    """Test the is_absolutely_continuous property."""
    # Nelsen18 is not absolutely continuous
    assert not nelsen18_copula.is_absolutely_continuous


def test_generator_function(nelsen18_copula):
    """Test the generator function of the Nelsen18 copula."""
    # Test generator at specific t values
    t_values = [0.1, 0.3, 0.5, 0.7, 0.9]

    for t in t_values:
        # Manual calculation using the formula: exp(theta/(t-1))
        theta = 3
        expected = np.exp(theta / (t - 1))
        actual = float(nelsen18_copula.generator(t))
        assert np.isclose(actual, expected, rtol=1e-5)


def test_nelsen18_inverse_generator_at_infinity():
    """Test the inverse generator function of Nelsen3."""
    copula = Nelsen18(2.5)
    y = sympy.exp(-2.5)
    inv_gen = copula.inv_generator(y=y)
    actual = float(inv_gen)
    assert np.isclose(actual, 0, rtol=1e-10)


def test_cdf_function(nelsen18_copula):
    """Test the CDF function of the Nelsen18 copula."""
    # Test CDF at specific points
    points = [(0.3, 0.4), (0.5, 0.5), (0.7, 0.2), (0.9, 0.8)]

    for u, v in points:
        # The CDF formula is complex and involves piecewise functions
        # We'll verify it returns a value in [0,1] for valid inputs
        result = float(nelsen18_copula.cdf(u, v))
        assert 0 <= result <= 1


def test_cdf_boundary_conditions(nelsen18_copula):
    """Test boundary conditions for the Nelsen18 copula CDF."""
    # Test points along the boundaries
    u_values = [0.1, 0.3, 0.5, 0.7, 0.9]

    # C(u,0) = 0 for any u in [0,1]
    for u in u_values:
        assert np.isclose(float(nelsen18_copula.cdf(u, 0)), 0, atol=1e-10)
        assert np.isclose(float(nelsen18_copula.cdf(0, u)), 0, atol=1e-10)

    # C(u,1) = u and C(1,v) = v for any u,v in [0,1]
    for u in u_values:
        assert np.isclose(float(nelsen18_copula.cdf(u, 1)), u, rtol=1e-5)
        assert np.isclose(float(nelsen18_copula.cdf(1, u)), u, rtol=1e-5)


def test_conditional_distribution_2(nelsen18_copula):
    """Test the second conditional distribution function."""
    # Test conditional distribution at some valid points
    points = [(0.3, 0.4), (0.5, 0.5), (0.7, 0.2)]

    for u, v in points:
        # Just verify it returns a value in [0,1] without errors
        result = float(nelsen18_copula.cond_distr_2(u, v))
        assert 0 <= result <= 1


def test_lambda_l(nelsen18_copula):
    """Test lower tail dependence coefficient (lambda_L)."""
    # For Nelsen18, lambda_L = 0
    assert float(nelsen18_copula.lambda_L()) == 0


def test_lambda_u(nelsen18_copula):
    """Test upper tail dependence coefficient (lambda_U)."""
    # For Nelsen18, lambda_U = 1
    assert float(nelsen18_copula.lambda_U()) == 1


def test_theta_dependent_behavior():
    """Test how behavior changes with different theta values."""
    # Create copulas with different theta values
    copula1 = Nelsen18(2)
    copula2 = Nelsen18(5)
    copula3 = Nelsen18(10)

    # Test at a specific point
    u, v = 0.3, 0.4

    # Calculate CDF values
    cdf1 = float(copula1.cdf(u, v))
    cdf2 = float(copula2.cdf(u, v))
    cdf3 = float(copula3.cdf(u, v))

    # For Nelsen18, different theta should give different CDF values
    # Verify they are different
    assert cdf1 != cdf2 or cdf2 != cdf3

    # Verify they are all in valid range
    assert 0 <= cdf1 <= 1
    assert 0 <= cdf2 <= 1
    assert 0 <= cdf3 <= 1


def test_large_theta_behavior():
    """Test behavior with a very large theta value."""
    # Create a copula with a large theta
    large_theta = Nelsen18(100)

    # Test at some points
    points = [(0.3, 0.4), (0.5, 0.5), (0.7, 0.2)]

    for u, v in points:
        # Verify computation doesn't raise errors
        result = float(large_theta.cdf(u, v))

        # Result should be within [0,1]
        assert 0 <= result <= 1


def test_numerical_stability():
    """Test numerical stability at challenging points."""
    copula = Nelsen18(3)

    # Test at points very close to 0
    edge_points = [
        (0.001, 0.5),  # u close to 0
        (0.5, 0.001),  # v close to 0
    ]

    for u, v in edge_points:
        # Verify CDF computation doesn't raise errors
        result = float(copula.cdf(u, v))

        # Result should be within [0,1]
        assert 0 <= result <= 1

        # For very small u or v, result should be close to 0
        assert result < 0.1

    # Points close to 1 may cause numerical issues due to the (t-1) term in generator
    # Test carefully approaching 1
    approach_points = [
        (0.95, 0.5),
        (0.97, 0.5),
        (0.5, 0.95),
        (0.5, 0.97),
    ]

    for u, v in approach_points:
        # Just verify computation doesn't raise errors
        try:
            result = float(copula.cdf(u, v))
            # If we get a result, it should be in [0,1]
            assert 0 <= result <= 1
        except Exception as e:
            # If there's a numerical issue, print it but don't fail the test
            print(f"Numerical issue at ({u}, {v}): {e}")
