# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs-dev/03_cnn_virus_data.ipynb.

# %% auto 0
__all__ = ['CODE_ROOT', 'PACKAGE_ROOT', 'OriginalLabels', 'FastaFileReader', 'FastqFileReader', 'AlnFileReader',
           'create_infer_ds_from_fastq', 'strings_to_tensors', 'split_kmer_into_50mers', 'tfrecord_from_fastq',
           'tfrecord_from_text', 'get_dataset_from_tfr', 'combine_predictions', 'DataGenerator_from_50mer',
           'get_learning_weights', 'get_params_50mer', 'get_params_150mer', 'get_kmer_from_50mer',
           'get_kmer_from_150mer']

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 3
import collections
import json
import numpy as np
import pandas as pd
import os
import re
import tensorflow as tf
import warnings

from ecutilities.core import validate_path
from functools import partial, partialmethod
from ..bio import q_score2prob_error
from ..core import TextFileBaseReader, ProjectFileSystem
from pathlib import Path
from tqdm.notebook import tqdm, trange
from typing import Any, Optional

os.environ['TF_CPP_MIN_LOG_LEVEL'] = '3' # or any {'0', '1', '2'}
import tensorflow as tf
from tensorflow.io import serialize_tensor, FixedLenFeature
from tensorflow.keras.utils import Sequence, to_categorical
from tensorflow.data import TextLineDataset, TFRecordDataset

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 4
# Retrieve the package root
from .. import __file__
CODE_ROOT = Path(__file__).parents[0]
PACKAGE_ROOT = Path(__file__).parents[1]

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 14
class OriginalLabels:
    """Converts labels to species for original data"""
    def __init__(self, p2mapping=None):
        if p2mapping is None:
            p2mapping = ProjectFileSystem().data / 'CNN_Virus_data/virus_name_mapping'
        assert p2mapping.is_file()
        df = pd.read_csv(p2mapping, sep='\t', header=None, names=['species', 'label'])
        self._label2species = df['species'].to_list()
        self._label2species.append('Unknown Virus Species')
        self._species2label = {specie:label for specie, label in zip(df['species'], df['label'])}
        self._species2label['Unknown Virus Species'] = len(self._label2species)

    def label2species(self, n:int):
        return self._label2species[n]

    def species2label(self, s:str):
        return self._species2label[s]

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 30
class FastaFileReader(TextFileBaseReader):
    """Wrap a FASTA file and retrieve its content in raw format and parsed format"""
    def __init__(
        self,
        path: str|Path,  # path to the Fasta file
    ):
        super().__init__(path, nlines=1)
        self.text_to_parse_key = 'definition line'
        self.set_parsing_rules(verbose=False)
        
    def __next__(self)-> dict[str, str]:   # `{'definition line': text in dfn line, 'sequence': full sequence as str}` 
        """Return one definition line and the corresponding sequence"""
        lines = []
        for i in range(2):
            lines.append(self._safe_readline())
        dfn_line = lines[0].replace('\n', '')   #remove the next line symbol at the end of the line
        sequence = lines[1].replace('\n', '')   #remove the next line symbol at the end of the line
        return {'definition line':dfn_line, 'sequence':f"{sequence}"}
    
    def print_first_chunks(
        self, 
        nchunks:int=3,  # number of chunks to print out
    ):
        """Print the first `nchunks` chunks of text from the file"""
        self.reset_iterator()
        for i, seq_dict in enumerate(self.__iter__()):
            print(f"\nSequence {i+1}:")
            print(seq_dict['definition line'])
            print(f"{seq_dict['sequence'][:80]} ...")
            if i >= nchunks-1: break
        self.reset_iterator()
            
    def parse_file(
        self,
        add_seq :bool=False,     # When True, add the full sequence to the parsed metadata dictionary
        save_json: bool=False    # When True, save the file metadata as a json file of same stem name
    )-> dict[str]:               # Metadata as Key/Values pairs
        """Read fasta file and return a dictionary with definition line metadata and optionally sequences"""
    
        self.reset_iterator()
        parsed = {}
        for d in self:
            dfn_line = d['definition line']
            seq = d['sequence']
            metadata = self._parse_text_fn(dfn_line, self.re_pattern, self.re_keys)
            if add_seq: metadata['sequence'] = seq         
            parsed[metadata['seqid']] = metadata
                        
        if save_json:
            p2json = self.path.parent / f"{self.path.stem}_metadata.json"
            with open(p2json, 'w') as fp:
                json.dump(parsed, fp, indent=4)
                print(f"Metadata for '{self.path.name}'> saved as <{p2json.name}> in  \n{p2json.parent.absolute()}\n")

        return parsed

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 76
class FastqFileReader(TextFileBaseReader):
    """Iterator going through a fastq file's sequences and return each section + prob error as a dict"""
    def __init__(
        self,
        path:str|Path,   # path to the fastq file
    )-> dict:           # key/value with keys: definition line; sequence; q score; prob error
        self.nlines = 4
        super().__init__(path, nlines=self.nlines)
        self.text_to_parse_key = 'definition line'
        self.set_parsing_rules(verbose=False)        
    
    def __next__(self):
        """Return definition line, sequence and quality scores"""
        lines = []
        for i in range(self.nlines):
            lines.append(self._safe_readline().replace('\n', ''))
        
        output = {
            'definition line':lines[0], 
            'sequence':f"{lines[1]}", 
            'read_qscores': f"{lines[3]}",
        }
        output['probs error'] = np.array([q_score2prob_error(q) for q in output['read_qscores']])
        
        return output
    
    def print_first_chunks(
        self, 
        nchunks:int=3,  # number of chunks to print out
    ):
        """Print the first `nchunks` chunks of text from the file"""
        for i, seq_dict in enumerate(self.__iter__()):
            print(f"\nSequence {i+1}:")
            print(seq_dict['definition line'])
            print(f"{seq_dict['sequence'][:80]} ...")
            if i >= nchunks: break
            
    def parse_file(
        self,
        add_readseq :bool=False,    # When True, add the full sequence to the parsed metadata dictionary
        add_qscores:bool=False,     # Add the read ASCII Q Scores to the parsed dictionary when True
        add_probs_error:bool=False, # Add the read probability of error to the parsed dictionary when True
        save_json: bool=False       # When True, save the file metadata as a json file of same stem name
    )-> dict[str]:                  # Metadata as Key/Values pairs
        """Read fastq file, return a dict with definition line metadata and optionally read sequence and q scores, ..."""
    
        self.reset_iterator()
        parsed = {}
        for d in self:
            dfn_line = d['definition line']
            seq, q_scores, prob_e = d['sequence'], d['read_qscores'], d['probs error']
            metadata = self._parse_text_fn(dfn_line, self.re_pattern, self.re_keys)
            if add_readseq: metadata['readseq'] = seq         
            if add_qscores: metadata['read_qscores'] = q_scores
            if add_probs_error: metadata['probs error'] = prob_e
            parsed[metadata['readid']] = metadata 
                        
        if save_json:
            p2json = self.path.parent / f"{self.path.stem}_metadata.json"
            with open(p2json, 'w') as fp:
                json.dump(parsed, fp, indent=4)
                print(f"Metadata for '{self.path.name}'> saved as <{p2json.name}> in  \n{p2json.parent.absolute()}\n")

        return parsed

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 91
class AlnFileReader(TextFileBaseReader):
    """Iterator going through an ALN file"""
    def __init__(
        self,
        path:str|Path,   # path to the aln file
    )-> dict:            # key/value with keys: 
        """Set TextFileBaseReader attributes and specific class attributes"""
        self.nlines = 1
        super().__init__(path, nlines=self.nlines)
        self.header = self.read_header()
        self.nlines = 3
        self.text_to_parse_key = 'definition line'
        self.set_parsing_rules(verbose=False)
        self.set_header_parsing_rules(verbose=False)
        self.ref_sequences = self.parse_header_reference_sequences()

    def __next__(self):
        """Return definition line, sequence and quality scores"""
        lines = []
        for i in range(self.nlines):
            lines.append(self._safe_readline().replace('\n', ''))

        output = {
            'definition line':lines[0], 
            'ref_seq_aligned':f"{lines[1]}", 
            'read_seq_aligned': f"{lines[2]}",
        }   
        return output
    
    def read_header(self):
        """Read ALN file Header and return each section parsed in a dictionary"""
        
        header = {}
        if self.fp is not None:
            self.fp.close()
        self.fp = open(self.path, 'r')
        
        line = self._safe_readline().replace('\n', '')
        if not line.startswith('##ART_Illumina'): 
            raise ValueError(f"Header of this file does not start with ##ART_Illumina")
        line = self._safe_readline().replace('\n', '')
        if not line.startswith('@CM'): 
            raise ValueError(f"First header line should start with @CM")
        else: 
            header['command'] = line[3:].replace('\t', '').strip()

        refseqs = []
        while True:
            line = self._safe_readline().replace('\n', '')
            if line.startswith('##Header End'): break
            else:
                refseqs.append(line)
        header['reference sequences'] = refseqs
        
        return header
    
    def reset_iterator(self):
        """Reset the iterator to point to the first line in the file, by recreating a new file handle.
        
        `AlnFileReader` requires a specific `reset_iterator` method, in order to skip the header every time it is reset
        """
        if self.fp is not None:
            self.fp.close()
        self.fp = open(self.path, 'r')
        while True:
            line = self._safe_readline().replace('\n', '')
            if line.startswith('##Header End'): break

    def parse_definition_line_with_position(
        self, 
        dfn_line:str    # fefinition line string to be parsed
        )-> dict:       # parsed metadata in key/value format + relative position of the read
        """Parse definition line and adds relative position"""
        read_meta = self.parse_text(dfn_line)
        read_refseqid = read_meta['refseqid']
        read_start_pos = int(read_meta['aln_start_pos'])
        read_refseq_lentgh = int(self.ref_sequences[read_refseqid]['refseq_length'])
        read_meta['read_pos'] = (read_start_pos *10)// read_refseq_lentgh + 1
        return read_meta
    
    def parse_file(
        self, 
        add_ref_seq_aligned:bool=False,   # Add the reference sequence aligned to the parsed dictionary when True
        add_read_seq_aligned:bool=False,  # Add the read sequence aligned to the parsed dictionary when True
    )-> dict[str]: 
        # Key/Values. Keys: 
        # `readid`,`seqid`,`seq_nbr`,`read_nbr`,`aln_start_pos`,`ref_seq_strand`
        # optionaly `ref_seq_aligned`,`read_seq_aligned`
        """Read ALN file, return a dict w/ alignment info for each read and optionaly aligned reference sequence & read"""
        self.reset_iterator()
        parsed = {}
        for d in self:
            dfn_line = d['definition line']
            ref_seq_aligned, read_seq_aligned = d['ref_seq_aligned'], d['read_seq_aligned']
            metadata = self.parse_text(dfn_line)
            if add_ref_seq_aligned: metadata['ref_seq_aligned'] = ref_seq_aligned         
            if add_read_seq_aligned: metadata['read_seq_aligned'] = read_seq_aligned
            parsed[metadata['readid']] = metadata 
        return parsed

    def parse_header_reference_sequences(
        self,
        pattern:str|None=None,     # regex pattern to apply to parse the reference sequence info
        keys:list[str]|None=None,  # list of keys: keys are both regex match group names and corresponding output dict keys 
        )->dict[str]:                  # parsed metadata in key/value format
        """Extract metadata from all header reference sequences"""
        if pattern is None and keys is None:
            pattern, keys = self.re_header_pattern, self.re_header_keys
        parsed = {}
        for seq_dfn_line in self.header['reference sequences']:
            metadata = self.parse_text(seq_dfn_line, pattern, keys)
            parsed[metadata['refseqid']] = metadata
            
        return parsed       
        
    def set_header_parsing_rules(
        self,
        pattern: str|bool=None,   # regex pattern to apply to parse the text, search in parsing rules json if None
        keys: list[str]=None,     # list of keys/group for regex, search in parsing rules json if None
        verbose: bool=False       # when True, provides information on each rule
    )-> None:
        """Set the regex parsing rule for reference sequence in ALN header.
               
        Updates 3 class attributes: `re_header_rule_name`, `re_header_pattern`, `re_header_keys`
        
        TODO: refactor this and the method in Core: to use a single function for the common part and a parameter for the text_to_parse 
        """
        
        P2JSON = Path(f"{PACKAGE_ROOT}/default_parsing_rules.json")
        
        self.re_header_rule_name = None
        self.re_header_pattern = None
        self.re_header_keys = None
        
        # get the first reference sequence definition line in header
        text_to_parse = self.header['reference sequences'][0]
        divider_line = f"{'-'*80}"

        if pattern is not None and keys is not None:  # When specific pattern and keys are passed
            try:
                metadata_dict = self.parse_text(text_to_parse, pattern, keys)
                self.re_header_rule_name = 'Custom Rule'
                self.re_header_pattern = pattern
                self.re_header_keys = keys
                if verbose:
                    print(divider_line)
                    print(f"Custom rule was set for header in this instance.")
            except Exception as err: 
                raise ValueError(f"The pattern generates the following error:\n{err}")
                
        else:  # automatic rule selection among rules saved in json file
            # Load all existing rules from json file
            with open(P2JSON, 'r') as fp:
                parsing_rules = json.load(fp)
                
            # test all existing rules and keep the one with highest number of matches
            max_nbr_matches = 0
            for k, v in parsing_rules.items():
                re_header_pattern = v['pattern']
                re_header_keys = v['keys'].split(' ')
                try:
                    metadata_dict = self.parse_text(text_to_parse, re_header_pattern, re_header_keys)
                    nbr_matches = len(metadata_dict)
                    if verbose:
                        print(divider_line)
                        print(f"Rule <{k}> generated {nbr_matches:,d} matches")
                        print(divider_line)
                        print(re_header_pattern)
                        print(re_header_keys)

                    if len(metadata_dict) > max_nbr_matches:
                        self.re_header_pattern = re_header_pattern
                        self.re_header_keys = re_header_keys
                        self.re_header_rule_name = k    
                except Exception as err:
                    if verbose:
                        print(divider_line)
                        print(f"Rule <{k}> generated an error")
                        print(err)
                    else:
                        pass
            if self.re_header_rule_name is None:
                msg = """
        None of the saved parsing rules were able to extract metadata from the first line in this file.
        You must set a custom rule (pattern + keys) before parsing text, by using:
            `self.set_parsing_rules(custom_pattern, custom_list_of_keys)`
                """
                warnings.warn(msg, category=UserWarning)
            
            if verbose:
                print(divider_line)
                print(f"Selected rule with most matches: {self.re_header_rule_name}")

            # We used the iterator, now we need to reset it to make all lines available
            self.reset_iterator()

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 128
def create_infer_ds_from_fastq(
    p2fastq: str|Path,             # Path to the fastq file (aln file path is inferred)
    output_dir:str|Path|None=None, # Path to directory where ds file will be saved
    overwrite_ds:bool=False,       # If True, overwrite existing ds file. If False, error is raised if ds file exists
    nsamples:int|None=None         # Used to limit the number of reads to use for inference, use all if None
)-> (Path, Path, pd.DataFrame):    # Paths to dataset file, path to metadata file, dataframe with metadata
    """Build an inference dataset file as required by the CNN Virus model from a simreads fastq (ART format).
    
    Also extract the fastq read sequence metadata, saves it in a metadata file and returns them as a DataFrame
    """
    fastq = FastqFileReader(p2fastq)
    aln = AlnFileReader(p2fastq.parent / f"{p2fastq.stem}.aln")
    
    if output_dir is None:
        p2outdir = Path()
    else:
        validate_path(output_dir, path_type='dir', raise_error=True)
        p2outdir = output_dir if isinstance(output_dir, Path) else Path(output_dir)
    
    p2dataset = p2outdir / f"{p2fastq.stem}_ds"
    p2metadata = p2outdir / f"{p2fastq.stem}_metadata.csv"
    
    if p2dataset.is_file():
        if overwrite_ds: 
            p2dataset.unlink()
            if p2metadata.is_file(): p2metadata.unlink()
        else:
            raise ValueError(f"{p2dataset.name} already exists in {p2dataset.absolute()}")
    p2dataset.touch()
    p2metadata.touch()
    
    read_ids = []
    read_refseqs = []
    read_start_pos = []
    read_strand = []
    
    with open(p2dataset, 'a') as fp:
        i = 1
        for fastq_chunk, aln_chunk in tqdm(zip(fastq, aln)):
            seq = fastq_chunk['sequence']
            
            aln_meta = aln.parse_text(aln_chunk['definition line'])
            read_ids.append(aln_meta['readid'])
            read_refseqs.append(aln_meta['refseqid'])
            read_start_pos.append(aln_meta['aln_start_pos'])
            read_strand.append(aln_meta['refseq_strand'])

            label = 0
            pos = 0
            fp.write(f"{seq}\t{label}\t{pos}\n")

            i += 1
            if nsamples:
                if i > nsamples: break
                    
    print(f"Dataset with {i-1:,d} reads")

    metadata = np.array(list(zip(read_ids, read_refseqs, read_start_pos, read_strand)))
    metadata = pd.DataFrame(data={
                'read_ids': read_ids,
                'read_refseqs': read_refseqs,
                'read_start_pos': read_start_pos,
                'read_strand': read_strand})
    metadata.to_csv(p2metadata, index=True)
    
    return p2dataset, p2metadata, metadata

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 134
def strings_to_tensors(
    b: tf.Tensor        # batch of strings 
    ):
    """Function converting a batch of bp strings into three tensors: (x_seqs, (y_labels, y_pos))"""
    
    # Split the string in three : returns a ragged tensor which needs to be converted into a normal tensor using .to_tensor()
    t = tf.strings.split(b, '\t').to_tensor(default_value = '', shape=[None, 3])

    # Split each sequence string into a list of single base strings:
    # 'TCAAAATAATCA' -> ['T','C','A','A','A','A','T','A','A','T','C','A']
    seqs = tf.strings.bytes_split(t[:, 0]).to_tensor(shape=(None, 50))


    # BHE sequences
    # Each base letter (A, C, G, T, N) is replaced by a OHE vector
    #     "A" converted into [1,0,0,0,0]
    #     "C" converted into [0,1,0,0,0]
    #     "G" converted into [0,0,1,0,0]
    #     "T" converted into [0,0,0,1,0]
    #     "N" converted into [0,0,0,0,1]
    # 
    # Technical Notes:
    # a. The batch of sequence `seqs` has a shape (batch_size, 50) after splitting each byte. 
    #    Must flatten it first, then apply the transform on each base, then reshape to original shape
    # b. We need to map each letter to one vector/tensor. 
    #    1. Cast bytes seqs into integer sequence (uint8 to work byte by byte)
    #    2. For each base letter (A, C, G, T, N) create one tensor (batch_size, 50) (seqs_A, _C, _G, _T, _N)
    #    3. Value is 1 if it is the base in the sequence, otherwise 0
    #    4. Concatenate these 5 tensors into a tensor of shape (batch_size, 50, 5)
 
    seqs_uint8 = tf.io.decode_raw(seqs, out_type=tf.uint8)
    # note: tf.io.decode_raw adds one dimension at the end in the process
    #       [b'C', b'A', b'T'] will return [[67], [65], [84]] and not [67, 65, 84]
    #       this is actually what we want to contatenate the values for each base letter

    A, C, G, T, N = 65, 67, 71, 84, 78

    seqs_A = tf.cast(seqs_uint8 == A, tf.float32)
    seqs_C = tf.cast(seqs_uint8 == C, tf.float32)
    seqs_G = tf.cast(seqs_uint8 == G, tf.float32)
    seqs_T = tf.cast(seqs_uint8 == T, tf.float32)
    seqs_N = tf.cast(seqs_uint8 == N , tf.float32)

    x_seqs = tf.concat([seqs_A, seqs_C, seqs_G, seqs_T, seqs_N], axis=2)

    # OHE labels
    n_labels = 187
    y_labels = tf.strings.to_number(t[:, 1], out_type=tf.int32)
    y_labels = tf.gather(tf.eye(n_labels), y_labels)

    # OHE positions
    n_pos = 10
    y_pos = tf.strings.to_number(t[:, 2], out_type=tf.int32)
    y_pos= tf.gather(tf.eye(n_pos), y_pos)

    return (x_seqs, (y_labels, y_pos))

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 137
def _bytes_feature(value):
    """Returns a bytes_list from a string / byte."""
    if isinstance(value, type(tf.constant(0))): # if value is tensor
        value = value.numpy() # get value of tensor
    return tf.train.Feature(bytes_list=tf.train.BytesList(value=[value]))

def _float_feature(value):
  """Returns a floast_list from a float / double."""
  return tf.train.Feature(float_list=tf.train.FloatList(value=[value]))

def _int64_feature(value):
  """Returns an int64_list from a bool / enum / int / uint."""
  return tf.train.Feature(int64_list=tf.train.Int64List(value=[value]))

def _serialize_array(array):
  array = serialize_tensor(array)
  return array

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 138
def _base_hot_encode(
    line: str        # one string (one line in text dataset)
    ):
    """Convert a line from text dataset into three tensors: read sequence (BHE), virus label and position"""
    
    # Split the line (string) in three : read, label, position
    t = tf.strings.split(line.replace('\n', ''), '\t')

    # Split the sequence string into a list of single base strings:
    # 'TCAAAATAATCA' -> ['T','C','A','A','A','A','T','A','A','T','C','A']
    read = tf.strings.bytes_split(t[0])

    # Base Hot Encode sequences (BHE)
    # Each base letter (A, C, G, T, N) is replaced by a OHE vector
    #     "A" converted into [1,0,0,0,0]
    #     "C" converted into [0,1,0,0,0]
    #     "G" converted into [0,0,1,0,0]
    #     "T" converted into [0,0,0,1,0]
    #     "N" converted into [0,0,0,0,1]
    
    # Decode the base letters A, C, ... into their ASCII code for easy conversion into BHE
    # ASCII code for A, C, G, T and N:
    A, C, G, T, N = 65, 67, 71, 84, 78
    read_uint8 = tf.io.decode_raw(read, out_type=tf.uint8)

    # Technical Notes: 
    #   tf.io.decode_raw adds one dimension at the end in the process
    #   [b'C', b'A', b'T'] will return [[67], [65], [84]] and not [67, 65, 84]
    #   this is actually what we want to contatenate the values for each base letter
    read_A = tf.cast(read_uint8 == A, tf.float32)
    read_C = tf.cast(read_uint8 == C, tf.float32)
    read_G = tf.cast(read_uint8 == G, tf.float32)
    read_T = tf.cast(read_uint8 == T, tf.float32)
    read_N = tf.cast(read_uint8 == N , tf.float32)
    x_reads = tf.concat([read_A, read_C, read_G, read_T, read_N], axis=1)

    # OHE labels
    n_labels = 187
    y_labels = tf.strings.to_number(t[1], out_type=tf.int32) # int32 so it can be used an index in gather
    y_labels = tf.gather(tf.eye(n_labels, dtype=tf.float32), y_labels)

    # OHE positions
    n_pos = 10
    y_pos = tf.strings.to_number(t[2], out_type=tf.int32) # int32 so it can be used an index in gather
    y_pos= tf.gather(tf.eye(n_pos, dtype=tf.float32), y_pos)

    return x_reads, y_labels, y_pos

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 139
def split_kmer_into_50mers(
    kmer: tf.Tensor        # tensor representing a single k-mer read, after base
    ):
    """Converts a k-mer read into several 50-mer reads, by shifting the k-mer one base at a time."""
    def fn(accumulated, elem):
        return tf.roll(accumulated, shift=-elem, axis=0)

    k, n = kmer.shape[0], 50
    nb_splits = (k-n+1)
    # Create a tensor of integers, with a 0 as first element and 1 for all other elements, for shifts
    shifts = np.ones(shape=nb_splits, dtype=int)
    shifts[0] = 0 
    shifts = tf.convert_to_tensor(shifts, dtype=tf.int64)
    # Use tf.scan to shift the original read nb_splits times
    shifted = tf.scan(fn, shifts, kmer, reverse=False)
    return shifted[:, :n, :]  # return the tensor with shifted kmer, sliced to only keep 50 bases

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 140
def tfrecord_from_fastq(
    p2fastq:Path,              # Path to the fastaq file (should be associated with a aln file)
    p2tfrds:Path|None=None,    # Path to the TFRecord file, default creates a file in saved directory
    overwrite:bool=False       # When True, overides any existing file, When False, raises an error
    ) -> (Path, Path):         # Paths to the saved TFRecord file and the metadata csv file
    """Creates a TFRecord dataset for inference from fastq and aln files, as well as a csv metadata file

    The TFRecord dataset can be used for training or prediction, using the original CNN Virus model.
    The metadata file is a Pandas DataFrame converted into csv
    """
    # Setup paths
    if p2tfrds is None:
        p2tfrds = ProjectFileSystem().data / 'saved/cnn_virus_datasets' / f"{p2fastq.stem}.tfrecords"
    p2metadata = p2tfrds.parent / f"{p2tfrds.stem}.metadata"

    if p2tfrds.exists():
        if overwrite:
            p2tfrds.unlink()
            if p2metadata.exists(): p2metadata.unlink()
        else: 
            raise ValueError(f"{p2tfrds.name} already exists. To overwrite, set parameter `overwrite` to True")

    p2aln = p2fastq.parent / f"{p2fastq.stem}.aln"
    assert p2aln.is_file(), f"No ALN file associated with {fastq.name}"

    # Create file readers
    fastq = FastqFileReader(p2fastq)
    aln = AlnFileReader(p2aln)

    read_ids, read_refseqs, read_start_pos, read_strand = [], [], [], []

    # Create TFRecord writer and loop through reads
    writer = tf.io.TFRecordWriter(str(p2tfrds.absolute())) 
    for i, (fastq_element, aln_element) in tqdm(enumerate(zip(fastq, aln))):
        # Extract read text sequence from fastq and metadata from aln files
        seq = fastq_element['sequence']           
        aln_meta = aln.parse_text(aln_element['definition line'])
        read_ids.append(aln_meta['readid'])
        read_refseqs.append(aln_meta['refseqid'])
        read_start_pos.append(aln_meta['aln_start_pos'])
        read_strand.append(aln_meta['refseq_strand'])

        # Create and write one Example, including BHE sequence, the label and the position
        bhe_seq, label, pos = _base_hot_encode(f"{seq}\t{0}\t{0}\n")
        data = {
            'read' : _bytes_feature(_serialize_array(bhe_seq)),
            'label' : _bytes_feature(_serialize_array(label)),
            'pos' : _bytes_feature(_serialize_array(pos))
        }
        out = tf.train.Example(features=tf.train.Features(feature=data))
        writer.write(out.SerializeToString())

    writer.close()
    print(f"Wrote {i+1} reads to TFRecord file {p2tfrds.name}")

    metadata = np.array(list(zip(read_ids, read_refseqs, read_start_pos, read_strand)))
    metadata = pd.DataFrame(data={
                'read_ids': read_ids,
                'read_refseqs': read_refseqs,
                'read_start_pos': read_start_pos,
                'read_strand': read_strand})
    metadata.to_csv(p2metadata, index=True)
    
    return p2tfrds, p2metadata, metadata

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 141
def tfrecord_from_text(
    p2ds,                      # Path to the text dataset, in the format of original CNN Virus data
    p2tfrds:Path|None=None,    # Path to the TFRecord file, default creates a file in savec directory
    overwrite:bool=False       # When True, overides any existing file, When False, raises an error
    ) -> Path:                 # Path to the saved TFRecord file
    # Setup paths
    if p2tfrds is None:
        p2tfrds = ProjectFileSystem().data / 'saved/cnn_virus_datasets' / f"{p2ds.stem}.tfrecords"
    # p2metadata = p2tfrds.parent / f"{p2tfrds.stem}.metadata"

    if p2tfrds.exists():
        if overwrite:
            p2tfrds.unlink()
            # if p2metadata.exists(): p2metadata.unlink()
        else: 
            raise ValueError(f"{p2tfrds.name} already exists. To overwrite, set parameter `overwrite` to True")

    reads = TextFileBaseReader(p2ds, nlines=1)
    writer = tf.io.TFRecordWriter(str(p2tfrds.absolute())) 

    for i, line in enumerate(reads):
        # Create and write one Example, including BHE sequence, the label and the position
        bhe_seq, label, pos = _base_hot_encode(line)
        data = {
            'read' : _bytes_feature(_serialize_array(bhe_seq)),
            'label' : _bytes_feature(_serialize_array(label)),
            'pos' : _bytes_feature(_serialize_array(pos))
            }
        out = tf.train.Example(features=tf.train.Features(feature=data))
        writer.write(out.SerializeToString())

    writer.close()
    print(f"Wrote {i+1} reads to TFRecord")
    return p2tfrds

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 142
def _parse_tfr_element(element):
    # Define the underlying structure of the data (mirror the dta structure above)
    data = {    
        'read' : FixedLenFeature([], tf.string),
        'label' : FixedLenFeature([], tf.string),
        'pos' : FixedLenFeature([], tf.string) 
    }

    content = tf.io.parse_single_example(element, data)
  
    read_bytes = content['read']
    label_bytes = content['label']
    pos_bytes = content['pos']
    
    # Parse the string tensor into a real tensors, with proper types
    read = tf.io.parse_tensor(read_bytes, out_type=tf.float32)
    label = tf.io.parse_tensor(label_bytes, out_type=tf.float32)
    pos = tf.io.parse_tensor(pos_bytes, out_type=tf.float32)
    
    return (read, (label, pos))

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 143
def get_dataset_from_tfr(
    p2tfrds:Path,      # Path to the TFRecord dataset
    batch_size:int = 1 # Desired batch side for the dataset
    ) -> tf.data.Dataset: # dataset, formated with the batch size
    # Create a dataset from the TFRecord file
    dataset = tf.data.TFRecordDataset(p2tfrds)
    # Convert the strings into the proper format using the parsing function
    dataset = dataset.map(_parse_tfr_element)
    return dataset.batch(batch_size)

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 150
def combine_predictions(labels, label_probs, positions):
    """Combine set of 50-mer predictions into one final prediction for label and position"""

    # Filter our any prediction with low predicted probability 
    valid_preds_mask = tf.reduce_max(label_probs, axis=1) >= 0.9
    valid_labels = labels[valid_preds_mask].numpy()
    valid_positions = positions[valid_preds_mask].numpy()

    # Return prediction outside the label and position ranges if no valid prediction
    if len(valid_labels) == 0:
        return 187, 10

    # Take most frequent label, and most frequent position for the selected label
    else:
        unique_labels, _, counts = tf.unique_with_counts(valid_labels)
        combined_label = unique_labels[tf.argmax(counts)].numpy()

        df = pd.DataFrame({'labels': valid_labels, 'positions': valid_positions})
        gb = df.groupby('labels')
        counter_pos = collections.Counter(df.positions.iloc[gb.groups[combined_label]].tolist())
        combined_pos = counter_pos.most_common(1)[0][0]
        return combined_label, combined_pos

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 154
class DataGenerator_from_50mer(Sequence):
    """data generator for generating batches of data from 50-mers"""

    d_nucl = {"A": 0,"C": 1,"G": 2,"T": 3,"N":4}

    def __init__(self, f_matrix, f_labels, f_pos, batch_size=1024,n_classes=187, shuffle=True):
        self.batch_size = batch_size
        self.labels = f_labels
        self.matrix = f_matrix
        self.pos = f_pos
        self.n_classes = n_classes
        self.shuffle = shuffle
        self.on_epoch_end()
    def __len__(self):
        return int(np.ceil(len(self.labels) / self.batch_size))
    def __getitem__(self, index):
        indexes = self.indexes[index*self.batch_size:(index+1)*self.batch_size]
        X, y= self.__data_generation(indexes)
        return X,y
    def on_epoch_end(self):
        self.indexes = np.arange(len(self.labels))
        if self.shuffle == True:
            np.random.shuffle(self.indexes)
    def __data_generation(self, index):
        x_train=[]
        for i in index:
            seq=self.matrix[i]
            seq_list=[j for j in seq]
            x_train.append(seq_list)
        x_train=np.array(x_train)
        x_tensor=np.zeros(list(x_train.shape)+[5])
        for row in range(len(x_train)):
            for col in range(50):
                x_tensor[row,col,self.d_nucl[x_train[row,col]]]=1
        y_pos=[]
        y_label=[self.labels[i] for i in index]
        y_label=np.array(y_label)
        y_label=to_categorical(y_label, num_classes=self.n_classes)
        y_pos=[self.pos[i] for i in index]
        y_pos=np.array(y_pos)
        y_pos=to_categorical(y_pos, num_classes=10)
        return x_tensor,{'labels': y_label, 'pos': y_pos}

# %% ../../nbs-dev/03_cnn_virus_data.ipynb 156
def get_learning_weights(filepath):
    """get different learning weights for different classes, from file"""
    f = open(filepath,"r").readlines()
    d_weights = {}
    for i in f:
        i = i.strip().split("\t")
        d_weights[float(i[0])]=float(i[1])
    return d_weights

def get_params_50mer():
    """set default params for generating batches of 50-mer"""
    params = {'batch_size': 1024,
    'n_classes': 187,
    'shuffle': True}
    return params

def get_params_150mer():
    """ set default params for generating batches of 150-mer"""
    params = {'batch_size': 101,
    'n_classes': 187,
    'shuffle': False}
    return params

def get_kmer_from_50mer(filepath, max_seqs=None):
    """Load data from sequence file and returns three tensors, with max nbr sequences"""
    f_matrix=[]
    f_labels=[]
    f_pos=[]
    with open(filepath, 'r') as fp:
        i = 0
        while True:
            line = fp.readline()
            i += 1
            # EOF
            if line == '':
                break
            # Reached max number of k-mers to load from file
            elif max_seqs is not None and i > max_seqs:
                break
            else:
                seq, label, pos = line.strip().split('\t')
                f_matrix.append(seq)
                f_labels.append(label)
                f_pos.append(pos)
    return f_matrix,f_labels,f_pos

def get_kmer_from_150mer(filepath, max_seqs=None):
    """Load data from sequence file and returns three tensors, with max nbr sequences"""
    f_matrix=[]
    f_labels=[]
    f_pos=[]
    with open(filepath,"r") as fp:
        i = 0
        while True:
            line = fp.readline()
            i += 1
            # EOF
            if line == '':
                break
            # Reached max number of k-mers to load from file
            elif max_seqs is not None and i > max_seqs:
                break
            else:
                seq, label, pos = line.strip().split('\t')
                # Split 150-mer into 101 50-mers, shifted by one nucleotide
                for i in range(len(seq)-49):
                    kmer=seq[i:i+50]
                    f_matrix.append(kmer)
                    f_labels.append(label)
                    f_pos.append(pos)
    return f_matrix,f_labels,f_pos
