#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
Search for a default credentials using the DefaultCreds-cheat-sheet dataset.
inspired by @ncrocfer whatportis project.
'''
__author__ = "ihebski"
__version__ = "0.4"
__status__ = "Development"
__codename__ = 'creds'
__source__ ="https://github.com/ihebski/DefaultCreds-cheat-sheet"

from tinydb import TinyDB,Query,where
from tinydb.storages import JSONStorage
from tinydb.middlewares import CachingMiddleware
from prettytable import PrettyTable
from pathlib import Path
import csv
import requests
import fire
import pathlib
import hashlib


path = pathlib.Path(__file__).parent
db = TinyDB(f"{path}/DefaultCreds_db.json",storage=CachingMiddleware(JSONStorage))
DefaultCreds_CSV_FILE = "https://raw.githubusercontent.com/ihebski/DefaultCreds-cheat-sheet/main/DefaultCreds-Cheat-Sheet.csv"

def get_db(path=path,msg="[+] Download database..."):
    """
    This function downloads the DefaultCreds-Cheat-Sheet.csv file and converted into a json database.
    https://tinydb.readthedocs.io/en/latest/usage.html
    """
    db = TinyDB(f"{path}/DefaultCreds_db.json",storage=CachingMiddleware(JSONStorage))
    vendor = []
    r = requests.get(DefaultCreds_CSV_FILE).content.decode("utf-8")
    print(msg)
    data = csv.reader(r.splitlines())
    for row in data:
        vendor.append(
            {
                "product" : row[0].lower().strip() if row[0] else "-",
                "username" : row[1].strip() if row[0] else "-",
                "password" : row[2].strip() if row[0] else "-"
            }
        )
    db.truncate()
    db.insert_multiple(vendor)
    db.close()

def export_creds(product,keyword):
    """
    Export usernames and passwords to separated files
    :param list(dict(product))
    :param keyword
    :return 2 files
    """
    # username and passwords default files paths
    username_path = f"/tmp/{keyword}-usernames.txt"
    passwords_path = f"/tmp/{keyword}-passwords.txt"

    # Create lists for (usernames,passwords) and remove duplicates
    username_list, password_list = set([row.get("username").replace('<blank>','') for row in product]), set([row.get("password").replace('<blank>','') for row in product])
    
    # Create files if not exist
    if not all([Path(username_path).is_file(),Path(passwords_path).is_file()]):
        with open(f'{username_path}', "w") as outfile_usernames, open(f'{passwords_path}', "w") as outfile_passwords:
            outfile_usernames.write("\n".join(username_list))
            outfile_passwords.write("\n".join(password_list))
        print(f"\n[+] Creds saved to {username_path} , {passwords_path} 📥")
    else:
        print(f"\n[!] Creds already exists under {username_path} , {passwords_path} ⛔️")

def print_table(product,keyword,export):
    """
    This function returns a pretty table used to display the results.
    :param list of searched products
    https://pypi.org/project/prettytable/
    """
    if len(product) == 0:
        print("[-] Product not found in database 🦕")
    else:
        table = PrettyTable(["Product", "username", "password"])
        table.align["Product"] = "l"
        table.padding_width = 1
        for row in product:
            table.add_row([row.get("product"),row.get("username"),row.get("password")])
        print(table)
        if export: export_creds(product,keyword)

def search(keyword,export=False):
    """
    This function search for a product using like statement
    :param keyword
    :return table
    """
    if len(db.all()) == 0:
        get_db()
        print_table(db.search(where("product").search(keyword.lower())),keyword,export)
    else:
        print_table(db.search(where("product").search(keyword.lower())),keyword,export)

def sha256sum(filename):
    """
    This function is used to compare two file hashes
    :param filename path
    :return sha256sum hash
    """
    h  = hashlib.sha256()
    b  = bytearray(128*1024)
    mv = memoryview(b)
    with open(filename, 'rb', buffering=0) as f:
        while n := f.readinto(mv):
            h.update(mv[:n])
    return h.hexdigest()

def update():
    """
    Update database records
    """
    get_db('/tmp',"Check for new updates...🔍")
    print('[+] No action needed, We are fine 🤘') if sha256sum('/tmp/DefaultCreds_db.json') == sha256sum(f"{path}/DefaultCreds_db.json") else get_db(path,"New updates are available 🚧\n[+] Download database...")


def run():
    fire.Fire()

if __name__ == "__main__":
    run()
